/*

   window.c - xslideshow main routine.

   Author:   Susumu Shiohara (shiohara@tpp.epson.co.jp)

		  Copyright 1993-1997 by Susumu Shiohara

			   All Rights Reserved

*/
#define IS_MAIN

#include "vroot.h"
#include "xslideshow.h"
#include "animate.h"
#include "resource.h"
#include "version.h"
#include "help.h"
#include <sys/signal.h>

#if defined ( _HAS_XPM )
#include <X11/xpm.h>
#include "xshiohara/xshiohara_color.xpm"
#endif
#include "xshiohara/xshiohara_bw.xbm"
#include "xshiohara/xshiohara_mask.xbm"

#if defined(__STDC__) || defined(__cplusplus)
# define P_(s) s
#else
# define P_(s) ()
#endif

/******************** External functions ********************/
extern ErrStatus animateShow P_((Widget,char *,int *, int *));
extern void initComm P_((Widget, Display *, Window));
extern Boolean initTrueColor P_(());
extern void PostDisplay P_(());
extern void initClock P_(());
extern void initBiff P_(());
extern void replaceBiffWindow P_(());
extern void replaceClockWindow P_(());
extern unsigned long  myrandom P_(());
extern void mysrandom P_((unsigned long));
extern void left_slide P_(());
extern void right_slide P_(());
extern void up_slide P_(());
extern void down_slide P_(());

/******************** External values ********************/
extern char setFilePath[];
extern char setFileName[];
extern char animMode[];
extern int  originalImageWidth, originalImageHeight;



/************************ Global values *************************/
char **file_listp;
int   *file_list_idxp;
int    file_list_count;
int    file_list_cur_count;

/************************ Local values *************************/
#if defined _EDIT
static Widget popWidget;
#endif
static Widget toplevel,topWidget;
static XtAppContext app_context;
static Atom wm_delete_window;
static Window fileWindow,sizeWindow;
static int filew,fileh,sizew,sizeh;
static Window messageWindow;
static Window specialWindow;
static Pixmap specialPixmap,specialShapePixmap;
static Cursor cursWatch, cursPirate, cursExchange, cursHeart, cursCoffee, cursGumby; 
static int valid_file_num=0,hasImage = False;

static char defAnim[] = "slide";
static char destinationPath[MAX_TEXT_LEN];

static char **anim_list;
static int   *anim_list_idxp;
static int    anim_list_count;
static int    anim_list_cur_count;

static char tmpBuf[MAX_TEXT_LEN];
static char targetFilePath[MAX_TEXT_LEN];

#if defined ( _HAS_XPM )
static XpmAttributes xpma;
static int xpmsuccess = False;
#endif


/******************** Type definitions ********************/
#define MYEVENT_TICKS 10000	/* Get the X Event every 10 msec */
#define MAX_WORK_PROCS 64
typedef struct {
	void (*proc)();
	dword interval;
	dword past;
	caddr_t client_data;
}WorkProc;

static dword pastTime;

typedef struct {
	dword numProc;
	WorkProc procs[MAX_WORK_PROCS];
} WorkProcs;
static WorkProcs work_procs;


/******************** functions ********************/

/*
* Get and proceed the X Event
*/
void myevent()
{
XtInputMask mask;
struct timeval tp;
struct timezone tzp;
dword now;
int i;


	/* Get now time */
	gettimeofday(&tp, &tzp);
	now = tp.tv_sec * 1000000 + tp.tv_usec;


	/* Check and exec the work procs */
	if(work_procs.numProc > 0){	

		for(i = 0; i < MAX_WORK_PROCS; i++){

			/* Check if it has a work proc */
			if(work_procs.procs[i].proc != NULL) {

				/* Check if it passed specified time */
				if((now - work_procs.procs[i].past) > work_procs.procs[i].interval){

					/* Exec the work proc */
					(void)(*work_procs.procs[i].proc)(work_procs.procs[i].client_data);
					work_procs.procs[i].past = now;

				}
			}
		}
	}


	/* Get the X Event every MYEVENT_TICKS */
	if(now - pastTime > MYEVENT_TICKS){
		pastTime = now;

		/* Check and proceed X event */
		while( (mask = XtAppPending(app_context)) ) {

			/*
			To use XtAppPending to determine whether an event is available
			on a particular application context, and then call XtAppProessEvent
			if there is an event to process.
			*/
			XtAppProcessEvent(app_context, mask);
		}
	}

}


/*
* make Cursor resources
*/
void makeCursors()
{
	if(app_data.root || app_data.override) return;
	cursWatch = XCreateFontCursor(theDisp, XC_watch);
	cursPirate = XCreateFontCursor(theDisp, XC_pirate);
	cursExchange = XCreateFontCursor(theDisp, XC_exchange);
	cursHeart = XCreateFontCursor(theDisp, XC_heart);
	cursCoffee = XCreateFontCursor(theDisp, XC_coffee_mug);
	cursGumby = XCreateFontCursor(theDisp, XC_gumby);
}

void makeNoCursor()
{
	XUndefineCursor(theDisp, theWindow);
}

/*
* make the WaitCursor
*/
void makeWaitCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursWatch);
}

/*
* make the RemoveCursor
*/
void makeRemoveCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursPirate);
}

/*
* make the MoveCursor
*/
void makeMoveCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursExchange);
}

/*
* make the PauseCursor
*/
void makePauseCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursHeart);
}

/*
* make the AppreciationCursor
*/
void makeAppreciationCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursCoffee);
}

/*
* make the GumbyCursor for reading files
*/
void makeGumbyCursor()
{
	if(app_data.root || app_data.override) return;
	XDefineCursor(theDisp, theWindow, cursGumby);
}


/*
* Check if it was paused
*/
void checkPause()
{

	if(IS_PAUSE_ON(click)){

		if(!app_data.root)
			makePauseCursor();

		while(1){
			myevent();		/* Get the X Event */

			if(IS_PAUSE_OFF(click)){

				if(!app_data.root)
					makeAppreciationCursor();

				break;
			}
		}
	}

}

/*
* micro seconds tick sleep
*/
#if defined(__STDC__) || defined(__cplusplus)
void myusleep(dword useconds)
#else
void myusleep(useconds)
dword useconds;
#endif
{
int i,j;

	myevent();			/* Get the X Event */

	i = useconds / MYEVENT_TICKS;
	j = useconds % MYEVENT_TICKS;

	if(i == 0 && j == 0)
		return;

	while(i--){
		checkPause();
		usleep(MYEVENT_TICKS);
		myevent();		/* Get the X Event */
	}
	checkPause();
	usleep(j);

	myevent();			/* Get the X Event */
}


/*
* wait
*/
ActionStatus mywait()
{
dword seconds=0;

	if(app_data.intervalTime == 0) {

		/* Do not return until accept some ACTION */
		while(True) {
			myusleep(1000000);  /* sleep 1000 msec */
			if(IS_SLEEP_OFF(click))
				return(click);
		}

	}
	else {
		while(seconds < app_data.intervalTime) {
			myusleep(1000000);  /* sleep 1000 msec */
			if(IS_SLEEP_OFF(click))
				return(click);
			else
				seconds++;
		}
	}

	return(click);
}


/*
* no wait
*/
ActionStatus mynowait()
{
	myusleep(0); /* Only get the X Event in myusleep() */
	return(click);
}


/*
* Remove specified work proc
*/
#if defined(__STDC__) || defined(__cplusplus)
void XSlideshowRemoveWorkProc( int id )
#else
void XSlideshowRemoveWorkProc( id )
int id;
#endif
{
	if(id != -1 && id < MAX_WORK_PROCS){
		work_procs.procs[id].proc = NULL;
		if(work_procs.numProc > 0)
			work_procs.numProc--;
	}
}


/*
* Add new work proc
*/
#if defined(__STDC__) || defined(__cplusplus)
int XSlideshowAddWorkProc( void (*func)() , dword interval, caddr_t client_data)
#else
int XSlideshowAddWorkProc( func, interval, client_data)
void (*func)();
dword interval;
caddr_t client_data;
#endif
{
struct timeval tp;
struct timezone tzp;
dword now;
int i;

	/* Get now time */
	gettimeofday(&tp, &tzp);
	now = tp.tv_sec * 1000000 + tp.tv_usec;

	/* Search a free entry */
	for(i = 0; i < MAX_WORK_PROCS; i++){
		if(work_procs.procs[i].proc == NULL)
			break;
	}

	if(i == MAX_WORK_PROCS) {
		fprintf(stderr,"xslideshow: Error: no more entry for new work proc\n");
		return(-1); /* No free entry */
	}
	else {
		work_procs.procs[i].past = now;
		work_procs.procs[i].proc = func;
		work_procs.procs[i].interval = interval;
		work_procs.procs[i].client_data = client_data;
		work_procs.numProc++;
		return(i);
	}
}


/*********************************************************************
  Thank you for giving these action idea
  Mr Peter Grina <grina@fai.com> and Dr Eberhard W Lisse <el@lisse.na>
*********************************************************************/

void action_dummy()
{
	;
}
void action_next_frame()
{
	ACT_SLEEP_OFF(click);
	ACT_PAUSE_OFF(click);
	ACT_PREV_OFF(click);
	ACT_NEXT_ON(click);
	if(app_data.verbose && app_data.debug)
		fprintf(stderr,"xslideshow: ACTION_NEXT_FRAME\n");
	if(app_data.forceChange)
		gotSomeAction = True;
}
void action_prev_frame()
{
	ACT_SLEEP_OFF(click);
	ACT_PAUSE_OFF(click);
	ACT_NEXT_OFF(click);
	ACT_PREV_ON(click);
	if(app_data.verbose && app_data.debug)
		fprintf(stderr,"xslideshow: ACTION_PREV_FRAME\n");
	if(app_data.forceChange)
		gotSomeAction = True;
}
void action_head_frame()
{
	ACT_SLEEP_OFF(click);
	ACT_PAUSE_OFF(click);
	ACT_TAIL_OFF(click);
	ACT_HEAD_ON(click);
	if(app_data.verbose && app_data.debug)
		fprintf(stderr,"xslideshow: ACTION_HEAD_FRAME\n");
	if(app_data.forceChange)
		gotSomeAction = True;
}
void action_tail_frame()
{
	ACT_SLEEP_OFF(click);
	ACT_PAUSE_OFF(click);
	ACT_HEAD_OFF(click);
	ACT_TAIL_ON(click);
	if(app_data.verbose && app_data.debug)
		fprintf(stderr,"xslideshow: ACTION_TAIL_FRAME\n");
	if(app_data.forceChange)
		gotSomeAction = True;
}
void action_pause_frame()
{
	if(pauseToggle == 0){
		pauseToggle = 1;
		ACT_PAUSE_ON(click);
		if(app_data.verbose && app_data.debug)
			fprintf(stderr,"xslideshow: ACTION_PAUSE_FRAME on\n");
	}
	else {
		pauseToggle = 0;
		ACT_PAUSE_OFF(click);
		if(app_data.verbose && app_data.debug)
			fprintf(stderr,"xslideshow: ACTION_PAUSE_FRAME off\n");
	}
}
void action_change_frame()
{
	ACT_SLEEP_OFF(click);
	ACT_PAUSE_OFF(click);
	if(app_data.forceChange)
		gotSomeAction = True;
}
void action_left()
{
	if(!strcmp(app_data.animate , "slide"))
		left_slide();
}
void action_right()
{
	if(!strcmp(app_data.animate , "slide"))
		right_slide();
}
void action_up()
{
	if(!strcmp(app_data.animate , "slide"))
		up_slide();
}
void action_down()
{
	if(!strcmp(app_data.animate , "slide"))
		down_slide();
}
void action_resize()
{
	if(!strcmp(app_data.animate , "slide")){
		if(!app_data.resize && !gotSomeAction && hasImage){
			prevWindowWidth = windowWidth;
			prevWindowHeight = windowHeight;
			ACT_SLEEP_OFF(click);
			ACT_PAUSE_OFF(click);
			app_data.resize = True;
			app_data.fit = False;
			gotSomeAction = True;

			/* Stay on the same picture */
			if(IS_PREV_ON(click))
				file_list_cur_count++;
			else
				file_list_cur_count--;
		}
	}
}
void action_fit()
{
Arg args[5];
int ac;

	if(!strcmp(app_data.animate , "slide")){
		if(!app_data.fit && !gotSomeAction && hasImage){
			ACT_SLEEP_OFF(click);
			ACT_PAUSE_OFF(click);
			app_data.fit = True;
			app_data.resize = False;
			gotSomeAction = True;

			/* Stay on the same picture */
			if(IS_PREV_ON(click))
				file_list_cur_count++;
			else
				file_list_cur_count--;

			newWindowWidth = prevWindowWidth;
			newWindowHeight = prevWindowHeight;

			/* Set and change window size */
			ac = 0;
			XtSetArg(args[ac],XtNwidth,newWindowWidth);ac++;
			XtSetArg(args[ac],XtNheight,newWindowHeight);ac++;
			XtSetValues(topWidget,args,ac);

			replaceBiffWindow();
			replaceClockWindow();
		}
	}
}


#if defined _EDIT
#if defined(__STDC__) || defined(__cplusplus)
static void cancel(Widget widget, caddr_t client_data, caddr_t call_data)
#else
static void cancel( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
	XtPopdown(popWidget);
	XtDestroyWidget(popWidget);
}

void moveTheFile()
{
char cmd[MAX_TEXT_LEN];

	if(file_list_count > 0){
		sprintf(cmd,"mv %s %s",targetFilePath,destinationPath);
		if(app_data.verbose)
			fprintf(stderr,"xslideshow: %s\n", cmd);
		if((system(cmd)) != 127){
			if(valid_file_num > 0) valid_file_num--;
			hasImage = False;
			if(!app_data.root)
				makeMoveCursor();
			XtFree((char *)file_listp[ file_list_idxp[ file_list_cur_count ] ]);
			file_listp[ file_list_idxp[ file_list_cur_count ] ] = NULL;
		}
		else {
			if(app_data.verbose)
				fprintf(stderr,"xslideshow: Can't exec %s\n", cmd);
		}
		ACT_SLEEP_OFF(click);
		ACT_PAUSE_OFF(click);
		if(app_data.forceChange)
			gotSomeAction = True;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
static void moveOk(Widget widget, caddr_t client_data, caddr_t call_data)
#else
static void moveOk( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
String string;

	string = XawDialogGetValueString((Widget)client_data);
	strcpy(destinationPath,string);
	XtPopdown(popWidget);
	XtDestroyWidget(popWidget);
	moveTheFile();
}


void action_move()
{
Widget dw,tmpw;
Dimension width, height;
Position x, y;
Arg args[10];
int ac;

	if(strcmp(app_data.animate , "slide"))
		return;

	if(app_data.confirm || destinationPath[0] == 0){
		ac = 0;
		XtSetArg(args[ac], XtNallowShellResize, (XtArgVal)True); ac++;
		popWidget = XtCreatePopupShell("popup",transientShellWidgetClass,toplevel,args,ac);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"Destination path"); ac++;
		XtSetArg(args[ac], XtNvalue, (XtArgVal)destinationPath); ac++;
		dw = XtCreateManagedWidget(NULL, dialogWidgetClass, popWidget, args, ac);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"  Ok  "); ac++;
		tmpw = XtCreateManagedWidget(NULL, commandWidgetClass, dw, args, ac);
		XtAddCallback(tmpw, XtNcallback, (XtPointer)moveOk, (caddr_t)dw);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"Cancel"); ac++;
		tmpw = XtCreateManagedWidget(NULL, commandWidgetClass, dw, args, ac);
		XtAddCallback(tmpw, XtNcallback, (XtPointer)cancel, (caddr_t)dw);

		XtRealizeWidget(popWidget);
		while(!(XtIsRealized(popWidget)));

		ac = 0;
		XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
		XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
		XtGetValues(toplevel, args, ac);

		XtTranslateCoords(toplevel, width/2-150, height/2-50, &x, &y);

		ac = 0;
		XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
		XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
		XtSetArg(args[ac], XtNwidth, (XtArgVal)300); ac++;
		XtSetArg(args[ac], XtNheight, (XtArgVal)100); ac++;
		XtSetValues(popWidget, args, ac);

		XtPopup(popWidget, XtGrabNonexclusive);
	}
	else{
		if(destinationPath[0] != 0)
			moveTheFile();
	}
}

/*
* remove the file
*/
void removeTheFile()
{
	if(file_list_count > 0){
		if((unlink(targetFilePath)) == 0){
			if(valid_file_num > 0) valid_file_num--;
			hasImage = False;
			if(!app_data.root)
				makeRemoveCursor();
			if(app_data.verbose)
				fprintf(stderr,"xslideshow: Remove %s\n", targetFilePath);
			XtFree((char *)file_listp[ file_list_idxp[ file_list_cur_count ] ]);
			file_listp[ file_list_idxp[ file_list_cur_count ] ] = NULL;
		}
		else {
			if(app_data.verbose)
				fprintf(stderr,"xslideshow: Can't remove %s\n", targetFilePath);
		}
		ACT_SLEEP_OFF(click);
		ACT_PAUSE_OFF(click);
		if(app_data.forceChange)
			gotSomeAction = True;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
static void removeOk(Widget widget, caddr_t client_data, caddr_t call_data)
#else
static void removeOk( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
	XtPopdown(popWidget);
	XtDestroyWidget(popWidget);
	removeTheFile();
}

void action_remove()
{
Widget dw,tmpw;
Dimension width, height;
Position x, y;
Arg args[10];
int ac;

	if(strcmp(app_data.animate , "slide"))
		return;

	if(targetFilePath[0] == 0)
		return;

	if(app_data.confirm){
		ac = 0;
		XtSetArg(args[ac], XtNallowShellResize, (XtArgVal)True); ac++;
		popWidget = XtCreatePopupShell("popup",transientShellWidgetClass,toplevel,args,ac);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"Remove the file"); ac++;
		XtSetArg(args[ac], XtNvalue, (XtArgVal)targetFilePath); ac++;
		dw = XtCreateManagedWidget(NULL, dialogWidgetClass, popWidget, args, ac);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"  Ok  "); ac++;
		tmpw = XtCreateManagedWidget(NULL, commandWidgetClass, dw, args, ac);
		XtAddCallback(tmpw, XtNcallback, (XtPointer)removeOk, (caddr_t)dw);

		ac = 0;
		XtSetArg(args[ac], XtNlabel, (XtArgVal)"Cancel"); ac++;
		tmpw = XtCreateManagedWidget(NULL, commandWidgetClass, dw, args, ac);
		XtAddCallback(tmpw, XtNcallback, (XtPointer)cancel, (caddr_t)dw);

		XtRealizeWidget(popWidget);
		while(!(XtIsRealized(popWidget)));

		ac = 0;
		XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
		XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
		XtGetValues(toplevel, args, ac);

		XtTranslateCoords(toplevel, width/2-150, height/2-50, &x, &y);

		ac = 0;
		XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
		XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
		XtSetArg(args[ac], XtNwidth, (XtArgVal)300); ac++;
		XtSetArg(args[ac], XtNheight, (XtArgVal)100); ac++;
		XtSetValues(popWidget, args, ac);

		XtPopup(popWidget, XtGrabNonexclusive);
	}
	else{
		removeTheFile();
	}
}
#endif /* _EDIT */


/*
* Error handlers
*/
void goodbyekiss()
{
	signal(SIGALRM,SIG_IGN);

#if defined ( _HAS_XPM )
	if( xpmsuccess == True ) {
		(void) XpmFreeAttributes( &xpma );
		xpmsuccess = False;
	}
#endif

	if(theDisp && theCmap)
		PostDisplay();

	XtCloseDisplay(theDisp);

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: goodbye kiss!\n");

	exit(0);
}


/*
* Exit from the window manager
*/
#if defined(__STDC__) || defined(__cplusplus)
void wm_quit(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
void wm_quit(w,event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
	signal(SIGALRM,SIG_IGN);

	if (event->type == ClientMessage &&
		(Atom)event->xclient.data.l[0] != wm_delete_window) {
		XBell (XtDisplay(w), 0);
		return;
	}

#if defined ( _HAS_XPM )
	if( xpmsuccess == True ) {
		(void) XpmFreeAttributes( &xpma );
		xpmsuccess = False;
	}
#endif

	if(theDisp && theCmap)
		PostDisplay();

	XCloseDisplay(theDisp);

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: wm quit!\n");

	exit (0);
}


/*
* Signal handlers
*/
void trace_trap()
{
	signal(SIGALRM,SIG_IGN);
	fprintf(stderr,"xslideshow : Caught SIGTRAP signal. Exit.\n");
	exit(1);
}
void segmentation_fault()
{
	signal(SIGALRM,SIG_IGN);
	fprintf(stderr,"xslideshow : Caught SIGSEGV signal. Exit.\n");
	exit(1);
}
void bus_error()
{
	signal(SIGALRM,SIG_IGN);
	fprintf(stderr,"xslideshow : Caught SIGBUS signal. Exit.\n");
	exit(1);
}
void terminate()
{
	signal(SIGALRM,SIG_IGN);
#if False
	fprintf(stderr,"xslideshow : Caught SIGTERM signal. Exit.\n");
#endif
	exit(1);
}



/*
* Move the window on the Bresenham line
*/
#if defined(__STDC__) || defined(__cplusplus)
static int sign(int x)
#else
static int sign(x)
int x;
#endif
{
	if(x == 0)     return(0);
	else if(x > 0) return(1);
	else		   return(-1);
}

#if defined(__STDC__) || defined(__cplusplus)
void move_on_bresenham_line(Window w,int x1,int y1,int x2,int y2, int step, int ticks)
#else
void move_on_bresenham_line(w,x1,y1,x2,y2,step,ticks)
Window w;
int x1,y1,x2,y2,step,ticks;
#endif
{
int  i,j;
int  x,y,dx,dy,sx,sy;
int  prex,prey,incli;
int  e,cons1,cons2;

	prex = x1;
	prey = y1;

	dx = x2 - x1;
	dy = y2 - y1;
	sx = sign(dx);
	sy = sign(dy);
	dx = (dx < 0) ? -dx : dx; 
	dy = (dy < 0) ? -dy : dy; 

	if ( dy > dx ){
		incli = 1;
		cons2 = dx << 1;
		cons1 = cons2 - (dy << 1);
		e = (dx << 1) - dy;
		x = x1;
		y = y1 + sy;
		j = dy;
	}
	else{
		incli = 0;
		cons2 = dy << 1;
		cons1 = cons2 - (dx << 1);
		e = (dy << 1) - dx;
		y = y1;
		x = x1 + sx;
		j = dx;
	}

	for(i = 1; i <= j; i++){
		if(e > 0){
			if ( incli == 1 )
				x += sx;
			else
				y += sy;
			e += cons1;
		}
		else{
			e += cons2;
		}

		if(ABS(x,prex) > step || ABS(y,prey) > step){
			XMoveWindow(theDisp, w, x, y);
			myusleep(ticks);	/* Get the X Event */

			prex = x;
			prey = y;
		}

		if(incli == 1)
			y += sy;
		else x += sx;
	}

	XMoveWindow(theDisp, w, x2, y2);
}


/*
* Special window handling
*/
void removeSpecialWindow()
{
	if(specialWindow != (Window)0){
		XDestroyWindow(theDisp, specialWindow);
		specialWindow = (Window)0;

#if defined ( _HAS_XPM )
		if( xpmsuccess == True ) {
			(void) XpmFreeAttributes( &xpma );
			xpmsuccess = False;
		}
#endif

	}
}


#if defined(__STDC__) || defined(__cplusplus)
void createSpecialWindow(int x, int y)
#else
void createSpecialWindow(x,y)
int x,y;
#endif
{
XSetWindowAttributes xswa;
#if defined ( _HAS_XPM )
int errf;
#endif

	/* The special window has been created */
	if(specialWindow != (Window)0)
		return;

	/* Create a New (Only one !) special window */
	specialWindow = XCreateSimpleWindow(	theDisp,
											theWindow,
											x,
											y,
											xshiohara_mask_width,
											xshiohara_mask_height,
											0,
											dbcol,
											dbcol	);

#if defined ( _HAS_XPM )
	/*
	This part is a hacked version of Stefan Gustavson's xteddy program
		Copyright Stefan Gustavson, ISY-LiTH, 1994.
		  Email address: stefang@isy.liu.se
	*/

	/* Create the pixmap from the xpm file */
	xpmsuccess     = False;
	xpma.visual    = theVisual;
	xpma.colormap  = defCmap;
	xpma.depth     = DefaultDepth(theDisp,theScreenNumber);
	xpma.valuemask = XpmVisual | XpmColormap | XpmDepth;
	errf = XpmCreatePixmapFromData(	theDisp,
									specialWindow,
									xshiohara_color,
									&specialPixmap,
									&specialShapePixmap,
									&xpma	);

	if(errf != XpmSuccess) {
		specialPixmap = XCreatePixmapFromBitmapData(	theDisp,
														specialWindow,
														xshiohara_bw_bits,
														xshiohara_bw_width,
														xshiohara_bw_height,
														dbcol,
														dwcol,
														xpma.depth	);
	}
	else {
		xpmsuccess = True;
	}

#else

	/* Create the pixmap from the bmp file */
	specialPixmap = XCreatePixmapFromBitmapData(theDisp,
												specialWindow,
												xshiohara_bw_bits,
												xshiohara_bw_width,
												xshiohara_bw_height,
												dbcol,
												dwcol,
												DefaultDepth(theDisp, theScreenNumber)	);
#endif

	/* Create the shape pixmap */
	xswa.override_redirect = True;
	xswa.background_pixmap = specialPixmap;
	XChangeWindowAttributes(theDisp,
							specialWindow,
							CWBackPixmap | CWOverrideRedirect,
							&xswa);

	specialShapePixmap = XCreateBitmapFromData(	theDisp,
												specialWindow,
												xshiohara_mask_bits,
												xshiohara_mask_width,
												xshiohara_mask_height	);

	XShapeCombineMask(	theDisp,
						specialWindow,
						ShapeBounding,
						0,
						0, 
						specialShapePixmap, 
						ShapeSet	);

	XClearWindow(theDisp, specialWindow);
	XLowerWindow(theDisp, specialWindow);
	XMapWindow(theDisp, specialWindow);
}


/*
* Image window handling
*/
#if defined(__STDC__) || defined(__cplusplus)
void removeImageWindow(Window *win)
#else
void removeImageWindow(win)
Window *win;
#endif
{
	if(*win != (Window)0){
		XDestroyWindow(theDisp, *win);
		*win = (Window)0;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
void ShowImageWindow(Window *win,int x,int y,int w,int h)
#else
void ShowImageWindow(win,x,y,w,h)
Window *win;
int x,y,w,h;
#endif
{
Pixmap pic;
XSetWindowAttributes xswa;
XGCValues gcv;
GC gc;

	/* Create a simple window */
	*win = XCreateSimpleWindow(	theDisp,
								theWindow,
								x,
								y,
								w,
								h,
								0,
								bcol,
								bcol	);
	xswa.override_redirect = True;
	XChangeWindowAttributes(	theDisp,
								*win,
								CWOverrideRedirect,
								&xswa	);

	/* Create a pixmap */
	pic = XCreatePixmap(	theDisp,
							*win,
							w,
							h,
							DefaultDepth(theDisp, theScreenNumber)	);

	gcv.function = GXcopy;
	gcv.foreground = dbcol;	/* use the default black pixel */
	gc = XCreateGC(	theDisp,
					pic,
					GCForeground | GCFunction,
					&gcv	);

	XFillRectangle(	theDisp,
					pic,
					gc,
					0,
					0,
					w,
					h	);

	XFreeGC(theDisp,gc);

	/* Put the image into the pixmap */
	gcv.function = GXcopy;
	gc = XCreateGC(	theDisp,
					pic,
					GCFunction,
					&gcv	);
	XPutImage(	theDisp,
				pic,
				gc,
				theImage,
				0, 0,
				0, 0,
				w, h	);

	/* Set the pixmap as the window background */
	XSetWindowBackgroundPixmap(theDisp, *win, pic);
	XClearWindow(theDisp, *win);

	XFreePixmap(theDisp, pic);
	XFreeGC(theDisp, gc);

	/* Hide the window */
	XLowerWindow(theDisp, *win);
	XMapWindow(theDisp, *win);
}

#if defined(__STDC__) || defined(__cplusplus)
void ShowImage(int x,int y,int w,int h)
#else
void ShowImage(x,y,w,h)
int x,y,w,h;
#endif
{
XGCValues   gcv;
GC	  gc;

	gcv.function = GXcopy;
	gc = XCreateGC(theDisp, theWindow, GCFunction, &gcv);
	/* Draw the image on the window */
	XPutImage(	theDisp,
				theWindow,
				gc,
				theImage,
				0, 0,
				x, y,
				w, h	);
	XFreeGC(theDisp, gc);
}


/*
* File name window handling
*/
void raiseFileNameWindow()
{
	if(fileWindow != (Window)0){
		XMapWindow(theDisp, fileWindow);

		/* move the file name window on the center and bottom of the window */
		move_on_bresenham_line( fileWindow,
						(windowWidth - filew) / 2,
						windowHeight + fileh,
						(windowWidth - filew) / 2,
						windowHeight - fileh,
						1,
						app_data.moveOnTheLineTicks	);
	}


	if(sizeWindow != (Window)0){
		/* Raise the Size Window */
		XMapWindow(theDisp, sizeWindow);

		/* move the file name window on the center and bottom of the window */
		move_on_bresenham_line( sizeWindow,
						(windowWidth - sizew) / 2,
						-fileh,
						(windowWidth - sizew) / 2,
						0,
						1,
						app_data.moveOnTheLineTicks	);
	}
}

void removeFileNameWindow()
{
	if(fileWindow != (Window)0){
		move_on_bresenham_line(
						fileWindow,
						(windowWidth - filew) / 2,
						windowHeight - fileh,
						(windowWidth - filew) / 2,
						windowHeight + fileh,
						1,
						app_data.moveOnTheLineTicks	);

		XDestroyWindow(theDisp, fileWindow);
		fileWindow = (Window)0;
	}

	if(sizeWindow != (Window)0){
		move_on_bresenham_line(
						sizeWindow,
						(windowWidth - sizew) / 2,
						0,
						(windowWidth - sizew) / 2,
						-fileh,
						1,
						app_data.moveOnTheLineTicks	);

		XDestroyWindow(theDisp, sizeWindow);
		sizeWindow = (Window)0;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
void createFileNameWindow(char *fname)
#else
void createFileNameWindow(fname)
char *fname;
#endif
{
Pixmap pic;
int d_h,f_a,f_d;
XCharStruct overall;
XSetWindowAttributes xswa;
XGCValues gcv;
GC gc;
char c,flen,i;
char fileName[MAX_TEXT_LEN];
char imageSize[MAX_TEXT_LEN];

	if(fileWindow != (Window)0){
		removeFileNameWindow();
	}

	/* Create the file name without its path name */
	if(!app_data.showFilePath){
		flen = strlen(fname);

		/* Some UNIX-OS does not have strspn() */
		for(c = 0,i = 0; i < flen; i++) {
			if(fname[i] == '/')
				c = i;
		}
		if(c != 0 && c < flen)
			strcpy(fileName, &fname[c + 1]);
		else
			strcpy(fileName, fname);
	}
	else{
		strcpy(fileName, fname);
	}

	/* Get text and font information */
	XTextExtents(	app_data.font,
					fileName,
					strlen(fileName),
					&d_h,
					&f_a,
					&f_d,
					&overall);
	/* Calc file name window size */
	filew = (overall.lbearing < 0 ? -overall.lbearing : overall.lbearing) + overall.rbearing;
	fileh = overall.descent + overall.ascent;

	/* Create a file name window */
	fileWindow = XCreateSimpleWindow(	theDisp,
										theWindow,
										(windowWidth - filew) / 2,
										windowHeight + fileh,
										filew,
										fileh,
										0,
										dbcol,
										dbcol	);

	xswa.override_redirect = True;
	XChangeWindowAttributes(theDisp, fileWindow, CWOverrideRedirect, &xswa);

	pic = XCreatePixmap(	theDisp,
							fileWindow,
							filew,
							fileh,
							DefaultDepth(theDisp,theScreenNumber)	);

	gcv.function = GXcopy;
	gcv.foreground = dbcol;	/* default black pixel */
	gc = XCreateGC(theDisp, pic, GCForeground | GCFunction, &gcv);
	XFillRectangle(theDisp, pic, gc, 0, 0, filew, fileh);
	XFreeGC(theDisp, gc);

	gcv.font = app_data.font->fid;
	if(dbcol == 1 && dwcol == 0)
		gcv.function = GXand;
	else if(dbcol == 0 && dwcol == 1)
		gcv.function = GXor;
	else
		gcv.function = GXcopy;
	gcv.foreground = dwcol;
	gcv.background = dbcol;
	gc = XCreateGC(theDisp, pic, GCFont | GCFunction | GCForeground | GCBackground, &gcv);

	/* Draw the file name string */
	XDrawImageString(	theDisp,
						pic,
						gc,
						(overall.lbearing < 0 ? -overall.lbearing : overall.lbearing),
						overall.ascent,
						fileName,
						strlen(fileName)	);

	/* Set the picsmap as the window background */
	XSetWindowBackgroundPixmap(theDisp, fileWindow, pic);
	XClearWindow(theDisp, fileWindow);

	XFreePixmap(theDisp, pic);
	XFreeGC(theDisp, gc);

	XLowerWindow(theDisp, fileWindow);


	if(!app_data.clock){

		sprintf(imageSize,"%dx%d",originalImageWidth,originalImageHeight);

		/* Get text and font information */
		XTextExtents(	app_data.font,
					imageSize,
					strlen(imageSize),
					&d_h,
					&f_a,
					&f_d,
					&overall);
		/* Calc file name window size */
		sizew = (overall.lbearing < 0 ? -overall.lbearing : overall.lbearing) + overall.rbearing;
		sizeh = overall.descent + overall.ascent;

		/* Create a size window */
		sizeWindow = XCreateSimpleWindow(	theDisp,
										theWindow,
										(windowWidth - sizew) / 2,
										windowHeight + sizeh,
										sizew,
										sizeh,
										0,
										dbcol,
										dbcol	);

		xswa.override_redirect = True;
		XChangeWindowAttributes(theDisp, sizeWindow, CWOverrideRedirect, &xswa);

		pic = XCreatePixmap(	theDisp,
							sizeWindow,
							sizew,
							sizeh,
							DefaultDepth(theDisp,theScreenNumber)	);

		gcv.function = GXcopy;
		gcv.foreground = dbcol;	/* default black pixel */
		gc = XCreateGC(theDisp, pic, GCForeground | GCFunction, &gcv);
		XFillRectangle(theDisp, pic, gc, 0, 0, sizew, sizeh);
		XFreeGC(theDisp, gc);

		gcv.font = app_data.font->fid;
		if(dbcol == 1 && dwcol == 0)
			gcv.function = GXand;
		else if(dbcol == 0 && dwcol == 1)
			gcv.function = GXor;
		else
			gcv.function = GXcopy;
		gcv.foreground = dwcol;
		gcv.background = dbcol;
		gc = XCreateGC(theDisp, pic, GCFont | GCFunction | GCForeground | GCBackground, &gcv);

		/* Draw the size tring */
		XDrawImageString(	theDisp,
						pic,
						gc,
						(overall.lbearing < 0 ? -overall.lbearing : overall.lbearing),
						overall.ascent,
						imageSize,
						strlen(imageSize)	);

		/* Set the picsmap as the window background */
		XSetWindowBackgroundPixmap(theDisp, sizeWindow, pic);
		XClearWindow(theDisp, sizeWindow);

		XFreePixmap(theDisp, pic);
		XFreeGC(theDisp, gc);

		XLowerWindow(theDisp, sizeWindow);
	}
}

/*
* Message window handling
*/
void removeMessageWindow()
{
	if(messageWindow != (Window)0){
		XDestroyWindow(theDisp, messageWindow);
		messageWindow = (Window)0;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
int showErrorMessages(int numlines, char *message1, char *message2)
#else
int showErrorMessages(numlines, message1, message2)
int numlines;
char *message1,*message2;
#endif
{
Pixmap pic;
int d_h,f_a,f_d,textwidth,textheight;
XCharStruct overall;
int tw1,tw2,th1,th2,as1,as2;
XSetWindowAttributes xswa;
XGCValues   gcv;
GC	  gc;

	/* remove previous message window */
	if(messageWindow != (Window)0){
		removeMessageWindow();
	}

	/* show message1 string */
	if(numlines == 1 && message1 != NULL){

		/* Get text and font information */
		XTextExtents(	app_data.font,
						message1,
						strlen(message1),
						&d_h,
						&f_a,
						&f_d,
						&overall);

		textwidth = (overall.lbearing < 0 ? -overall.lbearing : overall.lbearing) + overall.rbearing;
		textheight = overall.descent + overall.ascent;

		/* Create the message1 window */
		messageWindow = XCreateSimpleWindow(	theDisp,
												theWindow,
												(windowWidth - textwidth) / 2,
												(windowHeight - textheight) / 2,
												textwidth,
												textheight,
												0,
												dbcol, dbcol	);
		xswa.override_redirect = True;
		XChangeWindowAttributes(theDisp, messageWindow, CWOverrideRedirect, &xswa);

		pic = XCreatePixmap(	theDisp,
								messageWindow,
								textwidth,
								textheight,
								DefaultDepth(theDisp,theScreenNumber)	);

		gcv.function = GXcopy;
		gcv.foreground = dbcol;
		gc = XCreateGC(theDisp, pic, GCForeground | GCFunction, &gcv);
		XFillRectangle(theDisp, pic, gc, 0, 0, textwidth, textheight);
		XFreeGC(theDisp, gc);

		/* Draw the message1 string */
		XDrawImageString(	theDisp,
							pic,
							fontGC,
							(overall.lbearing < 0 ? -overall.lbearing : overall.lbearing),
							overall.ascent,
							message1,
							strlen(message1));

	}

	/* show both message1 and message2 string */
	else if(message1 != NULL && message2 != NULL){

		/* Get text and font information for message1 */
		XTextExtents(	app_data.font,
						message1,
						strlen(message1),
						&d_h,
						&f_a,
						&f_d,
						&overall	);
		tw1 = (overall.lbearing < 0 ? -overall.lbearing : overall.lbearing) + overall.rbearing;
		th1 = overall.descent + overall.ascent;
		as1 = overall.ascent;

		/* Get text and font information for message2 */
		XTextExtents(	app_data.font,
						message2,
						strlen(message2),
						&d_h,
						&f_a,
						&f_d,
						&overall	);
		tw2 = (overall.lbearing < 0 ? -overall.lbearing : overall.lbearing) + overall.rbearing;
		th2 = overall.descent + overall.ascent;
		as2 = overall.ascent;

		textwidth = (tw1 > tw2) ? tw1 : tw2;
		textheight = th1 + th2 + (th1 +th2 ) / 4;

		/* Create a message window */
		messageWindow = XCreateSimpleWindow(	theDisp,
												theWindow,
												(windowWidth - textwidth) / 2,
												(windowHeight - textheight) / 2,
												textwidth,
												textheight,
												0,
												dbcol, dbcol	);

		xswa.override_redirect = True;
		XChangeWindowAttributes(theDisp, messageWindow, CWOverrideRedirect, &xswa);

		pic = XCreatePixmap(	theDisp,
								messageWindow,
								textwidth,
								textheight,
								DefaultDepth(theDisp,theScreenNumber)	);

		gcv.function = GXcopy;
		gcv.foreground = dbcol;
		gc = XCreateGC(theDisp, pic, GCForeground | GCFunction,&gcv);
		XFillRectangle(theDisp, pic, gc, 0, 0, textwidth, textheight);
		XFreeGC(theDisp, gc);

		/* Draw the message1 string on the line 1 */
		XDrawImageString(	theDisp,
							pic,
							fontGC,
							(tw1 > tw2) ? 0 : (tw2 - tw1) / 2,
							as1,
							message1,
							strlen(message1)	);

		/* Draw the message2 string on the line 2 */
		XDrawImageString(	theDisp,
							pic,
							fontGC,
							(tw2 > tw1) ? 0 : (tw1 - tw2) / 2,
							textheight - overall.descent,
							message2,
							strlen(message2)	);
	}

	XSetWindowBackgroundPixmap(theDisp, messageWindow, pic);
	XClearWindow(theDisp, messageWindow);

	XFreePixmap(theDisp, pic);
	XMapWindow(theDisp, messageWindow);

	return(textheight);
}

/*
* Check the file type
*
* How to check JPEG format.
* You can identify a JFIF file by looking for the following sequence:
* X'FF', SOI, X'FF', APP0, <2 bytes to be skipped>, "JFIF", X'00'.
*
*/
#if defined(__STDC__) || defined(__cplusplus)
ErrStatus checkFileType(char *fname)
#else
ErrStatus checkFileType(fname)
char *fname;
#endif
{
int i, offs;
int tmp[12];
char tmpc[12];
struct stat fstatus;

	fileType = IS_NONE;

	/* Some UNIX-OS can open the file even if it has NULL name */
	if(fname[0] == 0)
		return(ERR_NO_ERR);

	if((stat(fname,&fstatus))== -1)
		return(ERR_NO_ERR);

	if( (fstatus.st_mode & S_IFDIR) == S_IFDIR )
		return(ERR_NO_ERR);

	if((fp = fopen(fname,"r")) == (FILE *)NULL)
		return(ERR_NO_ERR);

	for(i = 0;i < 12;i++){
		tmp[i] = getc(fp);
		if(feof(fp)) return(ERR_NO_ERR);
		tmpc[i] = (char)tmp[i];
	}

	if(strncmp(tmpc,"GIF",3) == 0) {
		fileType = IS_GIF;
	}
	else if(   (tmp[ 0] == 0xff)
			&& (tmp[ 1] == 0xd8) /* SOI (Start Of Image) */
			&& (tmp[ 2] == 0xff)
			/* APP0 */
			&& ((tmp[ 3] == 0xe0) || (tmp[ 3] == 0xe1))
				&& (
	   ((tmp[6]&0x5f)=='J' && (tmp[7]&0x5f)=='F' && (tmp[8]&0x5f)=='I' && (tmp[9]&0x5f)=='F')
	|| ((tmp[6]&0x5f)=='J' && (tmp[7]&0x5f)=='F' && (tmp[8]&0x5f)=='X' && (tmp[9]&0x5f)=='X')
	|| ((tmp[6]&0x5f)=='E' && (tmp[7]&0x5f)=='X' && (tmp[8]&0x5f)=='I' && (tmp[9]&0x5f)=='F')
				)
				&& (tmp[10] == 0x00)
								){ 
		fileType = IS_JPG;
	}
	else {
		rewind(fp);
		while(!feof(fp)){

			if((tmp[0] = getc(fp)) == 0xff){
				if(feof(fp)) break;
				offs = ftell(fp);
				if(offs > 1024) break;

				for(i=1; i<11; i++){
					tmp[i] = getc(fp);
					if(feof(fp)) break;
				}
				if(feof(fp)) break;

				if(	   (tmp[ 0] == 0xff)
					&& (tmp[ 1] == 0xd8) /* SOI (Start Of Image) */
					&& (tmp[ 2] == 0xff)
					/* APP0 */
					&& ((tmp[ 3] == 0xe0) || (tmp[ 3] == 0xe1))
					&& (
		   ((tmp[6]&0x5f)=='J' && (tmp[7]&0x5f)=='F' && (tmp[8]&0x5f)=='I' && (tmp[9]&0x5f)=='F')
		|| ((tmp[6]&0x5f)=='J' && (tmp[7]&0x5f)=='F' && (tmp[8]&0x5f)=='X' && (tmp[9]&0x5f)=='X')
		|| ((tmp[6]&0x5f)=='E' && (tmp[7]&0x5f)=='X' && (tmp[8]&0x5f)=='I' && (tmp[9]&0x5f)=='F')
					)
					&& (tmp[10] == 0x00)
										){
					fileType = IS_JPG;
					break;
				}
				else {
					(void)fseek(fp,offs+1,SEEK_SET);
				}
			}

		}
	}

	fclose(fp);
	return(ERR_NO_ERR);
}

/*
* Resize notify handler
*/
#if defined(__STDC__) || defined(__cplusplus)
void ResizeWindow(Widget w, XtPointer data, XEvent *event)
#else
void ResizeWindow(w, data, event)
Widget w;
XtPointer data;
XEvent *event;
#endif
{
XWindowAttributes xgwa;

	if(event->type == ConfigureNotify){
		XGetWindowAttributes(theDisp, theWindow, &xgwa);
		newWindowWidth = xgwa.width;
		newWindowHeight = xgwa.height;
		if(app_data.verbose && app_data.debug){
			fprintf(stderr,"xslideshow: Get Resize Notify Event (%d x %d)\n",
							newWindowWidth,newWindowHeight);
		}
	}
}


/*
* Shell sort alogrithm
*/
#if defined(__STDC__) || defined(__cplusplus)
static void shellSortListIndex(char **listp, int *idxp, int count)
#else
static void shellSortListIndex(listp, idxp, count)
char **listp;
int *idxp;
int count;
#endif
{
int h,i,j,k,tmp;

	if(app_data.verbose && app_data.debug) 
		fprintf(stderr,"shellSortListIndex()\n");

	/* Shell Sort */
	for(h = count >> 1; h > 0; h >>= 1) {
		for(i = count - h,k = 1; k <= i; k++) {
			for(j = k - 1;
				(j >= 0) && (strcmp(listp[idxp[j]], listp[idxp[j+1]]) > 0);
				j -= h)	{
				tmp       = idxp[j];
				idxp[j]   = idxp[j+1];
				idxp[j+1] = tmp;
			} 
		}
		myevent();	/* Get the X Event */
	}

	if(app_data.verbose && app_data.debug) {
		for(i = 0; i < count; i++) {
			fprintf( stderr, "%d ", idxp[i]);
		}
		fprintf( stderr, "\n");
	}
}


/*
* Search the directory and add files to the list
*/
#if defined(__STDC__) || defined(__cplusplus)
void SearchDirAndAddFileToList(char *fileDir, int needIndex)
#else
void SearchDirAndAddFileToList(fileDir, needIndex)
char *fileDir;
int needIndex;
#endif
{
int  i,j,k,size;
int  file_list_size = 10, cur_file_list_size;
char *d,pathDir[MAX_TEXT_LEN];

		/* Free previous file list */
		if(file_listp != (char **)NULL){
			if(file_listp[0] != (char *)NULL){
				for(i = 0; i < file_list_count; i++){
					XtFree((char *)file_listp[i]);
				}
			}
			XtFree((char *)file_listp);
		}

		cur_file_list_size = file_list_size;
		if((file_listp = (char **)XtMalloc(sizeof(char *) * cur_file_list_size)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}

		strcpy(pathDir, fileDir);
		file_listp[0] = (char *)NULL;
		file_list_count = 0;
		file_list_cur_count = 0;
		i = 0;
		d = pathDir;
		size = strlen(d);

		if(!app_data.root)
			makeGumbyCursor();

		while (i < size) {
			if (d[i] == ' ' || d[i] == '\t' || d[i] == '\n' || d[i] == 0) {
				i++;
				continue;
			}
			
			{
			DIR *dirp;
			struct dirent *direntp;
			struct stat buf;
			int first_name = 0;
			int offs = 0;
			char *newPath;
			char filePath[MAX_TEXT_LEN];

				newPath = d + i;

				/* Skip white space */
				while (d[i] != ' ' && d[i] != '\t' && d[i] != '\n' && d[i] != 0) i++;
				d[i++] = 0;

				if(app_data.verbose && app_data.debug)
					fprintf( stderr, "xslideshow: search %s\n", newPath);

				if((dirp = opendir( newPath )) == NULL) {
					file_list_count = 0;
					return;
				}

				while ( (direntp = readdir( dirp )) != NULL ){
					if(first_name == 0){

						/* Solaris ucbcc + SYSV lib does not work correctry.	    */
						/* So, if the first d_name is not ".", then adjust name pointer */

#if 0 /* Just Comment */
	This is a dirent structure of SYSV (Solaris)
	struct  dirent {
		ino_t	    d_ino;
		off_t	    d_off;
		unsigned short   d_reclen;
		char	     d_name[1];
	};

	This is a dirent structure of BSD
		struct  dirent {
			off_t   d_off;		  /* offset of next disk directory entry */
			u_long  d_fileno;	       /* file number of entry */
			u_short d_reclen;	       /* length of this record */
			u_short d_namlen;	       /* length of string in d_name */
			char    d_name[MAXNAMLEN + 1];  /* name (up to MAXNAMLEN + 1) */
		};
#endif

						first_name = 1;
						if(strcmp(direntp->d_name,"."))
							offs = -2;

	 				}
					if (!strcmp(direntp->d_name, ".") || !strcmp(direntp->d_name, ".."))
						continue;
					
					strcpy( filePath, newPath );
					strcat( filePath, "/"  );
					strcat( filePath, direntp->d_name + offs);
					stat(filePath, &buf);
					if( (buf.st_mode & S_IFDIR) == S_IFDIR )
						continue;

					checkFileType(filePath);
					if(fileType != IS_GIF && fileType != IS_JPG) {
						if(app_data.verbose && app_data.debug)
							fprintf( stderr, "xslideshow: %s is not a GIF or JPEG file\n", filePath);
						continue;
					}

					if(app_data.verbose && app_data.debug)
						fprintf( stderr, "xslideshow: %s\n", filePath);

					if (cur_file_list_size <= file_list_count){
						cur_file_list_size += file_list_size;
						file_listp = (char **)XtRealloc((char *)file_listp, sizeof (char *) * cur_file_list_size);
					}
					if((file_listp[file_list_count] = (char *)XtMalloc(sizeof(char) * strlen(filePath) + 1)) == NULL){
						fprintf(stderr,"xslideshow: memory allocation error\n");
						exit(1);
					}
					strcpy(file_listp[file_list_count], filePath);
					file_list_count++;

					myevent();
	 			}
	 			(void)closedir( dirp );
			}
		}

		if(file_list_count == 0 || needIndex == 0) {
			return;
		}

		/* Create the show order index */
		if(file_list_idxp != (int *)NULL)
			XtFree((char *)file_list_idxp);

		if((file_list_idxp = (int *) XtMalloc (sizeof (int) * file_list_count)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}

		if(strcmp(app_data.fileSelectionMode,"random") == 0){	/* random order */

			for(j = 0; j < file_list_count; j++)
				file_list_idxp[j] = -1;

			for(j = 0; j < file_list_count; j++){
				while(1){
					k = myrandom() % file_list_count;
					if(file_list_idxp[k] == -1) {
						file_list_idxp[k] = j;
						break;
					}
				}
				myevent();
			}

			if(app_data.verbose && app_data.debug){
				fprintf(stderr,"\nTotal file count = %d ,",file_list_count);
				fprintf(stderr,"Random show order is shown below.\n");
				for(j = 0;j < file_list_count;j++)
					fprintf(stderr,"%d ",file_list_idxp[j]);
				fprintf(stderr,"\n");
			}

		}
		else{	/* sequent order */

			for(j = 0;j < file_list_count;j++)
				file_list_idxp[j] = j;

			shellSortListIndex(file_listp, file_list_idxp, file_list_count);

			if(app_data.verbose && app_data.debug) 
				fprintf(stderr,"\nTotal file count = %d\n",file_list_count);

		}
}

void ShowNoVaildFileMessage()
{
	if(messageWindow != (Window)0)
		removeMessageWindow();

	if(theVisual->class == PseudoColor){
		if((app_data.root || app_data.override) 
			&& app_data.privateCmap) { 
			XUninstallColormap(theDisp,theCmap);
		}
		else {
			XSetWindowColormap(theDisp,theWindow,defCmap);
		}
	}
	XSetWindowBackground(theDisp,theWindow,dbcol);
	XClearWindow(theDisp,theWindow);

	/* Display the ERROR Messages */
	{int textheight,w,h;
		textheight = showErrorMessages(1, errstrs[ERR_NO_FILE], NULL);
		w = xshiohara_mask_width;
		h = xshiohara_mask_height;
		createSpecialWindow( (windowWidth-w)/2, (windowHeight-textheight)/2 - h );
	}

	if(!app_data.root)
		makeAppreciationCursor();

	/* Loop until get the other path name from xslideshow-control */
	while(1){
		myevent();	/* Get the X Event */
		if(setFilePath[0] != 0){
			SearchDirAndAddFileToList(setFilePath, 1);
			setFilePath[0] = 0;
			if(file_list_count > 0)
				break;
		}
	}
	removeSpecialWindow();
	removeMessageWindow();
}

/*
* Xslideshow main loop
*/
#if defined(__STDC__) || defined(__cplusplus)
Boolean XSlideshowMainLoop(Widget topWidget)
#else
Boolean XSlideshowMainLoop(topWidget)
Widget topWidget;
#endif
{
ErrStatus errf;
int action, i;
int reset_file_list_count;

	file_list_cur_count = 0;
	anim_list_cur_count = 0;
	valid_file_num = 0;

	/* Endless loop */
	while(1){

		if(app_data.verbose) {
			fprintf(stderr,
					"\nxslideshow: <%d/%d> %d : ",
					file_list_cur_count + 1,
					file_list_count,
					file_list_idxp[ file_list_cur_count ] + 1	);
		}

		if(anim_list_count != 0 && animMode[0] == 0) {
			if(anim_list_cur_count >= anim_list_count)
				anim_list_cur_count = 0;

			app_data.animate = (char *)anim_list[ anim_list_idxp[ anim_list_cur_count ] ];
			anim_list_cur_count++;
		}
		else if(animMode[0] != 0) {
			app_data.animate = animMode;
		}

		useGlobalCmap = False;
		if(theVisual->class == TrueColor){
			/* TrueColor can't animate below */
			if(	   !strcmp(app_data.animate , "fade")
				|| !strcmp(app_data.animate , "reverse")
				|| !strcmp(app_data.animate , "monotone")
				|| !strcmp(app_data.animate , "flush")
				|| !strcmp(app_data.animate , "coloranim")
				|| !strcmp(app_data.animate , "coloranim2")
				)
				app_data.animate = defAnim;
		}
		else {
			 /* PseudoColor */
			if(    !strcmp(app_data.animate , "block24")
				|| !strcmp(app_data.animate , "wipe24")
				|| !strcmp(app_data.animate , "float24")
				|| !strcmp(app_data.animate , "transparent")
				) {
				useGlobalCmap = True;
			}
		}

		if(app_data.verbose)
			fprintf(stderr,"%s\n",app_data.animate);

		/* Proceed animation */

		ACT_SLEEP_ON(click); /* enable display interval time */

		if(setFileName[0] != NULL && strcmp( setFileName, "(removed)" )) {
			/* It is set by xslideshow-command */
			strcpy(tmpBuf, setFileName);

			/* Search current file name index */
			file_list_cur_count = 0;
			for(i = 0; i < file_list_count; i++){
				if(file_listp[ file_list_idxp[ i ] ] == NULL)
					continue;
				if( !strcmp( file_listp[ file_list_idxp[ i ] ], tmpBuf ) ){
					file_list_cur_count = i;	
					break;
				}
			}
		}
		setFileName[0] = NULL;

		reset_file_list_count = file_list_cur_count;
		while(file_listp[ file_list_idxp[ file_list_cur_count ] ] == NULL){
		
			if(IS_PREV_ON(click)){
				--file_list_cur_count;
				if(file_list_cur_count < 0) {
					file_list_cur_count = file_list_count-1;
				}
			}
			else {
				file_list_cur_count++;
				if(file_list_cur_count >= file_list_count){
					file_list_cur_count = 0;
				}
			}
			if(reset_file_list_count == file_list_cur_count){
				ShowNoVaildFileMessage();
			}
		}
		strcpy(targetFilePath, file_listp[ file_list_idxp[ file_list_cur_count ] ]);
		errf = animateShow(	topWidget,
							targetFilePath,
							&hasImage,
							&action	);

		/* End display interval */
		ACT_SLEEP_OFF(click);

		/* Check exit status */
		if(errf != ERR_NO_ERR){
			if(messageWindow != (Window)0)
				removeMessageWindow();

			if(theVisual->class == PseudoColor){
				if((app_data.root || app_data.override) 
					&& app_data.privateCmap) {
					XUninstallColormap(theDisp,theCmap);
				}
				else {
					XSetWindowColormap(theDisp,theWindow,defCmap);
				}
			}
			XSetWindowBackground(theDisp,theWindow,dbcol);
			XClearWindow(theDisp,theWindow);

			/* Display ERROR Message */
			(void)showErrorMessages(	2,	/* line position */
										errstrs[errf],
										file_listp[ file_list_idxp [ file_list_cur_count ] ]	);

			myusleep(10000000);	/* Show error message for 10 sec */

			removeMessageWindow();
		}


		/* Check if got the new file path directory from xslideshow-control */
		if(setFilePath[0] != 0){

			/* Search file path directory */
			strcpy(tmpBuf, setFilePath);
			setFilePath[0] = 0;
			SearchDirAndAddFileToList(tmpBuf, 1);

			/* If there are no image file */
			if(file_list_count == 0){
				if(messageWindow != (Window)0)
					removeMessageWindow();

				if(theVisual->class == PseudoColor){
					if((app_data.root || app_data.override) 
						&& app_data.privateCmap) { 
						XUninstallColormap(theDisp,theCmap);
					}
					else {
						XSetWindowColormap(theDisp,theWindow,defCmap);
					}
				}
				XSetWindowBackground(theDisp,theWindow,dbcol);
				XClearWindow(theDisp,theWindow);

				/* Display the ERROR Messages */
				{int textheight,w,h;
					textheight = showErrorMessages(1, errstrs[ ERR_NO_FILE ], NULL);
					w = xshiohara_mask_width;
					h = xshiohara_mask_height;
					createSpecialWindow( (windowWidth-w)/2, (windowHeight-textheight)/2 - h );
				}

				if(!app_data.root)
					makeAppreciationCursor();

				/* Loop until get the other path name from xslideshow-control */
				while(1){
					myevent();	/* Get the X Event */
					if(setFilePath[0] != 0){
						strcpy(tmpBuf, setFilePath);
						setFilePath[0] = 0;
						SearchDirAndAddFileToList(tmpBuf, 1);
						if(file_list_count > 0)
							break;
					}
				}
				removeSpecialWindow();
				removeMessageWindow();
			}
		}
		else {

			if(hasImage)
				valid_file_num++;


			if(IS_HEAD_ON(action)){
				file_list_cur_count = 0;
				ACT_HEAD_OFF(click);
			}
			else if(IS_TAIL_ON(action)){
				file_list_cur_count = file_list_count-1;
				ACT_TAIL_OFF(click);
			}
			else if(IS_PREV_ON(action)){
				--file_list_cur_count;
				if(file_list_cur_count < 0) {
					if(valid_file_num == 0) {
						ShowNoVaildFileMessage();
					}
					file_list_cur_count = file_list_count-1;
				}
			}
			else if(IS_NEXT_ON(action)){
				file_list_cur_count++;
				if(file_list_cur_count >= file_list_count){
					if(valid_file_num == 0) {
						ShowNoVaildFileMessage();
					}
					file_list_cur_count = 0;
				}
			}
		}
		myevent();	/* Get the X Event */
	}
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
Bool JudgeMapNotify(Display *dpy, XEvent *ev, XPointer w)
#else
Bool JudgeMapNotify(dpy, ev, w)
Display *dpy;
XEvent *ev;
XPointer w;
#endif
{
	return(	ev->xany.type == MapNotify
			&& ev->xvisibility.window == (Window)w );
}

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
Bool JudgeConfigureNotify(Display *dpy, XEvent *ev, XPointer w)
#else
Bool JudgeConfigureNotify(dpy, ev, w)
Display *dpy;
XEvent *ev;
XPointer w;
#endif
{
	return(	ev->xany.type == ConfigureNotify
			&& ev->xvisibility.window == (Window)w );
}


/*
*
*/
void checkAndCreateAnimateList()
{
int  i,j,k,size;
int  anim_list_size = 10;
char *d;

	/* Reset the animation count */
	anim_list_count = 0;

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: animate = %s \n",app_data.animate);

	/* Create the animation list */
	if(	strcmp(app_data.animate,"random") == 0
			&& app_data.animateList[0] != 0){

		if(app_data.verbose)
			fprintf(stderr,"xslideshow: animateList = %s\n",app_data.animateList);

		/* Allocate the first list buffer */
		if((anim_list = (char **)XtMalloc(sizeof(char *) * anim_list_size)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}
		anim_list[0] = (char *)NULL;

		/* Read the animation string and create the list */
		i = 0;
		d = app_data.animateList;
		size = strlen(d);
		while (i < size) {
			/* Skip white space */
			if (d[i]==' ' || d[i]=='\t' || d[i]=='\n' || d[i]==0) {i++;continue;}

			/* Reallocate the list buffer */
			if (anim_list_size <= anim_list_count ) {
				anim_list_size = anim_list_size * 2;
				anim_list = (char **)XtRealloc((char *)anim_list,sizeof (char *) * anim_list_size);
			}

			/* Set start pointer of an animation string to the list */
			anim_list[anim_list_count++] = d + i;

			/* Search the end position of an animation string */
			while (d[i]!=' ' && d[i]!='\t' && d[i]!='\n' && d[i]!=0) i++;
			d[i++] = 0;
		}

		/* Create the show order index */
		if((anim_list_idxp = (int *) XtMalloc (sizeof (int) * anim_list_count)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}

		for(j = 0; j < anim_list_count; j++) anim_list_idxp[j] = -1;

		for(j = 0; j < anim_list_count; j++){
			while(1){
				k = myrandom() % anim_list_count;
				if(anim_list_idxp[k] == -1) {
					anim_list_idxp[k] = j;
					break;
				}
			}
		}

		if(app_data.verbose && app_data.debug){
			fprintf(stderr,"xslideshow: Total animation count = %d ,",anim_list_count);
			fprintf(stderr,"Animation order is shown below.\n");
			for(j = 0;j < anim_list_count;j++)
				fprintf(stderr,"%s ",anim_list[anim_list_idxp[j]]);
			fprintf(stderr,"\n");
		}
	}

}

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
void checkCommandlineOptions(int argc, char **argv)
#else
void checkCommandlineOptions(argc, argv)
int argc;
char **argv;
#endif
{
int  i,j,cmdlinefiles = 0;
int optc;
char **optv;

	optc = argc;
	optv = argv;
	while(--optc){
		++optv;
		if(strncmp(*optv,"-",1)){
			if(*optv[0] == 0)
				continue;
			else
				cmdlinefiles++;
		}
	}

	if(cmdlinefiles){

		/* Apply files from command line */
		if((file_listp = (char **)XtMalloc(sizeof(char *) * cmdlinefiles)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}

		file_list_cur_count = 0;
		file_list_count = 0;
		optc = argc;
		optv = argv;

		while(--optc){
			++optv;
			if(strncmp(*optv,"-",1)){
				if(*optv[0] == 0)
					continue;

				checkFileType(*optv);
				if(fileType != IS_GIF && fileType != IS_JPG) {
					if(app_data.verbose && app_data.debug)
						fprintf( stderr, "xslideshow: %s is not a GIF or JPEG file\n", *optv);
					continue;
				}

				if(app_data.verbose && app_data.debug)
					fprintf( stderr, "xslideshow: add %s to the list\n", *optv);

				if((file_listp[file_list_count] = (char *)XtMalloc(sizeof(char) * strlen(*optv) + 1)) == NULL){
					fprintf(stderr,"xslideshow: memory allocation error\n");
					exit(1);
				}
				strcpy(file_listp[file_list_count], *optv);
				file_list_count++;
			}
		}

	}
	else {
		if(app_data.verbose)
			fprintf( stderr, "xslideshow: path = %s\n", app_data.pathDir);

		SearchDirAndAddFileToList(app_data.pathDir, 0);
	}

	if(file_list_count == 0){ /* There are no file */
		char cbuf1[64],cbuf2[64];
		sprintf(cbuf1,"Xslideshow by Susumu Shiohara.");
		sprintf(cbuf2,"version %s, patchlevel %s",version,patchlevel);
		{
		int textheight, w,h;
			textheight = showErrorMessages(2,cbuf1,cbuf2);
			w = xshiohara_mask_width;
			h = xshiohara_mask_height;
			createSpecialWindow( (windowWidth-w)/2, (windowHeight-textheight)/2 - textheight/4 - h );
		}

		if(!app_data.root)
			makeAppreciationCursor();

		/* Loop until get the other path name from xslideshow-control */
		while(1){
			myevent();	/* Get the X Event */
			if(setFilePath[0] != 0){
				strcpy(tmpBuf, setFilePath);
				setFilePath[0] = 0;
				SearchDirAndAddFileToList(tmpBuf, 1);
				if(file_list_count > 0)
					break;
			}
		}
		removeSpecialWindow();
		removeMessageWindow();
	}

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: file selection mode = %s \n",app_data.fileSelectionMode);

	/* SearchDirAndAddFileToList() make random or sequent order */
	if(file_list_idxp == (int *)NULL) {

		/* Create the show order index */
		if((file_list_idxp = (int *) XtMalloc (sizeof (int) * file_list_count)) == NULL){
			fprintf(stderr,"xslideshow: memory allocation error\n");
			exit(1);
		}

		if(strcmp(app_data.fileSelectionMode,"random") == 0){	/* random order */

			for(j = 0; j < file_list_count; j++)
				file_list_idxp[j] = -1;

			for(j = 0; j < file_list_count; j++){
				while(1){
					i = myrandom() % file_list_count;
					if(file_list_idxp[i] == -1) {
						file_list_idxp[i] = j;
						break;
					}
				}
			}

			if(app_data.verbose && app_data.debug){
				fprintf(stderr,"\nTotal file count = %d ,",file_list_count);
				fprintf(stderr,"Random show order is shown below.\n");
				for(j = 0;j < file_list_count;j++)
					fprintf(stderr,"%d ",file_list_idxp[j]);
				fprintf(stderr,"\n");
			}

		}
		else{	/* sequent order */

			for(j = 0;j < file_list_count;j++)
				file_list_idxp[j] = j;

			shellSortListIndex(file_listp, file_list_idxp, file_list_count);

			if(app_data.verbose && app_data.debug) 
				fprintf(stderr,"\nTotal file count = %d\n",file_list_count);

		}
	}
}

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
void get_display_info(Display *dpy)
#else
void get_display_info(dpy)
Display *dpy;
#endif
{
int i, n;
XPixmapFormatValues *pmf;

	pmf = XListPixmapFormats (dpy, &n);
	if(app_data.verbose && app_data.debug){
		fprintf(stderr, "number of supported pixmap formats: %d\n", n);
	}
	if (pmf) {
		if(app_data.verbose && app_data.debug){
			fprintf(stderr, "supported pixmap formats:\n");
		}
		for (i = 0; i < n; i++) {
			if(app_data.verbose && app_data.debug){
				fprintf(stderr,"	depth %d, bits_per_pixel %d, scanline_pad %d\n",
						pmf[i].depth, pmf[i].bits_per_pixel, pmf[i].scanline_pad);
			}
			if(pmf[i].depth == theDepth){
				gim.pmf.depth = pmf[i].depth;
				gim.pmf.bits_per_pixel = pmf[i].bits_per_pixel;
				gim.pmf.scanline_pad = pmf[i].scanline_pad;
			}
		}
		XFree ((char *) pmf);
	}
	return;
}

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
void main(int argc, char **argv)
#else
void main(argc,argv)
int argc;
char **argv;
#endif
{
struct timeval tp;
struct timezone tzp;
XEvent event;
XSetWindowAttributes xswa;
XWindowAttributes xgwa;
XGCValues gcv;
int i, ac;
byte *bp;
Arg args[5];

	gettimeofday(&tp, &tzp);
	pastTime = tp.tv_sec * 1000000 + tp.tv_usec;
	mysrandom((unsigned long)tp.tv_sec);

	/* Initialize signals */

	signal(SIGTERM,terminate);
#if !defined (_DEBUG)
	signal(SIGBUS, bus_error);
	signal(SIGSEGV,segmentation_fault);
	signal(SIGTRAP,trace_trap);
#endif

	/* Initialize Instance */

	theDisp	 = NULL;
	theCmap	 = (Colormap)0;
	fileWindow	     = (Window)0;
	sizeWindow	     = (Window)0;
	messageWindow	  = (Window)0;
	specialWindow	  = (Window)0;
	click		  = ACTION_NEXT_FRAME;
	gotSomeAction	  = False;
	allocatedColorNum      = 0;
	allocatedSysColorNum   = 0;
	setFilePath[0]	 = 0;
	destinationPath[0] = 0;
	animMode[0]	    = 0;
	file_listp	     = (char **)NULL;
	file_list_idxp	 = (int *)NULL;
	file_list_count = 0;
	pauseToggle	    = 0;
	theImage	       = NULL;
	gim.subImageList       = (SUBGIMAGE *)XtMalloc(sizeof(SUBGIMAGE));
	gim.subImageList->next = (SUBGIMAGE *)NULL;
	gim.subImageList->image= (byte *)NULL;


	/* Initialize WorkProcs */
	work_procs.numProc = 0;
	for(i = 0; i < MAX_WORK_PROCS;i++){
		work_procs.procs[i].proc = NULL;
	}

	/* Initialize X Toolkit */
	
	toplevel = XtAppInitialize(
							&app_context,
							"Xslideshow",
							options,
							XtNumber(options),
							&argc,argv,
							defaults,
							0,0);

	if(toplevel == (Widget)NULL)
		exit(1);

	XtGetApplicationResources(toplevel,
							&app_data,
							resources,
							XtNumber(resources),NULL,0);

	if(app_data.help) {
		fprintf(stderr,"Usage:\n");
		for(i=0; help_message[i] != NULL; i++){
			fprintf(stderr,"%s\n",help_message[i]);
		}
		exit(0);
	}

	if(app_data.translations != NULL)
		XtOverrideTranslations(toplevel,
							XtParseTranslationTable(app_data.translations));

	XtAppAddActions(app_context,(XtActionList)actions,XtNumber(actions));

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: version %s, patchlevel %s\n",version,patchlevel);


	Endean = 0x01234567; bp = (byte *)&Endean;
	if(*bp == 0x01) Endean = IS_BIG_ENDEAN;
	else Endean = IS_LITTLE_ENDEAN;
	
	if(app_data.verbose && app_data.debug){
		if(Endean == IS_BIG_ENDEAN)
			fprintf(stderr,"xslideshow: I guess your machine is BIG_ENDEAN\n");
		else
			fprintf(stderr,"xslideshow: I guess your machine is LITTLE_ENDEAN\n");

		for(i=0; i<argc; i++)
			fprintf(stderr,"%s ",*(argv + i));
		fprintf(stderr,"\n");
	}



	theDisp	 = XtDisplay(toplevel);
	theVisual = DefaultVisual(theDisp,theScreenNumber);
	theDepth = DefaultDepth(theDisp,theScreenNumber);
	defCmap	 = DefaultColormap(theDisp,theScreenNumber);
	theScreenNumber = DefaultScreen(theDisp);

	gim.visual.class       = theVisual->class;
	gim.visual.red_mask    = theVisual->red_mask;
	gim.visual.green_mask  = theVisual->green_mask;
	gim.visual.blue_mask   = theVisual->blue_mask;
	gim.visual.map_entries = theVisual->map_entries;

	get_display_info(theDisp);

#if defined (_DEBUG)
# if defined (_DEPTH16)
	theDepth = 16;
	gim.pmf.bits_per_pixel = 16;
	gim.visual.red_mask  = 0xf800;
	gim.visual.green_mask= 0x07e0;
	gim.visual.blue_mask = 0x001f;
# elif defined (_DEPTH15)
	theDepth = 15;
	gim.pmf.bits_per_pixel = 16;
	gim.visual.red_mask  = 0x7c00;
	gim.visual.green_mask= 0x03e0;
	gim.visual.blue_mask = 0x001f;
# elif defined (_DEPTH12)
	theDepth = 12;
	gim.pmf.bits_per_pixel = 16;
	gim.visual.red_mask  = 0x0f00;
	gim.visual.green_mask= 0x00f0;
	gim.visual.blue_mask = 0x000f;
# elif defined (_DEPTH9)
	theDepth = 9;
	gim.pmf.bits_per_pixel = 16;
	gim.visual.red_mask  = 0x01c0;
	gim.visual.green_mask= 0x0038;
	gim.visual.blue_mask = 0x0007;
# endif
#endif


	/* Create the Window */

	if(app_data.root){

		/* RootWindow */
		theWindow = XCreateSimpleWindow(theDisp,
			RootWindowOfScreen(XtScreen(toplevel)),
			0,0,	/* Origin */
			DisplayWidth(theDisp,theScreenNumber),
			DisplayHeight(theDisp,theScreenNumber),
			0,
			BlackPixel(theDisp,theScreenNumber),
			BlackPixel(theDisp,theScreenNumber));
		xswa.override_redirect = True;
		XChangeWindowAttributes(theDisp, theWindow, CWOverrideRedirect, &xswa);
		XLowerWindow(theDisp,theWindow);
		XMapWindow(theDisp,theWindow);
		XtDestroyWidget(toplevel);
		topWidget = (Widget)0;
		makeNoCursor();

	}
	else{

		if(app_data.full) {

			if(app_data.override){

				/* Max size override windiow */
				topWidget = XtCreatePopupShell("topWidget",
											overrideShellWidgetClass,
											toplevel,NULL,0);
				XtRealizeWidget(topWidget);
				while(!(XtIsRealized(topWidget)));

				ac = 0;
				XtSetArg(args[ac],XtNborderWidth,0);ac++;
				XtSetArg(args[ac],XtNx,0);ac++;
				XtSetArg(args[ac],XtNy,0);ac++;
				XtSetArg(args[ac],XtNwidth,DisplayWidth(theDisp,theScreenNumber));ac++;
				XtSetArg(args[ac],XtNheight,DisplayHeight(theDisp,theScreenNumber));ac++;
				XtSetValues(topWidget,args,ac);
				XtMapWidget(topWidget);
				theWindow = XtWindow(topWidget);
				xswa.event_mask = ButtonPressMask | KeyPressMask;
				XChangeWindowAttributes(theDisp, theWindow, CWEventMask, &xswa);
				XtMoveWidget(topWidget,0,0);
				XSync(theDisp, False);
				makeNoCursor();

			}
			else {

				/* Max size windiow */
				topWidget = toplevel;
				XtRealizeWidget(topWidget);
				while(!(XtIsRealized(topWidget)));

				theWindow = XtWindow(topWidget);
				XMoveWindow(theDisp,theWindow,0,0);
				ac = 0;
				XtSetArg(args[ac],XtNwidth,DisplayWidth(theDisp,theScreenNumber));ac++;
				XtSetArg(args[ac],XtNheight,DisplayHeight(theDisp,theScreenNumber));ac++;
				XtSetValues(topWidget,args,ac);
				wm_delete_window = XInternAtom(XtDisplay(toplevel),"WM_DELETE_WINDOW",False);
				(void)XSetWMProtocols(XtDisplay(toplevel),XtWindow(toplevel),&wm_delete_window,1);
				XIfEvent(theDisp,&event,JudgeMapNotify,(XPointer)theWindow);
				XSync(theDisp, False);
				XIfEvent(theDisp,&event,JudgeConfigureNotify,(XPointer)theWindow);
				XSync(theDisp, False);
				XSetInputFocus(theDisp,theWindow,RevertToParent,CurrentTime);
				makeCursors();

			}

		}
		else {

			if(app_data.override){
				int x,y,width,height;
				XParseGeometry(app_data.geometry,&x,&y,&width,&height);

				/* Override windiow */
				topWidget = XtCreatePopupShell("topWidget",
											overrideShellWidgetClass,
											toplevel,NULL,0);
				XtRealizeWidget(topWidget);
				while(!(XtIsRealized(topWidget)));

				ac = 0;
				XtSetArg(args[ac],XtNborderWidth,0);ac++;
				XtSetArg(args[ac],XtNwidth,width);ac++;
				XtSetArg(args[ac],XtNheight,height);ac++;
				XtSetValues(topWidget,args,ac);
				XtMapWidget(topWidget);
				theWindow = XtWindow(topWidget);
				XtMoveWidget(topWidget,x,y);
				XSync(theDisp, False);
				makeNoCursor();

			}
			else {

				/* Normal windiow */
				topWidget = toplevel;
				XtRealizeWidget(topWidget);
				while(!(XtIsRealized(topWidget)));

				theWindow = XtWindow(topWidget);
				if(!app_data.resize) 
					XtAddEventHandler(topWidget, StructureNotifyMask, FALSE, 
						(XtEventHandler)ResizeWindow, (XtPointer)NULL);
				wm_delete_window = XInternAtom(XtDisplay(toplevel),"WM_DELETE_WINDOW",False);
				(void)XSetWMProtocols(XtDisplay(toplevel),XtWindow(toplevel),&wm_delete_window,1);
				XIfEvent(theDisp,&event,JudgeMapNotify,(XPointer)theWindow);
				XSync(theDisp, False);
				XSetInputFocus(theDisp,theWindow,RevertToParent,CurrentTime);
				makeCursors();

			}

		}
	}

	xswa.backing_store = WhenMapped;
	XChangeWindowAttributes(theDisp, theWindow, CWBackingStore, &xswa);

	/* Initialize the window */

	XGetWindowAttributes (theDisp,theWindow,&xgwa);
	prevWindowWidth = newWindowWidth = windowWidth = xgwa.width;
	prevWindowHeight = newWindowHeight = windowHeight = xgwa.height;
	dwcol = WhitePixel(theDisp,theScreenNumber);
	dbcol = BlackPixel(theDisp,theScreenNumber);
	if(app_data.verbose && app_data.debug)
		fprintf(stderr,"xslideshow: black=%d, white=%d\n",(int)dbcol,(int)dwcol);

	fontGC = DefaultGC(theDisp,theScreenNumber);
	gcv.font = app_data.font->fid;
	if(dbcol == 1 && dwcol == 0)
		gcv.function = GXand;
	else if(dbcol == 0 && dwcol == 1)
		gcv.function = GXor;
	else
		gcv.function = GXcopy;
	gcv.foreground = dwcol;
	gcv.background = dbcol;
	XChangeGC(theDisp,fontGC,GCFont | GCForeground | GCBackground | GCFunction,&gcv);

	if(app_data.privateCmap)
		theCmap = XCreateColormap(theDisp,theWindow,theVisual,AllocNone);
	else
		theCmap = DefaultColormap(theDisp,theScreenNumber);

	/* Check the visual class */

	/* XSlideshow works only PseudoColor and TrueColor */
	if( theVisual->class != PseudoColor && theVisual->class != TrueColor) {
		(void) showErrorMessages(1,errstrs[ERR_SCREEN],NULL);
		XtAppMainLoop(app_context);
	}

	if(app_data.verbose && app_data.debug){
		fprintf(stderr, "xslideshow: class = %d\n", theVisual->class);
		fprintf(stderr, "xslideshow: depth = %d\n", theDepth);
		fprintf(stderr, "xslideshow: red_mask = %04x\n", (int)theVisual->red_mask);
		fprintf(stderr, "xslideshow: green_mask = %04x\n", (int)theVisual->green_mask);
		fprintf(stderr, "xslideshow: blue_mask = %04x\n", (int)theVisual->blue_mask);
		fprintf(stderr, "xslideshow: map_entries = %d\n", theVisual->map_entries);
		fprintf(stderr, "xslideshow: bits_per_rgb = %d\n", theVisual->bits_per_rgb);
	}
	if( theVisual->class == TrueColor) {
		if(app_data.perfect) {
			if(!app_data.root)
				makeWaitCursor();
			if(initTrueColor() == False)
				app_data.perfect = False;
		}
	}
	else {
		app_data.perfect = False; /* Disable perfect mode */
	}

	/* Initialize Communication */

	if(topWidget != (Widget)0)
		initComm(topWidget, theDisp, theWindow);

	if(app_data.clock)
		initClock();

	if(app_data.biff) 
		initBiff();

	/* Create the animation mode list */

	checkAndCreateAnimateList();

	/* Create the file list */

	checkCommandlineOptions(argc, argv);


	if(app_data.destDir[0] != 0) 
		strcpy(destinationPath, app_data.destDir);

	/* Main loop */
	(void) XSlideshowMainLoop(topWidget);
}

#undef IS_MAIN

