#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* maincanvas.c: routines for configuring and handling the main canvas
 *
 * This module handles events on the main canvas, which is where the trees
 * are displayed, and all editing is done.  These routines create this
 * canvas, zoom in and zoom out, set various options, and manage the
 * drawing area.  They also manage the list of event functions, which are
 * functions that handle button and key presses.
 */

#include "interface.h"

#include "treedata.h"

/* this is a list of event functions.  When an event happens on the canvas
 * (such as button press or mouse enter), these handlers are called in order.
 * If a handler deals with an event, it returns 1, and that is the last
 * handler called.  If a handler doesn't deal with an event, it returns 0 and
 * the next handler is called */
typedef struct EFLIST {
	int (*f)();
	struct EFLIST *next;
	} eflistd, *eflist;

eflist eventfunctions;

int canvas_do_events(c, e)
tcanvas c;
tevent e;
{
	static eflist t;
	static int x, y;

	if(tevent_type(e)==te_repaint)
	{
		/* repaint all trees */
		draw_all_trees(c);
		if(globals->mode==MouseMove)
			draw_all_tree_boundaries(c);
		draw_selection(c);
	}
	else
	{
		seteventloc(e);
		tevent_location(e, &x, &y);

		/* call event functions in order, stopping if one deals with the
		 * event */
		for(t=eventfunctions->next;t!=NULL;t=t->next)
			if((*t->f)(c, e, x, y, NULL))
				break;
	}
	return(1);
}

void create_main_canvas(f, p, args)
tframe f;
tpanel p;
targs args;
{
	int pw, ph;

	if(PAGEW>tdisplay_width(f)-15)
		pw=tdisplay_width(f)-15;
	else
		pw=PAGEW;
	if(PAGEH>tdisplay_height(f)-60-tpanel_height(p))
		ph=tdisplay_height(f)-60-tpanel_height(p);
	else
		ph=PAGEH;
#ifdef DEBUGIGNORE
	if(1)
	globals->c=tcanvas_new(f, 0, 0, PAGEW, 200, NULL, p,
		PAGEW, PAGEH, 15, 15, 20, 10, globals->usebuffer, args);
	else
#endif
	globals->c=tcanvas_new(f, 0, 0, pw, ph, NULL, p,
		PAGEW, PAGEH, 15, 15, 20, 10, globals->usebuffer, args);
	tcanvas_set_event_procedure(globals->c, canvas_do_events);
	globals->depth=tdisplay_depth(titem_display(globals->c));

	eventfunctions=(eflist)malloc(sizeof(eflistd));
	eventfunctions->f=canvas_do_events;
	eventfunctions->next=NULL;
}

void resize_main_canvas(x, y, w, h, right, below)
int x, y, w, h;
titem right, below;
{
	tcanvas_resize(globals->c, x, y, w, h, right, below);
}


void canvas_zoom_in()
{
	static int mag;

	mag=tcanvas_zoom_level(globals->c);
	/* can only zoom in so far */
	if(mag<32)
	{
		/* check if there's a selection and if it's on screen */
		if(get_selection_type()!=NoSel &&
				tcanvas_point_on_screen(globals->c, globals->px, globals->py))
			/* if there is, zoom in on the selection */
			tcanvas_zoom_in(globals->c, globals->px, globals->py);
		else
			/* otherwise, zoom in on the center */
			tcanvas_zoom_in(globals->c, tcanvas_center_x(globals->c, 0),
				tcanvas_center_y(globals->c, 0));
	}
}
void canvas_zoom_out()
{
	static int mag;

	mag=tcanvas_zoom_level(globals->c);
	/* can only zoom in so far */
	if(mag>-32)
	{
		/* check if there's a selection and if it's on screen */
		if(get_selection_type()!=NoSel &&
				tcanvas_point_on_screen(globals->c, globals->px, globals->py))
			/* if there is, zoom in on the selection */
			tcanvas_zoom_out(globals->c, globals->px, globals->py);
		else
			/* otherwise, zoom in on the center */
			tcanvas_zoom_out(globals->c, tcanvas_center_x(globals->c, 0),
				tcanvas_center_y(globals->c, 0));
	}
}

void canvas_show_page_lines()
	/* turn on lines that indicate edges of pages */
{
	globals->pagelines=1;
	tcanvas_pages(globals->c, 1);
}
void canvas_hide_page_lines()
	/* turn off lines that indicate edges of pages */
{
	globals->pagelines=0;
	tcanvas_pages(globals->c, 0);
}

void canvas_show_handles()
{
	globals->handles=1;
	tcanvas_add_all(globals->c);
}
void canvas_hide_handles()
{
	globals->handles=0;
	tcanvas_add_all(globals->c);
}
void canvas_reset()
{
	tcanvas_reset_area(globals->c);
}

void canvas_add_all()
{
	tcanvas_add_all(globals->c);
}

void canvas_add(x, y, w, h)
int x, y, w, h;
{
	tcanvas_add_area(globals->c, x, y, w, h);
}

void canvas_add_d(x, y, w, h)
double x, y;
int w, h;
{
	int xx, yy;
	xx=x;
	yy=y;
	tcanvas_add_area(globals->c, xx, yy, w, h);
}

void canvas_add_scale_bar(t)
tree t;
{
	treetypedata td;
	int x, y, w;

	td=tdta(t)->td;
	ttext_add_area(td->percent);
	x=td->plx;
	y=td->ply-PCHOOSERADIUS;
	w=td->plw;
	tcanvas_add_area(globals->c, x, y, w, (PCHOOSERADIUS+1)*2);
}

void canvas_add_tree(t)
tree t;
{
	treetypedata td;
	int w, h, ww, hh;
	
	w=TRESIZERADIUS;
	h=TRESIZERADIUS;
	size_to_cvs(globals->c, &w, &h);
	td=tdta(t)->td;
	tcanvas_add_area(globals->c, td->x-w, td->y-h, td->w+w*2+1, td->h+h*2+1);
	canvas_add_scale_bar(t);
}

void canvas_add_text(t)
ttext t;
{
	ttext_add_area(t);
}

void canvas_add_branch(b)
treebranch b;
{
	treenode up, down;
	nodetypedata ud, dd;
	treetypedata td;
	int x, y, w, h;
	int lr;

	if(b->up==NULL)
		return;

	ud=ndta(b->up)->td;
	dd=ndta(b->down)->td;
	td=tdta(tree_get_tree(b))->td;

	lr=bdta(b)->td->thickness/2;
	if(lr<BPICKDISTANCE/2)
		lr=BPICKDISTANCE/2;
	x=nodex(ud, td);
	y=nodey(ud, td);
	w=nodex(dd, td)-nodex(ud, td);
	h=nodey(dd, td)-nodey(ud, td);
	normalize_rect(&x, &y, &w, &h);
	x-=lr+1+1;
	y-=lr+1+1;
	w+=2*lr+4+2;
	h+=2*lr+4+2;
	tcanvas_add_area(globals->c, x, y, w, h);
}

void canvas_add_branch_point(b, p)
treebranch b;
double p;
{
	nodetypedata up, down;
	treetypedata td;
	int x, y, w, h;

	if(b->up==NULL)
		return;
	up=ndta(b->up)->td;
	down=ndta(b->down)->td;
	td=tdta(tree_get_tree(b))->td;

	switch(td->bt)
	{
		case FlowerBranch:
			x=nodex(up, td)+(nodex(down, td)-nodex(up, td))*p;
			y=nodey(up, td)+(nodey(down, td)-nodey(up, td))*p;
			break;
		case Midway:
		case RoomForGroup:
		case OnScreen:
		default:
			x=nodex(up, td)+(nodex(down, td)-nodex(up, td))*p;
			y=nodey(down, td);
			break;
	}
	tcanvas_add_handle(globals->c, x, y, BPICKDISTANCE);
}

void canvas_add_node(n)
treenode n;
{
	int x, y;
	nodetypedata nd;
	treetypedata td;

	if(n==NULL)
		return;

	nd=ndta(n)->td;
	td=tdta(tree_get_tree(n))->td;

	x=nodex(nd, td);
	y=nodey(nd, td);
	tcanvas_add_handle(globals->c, x, y, NCHOOSERADIUS);
}

void canvas_add_subtree(n)
treenode n;
{
	list t;

	canvas_add_node(n);
	if(showlabel(n, NoTreeType))
		canvas_add_text(ndta(n)->td->label);
	tforsubtree(n, t)
		canvas_add_subtree(subtree(t));
}

void canvas_no_clipping()
{
	tcanvas_clipping_off(globals->c);
}

void canvas_remove_all_event_functions()
{
	eflist c, p;

	for(c=eventfunctions->next;c!=NULL;)
	{
		p=c;
		c=c->next;
		uninstall(*p->f);
	}
	eventfunctions->next=NULL;
}

int canvas_remove_event_function(f)
int (*f)();
{
	eflist c, p;

	for(c=eventfunctions;c->next!=NULL;c=c->next)
		if(c->next->f==f)
		{
			p=c->next;
			c->next=p->next;
			free(p);
			return(1);
		}
	return(0);
}

void canvas_add_event_function(f, a)
/* add f after a */
int (*f)();
int (*a)();
{
	eflist c, newf;

	if(a==NULL)
		c=eventfunctions;
	else
		for(c=eventfunctions;c->next!=NULL;c=c->next)
			if(c->f==a)
				break;
	
	if(c!=NULL)
	{
		newf=(eflist)malloc(sizeof(eflistd));
		newf->f=f;
		newf->next=c->next;
		c->next=newf;
	}
}

#ifdef DEBUG
pef()
{
	eflist t;
	for(t=eventfunctions->next;t!=NULL;t=t->next)
		printhandler(*t->f);
}
#endif
