#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/* hide.c:
 *
 * routines for hiding and showing nodes.  When you hide a node, you are
 * actually hiding all of its descendants.  When you hide a level, you are
 * hiding all descendants of nodes at that level.  This means that if you
 * hide level 2, all nodes that are more than 2 branches away from the root
 * node will be hidden.
 * i.e.:                       +---------+ node 1
 *            +----------------+
 *  +---------+                +--------+ node 5
 *  |         +------+ node 2
 *  + root
 *  |         +-------------+ node 3
 *  +---------+
 *            +--------------+ node 4
 *
 * hide_level(2) would make the tree appear like this:
 *                             
 *            +----------------+
 *  +---------+                
 *  |         +------+ node 2
 *  + root
 *  |         +-------------+ node 3
 *  +---------+
 *            +--------------+ node 4
 *
 * hide_level(1) would make the tree appear like this:
 *
 *  +---------+                
 *  |
 *  + root
 *  |
 *  +---------+
 *  
 *
 */

#include "interface.h"

#include "treedata.h"

int hide_node(n)
treenode n;
{
	list t;

	/* no sense in hiding a leaf, since there are no descendants */
	if(treeleaf(n))
		return(0);
	/* set the flag that says that this node has its children hidden */
	tndta(n, NoTreeType)->childrenhidden=1;
	/* set all descendants to be hidden (recursively) */
	tforsubtree(n, t)
	{
		tndta(subtree(t), NoTreeType)->hidden=1;
		hide_node(subtree(t));
	}
	/* invalidate formatting so that auto formatting can readjust */
	invalidate_refit(tree_get_tree(n), NoTreeType);
	invalidate_spacing(tree_get_tree(n), NoTreeType);
	/* specify that tree has been modified */
	modify();
	return(1);
}

int show_node(n)
	/* when you show a node, you are telling all of it's descendants to
	 * be visible.  It is an error to show a node that is not visible, and
	 * treetool won't even let you try
	 */
treenode n;
{
	list t;

	/* children are no longer hidden, so adjust flag */
	tndta(n, NoTreeType)->childrenhidden=0;
	/* set all descendants to be visible */
	tforsubtree(n, t)
	{
		tndta(subtree(t), NoTreeType)->hidden=0;
		show_node(subtree(t));
	}
	/* invalidate format and specify that tree has been modified */
	invalidate_refit(tree_get_tree(n), NoTreeType);
	invalidate_spacing(tree_get_tree(n), NoTreeType);
	modify();
	return(1);
}

int find_selected_level(n, l)
	/* finds the index of the highest level in the tree that has a selected
	 * node.  The root node is level 0, and level numbers increase as you
	 * traverse branches towards the nodes */
treenode n;
int l;
{
	int tl, tmp;
	list t;

	if(ndta(n)->selected)
		return(l);
	else
	{
		tl=-1;
		/* recursively, for each subtree, find the highest selected level,
		 * and return it */
		tforsubtree(n, t)
		{
			tmp=find_selected_level(subtree(t), l+1);
			if(tmp!=-1)
			{
				if(tl==-1)
					tl=tmp;
				else if(tl>tmp)
					tl=tmp;
			}
		}
		return(tl);
	}
}

void hide_selected_level(n, l, i)
	/* recursively goes down all branches from the root until it hits a node
	 * at level l, and then hides that node (all its descendants)
	 */
treenode n;
int l, i;
{
	list t;

	if(l==i)
		hide_node(n);
	else
		tforsubtree(n, t)
			hide_selected_level(subtree(t), l, i+1);
}

int hide_level(t)
	/* hides all the nodes at or below the level of the current selection */
tree t;
{
	int l;

	/* look for the level that is selected */
	l=find_selected_level(treeroot(t), 0);
	if(l==-1)
		return(0);
	/* if there is one, start hiding from there */
	hide_selected_level(treeroot(t), l, 0);
	modify();
	return(1);
}
