/*  fixpos/posfix  Conversion functions for lats and lons  */

/*****************************************************************************\

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

\*****************************************************************************/



#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#define NINT(a)   ((a)<0.0 ? (int) ((a) - 0.5) : (int) ((a) + 0.5))

/***************************************************************************\

    Module :        fixpos

    Programmer :    Jan C. Depner

    Date :          04/25/94

    Purpose :       This routine converts decimal degrees to degrees, minutes,
                    seconds, and hemisphere in the form specified by 'form'.

    Arguments:      degs        -   decimal degrees
                    type        -   "lat" or "lon"
                    form        -   "hdms", "hdm", "hd", "dms", "dm", or "d"
                    deg         -   degrees
                    min         -   minutes
                    sec         -   seconds
                    hem         -   hemisphere
                    
    Return Value:   char *      -   string containing the position in 'form'
                                    format

\***************************************************************************/

char *fixpos (double degs, double *deg, double *min, double *sec, char *hem,
char *c_type, char *c_form)
{
    double          abs_degs;
    static char     string[132];
    char            type[5], form[5];
    unsigned        i;


    strcpy (type, c_type);
    strcpy (form, c_form);

    for (i = 0 ; i < strlen (type) ; i++) type[i] = tolower (type[i]);
    for (i = 0 ; i < strlen (form) ; i++) form[i] = tolower (form[i]);


    /*  Compute the degrees, minutes, seconds, and hemisphere.  */
    
    abs_degs = fabs (degs) * 3600.0;

    *sec = fmod (abs_degs, 60.0);
    abs_degs -= *sec;
    abs_degs /= 60.0;
    *min = fmod (abs_degs, 60.0);
    abs_degs -= *min;
    abs_degs /= 60.0;
    *deg = abs_degs;

    *deg = (int) (*deg + 0.00001);
    *min = (int) (*min + 0.00001);
    *sec += 0.000000001;

    if (*sec >= 60.0)
    {
        *min += 1.0;
        *sec -= 60.0;
    }

    i = NINT (*sec * 100.0);
    *sec = i / 100.0;

    if (*min >= 60.0)
    {
        *deg += 1.0;
        *min -= 60.0;
    }

    if (!strncmp (type, "lon", 3))
    {
        if (degs < 0.0)
        {
            *hem = 'W';
        }
        else
        {
            *hem = 'E';
        }
    }
    else
    {
        if (degs < 0.0)
        {
            *hem = 'S';
        }
        else
        {
            *hem = 'N';
        }
    }

    /*  Build the output string and reset the min, sec, and hem if needed.  */
    
    if (!strncmp (form, "hdms", 4))
    {
        sprintf (string, "%1c %d %02d %05.2lf", *hem, (int) *deg, (int) *min, 
            *sec);
    }
    else if (!strncmp (form, "hdm", 3))
    {
        *min += *sec / 60.0;
        *sec = 0.0;
        sprintf (string, "%1c %d %lf", *hem, (int) *deg, *min);
    }
    else if (!strncmp (form, "hd", 3))
    {
        *deg += *min / 60.0 + *sec / 3600.0;
        *min = 0.0;
        *sec = 0.0;
        sprintf (string, "%1c %lf", *hem, *deg);
    }
    else if (!strncmp (form, "dms", 3))
    {
        if (degs < 0.0)
        {
            *hem = '-';
        }
        else
        {
            *hem = ' ';
        }
        sprintf (string, "%1c %d %02d %05.2lf", *hem, (int) *deg, (int) *min, *sec);

        if (degs < 0.0)
        {
            if ((int) *deg)
            {
                *deg = -*deg;
            }
            else if ((int) *min)
            {
                *min = -*min;
            }
            else
            {
                *sec = -*sec;
            }
        }
    }
    else if (!strncmp (form, "dm", 3))
    {
        if (degs < 0.0)
        {
            *hem = '-';
        }
        else
        {
            *hem = ' ';
        }
        *min += *sec / 60.0;
        *sec = 0.0;
        sprintf (string, "%1c %d %lf", *hem, (int) *deg, *min + *sec / 60.0);

        if (degs < 0.0)
        {
            if ((int) *deg)
            {
                *deg = -*deg;
            }
            else
            {
                *min = -*min;
            }
        }
    }
    else if (!strncmp (form, "d", 3))
    {
        if (degs < 0.0)
        {
            *hem = '-';
        }
        else
        {
            *hem = ' ';
        }
        sprintf (string, "%1c %lf", *hem, fabs (degs));

        *deg = degs;
        *min = 0.0;
        *sec = 0.0;
    }

    return (string);
}





/***************************************************************************\

    Module :        posfix

    Programmer :    Jan C. Depner

    Date :          04/25/94

    Purpose :       This routine converts a string containing the latitude or
                    longitude in any of the following formats to decimal
                    degrees.

                        Hemisphere Degrees Minutes Seconds.decimal
                        Hemisphere Degrees Minutes.decimal
                        Hemisphere Degrees.decimal
                        SignDegrees Minutes Seconds.decimal
                        SignDegrees Minutes.decimal
                        SignDegrees.decimal

    Arguments:      *string     -   character string
                    *degs       -   degrees decimal
                    type        -   "lat" or "lon"
                    
    Return Value:   None        -   string containing the position in 'form'
                                    format

\***************************************************************************/

void posfix (char *string, double *degs, char *c_type)
{
    unsigned    i, sign;
    double      fdeg, fmin, fsec, f1, f2, f3;
    char        type[5];


    strcpy (type, c_type);

    
    for (i = 0 ; i < strlen (type) ; i++) type[i] = tolower (type[i]);
    
    /*  Check for and clear sign or hemisphere indicators.          */
        
    sign = 0;
    for (i = 0 ; i < strlen (string) ; i++)
    {
        if (!strncmp (type, "lon", 3))
        {
            if (string[i] == 'W' || string[i] == 'w' || string[i] == '-')
            {
                string[i] = ' ';
                sign = 1;
            }
        }
        else
        {
            if (string[i] == 'S' || string[i] == 's' || string[i] == '-')
            {
                string[i] = ' ';
                sign = 1;
            }
        }

        if (string[i] == 'n' || string[i] == 'N' || string[i] == 'e' ||
            string[i] == 'E' || string[i] == '+') string[i] = ' ';
    }

    fdeg = 0.0;
    fmin = 0.0;
    fsec = 0.0;
    f1 = 0.0;
    f2 = 0.0;
    f3 = 0.0;

    /*  Convert the string to degrees, minutes, and seconds.        */
    
    i = sscanf (string, "%lf %lf %lf", &f1, &f2, &f3);

    /*  Based on the number of values scanned, compute the total    */
    /*  degrees.                                                    */
    
    switch (i)
    {
        case 3:
            fsec = f3 / 3600.0;

        case 2:
            fmin = f2 / 60.0;

        case 1:
            fdeg = f1;
    }

    fdeg += fmin + fsec;
    *degs = fdeg;
    if (sign) *degs = - *degs;
}

