///////////////////////////////////////////////////////////////////////////////
//
// Wall Paper Changer for GNOME
// Copyright (C) 2000-2001  Kenichi Shibata <ken_s@cat.zero.ad.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// Please read the file "COPYING" for more information.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////
//
// ken_s@cat.zero.ad.jp
// http://www3.might.co.jp/~sibata/top/
//
///////////////////////////////////////////////////////////////////////////////

#define __WPAC_H
#include "global.h"

// static
static WPAC_DATA wpac;
static WPAC_DATA wpac_tmp;


////////////////////////////////////////////////////////
//                                                    //
//  Set Change mode                                   //
//                                                    //
//    Parameter                                       //
//       mode : TRUE  = Auto                          //
//              FALSE = Manual                        //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_set_changeMode(gboolean mode)
{
  wpac_tmp.changeMode = mode;
}

////////////////////////////////////////////////////////
//                                                    //
//  Set Random mode                                   //
//                                                    //
//    Parameter                                       //
//       random : TRUE  = Random                      //
//                FALSE = Sequential                  //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_set_random(gboolean random)
{
  wpac_tmp.random = random;
}

////////////////////////////////////////////////////////
//                                                    //
//  Set timeout time                                  //
//                                                    //
//    Parameter                                       //
//       time : T.O. minute                           //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_set_timeout(gulong time)
{
  wpac_tmp.timeout = time;
}

////////////////////////////////////////////////////////
//                                                    //
//  Get change mode                                   //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = Auto                      //
//                  FALSE = Manual                    //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_get_changeMode(void)
{
  return (wpac_tmp.changeMode);
}

////////////////////////////////////////////////////////
//                                                    //
//  Get random mode                                   //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = random                    //
//                  FALSE = sequential                //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_get_random(void)
{
  return (wpac_tmp.random);
}

////////////////////////////////////////////////////////
//                                                    //
//  Get timeout minute                                //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gulong : minute                              //
//                                                    //
////////////////////////////////////////////////////////
gulong wpac_get_timeout(void)
{
  return (wpac_tmp.timeout);
}

////////////////////////////////////////////////////////
//                                                    //
//  Get Wallpaper data list                           //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       GList* : list pointer                        //
//                                                    //
////////////////////////////////////////////////////////
GList *wpac_get_wallPaper(void)
{
  return (wpac_tmp.wallPaper);
}

////////////////////////////////////////////////////////
//                                                    //
//  Find wallpaper data                               //
//                                                    //
//    Parameter                                       //
//       filename : wall paper file name              //
//    Return                                          //
//       WALL_PAPER* : find wallpaper data pointer    //
//                                                    //
////////////////////////////////////////////////////////
WALL_PAPER *wpac_find_wallPaper(gchar *filename)
{
  GList *wallPaper;
  WALL_PAPER *data;

  wallPaper = wpac_tmp.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if (strcmp(data->fileName, filename) == 0)
      return ((WALL_PAPER *)(wallPaper->data));
    wallPaper = g_list_next(wallPaper);
  }
  return (NULL);
}

////////////////////////////////////////////////////////
//                                                    //
//  Add wallpaper data                                //
//                                                    //
//    Parameter                                       //
//       add_data : add Wallpaper data                //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_add_image(WALL_PAPER *add_data)
{
  GList *wallPaper;
  WALL_PAPER *data;

  wallPaper = wpac_tmp.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if (strcmp(data->fileName, add_data->fileName) == 0) {
      data->layout = add_data->layout;
      data->dither = add_data->dither;
      data->gray = add_data->gray;
      data->zoom_x_eq_y = add_data->zoom_x_eq_y;
      data->zoom_x = add_data->zoom_x;
      data->zoom_y = add_data->zoom_y;
      data->rotate = add_data->rotate;
      data->gamma = add_data->gamma;
      data->brighten = add_data->brighten;
      data->bg_color[0] = add_data->bg_color[0];
      data->bg_color[1] = add_data->bg_color[1];
      data->bg_color[2] = add_data->bg_color[2];
      if (data->option)
	g_free(data->option);
      if ((add_data->option) && (strlen(add_data->option) > 0)) {
	data->option = (gchar *)g_malloc(strlen(add_data->option) + 1);
	strcpy(data->option, add_data->option);
      } else {
	data->option = (gchar *)NULL;
      }
      wpac_data_check(data);
      return;
    }
    wallPaper = g_list_next(wallPaper);
  }
  data = (WALL_PAPER *)g_malloc(sizeof(WALL_PAPER));
  data->fileName = (gchar *)g_malloc(strlen(add_data->fileName) + 1);
  strcpy(data->fileName, add_data->fileName);
  data->layout = add_data->layout;
  data->dither = add_data->dither;
  data->gray = add_data->gray;
  data->zoom_x_eq_y = add_data->zoom_x_eq_y;
  data->zoom_x = add_data->zoom_x;
  data->zoom_y = add_data->zoom_y;
  data->rotate = add_data->rotate;
  data->gamma = add_data->gamma;
  data->brighten = add_data->brighten;
  data->bg_color[0] = add_data->bg_color[0];
  data->bg_color[1] = add_data->bg_color[1];
  data->bg_color[2] = add_data->bg_color[2];
  if ((add_data->option) && (strlen(add_data->option) > 0)) {
    data->option = (gchar *)g_malloc(strlen(add_data->option) + 1);
    strcpy(data->option, add_data->option);
  } else {
    data->option = (gchar *)NULL;
  } 
  data->used = FALSE;
  wpac_data_check(data);
  wpac_tmp.wallPaper = g_list_append(wpac_tmp.wallPaper, (gpointer)data);
  wpac_tmp.wallPaper = g_list_sort(wpac_tmp.wallPaper, wpac_data_comp);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall paper data compare                           //
//                                                    //
//    Parameter                                       //
//       a : compare data                             //
//       b : compare data                             //
//    Return                                          //
//       gint : 0  = (a == b)                         //
//              0 != (a != b)                         //
//                                                    //
////////////////////////////////////////////////////////
static gint wpac_data_comp(gconstpointer a, gconstpointer b)
{
  gchar *fn1, *fn2;
  gint  len1, len2;
  WALL_PAPER *w1, *w2;

  w1 = (WALL_PAPER *)a;
  w2 = (WALL_PAPER *)b;
  fn1 = w1->fileName;
  fn2 = w2->fileName;
  while ((*fn1) && (*fn2)) {
    if ((*fn1) != (*fn2))
      return ((gint)((*fn2) - (*fn1)));
    fn1++;
    fn2++;
  }
  
  fn1 = w1->fileName;
  fn2 = w2->fileName;
  len1 = strlen(fn1);
  len2 = strlen(fn2);
  if (len1 != len2) {
    return (len2 - len1);
  }
  return (0);
}



////////////////////////////////////////////////////////
//                                                    //
//  Delete Wallpaper data                             //
//                                                    //
//    Parameter                                       //
//       filename : wallpaper file name               //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_del_image(gchar *filename)
{
  GList *wallPaper;
  WALL_PAPER *data;

  wallPaper = wpac_tmp.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if (strcmp(data->fileName, filename) == 0) {
      wpac_tmp.wallPaper = g_list_remove(wpac_tmp.wallPaper, wallPaper->data);
      g_free(data->fileName);
      if (data->option)
	g_free(data->option);
      g_free(data);
      return;
    }
    wallPaper = g_list_next(wallPaper);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Wallpaper data copy                               //
//                                                    //
//    Parameter                                       //
//       dst : distnation data                        //
//       src : source data                            //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void wpac_copy(WPAC_DATA *dst, WPAC_DATA *src)
{
  GList *src_glist, *dst_glist;
  WALL_PAPER *src_data, *dst_data;
  
  dst->changeMode = src->changeMode;
  dst->random = src->random;
  dst->timeout = src->timeout;
  src_glist = src->wallPaper;

  dst_glist = (GList *)NULL;
  while (src_glist) {
    src_data = (WALL_PAPER *)(src_glist->data);
    dst_data = (WALL_PAPER *)g_malloc(sizeof(WALL_PAPER));
    dst_data->fileName = (gchar *)g_malloc(strlen(src_data->fileName) + 1);
    strcpy(dst_data->fileName, src_data->fileName);
    dst_data->layout = src_data->layout;
    dst_data->dither = src_data->dither;
    dst_data->gray = src_data->gray;
    dst_data->zoom_x_eq_y = src_data->zoom_x_eq_y;
    dst_data->zoom_x = src_data->zoom_x;
    dst_data->zoom_y = src_data->zoom_y;
    dst_data->rotate = src_data->rotate;
    dst_data->gamma = src_data->gamma;
    dst_data->brighten = src_data->brighten;
    dst_data->bg_color[0] = src_data->bg_color[0];
    dst_data->bg_color[1] = src_data->bg_color[1];
    dst_data->bg_color[2] = src_data->bg_color[2];
    if (src_data->option) {
      dst_data->option = (gchar *)g_malloc(strlen(src_data->option) + 1);
      strcpy(dst_data->option, src_data->option);
    } else {
      dst_data->option = (gchar *)NULL;
    }
    dst_data->used = FALSE;
    wpac_data_check(dst_data);
    dst_glist = g_list_append(dst_glist, dst_data);
    src_glist = g_list_next(src_glist);
  }
  dst->wallPaper = dst_glist;
}

////////////////////////////////////////////////////////
//                                                    //
//  Wallpaper data All Free                           //
//                                                    //
//    Parameter                                       //
//       wpac_data : Wallpaper data                   //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void wpac_allFree(WPAC_DATA *wpac_data)
{
  GList *glist;
  WALL_PAPER *data;

  glist = wpac_data->wallPaper;
  while (glist) {
    data = (WALL_PAPER *)(glist->data);
    g_free(data->fileName);
    if (data->option)
      g_free(data->option);
    g_free(data);
    glist = g_list_next(glist);
  }
  g_list_free(wpac_data->wallPaper);
}

////////////////////////////////////////////////////////
//                                                    //
//  Initial Temporary Data                            //
//                                                    //
//    Parameter                                       //
//       argc : Command line Parameter count          //
//       argv : Command line parameter                //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_init_editData(void)
{
  wpac_copy(&wpac_tmp, &wpac);
}

////////////////////////////////////////////////////////
//                                                    //
//  Quit wapc                                         //
//                                                    //
//    Parameter                                       //
//       argc : Command line Parameter count          //
//       argv : Command line parameter                //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void wpac_quit(void)
{
  wpac_allFree(&wpac);
}

////////////////////////////////////////////////////////
//                                                    //
//  Load wallpaper data from file                     //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = OK                        //
//                  FALSE = NG                        //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_load_file(void)
{
  gchar dataFileName[1024], buff[2048], *datap;
  FILE *fp;
  gint section;
  WALL_PAPER wallpaper;

  // Default
  wpac_tmp.wallPaper = (GList *)NULL;
  wpac_tmp.changeMode = TRUE;
  wpac_tmp.random = FALSE;
  wpac_tmp.timeout = 1;

  wallpaper.fileName = NULL;
  wallpaper.option = NULL;
  section = -1;

  // file open
  strcpy(dataFileName, g_get_home_dir());
  strcat(dataFileName, "/.wpac/wallpaper.dat");

  if ((fp = fopen(dataFileName, "r")) == NULL) {
    return (FALSE);
  }

  // file read
  while (fgets(buff, 2048, fp) != NULL) {

    datap = buff;
    while ((*datap) && (*datap != '\n')) {
      datap++;
    }
    *datap = (gchar)NULL;

    if (strncmp(buff, "[common]", strlen("[common]")) == 0) {
      section = 0;
      continue;
    } else if (strncmp(buff, "[image file]", strlen("[image file]")) == 0) {
      section = 1;
      continue;
    }

    datap = wpac_seek_data(buff);
    switch (section) {
    // common section
    case 0:
      if (strncmp(buff, "<change mode>", strlen("<change mode>")) == 0) {
	if (strncmp(datap, "manual", strlen("manual")) == 0) {
	  wpac_tmp.changeMode = FALSE;
	} else {
	  wpac_tmp.changeMode = TRUE;
	}
      } else if (strncmp(buff, "<change time>", strlen("<change time>")) == 0) {
	  wpac_tmp.timeout = d_to_gulong(datap);
      } else if (strncmp(buff, "<random>", strlen("<random>")) == 0) {
	if (strncmp(datap, "yes", strlen("yes")) == 0) {
	  wpac_tmp.random = TRUE;
	} else {
	  wpac_tmp.random = FALSE;
	}
      }
      break;
    // file image section
    case 1:
      // filename
      if (strncmp(buff, "<file name>", strlen("<file name>")) == 0) {
	wallpaper.fileName = (gchar *)g_malloc(strlen(datap) + 1);
	strcpy(wallpaper.fileName, datap);
      // layout
      } else if (strncmp(buff, "<layout>", strlen("<layout>")) == 0) {
	if (strncmp(datap, "center", strlen("center")) == 0)
	  wallpaper.layout = LAYOUT_CENTER;
	else if (strncmp(datap, "fullscreen", strlen("fullscreen")) == 0)
	  wallpaper.layout = LAYOUT_FULLSCREEN;
	else
	  wallpaper.layout = LAYOUT_TILE;
      // dither
      } else if (strncmp(buff, "<dither>", strlen("<dither>")) == 0) {
	if (strncmp(datap, "no", strlen("no")) == 0)
	  wallpaper.dither = FALSE;
	else
	  wallpaper.dither = TRUE;
      // gray
      } else if (strncmp(buff, "<gray>", strlen("<gray>")) == 0) {
	if (strncmp(datap, "no", strlen("no")) == 0)
	  wallpaper.gray = FALSE;
	else
	  wallpaper.gray = TRUE;
      // X == Y
      } else if (strncmp(buff, "<zoom x eq y>", strlen("<zoom x eq y>")) == 0) {
	if (strncmp(datap, "yes", strlen("yes")) == 0)
	  wallpaper.zoom_x_eq_y = TRUE;
	else
	  wallpaper.zoom_x_eq_y = FALSE;
      // zoom x
      } else if (strncmp(buff, "<zoom x>", strlen("<zoom x>")) == 0) {
	wallpaper.zoom_x = (guchar)d_to_gulong(datap);
      // zoom y
      } else if (strncmp(buff, "<zoom y>", strlen("<zoom y>")) == 0) {
	wallpaper.zoom_y = (guchar)d_to_gulong(datap);
      // rotate
      } else if (strncmp(buff, "<rotate>", strlen("<rotate>")) == 0) {
	wallpaper.rotate = (gushort)d_to_gulong(datap);
      // gamma
      } else if (strncmp(buff, "<gamma>", strlen("<gamma>")) == 0) {
	wallpaper.gamma = (guchar)d_to_gulong(datap);
      // brighten
      } else if (strncmp(buff, "<brighten>", strlen("<brighten>")) == 0) {
	wallpaper.brighten = (gushort)d_to_gulong(datap);
      // background color
      } else if (strncmp(buff, "<background color>", strlen("<background color>")) == 0) {
	sscanf(datap, "%lf/%lf/%lf", &wallpaper.bg_color[0], &wallpaper.bg_color[1], &wallpaper.bg_color[2]);
      // option
      } else if (strncmp(buff, "<option>", strlen("<option>")) == 0) {
	wallpaper.option = (gchar *)g_malloc(strlen(datap) + 1);
	strcpy(wallpaper.option, datap);
      // start data
      } else if (strncmp(buff, "{", strlen("{")) == 0) {
	continue;
      // end data
      } else if (strncmp(buff, "}", strlen("}")) == 0) {
	if (wallpaper.fileName != NULL) {
	  wpac_add_image(&wallpaper);
	  g_free(wallpaper.fileName);
	  if (wallpaper.option)
	    g_free(wallpaper.option);
	  wallpaper.fileName = NULL;
	  wallpaper.option = NULL;
	}
      }
      break;
    default:
      break;
    }
  }
  
  fclose(fp);

  if (wallpaper.fileName != NULL) {
    g_free(wallpaper.fileName);
  }

  wpac_tmp.wallPaper = g_list_sort(wpac_tmp.wallPaper, wpac_data_comp);
  wpac_copy(&wpac, &wpac_tmp);
  wpac_allFree(&wpac_tmp);
  wpac_init_used_flag();
  return (TRUE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wallpaper data limit check                        //
//                                                    //
//    Parameter                                       //
//       WALL_PAPER* : wallpaper data                 //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void wpac_data_check(WALL_PAPER *data)
{
  gint i;

  if (data->layout > LAYOUT_TILE)
    data->layout = LAYOUT_TILE;

  if (data->zoom_x < 1)
    data->zoom_x = 1;

  if (data->zoom_x > 200)
    data->zoom_x = 100;

  if (data->zoom_x_eq_y == TRUE)
    data->zoom_x = data->zoom_y;

  switch (data->rotate) {
  case 0:
  case 90:
  case 180:
  case 270:
    break;
  default:
    data->rotate = 0;
    break;
  }

  if (data->gamma > 100)
    data->gamma = 10;

  if (data->brighten > 300)
    data->brighten = 100;

  for (i=0; i<3; i++) {
    if (data->bg_color[i] > 1.0)
      data->bg_color[i] = 1.0;
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Wallpaper file data seek                          //
//                                                    //
//    Parameter                                       //
//       str : seek data                              //
//    Return                                          //
//       gchar : data string pointer                  //
//                                                    //
////////////////////////////////////////////////////////
static gchar *wpac_seek_data(gchar *str)
{
  while ((*str) && (*str != '='))
    str++;
  str++;
  while (*str == ' ')
    str++;
  return (str);
}


////////////////////////////////////////////////////////
//                                                    //
//  Save Wallpaper data to file                       //
//                                                    //
//    Parameter                                       //
//       tmpFree : Temporary data Free                //
//    Return                                          //
//       gboolean : TRUE  = OK                        //
//                  FALSE = NG                        //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_save_file(gboolean tmpFree)
{
  DIR *datadir;
  gchar dataFileName[1024];
  GList *wallPaper;
  WALL_PAPER *data;
  FILE *fp;

  wpac_allFree(&wpac);
  wpac_copy(&wpac, &wpac_tmp);
  if (tmpFree == TRUE)
    wpac_allFree(&wpac_tmp);
  wpac_init_used_flag();

  // make data directory
  strcpy(dataFileName, g_get_home_dir());
  strcat(dataFileName, "/.wpac");
  if ((datadir = opendir(dataFileName)) == NULL)
    mkdir(dataFileName, S_IFDIR | 0700);
  else
    closedir(datadir);

  // data file open
  strcat(dataFileName, "/wallpaper.dat");
  if ((fp = fopen(dataFileName, "w")) == NULL)
    return (FALSE);
  
  fprintf(fp, "[common]\n");

  // change mode
  fprintf(fp, "<change mode> = ");
  if (wpac.changeMode == TRUE)
    fprintf(fp, "auto\n");
  else
    fprintf(fp, "manual\n");

  // change time
  fprintf(fp, "<change time> = %ld\n", wpac.timeout);

  // random
  fprintf(fp, "<random> = ");
  if (wpac.random == TRUE)
    fprintf(fp, "yes\n");
  else
    fprintf(fp, "no\n");

  fprintf(fp, "[image file]\n");
  wallPaper = wpac.wallPaper;
  while (wallPaper) {
    fprintf(fp, "{\n");
    data = (WALL_PAPER *)(wallPaper->data);
    wpac_data_check(data);

    // filename
    fprintf(fp, "<file name> = %s\n", data->fileName);

    // layout
    fprintf(fp, "<layout> = ");
    switch (data->layout) {
    case LAYOUT_CENTER:
      fprintf(fp, "center\n");
      break;
    case LAYOUT_FULLSCREEN:
      fprintf(fp, "fullscreen\n");
      break;
    case LAYOUT_TILE:
    default:
      fprintf(fp, "tile\n");
      break;
    }

    // dither
    fprintf(fp, "<dither> = ");
    if (data->dither == TRUE)
      fprintf(fp, "yes\n");
    else
      fprintf(fp, "no\n");

    // gray
    fprintf(fp, "<gray> = ");
    if (data->gray == TRUE)
      fprintf(fp, "yes\n");
    else
      fprintf(fp, "no\n");

    // zoom_x_eq_y
    fprintf(fp, "<zoom x eq y> = ");
    if (data->zoom_x_eq_y == TRUE)
      fprintf(fp, "yes\n");
    else
      fprintf(fp, "no\n");

    // zoom_x
    fprintf(fp, "<zoom x> = %d\n", data->zoom_x);

    // zoom_y
    fprintf(fp, "<zoom y> = %d\n", data->zoom_y);

    // rotate
    fprintf(fp, "<rotate> = %d\n", data->rotate);

    // gamma
    fprintf(fp, "<gamma> = %d\n", data->gamma);

    // brighten
    fprintf(fp, "<brighten> = %d\n", data->brighten);

    // background color
    fprintf(fp, "<background color> = %f/%f/%f\n", data->bg_color[0], data->bg_color[1], data->bg_color[2]);

    // option
    if (data->option)
      fprintf(fp, "<option> = %s\n", data->option);

    fprintf(fp, "}\n\n");
    wallPaper = g_list_next(wallPaper);
  }
  fclose(fp);
  return (TRUE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Change wallpaper                                  //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = OK                        //
//                  FALSE = NG                        //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_change_wallpaper(void)
{
  GList *wallPaper;

  if (wpac_check_used_flag() == FALSE)
    wpac_init_used_flag();
  if (wpac.random == TRUE)
    wallPaper = wpac_get_wallpaper_random();
  else
    wallPaper = wpac_get_wallpaper_normal();
  if (wallPaper) {
    return (wpac_xloadimage((WALL_PAPER *)(wallPaper->data)));
  } else {
    return (FALSE);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Get wallpaper (random)                            //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       GList* : wallpaper data                      //
//                                                    //
////////////////////////////////////////////////////////
static GList *wpac_get_wallpaper_random(void)
{
  GList *wallPaper;
  GTimeVal currentTime;
  guint length, index;
  WALL_PAPER *data;
 
  if (wpac.wallPaper == NULL)
     return ((GList *)NULL);
  length = g_list_length(wpac.wallPaper);
  g_get_current_time(&currentTime);
  index = (guint)(currentTime.tv_sec) % length;
  wallPaper = g_list_nth(wpac.wallPaper, index);

  // search (index -> end)
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if ((data->used) == FALSE) {
      data->used = TRUE;
      return (wallPaper);
    }
    wallPaper = g_list_next(wallPaper);
    }

  // search (all)
  wallPaper = wpac.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if ((data->used) == FALSE) {
      data->used = TRUE;
      return (wallPaper);
    }
    wallPaper = g_list_next(wallPaper);
  }
  return ((GList *)NULL);
}

////////////////////////////////////////////////////////
//                                                    //
//  Get wallpaper (sequential)                        //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       GList* : wallpaper data                      //
//                                                    //
////////////////////////////////////////////////////////
static GList *wpac_get_wallpaper_normal(void)
{
  GList *wallPaper;
  WALL_PAPER *data;

  // search (all)
  wallPaper = wpac.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if ((data->used) == FALSE) {
      data->used = TRUE;
      return (wallPaper);
    }
    wallPaper = g_list_next(wallPaper);
  }
  return ((GList *)NULL);
}

////////////////////////////////////////////////////////
//                                                    //
//  Initial used flag                                 //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void wpac_init_used_flag(void)
{
  GList *wallPaper;
  WALL_PAPER *data;

  wallPaper = wpac.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    data->used = FALSE;
    wallPaper = g_list_next(wallPaper);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  check used flag                                   //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = not all used              //
//                  FALSE = all used                  //
//                                                    //
////////////////////////////////////////////////////////
static gboolean wpac_check_used_flag(void)
{
  GList *wallPaper;
  WALL_PAPER *data;

  wallPaper = wpac.wallPaper;
  while (wallPaper) {
    data = (WALL_PAPER *)(wallPaper->data);
    if (data->used == FALSE)
      return (TRUE);
    wallPaper = g_list_next(wallPaper);
  }
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  xloadimage run                                    //
//                                                    //
//    Parameter                                       //
//       data : wallpaper data                        //
//    Return                                          //
//       gboolean : TRUE  = OK                        //
//                  FALSE = NG                        //
//                                                    //
////////////////////////////////////////////////////////
gboolean wpac_xloadimage(WALL_PAPER *data)
{
  gchar buff[1024];
  guchar red, green, blue;
  static gchar command[] = "xloadimage ";
  static gchar opt_onroot[] = "-onroot ";
  static gchar opt_layout[3][13] = {
    "-center ", "-fullscreen ", " "
  };
  static gchar opt_dither[] = "-dither ";
  static gchar opt_gray[] = "-gray ";
  static gchar opt_zooom[] = "-zoom ";
  static gchar opt_xzoom[] = "-xzoom ";
  static gchar opt_yzoom[] = "-yzoom ";
  static gchar opt_rotate[] = "-rotate ";
  static gchar opt_gamma[] = "-gamma ";
  static gchar opt_brighten[] = "-brighten ";
  static gchar opt_bg_color[] = "-border rgb:";

  strcpy(buff, command);
  strcat(buff, opt_onroot);

  strcat(buff, opt_layout[data->layout]);

  if (data->dither == TRUE)
    strcat(buff, opt_dither);

  if (data->gray == TRUE)
    strcat(buff, opt_gray);

  if (data->layout != LAYOUT_FULLSCREEN) {
    if (data->zoom_x_eq_y == TRUE) {
      if (data->zoom_x != 100) {
	strcat(buff, opt_zooom);
	sprintf(buff, "%s%d ", buff, data->zoom_x);
      }
    } else {
      if (data->zoom_x != 100) {
	strcat(buff, opt_xzoom);
	sprintf(buff, "%s%d ", buff, data->zoom_x);
      }
      if (data->zoom_y != 100) {
	strcat(buff, opt_yzoom);
	sprintf(buff, "%s%d ", buff, data->zoom_y);
      }
    }
  }

  if (data->rotate != 0) {
    strcat(buff, opt_rotate);
    sprintf(buff, "%s%d ", buff, data->rotate);
  }

  if (data->gamma != 10) {
    strcat(buff, opt_gamma);
    sprintf(buff, "%s%d.%d ", buff, (data->gamma) / 10, (data->gamma) % 10);
  }
 
  if (data->brighten != 100) {
    strcat(buff, opt_brighten); 
    sprintf(buff, "%s%d ", buff, data->brighten);
  }

  if (data->layout != LAYOUT_TILE) {
    strcat(buff, opt_bg_color);
    red   = (guchar)(data->bg_color[0] * 256.0);
    green = (guchar)(data->bg_color[1] * 256.0);
    blue  = (guchar)(data->bg_color[2] * 256.0);
    sprintf(buff, "%s%02x/%02x/%02x ", buff, red, green, blue);
  }

  if (data->option) {
    sprintf(buff, "%s %s ", buff, data->option);
  }

  strcat(buff, data->fileName);
  //strcat(buff, " > /dev/null");
  printf("\n%s\n", buff);
  system(buff);
  return (TRUE);
}

