/* awindow.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: awindow.cc,v 1.16 2002/03/18 01:14:13 ralf Exp $ */

#include "awindow.h"

#include "text.h"
#include "button.h"

AWindow::AWindow(AGUIX *parent)
{
  this->aguix=parent;
  title=NULL;
  name=NULL;
  iconname=NULL;
  gui=new List();
  subwins=new List();
  subawins=new List();
  onScreen=false;
  parentawindow=NULL;
  focusOwner=NULL;
  border = 5;
}

int AWindow::create(AWindow *parent,int tx,int ty,int width,int height,int tbg,const char *ttitle)
{
  Display *dsp=aguix->getDisplay();
  Window ParentWnd;
  XTextProperty windowname,ticonname;
  int scr=aguix->getScreen();
  parentawindow=parent;
  if(parent==NULL) {
    ParentWnd=RootWindow(dsp,scr);
  } else {
    ParentWnd=parent->win;
  }
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask|CWBackPixel|CWColormap;
  XSetWindowAttributes attr;
  attr.event_mask=ExposureMask|ButtonPressMask|ButtonReleaseMask|StructureNotifyMask|KeyPressMask|KeyReleaseMask|ButtonMotionMask|PointerMotionHintMask;
  attr.background_pixel=aguix->getPixel(tbg);
  attr.colormap=aguix->getColormap();
  win=XCreateWindow(dsp,ParentWnd,tx,ty,width,height,0,aguix->getDepth(),InputOutput,vis,mask,&attr);
  if(!win) {
    return 1;
  }

  this->name=dupstring(ttitle);
  this->title=dupstring(ttitle);
  this->iconname=dupstring(ttitle);

  XClassHint classhint;
  classhint.res_name=dupstring(ttitle);
  classhint.res_class=aguix->getClassname();
  XSetClassHint(dsp,win,&classhint);
  _freesafe(classhint.res_class);
  _freesafe(classhint.res_name);
  XSetWMProtocols(dsp,win,aguix->getCloseAtom(),1);
  XSizeHints *SizeHints;
  XWMHints *WMHints;
  SizeHints=XAllocSizeHints();
  WMHints=XAllocWMHints();
  SizeHints->flags=PSize;
  SizeHints->width=width;
  SizeHints->height=height;
  WMHints->input=True;

  WMHints->window_group=aguix->getGroupWin();

  WMHints->flags=InputHint|WindowGroupHint;
  XStringListToTextProperty(&title,1,&windowname);
  XStringListToTextProperty(&title,1,&ticonname);
  XSetWMName(dsp,win,&windowname);
  XSetWMIconName(dsp,win,&ticonname);
  XSetWMHints(dsp,win,WMHints);
  XSetWMNormalHints(dsp,win,SizeHints);
  XStoreName(dsp,win,ttitle);
  XFree(SizeHints);
  XFree(WMHints);
  w=width;
  h=height;
  this->x=tx;
  this->y=ty;
  maxw=-1;
  maxh=-1;
  minw=-1;
  minh=-1;
  XClearWindow(dsp,win);
  XSetCommand(dsp,win,aguix->getargv(),aguix->getargc());
  /* jetzt this noch in aguix anhngen */
  aguix->insertWindow(this);
  if(parent!=NULL) {
    parent->addSubAWindow(this);
  }
  XFlush(dsp);
  this->bg=tbg;
  return 0;
}

AGUIX *AWindow::getAGUIX()
{
  return aguix;
}

void AWindow::sizeChanged(int width,int height)
{
  if((width!=w)||(height!=h)) {
    AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
    agmsg->type=AG_SIZECHANGED;
    agmsg->size.window=win;
    agmsg->size.neww=width;
    agmsg->size.newh=height;
    aguix->putAGMsg(agmsg);
    w=width;
    h=height;
  }
}

void AWindow::redraw()
{
  aguix->SetWindowBG(win,bg);
  Display *dsp=aguix->getDisplay();
  int scr=aguix->getScreen();
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask|CWBackPixel;
  XSetWindowAttributes attr;
  attr.event_mask=ExposureMask|ButtonPressMask|ButtonReleaseMask|StructureNotifyMask|KeyPressMask|KeyReleaseMask|ButtonMotionMask;
  attr.background_pixel=aguix->getPixel(bg);
  Window twin=XCreateWindow(dsp,win,0,0,w,h,0,aguix->getDepth(),InputOutput,vis,mask,&attr);
  if(!twin) {
    return;
  }
  XMapRaised(dsp,twin);
  XDestroyWindow(dsp,twin);
}

Window AWindow::getWindow()
{
  return win;
}

void AWindow::close()
{
  int id=gui->initEnum();
  GUIElement *telem=(GUIElement*)gui->getFirstElement(id);
  while(telem!=NULL) {
    delete telem;
    gui->removeElementAt(0);
    telem=(GUIElement*)gui->getFirstElement(id);
  }
  gui->closeEnum(id);
  Display *dsp=aguix->getDisplay();
  aguix->removeWindow(this);
  XUnmapWindow(dsp,win);
  XDestroyWindow(dsp,win);
  aguix->Flush();
  if(parentawindow!=NULL) parentawindow->removeSubAWindow(this);
}

AWindow::~AWindow()
{
  if(title!=NULL) _freesafe(title);
  if(name!=NULL) _freesafe(name);
  if(iconname!=NULL) _freesafe(iconname);
  int id=gui->initEnum();
  GUIElement *te=(GUIElement*)gui->getFirstElement(id);
  while(te!=NULL) {
    delete te;
    te=(GUIElement*)gui->getNextElement(id);
  }
  gui->closeEnum(id);
  delete gui;
  delete subwins;
  delete subawins;
}

int AWindow::getBG()
{
  return bg;
}

void AWindow::setBG(int tbg)
{
  this->bg=tbg;
}

void AWindow::ReactMessage(Message *msg)
{
  if((msg->window)!=win) return;
  // diese Message interressiert uns
}

GUIElement *AWindow::add(GUIElement *new_element)
{
  gui->addElement(new_element);
  new_element->setParent(this);
  return new_element;
}

void AWindow::setMaxSize(int mwidth,int mheight)
{
  XSizeHints *SizeHints;
  Display *dsp=aguix->getDisplay();
  SizeHints=XAllocSizeHints();
  if(minw>=0) {
    SizeHints->flags=PMaxSize|PMinSize;
    SizeHints->min_width=minw;
    SizeHints->min_height=minh;
  } else {
    SizeHints->flags=PMaxSize;
  }
  SizeHints->max_width=mwidth;
  SizeHints->max_height=mheight;
  maxw=mwidth;
  maxh=mheight;
  XSetWMNormalHints(dsp,win,SizeHints);
  XFree(SizeHints);
}

void AWindow::setMinSize(int mwidth,int mheight)
{
  XSizeHints *SizeHints;
  Display *dsp=aguix->getDisplay();
  SizeHints=XAllocSizeHints();
  if(maxw>=0) {
    SizeHints->flags=PMaxSize|PMinSize;
    SizeHints->max_width=maxw;
    SizeHints->max_height=maxh;
  } else {
    SizeHints->flags=PMinSize;
  }
  SizeHints->min_width=mwidth;
  SizeHints->min_height=mheight;
  minw=mwidth;
  minh=mheight;
  XSetWMNormalHints(dsp,win,SizeHints);
  XFree(SizeHints);
}

void AWindow::show()
{
  if(onScreen==true) return;
  Display *dsp=aguix->getDisplay();
  XMapWindow(dsp,win);
  onScreen=true;
}

void AWindow::hide()
{
  if(onScreen==false) return;
  Display *dsp=aguix->getDisplay();
  XUnmapWindow(dsp,win);
  onScreen=false;
}

void AWindow::hide(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XUnmapWindow(dsp,subwin);
  }
}

void AWindow::show(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XMapWindow(dsp,subwin);
  }
}

const char *AWindow::getTitle()
{
  return title;
}

void AWindow::setTitle(const char *newtitle)
{
  if(title!=NULL) _freesafe(title);
  title=dupstring(newtitle);
  Display *dsp=aguix->getDisplay();
  XClassHint classhint;
  classhint.res_name=dupstring(title);
  classhint.res_class=aguix->getClassname();
  XSetClassHint(dsp,win,&classhint);
  _freesafe(classhint.res_class);
  _freesafe(classhint.res_name);
  XTextProperty windowname,ticonname;
  XStringListToTextProperty(&title,1,&windowname);
  XStringListToTextProperty(&title,1,&ticonname);
  XSetWMName(dsp,win,&windowname);
  XSetWMIconName(dsp,win,&ticonname);
  XStoreName(dsp,win,title);
}

void AWindow::resizeSubWin(Window twin,int tw,int th)
{
  Display *dsp=aguix->getDisplay();
  XResizeWindow(dsp,twin,tw,th);
}

void AWindow::moveSubWin(Window twin,int tx,int ty)
{
  Display *dsp=aguix->getDisplay();
  XMoveWindow(dsp,twin,tx,ty);
}

bool AWindow::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue=false;
  if(msg->window==win) {
    if(msg->type==ConfigureNotify) {
      sizeChanged(E->xconfigure.width,E->xconfigure.height);
    }
  }
  // Alle Element bekommen Nachricht
  int id=gui->initEnum();
  GUIElement *guie=(GUIElement*)gui->getFirstElement(id);
  while((guie!=NULL)&&(returnvalue==false)) {
    returnvalue=guie->handleMessage(E,msg);
    guie=(GUIElement*)gui->getNextElement(id);
  }
  gui->closeEnum(id);
  return returnvalue;
}

bool AWindow::hasSubWin(Window twin)
{
  int pos=subwins->getIndex((void*)twin);
  if(pos==-1) return false;
  else return true;
}

Window AWindow::getSubWindow(Window parent,int tx,int ty,int tw,int th)
{
  Window newwin,pwin;
  Display *dsp=aguix->getDisplay();
  int scr=aguix->getScreen();
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask|CWBackPixel|CWColormap;

  XSetWindowAttributes attr;
  attr.event_mask=ExposureMask|ButtonPressMask|ButtonReleaseMask|StructureNotifyMask|KeyPressMask|KeyReleaseMask|ButtonMotionMask|EnterWindowMask|LeaveWindowMask|PointerMotionHintMask;
  attr.background_pixel=aguix->getPixel(bg);
  attr.colormap=aguix->getColormap();
  if(parent!=0) pwin=parent; else pwin=win;
  newwin=XCreateWindow(dsp,pwin,tx,ty,tw,th,0,aguix->getDepth(),InputOutput,vis,mask,&attr);
  if(!newwin) {
    return newwin;
  }
  XMapRaised(dsp,newwin);
  XClearWindow(dsp,newwin);
  subwins->addElement((void*)newwin);
  return newwin;
}

void AWindow::removeSubWin(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XDestroyWindow(dsp,subwin);
    subwins->removeElement((void*)subwin);
  }
}

void AWindow::addSubAWindow(AWindow *child)
{
  subawins->addElement(child);
}

void AWindow::removeSubAWindow(AWindow *child)
{
  subawins->removeElement(child);
}

bool AWindow::isParent(Window childwin,bool direct)
{
  int id=subawins->initEnum();
  bool found=false;
  if(direct==false) {
    AWindow *tawin=(AWindow*)subawins->getFirstElement(id);
    while(tawin!=NULL) {
      if(tawin->isParent(childwin,true)==true) found=true;
      tawin=(AWindow*)subawins->getNextElement(id);
    }
  }
  if(hasSubWin(childwin)==true) found=true;
  if(childwin==win) found=true;
  subawins->closeEnum(id);
  return found;
}

bool AWindow::remove(GUIElement *elem)
{
  bool returnvalue=false;
  int pos=gui->getIndex(elem);
  if(pos>=0) {
    returnvalue=true;
    gui->removeElementAt(pos);
    delete elem;
  }
  return returnvalue;
}

void AWindow::move(int nx,int ny)
{
  Display *dsp=aguix->getDisplay();
  XMoveWindow(dsp,win,nx,ny);
  x=nx;
  y=ny;
}

void AWindow::resize(int nw,int nh)
{
  if ( ( nw != w ) || ( nh != h ) ) {
    Display *dsp=aguix->getDisplay();
    XResizeWindow(dsp,win,nw,nh);
    w=nw;
    h=nh;
    AGMessage *agmsg = (AGMessage*)_allocsafe( sizeof( AGMessage ) );
    agmsg->type = AG_SIZECHANGED;
    agmsg->size.window = win;
    agmsg->size.neww = w;
    agmsg->size.newh = h;
    aguix->putAGMsg( agmsg );
  }
}

GUIElement *AWindow::findGUIElement(Window child)
{
  int id;
  id=gui->initEnum();
  GUIElement *guiel=(GUIElement*)gui->getFirstElement(id);
  while(guiel!=NULL) {
    if(guiel->isParent(child)==true) break;
    guiel=(GUIElement*)gui->getNextElement(id);
  }
  gui->closeEnum(id);
  return guiel;
}

int AWindow::getWidth()
{
  return w;
}

int AWindow::getHeight()
{
  return h;
}

void AWindow::setCursor(int type)
{
  aguix->setCursor(win,type);
}

void AWindow::unsetCursor()
{
  aguix->unsetCursor(win);
}

void AWindow::invalidFocus()
{
  // we could had a focus owner before getting a parent awindow
  // so first check if we have not one
  if(parentawindow!=NULL) {
    // parent awindow so he should store the focus owner
    parentawindow->invalidFocus();
  }
  // if we have a focus owner, release it
  if(focusOwner!=NULL) {
    GUIElement *oldf=focusOwner;
    focusOwner=NULL; // to prevent leaveFokus() to recall this method
    oldf->leaveFocus();
  }
}

void AWindow::applyFocus(GUIElement *newfocus)
{
  invalidFocus();
  if(newfocus!=NULL) {
    if(parentawindow!=NULL) {
      parentawindow->applyFocus(newfocus);
    } else {
      focusOwner=newfocus;
      newfocus->takeFocus();
    }
  }
}

bool AWindow::isOwner(GUIElement *f)
{
  if(parentawindow!=NULL) {
    return parentawindow->isOwner(f);
  } else {
    if(focusOwner==f) return true;
  }
  return false;
}

void AWindow::maximizeX()
{
  int id;
  int mw,tw;

  mw = 0;
  id = gui->initEnum();
  GUIElement *guiel = (GUIElement*)gui->getFirstElement( id );
  while ( guiel != NULL ) {
    tw = guiel->getX() + guiel->getWidth() + border;
    if ( tw > mw )
      mw = tw;
    guiel = (GUIElement*)gui->getNextElement( id );
  }
  gui->closeEnum( id );

  id = subawins->initEnum();
  AWindow *swin = (AWindow*)subawins->getFirstElement( id );
  while ( swin != NULL ) {
    tw = swin->getX() + swin->getWidth() + border;
    if ( tw > mw )
      mw = tw;
    swin = (AWindow*)subawins->getNextElement( id );
  }
  subawins->closeEnum( id );
  
  if ( mw > w ) {
    resize( mw, h );
  }
}

void AWindow::maximizeY()
{
  int id;
  int mh,th;

  mh = 0;
  id = gui->initEnum();
  GUIElement *guiel = (GUIElement*)gui->getFirstElement( id );
  while ( guiel != NULL ) {
    th = guiel->getY() + guiel->getHeight() + border;
    if ( th > mh )
      mh = th;
    guiel = (GUIElement*)gui->getNextElement( id );
  }
  gui->closeEnum( id );
  
  id = subawins->initEnum();
  AWindow *swin = (AWindow*)subawins->getFirstElement( id );
  while ( swin != NULL ) {
    th = swin->getY() + swin->getHeight() + border;
    if ( th > mh )
      mh = th;
    swin = (AWindow*)subawins->getNextElement( id );
  }
  subawins->closeEnum( id );

  if ( mh > h ) {
    resize( w, mh );
  }
}

int AWindow::getBorderWidth()
{
  return border;
}

void AWindow::setBorderWidth( int nv )
{
  if ( ( nv < 0 ) ||
       ( nv >= w/2 ) ) return;
  border = nv;
}

int AWindow::addTextFromString( const char *text,
				int tx,
				int ty,
				int vspace,
				Text ***return_texts,
				int *return_count,
				int *return_y )
{
  int i;
  int lines;
  char **liness;
  Text **text_elems;

  if ( text == NULL ) return 1;

  if ( tx < 0 ) tx = 0;
  if ( ty < 0 ) ty = 0;
  if ( vspace < 0 ) vspace = 0;

  lines = createLines( text, &liness );

  text_elems = (Text**)_allocsafe( sizeof( Text* ) * lines );

  for ( i = 0; i < lines; i++ ) {
    text_elems[i] = (Text*)add( new Text( aguix, tx, ty, liness[i], 1 ) );
    ty += text_elems[i]->getHeight() + vspace;
  }

  if ( return_texts != NULL ) {
    // caller want the Text list
    *return_texts = text_elems;
  } else _freesafe( text_elems );

  if ( return_count != NULL ) {
    *return_count = lines;
  }
  
  if ( return_y != NULL ) {
    // ty is not the real y value because vspace is always added
    *return_y = ty - vspace;
  }

  for ( i = 0; i < lines; i++ ) _freesafe( liness[i] );
  _freesafe( liness );

  return 0;
}

int AWindow::getX()
{
  return x;
}

int AWindow::getY()
{
  return y;
}

void AWindow::centerScreen()
{
  int rw, rh, nx, ny;

  rw = aguix->getRootWindowWidth();
  rh = aguix->getRootWindowHeight();
  if ( ( rw < 1 ) || ( rh < 1 ) ) return;
  
  nx = ( rw / 2 ) - ( w / 2 );
  ny = ( rh / 2 ) - ( h / 2 );
  move( nx, ny );
}
