/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "settingsdialog.h"

#include <QLayout>
#include <QLabel>
#include <QTabWidget>
#include <QCheckBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QIntValidator>
#include <QDialogButtonBox>
#include <QPushButton>
#include <QMessageBox>

#include "widgets/separatorcombobox.h"
#include "configdata.h"
#include "connectionmanager.h"
#include "iconloader.h"

SettingsDialog::SettingsDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Settings" ) );

    QVBoxLayout* mainLayout = new QVBoxLayout( this );

    QTabWidget* tabWidget = new QTabWidget( this );

    QWidget* appearanceTab = new QWidget( tabWidget );
    QGridLayout* appearanceLayout = new QGridLayout( appearanceTab );
    appearanceLayout->setMargin( 15 );

    QLabel* layoutLabel = new QLabel( tr( "Layout of the &main window:" ), appearanceTab );
    appearanceLayout->addWidget( layoutLabel, 0, 0 );

    m_layoutComboBox = new QComboBox( appearanceTab );
    m_layoutComboBox->addItem( tr( "Single Pane" ) );
    m_layoutComboBox->addItem( tr( "Three Panes" ) );
    appearanceLayout->addWidget( m_layoutComboBox, 0, 1 );

    layoutLabel->setBuddy( m_layoutComboBox );

    QLabel* detailsLabel = new QLabel( tr( "Layout of the issue details &view:" ), appearanceTab );
    appearanceLayout->addWidget( detailsLabel, 1, 0 );

    m_detailsComboBox = new QComboBox( appearanceTab );
    m_detailsComboBox->addItem( tr( "Single Pane" ) );
    m_detailsComboBox->addItem( tr( "Split Vertically" ) );
    m_detailsComboBox->addItem( tr( "Split Horizontally" ) );
    appearanceLayout->addWidget( m_detailsComboBox, 1, 1 );

    detailsLabel->setBuddy( m_detailsComboBox );

    QFrame* separator7 = new QFrame( appearanceTab );
    separator7->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator7->setFixedHeight( 20 );
    appearanceLayout->addWidget( separator7, 2, 0, 1, 2 );

    QLabel* languageLabel = new QLabel( tr( "&Language of user interface:" ), appearanceTab );
    appearanceLayout->addWidget( languageLabel, 3, 0 );

    m_languageComboBox = new SeparatorComboBox( appearanceTab );
    m_languageComboBox->addItem( tr( "Default" ) );
    m_languageComboBox->addSeparator();
    addLanguage( tr( "English" ), "en" );
    addLanguage( tr( "German" ), "de" );
    addLanguage( tr( "Polish" ), "pl" );
    addLanguage( tr( "Portuguese (Brazil)" ), "pt_BR" );
    addLanguage( tr( "Russian" ), "ru" );
    addLanguage( tr( "Spanish" ), "es" );
    addLanguage( tr( "Czech" ), "cs" );
    appearanceLayout->addWidget( m_languageComboBox, 3, 1 );

    languageLabel->setBuddy( m_languageComboBox );

    appearanceLayout->setRowStretch( appearanceLayout->rowCount(), 1 );
    appearanceLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( appearanceTab, IconLoader::icon( "appearance" ), tr( "Appearance" ) );

    QWidget* behaviorTab = new QWidget( tabWidget );
    QGridLayout* behaviorLayout = new QGridLayout( behaviorTab );
    behaviorLayout->setMargin( 15 );

    m_dockCheckBox = new QCheckBox( tr( "&Dock main window in system tray" ), behaviorTab );
    behaviorLayout->addWidget( m_dockCheckBox, 0, 0, 1, 2 );

    connect( m_dockCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( dockChanged() ) );

    QFrame* separator6 = new QFrame( behaviorTab );
    separator6->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator6->setFixedHeight( 20 );
    behaviorLayout->addWidget( separator6, 1, 0, 1, 2 );

    QLabel* showLabel = new QLabel( tr( "&Show main window at startup:" ), behaviorTab );
    behaviorLayout->addWidget( showLabel, 2, 0 );

    m_showComboBox = new QComboBox( behaviorTab );
    m_showComboBox->addItem( tr( "Never" ) );
    m_showComboBox->addItem( tr( "Automatically" ) );
    m_showComboBox->addItem( tr( "Always" ) );
    behaviorLayout->addWidget( m_showComboBox, 2, 1 );

    showLabel->setBuddy( m_showComboBox );

    QLabel* reconnectLabel = new QLabel( tr( "&Restore last connection at startup:" ), behaviorTab );
    behaviorLayout->addWidget( reconnectLabel, 3, 0 );

    m_reconnectComboBox = new QComboBox( behaviorTab );
    m_reconnectComboBox->addItem( tr( "Never" ) );
    m_reconnectComboBox->addItem( tr( "Automatically" ) );
    m_reconnectComboBox->addItem( tr( "Always" ) );
    behaviorLayout->addWidget( m_reconnectComboBox, 3, 1 );

    reconnectLabel->setBuddy( m_reconnectComboBox );

    QFrame* separator2 = new QFrame( behaviorTab );
    separator2->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator2->setFixedHeight( 20 );
    behaviorLayout->addWidget( separator2, 4, 0, 1, 2 );

    QLabel* actionLabel = new QLabel( tr( "D&efault action for attachments:" ), behaviorTab );
    behaviorLayout->addWidget( actionLabel, 5, 0 );

    m_actionComboBox = new QComboBox( behaviorTab );
    m_actionComboBox->addItem( tr( "Ask what to do" ) );
    m_actionComboBox->addItem( tr( "Open" ) );
    m_actionComboBox->addItem( tr( "Save As..." ) );
    behaviorLayout->addWidget( m_actionComboBox, 5, 1 );

    actionLabel->setBuddy( m_actionComboBox );

#if defined( Q_WS_WIN )
    if ( !configData->isLocal() ) {
        QFrame* separator3 = new QFrame( behaviorTab );
        separator3->setFrameStyle( QFrame::HLine | QFrame::Sunken );
        separator3->setFixedHeight( 20 );
        behaviorLayout->addWidget( separator3, 6, 0, 1, 2 );

        m_autoStartCheckBox = new QCheckBox( tr( "Start &WebIssues when I start Windows" ), behaviorTab );
        behaviorLayout->addWidget( m_autoStartCheckBox, 7, 0, 1, 2 );
    }
#endif

    behaviorLayout->setRowStretch( behaviorLayout->rowCount(), 1 );
    behaviorLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( behaviorTab, IconLoader::icon( "configure" ), tr( "Behavior" ) );

    QWidget* connectionTab = new QWidget( tabWidget );
    QGridLayout* connectionLayout = new QGridLayout( connectionTab );
    connectionLayout->setMargin( 15 );

    QLabel* proxyLabel = new QLabel( tr( "&Proxy used for Internet connections:" ), connectionTab );
    connectionLayout->addWidget( proxyLabel, 0, 0, 1, 2 );

    m_proxyCombo = new QComboBox( connectionTab );
    m_proxyCombo->addItem( tr( "No Proxy" ) );
    m_proxyCombo->addItem( tr( "SOCKS5 Proxy" ) );
#if ( QT_VERSION >= 0x040300 )
    m_proxyCombo->addItem( tr( "HTTP Proxy" ) );
#endif
    connectionLayout->addWidget( m_proxyCombo, 0, 2 );

    connect( m_proxyCombo, SIGNAL( currentIndexChanged( int ) ), this, SLOT( proxyChanged() ) );

    proxyLabel->setBuddy( m_proxyCombo );

    QLabel* hostLabel = new QLabel( tr( "&Host name:" ), connectionTab );
    connectionLayout->addWidget( hostLabel, 1, 0 );

    m_proxyHostEdit = new QLineEdit( connectionTab );
    connectionLayout->addWidget( m_proxyHostEdit, 1, 1, 1, 2 );

    hostLabel->setBuddy( m_proxyHostEdit );

    QLabel* portLabel = new QLabel( tr( "Port &number:" ), connectionTab );
    connectionLayout->addWidget( portLabel, 2, 0, 1, 2 );

    m_proxyPortEdit = new QLineEdit( connectionTab );
    QIntValidator* portValidator = new QIntValidator( 1, 65535, m_proxyPortEdit );
    m_proxyPortEdit->setValidator( portValidator );
    connectionLayout->addWidget( m_proxyPortEdit, 2, 2 );

    portLabel->setBuddy( m_proxyPortEdit );

    QLabel* userLabel = new QLabel( tr( "&User:" ), connectionTab );
    connectionLayout->addWidget( userLabel, 3, 0 );

    m_proxyUserEdit = new QLineEdit( connectionTab );
    connectionLayout->addWidget( m_proxyUserEdit, 3, 1, 1, 2 );

    userLabel->setBuddy( m_proxyUserEdit );

    QLabel* passwordLabel = new QLabel( tr( "Pa&ssword:" ), connectionTab );
    connectionLayout->addWidget( passwordLabel, 4, 0 );

    m_proxyPasswordEdit = new QLineEdit( connectionTab );
    m_proxyPasswordEdit->setEchoMode( QLineEdit::Password );
    connectionLayout->addWidget( m_proxyPasswordEdit, 4, 1, 1, 2 );

    passwordLabel->setBuddy( m_proxyPasswordEdit );

    connectionLayout->setRowStretch( connectionLayout->rowCount(), 1 );
    connectionLayout->setColumnStretch( 1, 1 );

    tabWidget->addTab( connectionTab, IconLoader::icon( "connection" ), tr( "Connection" ) );

    QWidget* advancedTab = new QWidget( tabWidget );
    QGridLayout* advancedLayout = new QGridLayout( advancedTab );
    advancedLayout->setMargin( 15 );

    QLabel* intervalLabel = new QLabel( tr( "I&nterval of periodic data updates:" ), advancedTab );
    advancedLayout->addWidget( intervalLabel, 0, 0 );

    m_intervalSpin = new QSpinBox( advancedTab );
    m_intervalSpin->setRange( 1, 120 );
    m_intervalSpin->setSuffix( tr( " min" ) );
    advancedLayout->addWidget( m_intervalSpin, 0, 1 );

    intervalLabel->setBuddy( m_intervalSpin );

    QFrame* separator4 = new QFrame( advancedTab );
    separator4->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator4->setFixedHeight( 20 );
    advancedLayout->addWidget( separator4, 1, 0, 1, 2 );

    QLabel* cacheLabel = new QLabel( tr( "&Maximum size of attachments cache:" ), advancedTab );
    advancedLayout->addWidget( cacheLabel, 2, 0 );

    m_cacheSpin = new QSpinBox( advancedTab );
    m_cacheSpin->setRange( 1, 200 );
    m_cacheSpin->setSuffix( tr( " MB" ) );
    advancedLayout->addWidget( m_cacheSpin, 2, 1 );

    cacheLabel->setBuddy( m_cacheSpin );

    QFrame* separator5 = new QFrame( advancedTab );
    separator5->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator5->setFixedHeight( 20 );
    advancedLayout->addWidget( separator5, 3, 0, 1, 2 );

    QLabel* warnAttachmentLabel = new QLabel( tr( "&Warn when adding attachments larger than:" ), advancedTab );
    advancedLayout->addWidget( warnAttachmentLabel, 4, 0 );

    m_warnAttachmentSpin= new QSpinBox( advancedTab );
    m_warnAttachmentSpin->setRange( 0, 50 );
    m_warnAttachmentSpin->setSuffix( tr( " MB" ) );
    m_warnAttachmentSpin->setSpecialValueText( tr( "Off" ) );
    advancedLayout->addWidget( m_warnAttachmentSpin, 4, 1 );

    warnAttachmentLabel->setBuddy( m_warnAttachmentSpin );

    QLabel* warnCommentLabel = new QLabel( tr( "Wa&rn when adding comments larger than:" ), advancedTab );
    advancedLayout->addWidget( warnCommentLabel, 5, 0 );

    m_warnCommentSpin = new QSpinBox( advancedTab );
    m_warnCommentSpin->setRange( 0, 200 );
    m_warnCommentSpin->setSuffix( tr( " kB" ) );
    m_warnCommentSpin->setSpecialValueText( tr( "Off" ) );
    advancedLayout->addWidget( m_warnCommentSpin, 5, 1 );

    warnCommentLabel->setBuddy( m_warnCommentSpin );

    advancedLayout->setRowStretch( advancedLayout->rowCount(), 1 );
    advancedLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( advancedTab, IconLoader::icon( "gear" ), tr( "Advanced" ) );

    mainLayout->addWidget( tabWidget );

    mainLayout->addSpacing( 10 );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Apply | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    mainLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SLOT( apply() ) );

    m_dockCheckBox->setChecked( configData->dockWindow() );
    m_reconnectComboBox->setCurrentIndex( configData->reconnectAtStartup() );

    int index = m_languageComboBox->findData( configData->language() );
    if ( index >= 2 )
        m_languageComboBox->setCurrentIndex( index );

    m_layoutComboBox->setCurrentIndex( configData->windowLayout() );

    if ( configData->detailsLayout() == LayoutSplitVertically )
        m_detailsComboBox->setCurrentIndex( 1 );
    else if ( configData->detailsLayout() == LayoutSplitHorizontally )
        m_detailsComboBox->setCurrentIndex( 2 );

    QNetworkProxy::ProxyType proxyType = configData->networkProxy().type();
    if ( proxyType == QNetworkProxy::Socks5Proxy )
        m_proxyCombo->setCurrentIndex( 1 );
#if ( QT_VERSION >= 0x040300 )
    else if ( proxyType == QNetworkProxy::HttpProxy )
        m_proxyCombo->setCurrentIndex( 2 );
#endif

    if ( proxyType != QNetworkProxy::NoProxy ) {
        m_proxyHostEdit->setText( configData->networkProxy().hostName() );
        if ( configData->networkProxy().port() != 0 )
            m_proxyPortEdit->setText( QString::number( configData->networkProxy().port() ) );
        m_proxyUserEdit->setText( configData->networkProxy().user() );
        m_proxyPasswordEdit->setText( configData->networkProxy().password() );
    }

    m_intervalSpin->setValue( configData->updateInterval() );
    m_actionComboBox->setCurrentIndex( configData->defaultAttachmentAction() );
    m_cacheSpin->setValue( configData->attachmentsCacheSize() );
    m_warnAttachmentSpin->setValue( configData->warnAttachmentSize() );
    m_warnCommentSpin->setValue( configData->warnCommentSize() );

#if defined( Q_WS_WIN )
    if ( !configData->isLocal() )
        m_autoStartCheckBox->setChecked( configData->checkAutoStart() );
#endif

    dockChanged();
    proxyChanged();

    resize( 450, 300 );
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::accept()
{
    if ( apply() )
        QDialog::accept();
}

bool SettingsDialog::apply()
{
    if ( m_proxyCombo->currentIndex() > 0 ) {
        if ( m_proxyHostEdit->text().isEmpty() || !m_proxyPortEdit->hasAcceptableInput() ) {
            QMessageBox::warning( this, tr( "Warning" ), tr( "Missing proxy host name or port number." ) );
            return false;
        }
    }

    configData->setDockWindow( m_dockCheckBox->isChecked() );
    configData->setShowAtStartup( (RestoreOption)m_showComboBox->currentIndex() );
    configData->setReconnectAtStartup( (RestoreOption)m_reconnectComboBox->currentIndex() );

    QString language = m_languageComboBox->itemData( m_languageComboBox->currentIndex() ).toString();
    if ( language != configData->language() )
        QMessageBox::warning( this, tr( "Warning" ), tr( "Language settings will be applied when the application is restarted." ) );
    configData->setLanguage( language );

    configData->setWindowLayout( (WindowLayout)m_layoutComboBox->currentIndex() );

    if ( m_detailsComboBox->currentIndex() == 1 )
        configData->setDetailsLayout( LayoutSplitVertically );
    else if ( m_detailsComboBox->currentIndex() == 2 )
        configData->setDetailsLayout( LayoutSplitHorizontally );
    else
        configData->setDetailsLayout( LayoutSinglePane );

    QNetworkProxy::ProxyType proxyType;
    if ( m_proxyCombo->currentIndex() == 1 )
        proxyType = QNetworkProxy::Socks5Proxy;
#if ( QT_VERSION >= 0x040300 )
    else if ( m_proxyCombo->currentIndex() == 2 )
        proxyType = QNetworkProxy::HttpProxy;
#endif
    else
        proxyType = QNetworkProxy::NoProxy;

    QNetworkProxy proxy( proxyType );

    if ( proxyType != QNetworkProxy::NoProxy ) {
        proxy.setHostName( m_proxyHostEdit->text() );
        proxy.setPort( (quint16)m_proxyPortEdit->text().toInt() );
        proxy.setUser( m_proxyUserEdit->text() );
        proxy.setPassword( m_proxyPasswordEdit->text() );
    }

    if ( connectionManager->isConnected() ) {
        if ( !compareProxies( configData->networkProxy(), proxy ) ) {
            QMessageBox::information( this, tr( "Notice" ),
                tr( "Changes in proxy settings will not be applied until you reconnect to the server." ) );
        }
    }

    configData->setNetworkProxy( proxy );

    configData->setUpdateInterval( m_intervalSpin->value() );
    configData->setDefaultAttachmentAction( (AttachmentAction)m_actionComboBox->currentIndex() );
    configData->setAttachmentsCacheSize( m_cacheSpin->value() );
    configData->setWarnAttachmentSize( m_warnAttachmentSpin->value() );
    configData->setWarnCommentSize( m_warnCommentSpin->value() );

#if defined( Q_WS_WIN )
    if ( !configData->isLocal() )
        configData->setAutoStart( m_autoStartCheckBox->isChecked() );
#endif

    configData->saveSettings();

    return true;
}

void SettingsDialog::dockChanged()
{
    if ( m_dockCheckBox->isChecked() ) {
        m_showComboBox->setCurrentIndex( configData->showAtStartup() );
        m_showComboBox->setEnabled( true );
    } else {
        m_showComboBox->setCurrentIndex( 2 );
        m_showComboBox->setEnabled( false );
    }
}

void SettingsDialog::proxyChanged()
{
    bool enabled = ( m_proxyCombo->currentIndex() > 0 );
    m_proxyHostEdit->setEnabled( enabled );
    m_proxyPortEdit->setEnabled( enabled );
    m_proxyUserEdit->setEnabled( enabled );
    m_proxyPasswordEdit->setEnabled( enabled );
}

void SettingsDialog::addLanguage( const QString& name, const QString& language )
{
    int index = 2;
    while ( index < m_languageComboBox->count() && QString::localeAwareCompare( name, m_languageComboBox->itemText( index ) ) > 0 )
        index++;

    m_languageComboBox->insertItem( index, name, language );
}

bool SettingsDialog::compareProxies( const QNetworkProxy& lhs, const QNetworkProxy& rhs )
{
    if ( lhs.type() != rhs.type() )
        return false;

    if ( lhs.type() != QNetworkProxy::NoProxy ) {
        if ( lhs.hostName() != rhs.hostName() || lhs.port() != rhs.port() )
            return false;
        if ( lhs.user() != rhs.user() || lhs.password() != rhs.password() )
            return false;
    }

    return true;
}
