#include "texteditor.h"

#include <qcanvas.h>
#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qaction.h>
#include <qapplication.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qtoolbar.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtextedit.h>
#include <qaccel.h>
#include <qtextstream.h>
#include <qpaintdevicemetrics.h>
#include <qsettings.h>
#include <qfontdialog.h>
#include <qmessagebox.h>
#include <qcolordialog.h>
#include <qregexp.h>
#include <qlineedit.h>
#include <qsimplerichtext.h>

#include <iostream>
#include <cstdlib>
#include <typeinfo>

#include "noteobj.h"
#include "version.h"

extern QCanvas* actCanvas;
extern int statusbarTime;
extern QSettings settings;

extern QAction *actionViewToggleNoteEditor;

extern QString iconPath;

using namespace std;


///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////


TextEditor::TextEditor()
{
    printer = new QPrinter( QPrinter::HighResolution );
	printer->setPrinterName (settings.readEntry("/vym/mainwindow/printerName",printer->printerName()));

	// Editor (use MyTextEdit to wrap verticalAlignmentChanged to 
	//         int argument. see header file)
    e = new MyTextEdit( this, "editor" );
    e->setFocus();
	e->setTextFormat(RichText);		// default
	e->setTabStopWidth (20);		// unit is pixel
	e->setColor (black);
	connect (e, SIGNAL( textChanged() ), this, SLOT( editorChanged() ) );
    setCentralWidget( e );
    statusBar()->message( "Ready", statusbarTime);
	setCaption(__VYM " - " +tr ("Note Editor"));

	// Toolbars
	setupFileActions();
	setupEditActions();
	setupFormatActions();
	setupSettingsActions();
	
	// Various states
	emptyPaper   = QBrush(gray);
	filledPaper  = QBrush(white);
	inactivePaper= QBrush(black);
	setInactive();

	// Load Settings
	resize (settings.readNumEntry( "/vym/noteeditor/geometry/width", 450),
	        settings.readNumEntry( "/vym/noteeditor/geometry/height",600));
	move   (settings.readNumEntry( "/vym/noteeditor/geometry/posX", 150),
	        settings.readNumEntry( "/vym/noteeditor/geometry/posY",  50));
	
	if (settings.readEntry( "/vym/noteeditor/showWithMain","yes") =="yes")
		setShowWithMain(true);
	else	
		setShowWithMain(false);

	varFont.fromString( settings.readEntry 
		("/vym/noteeditor/fonts/varFont",
		"Nimbus Sans l,14,-1,5,48,0,0,0,0,0") 
	);
	fixedFont.fromString (settings.readEntry (
		"/vym/noteeditor/fonts/fixedFont",
		"Courier,14,-1,5,48,0,0,0,1,0") 
	);
	QString s=settings.readEntry ("/vym/noteeditor/fonts/fonthintDefault","variable");
	if (s == "fixed")
	{	
		actionSettingsFonthintDefault->setOn (true);
		e->setCurrentFont (fixedFont);
	} else	
	{
		actionSettingsFonthintDefault->setOn (false);
		e->setCurrentFont (varFont);
	}	
	filenameHint="";

	// Save settings in vymrc
	settings.writeEntry("/vym/mainwindow/printerName",printer->printerName());
}


TextEditor::~TextEditor()
{
    if (printer) delete printer;
	// Save Settings
	settings.writeEntry( "/vym/noteeditor/geometry/width", width() );
	settings.writeEntry( "/vym/noteeditor/geometry/height", height() );
	settings.writeEntry( "/vym/noteeditor/geometry/posX", pos().x() );
	settings.writeEntry( "/vym/noteeditor/geometry/posY", pos().y() );
	
	if (showWithMain())
		settings.writeEntry( "/vym/noteeditor/showWithMain","yes");
	else	
		settings.writeEntry( "/vym/noteeditor/showWithMain","no");

	QString s;
	if (actionSettingsFonthintDefault->isOn() )
		s="fixed";
	else	
		s="variable";
	settings.writeEntry( "/vym/noteeditor/fonts/fonthintDefault",s );
	settings.writeEntry ("/vym/noteeditor/fonts/varFont",
		varFont.toString() );
	settings.writeEntry ("/vym/noteeditor/fonts/fixedFont",
		fixedFont.toString() );
}

bool TextEditor::isEmpty()
{
	if (e->text().length())
		return false;
	else
		return true;
}

void TextEditor::setShowWithMain(bool v)
{
	showwithmain=v;
}

bool TextEditor::showWithMain()
{
	return showwithmain;
}

void TextEditor::setFontHint (const QString &fh)
{
	if (fh=="fixed")
		actionFormatUseFixedFont->setOn (true);
	else
		actionFormatUseFixedFont->setOn (false);
}


QString TextEditor::getFontHint()
{
	if (actionFormatUseFixedFont->isOn())
		return "fixed";
	else	
		return "var";
}

QString TextEditor::getFontHintDefault()
{
	if (actionSettingsFonthintDefault->isOn())
		return "fixed";
	else	
		return "var";
}

void TextEditor::setFilename(const QString &fn)
{
	if (state==filledEditor)
		if (fn.isEmpty() )
		{
			filename="";
			statusBar()->message( "No filename available for this note.", statusbarTime );
		}	
		else
		{
			filename=fn;
			statusBar()->message( QString( "Current filename is %1" ).arg( filename ), statusbarTime );
		}	
}

QString TextEditor::getFilename()
{
	return filename;
}

void TextEditor::setFilenameHint(const QString &fnh)
{
	filenameHint=fnh;
}

QString TextEditor::getFilenameHint()
{
	return filenameHint;
}

bool TextEditor::findText(const QString &t, const bool &cs)
{
	bool wo=false;	// word matches
	if (e->find (t, cs, wo, true, 0, 0 ))
		return true;
	else	
		return false;
}

void TextEditor::setupFileActions()
{
    QToolBar *tb = new QToolBar( this );
    tb->setLabel( "File Actions" );
    QPopupMenu *menu = new QPopupMenu( this );
    menuBar()->insertItem( tr( "&File" ), menu );

    QAction *a;
    a = new QAction( tr( "Import" ), QPixmap( iconPath+"fileopen.png"), tr( "&Import..." ), CTRL + Key_O, this, "fileImport" );
    connect( a, SIGNAL( activated() ), this, SLOT( textLoad() ) );
	a->setEnabled(false);
    a->addTo( tb );
    a->addTo( menu );
	actionFileLoad=a;

    menu->insertSeparator();
    a = new QAction( tr( "Export Note (HTML)" ), QPixmap(iconPath+"filesave.png" ), tr( "&Export..." ), CTRL + Key_S, this, "fileSave" );
    connect( a, SIGNAL( activated() ), this, SLOT( textSave() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionFileSave=a;
	
    a = new QAction( tr( "Export Note As (HTML) " ), QPixmap(), tr( "Export &As... (HTML)" ), 0, this, "exportHTML" );
    connect( a, SIGNAL( activated() ), this, SLOT( textSaveAs() ) );
    a->addTo( menu );
	actionFileSaveAs=a;

    a = new QAction( tr( "Export Note As (ASCII) " ), QPixmap(), tr( "Export &As...(ASCII)" ), ALT + Key_X, this, "exportASCII" );
    connect( a, SIGNAL( activated() ), this, SLOT( textExportAsASCII() ) );
    a->addTo( menu );
	actionFileSaveAs=a;

    menu->insertSeparator();
    a = new QAction( tr( "Print Note" ), QPixmap(iconPath+"fileprint.png" ), tr( "&Print..." ), CTRL + Key_P, this, "filePrint" );
    connect( a, SIGNAL( activated() ), this, SLOT( textPrint() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionFilePrint=a;
}

void TextEditor::setupEditActions()
{
    QToolBar *tb = new QToolBar( this );
    tb->setLabel( "Edit Actions" );
    QPopupMenu *menu = new QPopupMenu( this );
    menuBar()->insertItem( tr( "&Edit" ), menu );

    QAction *a;
    a = new QAction( tr( "Undo" ), QPixmap(iconPath+"undo.png"), tr( "&Undo" ), CTRL + Key_Z, this, "undoEvent" );
    connect( a, SIGNAL( activated() ), e, SLOT( undo() ) );
    a->addTo( menu );
    a->addTo( tb);
	actionEditUndo=a;
	
    a = new QAction( tr( "Redo" ), QPixmap(iconPath+"redo.png" ), tr( "&Redo" ), CTRL + Key_Y, this, "editRedo" ); 
    connect( a, SIGNAL( activated() ), e, SLOT( redo() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionEditRedo=a;

    menu->insertSeparator();
    a = new QAction( tr( "Select and copy all" ), QPixmap(), tr( "Select and copy &all" ), CTRL + Key_A, this, "editcopyall" ); 
    connect( a, SIGNAL( activated() ), this, SLOT( editCopyAll() ) );
    a->addTo( menu );

    menu->insertSeparator();
    a = new QAction( tr( "Copy" ), QPixmap(iconPath+"editcopy.png" ), tr( "&Copy" ), CTRL + Key_C, this, "editCopy" );
    connect( a, SIGNAL( activated() ), e, SLOT( copy() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionEditCopy=a;
	
    a = new QAction( tr( "Cut" ), QPixmap(iconPath+"editcut.png" ), tr( "Cu&t" ), CTRL + Key_X, this, "editCut" );
    connect( a, SIGNAL( activated() ), e, SLOT( cut() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionEditCut=a;

    a = new QAction( tr( "Paste" ), QPixmap(iconPath+"editpaste.png" ), tr( "&Paste" ), CTRL + Key_V, this, "editPaste" );
    connect( a, SIGNAL( activated() ), e, SLOT( paste() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionEditPaste=a;
	
    a = new QAction( tr( "Delete all" ), QPixmap( iconPath+"edittrash.png"), tr( "&Delete All" ), 0, this, "editDeleteAll" );
    connect( a, SIGNAL( activated() ), e, SLOT( clear() ) );
    a->addTo( tb );
    a->addTo( menu );
	actionEditDeleteAll=a;

	a = new QAction( tr( "Convert paragraphs to linebreaks" ), QPixmap(), tr( "&Convert Paragraphs" ), ALT + Key_P, this, "editConvertPar" );
    connect( a, SIGNAL( activated() ), this, SLOT( textConvertPar() ) );
    a->addTo( menu );
	actionEditConvertPar=a;

	a = new QAction( tr( "Join all lines of a paragraph" ), QPixmap(), tr( "&Join lines" ), ALT + Key_J, this, "editJoinLines" );
    connect( a, SIGNAL( activated() ), this, SLOT( textJoinLines() ) );
    a->addTo( menu );
	actionEditJoinLines=a;
}

void TextEditor::setupFormatActions()
{
    QToolBar *tb = new QToolBar( this );
    tb->setLabel( "Format Actions" );
    QPopupMenu *menu = new QPopupMenu( this );
    menuBar()->insertItem( tr( "F&ormat" ), menu );

    QAction *a;

    a = new QAction( tr( "Toggle font hint for the whole text" ), QPixmap(iconPath+"formatfixedfont.png"), tr( "&Font hint" ), ALT + Key_I, this, "fontHint" );
	a->setToggleAction (true);
	a->setOn (settings.readBoolEntry ("/vym/noteeditor/fonts/useFixedByDefault",false) );
    connect( a, SIGNAL( activated() ), this, SLOT( toggleFonthint() ) );
    a->addTo( menu );
    a->addTo( tb );
	actionFormatUseFixedFont=a;

	menu->insertSeparator();

    comboFont = new QComboBox( true, tb );
    QFontDatabase db;
    comboFont->insertStringList( db.families() );
    connect( comboFont, SIGNAL( activated( const QString & ) ),
	     this, SLOT( textFamily( const QString & ) ) );
    comboFont->lineEdit()->setText( QApplication::font().family() );

    comboSize = new QComboBox( true, tb );
    QValueList<int> sizes = db.standardSizes();
    QValueList<int>::Iterator it = sizes.begin();
    for ( ; it != sizes.end(); ++it )
	comboSize->insertItem( QString::number( *it ) );
    connect( comboSize, SIGNAL( activated( const QString & ) ),
	     this, SLOT( textSize( const QString & ) ) );
    comboSize->lineEdit()->setText( QString::number( QApplication::font().pointSize() ) );

    menu->insertSeparator();

    QPixmap pix( 16, 16 );
    pix.fill( e->color());
    actionTextColor = new QAction( pix, tr( "&Color..." ), 0, this, "textColor" );
    connect( actionTextColor, SIGNAL( activated() ), this, SLOT( textColor() ) );
    actionTextColor->addTo( tb );
    actionTextColor->addTo( menu );

    actionTextBold = new QAction( QPixmap (iconPath+"text_bold.png"), tr( "&Bold" ), CTRL + Key_B, this, "textBold" );
    connect( actionTextBold, SIGNAL( activated() ), this, SLOT( textBold() ) );
    actionTextBold->addTo( tb );
    actionTextBold->addTo( menu );
    actionTextBold->setToggleAction( true );
    actionTextItalic = new QAction( QPixmap(iconPath+"text_italic.png"), tr( "&Italic" ), CTRL + Key_I, this, "textItalic" );
    connect( actionTextItalic, SIGNAL( activated() ), this, SLOT( textItalic() ) );
    actionTextItalic->addTo( tb );
    actionTextItalic->addTo( menu );
    actionTextItalic->setToggleAction( true );
    actionTextUnderline = new QAction( QPixmap (iconPath+"text_under.png"), tr( "&Underline" ), CTRL + Key_U, this, "textUnderline" );
    connect( actionTextUnderline, SIGNAL( activated() ), this, SLOT( textUnderline() ) );
    actionTextUnderline->addTo( tb );
    actionTextUnderline->addTo( menu );
    actionTextUnderline->setToggleAction( true );
    menu->insertSeparator();

    QActionGroup *grp = new QActionGroup( this );
    connect( grp, SIGNAL( selected( QAction* ) ), this, SLOT( textAlign( QAction* ) ) );

    actionAlignLeft = new QAction( QPixmap (iconPath+"text_left.png"), tr( "&Left" ), CTRL + Key_L, grp, "textLeft" );
    actionAlignLeft->setToggleAction( true );
    actionAlignCenter = new QAction( QPixmap (iconPath+"text_center.png"), tr( "C&enter" ), CTRL + Key_E, grp, "textCenter" );
    actionAlignCenter->setToggleAction( true );
    actionAlignRight = new QAction( QPixmap (iconPath+"text_right.png" ), tr( "&Right" ), CTRL + Key_R, grp, "textRight" );
    actionAlignRight->setToggleAction( true );
    actionAlignJustify = new QAction( QPixmap ( iconPath+"text_block.png"), tr( "&Justify" ), CTRL + Key_J, grp, "textjustify" );
    actionAlignJustify->setToggleAction( true );

    grp->addTo( tb );
    grp->addTo( menu );

    QActionGroup *grp2 = new QActionGroup( this );
    grp2->setExclusive(false);
    actionAlignSubScript = new QAction( QPixmap (iconPath+"text_sub.png"), tr( "Subs&cript" ), CTRL + SHIFT + Key_B, grp2, "textSubscript" );

    actionAlignSubScript->setToggleAction( true );
    connect(actionAlignSubScript, SIGNAL(activated()), this, SLOT(textVAlign()));

    actionAlignSuperScript = new QAction( QPixmap (iconPath+"text_super.png"), tr( "Su&perscript" ), CTRL + SHIFT + Key_P, grp2, "textSuperscript" );

    actionAlignSuperScript->setToggleAction( true );
    connect(actionAlignSuperScript, SIGNAL(activated()), this, SLOT(textVAlign()));

    menu->insertSeparator();
    
    grp2->addTo(tb);
    grp2->addTo(menu);

    connect( e, SIGNAL( currentFontChanged( const QFont & ) ),
	     this, SLOT( fontChanged( const QFont & ) ) );
    connect( e, SIGNAL( currentColorChanged( const QColor & ) ),
	     this, SLOT( colorChanged( const QColor & ) ) );
    connect( e, SIGNAL( currentAlignmentChanged( int ) ),
	     this, SLOT( alignmentChanged( int ) ) );
    connect( e, SIGNAL( currentVerticalAlignmentChanged(int)),
	     this, SLOT( verticalAlignmentChanged(int)));

}

void TextEditor::setupSettingsActions()
{
    QPopupMenu *menu = new QPopupMenu( this );
    menuBar()->insertItem( tr( "&Settings" ), menu );

    QAction *a;
    a = new QAction( tr( "Set fixed font" ), QPixmap(), tr( "Set &fixed font" ), 0, this, "setFixedFont" );
    connect( a, SIGNAL( activated() ), this, SLOT( setFixedFont() ) );
    a->addTo( menu );
	actionSettingsFixedFont=a;

    a = new QAction( tr( "Set variable font" ), QPixmap(), tr( "Set &variable font" ), 0, this, "setvariableFont" );
    connect( a, SIGNAL( activated() ), this, SLOT( setVarFont() ) );
    a->addTo( menu );
	actionSettingsVarFont=a;

    a = new QAction( tr( "Used fixed font by default" ), QPixmap(), tr( "&fixed font is default" ), 0, this, "fonthintDefault" );
	a->setToggleAction (true);
	// set state later in constructor...
    a->addTo( menu );
	actionSettingsFonthintDefault=a;
}

void TextEditor::textLoad()
{
	if (state!=inactiveEditor)
	{
		if (e->length()) 
		{
			QMessageBox mb( __VYM " - " +tr("Note Editor"),
				"Loading will overwrite the existing note",
				QMessageBox::Warning,
				QMessageBox::Yes | QMessageBox::Default,
				QMessageBox::Cancel,
				0 );
			mb.setButtonText( QMessageBox::Yes, "Load note" );
			switch( mb.exec() ) {
				case QMessageBox::Cancel:
					return;
					break;
			}
		} 
		// Load note
		QFileDialog *fd=new QFileDialog( this);
		fd->addFilter ("ASCII texts (*.txt)");
		fd->addFilter ("VYM notes (*.html)");
		fd->show();
		QString fn;
		if ( fd->exec() == QDialog::Accepted )
			fn = fd->selectedFile();

		if ( !fn.isEmpty() )
		{
			QFile f( fn );
			if ( !f.open( IO_ReadOnly ) )
			return;

			QTextStream ts( &f );
			setText( ts.read() );
			editorChanged();
		}
	}
}

void TextEditor::closeEvent( QCloseEvent* ce )
{
    ce->accept();	// TextEditor can be reopened with show()
    actionViewToggleNoteEditor->setOn (false);
    showwithmain=false;
    return;
}

QString TextEditor::getText()
{
	return e->text();
}

void TextEditor::editorChanged()
{
	if (isEmpty())
		state=emptyEditor;
	else
		state=filledEditor;

		if (state==emptyEditor)
			e->setPaper (emptyPaper);
		else
			e->setPaper (filledPaper);
	// SLOT is LinkableMapObj, which will update systemFlag
	emit (textHasChanged() );
}


void TextEditor::setText(QString t)
{
	if ( !QStyleSheet::mightBeRichText( t ) )
		t = QStyleSheet::convertFromPlainText( t, QStyleSheetItem::WhiteSpaceNormal );
	e->setReadOnly(false);
	e->setText(t);
//	editorChanged();	//not called automagically  FIXME or is it?

	enableActions();
}

void TextEditor::setInactive()
{
	state=inactiveEditor;
	setText("");
	e->setPaper (inactivePaper);
	e->setReadOnly (true);

	disableActions();
}

void TextEditor::editCopyAll()
{
	e->selectAll();
	e->copy();
}

void TextEditor::textSaveAs()
{
    QString fn = QFileDialog::getSaveFileName( QString::null, "VYM Note (HTML) (*.html);;All files (*)",
					       this,"export note dialog",tr("Export Note to single file") );

    if ( !fn.isEmpty() ) 
	{
		QFile file (fn);
		if (file.exists())
		{
			QMessageBox mb( __VYM,
				tr("The file ") + fn + 
				tr(" exists already. "
				"Do you want to overwrite it?"),
			QMessageBox::Warning,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape,
			QMessageBox::NoButton );
			mb.setButtonText( QMessageBox::Yes, tr("Overwrite") );
			mb.setButtonText( QMessageBox::No, tr("Cancel"));
			switch( mb.exec() ) {
				case QMessageBox::Yes:
					// save 
					filename = fn;
					textSave();
					return;
				case QMessageBox::Cancel:
					// do nothing
					break;
			}
		} else
		{
			filename = fn;
			textSave();
			return;
		}			
    }
	statusBar()->message(tr( "Couldn't export note ") + fn, statusbarTime );
}


void TextEditor::textSave()
{
    if ( filename.isEmpty() ) 
	{
		textSaveAs();
		return;
    }

    QString text = e->text();
    QFile f( filename );
    if ( !f.open( IO_WriteOnly ) ) 
	{
		statusBar()->message( QString("Could not write to %1").arg(filename),
					  statusbarTime );
		return;
    }

    QTextStream t( &f );
    t << text;
    f.close();

    e->setModified( FALSE );

    statusBar()->message( QString( "Note exported as %1" ).arg( filename ), statusbarTime );
}

void TextEditor::textConvertPar()
{
	// In X11 a copy&paste generates paragraphs, 
	// which is not always wanted
	// This function replaces paragraphs by linebreaks.
	int parFrom, parTo, indFrom, indTo;
	e->getSelection (&parFrom,&indFrom,&parTo,&indTo);
	QString t;
	if (parFrom>-1)
		t=e->selectedText();
	else
		t=e->text();

	QRegExp re("<p.*>");
	re.setMinimal(true);
	t.replace (re,"");
	t.replace ("</p>","<br />");
	if (parFrom>-1)
	{
		e->setCursorPosition (parFrom,indFrom);
		e->cut();
		// Tried to simply insert the changed text with
		// e->insert (t,(uint)(QTextEdit::RemoveSelected));
		// but then the html would be quoted. So I use the ugly
		// way: insert a marker, replace it in whole text of QTextEdit
		QString marker="R3PlAcEMeL4teR!";
		e->insert (marker);
		e->setText (e->text().replace(marker,t));
	} else
		e->setText(t);
}

void TextEditor::textJoinLines()
{
	int parFrom, parTo, indFrom, indTo;
	e->getSelection (&parFrom,&indFrom,&parTo,&indTo);
	QString t;
	if (parFrom>-1)
		t=e->selectedText();
	else
		t=e->text();
	// In addition to textConvertPar it is sometimes
	// useful to join all lines of a paragraph
	QRegExp re("</p>\n+<p>(?!</p>)");
	re.setMinimal(true);
	t.replace (re," ");

	// Above we may have introduced new " " at beginning of a
	// paragraph - remove it.
	re.setPattern("<p> ");
	t.replace (re,"<p>");
	if (parFrom>-1)
	{
		e->setCursorPosition (parFrom,indFrom);
		e->cut();
		// Tried to simply insert the changed text with
		// e->insert (t,(uint)(QTextEdit::RemoveSelected));
		// but then the html would be quoted. So I use the ugly
		// way: insert a marker, replace it in whole text of QTextEdit
		QString marker="R3PlAcEMeL4teR!";
		e->insert (marker);
		e->setText (e->text().replace(marker,t));
	} else
		e->setText(t);
}

void TextEditor::textExportAsASCII()
{
	QString text = NoteObj (e->text()).getNoteASCII();
	QString fn,s;
	if (!filenameHint.isEmpty())
	{
		if (!filenameHint.contains (".txt"))
			s=filenameHint+".txt";
		else	
			s=filenameHint;
	} else	
		s=QString::null;
	fn = QFileDialog::getSaveFileName( s, "VYM Note (ASCII) (*.txt);;All files (*)", this,"export note dialog",tr("Export Note to single file (ASCII)") );
	int ret=-1;

    if ( !fn.isEmpty() ) 
	{
		QFile file (fn);
		if (file.exists())
		{
			QMessageBox mb( "VYM",
				tr("The file ") + fn + 
				tr(" exists already. "
				"Do you want to overwrite it?"),
			QMessageBox::Warning,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape,
			QMessageBox::NoButton );
			mb.setButtonText( QMessageBox::Yes, tr("Overwrite") );
			mb.setButtonText( QMessageBox::No, tr("Cancel"));
			ret=mb.exec();
		}	
		if (ret==QMessageBox::Cancel)
			return;
			
		// save 
		if ( !file.open( IO_WriteOnly ) ) 
			statusBar()->message( QString("Could not write to %1").arg(filename),
						  statusbarTime );
		else
		{
			QTextStream t( &file );
			t << text;
			file.close();

			statusBar()->message( QString( "Note exported as %1" ).arg( fn ), statusbarTime );
		}
    }
}


void TextEditor::textPrint()
{
    printer->setFullPage(TRUE);
    if ( printer->setup( this ) ) 
	{
		QPainter p( printer );
		// Check that there is a valid device to print to.
		if ( !p.device() ) return;
		QPaintDeviceMetrics metrics( p.device() );
		int dpiy = metrics.logicalDpiY();
		int margin = (int) ( (2/2.54)*dpiy ); // 2 cm margins
		QRect body( margin, margin, metrics.width() - 2*margin, metrics.height() - 2*margin );
		QFont font( e->currentFont() );
		font.setPointSize( 10 ); // we define 10pt to be a nice base size for printing

		QSimpleRichText richText( e->text(), font,
					  e->context(),
					  e->styleSheet(),
					  e->mimeSourceFactory(),
					  body.height() );
		richText.setWidth( &p, body.width() );
		QRect view( body );
		int page = 1;
		do 
		{
			richText.draw( &p, body.left(), body.top(), view, colorGroup() );
			view.moveBy( 0, body.height() );
			p.translate( 0 , -body.height() );
			p.setFont( font );
			p.drawText( view.right() - p.fontMetrics().width( QString::number( page ) ),
				view.bottom() + p.fontMetrics().ascent() + 5, QString::number( page ) );
			if ( view.top()  >= richText.height() )
			break;
			printer->newPage();
			page++;
		} while (TRUE);
    }
}

void TextEditor::textEditUndo()
{
}

void TextEditor::toggleFonthint()
{
	setUpdatesEnabled (false);
	e->selectAll (true);
	if (!actionFormatUseFixedFont->isOn() ) 
		e->setCurrentFont (varFont);
	else	
		e->setCurrentFont (fixedFont);
	e->selectAll (false);
	setUpdatesEnabled (true);
	repaint();
}

void TextEditor::setFixedFont()
{
	bool ok;
	QFont font =QFontDialog::getFont(
                    &ok, fixedFont, this );
    if ( ok ) 
        // font is set to the font the user selected
		fixedFont=font;
}

void TextEditor::setVarFont()
{
	bool ok;
	QFont font =QFontDialog::getFont(
                    &ok, varFont, this );
    if ( ok ) 
        // font is set to the font the user selected
		varFont=font;
}

void TextEditor::textBold()
{
    e->setBold( actionTextBold->isOn() );
}

void TextEditor::textUnderline()
{
    e->setUnderline( actionTextUnderline->isOn() );
}

void TextEditor::textItalic()
{
    e->setItalic( actionTextItalic->isOn() );
}

void TextEditor::textFamily( const QString &f )
{
    e->setFamily( f );
}

void TextEditor::textSize( const QString &p )
{
    e->setPointSize( p.toInt() );
}


void TextEditor::textColor()
{
    QColor col = QColorDialog::getColor( e->color(), this );
    if ( !col.isValid() )
	return;
    e->setColor( col );
    QPixmap pix( 16, 16 );
    pix.fill( black );
    actionTextColor->setIconSet( pix );
}

void TextEditor::textAlign( QAction *a )
{
    if ( a == actionAlignLeft )
	e->setAlignment( AlignLeft );
    else if ( a == actionAlignCenter )
	e->setAlignment( AlignHCenter );
    else if ( a == actionAlignRight )
	e->setAlignment( AlignRight );
    else if ( a == actionAlignJustify )
	e->setAlignment( AlignJustify );
}

void TextEditor::textVAlign()
{
    if ( sender() == actionAlignSuperScript && actionAlignSuperScript->isOn()) {
	e->setVerticalAlignment( QTextEdit::AlignSuperScript);
    } else if (sender() == actionAlignSubScript && actionAlignSubScript->isOn()) {
	e->setVerticalAlignment( QTextEdit::AlignSubScript);
    } else {
	e->setVerticalAlignment( QTextEdit::AlignNormal);
    }
}


void TextEditor::fontChanged( const QFont &f )
{
    comboFont->lineEdit()->setText( f.family() );
    comboSize->lineEdit()->setText( QString::number( f.pointSize() ) );
    actionTextBold->setOn( f.bold() );
    actionTextItalic->setOn( f.italic() );
    actionTextUnderline->setOn( f.underline() );
}

void TextEditor::colorChanged( const QColor &c )
{
    QPixmap pix( 16, 16 );
    pix.fill( c );
    actionTextColor->setIconSet( pix );
}

void TextEditor::alignmentChanged( int a )
{
    if ( ( a == AlignAuto ) || ( a & AlignLeft ))
	actionAlignLeft->setOn( true );
    else if ( ( a & AlignHCenter ) )
	actionAlignCenter->setOn( true );
    else if ( ( a & AlignRight ) )
	actionAlignRight->setOn( true );
    else if ( ( a & AlignJustify ) )
	actionAlignJustify->setOn( true );
}

void TextEditor::verticalAlignmentChanged(int a) 
{
    if (a == QTextEdit::AlignSuperScript ) {
	actionAlignSuperScript->setOn(true);
	actionAlignSubScript->setOn(false);
    } else if (a == QTextEdit::AlignSubScript ) {
	actionAlignSuperScript->setOn(false);
	actionAlignSubScript->setOn(true);
    } else {
	actionAlignSuperScript->setOn(false);
	actionAlignSubScript->setOn(false);
    }
}



void TextEditor::enableActions()
{
	actionFileLoad->setEnabled(true);
	actionFileSave->setEnabled(true);
	actionFileSaveAs->setEnabled(true);
	actionFilePrint->setEnabled(true);
	actionEditUndo->setEnabled(true);
	actionEditRedo->setEnabled(true);
	actionEditCopy->setEnabled(true);
	actionEditCut->setEnabled(true);
	actionEditPaste->setEnabled(true);
	actionEditDeleteAll->setEnabled(true);
	actionEditConvertPar->setEnabled(true);
	actionEditJoinLines->setEnabled(true);
	actionFormatUseFixedFont->setEnabled(true);
}

void TextEditor::disableActions()
{
	actionFileLoad->setEnabled(false);
	actionFileSave->setEnabled(false);
	actionFileSaveAs->setEnabled(false);
	actionFilePrint->setEnabled(false);
	actionEditUndo->setEnabled(false);
	actionEditRedo->setEnabled(false);
	actionEditCopy->setEnabled(false);
	actionEditCut->setEnabled(false);
	actionEditPaste->setEnabled(false);
	actionEditDeleteAll->setEnabled(false);
	actionEditConvertPar->setEnabled(false);
	actionEditJoinLines->setEnabled(false);
	actionFormatUseFixedFont->setEnabled(false);
}


