/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <fstream>
#include <string>
#include <vector>
#include <set>
#include <queue>
#include <map>
#include <list>
#include <stack>
/* -- */
#include <QDir>
#include <QFileInfo>
#include <QStringList>
/* -- */
#include "common/Errors.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "parser/Line.hpp"
#include "parser/File.hpp"
#include "parser/Definition.hpp"
#include "parser/Parser.hpp"
#include "parser/ParserVite.hpp"
#include "parser/ParserPaje.hpp"
#include "parser/ParserEventPaje.hpp"
/* -- */
using namespace std;

ParserVite::ParserVite() {
    _is_finished = false;
    _is_canceled = false;
}

void ParserVite::parse(const string &filename, Trace &trace, bool finish_trace_after_parse){

    ParserPaje *parserpaje = new ParserPaje();
    
    set_file_to_parse(filename);

    try {
        parserpaje->parse(filename, trace, false);
    }
    catch(...) {
        finish();
        trace.finish();
        delete parserpaje;
        return ;
    }

    map<unsigned int, Definition> definitions = *parserpaje->get_definitions();

    Definition current_definition;

    // We get the directory name
    QFileInfo fileinfo(QString::fromStdString(filename));
    QDir      directory = fileinfo.absoluteDir();

    // We get all the filenames by searching them in the containers stored previously
    QStringList                      file_event_names;
    QString                          name;
    stack<Container *>               containers;
    const list <Container *>        *root_containers = trace.get_root_containers();
    const map<std::string, Value *> *extra_fields;
    String                          *filename_String = NULL;
    map<string, Value *>::const_iterator field;

    for (std::list<Container *>::const_iterator it = root_containers->begin();
         it != root_containers->end();
         ++ it) {
        containers.push(*it);
    }
    while (!containers.empty()) {
        Container *c = containers.top();
        containers.pop();
        extra_fields = c->get_extra_fields();
        field = extra_fields->find(string("Filename"));
        // Search the filename
        if (field != extra_fields->end()) {
            filename_String = (String *)(*field).second;
            name = QString::fromStdString(filename_String->to_string());
        }
        else {
            name = "";
        }
        if(name != "") {
            file_event_names << name;
        }
        // We add the children
        list<Container *>::const_iterator children_end = c->get_children()->end();
        for (list<Container *>::const_iterator i = c->get_children()->begin();
             i != children_end;
             ++ i) {
            containers.push(*i);
        }
    }

    QStringList::const_iterator it_end = file_event_names.constEnd();
    File event_file;
    Line event_line;
    string event_identity_string;
    unsigned int event_identity;

    for (QStringList::const_iterator it = file_event_names.constBegin() ; it != it_end ; ++ it) {

        string event_filename = (directory.absolutePath()+QDir::separator()+(*it)).toStdString();
        
        ParserEventPaje *parserevent = new ParserEventPaje();
        event_file.open(event_filename);
        
        if(!event_file.is_opened()) {
            Error::set(Error::_OPEN, Error::_WARNING);
            continue;
        }
        while(!(event_file.is_eof() || _is_canceled)) {
            event_file.get_line(event_line);
            
            if (!event_line.item(0, event_identity_string)) {
                continue; // We have \n 
            }

            sscanf(event_identity_string.c_str(), "%u", &event_identity);
            current_definition = definitions[event_identity];
            
            // If it is a end link we wait for the begin (PajeStartLink)
            if(current_definition.get_event_name() == "PajeEndLink") {
                _end_link.push_back(event_line);
                continue;
            }

            parserevent->store_event(current_definition,
                                     event_line,
                                     trace);
        }

        event_file.close();
        delete parserevent;
    }

    // We end the links
    ParserEventPaje *parserevent = new ParserEventPaje();
    const unsigned int number_of_links = _end_link.size();
    for(unsigned int i = 0 ; i < number_of_links ; i ++) {
        event_line = _end_link[i];
        event_line.item(0, event_identity_string);
        sscanf(event_identity_string.c_str(), "%u", &event_identity);
        
        current_definition = definitions[event_identity];
        parserevent->store_event(current_definition,
                                 event_line,
                                 trace);
    }

    delete parserevent;
    
    finish();

    if(finish_trace_after_parse) {
        trace.finish();
    }
    
    delete parserpaje;
}


void ParserVite::set_file_to_parse(const string &filename){
    _file_to_parse = filename;
}


float ParserVite::get_size_loaded() const{
    // TODO when multithread :) else we cannot determine it except in computing the sum of all the file sizes and storing already loaded ^^
    return 0.5;
}

bool ParserVite::is_end_of_parsing() const{
    return _is_finished;
}

void ParserVite::finish() {
    _is_finished = true;
}

void ParserVite::set_canceled() {
    _is_canceled = true;
}
