/*
 * ===========================
 * VDK Visual Development Kit
 * Version 1.0
 * Revision 7
 * January 2000
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
/*
OVERVIEW
--------
This file has the aim to be a footstep that shows how to make a 
gtk+ widget wrapper in vdk.
We choose here to wrap GtkExTextHighlight widget.
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if !HAVE_GNOME
  #if ENABLE_NLS
    #include <libintl.h>
#define _(str) gettext(str)
#define N_(str) str

  #else
    #define _(str) str
    #define N_(str) str 
  #endif
#else
 #include <gnome.h>
#endif
#include <vdkb/vdksynted.h>
#include <vdk/forms.h>
#include <stdio.h>
#include <sys/stat.h>
#include <vdkb/vdkb.h>
#include <ctype.h>
static Tipwin* tip_window = NULL;
static char buff[1024];
static char floating_token[256];
extern VDKBuilder* TheApp;
extern TokenList* tokenlist;
extern HintBTree* hint_tree;

/*
 */
static struct 
{ 
  char *key,*tpl;
} templates[] =  
{
  {"for"," ( ; ; ) { ; }"},
  {"while"," ( ) { ; }"},
  {"switch"," ( ) { case: ; break; default: ; break; }"},
  {"if"," ( ) { ; } else { ; }"},
  {"do"," { ; } while( );"},
  {NULL,NULL}
};

/*
*/
VDKHighlightTable::VDKHighlightTable(VDKSyntaxEditor* owner):
    list(NULL),table(NULL),owner(owner)
{
}
/*
*/
VDKHighlightTable::~VDKHighlightTable()
{
    if(table)
        gds_editor_table_free(table);
}
/*
*/
VDKPatternTable::VDKPatternTable(VDKSyntaxEditor* owner):
     VDKHighlightTable(owner)
{
}
/*
*/
void 
VDKPatternTable::Add( char *name,
                    char *regex,
                    VDKFont *font,
                    VDKColor *fore,
                    VDKColor *back)
{
  list = gds_editor_pattern_entry_new(list,
                                   name,
                                   regex,
                                   fore ? (GdkColor*) (*fore) : NULL,
                                   back ? (GdkColor*) (*back) : NULL,
                                   font ? (GdkFont*) (*font) : NULL,
                                   0,
                                   PATTERN_TABLE);

}

/*
*/

bool
VDKPatternTable::Install()
{
    if(list)
        table = gds_editor_pattern_table_new(list);
    if(table)
    {
        gds_editor_install_table(
            GDS_EDITOR(owner->ConnectingWidget()),table);
        return true;
    }
    else
        return false;
}

/*
*/
VDKSyntaxTable::VDKSyntaxTable(VDKSyntaxEditor* owner): 
    VDKHighlightTable(owner)
{

}

/*
*/
void 
VDKSyntaxTable::Add( char *name,
                  char *start,
                  char* end,
                  VDKFont *font,
                  VDKColor *fore,
                  VDKColor *back)
{
  list = gds_editor_syntax_entry_new(list,
                                   name,
                                   start,
                                   end,
                                   fore ? (GdkColor*) (*fore) : NULL,
                                   back ? (GdkColor*) (*back) : NULL,
                                   font ? (GdkFont*) (*font) : NULL,
                                   0,SYNTAX_TABLE);

}
/*
*/
bool
VDKSyntaxTable::Install()
{
    if(list)
        table = gds_editor_syntax_table_new(list);
    if(table)
    {
        gds_editor_install_table(
            GDS_EDITOR(owner->ConnectingWidget()),table);
        return true;
    }
    else
        return false;
}

/*
*/
VDKSyntaxEditor::VDKSyntaxEditor(VDKForm* owner):
    VDKExText(owner),
    Syntax("Syntax",this,false,&VDKSyntaxEditor::SetSyntax),
    TabSpaces("TabSpaces",this,4)

{
    LocalConnect();
    sTable = new VDKSyntaxTable(this);
    pTable = new VDKPatternTable(this);
    CurrentProp = NULL;
}


/*
*/
VDKSyntaxEditor::~VDKSyntaxEditor()
{
    if(sTable)
      delete sTable;
    if(pTable)
      delete pTable;
}

/*
 */
void 
VDKSyntaxEditor::ClearSyntaxAndPatternsTables()
{
  if(pTable->table)
    {
      gds_editor_table_free(pTable->table);
      pTable->list = NULL;
      pTable->table = NULL;
    }
  if(sTable->table)
    {
      gds_editor_table_free(sTable->table);
      sTable->list = NULL;
      sTable->table = NULL;
    }
}

/*
*/
void 
VDKSyntaxEditor::SetSyntax(bool flag)
{ 
    if(sigwid && flag && !sTable->table)
        sTable->Install();
    if(sigwid && flag && !pTable->table)
        pTable->Install();        
    if(sigwid)
      {
        gds_editor_set_highlight(GDS_EDITOR(sigwid), flag);
        gtk_widget_queue_draw(GTK_WIDGET(sigwid));
      }
}
/*
*/
void 
VDKSyntaxEditor::LocalConnect()
{

    VDKExText::LocalConnect();
      // handles tab
    gtk_signal_connect (GTK_OBJECT (sigwid), "key_press_event", 
			GTK_SIGNAL_FUNC (VDKSyntaxEditor::TabHandler), this);
    // handles tab
    gtk_signal_connect (GTK_OBJECT (sigwid), "key_release_event", 
			GTK_SIGNAL_FUNC (VDKSyntaxEditor::OnKeyRelease), this);
    // handle property mark
    gtk_signal_connect (GTK_OBJECT (sigwid), "property_mark", 
			GTK_SIGNAL_FUNC (VDKSyntaxEditor::MarkHandler), this);

}

/*
*/
void
VDKSyntaxEditor::MarkHandler (GtkWidget *widget,
			      GtkExTextProperty *prop,
			      gpointer gp)
{
  VDKSyntaxEditor* editor;
  g_return_if_fail (widget != NULL);
  g_return_if_fail (gp != NULL);
  editor = reinterpret_cast<VDKSyntaxEditor*>(gp);
  editor->CurrentProp = prop;
  if(tip_window)
    {
      tip_window->Close();
      tip_window->Destroy();
      tip_window = NULL;
    }
  *floating_token = '\0';
}
/*
 */
char*
VDKSyntaxEditor::ProcessMark(void)
{
  GtkExText *text = GTK_EXTEXT(sigwid);
  char* word = NULL;
  int start = -1,end = -1;
  start = gtk_editable_get_position(GTK_EDITABLE(text));
  // gets current or previous word
  gtk_extext_get_current_word(text,&start,&end);
  if(end < 0)
    gtk_extext_get_previous_word(text,&start,&end);
  if(start > 0 && end > 0)
    word =  gtk_extext_get_text(text,start,end);
  return word;
}

/*
  Overlaps src to tgt and return where overlaps ends.
  Overlap must be complete, returns NULL on failure
 */
static char* overlap(char* tgt, char* src)
{
  int t = 0;
  unsigned int z = strlen(src);
  if(z > strlen(tgt))
    return (char*) NULL;
  else
    {
      for(;src[t] && (src[t] == tgt[t]);t++) 
	;
      return t == z ? &tgt[t] : (char*) NULL;
    }
}
/*
 */
bool
VDKSyntaxEditor::MakeCompletion(char* word)
{
  TokenList local;
  TokenListIterator li(*tokenlist);
  for(;li;li++)
    {
      char* token = (char*) li.current();
      if(overlap(token,word))
	  local.add(li.current());
    }
  // one token found
  if(local.size() == 1)
    {
      char* token = (char*) local[0];
      char* p;
      if((strlen(word) < strlen(token)) &&
	 ( p = overlap(token,word)))
	  TextInsert(p);
    }
  // more than one token
  else if(local.size() > 1)
    {

      sprintf(buff,_("%2d more words:\n"),local.size());
      for(TokenListIterator li(local);li;li++)
	{
	  strcat(buff,(char*) li.current());
	  strcat(buff,"\n");
	}
      ShowTipWindow(buff);
    }
  // no tokens found, will be added to tokenlist
  else
    {
      sprintf(buff,_("\"%s\" isn't in word completion list.\n\
Hit ctrl-a to add it"),word);
      ShowTipWindow(buff);
      // copies word on floating buffer;
      strcpy(floating_token,word);
    }
  return local.size() > 0;
}
/*
 */
void
VDKSyntaxEditor::AddToken()
{
  if(!*floating_token)
    {
      sprintf(buff,_("Nothing to add to completion list"));
      ShowTipWindow(buff);
    }
  else
    {
      VDKString s(floating_token);
      if(!tokenlist->find(s))
	{
	  tokenlist->add(s);
	  sprintf(buff,_("Word: \"%s\" added to completion list"),floating_token);
	  ShowTipWindow(buff);
	}
      else
	{
	  sprintf(buff,_("%s already on completion list"),floating_token);
	  ShowTipWindow(buff);
	}
      *floating_token = '\0';
    }
}
/*
 */
bool
VDKSyntaxEditor::WriteTemplate(char* word)
{
  int t = 0;
  for(;templates[t].key;t++)
    {
      if(!strcmp(templates[t].key,word))
	{
	  if(CurrentProp)
	    Pointer = CurrentProp->endpos;
	  TextInsert(templates[t].tpl);
	  return true;
	}
    }
  return false;
}

/*
 */
void 
Tipwin::Setup(void) 
{
  VDKEventBox *vbox = new VDKEventBox(this,v_box);
  Add(vbox,0,1,1,0); 
  vbox->NormalBackground = VDKRgb(255,255,255);
  label = new VDKLabel(this,tip);
  vbox->Add(label,0,1,1,0);
}

/*
 */
char *
VDKSyntaxEditor::MakeTip(char* word)
{
  VDKHint hint(word,(char*) ""),*found;
  if( (found = hint_tree->find(hint)))
    sprintf(buff,found->hint);
  else
    sprintf(buff,_("%s: no tip available"),word);
  return buff;
}
/*
 */
void
VDKSyntaxEditor::ShowTipWindow(char* tip)
{
  if( tip && !tip_window)
    {
      int x, y;
      GtkExText *text = GTK_EXTEXT(sigwid);
      gdk_window_get_deskrelative_origin(GTK_WIDGET(text)->window,
					 &x,&y);
      x += text->cursor.x;
      y += text->cursor.y;
      tip_window = new Tipwin(Owner(),tip);
      tip_window->Setup();
      tip_window->Position = VDKPoint(x,y);
      tip_window->Show();
     }
}
/*
  some special key handling:
  - tab to smart indent
  - ctrl-tab or ctrl-a for words completion
  - ctrl-q for hints
  - ) or } for parenthesis matching
*/

static 
struct TIMERSTRUCT 
{ 
  GtkWidget* widget; 
  int match,pos; 
  char key;
  int timerid;
  bool insert;
} TimerStruct;

static bool timeron = false;

static int ParenMatch(GtkWidget* widget, int pos, char paren);

static int HandleTimeOut(gpointer gp)
{
  GtkEditableClass *klass;
  g_return_val_if_fail (gp != NULL,FALSE);
  struct TIMERSTRUCT* ts = reinterpret_cast<struct TIMERSTRUCT*>(gp);
  char key[2];
  key[0] = ts->key;
  key[1] = '\0';
  klass = GTK_EDITABLE_CLASS (GTK_OBJECT (ts->widget)->klass);
  klass->update_text (GTK_EDITABLE(ts->widget), ts->match, ts->match+1);
  gtk_extext_set_position(GTK_EDITABLE(ts->widget),ts->pos);
  if(ts->insert)
    gtk_extext_insert(GTK_EXTEXT(ts->widget),key,1);
  gtk_timeout_remove(ts->timerid);
  timeron = false;
  return FALSE;
}


/*
handle key release
for <alt> + key pad num strokes
*/
static char ansibuffer[32] = {'\0'};
static int ansicount = 0;
extern "C" {int GetEditorExtAsciiSupport(); };

extern int GetEditorExtAsciiSupport();
int
VDKSyntaxEditor::OnKeyRelease (GtkWidget *widget,
                             GdkEvent *ev,
                             gpointer gp)
{
  VDKSyntaxEditor* editor;
  GdkEventKey* event = NULL;
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (ev != NULL, FALSE);
  g_return_val_if_fail (gp != NULL, FALSE);
  editor = reinterpret_cast<VDKSyntaxEditor*>(gp);
  event = (GdkEventKey*) ev;
  bool isAlt = (event->keyval == GDK_Alt_L) &&
    (event->state & GDK_MOD1_MASK) ;
  bool isIndent = (event->keyval == GDK_KP_Enter || 
		  event->keyval == GDK_Return);
  VDKString Yes = CHECK_YES;
  bool autoindent = VDKBuilder::ideDefaults.project.code_autoindent == Yes;
  if(isAlt && *ansibuffer)
    {
      char key[2];
      int val = atoi(ansibuffer); 
      key[0] = val;
      key[1] = '\0';
      if(val <= 127 || GetEditorExtAsciiSupport())
	editor->TextInsert(key,1);
      else 
	{
	  char *p = (char*) g_malloc(8);
	  sprintf(p,"\\%o",atoi(ansibuffer));
	  editor->TextInsert(p);
	  g_free(p);
	}
      *ansibuffer = '\0';
      ansicount = 0;
      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
					  "key_release_event");
      return TRUE;
    }
  else if (isIndent && autoindent)
    {
      editor->NaiveIndent();
      return TRUE;
    }
  else
    return FALSE;
}
/*
  Originally was made to handle <tab> only,
  now handles more keystrokes, but the name 
  remained unchanged.
 */
int
VDKSyntaxEditor::TabHandler (GtkWidget *widget,
                             GdkEvent *ev,
                             gpointer gp)
{
    VDKSyntaxEditor* editor;
    char* word;
    GdkEventKey* event = NULL;
    g_return_val_if_fail (widget != NULL, FALSE);
    g_return_val_if_fail (ev != NULL, FALSE);
    g_return_val_if_fail (gp != NULL, FALSE);
    editor = reinterpret_cast<VDKSyntaxEditor*>(gp);
    event = (GdkEventKey*) ev;
    // destroy tip window if any
    if(tip_window)
      {
	tip_window->Close();
	tip_window->Destroy();
	tip_window = NULL;
      }
    // forces paren matching timeout during fast key presses
    if(timeron)
      HandleTimeOut(&TimerStruct);
    bool isCtrl  = event->state & GDK_CONTROL_MASK;
    bool isAlt = event->state & GDK_MOD1_MASK;
    bool isShift = event->state & GDK_SHIFT_MASK;
    // handles <alt> + key pad numeric sequence
    if (isAlt && 
	( event->keyval >= GDK_KP_0 && event->keyval <= GDK_KP_9) )
      {
	if(ansicount < (int) (sizeof(ansibuffer)-1))
	  {
	    ansibuffer[ansicount] = event->keyval - GDK_KP_0 + 48;
	    ansibuffer[ansicount+1] = '\0';
	    ansicount++;
	    // stops the signal here
	    // so won't reach underlying gtkextext
	    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
					  "key_press_event");
	  }
      }
    // handles paren match for () and {}
    else if (event->keyval == GDK_parenright || 
	event->keyval == GDK_braceright)
      // shows paren match and insert keystroke (last arg = true)
      return editor->ShowParenMatch(event->keyval,widget,true);
    // handles word completions and hints
    else if(event->keyval == GDK_Tab ||
	    (isShift && event->keyval == GDK_ISO_Left_Tab ) ||
	    (isCtrl && event->keyval == GDK_q) ||
	    (isCtrl && event->keyval == GDK_a) ||
	    (isCtrl && event->keyval == GDK_p)
	    )
      {
	// stops the signal here
	// so won't reach underlying gtkextext
	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
				      "key_press_event");
	// ctrl-q
	if(event->keyval == GDK_q)
	  {
	    word = editor->ProcessMark();
	    if(word)
	      {
		editor->ShowTipWindow(editor->MakeTip(word));
		g_free(word);
	      }
	    else
	      gdk_beep();
	  }
	// ctrl-a
	else if(event->keyval == GDK_a)
	  editor->AddToken();
	// ctrl-p
	// show paren match without inserting key
	else if(event->keyval == GDK_p)
	  {
	    int pos = gtk_editable_get_position(GTK_EDITABLE(widget));
	    char key =  gtk_extext_get_char_at_pos(GTK_EXTEXT(widget),
						   pos);
	    editor->ShowParenMatch(key,widget,false);
	  }
	// ctrl-tab
	else if(isCtrl)
	  {
	    word = editor->ProcessMark();
	    if(word && (!editor->WriteTemplate(word)) )
	      editor->MakeCompletion(word);
	    if(word)
	      g_free(word);
	  }
	// shift tab
	else if(event->keyval == GDK_ISO_Left_Tab)
	  // insert spaces as user have set
	  editor->NormalTab();
	// tab
	else
	  {
	    char* indent = 
	      (char*) VDKBuilder::ideDefaults.project.code_indent;
	    if(!strcmp(indent,CHECK_YES))
	      editor->NaiveIndent();
	    else
	      editor->DefaultIndent();
	  }
        return TRUE;
      }
    else
      return FALSE;
    return FALSE;
}

/*
 */
int
VDKSyntaxEditor::ShowParenMatch(char keyval,
				GtkWidget* widget,
				bool insert)
{
  int p = gtk_editable_get_position(GTK_EDITABLE(widget));
  int pos = insert ? p : p-1;
  int match;
  GtkExTextLineData *visible_line = NULL;
  GtkExTextLineData *match_line = NULL;
  // answers a match > 0 if a matching paren was found
  if ((match = ParenMatch(widget,pos,keyval)) > 0)
    {
      // stops the signal here
      // so won't reach underlying gtkextext
      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
				    "key_press_event");
      // check if matching position is visible or not
      visible_line = 
	gtk_extext_get_first_visible_line(GTK_EXTEXT(widget));
      match_line = 
	gtk_extext_get_line_by_char_pos(GTK_EXTEXT(widget),match);
      // is visible so enligths matching position
      // for a short period using a timeout.
      // timeout handling function will insert keystroke as well
      if( match_line->line_number >= visible_line->line_number)
	{
	  gtk_editable_select_region (GTK_EDITABLE(widget),
				      match,match+1);
	  timeron = true;
	  TimerStruct.widget = widget;
	  TimerStruct.match = match;
	  TimerStruct.pos = insert ? pos : pos+1;
	  TimerStruct.key = keyval;
	  TimerStruct.insert = insert;
	  TimerStruct.timerid = gtk_timeout_add(400,
						HandleTimeOut,&TimerStruct);
	}
      // matching position is not visible, prompts user
      else
	{
	  char key[2];
	  char* word = NULL;
	  GtkExTextLineData *prev_line = NULL;
	  key[0] = keyval;
	  key[1] = '\0';
	  gtk_extext_set_position(GTK_EDITABLE(widget),insert ? pos : pos+1);
	  if(insert)
	    gtk_extext_insert(GTK_EXTEXT(widget),key,1);
	  // prepare buffer
	  sprintf(buff,_("Match at line:%d\n"),
		  match_line->line_number+1);
	  // get line before matching one if any
	  if(match_line->line_number > 0)
	    prev_line = 
	      gtk_extext_get_line_by_char_pos(GTK_EXTEXT(widget),
					      match_line->startpos-1);
	  if(prev_line)
	    word = gtk_extext_get_text(GTK_EDITABLE(widget),
				       prev_line->startpos,
				       prev_line->endpos);
	  if(word)
	    {
	      strcat(buff,word);
	      g_free(word);
	    }
	  // get matching line
	  word = gtk_extext_get_text(GTK_EDITABLE(widget),
				     match_line->startpos,
				     match_line->endpos);
	  strcat(buff,word);
	  g_free(word);
	  ShowTipWindow(buff);
	}
      if(match_line)
	g_free(match_line);
      if(visible_line)
	g_free(visible_line);
      return TRUE;
    }
  else
    // most probably a paren mismatch, warns user
    {
      sprintf(buff,_("Humm.., probably a mismatch"));
      ShowTipWindow(buff);
      return FALSE; 
    }
}

/*
 */
int ParenMatch(GtkWidget* widget, int pos ,char paren)
{
  GtkExText* text = GTK_EXTEXT(widget);  
  char match;
  if(paren == GDK_parenright)
    match = GDK_parenleft;
  else if (paren == GDK_braceright)
    match = GDK_braceleft;
  else
    return 0;
  int stack = 1;
  do
    {
      char key = GTK_EXTEXT_INDEX(text,pos);
      if(key == match)
	stack--;
      else if (key == paren)
	stack++;
      if(stack > 0)
	pos--;
    } while( (stack > 0)  && (pos > 0));
  return pos;
}
/*
 */
static int
BraceRightInLine(GtkExText* text, int current_pos)
{
  int pos = 0;
  GtkExTextLineData * line = 
    gtk_extext_get_line_by_char_pos(text,current_pos);
  if(line)
    {
      int x = line->startpos;
      for(; x <= line->endpos; x++)
	{
	  if(GTK_EXTEXT_INDEX(text,x) == GDK_braceright) 
		      break;
	}
      pos = (x <= line->endpos) ? x - 1 : 0;
      g_free(line);
    }
  return pos;
}
/*
*/
static int
NearestBackwardBracePosition(GtkExText* text, int current_pos)
{
  // check if on the line there is a brace
  int pos = current_pos;
  GtkExTextLineData * line = gtk_extext_get_line_by_char_pos(text,pos);
  if(line)
    {
      int x = line->startpos;
      for(; x <= line->endpos; x++)
	  if(GTK_EXTEXT_INDEX(text,x) == GDK_braceright) 
	    break;
      pos = (x <= line->endpos) ? x - 1 : current_pos;
      g_free(line);
    }
  else
    pos = current_pos;
  // search previous nested matching {
  int stack = 0;
  do
    {
      char key = GTK_EXTEXT_INDEX(text,pos);
      if(key == GDK_braceleft)
	stack--;
      else if (key == GDK_braceright)
	stack++;
      if(stack >= 0)
	pos--;
    } while( (stack >= 0)  && (pos > 0));
  return pos;
}
/*
 */
static char* keywords[] =
{ "for","if","else","while","case","do","switch", 0 };

static bool 
in_keywords(char* key)
{
  for(int z = 0; keywords[z]; z++)
    {
      if(!strcmp(key,keywords[z]))
	return true;
    }
  return false;
}
/*
  searches a key word on previous line
 */
static int
NearestBackwardKeywordPosition(GtkExText* text, int current_pos)
{
  int pos = current_pos;

  GtkExTextLineData *line = gtk_extext_get_line_by_char_pos(text,pos);
  if(line)
    {
      int prev_line = line->line_number-1;
      if(prev_line >= 0)
	{
	  g_free(line);
	  gtk_extext_set_line(text,prev_line);  
	  line = gtk_extext_get_line_by_char_pos(text,
			 gtk_extext_get_position(GTK_EDITABLE(text)));
	  if(line)
	    pos = line->endpos;
	}
    }
  else
    return 0;

  GtkExTextProperty * prop =
      gtk_extext_property_nearest_backward(text,pos,NULL);
  while(prop && (prop->startpos >= line->startpos))
    {
      char* key = new char[prop->endpos - prop->startpos +1];
      char* propstring = gtk_extext_get_text(text,prop->startpos,prop->endpos);
      if(propstring && in_keywords(propstring))
	{
	  strcpy(key,propstring);
	  g_free(propstring);
	  printf("\nprop:%s",key);
	  fflush(stdout);
	  break;

	}
      delete [] key; 
      g_free(propstring);
      prop = prop->prev;
    }
  // restore position
  // gtk_extext_set_position(GTK_EDITABLE(text),current_pos);
  pos = prop && (prop->startpos >= line->startpos) ? prop->startpos: 0;
  g_free(line);
  return pos;
}
/*
 */
void
VDKSyntaxEditor::NaiveIndent(void)
{
  int spaces = TabSpaces, z = 0, len=0;
  char* tabs = NULL;
  char current_key = ' ';
  guint b_pos = 0,k_pos = 0,pos , iPos;
  GtkExText* text = GTK_EXTEXT(WrappedWidget());  
  // remember actual position
  guint current_pos = Pointer;
  guint desired_pos = 0;
  // search for a brace matching
  b_pos = NearestBackwardBracePosition(text, current_pos);
  if(!BraceRightInLine(text, current_pos))
    // search for a keyword match
    k_pos = NearestBackwardKeywordPosition(text, current_pos);
  // choose the nearest whichever comes first.
  pos = b_pos > k_pos ? b_pos : k_pos;
  // matched a closing brace or keyword
  if(pos > 0)
    {
      // set pointer to pos 
      Pointer = pos;
      // set pointer to column beginning
      SetColumn(0);
      // compute desired column offset
      desired_pos = pos - Pointer;
    }
  // restore current position
  Pointer = current_pos;
  // definetely no match (no brace nor keyword)
  if(pos <= 0)
    {
      DefaultIndent();
      return;
    }
  // move to beginning of present line
  pos = current_pos;
  SetColumn(0);
  iPos = Pointer;
  // counts how many white spaces there are from line beginning
  do
    {
      char key = GTK_EXTEXT_INDEX(text, iPos);
      if(key != ' ')
	{
	  current_key = key;
	  break;
	}
      iPos++;z++;
    } while(iPos < Length);
  // compute how many spaces have to be inserted/deleted
  // if spaces > 0  insert them
  // else if spaces < 0 delete them
  spaces = desired_pos - z + (current_key != GDK_braceright ? TabSpaces : 0);
  // restore pointer position
  Pointer = current_pos;
  // there are some space to insert/delete
  if(spaces != 0)
    {
      SetColumn(0);
      pos = Pointer;
    }
  else
    // no spaces to insert or delete, nothing to do
    return;
  // inserts spaces
  if(spaces > 0)
    {
      tabs = new char[spaces+1];
      memset(tabs,(char) 0, spaces+1); 
      memset(tabs,(char) ' ',spaces);
      TextInsert(tabs,spaces);
      delete[] tabs;
    }
  // delete spaces
  else
    gtk_editable_delete_text(GTK_EDITABLE(text),
			     GTK_EDITABLE(text)->current_pos,
			     GTK_EDITABLE(text)->current_pos+abs(spaces));
  // restore pointer to the right position
  Pointer = spaces >= 0 ? current_pos + spaces :
    pos + z - abs(spaces);
}


/*
*/
void
VDKSyntaxEditor::NormalTab(void)
{
  char* tabs = new char[TabSpaces+1];
  memset(tabs,(char) 0, TabSpaces+1); 
  memset(tabs,(char) ' ',TabSpaces);
  TextInsert(tabs,TabSpaces);
  delete[] tabs;  
}
/*
*/
void
VDKSyntaxEditor::DefaultIndent(void)
{
    int spaces = TabSpaces, z = 0, len=0;
    char* tabs = NULL,*buffer,*locbuff;
    // remeber actual position
    guint pos = Pointer;
    guint iPos;
    // move to beginning of present line
    SetColumn(0);
    iPos= Pointer;
    // gets all chars from iPos to pos
    locbuff = GetChars(iPos,pos);
    len = pos-iPos;
    buffer = new char[len+1];
    if(locbuff)
      {
        strcpy(buffer,locbuff);
	g_free(locbuff);
      }
    else
        *buffer = '\0';
    // counts how many white spaces there are from line beginning to pos
    while( (buffer[z]) && (isspace(buffer[z])) ) z++;
    delete[] buffer;
    // compute how many spaces to next tab
    spaces = (z > 0) ? spaces - (z%spaces): spaces;
    // inserts tabs only if needed
    if(spaces > 0)
        {
        tabs = new char[spaces+1];
        memset(tabs,(char) 0, spaces+1); 
        memset(tabs,(char) ' ',spaces);
        TextInsert(tabs,spaces);
        delete[] tabs;
        }
    // restablishes original position
    Pointer = pos+spaces;
}







