/*
 * ===========================
 * VDK Component Library
 * Version 0.2
 * ===========================
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 * ===========================================
 * This library is a component of:
 * VDK Visual Development Kit
 * Version 0.4.1
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 * ===========================================
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-130
 */

#include <vdk/chart.h>
#include <cstdlib>
#include <cstring>
/////////////////////////////////////
DEFINE_EVENT_LIST(VDKChart,VDKCanvas);
/*
captures configure event:
recompute chart allocated size
and redraw chart
 */
bool VDKChart::OnConfigure(VDKObject* sender, GdkEvent* event)
{
  if(!gc)
    gc = gdk_gc_new(pixmap);
  size = Usize;
  printf("\nsize:%d,%d",size.x,size.y);
  fflush(stdout);
  axis = ChartAxis(this,size.X(),size.Y());
  axis.Draw();
  DrawTitle();
  DrawChart();
  Redraw();
  return true;
}
/*
 */
bool VDKChart::OnClick(VDKObject* sender, GdkEvent* event)
{
  if(series.size() <= 0)
    return true;
  if(!tip_window)
    {
       char buff[64];
       GdkEventButton *ev = (GdkEventButton*) event;
       double x = ev->x;
       double y = ev->y;
       double sx = (x - xn1 + kx*xv1)/kx;
       double sy = (y - yn1 + ky*yv1)/ky;
       sprintf(buff,"%.3f,%.3f",sx,sy);
       tip_window = gtk_window_new (GTK_WINDOW_POPUP);
       gtk_window_position(GTK_WINDOW(tip_window),GTK_WIN_POS_MOUSE);
       GtkWidget* label = gtk_label_new(buff);
       gtk_container_add(GTK_CONTAINER(tip_window),label);
       gtk_widget_show(label);
       gtk_widget_show(tip_window);
    }
  return false;
}

/*
 */
bool VDKChart::OnClickRelease(VDKObject* sender, GdkEvent* event)
{
if(tip_window)
  {
    gtk_widget_destroy(tip_window);
    tip_window = NULL;
  }
return true;
}
/////////////////////////////////////////////////////////////
/*
constructor
 */
VDKChart::VDKChart(VDKForm* owner, int w = 100, int h = 100):
  VDKCanvas(owner,w,h),
  ChartBorder("ChartBorder",this,20,&VDKChart::SetChartBorder),
  Title("Title",this,"Untitled"),
  LabelX("LabelX",this,""),
  LabelY("LabelX",this,""),
  LabelXDigits("LabelXDigits",this,2),
  LabelYDigits("LabelYDigits",this,2)
{
  gc = NULL;
  tip_window = NULL;
  EventConnect("configure_event",&VDKChart::OnConfigure);
  EventConnect("button_press_event",&VDKChart::OnClick);
  EventConnect("button_release_event",&VDKChart::OnClickRelease);
  Font(new VDKFont(owner,"helvetica Medium 10"));
}

/*
set chart border
 */
void VDKChart::SetChartBorder(int b)
{
size = Usize;
axis = ChartAxis(this,size.X(),size.Y());
DrawChart();
}
/*
draw title
*/
void VDKChart::DrawTitle()
{
  VDKString s = Title;
  VDKPoint size = Usize;
  VDKPoint center(size.X()/2,ChartBorder/2);
  VDKFont* font = Font;
  int half_w = (gdk_string_width (font->AsGdkFont(), (char*) s))/2;
  VDKRgb fg = Foreground;
  if(fg.red >= 0)
    SetColor(fg);
  DrawString(center.X()-half_w,center.Y(), (char*) s);
}
/*
destructor:
frees series
and graphic context
 */
VDKChart::~VDKChart()
{
SeriesListIterator li(series);
for(;li;li++)
  delete li.current();
}
/*
  add serie to chart, unicity is checked,
  <s> will substitute actual series
  if a match is found. Old series will be destroyed
*/
void VDKChart::AddSeries(Series* s)
{
if(series.size() > 0)
  {
    SeriesListIterator li(series);
    for(;li;li++)
	if((*li.current() == *s))
	  break;
    if(li)
      {
	Series* p = li.current();
	// removes series li.current()
	series.remove(p);
	delete p;
      }
  }
 series.add(s);
 ComputeDomainLimits(s);
 DrawChart();
}
/*
update domain limits:
lowest and highest x and y
 */
void VDKChart::ComputeDomainLimits(Series* s)
{
if(series.size() == 1)
  {
    domainmax.x = s->Max().x;
    domainmax.y = s->Max().y;
    domainmin.x = s->Min().x;
    domainmin.y = s->Min().y;
  }
else
  {
    domainmax.x = domainmax.x < s->Max().x ? s->Max().x : domainmax.x;
    domainmax.y = domainmax.y < s->Max().y ? s->Max().y : domainmax.y;
    domainmin.x = domainmin.x > s->Min().x ? s->Min().x : domainmin.x;
    domainmin.y = domainmin.y > s->Min().y ? s->Min().y : domainmin.y;
  }
 domainmin.x = domainmin.x == domainmax.x ? 0 : domainmin.x;
 domainmin.y = domainmin.y == domainmax.y ? 0 : domainmin.y;

}
/*
clear chart, destroy series
*/
void VDKChart::Clear()
{
  SeriesListIterator li(series);
  for(;li;li++)
    delete li.current();
  series.flush();
  VDKCanvas::Clear(); 
  axis.Draw();
  DrawTitle();
  Redraw();
}
/*
actually draw chart:
1. Clear chart
2. Compute scaling factors
3. Calls a virtual drawing function: 
   Plot(Series* s, int n, Coord c) with scaled coordinates.
   VDKChart::Plot() does nothing.
   Subclasses should override Plot() in order 
   to draw different chart shapes (lines, bars, etc.)
*/
const int tickL = 8;
void VDKChart::DrawChart()
{
  if(series.size() <= 0)
    return;
  else
    {
      VDKCanvas::Clear(); 
      axis.Draw();
      DrawTitle();
    }
  // recompute domain with chart allocated size
  xn1 = axis.Domain().Origin().X();
  xn2 = xn1 + axis.Domain().W();
  xv1 = domainmin.x; xv2 = domainmax.x;
  //xv1 = xv1 == xv2 ? 0 : xv1;
  yn1 = axis.Domain().Origin().Y();
  yn2 = yn1 - axis.Domain().H();
  yv1 = domainmin.y; yv2 = domainmax.y;
  //yv1 = yv1 == yv2 ? 0 : yv1;
  kx  = (xn2-xn1)/(xv2-xv1);
  ky  = (yn2-yn1)/(yv2-yv1);
  // iterates over series
  // n-square algorithm :-(
  SeriesListIterator li(series);
  int z; 
  for(z=0;li;li++,z++)
    {
      int t;
      CoordListIterator lc(*li.current());
      // scales each point
      for(t = 0;lc;lc++,t++)
	{
	  VDKPoint p(
		     int(xn1+kx*(lc.current().x-xv1)),
		     int((yn1+ky*(lc.current().y-yv1)))
		     );
	  Plot(p,t,li.current());
	}
    }
  // draw x,y axis ticks
  DrawTicks();
  //
  DrawLabels();
  // redraw with no expose event
  Redraw();
}
/*
Draws x,y ticks (major and minor)
*/
void VDKChart::DrawTicks()
{
  char buff[32];
  // compute and draws x,y axis min,max value
  double lminx = xn1+kx*(domainmin.x-xv1);
  double lminy = yn1+ky*(domainmin.y-yv1);
  double lmaxx = xn1+kx*(domainmax.x-xv1);
  double lmaxy = yn1+ky*(domainmax.y-yv1);
  VDKFont* vdkfont = Font;
  GdkFont* font = vdkfont->AsGdkFont();
  int h = font->ascent + font->descent + 1;
  int w;
  // draws x ticks
  int t;
  VDKRgb fg = Foreground;
  if(fg.red >= 0)
    SetColor(fg);
  double j,lx,k = (domainmax.x-domainmin.x)/10.0;
  int x_digits = LabelXDigits;
  int y_digits = LabelYDigits;
  for(j = domainmin.x, t=0; j <= domainmax.x; j+=k,t++)
    {
      lx = xn1+kx*(j-xv1);
      if ( (t%2) == 0)
	{
	  sprintf(buff,"%.*f",x_digits,j);
	  w = gdk_string_width (font, buff);
	  DrawString(int(lx-w/2),int(lminy+tickL+h),buff);
	  // major tick
	  DrawLine(int(lx),int(lminy),int(lx),int(lminy+tickL));
	}
      else
	// minor tick
	DrawLine(int(lx),int(lminy),int(lx),int(lminy+tickL/2));
    }
  if(t <= 10)
    {
      sprintf(buff,"%.*f",x_digits,domainmax.x);
      w = gdk_string_width (font, buff);
      DrawString(int(lmaxx-w/2),int(lminy+tickL+h),buff); 
    }
  // draws y ticks
  double ly;
  k = (domainmax.y-domainmin.y)/10;
  for(j = domainmin.y, t=0; j <= domainmax.y ; j+=k,t++)
    {
      ly = yn1+ky*(j-yv1);
      if ( (t%2) == 0)
	{
	  sprintf(buff,"%.*f",y_digits,j);
	  w = gdk_string_width (font, buff);
	  DrawString(int(lminx-w-tickL),int(ly+h/3),buff);
	  // major tick
	  DrawLine(int(lminx),int(ly),int(lminx-tickL),int(ly));
	}
      else
	// minor tick
	DrawLine(int(lminx),int(ly),int(lminx-tickL/2),int(ly));
    }
  if(t <= 10)
    {
       sprintf(buff,"%.*f",y_digits,domainmax.y);
       w = gdk_string_width (font, buff);
       DrawString(int(lminx-w-tickL),int(lmaxy),buff);
    }
}
/*
draw x and y labels
*/
void VDKChart::DrawLabels()
{
  // x labels
  VDKString s = LabelX;
  VDKPoint size = Usize;
  VDKFont* vdkfont = Font;
  GdkFont* font = vdkfont->AsGdkFont();
  VDKRgb fg = Foreground;
  if(fg.red >= 0)
    SetColor(fg);
  if(! s.isNull())
    {
      VDKPoint center(size.X()/2,
		      axis.Domain().Origin().Y() + 
		      ChartBorder - 5);
      int half_w = (gdk_string_width (font, (char*) s))/2;
      DrawString(center.X()-half_w,center.Y(), (char*) s);
    }
  s = LabelY;
  if(! s.isNull())
    {
      int h = font->ascent+font->descent;
      int z = std::strlen((char*) s);
      int sh = h*z;
      char *p = (char*) s;
      VDKPoint center(axis.Domain().Origin().X() - 
		      ChartBorder + 5, size.Y()/2-sh/2);
      int t = 0;
      for(; t < z; t++,p++)
	DrawText(center.X(),center.Y()+t*h, p,1);
     
    }
}
/*
set drawing color.
Use this to set color for Plot()
function. (see VDKLineChart::Plot())
*/
void VDKChart::SetColor(VDKRgb rgb)
{
  GdkColor *color = (GdkColor*) malloc (sizeof (GdkColor));
  GdkColormap *colormap = gdk_window_get_colormap (Widget()->window);
  color->red    = rgb.red << 8;
  color->green  = rgb.green << 8;
  color->blue   = rgb.blue << 8;
  if (!gdk_color_alloc (colormap, color))
    gdk_color_black (colormap, color);
  gdk_gc_set_foreground(GC(), color);
  free(color);
}
/* 
set line style
Use this to set line style for Plot()
function. (see VDKLineChart::Plot())
*/
void VDKChart::SetLineAttributes(
				 gint lineWidth, 
				 GdkLineStyle lineStyle,
				 GdkCapStyle capStyle,
				 GdkJoinStyle joinStyle)
{
if(gc)
  gdk_gc_set_line_attributes(gc,
			     lineWidth,
			     lineStyle,
			     capStyle,
			     joinStyle);
}
//////////////////// CHART AXIS /////////////////////////////
/*
constructor:
compute axis origin and chart domain
based on chart size and chart border.
This will be surely changed by
configure event. (see OnConfigure())
*/
ChartAxis::ChartAxis(VDKChart* owner,int w, int h):
  owner(owner)
{
  domain = VDKRect(owner->ChartBorder,
		   h-owner->ChartBorder,
		   w-owner->ChartBorder*2,
		   h-owner->ChartBorder*2);
}
/*
copy-initializer
*/
ChartAxis::ChartAxis(ChartAxis& a)
{
  owner = a.owner;
  domain = a.domain;
}
/*
draws axis on owner canvas
*/
void ChartAxis::Draw()
{
if(!owner)
  return;
 VDKRgb fg = owner->Foreground;
 if(fg.red >= 0)
   owner-> SetColor(fg);
 owner->DrawLine(domain.Origin().X(),
		 domain.Origin().Y(),
		 domain.Origin().X(),
		 owner->ChartBorder);
 owner->DrawLine(domain.Origin().X(),
		 domain.Origin().Y(),
		 domain.Origin().X()+ domain.W(),
		 domain.Origin().Y());
}

/////////////////////// SERIES ///////////////////////
/*
add a point ot series,
computes on the fly
max and min x,y
*/
void Series::Add(double x, double y)
{
  if(size() < 1)
    {
      max.x = x; max.y = y;
      min.x = x; min.y = y;
    }
  else
    {
      max.x = max.x < x ? x : max.x;
      max.y = max.y < y ? y : max.y;
      min.x = min.x > x ? x : min.x;
      min.y = min.y > y ? y : min.y;
    }
  Coord c(x,y);
  CoordList::add(c);
}
/*
add a point array
 */
void Series::Add(double* x, double* y, int n)
{
int i;
for(i=0;i<n;i++)
  Add(x[i],y[i]);
}

///////////////////////// LINE CHART //////////////
/*
Plot a line chart
 */
void VDKLineChart::Plot(VDKPoint& p, int t, Series* series)
{
static int fx,fy;
// each time a new series should be plotted
// set canvas color and line attributes.
// This happens whenever t == 0
if(t == 0)
  {
    VDKRgb color = series->Color;
    SetColor(color);
    SetLineAttributes(
		      series->LineWidth,
		      series->LineStyle,
		      series->LineCapStyle,
		      series->LineJoinStyle);
    fx = p.X();
    fy = p.Y();
  }
 else
   {
     // pixmap is a protected member of VDKCanvas class
     if(pixmap)
       gdk_draw_line(pixmap, GC(), fx,fy,p.X(),p.Y());
     fx = p.X(); fy = p.Y();
   }
}
///////////////////////// SCATTERED CHART //////////////
/*
Plot a scattered chart
 */
void VDKScatteredChart::Plot(VDKPoint& p, int t, Series* series)
{ 
if(t == 0)
  {
    VDKRgb color = series->Color;
    SetColor(color);
    SetLineAttributes(
		      series->LineWidth,
		      series->LineStyle,
		      series->LineCapStyle,
		      series->LineJoinStyle);
  }
gdk_draw_rectangle(pixmap,GC(),TRUE,p.X()-2,p.Y()-2,4,4);
}
///////////////////////// BAR CHART //////////////
/*
Plot a bar chart
 */
void VDKBarChart::Plot(VDKPoint& p, int t, Series* series)
{ 
  if(t == 0)
    {
      VDKRgb color = series->Color;
      SetColor(color);
      SetLineAttributes(
			series->LineWidth,
			series->LineStyle,
			series->LineCapStyle,
			series->LineJoinStyle);
    }

  if(Labels)
    {
      char buff[64];
      VDKFont* vdkfont = Font;
      GdkFont* font = vdkfont->AsGdkFont();
      sprintf(buff,"%.1f",(p.Y() - yn1 + ky*yv1)/ky);
      int w = gdk_string_width (font, buff);
      gdk_draw_string(pixmap,font,GC(),p.X()- w/2,p.Y()-2,buff);
    }

  gdk_draw_rectangle(pixmap,GC(),TRUE,
		     p.X()-BarWidth/2,
		     p.Y(),
		     BarWidth,
		     axis.Domain().Origin().Y()-p.Y()
		     );
}







