/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "math.h"

#include "support.h"
#include "interface.h"
#include "gtk_main.h"
#include "visu_object.h"

#include "gtk_move.h"
#include "gtk_pick.h"
#include "gtk_interactive.h"
#include "gtk_renderingWindowWidget.h"
#include "openGLFunctions/interactive.h"
#include "extensions/marks.h"
#include "extraFunctions/plane.h"
#include "extraGtkFunctions/gtk_numericalEntryWidget.h"
#include "extraGtkFunctions/gtk_elementComboBox.h"
#include "coreTools/toolMatrix.h"
#include "visu_extension.h"

/**
 * SECTION: gtk_move
 * @short_description: The move tab in the interactive dialog.
 */

/* Callbacks. */
static void onNewData(GObject *obj, VisuData *dataObj, gpointer bool);
static void onDataReady(GObject *obj, VisuData *dataObj, gpointer bool);
static void refreshMoveAxesValues(VisuData *obj, VisuOpenGLView *view, gpointer data);
static void onMovePositionChanged(NumericalEntry *entry,
				  double oldValue, gpointer data);
static void onAxeChoosen(NumericalEntry *entry, double oldValue, gpointer data);
static void onMoveToOriginalClicked(GtkButton *button, gpointer data);
static void onRemoveNodeClicked(GtkButton *button, gpointer user_data);
static void onAddNodeClicked(GtkButton *button, gpointer user_data);
static void onMoveMethod(GtkToggleButton *toggle, gpointer data);
static void onPageEnter(GtkNotebook *notebook, GtkWidget *child,
			gint page_num, gpointer user_data);
static void onCheckDuplicate(GtkToggleButton *button, gpointer data);
static void onDuplicate(GtkButton *button, gpointer data);
static void onMoveToggled(GtkToggleButton *toggle, gpointer data);
static void onSpinBasis(GtkSpinButton *spin, gpointer data);
static gboolean removeHighlight(gpointer data);
static void onChooseBasis(GtkToggleButton *toggle, gpointer data);
static void onBasisSelected(VisuInteractive *inter, VisuInteractivePick pick,
			    VisuNode *nodes[3], gpointer data);
static void onApplyBasis(GtkButton *button, gpointer data);
static void onPopulationChanged(VisuData *dataObj, gint *nodes, gpointer data);
static void onPositionChanged(VisuData *dataObj, gpointer data);
static void onStartMove(VisuInteractive *inter, GList* nodeIds, gpointer data);
static void onMove(VisuInteractive *inter, float drag[3], gpointer data);
static void onGetAxisClicked(GtkButton *button, gpointer data);
static void onMoveClickStop(VisuInteractive *inter, gpointer data);
static void onPickClickStop(VisuInteractive *inter, gpointer data);

/* Local methods. */
static void setLabelsOrigin(VisuData *data, GList *nodeIds);
static void rebuildBasis(VisuData *dataObj);
static void drawBasisCell(VisuData *dataObj, float O[3], float mat[3][3]);

#define GTK_MOVE_INFO				\
  _("left-button\t\t\t\t: drag node(s) in the screen plane\n"	\
    "middle-button (wheel)\t\t: drag node(s) along specific axis\n"	\
    "shift-left-button\t\t\t: drag node(s) along x axis\n"	\
    "control-left-button\t\t\t: drag node(s) along y axis\n"	\
    "control-shift-left-button\t: drag node(s) along z axis\n"	\
    "right-button\t\t\t\t: switch to observe")
#define EXT_BASIS_ID "NewBasis"

/* Pick informations hook */
static VisuInteractive *interPick, *interMove;
static int movedNode = -1;
static float moveNodeOrigin[3];
#define GTK_MOVE_NO_NODE         _("(<i>none</i>)")
static gulong onSpin_id[4];
static VisuInteractiveId currentMode = interactive_move;
static guint currentAxe;

/* Widgets */
static GtkWidget *notebookAction;
static GtkWidget *observeWindow;
static GtkWidget *entryMoveXYZ[3];
static GtkWidget *entryAddXYZ[3];
static GtkWidget *entryAxeXYZ[3];
static GtkWidget *removeButton, *cancelButton;
static GtkWidget *comboElements;
static GtkWidget *labelNMoves;
static GtkWidget *radioMovePick, *radioMoveRegion;
enum
  {
    COLUMN_NAME,             /* The label shown */
    COLUMN_POINTER_TO_DATA,  /* Pointer to the VisuElement. */
    N_COLUMNS
  };
static GtkWidget *checkDuplicate, *comboDuplicate, *buttonDupplicate;
static GtkWidget *labelOriginX, *labelOriginY, *labelOriginZ;
static GtkWidget *labelScreenHorizontal, *labelScreenVertical;
static GtkWidget *spinABC[4], *toggleABC[4], *applyBasis;
static gchar *lblSpinABC[4] = {"orig.:", "X:", "Y:", "Z:"};
static gint prevBasis[4] = {0, 0, 0, 0};
static guint timeoutBasis[4];
static VisuExtension *extBasis;


/********************/
/* Public routines. */
/********************/
GtkWidget* gtkMoveBuild_interface(VisuUiMain *main, gchar **label,
				  gchar **help, GtkWidget **radio)
{
  GtkWidget *wd, *hbox, *vbox, *bt;
  VisuData *data;
  VisuDataIter dataIter;
  int i;
  VisuRenderingWindow *window;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  *label = g_strdup("Pick");
  *help  = g_strdup(GTK_MOVE_INFO);
  *radio = lookup_widget(main->interactiveDialog, "radioMove");
  g_signal_connect(G_OBJECT(*radio), "toggled",
		   G_CALLBACK(onMoveToggled), (gpointer)main->interactiveDialog);

  window = visu_uiMainClass_getDefaultRendering();
  data = visu_rendering_window_getVisuData(window);
  g_return_val_if_fail(data, (GtkWidget*)0);
  visu_data_iterNew(data, &dataIter);

  /* We create here the two interactives. */
  interMove = visuInteractiveNew(interactive_move);
  g_object_ref(G_OBJECT(interMove));
  g_signal_connect(G_OBJECT(interMove), "move",
		   G_CALLBACK(onMove), (gpointer)0);
  g_signal_connect(G_OBJECT(interMove), "start-move",
		   G_CALLBACK(onStartMove), (gpointer)window);
  g_signal_connect(G_OBJECT(interMove), "stop",
		   G_CALLBACK(onMoveClickStop), (gpointer)0);
  interPick = visuInteractiveNew(interactive_pick);
  g_object_ref(G_OBJECT(interPick));
  g_signal_connect(G_OBJECT(interPick), "node-selection",
		   G_CALLBACK(onBasisSelected), (gpointer)0);
  g_signal_connect(G_OBJECT(interPick), "stop",
		   G_CALLBACK(onPickClickStop), (gpointer)0);

  observeWindow = main->interactiveDialog;
  
  vbox = lookup_widget(main->interactiveDialog, "vbox21");

  notebookAction = lookup_widget(main->interactiveDialog, "notebookAction");
  g_signal_connect(G_OBJECT(notebookAction), "switch-page",
		   G_CALLBACK(onPageEnter), (gpointer)0);

  /* The move node action. */
  removeButton = gtk_button_new();
  gtk_box_pack_end(GTK_BOX(lookup_widget(main->interactiveDialog, "hbox72")),
                   removeButton, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(removeButton,
                              _("Suppress node (either picked one or selected ones)."));
  gtk_container_add(GTK_CONTAINER(removeButton),
                    gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(removeButton), "clicked",
		   G_CALLBACK(onRemoveNodeClicked), (gpointer)0);
  gtk_widget_show_all(removeButton);

  labelNMoves = lookup_widget(main->interactiveDialog, "labelNMoves");
  gtk_label_set_markup(GTK_LABEL(labelNMoves), GTK_MOVE_NO_NODE);

  radioMovePick = lookup_widget(main->interactiveDialog, "radioMovePick");
  gtk_widget_set_name(radioMovePick, "message_radio");
  radioMoveRegion = lookup_widget(main->interactiveDialog, "radioMoveRegion");
  gtk_widget_set_name(radioMoveRegion, "message_radio");

  labelOriginX = lookup_widget(main->interactiveDialog, "labelOriginalX");
  labelOriginY = lookup_widget(main->interactiveDialog, "labelOriginalY");
  labelOriginZ = lookup_widget(main->interactiveDialog, "labelOriginalZ");

  labelScreenHorizontal = lookup_widget(main->interactiveDialog, "labelHorizontalAxe");
  labelScreenVertical   = lookup_widget(main->interactiveDialog, "labelVerticalAxe");

  wd = lookup_widget(main->interactiveDialog, "tableMovePick");
  entryMoveXYZ[0] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[0]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[0], 1, 2, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  entryMoveXYZ[1] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[1]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[1], 3, 4, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  entryMoveXYZ[2] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[2]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[2], 5, 6, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  entryAxeXYZ[0] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAxeXYZ[0]), 11);
  gtk_table_attach(GTK_TABLE(wd), entryAxeXYZ[0], 1, 3, 1, 2,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  entryAxeXYZ[1] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAxeXYZ[1]), 11);
  gtk_table_attach(GTK_TABLE(wd), entryAxeXYZ[1], 3, 5, 1, 2,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  entryAxeXYZ[2] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAxeXYZ[2]), 11);
  gtk_table_attach(GTK_TABLE(wd), entryAxeXYZ[2], 5, 7, 1, 2,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  bt = gtk_button_new();
  gtk_table_attach(GTK_TABLE(wd), bt, 7, 8, 1, 2,
                   GTK_FILL, GTK_SHRINK, 0, 0);
  gtk_widget_set_tooltip_text(bt, _("Capture the perpendicular axis to the current view."));
  gtk_container_add(GTK_CONTAINER(bt),
                    gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT, GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(bt), "clicked",
		   G_CALLBACK(onGetAxisClicked), (gpointer)window);
  cancelButton = gtk_button_new();
  gtk_table_attach(GTK_TABLE(wd), cancelButton, 7, 8, 2, 3,
                   GTK_FILL, GTK_SHRINK, 0, 0);
  gtk_widget_set_tooltip_text(cancelButton, _("Return coordinates to initial values."));
  gtk_container_add(GTK_CONTAINER(cancelButton),
                    gtk_image_new_from_stock(GTK_STOCK_UNDO, GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(cancelButton), "clicked",
		   G_CALLBACK(onMoveToOriginalClicked), (gpointer)0);
  gtk_widget_show_all(wd);

  /* The add line. */
  wd = lookup_widget(main->interactiveDialog, "hboxAddNode");
  entryAddXYZ[0] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[0]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[0], FALSE, FALSE, 0);
  entryAddXYZ[1] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[1]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[1], FALSE, FALSE, 0);
  entryAddXYZ[2] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[2]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[2], FALSE, FALSE, 0);
  /* We create the structure that store the VisuElements */
  comboElements = elementComboBox_new(FALSE, FALSE, (const gchar*)0);
  gtk_box_pack_start(GTK_BOX(wd), comboElements, FALSE, FALSE, 0);
  gtk_box_reorder_child(GTK_BOX(wd), comboElements, 0);
  bt = gtk_button_new();
  gtk_box_pack_end(GTK_BOX(wd), bt, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(bt, _("Add a new node."));
  gtk_container_add(GTK_CONTAINER(bt),
                    gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(bt), "clicked",
		   G_CALLBACK(onAddNodeClicked), (gpointer)0);
  gtk_widget_show_all(wd);

  /* The duplicate line. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
  wd = gtk_label_new(_("<b>Duplicate nodes:</b>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 0);
  wd = gtk_label_new(_("<span size=\"smaller\">(the nodes listed in the pick tab)</span>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  gtk_widget_show_all(hbox);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  wd = gtk_check_button_new_with_mnemonic(_("du_plicate nodes as they are"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  checkDuplicate = wd;
  wd = gtk_label_new(_(" or as new: "));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  wd = elementComboBox_new(FALSE, FALSE, (const gchar*)0);
  gtk_widget_set_sensitive(wd, FALSE);
  comboDuplicate = wd;
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  wd = gtk_button_new_with_mnemonic(_("_duplicate"));
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  buttonDupplicate = wd;
  gtk_widget_show_all(hbox);

  /* The Basis line. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
  wd = gtk_label_new(_("<b>Change the basis set:</b>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  gtk_widget_show_all(hbox);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  for (i = 0; i < 4; i++)
    {
      wd = gtk_label_new(_(lblSpinABC[i]));
      gtk_misc_set_alignment(GTK_MISC(wd), 1., 0.5);
      gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 5);
      spinABC[i] = gtk_spin_button_new_with_range(0, dataIter.idMax, 1);
      onSpin_id[i] = g_signal_connect(G_OBJECT(spinABC[i]), "value-changed",
				      G_CALLBACK(onSpinBasis), GINT_TO_POINTER(i));
      gtk_box_pack_start(GTK_BOX(hbox), spinABC[i], FALSE, FALSE, 0);
      toggleABC[i] = gtk_toggle_button_new();
      gtk_button_set_relief(GTK_BUTTON(toggleABC[i]), GTK_RELIEF_NONE);
      gtk_box_pack_start(GTK_BOX(hbox), toggleABC[i], FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(toggleABC[i]),
			gtk_image_new_from_stock(GTK_STOCK_FIND,
						 GTK_ICON_SIZE_MENU));
      gtk_widget_set_tooltip_text(toggleABC[i],
                                  _("Select node by picking it on the rendering area."));
      g_signal_connect(G_OBJECT(toggleABC[i]), "toggled",
		       G_CALLBACK(onChooseBasis), GINT_TO_POINTER(i));
    }
  applyBasis = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), applyBasis, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(applyBasis),
		    gtk_image_new_from_stock(GTK_STOCK_APPLY, GTK_ICON_SIZE_MENU));
  gtk_widget_set_sensitive(applyBasis, FALSE);
  g_signal_connect(G_OBJECT(applyBasis), "clicked",
		   G_CALLBACK(onApplyBasis), (gpointer)0);
  gtk_widget_show_all(hbox);


  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);
  onNewData((GObject*)0, data, (gpointer)0);

  for (i = 0; i < 3; i++)
    {
      g_signal_connect(G_OBJECT(entryMoveXYZ[i]), "value-changed",
		       G_CALLBACK(onMovePositionChanged), GINT_TO_POINTER(i));
      g_signal_connect(G_OBJECT(entryAxeXYZ[i]), "value-changed",
		       G_CALLBACK(onAxeChoosen), (gpointer)0);
    }
  wd = lookup_widget(main->interactiveDialog, "radioMovePick");
  g_signal_connect(G_OBJECT(wd), "toggled",
		   G_CALLBACK(onMoveMethod), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "radioMoveRegion");
  g_signal_connect(G_OBJECT(wd), "toggled",
		   G_CALLBACK(onMoveMethod), (gpointer)window);

  g_signal_connect(G_OBJECT(checkDuplicate), "toggled",
		   G_CALLBACK(onCheckDuplicate), comboDuplicate);
  g_signal_connect(G_OBJECT(buttonDupplicate), "clicked",
		   G_CALLBACK(onDuplicate), (gpointer)0);

  extBasis = visu_extension_new(EXT_BASIS_ID, _("New basis set highlight"),
                                _("Draw a box representing the limit of the new basis set."),
                                visu_openGL_objectList_new(1), rebuildBasis);
  visu_extension_setPriority(extBasis, VISU_EXTENSION_PRIORITY_LOW);
  visuExtensions_add(extBasis);

  return (GtkWidget*)0;
}


static void setLabelsOrigin(VisuData *data, GList *nodeIds)
{
  gchar numero[256];
  VisuNode *node;

  if (nodeIds)
    {
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
	{
	  movedNode = GPOINTER_TO_INT(nodeIds->data);
	  node = visu_data_getNodeFromNumber(data, GPOINTER_TO_INT(nodeIds->data));
	  g_return_if_fail(node);

	  sprintf(numero, _("(node %d)"), GPOINTER_TO_INT(nodeIds->data) + 1);
	  gtk_label_set_markup(GTK_LABEL(labelNMoves), numero);
	  /* Set the origin position values. */
	  moveNodeOrigin[0] = node->xyz[0];
	  moveNodeOrigin[1] = node->xyz[1];
	  moveNodeOrigin[2] = node->xyz[2];
	  sprintf(numero, "<span size=\"small\">/ %5.2f</span>", moveNodeOrigin[0]);
	  gtk_label_set_markup(GTK_LABEL(labelOriginX), numero);
	  sprintf(numero, "<span size=\"small\">/ %5.2f</span>", moveNodeOrigin[1]);
	  gtk_label_set_markup(GTK_LABEL(labelOriginY), numero);
	  sprintf(numero, "<span size=\"small\">/ %5.2f</span>", moveNodeOrigin[2]);
	  gtk_label_set_markup(GTK_LABEL(labelOriginZ), numero);
	  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), node->xyz[0]);
	  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), node->xyz[1]);
	  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), node->xyz[2]);
	}
      else
	{
	  movedNode = -1;
	  sprintf(numero, _("(%d nodes)"), g_list_length(nodeIds));
	  gtk_label_set_markup(GTK_LABEL(labelNMoves), numero);
	  gtk_label_set_markup(GTK_LABEL(labelOriginX), "");
	  gtk_label_set_markup(GTK_LABEL(labelOriginY), "");
	  gtk_label_set_markup(GTK_LABEL(labelOriginZ), "");
	}
    }
  else
    {
      movedNode = -1;
      gtk_label_set_markup(GTK_LABEL(labelNMoves), GTK_MOVE_NO_NODE);
      gtk_label_set_markup(GTK_LABEL(labelOriginX), "");
      gtk_label_set_markup(GTK_LABEL(labelOriginY), "");
      gtk_label_set_markup(GTK_LABEL(labelOriginZ), "");
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
    }
  /* Set the sensitivity. */
  gtk_widget_set_sensitive(removeButton, (nodeIds != (GList*)0));
  gtk_widget_set_sensitive(cancelButton, (nodeIds != (GList*)0));
}

static void setMovingNodes()
{
  GList *nodeIds;

  DBG_fprintf(stderr, "Gtk Move: set the moving list.\n");

  nodeIds = (GList*)0;
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    nodeIds = gtkPickGet_nodeSelection();
  visuInteractiveSet_movingNodes(interMove, nodeIds);
  setLabelsOrigin((VisuData*)0, nodeIds);
  if (nodeIds)
    g_list_free(nodeIds);
}
/**
 * gtkMoveStart:
 * @window: a #VisuRenderingWindow object.
 *
 * Initialise a moving session.
 */
void gtkMoveStart(VisuRenderingWindow *window)
{
  DBG_fprintf(stderr, "Gtk Move: start the move panel.\n");
  renderingWindowPush_interactive(window,
				  (currentMode == interactive_move)?
				  interMove:interPick);

  if (currentMode == interactive_pick)
    renderingWindowPush_message(window, _("Pick a node with the mouse"));

  /* Set the moving list. */
  setMovingNodes();
}
/**
 * gtkMoveStop:
 * @window: a #VisuRenderingWindow object.
 *
 * Finalise a moving session.
 */
void gtkMoveStop(VisuRenderingWindow *window)
{
  DBG_fprintf(stderr, "Gtk Move: unset the move panel.\n");

  renderingWindowPop_interactive(window, (currentMode == interactive_move)?
				 interMove:interPick);

  if (currentMode == interactive_pick)
    renderingWindowPop_message(window);
}
static void onMove(VisuInteractive *inter _U_, float drag[3], gpointer data _U_)
{
/*   GList *nodes; */
  float val;
/*   VisuNode *node; */

  DBG_fprintf(stderr, "Gtk Move: callback on move action.\n");

  /* Modify the entry labels. */
  DBG_fprintf(stderr, "Gtk Move: update label of positions.\n");
  val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]));
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), val + drag[0]);
  val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]));
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), val + drag[1]);
  val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]));
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), val + drag[2]);
  return;
}
static void onStartMove(VisuInteractive *inter _U_, GList* nodeIds, gpointer data)
{
  DBG_fprintf(stderr, "Gtk Move: callback on start-move action.\n");
  setLabelsOrigin(visu_rendering_window_getVisuData(RENDERING_WINDOW(data)), nodeIds);
  return;
}
static void onMoveClickStop(VisuInteractive *inter _U_, gpointer data _U_)
{
  gtkInteractiveToggle();
}

/****************/
/* Private part */
/****************/

static void onPageEnter(GtkNotebook *notebook _U_, GtkWidget *child _U_,
			gint page_num, gpointer user_data _U_)
{
  gchar numero[256];
  GList *lst;

  if (page_num != 1)
    return;

  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      lst = gtkPickGet_nodeSelection();
      sprintf(numero, _("(%d nodes)"), g_list_length(lst));
      gtk_label_set_markup(GTK_LABEL(labelNMoves), numero);
      g_list_free(lst);
    }
}

static void onNewData(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLThetaPhiOmega",
		   G_CALLBACK(refreshMoveAxesValues), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		   G_CALLBACK(onPopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationIncrease",
		   G_CALLBACK(onPopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePositionChanged",
		   G_CALLBACK(onPositionChanged), (gpointer)0);
}

static void onDataReady(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  int i;
  VisuDataIter dataIter;

  if (dataObj)
    {
      DBG_fprintf(stderr, "Gtk Move: caught 'dataReadyForRendering' signal.\n");
      visu_data_iterNew(dataObj, &dataIter);
      for (i = 0; i < 4; i++)
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[i]), 0,
				  dataIter.idMax);
    }
  /* We remove the basisset drawing. */
  extBasis->used = FALSE;
}

static void refreshMoveAxesValues(VisuData *obj _U_, VisuOpenGLView *view, gpointer data)
{
  gint id;
  float x[3], y[3];
  char tmpChr[20];

  id = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebookAction));
  DBG_fprintf(stderr, "Gtk Move: refresh screen basis set.\n");
  DBG_fprintf(stderr, " | %d %d\n", id + 1, action_move);
  if (id + 1 == action_move || GPOINTER_TO_INT(data))
    {
      VisuOpenGLViewGet_screenAxes(view, x, y);
      sprintf(tmpChr, "(%4.2f;%4.2f;%4.2f)", x[0], x[1], x[2]);
      gtk_label_set_text(GTK_LABEL(labelScreenHorizontal), tmpChr);
      sprintf(tmpChr, "(%4.2f;%4.2f;%4.2f)", y[0], y[1], y[2]);
      gtk_label_set_text(GTK_LABEL(labelScreenVertical), tmpChr);
    }
}
static void onMoveMethod(GtkToggleButton *toggle, gpointer data _U_)
{
  if (!gtk_toggle_button_get_active(toggle))
    return;

  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);

  /* Set the moving list. */
  setMovingNodes();
}
static void onMovePositionChanged(NumericalEntry *entry, double oldValue,
				  gpointer data)
{
  float *valOfNode;
  VisuData *dataObj;
  VisuNode *node;
  GList *lst, *tmpLst;

  g_return_if_fail(GPOINTER_TO_INT(data) >= 0 && GPOINTER_TO_INT(data) < 3);

  /* We are not typing. */
  if (!gtk_widget_is_focus(GTK_WIDGET(entry)))
    return;

  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visu_data_getNodeFromNumber(dataObj, movedNode);
      g_return_if_fail(node);

      valOfNode = node->xyz + GPOINTER_TO_INT(data);
  
      if (*valOfNode == (float)numericalEntryGet_value(entry))
	return;

      *valOfNode = (float)numericalEntryGet_value(entry);
      visu_data_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);
      visu_data_emitNodePositionChanged(dataObj);
    }
  else
    {
      tmpLst = lst = gtkPickGet_nodeSelection();
      while(tmpLst)
	{
	  node = visu_data_getNodeFromNumber(dataObj, GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    node->xyz[GPOINTER_TO_INT(data)] +=
	      numericalEntryGet_value(entry) - oldValue;
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);

      visu_data_createAllNodes(dataObj);

    }

  VISU_ADD_REDRAW;
}
static void onMoveToOriginalClicked(GtkButton *button _U_, gpointer data _U_)
{
  VisuData *dataObj;
  VisuNode *node;
  float trans[3];
  GList *lst, *tmpLst;

  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());

  /* We are not typing. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visu_data_getNodeFromNumber(dataObj, movedNode);
      g_return_if_fail(node);

      node->xyz[0] = moveNodeOrigin[0];
      node->xyz[1] = moveNodeOrigin[1];
      node->xyz[2] = moveNodeOrigin[2];
      visu_data_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);

      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), moveNodeOrigin[0]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), moveNodeOrigin[1]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), moveNodeOrigin[2]);
    }
  else
    {
      trans[0] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]));
      trans[1] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]));
      trans[2] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]));

      tmpLst = lst = gtkPickGet_nodeSelection();
      while(tmpLst)
	{
	  node = visu_data_getNodeFromNumber(dataObj, GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    {
	      node->xyz[0] -= trans[0];
	      node->xyz[1] -= trans[1];
	      node->xyz[2] -= trans[2];
	    }
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);

      visu_data_createAllNodes(dataObj);

      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
    }
  visu_data_emitNodePositionChanged(dataObj);

  VISU_ADD_REDRAW;
}
static void onRemoveNodeClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  int *nodes;
  VisuData *dataObj;
  VisuNode *node;
  int iEle, i, n;
  GList *lst, *tmpLst;

  DBG_fprintf(stderr, "Gtk Observe/pick: remove the selected node %d.\n", movedNode);
  
  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visu_data_getNodeFromNumber(dataObj, movedNode);
      if (!node)
	return;
      iEle = node->posElement;

      nodes = g_malloc(sizeof(int) * 2);
      nodes[0] = movedNode;
      nodes[1] = -1;
      visu_data_removeNodes(dataObj, nodes);
      g_free(nodes);

      /* Copy the coordinates to the add entries. */
      for (i = 0; i < 3; i++)
	numericalEntrySet_value(NUMERICAL_ENTRY(entryAddXYZ[i]),
				numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[i])));
      elementComboBoxSet_selection(ELEMENT_COMBOX(comboElements),
                                   dataObj->fromIntToVisuElement[iEle]->name);

      visu_data_createNodes(dataObj, dataObj->fromIntToVisuElement[iEle]);
    }
  else
    {
      DBG_fprintf(stderr, "Gtk Move: get list of selected nodes.\n");
      tmpLst = lst = gtkPickGet_nodeSelection();
      n = g_list_length(lst);
      if (n > 0)
	{
	  nodes = g_malloc(sizeof(int) * (n + 1));
	  i = 0;
	  while(tmpLst)
	    {
	      nodes[i] = GPOINTER_TO_INT(tmpLst->data);
	      DBG_fprintf(stderr, " | will remove %3d %d\n", i, nodes[i]);
	      i += 1;
	      tmpLst = g_list_next(tmpLst);
	    }
	  nodes[i] = -1;
	  visu_data_removeNodes(dataObj, nodes);
	  g_free(nodes);
	}
      g_list_free(lst);

      visu_data_createAllNodes(dataObj);
    }
  /* We remove the node as selected one. */
  setLabelsOrigin((VisuData*)0, (GList*)0);

  VISU_ADD_REDRAW;
}
static void onAddNodeClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  VisuData *dataObj;
  GList *elements;
  VisuElement *element;
  float xyz[3];

  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());

  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(comboElements));
  g_return_if_fail(elements);
  element = (VisuElement*)elements->data;
  g_list_free(elements);

  xyz[0] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[0]));
  xyz[1] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[1]));
  xyz[2] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[2]));
  visu_data_addNodeFromElement(dataObj, element, xyz, FALSE, TRUE);

  /* We update all drawings. */
  visu_data_emitNodePositionChanged(dataObj);
  visu_data_createNodes(dataObj, element);
  VISU_ADD_REDRAW;
}
static void onCheckDuplicate(GtkToggleButton *button, gpointer data)
{
  gtk_widget_set_sensitive(GTK_WIDGET(data), !gtk_toggle_button_get_active(button));
}
static void onDuplicate(GtkButton *button _U_, gpointer data _U_)
{
  VisuData *dataObj;
  GList *tmpLst, *lst;
  gboolean custom;
  VisuElement *element;
  float coord[3], *trans;
  VisuNode *node;

  DBG_fprintf(stderr, "Gtk Move: duplicate selected nodes.\n");

  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());

  element = (VisuElement*)0;
  custom = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDuplicate));
  if (custom)
    {
      lst = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(comboDuplicate));
      g_return_if_fail(lst);
      element = (VisuElement*)lst->data;
      DBG_fprintf(stderr, "Gtk Move: change element to '%s'.\n", element->name);
      g_list_free(lst);
    }

  lst = gtkPickGet_nodeSelection();
  if (!lst)
    return;

  trans = visu_data_getXYZtranslation(dataObj);
  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      node = visu_data_getNodeFromNumber
	(dataObj, (guint)GPOINTER_TO_INT(tmpLst->data));
      visu_data_getNodePosition(dataObj, node, coord);
      coord[0] -= trans[0];
      coord[1] -= trans[1];
      coord[2] -= trans[2];
      if (element)
	visu_data_addNodeFromElement(dataObj, element, coord, FALSE, !tmpLst->next);
      else
	visu_data_addNodeFromIndex(dataObj, node->posElement, coord, FALSE, !tmpLst->next);
    }
  g_list_free(lst);
  g_free(trans);

  /* We update all drawings. */
  if (custom)
    visu_data_createNodes(dataObj, element);
  else
    visu_data_createAllNodes(dataObj);
  VISU_ADD_REDRAW;
}
static void onMoveToggled(GtkToggleButton *toggle, gpointer data)
{
  GtkWidget *wd;
  gboolean value;

  value = gtk_toggle_button_get_active(toggle);
  wd = lookup_widget(GTK_WIDGET(data), "alignment41");
  gtk_widget_set_sensitive(wd, value);
}
static gboolean setupBasisMatrix(VisuData *dataObj, float mat[3][3], float O[3])
{
  VisuNode *orig, *nodeA, *nodeB, *nodeC;
  float xyz[3];

  orig  =
    visu_data_getNodeFromNumber(dataObj, (guint)gtk_spin_button_get_value
                                (GTK_SPIN_BUTTON(spinABC[0])) - 1);
  nodeA =
    visu_data_getNodeFromNumber(dataObj, (guint)gtk_spin_button_get_value
                                (GTK_SPIN_BUTTON(spinABC[1])) - 1);
  nodeB =
    visu_data_getNodeFromNumber(dataObj, (guint)gtk_spin_button_get_value
                                (GTK_SPIN_BUTTON(spinABC[2])) - 1);
  nodeC =
    visu_data_getNodeFromNumber(dataObj, (guint)gtk_spin_button_get_value
                                (GTK_SPIN_BUTTON(spinABC[3])) - 1);
  g_return_val_if_fail(orig && nodeA && nodeB && nodeC, FALSE);

  visu_data_getNodePosition(dataObj, orig, O);
  visu_data_getNodePosition(dataObj, nodeA, xyz);
  mat[0][0] = xyz[0] - O[0];
  mat[1][0] = xyz[1] - O[1];
  mat[2][0] = xyz[2] - O[2];
  visu_data_getNodePosition(dataObj, nodeB, xyz);
  mat[0][1] = xyz[0] - O[0];
  mat[1][1] = xyz[1] - O[1];
  mat[2][1] = xyz[2] - O[2];
  visu_data_getNodePosition(dataObj, nodeC, xyz);
  mat[0][2] = xyz[0] - O[0];
  mat[1][2] = xyz[1] - O[1];
  mat[2][2] = xyz[2] - O[2];

  return TRUE;
}
static void onSpinBasis(GtkSpinButton *spin, gpointer data)
{
  int i;
  VisuRenderingWindow *window;
  VisuMarks *marks;
  gboolean valid;
  VisuData *dataObj;
  GList lst;
  float mat[3][3], O[3];
  
  i = GPOINTER_TO_INT(data);

  window = visu_uiMainClass_getDefaultRendering();
  dataObj = visu_rendering_window_getVisuData(window);
  marks = visu_rendering_window_getMarks(window);

  lst.next = (GList*)0;
  lst.prev = (GList*)0;
  /* Remove the previous one. */
  if (prevBasis[i] > 0)
    {
      lst.data = GINT_TO_POINTER(prevBasis[i] - 1);
      visu_marks_setHighlightedList(marks, &lst, MARKS_STATUS_UNSET);
    }
  if (timeoutBasis[i] > 0)
    g_source_remove(timeoutBasis[i]);

  prevBasis[i] = (gint)gtk_spin_button_get_value(spin);
  if (prevBasis[i] > 0 && visu_data_getNodeFromNumber(dataObj, prevBasis[i]))
    {
      lst.data = GINT_TO_POINTER(prevBasis[i] - 1);
      visu_marks_setHighlightedList(marks, &lst, MARKS_STATUS_SET);
      /* Add a new highlight. */
#if GLIB_MINOR_VERSION > 13
      timeoutBasis[i] = g_timeout_add_seconds(1, removeHighlight,
					      GINT_TO_POINTER(prevBasis[i]));
#else
      timeoutBasis[i] = g_timeout_add(1000, removeHighlight,
				      GINT_TO_POINTER(prevBasis[i]));
#endif
    }

  valid = TRUE;
  for(i = 0; i < 4; i++)
    valid = valid && (gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[i])) > 0.);
  gtk_widget_set_sensitive(applyBasis, valid);
  if (valid && setupBasisMatrix(dataObj, mat, O))
    {
      drawBasisCell(dataObj, O, mat);
      extBasis->used = TRUE;
      if (tool_matrix_determinant(mat) < 0.f)
        gtkInteractive_setMessage(_("The new basis set will be indirect."), GTK_MESSAGE_WARNING);
      else
        gtkInteractive_unsetMessage();
    }
  else
    extBasis->used = FALSE;
  
  VISU_FORCE_REDRAW;
}
static gboolean removeHighlight(gpointer data)
{
  int i;
  VisuMarks *marks;
  GList lst;

  i = GPOINTER_TO_INT(data);
  g_return_val_if_fail(i > 0, FALSE);

  marks = visu_rendering_window_getMarks(visu_uiMainClass_getDefaultRendering());

  /* Remove the previous one. */
  lst.next = (GList*)0;
  lst.prev = (GList*)0;
  lst.data = GINT_TO_POINTER(i - 1);
  visu_marks_setHighlightedList(marks, &lst, MARKS_STATUS_UNSET);

  VISU_FORCE_REDRAW;

  return FALSE;
}
static void onChooseBasis(GtkToggleButton *toggle, gpointer data)
{
  guint i;
  VisuRenderingWindow *window;
  gboolean valid;

  window = visu_uiMainClass_getDefaultRendering();

  DBG_fprintf(stderr, "Gtk Move: one toggle chooser has been toggled.\n");
  valid = gtk_toggle_button_get_active(toggle);
  if (valid)
    {
      currentMode = interactive_pick;
      currentAxe = GPOINTER_TO_INT(data);
      renderingWindowPush_interactive(window, interPick);
      renderingWindowPush_message(window, _("Pick a node with the mouse"));
    }
  else
    {
      currentMode = interactive_move;
      renderingWindowPop_interactive(window, interPick);
      renderingWindowPop_message(window);
    }

  for(i = 0; i < 4; i++)
    gtk_widget_set_sensitive(toggleABC[i], (!valid || (i == currentAxe)));
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "hbox72"), !valid);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "alignment44"), !valid);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "alignment27"), !valid);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "hboxAddNode"), !valid);
}
static void onBasisSelected(VisuInteractive *inter _U_, VisuInteractivePick pick _U_,
			    VisuNode *nodes[3], gpointer data _U_)
{
  g_return_if_fail(nodes[0]);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinABC[currentAxe]),
			    nodes[0]->number + 1);
  gtk_toggle_button_set_active
    (GTK_TOGGLE_BUTTON(toggleABC[currentAxe]), FALSE);
}
static void onPickClickStop(VisuInteractive *inter _U_, gpointer data _U_)
{
  int i;

  for(i = 0; i < 4; i++)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggleABC[i]), FALSE);
}
static void onApplyBasis(GtkButton *button _U_, gpointer data _U_)
{
  VisuData *dataObj;
  float matA[3][3], O[3];

  dataObj = visu_rendering_window_getVisuData(visu_uiMainClass_getDefaultRendering());
  if (setupBasisMatrix(dataObj, matA, O) && !visu_data_setNewBasis(dataObj, matA, O))
    gtkInteractive_setMessage(_("Cannot change the basis: given matrix is singular."),
                              GTK_MESSAGE_ERROR);
  extBasis->used = FALSE;
}
static void onPopulationChanged(VisuData *dataObj, gint *nodes _U_, gpointer data _U_)
{
  VisuDataIter iter;

  DBG_fprintf(stderr, "Gtk Move: caught 'population changed'.\n");
  visu_data_iterNew(dataObj, &iter);
  g_signal_handler_block(G_OBJECT(spinABC[0]), onSpin_id[0]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[0]), 0, iter.idMax + 1);
  g_signal_handler_unblock(G_OBJECT(spinABC[0]), onSpin_id[0]);

  g_signal_handler_block(G_OBJECT(spinABC[1]), onSpin_id[1]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[1]), 0, iter.idMax + 1);
  g_signal_handler_unblock(G_OBJECT(spinABC[1]), onSpin_id[1]);

  g_signal_handler_block(G_OBJECT(spinABC[2]), onSpin_id[2]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[2]), 0, iter.idMax + 1);
  g_signal_handler_unblock(G_OBJECT(spinABC[2]), onSpin_id[2]);

  g_signal_handler_block(G_OBJECT(spinABC[3]), onSpin_id[3]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[3]), 0, iter.idMax + 1);
  g_signal_handler_unblock(G_OBJECT(spinABC[3]), onSpin_id[3]);
}
static void onPositionChanged(VisuData *dataObj, gpointer data _U_)
{
  rebuildBasis(dataObj);
}
static void onAxeChoosen(NumericalEntry *entry _U_, double oldValue _U_,
			 gpointer data _U_)
{
  float axe[3];

  axe[0] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAxeXYZ[0]));
  axe[1] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAxeXYZ[1]));
  axe[2] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAxeXYZ[2]));

  visuInteractiveSet_movingAxe(interMove, axe);
}
static void onGetAxisClicked(GtkButton *button _U_, gpointer data)
{
  VisuData *dataObj;
  VisuOpenGLView *view;

  dataObj = visu_rendering_window_getVisuData(RENDERING_WINDOW(data));
  view = visu_data_getOpenGLView(dataObj);
  g_return_if_fail(view);
  
  numericalEntrySet_value(NUMERICAL_ENTRY(entryAxeXYZ[0]), view->camera->eye[0]);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryAxeXYZ[1]), view->camera->eye[1]);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryAxeXYZ[2]), view->camera->eye[2]);
}

/********************/
/* Drawing methods. */
/********************/
static void rebuildBasis(VisuData *dataObj)
{
  float mat[3][3], O[3];

  if (setupBasisMatrix(dataObj, mat, O))
    drawBasisCell(dataObj, O, mat);
}
static void drawBasisCell(VisuData *dataObj, float O[3], float mat[3][3])
{
  Plane *dataBox[7];
  int i, j;
  float ext[3], u[3], n[3], m[3], v[12][2][3], b[8][3];
  
  DBG_fprintf(stderr, "Gtk Move: draw the new basis cell.\n");

  /* We build the planes of the bounding box. */
  visu_data_getExtension(dataObj, ext);
  j = 0;
  for (i = 0; i < 3; i++)
    {
      u[0] = 0.f;
      u[1] = 0.f;
      u[2] = 0.f;
      u[i] = 1.f;
      visu_data_convertBoxCoordinatestoXYZ(dataObj, n, u);
      dataBox[j] = planeNew_undefined();
      planeSet_normalVector(dataBox[j], n);
      u[i] = 1 + ext[i];
      visu_data_convertBoxCoordinatestoXYZ(dataObj, m, u);
      planeSet_distanceFromOrigin(dataBox[j], sqrt(m[0] * m[0] + m[1] * m[1] + m[2] * m[2]));
      j += 1;
      dataBox[j] = planeNew_undefined();
      planeSet_normalVector(dataBox[j], n);
      u[i] = - ext[i];
      visu_data_convertBoxCoordinatestoXYZ(dataObj, m, u);
      planeSet_distanceFromOrigin(dataBox[j], -sqrt(m[0] * m[0] + m[1] * m[1] + m[2] * m[2]));
      j += 1;
    }
  dataBox[j] = (Plane*)0;
  /* We build the vertex array. */
  b[0][0] = O[0];
  b[0][1] = O[1];
  b[0][2] = O[2];
  b[1][0] = O[0] + mat[0][0];
  b[1][1] = O[1] + mat[1][0];
  b[1][2] = O[2] + mat[2][0];
  b[2][0] = O[0] + mat[0][0] + mat[0][1];
  b[2][1] = O[1] + mat[1][0] + mat[1][1];
  b[2][2] = O[2] + mat[2][0] + mat[2][1];
  b[3][0] = O[0] + mat[0][1];
  b[3][1] = O[1] + mat[1][1];
  b[3][2] = O[2] + mat[2][1];
  b[4][0] = O[0] + mat[0][2];
  b[4][1] = O[1] + mat[1][2];
  b[4][2] = O[2] + mat[2][2];
  b[5][0] = O[0] + mat[0][0] + mat[0][2];
  b[5][1] = O[1] + mat[1][0] + mat[1][2];
  b[5][2] = O[2] + mat[2][0] + mat[2][2];
  b[6][0] = O[0] + mat[0][0] + mat[0][1] + mat[0][2];
  b[6][1] = O[1] + mat[1][0] + mat[1][1] + mat[1][2];
  b[6][2] = O[2] + mat[2][0] + mat[2][1] + mat[2][2];
  b[7][0] = O[0] + mat[0][1] + mat[0][2];
  b[7][1] = O[1] + mat[1][1] + mat[1][2];
  b[7][2] = O[2] + mat[2][1] + mat[2][2];

  planesGet_intersection(dataBox, b[0], b[1], v[0][0], FALSE);
  planesGet_intersection(dataBox, b[1], b[0], v[0][1], FALSE);
  planesGet_intersection(dataBox, b[1], b[2], v[1][0], FALSE);
  planesGet_intersection(dataBox, b[2], b[1], v[1][1], FALSE);
  planesGet_intersection(dataBox, b[2], b[3], v[2][0], FALSE);
  planesGet_intersection(dataBox, b[3], b[2], v[2][1], FALSE);
  planesGet_intersection(dataBox, b[3], b[0], v[3][0], FALSE);
  planesGet_intersection(dataBox, b[0], b[3], v[3][1], FALSE);
  planesGet_intersection(dataBox, b[4], b[5], v[4][0], FALSE);
  planesGet_intersection(dataBox, b[5], b[4], v[4][1], FALSE);
  planesGet_intersection(dataBox, b[5], b[6], v[5][0], FALSE);
  planesGet_intersection(dataBox, b[6], b[5], v[5][1], FALSE);
  planesGet_intersection(dataBox, b[6], b[7], v[6][0], FALSE);
  planesGet_intersection(dataBox, b[7], b[6], v[6][1], FALSE);
  planesGet_intersection(dataBox, b[7], b[4], v[7][0], FALSE);
  planesGet_intersection(dataBox, b[4], b[7], v[7][1], FALSE);
  planesGet_intersection(dataBox, b[0], b[4], v[8][0], FALSE);
  planesGet_intersection(dataBox, b[4], b[0], v[8][1], FALSE);
  planesGet_intersection(dataBox, b[1], b[5], v[9][0], FALSE);
  planesGet_intersection(dataBox, b[5], b[1], v[9][1], FALSE);
  planesGet_intersection(dataBox, b[2], b[6], v[10][0], FALSE);
  planesGet_intersection(dataBox, b[6], b[2], v[10][1], FALSE);
  planesGet_intersection(dataBox, b[3], b[7], v[11][0], FALSE);
  planesGet_intersection(dataBox, b[7], b[3], v[11][1], FALSE);
  DBG_fprintf(stderr, "Gtk Move: new basis axes.\n");
  for (i = 0; i < 12; i++)
    DBG_fprintf(stderr, " | %6.2f x %6.2f x %6.2f -> %6.2f x %6.2f x %6.2f\n",
                v[i][0][0], v[i][0][1], v[i][0][2], v[i][1][0], v[i][1][1], v[i][1][2]);

  for (i = 0; i < 6; i++)
    g_object_unref(G_OBJECT(dataBox[i]));

  glDeleteLists(extBasis->objectListId, 1);
  glNewList(extBasis->objectListId, GL_COMPILE);
  glDisable(GL_LIGHTING);
  glDisable(GL_DITHER);
  glEnable(GL_BLEND);

  glLineWidth(1);
  glColor3f(1.f, 1.f, 1.f);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glEnable(GL_LINE_STIPPLE);
  glLineStipple(1, 57568);
  glBegin(GL_LINES);
  glVertex3fv(v[0][0]);
  glVertex3fv(v[0][1]);
  glVertex3fv(v[1][0]);
  glVertex3fv(v[1][1]);
  glVertex3fv(v[2][0]);
  glVertex3fv(v[2][1]);
  glVertex3fv(v[3][0]);
  glVertex3fv(v[3][1]);
  glVertex3fv(v[4][0]);
  glVertex3fv(v[4][1]);
  glVertex3fv(v[5][0]);
  glVertex3fv(v[5][1]);
  glVertex3fv(v[6][0]);
  glVertex3fv(v[6][1]);
  glVertex3fv(v[7][0]);
  glVertex3fv(v[7][1]);
  glVertex3fv(v[8][0]);
  glVertex3fv(v[8][1]);
  glVertex3fv(v[9][0]);
  glVertex3fv(v[9][1]);
  glVertex3fv(v[10][0]);
  glVertex3fv(v[10][1]);
  glVertex3fv(v[11][0]);
  glVertex3fv(v[11][1]);
  glEnd();
  glDisable(GL_LINE_STIPPLE);

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_CULL_FACE);
  glColor4f(0.f, 0.f, 0.f, 0.2f);
  glBegin(GL_POLYGON);
  glVertex3fv(b[0]);
  glVertex3fv(b[1]);
  glVertex3fv(b[2]);
  glVertex3fv(b[3]);
  glEnd();
  glBegin(GL_POLYGON);
  glVertex3fv(b[0]);
  glVertex3fv(b[1]);
  glVertex3fv(b[5]);
  glVertex3fv(b[4]);
  glEnd();
  glBegin(GL_POLYGON);
  glVertex3fv(b[0]);
  glVertex3fv(b[3]);
  glVertex3fv(b[7]);
  glVertex3fv(b[4]);
  glEnd();
  glBegin(GL_POLYGON);
  glVertex3fv(b[4]);
  glVertex3fv(b[5]);
  glVertex3fv(b[6]);
  glVertex3fv(b[7]);
  glEnd();
  glBegin(GL_POLYGON);
  glVertex3fv(b[6]);
  glVertex3fv(b[5]);
  glVertex3fv(b[1]);
  glVertex3fv(b[2]);
  glEnd();
  glBegin(GL_POLYGON);
  glVertex3fv(b[6]);
  glVertex3fv(b[7]);
  glVertex3fv(b[3]);
  glVertex3fv(b[2]);
  glEnd();
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);

  glEnable(GL_LIGHTING);
  glEnable(GL_DITHER);
  glEndList();
}
