/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToAscii.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <visu_tools.h>
#include <extraFunctions/extraNode.h>

/**
 * SECTION:dumpToAscii
 * @short_description: add an export capability of current positions.
 *
 * This provides a write routine to export V_Sim current
 * coordinates. It has several options to output or not hiddden nodes
 * or replicated nodes.
 */

static gboolean writeDataInAscii(FileFormat *format, const char* filename,
				 int width, int height, VisuData *dataObj,
				 guchar* imageData, GError **error,
				 voidDataFunc functionWait, gpointer data);
static gpointer waitData;
static voidDataFunc waitFunc;

DumpType* dumpToAscii_init()
{
  DumpType *ascii;
  char *typeASCII[] = {"*.ascii", (char*)0};
#define descrASCII _("Ascii file (current positions)")
  FileFormat* fmt;

  ascii = g_malloc(sizeof(DumpType));
  fmt = fileFormatNew(descrASCII, typeASCII);
  if (!fmt)
    {
      g_error("Can't initialize the ASCII dump module, aborting.\n");
    }

  ascii->bitmap = FALSE;
  ascii->fileType = fmt;
  ascii->writeFunc = writeDataInAscii;
  
  fileFormatAdd_propertyBoolean(fmt, "delete_hidden_nodes",
				_("Don't output hidden nodes"), FALSE);
  fileFormatAdd_propertyBoolean(fmt, "comment_hidden_nodes",
				_("Comment hidden nodes (if output)"), TRUE);
  fileFormatAdd_propertyBoolean(fmt, "expand_box",
				_("Keep primitive box (in case of node expansion)"), FALSE);
  fileFormatAdd_propertyBoolean(fmt, "reduced_coordinates",
				_("Export positions in reduced coordinates"), FALSE);
  fileFormatAdd_propertyBoolean(fmt, "angdeg_box",
				_("Export box as lengths and angles"), FALSE);
  fileFormatAdd_propertyBoolean(fmt, "type_alignment",
				_("Export nodes sorted by elements"), FALSE);

  waitData = (gpointer)0;
  waitFunc = (voidDataFunc)0;

  return ascii;
}

static gboolean writeDataInAscii(FileFormat *format, const char* filename,
				 int width _U_, int height _U_, VisuData *dataObj,
				 guchar* imageData _U_, GError **error,
				 voidDataFunc functionWait, gpointer data)
{
  gchar *nom, *prevFile;
  OpenGLView *view;
  char tmpChr;
  GList *prop;
  gchar* name;
  gboolean suppr, comment, expand, reduced, angdeg, eleSort;
  float xyz[3], ext[3], uvw[3], vertices[8][3], box[6];
  VisuDataIter iter;
  GString *output;
  const gchar *nodeComment;

  g_return_val_if_fail(error && !*error, FALSE);

  comment = TRUE;
  expand  = FALSE;
  suppr   = FALSE;
  reduced = FALSE;
  angdeg  = FALSE;
  eleSort = FALSE;
  prop = fileFormatGet_propertiesList(format);
  while (prop)
    {
      name = fileFormatGet_propertyName((FileFormatProperty*)(prop->data));
      if (strcmp(name, "comment_hidden_nodes") == 0)
	comment = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "expand_box") == 0)
	expand  = !fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "delete_hidden_nodes") == 0)
	suppr   = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "reduced_coordinates") == 0)
	reduced = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "angdeg_box") == 0)
	angdeg  = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "type_alignment") == 0)
	eleSort  = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      prop = g_list_next(prop);
    }

  waitData = data;
  waitFunc = functionWait;

  DBG_fprintf(stderr, "Dump ASCII: begin export of current positions...\n");

  output = g_string_new("");

  nom = visuDataGet_file(dataObj, 0, (FileFormat**)0);
  if (nom)
    {
      prevFile = g_path_get_basename(nom);
      g_string_append_printf(output, "# V_Sim export to ascii from '%s'\n", prevFile);
      g_free(prevFile);
    }
  else
    {
      g_warning("Can't get the name of the file to export.");
      g_string_append(output, "# V_Sim export to ascii\n");
    }

  view = visuDataGet_openGLView(dataObj);
  if (expand)
    visuDataGet_extension(dataObj, ext);
  else
    {
      ext[0] = 0.;
      ext[1] = 0.;
      ext[2] = 0.;
    }
  if (angdeg)
    {
      visuDataGet_boxVertices(dataObj, vertices, expand);
      box[0] = sqrt((vertices[1][0] - vertices[0][0]) *
		    (vertices[1][0] - vertices[0][0]) +
		    (vertices[1][1] - vertices[0][1]) *
		    (vertices[1][1] - vertices[0][1]) +
		    (vertices[1][2] - vertices[0][2]) *
		    (vertices[1][2] - vertices[0][2]));
      box[1] = sqrt((vertices[3][0] - vertices[0][0]) *
		    (vertices[3][0] - vertices[0][0]) +
		    (vertices[3][1] - vertices[0][1]) *
		    (vertices[3][1] - vertices[0][1]) +
		    (vertices[3][2] - vertices[0][2]) *
		    (vertices[3][2] - vertices[0][2]));
      box[2] = sqrt((vertices[4][0] - vertices[0][0]) *
		    (vertices[4][0] - vertices[0][0]) +
		    (vertices[4][1] - vertices[0][1]) *
		    (vertices[4][1] - vertices[0][1]) +
		    (vertices[4][2] - vertices[0][2]) *
		    (vertices[4][2] - vertices[0][2]));
      box[3] = acos(CLAMP(((vertices[3][0] - vertices[0][0]) *
			   (vertices[4][0] - vertices[0][0]) +
			   (vertices[3][1] - vertices[0][1]) *
			   (vertices[4][1] - vertices[0][1]) +
			   (vertices[3][2] - vertices[0][2]) *
			   (vertices[4][2] - vertices[0][2])) /
			  box[1] / box[2], -1.f, 1.f)) * 180.f / G_PI;
      box[4] = acos(CLAMP(((vertices[1][0] - vertices[0][0]) *
			   (vertices[4][0] - vertices[0][0]) +
			   (vertices[1][1] - vertices[0][1]) *
			   (vertices[4][1] - vertices[0][1]) +
			   (vertices[1][2] - vertices[0][2]) *
			   (vertices[4][2] - vertices[0][2])) /
			  box[0] / box[2], -1.f, 1.f)) * 180.f / G_PI;
      box[4] *= (vertices[4][2] < 0.)?-1.:+1.;
      box[5] = acos(CLAMP(((vertices[3][0] - vertices[0][0]) *
			   (vertices[1][0] - vertices[0][0]) +
			   (vertices[3][1] - vertices[0][1]) *
			   (vertices[1][1] - vertices[0][1]) +
			   (vertices[3][2] - vertices[0][2]) *
			   (vertices[1][2] - vertices[0][2])) /
			  box[0] / box[1], -1.f, 1.f)) * 180.f / G_PI;
    }
  else
    {
      box[0] = view->box->p2[0] * (1. + 2. * ext[0]);
      box[1] = view->box->p4[0] * (1. + 2. * ext[1]);
      box[2] = view->box->p4[1] * (1. + 2. * ext[1]);
      box[3] = view->box->p5[0] * (1. + 2. * ext[2]);
      box[4] = view->box->p5[1] * (1. + 2. * ext[2]);
      box[5] = view->box->p5[2] * (1. + 2. * ext[2]);
    }
  g_string_append_printf(output, "%15g %15g %15g\n", box[0], box[1], box[2]);
  g_string_append_printf(output, "%15g %15g %15g\n", box[3], box[4], box[5]);
  if (angdeg)
    g_string_append(output, "#keyword: angdeg\n");
  if (reduced)
    g_string_append(output, "#keyword: reduced\n");
  switch (visuDataGet_unit(dataObj))
    {
    case unit_angstroem:
      g_string_append(output, "#keyword: angstroem\n");
      break;
    case unit_bohr:
      g_string_append(output, "#keyword: atomic\n");
      break;
    default:
      break;
    }
  switch (visuDataGet_boundaryConditions(dataObj))
    {
    case BOX_SURFACE_XY:
    case BOX_SURFACE_YZ:
    case BOX_SURFACE_ZX:
      g_string_append(output, "#keyword: surface\n");
      break;
    case BOX_FREE:
      g_string_append(output, "#keyword: freeBC\n");
      break;
    default:
      break;
    }

  if (ext[0] != 0. || ext[1] != 0. || ext[2] != 0.)
    {
      g_string_append(output, "# Box is expanded, previous box size:\n");
      g_string_append_printf(output, "# %15g %15g %15g\n", view->box->p2[0], view->box->p4[0],
	      view->box->p4[1]);
      g_string_append_printf(output, "# %15g %15g %15g\n", view->box->p5[0], view->box->p5[1],
	      view->box->p5[2]);
      g_string_append(output, "# and box extension:\n");
      g_string_append_printf(output, "# %15g %15g %15g\n", ext[0], ext[1], ext[2]);
    }
  if (comment)
    {
      g_string_append(output, "# Statistics are valid for all nodes (hidden or not).\n");
      g_string_append(output, "# Hidden nodes are printed, but commented.\n");
    }
  visuDataIter_new(dataObj, &iter);
  g_string_append_printf(output, "# Box contains %d element(s).\n", iter.nElements);
  g_string_append_printf(output, "# Box contains %d nodes.\n", iter.nAllStoredNodes);
  for (visuDataIter_start(dataObj, &iter); iter.element;
       visuDataIter_nextElement(dataObj, &iter))
    g_string_append_printf(output, "#  | %d nodes for element '%s'.\n",
	    iter.nStoredNodes[iter.iElement], iter.element->name);
  if (suppr)
    g_string_append(output, "# Hidden nodes have been suppressed.\n");

  for (visuDataIter_startNumber(dataObj, &iter); iter.node && iter.element;
       (eleSort)?visuDataIter_next(dataObj, &iter):
       visuDataIter_nextNodeNumber(dataObj, &iter))
    {
      if ((iter.element->rendered && iter.node->rendered) || !comment)
	tmpChr = ' ';
      else
	tmpChr = '#';
      visuDataGet_nodePosition(dataObj, iter.node, xyz);
      if ((iter.element->rendered && iter.node->rendered) || !suppr)
	{
	  xyz[0] += ext[0] * view->box->p2[0] + ext[1] * view->box->p4[0] +
	    ext[2] * view->box->p5[0];
	  xyz[1] += ext[1] * view->box->p4[1] + ext[2] * view->box->p5[1];
	  xyz[2] += ext[2] * view->box->p5[2];
	  
	  if (reduced)
	    {
	      visuDataConvert_XYZtoBoxCoordinates(dataObj, uvw, xyz);
	      uvw[0] /= (1.f + 2.f * ext[0]);
	      uvw[1] /= (1.f + 2.f * ext[1]);
	      uvw[2] /= (1.f + 2.f * ext[2]);
	    }
	  else
	    {
	      uvw[0] = xyz[0];
	      uvw[1] = xyz[1];
	      uvw[2] = xyz[2];
	    }
	  g_string_append_printf(output, "%c%15g %15g %15g %s", tmpChr,
		  uvw[0], uvw[1], uvw[2], iter.element->name);
	  nodeComment = extraNodeGet_label(dataObj, iter.node);
	  if (nodeComment)
	    g_string_append_printf(output, " # %s", nodeComment);
	  g_string_append(output, "\n");
	}
    }
      
  g_file_set_contents(filename, output->str, -1, error);
  g_string_free(output, TRUE);

  if (*error)
    return FALSE;
  else
    return TRUE;
}
