/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_NODES_H
#define VISU_NODES_H

#include <glib.h>
#include <glib-object.h>

/**
 * node_struct:
 * @xyz: three floating point values that positions the node on x, y
 *       and z axis ;
 * @translation: an array of three floating point values that translates the
 *               node to its drawn position from (x, y, z) ;
 * @number: an integer that corresponds to its position in the entry file, it references
 *          also the node itself in the array 'fromNumberToVisuNode' of the #VisuData
 *          that contains the node ;
 * @posElement: an integer that is the position of the #VisuElement of the node
 *              in the array 'fromIntToVisuElement' of the #VisuData object that
 *              contains the node ;
 * @posNode: an integer that is the position of the node itself in the array
 *           'nodes' of the #VisuData object that contains the node ;
 * @rendered: a boolean to store if the node is drwn or not.
 *
 * Structure to store primary data of a node.
 */
struct node_struct
{
  /* coordinates of the node in cartesian coordinates. */
  float xyz[3];

  /* translation */
  float translation[3];

  /* Number of this element in the input file. */
  unsigned int number;
  /* Position in the #VisuData structure. */
  unsigned int posElement, posNode;

  /* A boolean to specify if this node is rendered or not. */
  gboolean rendered;
};
/**
 * VisuNode:
 *
 * Short way to address #node_struct objects.
 */
typedef struct node_struct VisuNode;

/**
 * VisuNodeProperty_struct:
 *
 * This structure defines a storage for one property for each node of a given
 * #VisuNodeArray. Use visuNodeNew_property() to create one property.
 */
struct VisuNodeProperty_struct;
/**
 * VisuNodeProperty:
 *
 * A convenient name for #VisuNodeProperty_struct structures.
 */
typedef struct VisuNodeProperty_struct VisuNodeProperty;

/**
 * VisuNodeArray:
 * @ntype: number of #VisuElement used in this object.
 * @idCounter: an internal counter used to give an id to new nodes.
 * @nodeTable: give the #VisuNode knowing its element number and
 *             its own position.
 * @nodeTableSize: the size of the previous array.
 * @nbOfAllStoredNodes: number of registered #VisuNode in this object.
 * @numberOfNodes: give the number of allocated #VisuNode for each
 *                 #VisuElement.
 * @numberOfStoredNodes: give the number of stored #VisuNode for
 *                       each #VisuElement.
 * @nodes: array that stores all the nodes.
 * @properties: a list of properties (see #VisuNodeProperty).
 *
 * This structure describes a set of nodes of different #VisuElement types.
 * It is optimized for quick access, allocation and reallocation.
 */
typedef struct nodes_struct
{
  /* Number of VisuElements used by these data. */
  unsigned int ntype;
  /* A counter. */
  unsigned int idCounter;
  /* This array gives access to the good VisuNode
     when one has its number. This number is an integer ranging in
     [0;idCounter[. This value is readable in the #VisuNode structure
     as the number attribute. The current allocated size is stored in
     @nodeTableSize. */
  VisuNode   **nodeTable;
  unsigned int nodeTableSize;
  /* The total of allocated VisuNodes. */
  unsigned int nNodes;
  /* The total of stored VisuNodes. */
  unsigned int nbOfAllStoredNodes;

  /* Number of nodes allocated (size of the nodes array) per VisuElement. */
  unsigned int* numberOfNodes;
  /* Number of nodes physically present in the array per VisuElement. */
  unsigned int* numberOfStoredNodes;
  /* Coordinates of all these nodes. */
  VisuNode **nodes;

  /* This is a table to store data, reachable with string keys.
     It should be accessed via visuNodeSet_property()
     and visuNodeGet_property(). */
  GHashTable *properties;

  /* A convenient pointer on the original node property. */
  VisuNodeProperty *origProp;
} VisuNodeArray;

/**
 * visuNodeNew_values:
 * @node: an allocated #VisuNode object ;
 * @xyz: the coordinates to set.
 * 
 * Set the coordinates and set all other values to default.
 */
void visuNodeNew_values(VisuNode *node, float xyz[3]);

/**
 * visuNodeCopy:
 * @nodeTo: an allocated #VisuNode object ;
 * @nodeFrom: an allocated #VisuNode object.
 * 
 * Copy all attributes of the object @nodeFrom to @nodeTo.
 */
void visuNodeCopy(VisuNode *nodeTo, VisuNode *nodeFrom);
/**
 * visuNodeSet_visibility:
 * @node: a #VisuNode object ;
 * @visibility: a boolean.
 *
 * This method is used to turn on or off the drawing of the specified node.
 *
 * Returns: true if the calling method should recreate the node list with
 * a call to the visuData_createAllNodes() method.
 */
int visuNodeSet_visibility(VisuNode* node, gboolean visibility);
/**
 * visuNodeGet_visibility:
 * @node: a #VisuNode object.
 *
 * This method is used get the status of the drawing state of a node.
 *
 * Returns: true if the node is rendered, false otherwise.
 */
gboolean visuNodeGet_visibility(VisuNode* node);


/**
 * visuNodeNew_nodes:
 * @nTypes: the size of nNodes.
 * @nNodes: an array giving the number of nodes per element.
 *
 * Create a new #VisuNodeArray structure, allocate all necessary values.
 *
 * Returns: a newly created #VisuNodeArray object.
 */
VisuNodeArray* visuNodeNew_nodes(unsigned int nTypes, unsigned int *nNodes);
/**
 * visuNodeRemove_nodes:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeNumbers: an array of integers (negative terminated).
 *
 * Remove the given #VisuNode from the @nodeArray. The properties
 * are also updated.
 */
void visuNodeRemove_nodes(VisuNodeArray *nodeArray, int *nodeNumbers);
/**
 * visuNodeRemove_allDuplicateNodes:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeNumbers: a location to create an arry of int.
 *
 * Remove all nodes that are not original in the box. The list of
 * removed nodes ids are stored in @nodeNumbers. This array is
 * allocated and should be freed with g_free(). If no nodes are
 * removed, this array is not allocated.
 *
 * Returns: TRUE if some nodes have been removed (and @nodeNumbers allocated).
 */
gboolean visuNodeRemove_allDuplicateNodes(VisuNodeArray *nodeArray,
					  int **nodeNumbers);
/**
 * visuNodeFree_nodes:
 * @nodeArray: a #VisuNodeArray object.
 *
 * Free the given object and all associated memory.
 */
void visuNodeFree_nodes(VisuNodeArray *nodeArray);
/**
 * visuNodeAllocate_newNodes:
 * @nodeArray: a #VisuNodeArray object ;
 * @iEle: the id of a #VisuElement to expand the number of allocated nodes ;
 * @step: the number of newly allocated nodes.
 *
 * When a new node is required, using visuNodeGet_newNode() or visuNodeGet_copyNode()
 * the storing arrays are expand automatically with a fixed increment. If the user
 * wants to control this increment, he should call this routine before the get() ones
 * with the appropriated @step value.
 */
void visuNodeAllocate_newNodes(VisuNodeArray *nodeArray, unsigned int iEle,
			       unsigned int step);

/**
 * visuNodeGet_newNode:
 * @nodeArray: a #VisuNodeArray object ;
 * @iEle: an integer between 0 and @nodeArray->ntypes - 1.
 *
 * Return the location of an unstored node for the given #VisuElement.
 * The returned node is then added in the list of used nodes.
 *
 * Returns: the location of a newly used node.
 */
VisuNode* visuNodeGet_newNode(VisuNodeArray *nodeArray, unsigned int iEle);
/**
 * visuNodeGet_copyNode:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a node of the given #VisuNodeArray.
 *
 * Return the location of an unstored node that is the deep copy of the given node.
 * The returned node is then added in the list of used nodes.
 *
 * Returns: the location of a newly used node.
 */
VisuNode* visuNodeGet_copyNode(VisuNodeArray *nodeArray, VisuNode *node);

/**
 * visuNodeTrace_property:
 * @array: a #VisuNodeArray object ;
 * @id: a property name.
 *
 * This is a debug method. It outputs on stderr the values for all
 * nodes of the property @id.
 */
void visuNodeTrace_property(VisuNodeArray *array, const gchar *id);


/*************************/
/* The property methods. */
/*************************/

/**
 * visuNodeNew_pointerProperty:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string ;
 * @freeFunc: a method to free each token (can be NULL).
 * @newAndCopyFunc: a method to create or copy each token.
 * @user_data: a user defined pointer that will be given to the free and copy routine.
 * 
 * This method creates and allocates a new area to store nodes associated data that
 * can be retrieve with the @key. These data are pointers on allocated memory
 * locations. When the property is removed with the #visuNodeFree_propertry (or the
 * associated #VisuNodeArray is free) the area is free and @freeFunc is called for
 * each token (or g_free() if @freeFunc is NULL).
 *
 * The method @newAndCopyFunc is used when the number of nodes is increased,
 * if the const gpointer of the GCopyFunc is not NULL, then we require a copy,
 * if it is NULL, then the routine must create a new token with default values.
 *
 * Returns: the newly created #VisuNodeProperty object.
 */
VisuNodeProperty* visuNodeNew_pointerProperty(VisuNodeArray* nodeArray,
					      const char* key, 
					      GFunc freeFunc,
					      GCopyFunc newAndCopyFunc,
					      gpointer user_data);
/**
 * visuNodeNew_intProperty:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string.
 * 
 * This method creates and allocates a new area to store nodes associated integer
 * values. This is the same than visuNodeNew_pointerProperty() but for static
 * integers instead of pointers as data.
 *
 * Returns: the newly created #VisuNodeProperty object.
 */
VisuNodeProperty* visuNodeNew_intProperty(VisuNodeArray* nodeArray,
					  const char* key);
/**
 * visuNodeFree_property:
 * @nodeArray: a #VisuNodeArray object.
 * @key: the name of the property to be removed.
 * 
 * This method free the given property and all associated data.
 */
void visuNodeFree_property(VisuNodeArray* nodeArray, const char* key);

/**
 * visuNodeGet_property:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string.
 *
 * This method is used to retrieve the node property associated to the given @key.
 *
 * Returns: a #VisuNodeProperty.
 */
VisuNodeProperty* visuNodeGet_property(VisuNodeArray* nodeArray, const char* key);

/**
 * visuNodeSet_propertyValue:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @key: a string ;
 * @value: A GValue pointer this the value to be stored.
 *
 * This method is used to store some values associated with
 * the given @node of the given @nodeArray. These values can be pointers to
 * anything allocated (will be free automatically when the property is deleted) or
 * they can be static values. This depends on the construction of the node property.
 * These values are described by the @key, and can be retrieved with the
 * visuNodeGet_propertyValue() method.
 *
 * See visuNodePropertySet_value() to directly set a value associated to a node.
 */
#define visuNodeSet_propertyValue(nodeArray, node, key, value)	\
  visuNodePropertySet_value(visuNodeGet_property(nodeArray, key), node, value)
/**
 * visuNodePropertySet_value:
 * @nodeProp: a #VisuNodeProperty object ;
 * @node: a #VisuNode object ;
 * @value: A GValue pointer this the value to be stored.
 *
 * This method is used to store some values associated with
 * the given @node of the given @nodeArray. These values can be pointers to
 * anything allocated (will be free automatically when the property is deleted) or
 * they can be static values. This depends on the construction of the node property.
 * These values can be retrieved with the visuNodePropertyGet_value() method.
 *
 * See visuNodeGet_property() to get a property by its name.
 */
void visuNodePropertySet_value(VisuNodeProperty* nodeProp, VisuNode* node,
			       GValue *value);
/**
 * visuNodeGet_propertyValue:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @key: a string ;
 * @value: an initialise GValue location.
 *
 * This method is used to retrieve some data associated to
 * the specified @node, stored in the given @data. These return data
 * should not be freed after used. The read value is stored in the given
 * GValue pointer. This GValue must be of the right type, depending on the
 * creation of the #VisuNodeProperty.
 *
 * Returns: some data associated to the key, stored the given GValue location.
 */
#define visuNodeGet_propertyValue(nodeArray, node, key, value)		\
  visuNodePropertyGet_value(visuNodeGet_property(nodeArray, key), node, value)
/**
 * visuNodePropertyGet_value:
 * @nodeProp: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @value: an initialise GValue location.
 *
 * This method is used to retrieve some data associated to
 * the specified @node, stored in the given @data. These return data
 * should not be freed after used. The read value is stored in the given
 * GValue pointer. This GValue must be of the right type, depending on the
 * creation of the #VisuNodeProperty.
 *
 * Returns: some data associated to the key, stored the given GValue location.
 */
GValue* visuNodePropertyGet_value(VisuNodeProperty* nodeProp, VisuNode* node,
				  GValue *value);

#endif
