/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#ifndef VISU_CONFIGFILE_H
#define VISU_CONFIGFILE_H

#include <glib.h>
#include "visu_data.h"

/**
 * VisuConfigFileEntry:
 *
 * This is the common name of the structure.
 */
typedef struct _VisuConfigFileEntry VisuConfigFileEntry;

/**
 * VISU_CONFIGFILE_PARAMETER:
 *
 * This defines a parameter entry in the config files.
 */
#define VISU_CONFIGFILE_PARAMETER 0
/**
 * VISU_CONFIGFILE_RESOURCE:
 *
 * This defines a resource entry in the config files.
 */
#define VISU_CONFIGFILE_RESOURCE 1

/**
 * visuConfigFileReadFunc:
 * @lines: an array of strings ;
 * @nbLines: an integer ;
 * @position: an integer ;
 * @dataObj: a #VisuData object ;
 * @error: a pointer to a GError pointer.
 *
 * This prototype corresponds to methods called when an entry is
 * found. The @lines argument is an array of lines read from the files.
 * These strings are copies and can be modified but not freed. There are
 * @nbLines and this value correspond to the number of lines defined
 * for the entry. The @error argument is used to store some text
 * messages and error ids. They should be in UTF8. The @error argument
 * must be initialised with (GError*)0. The @position argument give the number
 * of the first line given in @lines argument. If the @dataObj argument is not null,
 * some updates should be done with read values.
 *
 * Returns: TRUE if everything goes right, FALSE otherwise.
 */
typedef gboolean (*visuConfigFileReadFunc)(gchar **lines, int nbLines,
					   int position, VisuData *dataObj,
					   GError **error);
/**
 * visuConfigFileExportFunc:
 * @data: an empty GString to store the export ;
 * @nbLinesWritten: a pointer to store the number of lines written ;
 * @dataObj: a #VisuData object.
 *
 * This prototype defines a method that is used to export some resources
 * or parameters. The @data argument is an empty GString where the export has
 * to be written. If the argument @dataObj is not null, only resources related
 * to the #VisuData object should be exported (for parameters files, @dataObj is
 * always NULL).
 *
 * Returns: TRUE if everything goes right, FALSE otherwise.
 */
typedef gboolean (*visuConfigFileExportFunc)(GString *data, int *nbLinesWritten,
					     VisuData* dataObj);
/**
 * foreachFuncExport_struct:
 * @data: the string where the values are exported to ;
 * @nbLinesWritten: a pointer on an integer to be increased with the
 * number of written lines in @data.
 * @dataObj: the current #VisuData object, values are related to.
 *
 * This structure can be used to encapsulate the arguments of an export method
 * when used in a foreach glib loop.
 */
struct foreachFuncExport_struct
{
  GString *data;
  int *nbLinesWritten;
  VisuData *dataObj;
};


/**
 * visuConfigFileAdd_entry:
 * @kind: an integer ;
 * @key: a string (should not be NULL) ;
 * @description: a string (can be NULL) ;
 * @nbLines: an integer ;
 * @readFunc: a visuConfigFileReadFunc.
 *
 * This creates a new #VisuConfigFileEntry object with the given
 * values. The key and description arguments are copied.
 *
 * Returns: the newly created #VisuConfigFileEntry object.
 */
VisuConfigFileEntry* visuConfigFileAdd_entry(int kind, const gchar *key,
					     const gchar* description, int nbLines,
					     visuConfigFileReadFunc readFunc);
/**
 * visuConfigFileAdd_exportFunction:
 * @kind: an integer to identify resources or parameters ;
 * @writeFunc: a visuConfigFileExportFunc method.
 *
 * This stores the @writeFunc given. It will be called when resources or parameters
 * will be exported to disk.
 */
void visuConfigFileAdd_exportFunction(int kind, visuConfigFileExportFunc writeFunc);

/**
 * visuConfigFileSet_tag:
 * @entry: a #VisuConfigFileEntry object ;
 * @tag: a string.
 *
 * This method is used to set a tag to the given entry. This tag is used
 * to ignore or not the entry when the file is read. The @tag argument
 * is copied.
 */
void visuConfigFileSet_tag(VisuConfigFileEntry *entry, gchar *tag);
/**
 * visuConfigFileSet_version:
 * @entry: a #VisuConfigFileEntry object ;
 * @version: the version the entry appear in.
 *
 * Set the version number the entry appear in.
 */
void visuConfigFileSet_version(VisuConfigFileEntry *entry, float version);
/**
 * visuConfigFileSet_replace:
 * @newEntry: a #VisuConfigFileEntry object ;
 * @oldEntry: idem.
 *
 * Use this method to declare that @oldEntry has become obsolete and
 * has been replaced by @newEntry.
 */
void visuConfigFileSet_replace(VisuConfigFileEntry *newEntry,
			       VisuConfigFileEntry *oldEntry);

/**
 * visuConfigFileLoad:
 * @kind: an integer to identify the kind of file ;
 * @fileName: the path to file to read ;
 * @dataObj: a #VisuData object, sometime needed to update values (can
 * be NULL) ;
 * @error: a pointer to a GError pointer.
 *
 * Try to load the resources/parameters from the file name given in
 * parameter.
 *
 * Returns: TRUE if everything goes right. If @error is not NULL it
 *          should be freed with g_error_free().
 */
gboolean visuConfigFileLoad(int kind, const char* fileName,
			    VisuData *dataObj, GError **error);
/**
 * visuConfigFileSave:
 * @kind: an integer to identify the kind of file ;
 * @fileName: the path to file to read ;
 * @lines: a pointer to an integer (can be NULL) ;
 * @dataObj: a #VisuData object (can be NULL) ;
 * @error: a location to store a possible error.
 *
 * Try to export the resources/parameters to the file name given in
 * parameter. If @lines argument
 * is not NULL, and everything went right, it stores the number of written lines.
 * If the argument @dataObj is not null, only resources related
 * to the #VisuData object should be exported (for parameters files, @dataObj is
 * always NULL).
 *
 * Returns: TRUE if everything goes right.
 */
gboolean visuConfigFileSave(int kind, const char* fileName, int *lines,
			    VisuData *dataObj, GError **error);

/**
 * visuConfigFileAdd_knownTag:
 * @tag: a string (not nul or empty).
 *
 * If parameter entries have a tag, they are ignored except if their tag
 * has been declared using this method.
 */
void visuConfigFileAdd_knownTag(gchar* tag);

/**
 * visuConfigFileGet_validPath:
 * @kind: an integer identifier ;
 * @mode: a value from R_OK, W_OK and X_OK as described in unistd.h.
 * @utf8: if 1, the path is return in UTF-8 format, otherwise, the locale
 * of the file system is used.
 *
 * Test the entries of the hadoc list to find
 * a valid position to read or write a config file.
 * It tests access for the specified file.
 *
 * Returns: the first valid path find in the list of known paths.
 */
gchar* visuConfigFileGet_validPath(int kind, int mode, int utf8);
/**
 * visuConfigFileGet_nextValidPath:
 * @kind: an integer identifier ;
 * @accessMode: a value from R_OK, W_OK and X_OK as described in unistd.h ;
 * @list: a pointer to a valid *GList ;
 * @utf8: if 1, the path is return in UTF-8 format, otherwise, the locale
 * of the file system is used.
 *
 * Test the entries of the given list to find
 * a valid position to read or write a config file.
 * It tests access for the specified file. After a call to this
 * method the @list argument points to the next entry in the list, after
 * the one found.
 *
 * Returns: the first valid path find in the given list of paths.
 */
gchar* visuConfigFileGet_nextValidPath(int kind, int accessMode, GList **list, int utf8);
/**
 * visuConfigFileGet_defaultFileName:
 * @kind: an integer identifier.
 *
 * This methods is used to get the filename used for different
 * config files.
 *
 * Returns: the filename of config file. The returned *gchar is
 *          owned by V_Sim and should not be freed or modified.
 */
gchar* visuConfigFileGet_defaultFileName(int kind);
/**
 * visuConfigFileGet_pathList:
 * @kind: an integer identifier.
 *
 * V_Sim stores a list of paths where to look for resources or parameters
 * files, this method is used to get these lists.
 *
 * Returns: the list of the parameters or resources paths. This list is read-only.
 */
GList* visuConfigFileGet_pathList(int kind);


/**
 * visuConfigFile_init:
 *
 * Initialization, should not be called.
 *
 * Returns: 1 if everything goes right.
 */
int visuConfigFile_init();

/**
 * visuConfigFileGet_entries:
 * @kind: either #VISU_CONFIGFILE_PARAMETER or
 * #VISU_CONFIGFILE_RESOURCE ;
 *
 * This routine should be used for introspections purpose, to know
 * what resources or parameters are available.
 *
 * Returns: a #GList own by V_Sim.
 */
GList* visuConfigFileGet_entries(int kind);
/**
 * visuConfigFileExport_toXML:
 * @filename: a string in the encoding of the file system ;
 * @kind: either #VISU_CONFIGFILE_PARAMETER or
 * #VISU_CONFIGFILE_RESOURCE ;
 * @error: a location to store an error.
 *
 * Export all the registered entries for resources or parameters to an
 * XML file.
 *
 * Returns: TRUE if the file is written with success.
 */
gboolean visuConfigFileExport_toXML(const gchar *filename, int kind, GError **error);


#endif
