/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtk.h>
#include <glib.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */
#include <sys/stat.h>
#include <sys/types.h>

#include "interface.h"
#include "support.h"
#include "gtk_main.h"
#include "gtk_interactive.h"
#include "gtk_pairs.h"
#include "gtk_about.h"
#include "gtk_save.h"
#include "gtk_renderingWindowWidget.h"
#include "extraGtkFunctions/gtk_colorComboBoxWidget.h"

#include "panelModules/externalModules.h"
#include "panelModules/panelBrowser.h"
#include "panelModules/panelDataFile.h"
#include "panelModules/panelPlanes.h"
#include "panelModules/panelSurfaces.h"
#include "panelModules/gtkAtomic.h"
#include "panelModules/gtkSpin.h"

#include "visu_gtk.h"
#include "opengl.h"
#include "visu_object.h"
#include "visu_data.h"
#include "visu_configFile.h"
#include "visu_basic.h"            /* To have loadDataFromFile */
#include "visu_pairs.h"            /* To have visuPairSet_status */
#include "visu_commandLine.h"      /* To have getArgFilename and getXWindowGeometry*/
#include "visu_extension.h"
#include "visu_pickMesure.h"
#include "renderingBackend/visu_windowInterface.h"
#include "coreTools/toolOptions.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/dataFile.h"

/* Functions used to set some methods, related to gtkMain, such as a
 * custom load file chooser for each rendering method.
 */
VisuToolsInitFunc listInitRendenringGtkFunc[] = {
  panelElementAtomicInit_gtkMain,
  panelElementSpinInit_gtkMain,
  (VisuToolsInitFunc)0};

/* Store some informations on gtk method associated
   with different rendering methods. */
struct renderingMethod_gtkMain_struct
{
  GtkMainSetFilesFunc loadMethod;
};

enum {
  SPIN_BOUNDS_CHANGED_SIGNAL,
  LAST_SIGNAL
};

struct _GtkMainClass
{
  GtkWindowClass parent;

  guint signal_ids[LAST_SIGNAL];

  /* This hashtable associate a #RenderingMethod with some
     usefull Gtk informations, as stored in the
     #renderingMethod_gtkMain_struct structure. */
  GHashTable *renderingMethod_gtkMain;

  /* Store the last open directory. It is initialised
     to current working directory. */
  gchar *lastOpenDirectory;

  /* Gestion of the main window positioning. */
  gboolean rememberWindowPosition;
  GHashTable *windowPosition;

  /* Alert or not when quiting. */
  gboolean warningWhenQuit;
};

struct GtkMain_private_struct
{
  gboolean dispose_has_run;

  /* Pointers on different inside widgets. */
  GtkWidget *loadButton;
  GtkWidget *checkPairs;
  GtkWidget *pairsButton;
  GtkWidget *mouseActions;
  GtkWidget *saveButton;
  GtkWidget *quitButton;
  GtkWidget *aboutButton;
  GtkWidget *vboxMain;
};

/* Local object method. */
static void gtkMainBase_init     (GtkMainClass *klass);
static void gtkMainBase_finalise (GtkMainClass *klass);
static void gtkMainClass_init (GtkMainClass *klass);
static void gtkMain_init      (GtkMain *obj);
static void gtkMain_dispose   (GObject *obj);
static void gtkMain_finalize  (GObject *obj);

/* Local variables. */
static GtkWindowClass *parent_class = NULL;
static GtkMainClass   *my_class     = NULL;
static GtkMain *currentGtkMain      = NULL;

/* Miscelaneous functions with gtk */
static void setRenderingButtonSensitive(GtkMain *main, gboolean bool);
static void gtkMainHide_window(GtkWindow *win);
static void gtkMainShow_window(GtkWindow *win);


/* Local callbacks */
static void onLoadButtonClicked(GtkMain *main, GtkButton *button);
static void onPairsCheckToggled(GtkMain *main, GtkToggleButton *toggle);
static void onPairsButtonClicked(GtkMain *main, GtkButton *button);
static void onMouseActionsClicked(GtkMain *main, GtkButton *button);
static void onSaveButtonClicked(GtkMain *main, GtkButton *button);
static void onQuitButtonClicked(GtkMain *main, GtkButton *button);
static void onAboutButtonClicked(GtkMain *main, GtkButton *button);
static void onResourcesLoaded(GtkMain *main, VisuData *dataObj, gpointer data);
static void onRenderingChanged(GtkMain *main, RenderingMethod *method, gpointer data);
static void onFileChange(GtkMain *main, VisuData *dataObj, gpointer data);
static gboolean onKillPairsDialog(GtkMain *main, GdkEvent *event, gpointer data);
static gboolean onKillAboutDialog(GtkMain *main, GdkEvent *event, gpointer data);
static gboolean onKillInteractiveDialog(GtkMain *main, GdkEvent *event, gpointer data);
static gboolean onKillMainWindowEvent(GtkWidget *widget, GdkEvent *event,
					   gpointer user_data);

void onHideNextTime(GtkToggleButton *button, gpointer data);

/* Parameter to store the position of windows. */
#define PARAMETER_GTKMAIN_REMEMBER_DEFAULT TRUE

/* Parameter to change the policy of the warning quit dialog. */
#define FLAG_PARAMETER_GTKMAIN_QUIT    "main_confirmQuit"
#define DESC_PARAMETER_GTKMAIN_QUIT    "Show up a dialog to confirm when quit button is clicked ; boolean 0 or 1"
#define PARAMETER_GTKMAIN_QUIT_DEFAULT TRUE
static gboolean readMainQuitPolicy(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error);
#define FLAG_PARAMETER_GTKMAIN_PANEL   "main_panelStatus"
#define DESC_PARAMETER_GTKMAIN_PANEL   "Attach a panel to a tool window ; panel_name window_name (or None or Main)"
static gboolean readMainPanelStatus(gchar **lines, int nbLines, int position,
				    VisuData *dataObj, GError **error);
#define FLAG_PARAMETER_GTKMAIN_DOCK   "main_dock"
#define DESC_PARAMETER_GTKMAIN_DOCK   "Define the characteristic of a dock window ; visibility size(x,y) position(w,h) window_name"
static gboolean readMainDock(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static gboolean exportParametersGtkMain(GString *data, int *nbLinesWritten,
					VisuData *dataObj);


/* Number of time V_Sim try to attach the OpenGL context. */
#define GTK_MAIN_N_MAX_OPENGL 10

/*******************/
/* Object methods. */
/*******************/
GType gtkMain_get_type(void)
{
  static GType gtkMain_type = 0;

  if (!gtkMain_type)
    {
      static const GTypeInfo gtkMain_info =
      {
        sizeof (GtkMainClass),
        (GBaseInitFunc)gtkMainBase_init,
        (GBaseFinalizeFunc)gtkMainBase_finalise,
        (GClassInitFunc)gtkMainClass_init,
        NULL, /* class_finalise */
        NULL, /* class_data */
        sizeof (GtkMain),
        0, /* number of prealloc instances */
        (GInstanceInitFunc)gtkMain_init,
	NULL
      };
      gtkMain_type = g_type_register_static(GTK_TYPE_WINDOW, "GtkMain",
					    &gtkMain_info, 0);
      DBG_fprintf(stderr, "Gtk Main: creating the type GtkMain %d.\n",
		  (int)gtkMain_type);
    }

  return gtkMain_type;
}

static void gtkMainBase_init     (GtkMainClass *klass)
{
  DBG_fprintf(stderr, "Gtk Main: creating the class of the object.\n");

  DBG_fprintf(stderr, "                - allocate dynamic pointers ;\n");
  /* Allocate dynamical part of the class. */
  /* Create the hashtable of rendering method gtk functions. */
  klass->renderingMethod_gtkMain = g_hash_table_new_full(g_direct_hash,
							 g_direct_equal,
							 NULL, g_free);
  /* Retrieve the current working directory. */
  klass->lastOpenDirectory = g_get_current_dir();

  /* Create the hashtable to store windows and their position. */
  klass->windowPosition = g_hash_table_new_full(g_direct_hash,
						g_direct_equal,
						NULL, g_free);
}
static void gtkMainBase_finalise (GtkMainClass *klass)
{
  /* Free dynamical part of the class. */
  g_hash_table_destroy(klass->renderingMethod_gtkMain);
  if (klass->lastOpenDirectory)
    g_free(klass->lastOpenDirectory);
  g_hash_table_destroy(klass->windowPosition);
}

static void gtkMainClass_init(GtkMainClass *klass)
{
  GType paramFloat[1] = {G_TYPE_FLOAT};
  int i;
  VisuConfigFileEntry *resourceEntry;

  parent_class = g_type_class_peek_parent(klass);
  my_class     = klass;

  DBG_fprintf(stderr, "                - adding new signals ;\n");
  klass->signal_ids[SPIN_BOUNDS_CHANGED_SIGNAL] = 
    g_signal_newv ("spinBoundsChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__FLOAT,
		   G_TYPE_NONE, 1, paramFloat);

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose  = gtkMain_dispose;
  G_OBJECT_CLASS(klass)->finalize = gtkMain_finalize;

  /* Initialise internal variables. */
  for (i = 0; listInitRendenringGtkFunc[i]; i++)
    listInitRendenringGtkFunc[i]();

  /* Add the config file entries. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_QUIT,
					  DESC_PARAMETER_GTKMAIN_QUIT,
					  1, readMainQuitPolicy);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_PANEL,
					  DESC_PARAMETER_GTKMAIN_PANEL,
					  1, readMainPanelStatus);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_DOCK,
					  DESC_PARAMETER_GTKMAIN_DOCK,
					  1, readMainDock);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersGtkMain);

  /* Set the pixmap directory for the Glade stuff. */
  add_pixmap_directory(v_sim_pixmaps_dir);

  /* Set static parameters. */
  klass->rememberWindowPosition = PARAMETER_GTKMAIN_REMEMBER_DEFAULT;
  klass->warningWhenQuit        = PARAMETER_GTKMAIN_QUIT_DEFAULT;
}

static void gtkMain_init(GtkMain *obj)
{
  int width, height;
  int i;
  GdkPixbuf *obj_icon_pixbuf;
  GtkWidget *vbox, *hbox, *hbox2, *wd, *align, *image, *label;
  GtkTooltips *tooltips;
  ToolPanel *panel;
  DockWindow *dockMain;

  DBG_fprintf(stderr, "Gtk Main: initializing a new object (%p).\n",
	      (gpointer)obj);

  obj->private = g_malloc(sizeof(GtkMain_private));
  obj->private->dispose_has_run = FALSE;

  /* Public data. */
  obj->renderingWindow   = (GtkWidget*)0;
  obj->pairsDialog       = (GtkWidget*)0;
  obj->interactiveDialog = (GtkWidget*)0;
  obj->aboutDialog       = (GtkWidget*)0;

  /* Others elements linked to the main window. */
  DBG_fprintf(stderr, " | Gtk Pick and Observe\n");
  gtkInteractiveInit();
  DBG_fprintf(stderr, " | Gtk Pairs\n");
  gtkPairsInit();

  DBG_fprintf(stderr,"--- Initialising the rendering window ---\n");
  commandLineGet_XWindowGeometry(&width, &height);
  obj->renderingWindow =
    renderingWindow_new("V_Sim", "v_sim_render", "V_Sim", width, height);

  /* Private data. */
  DBG_fprintf(stderr, "--- Initialising the GTK interface ---\n");
  DBG_fprintf(stderr, " | Gtk Main");
  tooltips = gtk_tooltips_new ();

/*   main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL); */

  gtk_container_set_border_width(GTK_CONTAINER(obj), 7);
  gtk_window_set_title(GTK_WINDOW(obj), _("Command panel"));
  gtk_window_set_wmclass(GTK_WINDOW(obj), "v_sim_commandPanel", "V_Sim");
  obj_icon_pixbuf = create_pixbuf("icone-panel.png");
  if(obj_icon_pixbuf)
    {
      gtk_window_set_icon(GTK_WINDOW(obj), obj_icon_pixbuf);
      gdk_pixbuf_unref(obj_icon_pixbuf);
    }

  obj->private->vboxMain = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(obj), obj->private->vboxMain);
  gtk_widget_set_size_request(obj->private->vboxMain, 350, 550);

  wd = gtk_frame_new(_("Actions"));
  gtk_box_pack_end(GTK_BOX(obj->private->vboxMain), wd, FALSE, FALSE, 0);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(wd), vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

  obj->private->loadButton = gtk_button_new_from_stock(GTK_STOCK_OPEN);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->loadButton, TRUE, TRUE, 0);
  gtk_widget_set_sensitive(obj->private->loadButton, FALSE);
  gtk_tooltips_set_tip(tooltips, obj->private->loadButton,
		       _("Select a file to render."), NULL);

  align = gtk_alignment_new(0.5, 0.5, 1., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, TRUE, TRUE, 10);

  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(align), hbox2);

  obj->private->checkPairs = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hbox2), obj->private->checkPairs, FALSE, FALSE, 0);
  gtk_tooltips_set_tip(tooltips, obj->private->checkPairs,
		       _("Check to draw pairs between elements."), NULL);

  obj->private->pairsButton = gtk_button_new();
  gtk_widget_set_sensitive(obj->private->pairsButton, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox2), obj->private->pairsButton, TRUE, TRUE, 0);
  gtk_tooltips_set_tip(tooltips, obj->private->pairsButton,
		       _("Configure parameters for bindings such as color, thickness..."), NULL);

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->pairsButton), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_CONVERT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("_Pairs"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  obj->private->mouseActions = gtk_button_new();
  gtk_widget_set_sensitive(obj->private->mouseActions, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->mouseActions, TRUE, TRUE, 0);
  gtk_tooltips_set_tip(tooltips, obj->private->mouseActions,
		       _("Use the mouse to change the view and get position informations."), NULL);

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->mouseActions), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("Mouse _actions"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

  obj->private->saveButton = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->saveButton, TRUE, TRUE, 5);
  gtk_tooltips_set_tip(tooltips, obj->private->saveButton,
		       _("Click to save the parameters or the resources."), NULL);

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->saveButton), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_EDIT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("_Config. files"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  obj->private->quitButton = gtk_button_new_from_stock(GTK_STOCK_QUIT);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->quitButton, TRUE, TRUE, 5);

  obj->private->aboutButton = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->aboutButton, FALSE, FALSE, 15);
  gtk_tooltips_set_tip(tooltips, obj->private->aboutButton,
		       _("V_Sim program. Written by L. Billard, modified by D. Caliste."), NULL);
  gtk_button_set_relief(GTK_BUTTON(obj->private->aboutButton), GTK_RELIEF_NONE);

  image = create_pixmap(GTK_WIDGET(obj), "logo_petit.png");
  gtk_container_add(GTK_CONTAINER(obj->private->aboutButton), image);

  g_signal_connect_swapped(G_OBJECT(obj->private->loadButton), "clicked",
			   G_CALLBACK(onLoadButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->checkPairs), "toggled",
			   G_CALLBACK(onPairsCheckToggled), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->pairsButton), "clicked",
			   G_CALLBACK(onPairsButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->mouseActions), "clicked",
			   G_CALLBACK(onMouseActionsClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->saveButton), "clicked",
			   G_CALLBACK(onSaveButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->quitButton), "clicked",
			   G_CALLBACK(onQuitButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->aboutButton), "clicked",
			   G_CALLBACK(onAboutButtonClicked), (gpointer)obj);
  g_signal_connect(G_OBJECT(obj), "delete-event",
		   G_CALLBACK(onKillMainWindowEvent), (gpointer)0);
  g_signal_connect(G_OBJECT(obj), "destroy-event",
		   G_CALLBACK(onKillMainWindowEvent), (gpointer)0);
  g_signal_connect_swapped(G_OBJECT(visu), "resourcesLoaded",
			   G_CALLBACK(onResourcesLoaded), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(visu), "dataReadyForRendering", 
			   G_CALLBACK(onFileChange), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(visu), "renderingChanged",
			   G_CALLBACK(onRenderingChanged), (gpointer)obj);

  DBG_fprintf(stderr, " ... OK.\n");

  /* init the sub panel contains. */
  dockMain = toolPanelClassGet_commandPanel();
  wd = dockWindowGet_container(dockMain);
  gtk_box_pack_start(GTK_BOX(obj->private->vboxMain), wd, TRUE, TRUE, 0);

  /* Show all. */
  gtk_widget_show_all(obj->private->vboxMain);

  for (i = 0; panelListAll[i]; i++)
    {
      panel = panelListAll[i]();
      if (!panel)
	{
	  g_error("Can't initialise subpanel number %d.\n", i);
	}

      toolPanelAttach(panel, dockMain);
      
      DBG_fprintf(stderr, "Gtk Main : initialise '%s' subpanel OK.\n",
		  toolPanelGet_label(panel));
    }

}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void gtkMain_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Gtk Main: dispose object %p.\n", (gpointer)obj);

  if (GTK_MAIN(obj)->private->dispose_has_run)
    return;

  GTK_MAIN(obj)->private->dispose_has_run = TRUE;

  /* Chain up to the parent class */
  G_OBJECT_CLASS(parent_class)->dispose(obj);
}
/* This method is called once only. */
static void gtkMain_finalize(GObject* obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk Main: finalize object %p.\n", (gpointer)obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Gtk Main: freeing ... OK.\n");
}

GtkWidget* gtkMainNew()
{
  GtkMain *commandPanel;

  commandPanel = GTK_MAIN(g_object_new(GTK_MAIN_TYPE, NULL));

  if (!commandPanel)
    return (GtkWidget*)0;

  return GTK_WIDGET(commandPanel);
}


/*******************/
/* Local callbacks */
/*******************/
static void onLoadButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  renderingWindowLoad_file(RENDERING_WINDOW(main->renderingWindow), GTK_WINDOW(main));
}
static void onPairsCheckToggled(GtkMain *main, GtkToggleButton *toggle)
{
  gboolean res;
  VisuData *dataObj;

  res = gtk_toggle_button_get_active(toggle);
  visuPairSet_status(res);
  if (res)
    {
      dataObj = renderingWindowGet_visuData(RENDERING_WINDOW(main->renderingWindow));
      if (dataObj && visuPairBuild(dataObj))
	g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  else
    g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onPairsCloseClicked(GtkMain *main, GtkButton *button _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->pairsDialog));
}
static gboolean onKillPairsDialog(GtkMain *main, GdkEvent *event _U_,
				  gpointer data _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->pairsDialog));

  return TRUE;
}
static void onPairsButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  GtkWidget *wd;

  if (!main->pairsDialog)
    {
      gtkPairsBuild_window(main);
      wd = lookup_widget(main->pairsDialog, "closebutton3");
      g_signal_connect_swapped(G_OBJECT(wd), "clicked",
			       G_CALLBACK(onPairsCloseClicked),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->pairsDialog), "delete-event",
			       G_CALLBACK(onKillPairsDialog),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->pairsDialog), "destroy-event",
			       G_CALLBACK(onKillPairsDialog),
			       (gpointer)main);
    }

  gtkPairsBuild_pairs
    (main,
     renderingWindowGet_visuData(RENDERING_WINDOW(main->renderingWindow)), TRUE);

  gtkMainShow_window(GTK_WINDOW(main->pairsDialog));
}
static void onActionsCloseClicked(GtkMain *main, GtkButton *button _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->interactiveDialog));
  gtkMainShow_window(GTK_WINDOW(main));
}
static gboolean onKillInteractiveDialog(GtkMain *main,
					GdkEvent *event _U_, gpointer data _U_)
{
  onActionsCloseClicked(main, (GtkButton*)0);

  return TRUE;
}
static void onMouseActionsClicked(GtkMain *main, GtkButton *button _U_)
{
  GtkWidget *wd;
  gint posx, posy;

  if (!main->interactiveDialog)
    {
      gtkInteractiveBuild_window(main);
      wd = lookup_widget(main->interactiveDialog, "buttonBackToCommandPanel");
      g_signal_connect_swapped(G_OBJECT(wd), "clicked",
			       G_CALLBACK(onActionsCloseClicked),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->interactiveDialog), "delete-event",
			       G_CALLBACK(onKillInteractiveDialog),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->interactiveDialog), "destroy-event",
			       G_CALLBACK(onKillInteractiveDialog),
			       (gpointer)main);
    }

  /* Start new pick & observe session. */
  gtkInteractiveInit_session();

  gtk_window_get_position(GTK_WINDOW(main), &posx, &posy);
  DBG_fprintf(stderr, "Gtk Main: get command panel position (%d,%d).\n", posx, posy);
  gtkMainHide_window(GTK_WINDOW(main));
  DBG_fprintf(stderr, "Gtk Main: set observe position (%d,%d).\n", posx, posy);
  gtk_window_move(GTK_WINDOW(main->interactiveDialog), posx, posy);
  gtkMainShow_window(GTK_WINDOW(main->interactiveDialog));
}
static void onSaveButtonClicked(GtkMain *main _U_, GtkButton *button _U_)
{
  gtkSaveInit();
}
static void onQuitButtonClicked(GtkMain *main _U_, GtkButton *button _U_)
{
  gtkMain_quit();
}
static gboolean onKillAboutDialog(GtkMain *main, GdkEvent *event _U_,
				  gpointer data _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->aboutDialog));

  return TRUE;
}
static void onAboutButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  if (!main->aboutDialog)
    {
      gtkAboutBuild_window(main);
      g_signal_connect_swapped(G_OBJECT(main->aboutDialog), "delete-event",
			       G_CALLBACK(onKillAboutDialog),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->aboutDialog), "destroy-event",
			       G_CALLBACK(onKillAboutDialog),
			       (gpointer)main);
    }
  gtkMainShow_window(GTK_WINDOW(main->aboutDialog));
}

static void setRenderingButtonSensitive(GtkMain *main, gboolean bool)
{
  g_return_if_fail(IS_GTK_MAIN_TYPE(main));

  gtk_widget_set_sensitive(main->private->pairsButton, bool);
  gtk_widget_set_sensitive(main->private->mouseActions, bool);
}
static void onFileChange(GtkMain *main, VisuData *dataObj, gpointer data _U_)
{ 
  setRenderingButtonSensitive(main, (dataObj != (VisuData*)0));
  toolPanelClassSet_visuData(dataObj);
}
static void onResourcesLoaded(GtkMain *main, VisuData *dataObj _U_, gpointer data _U_)
{
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(main->private->checkPairs),
			       visuPairGet_status());
}
static void onRenderingChanged(GtkMain *main, RenderingMethod *method,
			       gpointer data _U_)
{
  gtk_widget_set_sensitive(main->private->loadButton,
			   (method != (RenderingMethod*)0));
  setRenderingButtonSensitive(main, FALSE);
}
static gboolean onKillMainWindowEvent(GtkWidget *widget _U_, GdkEvent *event _U_,
				      gpointer user_data _U_)
{
  gtkMain_quit();
  return TRUE;
}



/*****************/
/* Miscellaneous */
/*****************/
/* This method hides the specified window and save its coordinates if
   it is specified by a parameter. */
static void gtkMainHide_window(GtkWindow *win)
{
  int *val;

  if (!win)
    return;

  if (my_class->rememberWindowPosition)
    {
      val = (int*)g_hash_table_lookup(my_class->windowPosition, win);
      if (!val)
	{
	  val = g_malloc(sizeof(int) * 2);
	  g_hash_table_insert(my_class->windowPosition, (gpointer)win, (gpointer)val);
	}
      gtk_window_get_position(win, &val[0], &val[1]);
      DBG_fprintf(stderr, "Gtk Main : store position (%d,%d) for window %d.\n", 
		  val[0], val[1], GPOINTER_TO_INT(win));
    }
  gtk_widget_hide(GTK_WIDGET(win));
}

/* This method shows the specified window and try to put it at its
   former position if the good parameter is used. */
static void gtkMainShow_window(GtkWindow *win)
{
  int *val;

  if (!win)
    return;

  if (my_class->rememberWindowPosition)
    {
      val = (int*)g_hash_table_lookup(my_class->windowPosition, win);
      if (val)
	{
	  gtk_window_move(win, val[0], val[1]);
	  DBG_fprintf(stderr, "Gtk Main : set position (%d,%d) for window %d.\n", 
		      val[0], val[1], GPOINTER_TO_INT(win));
	}
    }
  gtk_window_present(win);
}

void onHideNextTime(GtkToggleButton *button, gpointer data)
{
  char *posNext;
  gchar *path, *bufferR, *pos;
  GString *bufferW, *bufferW2;
  gboolean resOk;
  int lines;
  GIOChannel *file;
  GError *err;
  gsize taille;
  GIOStatus statOK;

  g_return_if_fail(data);
  path = (gchar*)data;
  
  DBG_fprintf(stderr, "Gtk Main : change the warning dialog parameter in file '%s'.\n", path);
  my_class->warningWhenQuit = !gtk_toggle_button_get_active(button);

  /* If no file exists in the given path, we create it. */
  if (!g_file_test(path, G_FILE_TEST_EXISTS))
    {
      err = (GError*)0;
      resOk = visuConfigFileSave(VISU_CONFIGFILE_PARAMETER, path,
				 &lines, (VisuData*)0, &err);
      if (!resOk)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message,
				   (GtkWindow*)0);
	  g_error_free(err);
	}
      return;
    }

  /* If a parameter file already exist, we then just change the right line. */
  bufferR = (gchar*)0;
  err = (GError*)0;
  if (!g_file_get_contents(path, &bufferR, &taille, &err))
    {
      visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
      g_error_free(err);
      return;
    }

  /* We reopen the channel in write acces. */
  err = (GError*)0;
  file = g_io_channel_new_file(path, "w", &err);
  if (err)
    {
      visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
      g_error_free(err);
      return;
    }

  g_return_if_fail(bufferR);
  bufferW = g_string_new(bufferR);
  g_free(bufferR);
  /* Try to find the flag of the parameter. */
  pos = g_strrstr(bufferW->str, "\n"FLAG_PARAMETER_GTKMAIN_QUIT);
  if (!pos)
    {
      /* We append it at the end of the file. */
      DBG_fprintf(stderr, " | Can't find the option, appending it.\n");
      exportParametersGtkMain(bufferW, &lines, (VisuData*)0);

      err = (GError*)0;
      statOK = g_io_channel_write_chars(file, bufferW->str, -1,
                                        &taille, &err);
      if (err)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
	  g_error_free(err);
	}
    }
  else
    {
      DBG_fprintf(stderr, " | Option found, changing its value.\n");
      /* We erase the line and rewrite it. */
      *(pos + 1) = '\0';
      bufferW2 = g_string_new(bufferW->str);
      g_string_append_printf(bufferW2, "%s[gtk]: %i\n", FLAG_PARAMETER_GTKMAIN_QUIT,
			     (int)my_class->warningWhenQuit);
      posNext = strstr(pos + 2, "\n");
      if (posNext)
	g_string_append(bufferW2, posNext + 1);
      
      err = (GError*)0;
      statOK = g_io_channel_write_chars(file, bufferW2->str, -1,
                                        &taille, &err);
      if (err)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
	  g_error_free(err);
	}
      g_string_free(bufferW2, TRUE);
    }
  g_io_channel_shutdown(file, TRUE, (GError**)0);
  g_io_channel_unref(file);

  g_string_free(bufferW, TRUE);
}
static void onAddHomedir(GtkButton *button _U_, gpointer quitDialog)
{
  GtkWidget *wd;
  GList *dirs, *tmplst;
  gchar *path;

  DBG_fprintf(stderr, "Gtk Main: try to create the local home directory.\n");
#if GLIB_MINOR_VERSION > 7
  if (g_mkdir_with_parents(v_sim_local_conf_dir, 750))
#else
#if SYSTEM_X11 == 1
  if (mkdir(v_sim_local_conf_dir, 750))
#endif
#if SYSTEM_WIN32 == 1
  if (mkdir(v_sim_local_conf_dir))
#endif
#endif
    /* Failed. */
    visuGtkRaise_warning(_("I/O"),
			 _("Can't create the directory '$XDG_CONFIG_HOME/v_sim'."),
			 (GtkWindow*)0);
  else
    {
      /* Succeed hide the warning. */
      wd = lookup_widget(GTK_WIDGET(quitDialog), "hboxHomedir");
      gtk_widget_hide(wd);
      /* Retest the path. */
      dirs = (GList*)0;
      dirs = g_list_prepend(dirs, (gpointer)v_sim_data_dir);
      dirs = g_list_prepend(dirs, (gpointer)v_sim_local_conf_dir);
      tmplst = dirs;

      path = (gchar*)0;
      path = visuConfigFileGet_nextValidPath(VISU_CONFIGFILE_PARAMETER,
					     W_OK, &tmplst, 0);
      if (path)
	{
	  wd = lookup_widget(GTK_WIDGET(quitDialog), "hboxWarning");
	  gtk_widget_hide(wd);
	  wd = lookup_widget(GTK_WIDGET(quitDialog), "checkbuttonHideNextTime");
	  gtk_widget_set_sensitive(wd, TRUE);
	  g_signal_connect(G_OBJECT(wd), "toggled",
			   G_CALLBACK(onHideNextTime), (gpointer)path);
	}
      g_list_free(dirs);
    }
}

void gtkMain_quit()
{
  GtkWidget *quitDialog, *wd;
  GList *dirs, *tmplst;
  gchar *path;

  if (!my_class->warningWhenQuit)
    {
      gtk_main_quit();
      return;
    }

  quitDialog = create_quitDialog();
  
  /* Try to find installDir/v_sim.par or $XDG_CONFIG_HOME/v_sim/v_sim.par
     that is writable to store the preference of the hiding
     mode of the dialog. */
  dirs = (GList*)0;
  dirs = g_list_prepend(dirs, (gpointer)v_sim_data_dir);
  dirs = g_list_prepend(dirs, (gpointer)v_sim_local_conf_dir);
  tmplst = dirs;

  path = (gchar*)0;
  path = visuConfigFileGet_nextValidPath(VISU_CONFIGFILE_PARAMETER, W_OK, &tmplst, 0);
  if (!path)
    {
      wd = lookup_widget(quitDialog, "hboxWarning");
      gtk_widget_show(wd);
    }
  g_list_free(dirs);

  /* Attach a create the homedir method to the button. */
  wd = lookup_widget(quitDialog, "buttonAddHomedir");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onAddHomedir), (gpointer)quitDialog);
  /* Show the warning if the homedir is not existing and no path was found. */
  if (!g_file_test(v_sim_local_conf_dir, G_FILE_TEST_IS_DIR) && !path)
    {
      wd = lookup_widget(quitDialog, "hboxHomedir");
      gtk_widget_show(wd);
    }

  /* Attach a modify the parameter to the checkbox. */
  wd = lookup_widget(quitDialog, "checkbuttonHideNextTime");
  if (!path)
    gtk_widget_set_sensitive(wd, FALSE);
  else
    g_signal_connect(G_OBJECT(wd), "toggled",
		     G_CALLBACK(onHideNextTime), (gpointer)path);
  
  if (gtk_dialog_run(GTK_DIALOG(quitDialog)) == GTK_RESPONSE_OK)
    gtk_main_quit();
  else
    gtk_widget_destroy(quitDialog);
}

static gboolean readMainQuitPolicy(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  gboolean bool;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &bool, 1, error))
    return FALSE;
  my_class->warningWhenQuit = bool;
  
  return TRUE;
}
static gboolean readMainPanelStatus(gchar **lines, int nbLines, int position,
				    VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  ToolPanel *toolpanel;
  char *pt;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_string(lines[0], position, &tokens, 2, TRUE, error))
    return FALSE;
  toolpanel = toolPanelClassGet_toolPanelById(tokens[0]);
  if (!toolpanel)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the panel '%s' is unknown.\n"),
			   position, tokens[0]);
      g_strfreev(tokens);
      return FALSE;
    }
  pt = strchr(tokens[1], '\n');
  if (pt)
    *pt = ' ';
  toolPanelSet_containerId(toolpanel, g_strchomp(tokens[1]));

  g_strfreev(tokens);
  return TRUE;
}
static gboolean readMainDock(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  int res;
  gchar **tokens, *values;
  gboolean visible;
  int x, y, width, height;

  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit(g_strchug(lines[0]), " ", 4);
  if (!tokens[0] || !tokens[1] || !tokens[2] || !tokens[3])
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: awaited "
			     "'id visible pos size'.\n"),
			   position);
      g_strfreev(tokens);
      return FALSE;
    }
  values = g_strjoin(" ", tokens[0], tokens[1], tokens[2], NULL);
  res = sscanf(values, "%d %dx%d %dx%d", (int*)&visible, &x, &y, &width, &height);
  if (res != 5)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: can't read dock"
			     " characteristic values from '%s'.\n"),
			   position, values);
      g_strfreev(tokens);
      g_free(values);
      return FALSE;
    }
  g_free(values);
  toolPanelClassSet_windowSize(g_strchomp(tokens[3]), (guint)width, (guint)height);
  toolPanelClassSet_windowPosition(tokens[3], (guint)x, (guint)y);
  toolPanelClassSet_windowVisibility(tokens[3], visible);

  g_strfreev(tokens);
  return TRUE;
}
static gboolean exportParametersGtkMain(GString *data, int *nbLinesWritten,
					VisuData* dataObj _U_)
{
  GList *tmplst, *panelLst, *dockLst;
  gint x, y, width, height;
  gboolean visilibity;
  gchar *id;

  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_QUIT);
  g_string_append_printf(data, "%s[gtk]: %i\n\n", FLAG_PARAMETER_GTKMAIN_QUIT,
			 (int)my_class->warningWhenQuit);
  *nbLinesWritten = 3;

  /* Write the panel list. */
  panelLst = toolPanelClassGet_allToolPanels();
  if (panelLst)
    {
      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_PANEL);
      *nbLinesWritten += 1;
    }
  tmplst = panelLst;
  while(tmplst)
    { 
      g_string_append_printf(data, "%s[gtk]: %s %s\n", FLAG_PARAMETER_GTKMAIN_PANEL,
			     toolPanelGet_id(TOOL_PANEL(tmplst->data)),
			     toolPanelGet_containerId(TOOL_PANEL(tmplst->data)));
      *nbLinesWritten += 1;
      tmplst = g_list_next(tmplst);
    }
  if (panelLst)
    {
      g_string_append_printf(data, "\n");
      *nbLinesWritten += 1;
    }
  g_list_free(panelLst);

  /* Write the panel list. */
  dockLst = toolPanelClassGet_allWindows();
  if (dockLst)
    {
      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_DOCK);
      *nbLinesWritten += 1;
    }
  tmplst = dockLst;
  while(tmplst)
    { 
      toolPanelClassGet_windowCharacteristics((DockWindow*)(tmplst->data), &id,
					      &visilibity, &x, &y, &width, &height);
      g_string_append_printf(data, "%s[gtk]: %d %dx%d %dx%d %s\n",
			     FLAG_PARAMETER_GTKMAIN_DOCK,
			     (int)visilibity, x, y, width, height, id);
      *nbLinesWritten += 1;
      tmplst = g_list_next(tmplst);
    }
  if (dockLst)
    {
      g_string_append_printf(data, "\n");
      *nbLinesWritten += 1;
    }
  g_list_free(dockLst);

  return TRUE;
}


/*******************/
/* Class routines. */
/*******************/
GtkMain* gtkMainClassGet_currentPanel()
{
  return currentGtkMain;
}
void gtkMainClassSet_currentPanel(GtkMain *main)
{
  g_return_if_fail(IS_GTK_MAIN_TYPE(main));
  currentGtkMain = main;
}

char* gtkMainClassGet_lastOpenDirectory()
{
  DBG_fprintf(stderr, "Gtk Main : get the last open directory : '%s'.\n",
	      my_class->lastOpenDirectory);
  return my_class->lastOpenDirectory;
}
void gtkMainClassSet_lastOpenDirectory(char* directory)
{
  if (my_class->lastOpenDirectory)
    g_free(my_class->lastOpenDirectory);

  if (!g_path_is_absolute(directory))
    my_class->lastOpenDirectory = g_build_filename(g_get_current_dir(),
						   directory, NULL);
  else
    my_class->lastOpenDirectory = g_build_filename(directory, NULL);
  DBG_fprintf(stderr, "Gtk Main : set the last open directory to '%s', OK.\n",
	      my_class->lastOpenDirectory);
}
void gtkMainClassSet_renderingSpecificMethods(RenderingMethod *method,
					      GtkMainSetFilesFunc methodLoad)
{
  struct renderingMethod_gtkMain_struct *container;

  g_return_if_fail(method && my_class);

  container = g_malloc(sizeof(struct renderingMethod_gtkMain_struct));
  container->loadMethod = methodLoad;

  DBG_fprintf(stderr, "Gtk Main: set rendering specific for method '%s'.\n",
	      method->name);
  g_hash_table_insert(my_class->renderingMethod_gtkMain,
		      (gpointer)method, (gpointer)container);
}
GtkMainSetFilesFunc gtkMainClassGet_renderingSpecificOpen(RenderingMethod *method)
{
  struct renderingMethod_gtkMain_struct *container;

  g_return_val_if_fail(my_class, (GtkMainSetFilesFunc)0);
  
  if (!method)
    return (GtkMainSetFilesFunc)0;

  DBG_fprintf(stderr, "Gtk Main : looking for a specific load interface for rendering"
	      " method '%s'...\n", method->name);
  
  container = (struct renderingMethod_gtkMain_struct *)
    g_hash_table_lookup(my_class->renderingMethod_gtkMain, (gpointer)method);

  g_return_val_if_fail(container, (GtkMainSetFilesFunc)0);
  
  if (container->loadMethod)
    return container->loadMethod;
  else
    return visuGtkGet_fileFromDefaultFileChooser;
}
void gtkMainClassSet_rememberPosition(gboolean val)
{
  g_return_if_fail(my_class);

  DBG_fprintf(stderr, "GTK Main : set the remember parameter to %d.\n", val);
  my_class->rememberWindowPosition = val;
}
gboolean gtkMainClassGet_rememberPosition()
{
  g_return_val_if_fail(my_class, PARAMETER_GTKMAIN_REMEMBER_DEFAULT);
  return my_class->rememberWindowPosition;
}
