/* ========================================================================== */
/* === UMFPACK_transpose ==================================================== */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 3.2 (Jan. 1, 2002), Copyright (c) 2002 by Timothy A.       */
/* Davis, University of Florida, davis@cise.ufl.edu.  All Rights Reserved.    */
/* See README, umfpack.h, or type "umfpack_details" in Matlab for License.    */
/* -------------------------------------------------------------------------- */

/*

    User callable.  Computes a permuted transpose, C = (A (P,Q))' in Matlab
    notation, where B is in column-form.  Alternatively, this routine can be
    viewed as constructing the row-form of B = A (P,Q).  See umfpack_transpose.h
    for details.  The matrix A need not have sorted columns.  B does have sorted
    rows (equivalently, C has sorted columns).  A and B are square.
    The matrix A may be singular.

    Dynamic memory usage:

	A single call to UMF_malloc is made, for a workspace of size
	max (n,1) * sizeof(Int).  This is then free'd on return,
	via UMF_free.

*/

#include "umf_internal.h"
#include "umf_transpose.h"
#include "umf_malloc.h"
#include "umf_free.h"

#ifndef NDEBUG
PRIVATE Int init_count ;
#endif

/* ========================================================================== */

GLOBAL Int UMFPACK_transpose
(
    Int n,
    const Int Ap [ ],	/* size n+1 */
    const Int Ai [ ],	/* size nz = Ap [n] */
    const double Ax [ ], /* size nz, pattern only if Ax and/or Cx not present */

    const Int P [ ],	/* P [k] = i means original row i is kth row in A(P,Q)*/
			/* P is identity if not present */
			/* size n, if present */

    const Int Q [ ],	/* Q [k] = j means original col j is kth col in A(P,Q)*/
			/* Q is identity if not present */
			/* size n, if present */

    Int Cp [ ],		/* size n+1 */
    Int Ci [ ],		/* size max (n,nz) */
    double Cx [ ]	/* size nz, if present */
)
{

    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int status, *W ;

#ifndef NDEBUG
    init_count = UMF_malloc_count ;
#endif

    /* ---------------------------------------------------------------------- */
    /* allocate workspace */
    /* ---------------------------------------------------------------------- */

    W = (Int *) UMF_malloc (n, sizeof (Int)) ;
    if (!W)
    {
	ASSERT (UMF_malloc_count == init_count) ;
	return (UMFPACK_ERROR_out_of_memory) ;
    }
    ASSERT (UMF_malloc_count == init_count + 1) ;

    /* ---------------------------------------------------------------------- */
    /* B = row-form of A (P,Q) */
    /* ---------------------------------------------------------------------- */

    status = UMF_transpose (n, Ap, Ai, Ax, P, Q, n, Cp, Ci, Cx, W, TRUE) ;

    /* ---------------------------------------------------------------------- */
    /* free the workspace */
    /* ---------------------------------------------------------------------- */

    (void) UMF_free ((void *) W) ;
    ASSERT (UMF_malloc_count == init_count) ;

    return (status) ;
}

