/* ==================================================== ======== ======= *
 *
 *  uprop.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uprop_hh
#define	_uprop_hh
//pragma ident	"@(#)uprop.hh	ubit:b1.11.7"


class UProp: public UBrick {
  friend class UBox;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  virtual class UProp*  propCast() {return this;}

  // NOTE on UProp callbacks: 
  // the following callbacks are fired when object's value is changed:
  // 1. UOn::change      callbacks of THIS object
  // 2. UOn::changeProp  callbacks of its PARENTS

  UProp(u_modes b_modes) : UBrick(b_modes) {}
  //virtual ~UProp() {clean();}

  // Updates graphics
  virtual void update() = 0;

  //package_private: ====[ubit intrinsics]=====

  // updates grahics if arg is true the fires object's UOn::change callbacks 
  // fires then parents' UOn::changeProp callbacks
  virtual void changed(u_bool update_now);
  // changes corresponding value in the UContext 
  virtual void putProp(class UContext*, class UCtrl*) = 0;
};

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
///TEMPORARIREMENT EN Uprop

class UComment: public UProp {
  char *value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UComment(const char* = null, u_modes = 0);
  void set(const char*);
  const char* get() const {return value;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

UComment& ucomment(const char *s);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UScale : public UProp {
  int value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  //positive scale --> larger than normal size
  //negative scale --> smaller than normal size
  UScale(int = 0, u_modes = 0);

  // sets the (relative) scale value (the absolute scale value
  // results from the addition of all uscales if the object tree
  void set(int, u_bool update_now = true);
  int get() const {return value;}

  // increments from delta if delta is >0, decrements if delta is <0
  void incr(int delta);

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

inline UScale& uscale(int val = 0) {return *(new UScale(val));}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UAlpha : public UProp {
  float value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // alpha blending for this component (and its children taken as a group):
  // 1.0 : opaque object
  // 0.0 : fully transparent object
  UAlpha(float = 1.0, u_modes = 0);

  void set(float, u_bool update_now = true);
  float get() const {return value;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

UAlpha& ualpha(float = 1.0);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UBackground : public UProp {
protected:
  const class UColor  *bgcolor;
  const class UIma    *bgima;
  const class UHalign *halign;
  const class UValign *valign;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // NOTE that the background image can be a transparent UPix
  UBackground(u_modes = 0);
  UBackground(const UIma &bgima, u_modes = 0);
  UBackground(const UColor &bgcolor, u_modes = 0);

  // uniform background color
  void set(const UColor &bgcolor);
  void set(const UColor *bgcolor);

  // tiled image (tiling is specified by setTiling)
  void set(const UIma &bgima);
  void set(const UIma *bgima);

  // tiled image with a background color (useful if the image is transparent
  // or if the image is centered instead of being tiled)
  void set(const UIma *bgima, const UColor *bgcolor);

  // if null, the bg image is tiled in this direction
  // if &UHalign::center, it is centered in the horizontal direction
  // etc.
  void setLayout(const UHalign*, const UValign*);

  const class UColor*  getColor()  const {return bgcolor;}
  const class UIma*    getIma()    const {return bgima;}
  const class UHalign* getHalign() const {return halign;}
  const class UValign* getValign() const {return valign;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

UBackground& ubackground(const UColor &bgcolor);
UBackground& ubackground(const UIma &bgima);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// does not work in current version!!!

class UShape : public UProp {
  friend class UView;
protected:
  const UPix *pix;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // The shape of the component will adapt to the shape of the UPix argument
  UShape(const UPix&, u_modes = 0);

  void set(const UPix*);
  void set(const UPix&);
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

UShape& ushape(UPix&);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UFlagFilter : public UProp {
  const UFlag *flag;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UFlagFilter(const UFlag&, u_modes = 0);
  UFlagFilter(const UFlag* = null, u_modes = 0);

  //NB: set(null) unsets the current flag
  void set(const UFlag*);
  const UFlag* getFlag() {return flag;}

  virtual void update();
  virtual void putProp(UContext*, class UCtrl*);
};

UFlagFilter& uflagFilter(const UFlag&);
UFlagFilter& uflagFilter(const UFlag* = null);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UPos : public UProp {
  u_pos x, y;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // disable automatic layout by specifying a specific position
  // that is relative to the parent object
  UPos(u_pos x, u_pos y, u_modes = 0);
  UPos();
  virtual ~UPos() {clean();}

  void set(u_pos _x, u_pos _y, u_bool update_now = true);
  void setX(u_pos _x) {set(_x, y, true);}
  void setY(u_pos _y) {set(x, _y, true);}

  u_pos getX() const {return x;}
  u_pos getY() const {return y;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
  virtual void addingTo(ULink *selflink, UGroup *parent);
  virtual void removingFrom(ULink *selflink, UGroup *parent);
};

inline UPos& upos(u_pos x, u_pos y) {return *new UPos(x, y);}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UWidth : public UProp {
  u_dim value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  static const u_dim autoResize, keepSize;
  // Usage:
  // -- uwidth(int) with int >= 0  : size fixed by argument.
  //    horizontal size won't change except if the box is uhflex()
  // -- uwidth(UWidth::keepSize)   : keep initial size.
  //    size is computed automatically the first time, then does not change.
  // -- uwidth(UWidth::autoResize) : size automatically adapts
  //    size is re-computed automatically each time

  UWidth(u_dim value, u_modes = 0);

  void  set(u_dim);
  u_dim get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UWidth& uwidth(u_dim value);

/* ==================================================== ======== ======= */

class UHeight : public UProp {
  u_dim value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  static const u_dim autoResize, keepSize;
  // Usage:
  // -- same as UWidth (see above)

  UHeight(u_dim value, u_modes = 0);

  void  set(u_dim);
  u_dim get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UHeight& uheight(u_dim value);

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

class UOrient : public UProp {
  char value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  static UOrient vertical, horizontal;

  UOrient(const UOrient&, u_modes = 0);
  void set(const UOrient&);
  char get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
  UOrient(const char *instname, char value, u_modes);
};

UOrient& uorient(const UOrient&);

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

class UValign : public UProp {
  char value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // "flex" means "flexible object"
  static UValign top, bottom, center, flex;

  UValign(const UValign&, u_modes = 0);
  void set(const UValign&);
  char get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
  UValign(const char *instname, char value, u_modes);
};

UValign& uvalign(const UValign&);
inline UValign& utop()     {return UValign::top;}
inline UValign& ubottom()  {return UValign::bottom;}
inline UValign& uvcenter() {return UValign::center;}
inline UValign& uvflex()   {return UValign::flex;}

/* ==================================================== ======== ======= */

class UHalign : public UProp {
  char value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // "flex" means "flexible object"
  static UHalign left, right, center, flex;

  UHalign(const UHalign&, u_modes = 0);
  void set(const UHalign&);
  char get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
  UHalign(const char *instname, char value, u_modes);
};

UHalign& uhalign(const UHalign&);
inline UHalign& uleft()    {return UHalign::left;}
inline UHalign& uright()   {return UHalign::right;}
inline UHalign& uhcenter() {return UHalign::center;}
inline UHalign& uhflex()   {return UHalign::flex;}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

class UVspacing : public UProp {
  friend class UBox;
  int value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UVspacing(int, u_modes = 0);
  void set(int);
  int get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UVspacing& uvspacing(int);

/* ==================================================== ======== ======= */

class UHspacing : public UProp {
  friend class UBox;
  int value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UHspacing(int, u_modes = 0);
  void set(int);
  int get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UHspacing& uhspacing(int);

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

class UVmargin : public UProp {
  friend class UBox;
  int value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UVmargin(int, u_modes = 0);
  void set(int);
  int get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UVmargin& uvmargin(int);

/* ==================================================== ======== ======= */

class UHmargin : public UProp {
  friend class UBox;
  int value;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UHmargin(int, u_modes = 0);
  void set(int);
  int get() const {return value;}
  virtual void update();

  //==== Ubit Intrinsics
  virtual void putProp(class UContext*, class UCtrl*);
};

UHmargin& uhmargin(int);

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:00] ======= */
