/* ==================================================== ======== ======= *
 *
 *  ubrick.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ubrick_hh
#define	_ubrick_hh
//pragma ident	"@(#)ubrick.hh	ubit:b1.11.6"
#include <udefs.hh>
#include <ubrickImpl.hh>

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

extern "C" {
  const char* uversion();
  long uversionNo();
}

// Same syntax as printf()
// Note: 'funname' starting with a "!" provokes a Fatal Error!

void uerror(const char *funname, const char *format, ...);
void uwarning(const char *funname, const char *format, ...);

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

class UClass {
public:
  const char *name;
  UClass(const char *classname);
  static const UClass* getClass(const char *classname);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// NOTES on DELETION:
//
// * The 'udelete()' or 'udeletep()' functions MUST ALWAYS be used 
//   instead of the 'delete' primitive to destroy Bricks. 
//
// * These fcts. unlink the object given as an arg.from the instance graph.
//   They also destroy the object's CHILDREN (and descendants) recursively:
//   - if they are in the AUTODEL mode (the default, see 'UBrick::autodel()')
//   - and if they do NOT have other (undeleted) parents.
//
// * These fcts. never destroy predefined constants (such as UColor::red...)
//   nor objects in the UCONST or NO_DEL modes
//
// * The argument of 'udelete()' MUST be a VARIABLE. This variable is set
//   to null if the object could be destroyed.

// * The argument of 'udeletep()' does not need to be a variable (it can
//   be a value returned by another function). Thus, this arg. is NOT set
//   to null but the fct. returns true if the object could be destroyed.

// Usage:
//     UButton* b = &ubutton("foo");
//     udelete(b);  // sets b to null if *b was actually destroyed
//
//     // event->getSource() is not a variable so we use udeletep()
//     udeletep(event->getSource());

template <class X> void udelete(X*& obj) {
  if (udeletep(obj)) obj = null;
}

u_bool udeletep(UBrick* obj);

// Base class for objects that can be added to the instance graph
class UBrick {  
  friend class UGroup;
  friend class UBox;
  friend class UWin;
  friend class ULink;
  friend class UArgs;
  friend u_bool udeletep(UBrick*);
  // NOTE: 
  // if 'a' and 'b' are (sub-)Brick objects, the following expressions:
  //    A a = b;
  //    a = b;   
  //
  // are illegal and won't compile (as the semantics of = is undefined
  // in the general case). Use the 'set()' method instead.
  // Note howether that the following expressions are correct:
  //    A &c = b;
  //    A *a = &b;
  //    a = &b;   
 
  UBrick(const UBrick&);
  UBrick& operator=(const UBrick&);

public:
  static  const UClass  uclass;
  virtual const UClass* getClass()     const;
  virtual const char*   getClassName() const;
  virtual const char*   cname()        const; //synonym of getClassName()

  // default: set bmodes to 0
  UBrick(u_modes b_modes = 0)  {bmodes = b_modes; cache = null;}
  virtual ~UBrick() {clean();}
  virtual void clean();

  // OBSOLETE: simulates Dynamic Cast for non ANSI compilers:
  // xxxCast() checks if object's class is 'xxx' or derives from 'xxx'
  // -- if true:  returns object pointer casted in the appropriate way
  // -- if false: returns null

  virtual class UProp*  propCast()   {return null;}
  virtual class UItem*  itemCast()   {return null;}
  virtual class UStr*   strCast()    {return null;}
  virtual class UGroup* groupCast()  {return null;}
  virtual class UBox*   boxCast()    {return null;}
  virtual class UWin*   winCast()    {return null;}

  // changes the AUTODEL mode of the Brick (see NOTES on DELETION above)
  // AUTODEL bricks are implicitely destroyed when function 'udelete()'
  // or method 'UGroup::remove()' are called by their parents.
  // Mode AUTODEL is ON by default.
  virtual void autodel(u_bool on_off);

  // Returns the nth parent (as bricks may have several parents)
  // -- 'pos' = 0 means "first parent" and 'pos' = -1 means "last parent"
  // -- returns null if 'pos' is out of range
  // -- prints a warning and returns null if 'pos' < -1

  virtual UGroup* getParent(int pos);

  // returns the number of parents
  virtual int getParentCount();

  // Returns a copy of the object's parent list
  // -- returns a null terminated table that must be destroyed after use
  //    by using the delete[] primitive
  // -- the table is always created even if the object has no parent
  //    (in this case it will just contain a null pointer in first position)

  virtual UGroup** getParents();
  virtual UGroup** getParents(int &count);

  // removes the brick from (all) its parents
  virtual void removeFromParents();

  // returns true if argument is a direct parent (case indirect = false)
  // or an ancestor (case indirect = true) of 'this' object
  virtual u_bool isChildOf(class UGroup *possible_parent, u_bool indirect);

  // Object intialization
  virtual void init(ULink *selflink, ULink *parlink, UView *parview) {}

  // Updates graphics
  //virtual void update() = 0;

  // printf() syntax
  // -- 'funname' starting with a "!" provokes a Fatal Error!
  virtual void error(const char *funname, const char *format, ...) const;
  virtual void warning(const char *funname, const char *format, ...) const;

  // conditional operator /
  // (the condition can be a UOn::xxx logical event or an UFlag
  // please refer to file "uctrl.hh for details on logical conditions)
  // syntax: 
  //    UOn::enter  / UBgcolor::yellow
  //    UOn::arm    / ucall(function, arg)        [see file ucall.hh]
  //    UOn::action / ucall(object, method, arg)  [for details]

  friend ULink& operator/(const UCond&, UBrick&);
  friend ULink& operator/(const UCond&, UBrick*);

  // callback operator >> 
  // adds callbacks to the brick with an UOn::change condition
  // syntax: 
  //    mybrick >> ucall(function, arg)        [see file ucall.hh]
  //    mybrick >> ucall(object, method arg)   [for details]

  friend UBrick& operator>>(UBrick&, UCall&);
  friend UBrick& operator>>(UBrick*, UCall&);

  //package_private: ====[ubit intrinsics]=====

  // calls the callbacks that are registered on this UOn condition
  virtual void fire(class UEvent&, const class UOn&);
  ULLChain& getParentList() {return parents;}
  UChain*   getCache()      {return cache;}
  u_modes   getBmodes() const {return bmodes;}
  void      setBmodes(u_modes bmodes, u_bool on_off);
  virtual ULink* makeLink() {return new ULink(this);}
  u_bool    hasMultipleTrueParents();
protected:
  ULLChain parents;
  UChain   *cache;  // fast acces list (callbacks + special attributes)
  u_modes  bmodes;
  virtual void addingTo(ULink *selflink, UGroup *parent);
  //NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);
};

//OBSOLETE: prevents deletion by udelete() and sets the AUTODEL mode to false
void ucannotDelete(class UBrick*);
void ucannotDelete(class UBrick&);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UItem: public UBrick {	// Abstract Class
  friend class UItemLink;
  friend class UArgs;
  friend class UBox;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass()  const {return &uclass;}
  virtual class UItem*  itemCast()  {return this;}

  // NOTE on UItem callbacks: 
  // the following callbacks are fired when object's value is changed:
  // 1. UOn::change      callbacks of THIS object
  // 2. UOn::changeItem  callbacks of its PARENTS
  // in addition, an UStr object (or a subclass) also fires:
  // 3. UOn::changeStr   callbacks of its PARENTS

  UItem(u_modes b_modes): UBrick(b_modes) {};
  //virtual ~UItem() {clean();}

  // Updates graphics
  virtual void update() = 0;

  //package_private: ====[ubit intrinsics]=====

  // updates grahics if arg is true the fires object's UOn::change callbacks 
  // fires then parents' UOn::changeItem callbacks
  virtual void changed(u_bool update_now);
  virtual void getSize(UContext*, u_dim *w, u_dim *h) const = 0;
  virtual void paint(UWinGraph&, UContext*, const URegion &r) const = 0;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
