/* Tracker
 * routines for emails with Evolution
 * Copyright (C) 2006, Laurent Aguerreche (laurent.aguerreche@free.fr)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <glib/gstdio.h>

#include "tracker-email-evolution.h"
#include "tracker-email-utils.h"
#include "tracker-db-email.h"
#include "tracker-cache.h"
#include "tracker-dbus.h"
#include "tracker-watch.h"


#define EVOLUTION_MAIL_DIR_S ".evolution/mail"


typedef enum {
	EVOLUTION_MAIL_PROTOCOL_UNKNOWN,
	EVOLUTION_MAIL_PROTOCOL_MBOX,
	EVOLUTION_MAIL_PROTOCOL_IMAP,
	EVOLUTION_MAIL_PROTOCOL_MH,
	EVOLUTION_MAIL_PROTOCOL_MAILDIR
} EvolutionMailProtocol;

typedef struct {
	gchar                 *name;       /* laurent.aguerreche@free.fr */
	gchar                 *uid;        /* unique ID generated by Evolution, looks like "1134161347.7985.16@foo.bar" */
	gboolean              enabled;	   /* enabled account or not */
	EvolutionMailProtocol protocol;	   /* protocol used: MBox, IMAP, MH, Maildir */
	gchar                 *source_url; /* mbox:///var/mail/laurent (used to find where root directory for email files is) */
} EvolutionAccount;

typedef struct {
	gchar		*mail_dir;	/* something like "/home/laurent.evolution/mail" */
	GSList		*accounts;	/* list of EvolutionAccount elements */
	gchar		*dir_local;	/* something like "/home/laurent.evolution/mail/local" */
	GSList		*imap_dirs;	/* a directory looks like "/home/laurent.evolution/mail/imap/laurent.aguerreche@free.fr".
					   Email address acts as an account name. */
	GSList		*mh_dirs;	/* list of MH directories */
	GSList		*maildir_dirs;	/* list of maildir directories */
} EvolutionConfig;

typedef struct {
	GQueue		 *elements;	/* Stack of elements in XML files.
					   Eg. <source><url>foo</url></source> where "url" and "source" will appear in that stack. */
	EvolutionAccount *account;	/* Parsing of an XML entity feeds that account. */
} ParsingAccountState;

enum {
	EVOLUTION_MESSAGE_ANSWERED     = 1 << 0,
	EVOLUTION_MESSAGE_DELETED      = 1 << 1,
	EVOLUTION_MESSAGE_DRAFT        = 1 << 2,
	EVOLUTION_MESSAGE_FLAGGED      = 1 << 3,
	EVOLUTION_MESSAGE_SEEN         = 1 << 4,
	EVOLUTION_MESSAGE_ATTACHMENTS  = 1 << 5,
	EVOLUTION_MESSAGE_ANSWERED_ALL = 1 << 6,
	EVOLUTION_MESSAGE_JUNK         = 1 << 7,
	EVOLUTION_MESSAGE_SECURE       = 1 << 8
};

typedef struct {
	gchar            *path;                 /* path the summary file */
	FILE             *f;                    /* opened file descriptor for the file */
	EvolutionAccount *associated_account;   /* pointer to the associated Evo account of this file (so we will not free
                                                   this pointer...) */
} SummaryFile;

typedef struct {
	gint32		version;
	gboolean	legacy;
	gint32		flags;
	gint32		nextuid;
	time_t		time;
	gint32		saved_count;
	gint32		unread_count;
	gint32		deleted_count;
	gint32		junk_count;
	gchar 		*uri_prefix;
} SummaryFileHeader;

/* Some infos are only accessible throw a deep code path but we need to retreive them. */
typedef struct {
	gchar		*mail_uid;
} EvolutionAdHocInfos;


extern Tracker		*tracker;

static EvolutionConfig	*evolution_config = NULL;


static gboolean	load_evolution_config			(EvolutionConfig **conf);
static void	free_evolution_config			(EvolutionConfig *conf);

static EvolutionAccount * copy_evolution_account	(EvolutionAccount *account);
static void	free_evolution_account			(EvolutionAccount *account);
static GSList *	find_accounts_by_gconf			(GSList *found_accounts);
static void	account_start_element_handler		(GMarkupParseContext	*context,
							 const gchar		*element_name,
							 const gchar		**attr_names,
							 const gchar		**attr_values,
							 gpointer			user_data,
							 GError			**error);
static void	account_text_handler			(GMarkupParseContext	*context,
							 const gchar		*text,
							 gsize			text_len,
							 gpointer			user_data,
							 GError			**error);
static void	account_end_element_handler		(GMarkupParseContext	*context,
							 const gchar		*element_name,
							 gpointer			user_data,
							 GError			**error);

static gboolean	is_in_dir_local				(const gchar *dir);
static gboolean	is_in_dir_imap				(const gchar *dir);
static gboolean	is_in_dir_imap4				(const gchar *dir);

/*
static gboolean	is_in_dir_mh				(const gchar *path);
static gboolean	is_in_dir_maildir			(const gchar *path);
*/

static gchar *	get_account_name_in_imap_path		(const gchar *path);
static gchar * get_account_name_from_imap_uri          (const gchar *imap_uri);

typedef gboolean (* LoadSummaryFileMetaHeaderFct) (SummaryFile *summary, SummaryFileHeader *header);
typedef gboolean (* LoadMailMessageFct) (SummaryFile *summary, MailMessage **mail_msg);
typedef gboolean (* SkipMailMessageFct) (SummaryFile *summary);
typedef gboolean (* SaveOnDiskMailMessageFct) (DBConnection *db_con, MailMessage *msg);

static void	index_mail_messages_by_summary_file	(DBConnection *db_con, MailType mail_type,
							 const gchar *summary_file_path,
							 LoadSummaryFileMetaHeaderFct load_meta_header,
							 LoadMailMessageFct load_mail,
							 SkipMailMessageFct skip_mail,
							 SaveOnDiskMailMessageFct save_ondisk_mail);

static void	load_uri_and_status_of_mbox_mail_message (GMimeMessage *g_m_message, MailMessage *msg, gpointer user_data);

static gboolean	open_summary_file			(const gchar *path, SummaryFile **summary);
static void	free_summary_file			(SummaryFile *summary);

static gboolean	load_summary_file_header		(SummaryFile *summary, SummaryFileHeader **header);
static void	free_summary_file_header		(SummaryFileHeader *header);
static gboolean	load_summary_file_meta_header_for_local	(SummaryFile *summary, SummaryFileHeader *header);
static gboolean	load_summary_file_meta_header_for_imap	(SummaryFile *summary, SummaryFileHeader *header);
//static gboolean	load_summary_file_meta_header_for_maildir (SummaryFile *summary, SummaryFileHeader *header);

/* static gboolean	load_mail_message_for_local		(SummaryFile *summary, MailMessage **mail_msg); */
static gboolean	load_mail_message_for_imap		(SummaryFile *summary, MailMessage **mail_msg);
static gboolean	load_mail_message_for_imap4		(SummaryFile *summary, MailMessage **mail_msg);
static gboolean	do_load_mail_message_for_imap		(SummaryFile *summary, MailMessage **mail_msg, gboolean do_skipping_of_content_info);
//static gboolean	load_mail_message_for_maildir		(SummaryFile *summary, MailMessage **mail_msg);
static gboolean	load_mail_message			(SummaryFile *summary, MailMessage *mail_msg);

/* static gboolean	skip_mail_message_for_local		(SummaryFile *summary); */
static gboolean	skip_mail_message_for_imap		(SummaryFile *summary);
static gboolean	skip_mail_message_for_imap4		(SummaryFile *summary);
static gboolean	do_skip_mail_message_for_imap		(SummaryFile *summary, gboolean do_skipping_of_content_info);
//static gboolean	skip_mail_message_for_maildir		(SummaryFile *summary);
static gboolean	skip_mail_message			(SummaryFile *summary);

static gboolean	skip_loading_content_info		(SummaryFile *summary);
static gboolean	do_skip_loading_content_info		(SummaryFile *summary);

static gboolean	save_ondisk_email_message_for_imap	(DBConnection *db_con, MailMessage *mail_msg);
static gboolean	save_ondisk_email_message_for_imap4	(DBConnection *db_con, MailMessage *mail_msg);
static gboolean	do_save_ondisk_email_message_for_imap	(DBConnection *db_con, MailMessage *mail_msg);
static gboolean	do_save_ondisk_email_message		(DBConnection *db_con, MailMessage *mail_msg);
static gboolean	index_mail_parts			(DBConnection *db_con, MailMessage *mail_msg, const gchar *mail_part_radix);

static GSList *	add_persons_from_internet_address_list_string_parsing	(GSList *list, const gchar *s);

static gboolean scan_summary_for_junk (SummaryFile *summary, guint32 *uid, MailType type);

static inline gboolean	decode_gint32		(FILE *f, gint32 *n);
static inline gboolean	skip_gint32_decoding	(FILE *f);
static inline gboolean	decode_guint32		(FILE *f, guint32 *n);
static inline gboolean	skip_guint32_decoding	(FILE *f);
static inline gboolean	decode_time_t		(FILE *f, time_t *t);
static inline gboolean	skip_time_t_decoding	(FILE *f);
static inline gboolean	decode_off_t		(FILE *f, off_t *t);
static inline gboolean	skip_off_t_decoding	(FILE *f);
static inline gboolean	decode_string		(FILE *f, gchar **str);
static inline gboolean	skip_string_decoding	(FILE *f);
static inline gboolean	skip_token_decoding	(FILE *f);

static gchar * g_unescape_uri_string (const gchar *escaped, const gchar *illegal_characters);
static void  check_summary_file (DBConnection *db_con, const gchar *filename, MailStore *store);


/********************************************************************************************
 Public functions
*********************************************************************************************/

gboolean
evolution_init_module (void)
{
	EvolutionConfig *conf;

	if (evolution_config) {
		return TRUE;
	}

	conf = NULL;

	if (load_evolution_config (&conf)) {
		evolution_config = conf;
	}

	return evolution_module_is_running ();
}


gboolean
evolution_module_is_running (void)
{
	return evolution_config != NULL;
}


gboolean
evolution_finalize_module (void)
{
	if (!evolution_config) {
		return TRUE;
	}

	free_evolution_config (evolution_config);
	evolution_config = NULL;

	return !evolution_module_is_running ();
}


void
evolution_watch_emails (DBConnection *db_con)
{
	gchar ***res, **row;
	gint  j;

	/* check all registered mbox/paths for deletions */
	res = tracker_db_email_get_mboxes (db_con);

	for (j = 0; (row = tracker_db_get_row (res, j)); j++) {

		if (row[2] && row[3]) {
			MailStore *store = tracker_db_email_get_mbox_details (db_con, row[3]);

			if (store) {
				check_summary_file (db_con, row[2], store);
				tracker_db_email_free_mail_store (store);
			}
		}
	}

	if (res) {
		tracker_db_free_result (res);
	}

	email_watch_directory (evolution_config->dir_local, "EvolutionEmails");

	g_slist_foreach (evolution_config->imap_dirs, (GFunc) email_watch_directory, "EvolutionEmails");
	g_slist_foreach (evolution_config->mh_dirs, (GFunc) email_watch_directory, "EvolutionEmails");
	g_slist_foreach (evolution_config->maildir_dirs, (GFunc) email_watch_directory, "EvolutionEmails");
}


gboolean
evolution_file_is_interesting (FileInfo *info, const gchar *service)
{
	GSList *dir;

	g_return_val_if_fail (info, FALSE);
	g_return_val_if_fail (info->uri, FALSE);
	g_return_val_if_fail (evolution_config, FALSE);
	g_return_val_if_fail (evolution_config->mail_dir, FALSE);

	/* mbox/pop/imap all have summary files (*.ev-summary or "summary") */
	if ((strcmp (info->uri, "summary") == 0) || g_str_has_suffix (info->uri, "summary")) {
		return (!g_str_has_suffix (info->uri, "Drafts.ev-summary") && !g_str_has_suffix (info->uri, "Outbox.ev-summary"));
	} else {
		return FALSE;
	}

	/* But emails into MH dirs need to be treat separately since they can
	   be anywhere! */
	for (dir = evolution_config->mh_dirs; dir; dir = dir->next) {
		const gchar *dir_path;
		gchar       *mime;
		gboolean    result;

		dir_path = dir->data;

		if (g_str_has_prefix (info->uri, dir_path)) {
			mime = tracker_get_mime_type (info->uri);
			result = (strcmp (mime, "message/rfc822") == 0);
			g_free (mime);

			return result;
		}
	}

	/* and same thing for maildir dirs */
	for (dir = evolution_config->maildir_dirs; dir; dir = dir->next) {
		const gchar *dir_path;
		gchar       *mime;
		gboolean    result;

		dir_path = dir->data;

		if (g_str_has_prefix (info->uri, dir_path)) {
			mime = tracker_get_mime_type (info->uri);
			result = (strcmp (mime, "message/rfc822") == 0);
			g_free (mime);

			return result;
		}
	}

	return FALSE;
}


void
evolution_index_file (DBConnection *db_con, FileInfo *info)
{
	gchar *file_name;

	g_return_if_fail (db_con);
	g_return_if_fail (info);

	file_name = g_path_get_basename (info->uri);

	if (g_str_has_prefix (file_name, "junk")) goto end_index;
	if (g_str_has_prefix (file_name, "trash")) goto end_index;
	if (g_str_has_prefix (file_name, "outbox")) goto end_index;
	if (g_str_has_prefix (file_name, "drafts")) goto end_index;

	tracker_debug ("indexing email summary %s", info->uri);

	if (is_in_dir_local (info->uri) && g_str_has_suffix (file_name, ".ev-summary")) {
		/* a MBox file changed */
		gchar       *mbox_file;
		SummaryFile *summary;

		mbox_file = tracker_get_radix_by_suffix (info->uri, ".ev-summary");

		tracker_debug ("summary %s is an mbox", info->uri);

		/* check mbox is registered */
		if (tracker_db_email_get_mbox_id (db_con, mbox_file) == -1) {
			gchar *uri_prefix, *mail_path1, *mail_path2;

			gchar *mbox_dir = g_strconcat (evolution_config->dir_local, "/", NULL);
			mail_path1 =  tracker_string_replace (mbox_file, mbox_dir, NULL);
			g_free (mbox_dir);


			mail_path2 = tracker_string_replace (mail_path1, ".sbd", NULL);

			uri_prefix = g_strdup_printf ("email://%s/%s;uid=", "local@local", mail_path2);

			g_free (mail_path1);
			g_free (mail_path2);

			tracker_db_email_register_mbox (db_con, MAIL_APP_EVOLUTION, MAIL_TYPE_MBOX ,mbox_file, info->uri, uri_prefix);

			g_free (uri_prefix);

		}

		MailStore *store = tracker_db_email_get_mbox_details (db_con, mbox_file);

		summary = NULL;

		if (store && open_summary_file (info->uri, &summary)) {
			SummaryFileHeader *header = NULL;

			tracker_info ("investigating summary file %s", mbox_file);

			if (!load_summary_file_header (summary, &header)) {
				tracker_error ("ERROR: failed to load summary file %s", info->uri);
				free_summary_file (summary);
				goto end_index;
			}

			if (!load_summary_file_meta_header_for_local (summary, header)) {
				tracker_error ("ERROR: failed to load summary header file %s", info->uri);
				free_summary_file_header (header);
				free_summary_file (summary);
				goto end_index;
			}


			if (!email_parse_mail_file_and_save_new_emails (db_con, MAIL_APP_EVOLUTION, mbox_file,
                                                                        load_uri_and_status_of_mbox_mail_message, NULL,
                                                                        NULL, NULL,
                                                                        store)) {
				tracker_info ("setting junk status on email file %s", mbox_file);
				tracker_db_email_flag_mbox_junk (db_con, mbox_file);
			} else {
				tracker_db_email_set_message_counts (db_con, mbox_file, store->mail_count, store->junk_count, store->delete_count);
				tracker_debug ("Number of existing messages in %s are %d, %d junk, %d deleted and header totals are %d, %d, %d", mbox_file,
					store->mail_count, store->junk_count, store->delete_count, header->saved_count, header->junk_count, header->deleted_count);
			}

			free_summary_file_header (header);
			free_summary_file (summary);
		} else {
			tracker_error ("ERROR: failed to open summary file %s", info->uri);
		}

		g_free (mbox_file);

		if (store) {
			tracker_db_email_free_mail_store (store);
		}


	} else if (is_in_dir_imap (info->uri) || is_in_dir_imap4 (info->uri)) {

		if (strcmp (file_name, "summary") == 0) {

			if (is_in_dir_imap4 (info->uri)) {
				index_mail_messages_by_summary_file (db_con, MAIL_TYPE_IMAP4, info->uri,
								     load_summary_file_meta_header_for_imap,
								     load_mail_message_for_imap4,
								     skip_mail_message_for_imap4,
								     save_ondisk_email_message_for_imap4);
			} else {
				index_mail_messages_by_summary_file (db_con, MAIL_TYPE_IMAP, info->uri,
								     load_summary_file_meta_header_for_imap,
								     load_mail_message_for_imap,
								     skip_mail_message_for_imap,
								     save_ondisk_email_message_for_imap);
			}
		}



/* TO DO : add support for mail_dir and mh */
/*	} else if (info->mime && strcmp (info->mime, "message/rfc822") == 0 && g_str_has_suffix (file_name, ".")) {

		MailMessage *ondisk_msg;

		ondisk_msg = email_parse_mail_message_by_path (MAIL_APP_EVOLUTION, info->uri, NULL);
		if (ondisk_msg) {
			tracker_db_email_update_email (db_con, ondisk_msg);
			email_free_mail_message (ondisk_msg);
		}


	} else if (is_in_dir_maildir (info->uri)) {

		if (g_str_has_suffix (info->uri, ".ev-summary")) {
			index_mail_messages_by_summary_file (db_con, info->uri, MAIL_TYPE_MAILDIR,
							     load_summary_file_meta_header_for_maildir,
							     load_mail_message_for_maildir,
							     skip_mail_message_for_maildir);

		} else if (info->mime && strcmp (info->mime, "message/rfc822") == 0) {
			MailMessage *ondisk_msg;

			ondisk_msg = email_parse_mail_message_by_path (MAIL_APP_EVOLUTION, info->uri, NULL);
			if (ondisk_msg) {
				tracker_db_email_update_email (db_con, ondisk_msg);
				email_free_mail_message (ondisk_msg);
			}
		}

	} else if (info->mime && strcmp (info->mime, "message/rfc822") == 0) {


		if (email_mh_is_in_a_mh_dir (info->uri)) {

			try_to_save_ondisk_email_message (db_con, info->uri, NULL);

		}
*/
	}

 end_index:
	g_free (file_name);
}




/********************************************************************************************
 Private functions
*********************************************************************************************/

#define HEX_ESCAPE '%'

static gint
hex_to_int (gchar c)
{
	return  c >= '0' && c <= '9' ? c - '0'
    	: c >= 'A' && c <= 'F' ? c - 'A' + 10
    	: c >= 'a' && c <= 'f' ? c - 'a' + 10
    	: -1;
}


static gint
unescape_character (const gchar *scanner)
{
	gint first_digit;
	gint second_digit;

	first_digit = hex_to_int (*scanner++);

	if (first_digit < 0) {
		return -1;
	}

	second_digit = hex_to_int (*scanner++);
	if (second_digit < 0) {
		return -1;
	}

	return (first_digit << 4) | second_digit;
}


static gchar *
g_unescape_uri_string (const gchar *escaped, const gchar *illegal_characters)
{
	const gchar *in;
	gchar       *out, *result;
	gint        character;

	if (escaped == NULL) {
		return NULL;
	}

	result = g_malloc (strlen (escaped) + 1);

	out = result;
	for (in = escaped; *in != '\0'; in++) {
		character = *in;
		if (character == HEX_ESCAPE) {

			character = unescape_character (in + 1);

			/* Check for an illegal character. We consider '\0' illegal here. */
			if (character == 0 || (illegal_characters != NULL && strchr (illegal_characters, (char)character) != NULL)) {
				g_free (result);
				return NULL;
			}
			in += 2;
		}
		*out++ = character;
	}

	*out = '\0';

	//g_assert (out - result <= strlen (escaped));

	if (!g_utf8_validate (result, -1, NULL)) {
		g_free (result);
		return NULL;
	}

	return result;
}


static void
check_summary_file (DBConnection *db_con, const gchar *filename, MailStore *store)
{
	SummaryFile *summary = NULL;

	g_return_if_fail (store);

	if (open_summary_file (filename, &summary)) {
		SummaryFileHeader *header;
		gchar             *path;

		header = NULL;

		tracker_log ("Scanning summary file %s for junk", filename);

		if (!load_summary_file_header (summary, &header)) {
			free_summary_file (summary);
			tracker_error ("ERROR: failed to open summary file %s", filename);
			return;
		}

		if (store->type == MAIL_TYPE_MBOX) {

			if (!load_summary_file_meta_header_for_local (summary, header)) {
				free_summary_file_header (header);
				free_summary_file (summary);
				tracker_error ("ERROR: failed to open summary header file %s", filename);
				return;
			}
		} else if (store->type == MAIL_TYPE_IMAP || store->type == MAIL_TYPE_IMAP4) {

			if (!load_summary_file_meta_header_for_imap (summary, header)) {
				free_summary_file_header (header);
				free_summary_file (summary);
				tracker_error ("ERROR: failed to open summary header file %s", filename);
				return;
			}
		} else {
			tracker_error ("ERROR: summary file not supported");
			free_summary_file_header (header);
			free_summary_file (summary);
			return;

		}

		path = tracker_db_email_get_mbox_path (db_con, filename);

		if ((header->junk_count > store->junk_count) || (header->deleted_count > store->delete_count)) {

			gchar *mbox_id = tracker_int_to_str (tracker_db_email_get_mbox_id (db_con, path));
			gint i;

			for (i = 0; i < header->saved_count ; i++) {
				guint32 uid = 0;

				if (scan_summary_for_junk (summary, &uid, store->type)) {
					if (uid > 0) {
						gchar *uri, *str_uid;

						str_uid = tracker_uint_to_str (uid);

						tracker_db_email_insert_junk (db_con, path, uid);

						uri = g_strconcat (store->uri_prefix, str_uid, NULL);

						tracker_db_email_delete_email (db_con, uri);

						g_free (uri);
						g_free (str_uid);
						
					}
				} else {
					tracker_error ("ERROR: whilst scanning summary file");
					return;
					break;
				}
			}
			g_free (mbox_id);
		}
		tracker_db_email_reset_mbox_junk (db_con, path);
		tracker_db_email_set_message_counts (db_con, path, store->mail_count, header->junk_count, header->deleted_count);

		g_free (path);

		free_summary_file_header (header);
		free_summary_file (summary);
	}
}


static gchar *
get_account_name_from_imap_uri (const gchar *imap_uri)
{					  
	/* Assume url schema is:
           imap://foo@imap.free.fr/;etc
           or
           imap://foo;auth=DIGEST-MD5@imap.bar.com/;etc

           We try to get "foo@imap.free.fr".
        */
  
        /* check for embedded @ and then look for first colon after that */
  
        const gchar *start = imap_uri + 7;
        const gchar *at = strchr (start, '@');
        const gchar *semic = strchr (start, ';');

        gchar *user_name = NULL;
        gchar *at_host_name = NULL;
        gchar *account_name = NULL;

        if ( strlen (imap_uri) < 7 || at == NULL ) {
                return NULL;
        }

        if (semic < at) {
                /* we have a ";auth=FOO@host" schema
                   Set semic to the next semicolon, which ends the hostname. */
                user_name = g_strndup (start, semic - start);
                /* look for ';' at the end of the domain name */
                semic = strchr (at, ';');
        } else {
                user_name = g_strndup (start, at - start);
        }

        at_host_name = g_strndup (at, (semic - 1) - at);

        account_name = g_strconcat (user_name, at_host_name, NULL);

        g_free (user_name);
        g_free (at_host_name);

        return account_name;
}


static gboolean
load_evolution_config (EvolutionConfig **conf)
{
	char		*dir_imap, *dir_imap4;
	EvolutionConfig	*m_conf;
	GSList		*accounts;
	const GSList	*account;

	if (*conf) {
		free_evolution_config (*conf);
	}

	*conf = g_slice_new0 (EvolutionConfig);
	m_conf = *conf;

	m_conf->mail_dir = g_build_filename (g_get_home_dir (), EVOLUTION_MAIL_DIR_S, NULL);
	m_conf->dir_local = g_build_filename (m_conf->mail_dir, "local", NULL);

	dir_imap = g_build_filename (m_conf->mail_dir, "imap", NULL);
	dir_imap4 = g_build_filename (m_conf->mail_dir, "imap4", NULL);

	accounts = find_accounts_by_gconf (NULL);

	tracker_log ("Checking for Evolution email accounts...");

	for (account = accounts; account; account = account->next) {
		EvolutionAccount *evo_acc = account->data;

		if (evo_acc->enabled) {
			m_conf->accounts = g_slist_prepend (m_conf->accounts, copy_evolution_account (evo_acc));

			if (evo_acc->source_url) {

				switch (evo_acc->protocol) {
					case EVOLUTION_MAIL_PROTOCOL_MBOX: {
						/* currently nothing */
						break;
					}

					case EVOLUTION_MAIL_PROTOCOL_IMAP: {

					        gchar *account_name = get_account_name_from_imap_uri (evo_acc->source_url);

						if (account_name) {

							tracker_log ("Found imap account %s", account_name);

							m_conf->imap_dirs = g_slist_prepend (m_conf->imap_dirs, g_build_filename (dir_imap, account_name, NULL));

							g_free (account_name);

						}

						break;
					}

					case EVOLUTION_MAIL_PROTOCOL_MH: {
						m_conf->mh_dirs = g_slist_prepend (m_conf->mh_dirs, g_strdup (evo_acc->source_url + 5));
						break;
					}

					case EVOLUTION_MAIL_PROTOCOL_MAILDIR: {
						m_conf->maildir_dirs = g_slist_prepend (m_conf->maildir_dirs, g_strdup (evo_acc->source_url + 10));
						break;
					}

					case EVOLUTION_MAIL_PROTOCOL_UNKNOWN:
					default:
						break;
				}
			}
		}
	}

	g_slist_foreach (accounts, (GFunc) free_evolution_account, NULL);
	g_slist_free (accounts);

	g_free (dir_imap);
	g_free (dir_imap4);

	return TRUE;
}


static void
free_evolution_config (EvolutionConfig *conf)
{
	if (!conf) {
		return;
	}

	if (conf->mail_dir) {
		g_free (conf->mail_dir);
	}

	if (conf->dir_local) {
		g_free (conf->dir_local);
	}

	#define FREE_MY_LIST(list, free_fct)				\
		g_slist_foreach (list, (GFunc) free_fct, NULL);		\
		g_slist_free (list);

	FREE_MY_LIST (conf->accounts, free_evolution_account);
	FREE_MY_LIST (conf->imap_dirs, g_free);
	FREE_MY_LIST (conf->mh_dirs, g_free);
	FREE_MY_LIST (conf->maildir_dirs, g_free);

	#undef FREE_MY_LIST

	g_slice_free (EvolutionConfig, conf);
}


static EvolutionAccount *
copy_evolution_account (EvolutionAccount *account)
{
	EvolutionAccount *ret;

	g_return_val_if_fail (account, NULL);

	ret = g_slice_new0 (EvolutionAccount);

	#define DEEP_COPY_STR(val)				\
		if (account->val) {				\
			ret->val = g_strdup (account->val);	\
		}

	DEEP_COPY_STR (name);
	DEEP_COPY_STR (uid);
	ret->enabled = account->enabled;
	ret->protocol = account->protocol;
	DEEP_COPY_STR (source_url);

	#undef DEEP_COPY_STR

	return ret;
}


static void
free_evolution_account (EvolutionAccount *account)
{
	if (!account) {
		return;
	}

	if (account->name) {
		g_free (account->name);
	}

	if (account->uid) {
		g_free (account->uid);
	}

	if (account->source_url) {
		g_free (account->source_url);
	}

	g_slice_free (EvolutionAccount, account);
}


static GSList *
find_accounts_by_gconf (GSList *found_accounts)
{
	gchar *argv[4];
	gchar *text;

	argv[0] = "gconftool";
	argv[1] = "--get";
	argv[2] = "/apps/evolution/mail/accounts";
	argv[3] = NULL;

	text = NULL;

	if (!g_spawn_sync (NULL,
			  argv,
			  NULL,
			  G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL,
			  NULL,
			  NULL,
			  &text,
			  NULL,
			  NULL,
			  NULL)) {

		argv[0] = "gconftool-2";

		if (!g_spawn_sync (NULL,
			  argv,
			  NULL,
			  G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL,
			  NULL,
			  NULL,
			  &text,
			  NULL,
			  NULL,
			  NULL)) {

			return found_accounts;
		}
	}


	if (text) {
		gchar               *to_parse;
		GMarkupParser       *parser;
		GMarkupParseContext *parser_context;
		gchar               **accounts;
		ParsingAccountState *state;

		if (text[0] == '[') {
			size_t len;

			/* Usually a string returned by gconftool begins by character '['.
			   We remove this character. */

			to_parse = text + 1;

			/* and then it probably ends with string "]\n"... */
			len = strlen (text);
			if (text[len - 1] == '\n' && text[len - 2] == ']') {
				text[len - 2] = '\0';
			}
		} else {
			to_parse = text;
		}

		parser = g_slice_new0 (GMarkupParser);
		parser->start_element = account_start_element_handler;
		parser->text = account_text_handler;
		parser->end_element = account_end_element_handler;

		state = g_slice_new0 (ParsingAccountState);
		state->elements = g_queue_new ();

		parser_context = g_markup_parse_context_new (parser, 0, state, NULL);

		/* Accounts look that way:

		   <?xml version="1.0"?>
		   <account name="laurent.aguerreche@free.fr" uid="1134161347.7985.16@foo.bar" enabled="true"><identity><...etc...></account>
		   ,<?xml version="1.0"?>
		   <account name="whrahahaaa@free.fr" uid="1147098156.5252.0@foo.bar" enabled="true"><identity><...etc...></account>

		   so we split accounts around string "\n,".
		 */

		accounts = g_strsplit (to_parse, "\n,", -1);

		if (accounts) {
			gchar **account;

			for (account = accounts; *account; account++) {
				state->account = g_slice_new0 (EvolutionAccount);
				g_markup_parse_context_parse (parser_context, *account, -1, NULL);
				found_accounts = g_slist_prepend (found_accounts, state->account);
			}

			g_strfreev (accounts);
		}

		g_markup_parse_context_free (parser_context);

		g_queue_free (state->elements);
		g_slice_free (ParsingAccountState, state);
		g_slice_free (GMarkupParser, parser);
		g_free (text);
	}

	return found_accounts;
}


#define ELEMENT_ACCOUNT		"account"
#define ELEMENT_SOURCE		"source"

#define ELEMENT_URL		"url"

#define ELEMENT_IS(name) (element_name != NULL && strcmp (element_name, (name)) == 0)
#define PREV_ELEMENT_IS(name) (prev_element_name != NULL && strcmp (prev_element_name, (name)) == 0)

#define ATTRIBUTE_NAME		"name"
#define ATTRIBUTE_UID		"uid"
#define ATTRIBUTE_ENABLED	"enabled"

#define ATTRIBUTE_IS(name) (strcmp (attr_names[i], (name)) == 0)


static void
account_start_element_handler (GMarkupParseContext *context,
			       const gchar         *element_name,
			       const gchar         **attr_names,
			       const gchar         **attr_values,
			       gpointer	           user_data,
			       GError              **error)
{
	ParsingAccountState *state;
	EvolutionAccount    *account;

	state = user_data;

	g_return_if_fail (state);

	account = state->account;

	g_queue_push_head (state->elements, g_strdup (element_name));

	if (ELEMENT_IS (ELEMENT_ACCOUNT)) {
		guint i;
		for (i = 0; attr_names[i]; i++) {
			if (ATTRIBUTE_IS (ATTRIBUTE_NAME)) {
				account->name = g_strdup (attr_values[i]);

			} else if (ATTRIBUTE_IS (ATTRIBUTE_UID)) {
				account->uid = g_strdup (attr_values[i]);

			} else if (ATTRIBUTE_IS (ATTRIBUTE_ENABLED)) {
				account->enabled = (strcmp (attr_values[i], "true") == 0);
			}
		}
	}
}


static void
account_text_handler (GMarkupParseContext *context,
		      const gchar         *text,
		      gsize               text_len,
		      gpointer            user_data,
		      GError              **error)
{
	ParsingAccountState *state;
	const gchar         *element_name, *prev_element_name;

	state = user_data;

	g_return_if_fail (state);

	element_name = g_queue_peek_head (state->elements);
	prev_element_name = g_queue_peek_nth (state->elements, 1);

	if (ELEMENT_IS (ELEMENT_URL) && PREV_ELEMENT_IS (ELEMENT_SOURCE)) {

		if (text_len > 0 && text) {
			gchar *source_url, *esc_source_url;

			source_url = g_strndup (text, text_len);

			tracker_log ("Found email account uri %s", source_url);

			if (strncmp (source_url, "mbox:", 5) == 0 || strncmp (source_url, "pop", 3) == 0) {
				state->account->protocol = EVOLUTION_MAIL_PROTOCOL_MBOX;
			} else if ((strncmp (source_url, "imap:", 5) == 0) || (strncmp (source_url, "imap4:", 6) == 0)) {
				state->account->protocol = EVOLUTION_MAIL_PROTOCOL_IMAP;
			} else if (strncmp (source_url, "mh:", 3) == 0) {
				state->account->protocol = EVOLUTION_MAIL_PROTOCOL_MH;
			} else if (strncmp (source_url, "maildir:", 8) == 0) {
				state->account->protocol = EVOLUTION_MAIL_PROTOCOL_MAILDIR;
			} else {
				state->account->protocol = EVOLUTION_MAIL_PROTOCOL_MBOX;
			}

			esc_source_url = g_unescape_uri_string (source_url, "");
			g_free (source_url);

			state->account->source_url = esc_source_url;
		} else {
			state->account->source_url = NULL;
			state->account->protocol = EVOLUTION_MAIL_PROTOCOL_UNKNOWN;
		}
	}
}


static void
account_end_element_handler (GMarkupParseContext *context,
			     const gchar         *element_name,
			     gpointer            user_data,
			     GError              **error)
{
	ParsingAccountState *state;
	gchar               *my_element_name;

	state = user_data;

	g_return_if_fail (state);

	my_element_name = g_queue_pop_head (state->elements);

	g_free (my_element_name);
}


static gboolean
is_in_dir_local (const gchar *path)
{
	g_return_val_if_fail (path, FALSE);

	return (g_str_has_prefix (path, evolution_config->dir_local));
}


static gboolean
is_in_dir_imap (const gchar *path)
{
	g_return_val_if_fail (path, FALSE);

	return strstr (path, G_DIR_SEPARATOR_S "imap") != NULL;
}


static gboolean
is_in_dir_imap4 (const gchar *path)
{
	g_return_val_if_fail (path, FALSE);

	return strstr (path, G_DIR_SEPARATOR_S "imap4" G_DIR_SEPARATOR_S) != NULL;
}

/*
static gboolean
is_in_dir_mh (const gchar *path)
{
	const GSList *dir;

	g_return_val_if_fail (path, FALSE);

	for (dir = evolution_config->mh_dirs; dir; dir = dir->next) {
		const gchar *dir_path;

		dir_path = dir->data;
		if (g_str_has_prefix (path, dir_path)) {
			return TRUE;
		}
	}

	return FALSE;
}


static gboolean
is_in_dir_maildir (const gchar *path)
{
	const GSList *dir;

	g_return_val_if_fail (path, FALSE);

	for (dir = evolution_config->maildir_dirs; dir; dir = dir->next) {
		const gchar *dir_path;

		dir_path = dir->data;
		if (g_str_has_prefix (path, dir_path)) {
			return TRUE;
		}
	}

	return FALSE;
}
*/


static gchar *
get_account_name_in_imap_path (const gchar *path)
{
	const gchar *pos_imap;

	g_return_val_if_fail (path, NULL);

	pos_imap = strstr (path, G_DIR_SEPARATOR_S "imap");

	if (pos_imap) {
		const gchar *s, *s_end;

		s = pos_imap + 5;	/* strlen (G_DIR_SEPARATOR_S "imap") == 5 */

		if (s[0] == '4') {
			/* we are reading the end of word "imap4" */
			s++;

			if (s[0] == '\0') {
				return NULL;
			}
		}

		/* we are currently on a G_DIR_SEPARATOR character. Let's go further. */
		s++;

		if (s[0] == G_DIR_SEPARATOR || s[0] == '\0') {
			return NULL;
		}

		s_end = strchr (s, G_DIR_SEPARATOR);

		return (s_end) ? g_strndup (s, s_end - s) : g_strdup (s);
	}

	return NULL;
}


static void
index_mail_messages_by_summary_file (DBConnection                 *db_con,
                                     MailType                     mail_type,
				     const gchar                  *summary_file_path,
				     LoadSummaryFileMetaHeaderFct load_meta_header,
				     LoadMailMessageFct           load_mail,
				     SkipMailMessageFct           skip_mail,
				     SaveOnDiskMailMessageFct     save_ondisk_mail)
{
	SummaryFile *summary = NULL;

	if (!tracker->is_running || !tracker->enable_indexing) return; 

	if (open_summary_file (summary_file_path, &summary)) {
		SummaryFileHeader *header;
		gint32            mail_count, junk_count, delete_count;
		gchar             *dir;

		header = NULL;

		if (!load_summary_file_header (summary, &header)) {
			free_summary_file (summary);
			return;
		}

		if (!(*load_meta_header) (summary, header)) {
			free_summary_file_header (header);
			free_summary_file (summary);
			return;
		}

		dir = g_path_get_dirname (summary->path);

		/* check summary file is registered */
		if (tracker_db_email_get_mbox_id (db_con, dir) == -1) {
			const gchar *pos_folders = strstr (dir, G_DIR_SEPARATOR_S "folders" G_DIR_SEPARATOR_S);

			if (pos_folders) {
				size_t len_pos_folders = strlen (pos_folders);

				if (len_pos_folders > 9) {	/* strlen ("/folders/") == 9 */
					gchar *uri_dir, *clean_uri_dir, *uri_prefix;

					pos_folders += 9;

					uri_dir = NULL;

					/* we find relative summary file path from directory INBOX (or Sent, etc.).
					   It can be INBOX/, INBOX/sent-mail ,etc. */
					if (pos_folders) {
						uri_dir = g_strdup (pos_folders);
					}

					clean_uri_dir = NULL;

					if (uri_dir) {
						gchar *tmp_str = tracker_string_replace (uri_dir, "subfolders/", NULL);
						clean_uri_dir = tracker_string_replace (tmp_str, "folders/", NULL);
						g_free (tmp_str);
					}

					if (clean_uri_dir) {
						uri_prefix = g_strdup_printf ("email://%s/%s;uid=", summary->associated_account->uid, clean_uri_dir);
						g_free (clean_uri_dir);
					} else {
						uri_prefix = g_strdup_printf ("email://%s;uid=", summary->associated_account->uid);
					}

					g_free (uri_dir);

					tracker_db_email_register_mbox (db_con, MAIL_APP_EVOLUTION, mail_type, dir, summary_file_path, uri_prefix);

					g_free (uri_prefix);
				}
			}
		}

		MailStore *store = tracker_db_email_get_mbox_details (db_con, dir);

                if (!store) {
			tracker_error ("ERROR: could not retrieve store for file %s", dir);
			free_summary_file (summary);
			free_summary_file_header (header);

			g_free (dir);
			return;
		}

		tracker_debug ("Number of existing messages in %s are %d, %d junk, %d deleted and header totals are %d, %d, %d", dir,
				store->mail_count, store->junk_count, store->delete_count, header->saved_count, header->junk_count, header->deleted_count);

		tracker->mbox_count++;
		tracker_dbus_send_index_progress_signal ("Emails", dir);

		if (header->saved_count > store->mail_count) {
			/* assume new emails received */

			gint i;

			/* skip already indexed emails */
			for (i = 0; i < store->mail_count; i++) {
				if (!(*skip_mail) (summary)) {
					tracker_error ("ERROR: skipping email no. %d in summary file", i+1);
					tracker_db_email_free_mail_store (store);
					free_summary_file (summary);
					free_summary_file_header (header);
					tracker->mbox_processed++;
					g_free (dir);
					return;
				}
			}

			mail_count = 0;
			junk_count = 0;
			delete_count = 0;

			/* now we will read the new emails */
			for (i = store->mail_count; i < header->saved_count; i++) {
				MailMessage *mail_msg = NULL;

				mail_count++;
				tracker_debug ("processing email no. %d / %" G_GINT32_FORMAT, store->mail_count + 1, header->saved_count);

				if (!(*load_mail) (summary, &mail_msg)) {
					tracker_error ("ERROR: loading email no. %d in summary file", mail_count);
					tracker_db_email_free_mail_store (store);
					free_summary_file (summary);
					free_summary_file_header (header);
					tracker->mbox_processed++;
					g_free (dir);
					return;
				}

				gchar *sum_file_dir = g_path_get_dirname (summary->path);
				mail_msg->path = g_strdup_printf ("%s%s%d.", sum_file_dir, G_DIR_SEPARATOR_S, mail_msg->id);
				g_free (sum_file_dir);

				gchar *str_id = tracker_int_to_str (mail_msg->id);
				mail_msg->uri = g_strconcat (store->uri_prefix, str_id, NULL);
				g_free (str_id);

				mail_msg->store = store;

				if (!(*save_ondisk_mail) (db_con, mail_msg)) {
					tracker_log ("WARNING: Message, or message parts, could not be found locally - if you are using IMAP make sure you have selected the \"copy folder content locally for offline operation\" option in Evolution");
					/* we do not have all infos but we still save them */
					if (!tracker_db_email_save_email (db_con, mail_msg)) {
						tracker_log ("Failed to save email");
					}
				}

				if (mail_msg->junk) {
					junk_count++;
				}

				if (mail_msg->deleted) {
					delete_count++;
				}

				email_free_mail_file (mail_msg->parent_mail_file);
				email_free_mail_message (mail_msg);

				if (!tracker_cache_process_events (db_con->data, TRUE)) {
					tracker->status = STATUS_SHUTDOWN;
					tracker->shutdown = TRUE;
					tracker_dbus_send_index_status_change_signal ();
					return;
				}

				if (tracker_db_regulate_transactions (db_con->data, 500)) {
					if (tracker->verbosity == 1) {
						tracker_log ("indexing #%d - Emails in %s", tracker->index_count, dir);
					}

					if (tracker->index_count % 1000 == 0) {
						tracker_db_end_index_transaction (db_con->data);
						tracker_db_refresh_all (db_con->data);
						tracker_db_start_index_transaction (db_con->data);
					}
					
					tracker_dbus_send_index_progress_signal ("Emails", dir);
								
				}

			
			}

			tracker_log ("No. of new emails indexed in summary file %s is %d, %d junk, %d deleted", dir, mail_count, junk_count, delete_count);

			tracker_db_email_set_message_counts (db_con, dir, store->mail_count, store->junk_count, store->delete_count);

		} else {
			/* schedule check for junk */
			tracker_db_email_flag_mbox_junk (db_con, dir);
		}

		tracker->mbox_processed++;
		tracker_dbus_send_index_progress_signal ("Emails", dir);

		tracker_db_email_free_mail_store (store);
		free_summary_file (summary);
		free_summary_file_header (header);

		g_free (dir);
	}
}


static void
load_uri_and_status_of_mbox_mail_message (GMimeMessage *g_m_message, MailMessage *msg, gpointer user_data)
{
	const gchar	  *field;
	gchar		  **parts;
	unsigned long int flags;

	g_return_if_fail (g_m_message);
	g_return_if_fail (msg);

	field = g_mime_message_get_header (g_m_message, "X-Evolution");

	g_return_if_fail (field);

	/* we want to split lines with that form: 00001fd3-0100 into 00001fd3 and 0100 */
	parts = g_strsplit (field, "-", -1);

	if (!parts || !parts[0] || !parts[1]) {
		g_strfreev (parts);
		return;
	}

	/* parts[0] contains UID to contruct mail URI */

	msg->id = strtoul (parts[0], NULL, 16);

	/* parts[1] contains flags for status */

	flags = strtoul (parts[1], NULL, 16);

	if ((flags & EVOLUTION_MESSAGE_DELETED) == EVOLUTION_MESSAGE_DELETED) {
		msg->deleted = TRUE;
	}

	if ((flags & EVOLUTION_MESSAGE_JUNK) == EVOLUTION_MESSAGE_JUNK) {
		msg->junk = TRUE;
	}

	g_strfreev (parts);

}


static gboolean
open_summary_file (const gchar *path, SummaryFile **summary)
{
        gint fd;
        FILE *f;

	g_return_val_if_fail (path, FALSE);

	if (!tracker_file_is_indexable (path)) {
		return FALSE;
	}

	if (*summary) {
		free_summary_file (*summary);
                *summary = NULL;
	}

	fd = tracker_file_open (path, TRUE);

        if (fd == -1) {
        	return FALSE;
        }

        f = fdopen (fd, "r");
        if (!f) {
                tracker_file_close (fd, TRUE);
                return FALSE;
        }

	*summary = g_slice_new0 (SummaryFile);
	(*summary)->f = f;
	(*summary)->path = g_strdup (path);

	if (is_in_dir_local ((*summary)->path)) {
		(*summary)->associated_account = NULL;
		return TRUE;
	}

	/* find associated Evo account */
	{
		const GSList *account;

		for (account = evolution_config->accounts; account; account = account->next) {
			EvolutionAccount *evo_acc = account->data;

			switch (evo_acc->protocol) {
				case EVOLUTION_MAIL_PROTOCOL_MBOX:

					if (is_in_dir_local ((*summary)->path)) {
						/* is it possible to have more than one MBox account? */
						(*summary)->associated_account = evo_acc;
						goto loop_exit;
					}
					break;

				case EVOLUTION_MAIL_PROTOCOL_IMAP: {
					gchar *account_name = get_account_name_in_imap_path ((*summary)->path);

					if (!evo_acc->source_url || !account_name) {
						continue;
					}

					tracker_debug ("Account name for summary file is %s and path is %s", account_name, (*summary)->path);

					gchar *source_account_name = get_account_name_from_imap_uri (evo_acc->source_url);

					if (source_account_name) {
                                                if (!strcmp (source_account_name, account_name)) {
                                                        (*summary)->associated_account = evo_acc;
                                                        g_free (source_account_name);
                                                        g_free (account_name);
                                                        goto loop_exit;
                                                }
                                                g_free (source_account_name);
                                        }
					g_free (account_name);
					break;
				}

				case EVOLUTION_MAIL_PROTOCOL_MH:
				case EVOLUTION_MAIL_PROTOCOL_MAILDIR: {
					gchar *sum_file_dir = g_path_get_dirname ((*summary)->path);
					if (g_str_has_prefix (evo_acc->source_url, sum_file_dir)) {
						(*summary)->associated_account = evo_acc;
						g_free (sum_file_dir);
						goto loop_exit;
					}
					g_free (sum_file_dir);
					break;
				}

				default:
					tracker_error ("ERROR: no matching account found for email");
					continue;
					break;
			}
		}
	}

	loop_exit:

	if (!(*summary)->associated_account) {
		free_summary_file (*summary);
		*summary = NULL;

		return FALSE;
	} else {
		return TRUE;
	}
}


static void
free_summary_file (SummaryFile *summary)
{
	if (!summary) {
		return;
	}

	fclose (summary->f);

	g_free (summary->path);
	g_slice_free (SummaryFile, summary);
}


static gboolean
load_summary_file_header (SummaryFile *summary, SummaryFileHeader **header)
{
	SummaryFileHeader *h;
	FILE              *f;

	g_return_val_if_fail (summary, FALSE);

	if (*header) {
		free_summary_file_header (*header);
	}

	*header = g_slice_new0 (SummaryFileHeader);

	h = *header;

	h->uri_prefix = NULL;

	f = summary->f;


	if (!decode_gint32 (f, &h->version)) {
		goto error;
	}

	tracker_debug ("summary.version = %d", h->version);

	if (h->version > 0xff && (h->version & 0xff) < 12) {
		tracker_error ("ERROR: summary file header version too low");
		goto error;
	}

	h->legacy = !(h->version < 0x100 && h->version >= 13);

	if (h->legacy) {
		tracker_debug ("WARNING: summary file is a legacy version");
	}


	if (!decode_gint32 (f, &h->flags) ||
	    !decode_gint32 (f, &h->nextuid) ||
	    !decode_time_t (f, &h->time) ||
	    !decode_gint32 (f, &h->saved_count)) {
		goto error;
	}

	tracker_debug ("summary.flags = %d", h->flags);
	tracker_debug ("summary.nextuid = %d", h->nextuid);
	tracker_debug ("summary.time = %d", h->time);
	tracker_debug ("summary.count = %" G_GINT32_FORMAT, h->saved_count);

	if (!h->legacy) {
		if (!decode_gint32 (f, &h->unread_count) ||
		    !decode_gint32 (f, &h->deleted_count) ||
		    !decode_gint32 (f, &h->junk_count)) {
			goto error;
		}
	}

	tracker_debug ("summary.Unread = %d", h->unread_count);
	tracker_debug ("summary.deleted = %d", h->deleted_count);
	tracker_debug ("summary.junk = %d", h->junk_count);

	return TRUE;

 error:
	free_summary_file_header (*header);
	*header = NULL;

	return FALSE;
}


static void
free_summary_file_header (SummaryFileHeader *header)
{
	if (!header) {
		return;
	}

	if (header->uri_prefix) {
		g_free (header->uri_prefix);
	}

	g_slice_free (SummaryFileHeader, header);
}


static gboolean
load_summary_file_meta_header_for_local (SummaryFile *summary, SummaryFileHeader *header)
{
	FILE   *f;
	gint32 dummy;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	if (!decode_gint32 (f, &dummy) ||	/* major */
	    !decode_gint32 (f, &dummy) ||	/* minor */
	    !decode_gint32 (f, &dummy)) {	/* uid_len */
		return FALSE;
	}

	return TRUE;
}


static gboolean
load_summary_file_meta_header_for_imap (SummaryFile *summary, SummaryFileHeader *header)
{
	FILE    *f;
	guint32 dummy0;

	g_return_val_if_fail (summary, FALSE);
	g_return_val_if_fail (header, FALSE);

	f = summary->f;

	/* check for legacy version */
	if (header->version != 0x30c) {
		gint32 version, dummy1;

		if (!decode_gint32 (f, &version)) {
			return FALSE;
		}

		if (version < 0) {
			tracker_error ("ERROR: summary file version too low");
			return FALSE;
		}

		/* Right now we only support summary versions 1 through 3 */
		if (version > 3) {
			tracker_error ("ERROR: reported summary version (%" G_GINT32_FORMAT ") is too new", version);
			return FALSE;
		}

		if (version == 2) {
			if (!skip_gint32_decoding (f)) {
				return FALSE;
			}
		}

		/* validity */
		if (!decode_gint32 (f, &dummy1)) {
			return FALSE;
		}
	} else {
		/* validity */
		if (!decode_guint32 (f, &dummy0)) {
			return FALSE;
		}
	}

	return TRUE;
}

/*
static gboolean
load_summary_file_meta_header_for_maildir (SummaryFile *summary, SummaryFileHeader *header)
{
	gint32 version;

	g_return_val_if_fail (summary, FALSE);
	g_return_val_if_fail (header, FALSE);

	return decode_gint32 (summary->f, &version);
}
*/


/*
static gboolean
load_mail_message_for_local (SummaryFile *summary, MailMessage **mail_msg)
{
	off_t offset;

 	g_return_val_if_fail (summary, FALSE);

 	if (*mail_msg) {
 		email_free_mail_message (*mail_msg);
 	}

 	*mail_msg = email_allocate_mail_message ();

 	if (!load_mail_message (summary, *mail_msg)) {
 		goto error;
 	}

	if (!decode_off_t (summary->f, &offset)) {
 		goto error;
 	}

 	return TRUE;

 error:
 	email_free_mail_message (*mail_msg);
 	*mail_msg = NULL;
 	return FALSE;
}
*/


static gboolean
load_mail_message_for_imap (SummaryFile *summary, MailMessage **mail_msg)
{
	return do_load_mail_message_for_imap (summary, mail_msg, TRUE);
}


static gboolean
load_mail_message_for_imap4 (SummaryFile *summary, MailMessage **mail_msg)
{
	return do_load_mail_message_for_imap (summary, mail_msg, FALSE);
}


static gboolean
do_load_mail_message_for_imap (SummaryFile *summary, MailMessage **mail_msg, gboolean do_skipping_of_content_info)
{
	guint32 server_flags;

	g_return_val_if_fail (summary, FALSE);

	if (*mail_msg) {
		email_free_mail_message (*mail_msg);
	}

	*mail_msg = email_allocate_mail_message ();

	if (!load_mail_message (summary, *mail_msg)) {
		goto error;
	}

	if (!decode_guint32 (summary->f, &server_flags)) {
		goto error;
	}

	if (do_skipping_of_content_info) {
		if (!skip_loading_content_info (summary)) {
			goto error;
		}
	}

	return TRUE;

 error:
	email_free_mail_message (*mail_msg);
	*mail_msg = NULL;
	return FALSE;
}


/*
static gchar *
get_relative_path_from_source_url (const gchar *url)
{
	size_t url_len, i;

	g_return_val_if_fail (url, NULL);

	url_len = strlen (url);
	if (url_len == 0) {
		return NULL;
	}

	for (i = url_len - 1; i >= 0; i--) {
		if (url[i] == '#') {
			return g_strdup (url + i);
		}
	}

	return g_strdup ("");
}
*/
/*
static gboolean
load_mail_message_for_maildir (SummaryFile *summary, MailMessage **mail_msg)
{
	EvolutionAdHocInfos *adhoc_infos;

	g_return_val_if_fail (summary, FALSE);

	if (*mail_msg) {
		email_free_mail_message (*mail_msg);
	}

	*mail_msg = email_allocate_mail_message ();

	adhoc_infos = g_slice_new0 (EvolutionAdHocInfos);

	if (!load_mail_message (summary, *mail_msg, adhoc_infos)) {
		goto error;
	}

	if (adhoc_infos->mail_uid) {
		gchar *uri_dir;

		uri_dir = get_relative_path_from_source_url (summary->associated_account->source_url);
		if (!uri_dir) {
			goto error;
		}

		(*mail_msg)->uri = make_uri (summary->associated_account->uid, uri_dir, adhoc_infos->mail_uid);
		g_free (uri_dir);
	}

	free_evolution_adhoc_infos (adhoc_infos);

	return TRUE;

 error:
	email_free_mail_message (*mail_msg);
	*mail_msg = NULL;
	return FALSE;
}
*/


static gboolean
load_mail_message (SummaryFile *summary, MailMessage *mail_msg)
{
	FILE    *f;
	guint32	flags, size, count;
	time_t  date_sent, date_received;
	gchar   *uid, *to, *cc, *mlist;

	g_return_val_if_fail (summary, FALSE);
	g_return_val_if_fail (mail_msg, FALSE);

	f = summary->f;

	if (!decode_string (f, &uid) ||
	    !decode_guint32 (f, &flags) ||
	    !decode_guint32 (f, &size) ||
	    !decode_time_t (f, &date_sent) ||
	    !decode_time_t (f, &date_received) ||
	    !decode_string (f, &mail_msg->subject) ||
	    !decode_string (f, &mail_msg->from) ||
	    !decode_string (f, &to) ||
	    !decode_string (f, &cc) ||
	    !decode_string (f, &mlist)) {

		return FALSE;
	}

	g_free (mlist);


	mail_msg->id = strtoul (uid, NULL, 10);

	g_free (uid);

	if ((flags & EVOLUTION_MESSAGE_DELETED) == EVOLUTION_MESSAGE_DELETED) {
		mail_msg->deleted = TRUE;
	}

	if ((flags & EVOLUTION_MESSAGE_JUNK) == EVOLUTION_MESSAGE_JUNK) {
		mail_msg->junk = TRUE;
	}

	mail_msg->date = (long) date_received;

	mail_msg->to = add_persons_from_internet_address_list_string_parsing (NULL, to);
	mail_msg->cc = add_persons_from_internet_address_list_string_parsing (NULL, cc);

	g_free (to);
	g_free (cc);

	skip_gint32_decoding (f);	/* mi->message_id.id.part.hi */
	skip_gint32_decoding (f);	/* mi->message_id.id.part.lo */

	/* references */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.hi */
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.lo */
		}
	} else {
		return FALSE;
	}

	/* user flags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* a flag */
		}
	} else {
		return FALSE;
	}

	/* user tags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* tag name */
			skip_string_decoding (f);	/* tag value */
		}
	} else {
		return FALSE;
	}

	return TRUE;
}


static gboolean
skip_mail_message_for_imap (SummaryFile *summary)
{
	return do_skip_mail_message_for_imap (summary, TRUE);
}


static gboolean
skip_mail_message_for_imap4 (SummaryFile *summary)
{
	return do_skip_mail_message_for_imap (summary, TRUE);
}


static gboolean
do_skip_mail_message_for_imap (SummaryFile *summary, gboolean do_skipping_of_content_info)
{
	g_return_val_if_fail (summary, FALSE);

	if (!skip_mail_message (summary)) {
		return FALSE;
	}

	if (!skip_guint32_decoding (summary->f)) {
		return FALSE;
	}

	if (do_skipping_of_content_info) {
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}
	}

	return TRUE;
}


/*
static gboolean
skip_mail_message_for_maildir (SummaryFile *summary)
{
	return skip_mail_message (summary);
}
*/


static gboolean
skip_mail_message (SummaryFile *summary)
{
	FILE    *f;
	guint32 count;
	time_t  tt;
	guint   n;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

/*	if (!skip_over_mail (f)) {
		return FALSE;
	}
*/

	if (!skip_string_decoding (f))
		return FALSE;

	if (!decode_guint32 (f, &n))
		return FALSE;

	if (!decode_guint32 (f, &n))
		return FALSE;

	if (!decode_time_t (f, &tt))
		return FALSE;

	if (!decode_time_t (f, &tt))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;


	skip_gint32_decoding (f);	/* mi->message_id.id.part.hi */
	skip_gint32_decoding (f);	/* mi->message_id.id.part.lo */

	/* references */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.hi */
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.lo */
		}
	} else {
		return FALSE;
	}

	/* user flags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* a flag */
		}
	} else {
		return FALSE;
	}

	/* user tags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* tag name */
			skip_string_decoding (f);	/* tag value */
		}
	} else {
		return FALSE;
	}

	return TRUE;
}


static inline gboolean
skip_over_mail (FILE *f)
{
	return (skip_string_decoding (f) &&  /* uid */
                skip_guint32_decoding (f) && /* flags */
                skip_guint32_decoding (f) && /* size */
                skip_time_t_decoding (f) &&  /* date_sent */
                skip_time_t_decoding (f) &&  /* date_received */
                skip_string_decoding (f) &&  /* subject */
                skip_string_decoding (f) &&  /* from */
                skip_string_decoding (f) &&  /* to */
                skip_string_decoding (f) &&  /* cc */
                skip_string_decoding (f));   /* mlist */
}


static gboolean
skip_loading_content_info (SummaryFile *summary)
{
	guint32 count, i;

	g_return_val_if_fail (summary, FALSE);

	if (!do_skip_loading_content_info (summary)) {
		return FALSE;
	}

	if (!decode_guint32 (summary->f, &count)) {
		return FALSE;
	}

	if (count > 500) {
		return FALSE;
	}

	for (i = 0; i < count; i++) {
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}
	}

	return TRUE;
}


static gboolean
do_skip_loading_content_info (SummaryFile *summary)
{
	FILE    *f;
	guint32	count;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	if (!fgetc (f)) {	/* one byte to read... */
		return TRUE;
	}

	skip_token_decoding (f);	/* type */
	skip_token_decoding (f);	/* subtype */

	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_token_decoding (f);	/* name */
			skip_token_decoding (f);	/* value */
		}
	} else {
		return FALSE;
	}

	if (!skip_token_decoding (f) ||		/* id */
	    !skip_token_decoding (f) ||		/* description */
	    !skip_token_decoding (f) ||		/* encoding */
	    !decode_guint32 (f, &count)) {	/* size */
		return FALSE;
	}

	return TRUE;
}


static gboolean
save_ondisk_email_message_for_imap (DBConnection *db_con, MailMessage *mail_msg)
{
	return do_save_ondisk_email_message_for_imap (db_con, mail_msg);
}


static gboolean
save_ondisk_email_message_for_imap4 (DBConnection *db_con, MailMessage *mail_msg)
{
	return do_save_ondisk_email_message_for_imap (db_con, mail_msg);
}


static gboolean
do_save_ondisk_email_message_for_imap (DBConnection *db_con, MailMessage *mail_msg)
{
	g_return_val_if_fail (db_con, FALSE);
	g_return_val_if_fail (mail_msg, FALSE);

	tracker_log ("Trying to index mail \"%s\"", mail_msg->uri);

	if (!do_save_ondisk_email_message (db_con, mail_msg)) {
		/* Mail not found... So two cases:
		 * 1/ mail naming uses this schema:
		 *    - id.HEADER
		 *    - id.1.MIME
		 *    - id.1     <- contains a mail part
		 *    - id.2.MIME
		 *    - id.2
		 *    etc.
		 *    But we may also have something like
		 *    id.1.1.MIME, etc.
		 * 2/ mail do not exist on disk.
		 */
		gchar    *header_file;
		gboolean ret;

		ret = FALSE;

		tracker_log ("... Trying to index mail parts");

		header_file = g_strconcat (mail_msg->path, "HEADER", NULL);

		if (tracker_file_is_indexable (header_file)) {
			/* email is on disk */
			tracker_log ("... Indexing mail parts of email \"%s\"", mail_msg->uri);
			ret = index_mail_parts (db_con, mail_msg, mail_msg->path);
			tracker_log ("... Treatment of mail parts of \"%s\" finished", mail_msg->uri);

			if (ret) {
				tracker_db_email_save_email (db_con, mail_msg);
			}
		} else {
                        tracker_log ("...Indexing of mail parts failed");
                }

		g_free (header_file);

		return ret;

	} else {
		tracker_log ("Simple index of mail \"%s\" finished", mail_msg->uri);
		return TRUE;
	}
}


static gboolean
do_save_ondisk_email_message (DBConnection *db_con, MailMessage *mail_msg)
{
	g_return_val_if_fail (db_con, FALSE);
	g_return_val_if_fail (mail_msg, FALSE);
	g_return_val_if_fail (mail_msg->path, FALSE);

	if (tracker_file_is_indexable (mail_msg->path)) {
		/* we have downloaded the mail message on disk so we can fully index it. */
		MailMessage *mail_msg_on_disk = NULL;

		mail_msg_on_disk = email_parse_mail_message_by_path (MAIL_APP_EVOLUTION,
                                                                     mail_msg->path, NULL, NULL);

		if (mail_msg_on_disk && mail_msg_on_disk->parent_mail_file) {

			mail_msg_on_disk->parent_mail_file->next_email_offset = 0;
			mail_msg_on_disk->uri = g_strdup (mail_msg->uri);
			mail_msg_on_disk->store = mail_msg->store;

			tracker_db_email_save_email (db_con, mail_msg_on_disk);

                        email_free_mail_file (mail_msg_on_disk->parent_mail_file);
			email_free_mail_message (mail_msg_on_disk);

			return TRUE;
		}
	}

	return FALSE;
}


static gboolean
index_mail_parts (DBConnection *db_con, MailMessage *mail_msg, const gchar *mail_part_radix)
{
	GQueue *mail_parts;
	gint   i, num_parts;

	typedef struct {
		gchar	  *mail_file;
		gchar	  *mime_file;
		MimeInfos *mime_infos;
	} MailPart;

	#define free_mail_part(x)			\
		g_free (x->mail_file);			\
		g_free (x->mime_file);			\
		email_free_mime_infos (x->mime_infos);	\
		g_slice_free (MailPart, x);

	g_return_val_if_fail (db_con, FALSE);
	g_return_val_if_fail (mail_msg, FALSE);
	g_return_val_if_fail (mail_part_radix, FALSE);

	mail_parts = g_queue_new ();

	/* makes first names of mail parts to index  */
	for (i = 1; ; i++) {
		gchar    *str_i;
		MailPart *m;

		str_i = tracker_int_to_str (i);
		m = g_slice_new0 (MailPart);
		m->mail_file = g_strconcat (mail_part_radix, str_i, NULL);
		m->mime_file = g_strconcat (m->mail_file, ".MIME", NULL);
		g_free (str_i);

		if (tracker_file_is_indexable (m->mime_file)) {
			m->mime_infos = email_get_mime_infos_from_mime_file (m->mime_file);
			if (! m->mime_infos) {
				goto break_multipart_loop;
			}
			g_queue_push_tail (mail_parts, m);
		} else {
break_multipart_loop:
			g_free (m->mail_file);
			g_free (m->mime_file);
			g_slice_free (MailPart, m);
			break;
		}
	}

	mail_msg->body = NULL;

	num_parts = 0;

	/* find mail parts in queue */
	while (!g_queue_is_empty (mail_parts)) {
		MailPart *m;

		num_parts++;

		m = g_queue_pop_head (mail_parts);

		if (strcmp (m->mime_infos->type, "multipart") == 0) {
			gint j;

			for (j = 1; ; j++) {
				gchar		*str_j;
				MailPart	*inner_m;

				str_j = tracker_int_to_str (j);
				inner_m = g_slice_new0 (MailPart);
				inner_m->mail_file = g_strconcat (m->mail_file, ".", str_j, NULL);
				inner_m->mime_file = g_strconcat (inner_m->mail_file, ".MIME", NULL);
				g_free (str_j);

				if (tracker_file_is_indexable (inner_m->mime_file)) {
					inner_m->mime_infos = email_get_mime_infos_from_mime_file (inner_m->mime_file);
					g_queue_push_tail (mail_parts, inner_m);

				} else {
					g_free (inner_m->mail_file);
					g_free (inner_m->mime_file);
					g_slice_free (MailPart, inner_m);
					break;
				}
			}

		} else{
			if (tracker_file_is_indexable (m->mail_file)) {
				tracker_log ("Indexing message part \"%s\"", m->mail_file);

				if (num_parts == 1 && strcmp (m->mime_infos->type, "text") == 0) {
					/* it is the first mail part so it is assume to be body mail */
					tracker_log ("Message part \"%s\" will be indexed as mail body...", m->mail_file);
					g_file_get_contents (m->mail_file, &mail_msg->body, NULL, NULL);

				} else {
					if (m->mime_infos) {
						MailAttachment	*ma;
                                                gchar           *tmp_filename;

						ma = email_allocate_mail_attachment ();

						ma->attachment_name = (m->mime_infos->name ?
                                                                       g_strdup (m->mime_infos->name) :
                                                                       g_path_get_basename (m->mail_file)       /* this attachment has not a name... */
                                                                       );
                                                tmp_filename = (m->mime_infos->name ?
                                                                g_path_get_basename (m->mime_infos->name) :     /* we only take end of name to avoid to give
                                                                                                                   access to anywhere on disk if the name
                                                                                                                   contains characters from a path... */
                                                                g_strdup (ma->attachment_name)
                                                                );
						ma->mime = g_strconcat (m->mime_infos->type, "/", m->mime_infos->subtype, NULL);
						ma->tmp_decoded_file  = email_make_tmp_name_for_mail_attachment (tmp_filename);
                                                g_free (tmp_filename);

						if (email_decode_mail_attachment_to_file (m->mail_file, ma->tmp_decoded_file, m->mime_infos->encoding)) {
							email_add_saved_mail_attachment_to_mail_message (mail_msg, ma);

						} else {
							tracker_error ("ERROR: failed to decode mail part \"%s\"", m->mail_file);
							email_free_mail_attachment (ma);
						}
					}
				}
			}
		}

		free_mail_part (m);
	}

	#undef free_mail_part

	g_queue_free (mail_parts);

	return num_parts != 0;
}


static GSList *
add_persons_from_internet_address_list_string_parsing (GSList *list, const gchar *s)
{
	InternetAddressList *addrs_list, *tmp;

	g_return_val_if_fail (s, NULL);

	addrs_list = internet_address_parse_string (s);

	for (tmp = addrs_list; tmp; tmp = tmp->next) {
		MailPerson *mp;

		mp = email_allocate_mail_person ();

		mp->addr = g_strdup (tmp->address->value.addr);
		if(tmp->address->name)
                	mp->name = g_strdup (tmp->address->name);
		else
			mp->name = g_strdup (tmp->address->value.addr);

		list = g_slist_prepend (list, mp);
	}

	internet_address_list_destroy (addrs_list);

	return list;
}


static gboolean
scan_summary_for_junk (SummaryFile *summary, guint32 *uid, MailType type)
{
	FILE    *f;
	guint32	count, flags;
	time_t  tt;
	guint   n;
	gchar   *return_uid;

	g_return_val_if_fail (summary, FALSE);

	f = summary->f;

	if (!decode_string (f, &return_uid))
		return FALSE;

	if (!decode_guint32 (f, &flags))
		return FALSE;

	if (!decode_guint32 (f, &n))
		return FALSE;

	if (!decode_time_t (f, &tt))
		return FALSE;

	if (!decode_time_t (f, &tt))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	if (!skip_string_decoding (f))
		return FALSE;

	*uid = 0;

	skip_gint32_decoding (f);	/* mi->message_id.id.part.hi */
	skip_gint32_decoding (f);	/* mi->message_id.id.part.lo */

	/* references */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.hi */
			skip_gint32_decoding (f);	/* mi->references->references[i].id.part.lo */
		}
	} else {
		return FALSE;
	}

	/* user flags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* a flag */
		}
	} else {
		return FALSE;
	}

	/* user tags */
	if (decode_guint32 (f, &count) && count <= 500) {
		guint32 i;
		for (i = 0; i < count; i++) {
			skip_string_decoding (f);	/* tag name */
			skip_string_decoding (f);	/* tag value */
		}
	} else {
		return FALSE;
	}


	if ((flags & EVOLUTION_MESSAGE_DELETED) == EVOLUTION_MESSAGE_DELETED) {
		*uid = strtoul (return_uid, NULL, 10);


	} else if ((flags & EVOLUTION_MESSAGE_JUNK) == EVOLUTION_MESSAGE_JUNK) {
		*uid = strtoul (return_uid, NULL, 10);
	}

	if (type == MAIL_TYPE_MBOX) {
		skip_off_t_decoding (summary->f);

	} else 	if (type == MAIL_TYPE_IMAP) {
		skip_guint32_decoding (summary->f);
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}


	} else 	if (type == MAIL_TYPE_IMAP4) {
		skip_guint32_decoding (summary->f);
		if (!skip_loading_content_info (summary)) {
			return FALSE;
		}
	}

	return TRUE;
}




/***
  Functions to decode summary (or ev-summary) files.
***/


#define CFU_DECODE_T(type)				\
static inline gboolean					\
decode_##type (FILE *in, type *dest)			\
{							\
	type save = 0;					\
	gint i = sizeof (type) - 1;			\
	gint v = EOF;					\
							\
        while (i >= 0 && (v = fgetc (in)) != EOF) {	\
		save |= ((type)v) << (i * 8);		\
		i--;					\
	}						\
	*dest = save;					\
	if (v == EOF) {					\
		return FALSE;				\
	}						\
	return TRUE;					\
}


CFU_DECODE_T(time_t)
CFU_DECODE_T(off_t)
CFU_DECODE_T(size_t)


static inline gboolean
decode_guint32 (FILE *f, guint32 *n)
{
	guint32	value;
	gint    v;

	value = 0;

	/* until we get the last byte, keep decoding 7 bits at a time */
	while ( ((v = fgetc (f)) & 0x80) == 0 && v != EOF) {
		value |= v;
		value <<= 7;
	}

	if (v == EOF) {
		*n = value >> 7;
		return FALSE;
	}

	*n = value | (v & 0x7f);

	return TRUE;
}


static inline gboolean
decode_string (FILE *in, gchar **str)
{
	guint32 len;
	register gchar *ret;

	if (!decode_guint32 (in, &len)) {
		*str = NULL;
		return FALSE;
	}

	len--;
	if (len > 65536) {
		*str = NULL;
		return FALSE;
	}

	ret = g_malloc (len+1);
	if (len > 0 && fread (ret, len, 1, in) != 1) {
		g_free (ret);
		*str = NULL;
		return FALSE;
	}

	ret[len] = 0;
	*str = ret;
	return TRUE;
}


static inline gboolean
decode_gint32 (FILE *f, gint32 *dest)
{
	guint32 save;

	if (!f) return FALSE;

	if (fread (&save, sizeof (save), 1, f) == 1) {
		*dest = g_ntohl (save);
		return TRUE;
	} else {
		return FALSE;
	}
}


static inline gboolean
skip_gint32_decoding (FILE *f)
{
	return fseek (f, 4, SEEK_CUR) > 0;
}


static inline gboolean
skip_guint32_decoding (FILE *f)
{
	guint32 n;

	return decode_guint32 (f, &n);
}


static inline gboolean
skip_time_t_decoding (FILE *f)
{
	return fseek (f, sizeof (time_t), SEEK_CUR) == 0;
}


static inline gboolean
skip_off_t_decoding (FILE *f)
{
	return fseek (f, sizeof (off_t), SEEK_CUR) == 0;
}


static inline gboolean
skip_string_decoding (FILE *f)
{
	guint32 len;

	if (!decode_guint32 (f, &len)) {
		return FALSE;
	}

	if (fseek (f, len - 1, SEEK_CUR) != 0) {
		tracker_error ("ERROR: seek failed for string with length %d with error code %d", len - 1, errno);
		return FALSE;
	}

	return TRUE;
}


static inline gboolean
skip_token_decoding (FILE *f)
{
	guint32 len;

	if (!decode_guint32 (f, &len)) {
		return FALSE;
	}

	if (len < 32) {
		return TRUE;
	}

	len -= 32;
        return fseek (f, len, SEEK_CUR) == 0;
}
