// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "Linear.h"

namespace Torch {

Linear::Linear(int n_inputs_, int n_outputs_)
{
  addROption("weight decay", &weight_decay, 0, "weight decay", true);
  n_inputs = n_inputs_;
  n_outputs = n_outputs_;
}

int Linear::numberOfParams()
{
  return((n_inputs+1)*n_outputs);
}

void Linear::init()
{
  GradientMachine::init();
  reset();
}

void Linear::reset()
{
  real *params_ = (real *)params->ptr;
  real bound = 1./sqrt((real)n_inputs);

  for(int i = 0; i < n_params; i++)
    params_[i] = bounded_uniform(-bound, bound);
}

void Linear::forward(List *inputs)
{
  real *ptr_params = (real *)params->ptr;
  real *ptr_outputs = (real *)outputs->ptr;

  for(int s = 0; s < n_outputs; s++)
  {
    real out = 0;
    List *inputs_ = inputs;
    while(inputs_)
    {
      real *x = (real *)inputs_->ptr;
      for(int j = 0; j < inputs_->n; j++)
        out += *ptr_params++ * *x++;
      inputs_ = inputs_->next;
    }
    out += *ptr_params++;
    *ptr_outputs++ = out;
  }
}

void Linear::backward(List *inputs, real *alpha)
{
  real *beta_ptr = beta;
  for(int k = 0; k < n_inputs; k++)
    *beta_ptr++ = 0;

  real *alpha_ptr = alpha;
  real *params_ptr = (real *)params->ptr;
  for(int j = 0; j < n_outputs; j++, alpha_ptr++)
  {
    beta_ptr = beta;
    real z = *alpha_ptr;
    for(int k = 0; k < n_inputs; k++)
      *beta_ptr++ += z * *params_ptr++;
    params_ptr++; // le bias
  }

  alpha_ptr = alpha;
  real *der_params_ptr = (real *)der_params->ptr;
  for(int j = 0; j < n_outputs; j++, alpha_ptr++)
  {
    real z = *alpha_ptr;

    List *inputs_ = inputs;
    while(inputs_)
    {
      real *x = (real *)inputs_->ptr;
      for(int k = 0; k < inputs_->n; k++)
        *der_params_ptr++ = z * *x++;
      inputs_ = inputs_->next;
    }
    *der_params_ptr++ = z;
  }

  if(weight_decay != 0)
  {
    params_ptr = (real *)params->ptr;
    der_params_ptr = (real *)der_params->ptr;
    for(int i = 0; i < n_params; i++)
      *der_params_ptr++ += weight_decay * *params_ptr++;
  }
}

Linear::~Linear()
{
}

}

