// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef CMD_LINE_INC
#define CMD_LINE_INC

#include "Object.h"

namespace Torch {

// Internal
#define CMD_TEXT   0
#define CMD_OPTION 1
#define CMD_PARAMS 2

typedef struct CmdOption_
{
    char *name;
    char *help;
    void *ptr;
    int type;
    int status;
} CmdOption;
//

/** This class provides a useful interface for the user,
    to easily read some arguments/options from the command-line.
    
    Note that here, we make a difference between:
    \begin{itemize}
      \item {\bf options} which aren't not required.
      \item {\bf arguments} which are required.
    \end{itemize}

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class CmdLine : public Object
{
  public:
    int n_cmd_options;
    int n_cmd_params;
    CmdOption *cmd_options;
    char *text_info;

    // -----

    ///
    CmdLine();

    /** Read the command-line.
        Call this function {\bf after} adding options/arguments
        that you need, with the help of the following functions.
    */
    void read(int argc, char **argv);

    /** Print the help.
        Call this function {\bf after} adding options/arguments
        that you need, with the help of the following functions.
    */
    void help(char *name);

    //-----

    /** Functions for adding options.
        The calling order of the following functions will
        define the text order associated when you will call #help()#.
        
        Add an option (Int, Bool, Real, String).
        \begin{itemize}
          \item #name# the name of the option (must be unique).
          \item #ptr# is the pointer on the optional variable.
          \item #initvalue# is the initialization value.
          \item #help# is the help text for this option.
        \end{itemize}
        
        The option will be setted to #value# in the command-line
        by printing "#name# #value#"
    */
    ///
    void addICmdOption(const char *name, int *ptr, int initvalue, const char *help="");
    ///
    void addBCmdOption(const char *name, bool *ptr, bool initvalue, const char *help="");
    ///
    void addRCmdOption(const char *name, real *ptr, real initvalue, const char *help="");
    ///
    void addSCmdOption(const char *name, char **ptr, const char *initvalue, const char *help="");

    /** Functions for adding an argument.
        The argument will be setted to #value# in the command-line
        by writting "#value#" {\bf after} all the options.
        If there are N arguments, you have to write
        "#value1# #value2# #value3# ... #valueN#" to set them in the
        command-line.
    */
    ///
    void addICmdArg(const char *name, int *ptr, const char *help="");
    ///
    void addBCmdArg(const char *name, bool *ptr, const char *help="");
    ///
    void addRCmdArg(const char *name, real *ptr, const char *help="");
    ///
    void addSCmdArg(const char *name, char **ptr, const char *help="");    

    /// Add a text line in the help message.
    void addText(const char *text);

    /// Add a text at the beginnig of the help.
    void info(const char *text);

    //-----

    void addCmdOption(const char *name, void *ptr, int type, const char *help="", int status=CMD_OPTION);
    void setCmdOption(int argc, char **argv, int *current, CmdOption *ptro);
    void printCmdOption(CmdOption *ptro);
    virtual ~CmdLine();
};


}

#endif
