package Initrd;

#   $Header: /cvsroot/systemconfig/systemconfig/lib/Initrd.pm,v 1.9 2001/08/17 15:14:02 greekboy Exp $

#   Copyright (c) 2001 International Business Machines

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#   Vasilios Hoffman <greekboy@users.sourceforge.net>

use strict;
use Carp;
use File::Copy;
use Util::Log qw(:all);
use Util::Cmd qw(:all);

use Initrd::Ramdisk;

use vars qw($VERSION);

$VERSION = sprintf("%d.%02d", q$Revision: 1.9 $ =~ /(\d+)\.(\d+)/);

=head1 NAME

Initrd - builds initial ramdisks for kernels

=head1 SYNOPSIS

Initrd::setup($config);

=head1 DESCRIPTION

The Initrd module generates initial ramdisks for kernels, provided there is enough userland tools and kernel support to allow this.

=head1 AUTHOR

  Vasilios Hoffman <greekboy@users.sourceforge.net>

=head1 SEE ALSO

L<Config>, 
L<perl>

=cut

sub setup {

    my $config = shift;

    my %kernels = $config->varlist("^kernel");
    my $root = $config->get("root");

    my (@ramdisks,@files) = ();
    my $ramdisk;

    my %tools = Initrd::fetch_tools($config);

    if (not scalar keys %tools) {
        verbose("Missing the tools for building ramdisks, skipping.");
        return;
    }

    foreach (sort keys %kernels) {
        /(kernel\d+)_path/ and do {
            my $disk = $1 . "_initrd";
            my $kernel = $kernels{$1 . "_path"};
            if (not exists $kernels{$disk}) {
                debug("configuring ramdisk for $1");
                $ramdisk = new Initrd::Ramdisk(
                                                             kernel => $1,
                                                             kernel_path => $root . $kernel,
                                                             %tools,
                                                             $config->varlist("^(root|kernel|boot)")
                                                            );
                eval { $ramdisk->configure() };
                if (not "$@") {
                    push @ramdisks,$ramdisk;
                } else {
                    debug($@);
                    verbose("Not building ramdisk for $ramdisk->{kernel}");
                }
            }
        };
    }

    foreach $ramdisk (@ramdisks) {
        debug("creating ramdisk for $ramdisk->{kernel}");
        my $initrd;
        eval { $initrd = $ramdisk->create() };
        if ( not "$@" ) {
            verbose("Sucessfully created ramdisk for $ramdisk->{kernel} at $initrd");
            $config->set("$ramdisk->{kernel}_initrd",$initrd);
        } else {
            debug($@);
            verbose("Failed to create ramdisk for $ramdisk->{kernel}");
        }
        push @files, $ramdisk->files();
    }

    return @files;

}

sub fetch_tools {

    my $config = shift;

    my $root = $config->get('root');
    my %tools;

    #we need to have:
    #  loopback devices & kernel capabilities
    #  dd, insmod, losetup, mount & mknod commands
    #  a shell & libraries
    #  ability to write to temp
    #  /dev/zero

    #what we would like to know about/have
    #  gzip


    #first we do the easy showstoppers

    #are the commands we need present?
    my $mount = which("mount");
    my $umount = which("umount");
    my $mknod = which("mknod");
    my $losetup = which("losetup");
    my $insmod = which("insmod");
    my $insmod_static = which ("insmod.static");
    my $dd = which("dd");

    if ( not defined $insmod and defined $insmod_static ) {
	$insmod = $insmod_static;
    }

    if (not defined $mount) {
        verbose("Failed to find mount command.  Cannot build initial ramdisks\n");
        return;
    }

    if (not defined $umount) {
        verbose("Failed to find umount command.  Cannot build initial ramdisks\n");
        return;
    }

    if (not defined $mknod) {
        verbose("Failed to find mknod command.  Cannot build initial ramdisks\n");
        return;
    }

    if (not defined $losetup) {
        verbose("Failed to find losetup command.  Cannot build initial ramdisks\n");
        return;
    }

    if (not defined $insmod) {
        verbose("Failed to find losetup command.  Cannot build initial ramdisks\n");
        return;
    }

    if (not defined $dd) {
        verbose("Failed to find dd command.  Cannot build initial ramdisks\n");
        return;
    }

    $tools{"mount"} = $mount;
    $tools{"mknod"} = $mknod;
    $tools{"losetup"} = $losetup;
    $tools{"insmod"} = $insmod;
    $tools{"dd"} = $dd;

    #gzip is optional ... ramdisks don't HAVE to be compressed.
    $tools{"gzip"} = which("gzip");

    #okay, can we do loopback mounting?  First let's make sure loop.o is loaded,
    #incase we have a system that needs it and won't autoload.
    system("$tools{insmod} loop > /dev/null 2>&1");

    #now we dd into temp -- if this fails, /dev/zero is missing or we
    #can't write to /tmp
    my $time = join('',localtime());
    system("$tools{dd} if=/dev/zero of=$root/tmp/sctest.$time bs=1k count=1 >/dev/null 2>&1");

    if (not -e "$root/tmp/sctest.$time") {
      verbose("dd command failed to write to /tmp.  Cannot build initial ramdisks\n");
      return;
    }

    foreach (0 .. 7) {
        my $return_value;
	$return_value = system("$tools{losetup} /dev/loop$_ $root/tmp/sctest.$time > /dev/null 2>&1");
        $return_value /= 256;

        if ($return_value == 0) {
	  system("$tools{losetup} -d /dev/loop$_ > /dev/null 2>&1");
	  $tools{"loop"} = "/dev/loop$_";
	  debug("loop device is $tools{loop}");
	  last;
        }
    }

    system("rm $root/tmp/sctest.$time > /dev/null 2>&1");

    if (not exists $tools{"loop"}) {
        verbose("Unable to find valid loopback device.  Cannot build Initial ramdisks.");
        return;
    }

    #finally:  figure out what shell we are going to use, and what libraries we need.
    #if we have ldd, this is easy.  Otherwise we fall back on defaults.
    my $ldd = which("ldd");

    if (not defined $ldd) {
        return %tools;
    } else {
        $tools{ldd} = $ldd;
    }

    my @shells = qw(nash ash.static bash.static ash sash bash);

    foreach (@shells) {
        my $path;
        $path = which($_);

        if (not $path) {
            #debug("$_ not found");
            next;
        } else {
            debug("shell is $path");
            $tools{"shell"} = $path;
	    #if our shell is static, so should our insmod be
	    if ( ( /static$/ or /^nash$/ or /^sash$/ )
		   and defined $insmod_static ) {
	        $tools{insmod} = $insmod_static;
	    }
            last;
        }

    }

    $tools{"libs"} = "";

    my @binaries = ($tools{shell}, $tools{insmod});
    my $binary;

    foreach $binary (@binaries) {
        open(OUT,"$tools{ldd} $binary |") or croak("Couldn't run ldd $tools{shell}");
        while (<OUT>) {
            /not a dynamic executable/i and last;
            /=> (.*?) \(/ and do {
                if ( $tools{"libs"} =~ /$1/ ) {
                    next;
                } else {
                    $tools{"libs"} .= "$1 "
                }
            };
        }
        close(OUT);

    }

    debug("shared libraries needed are $tools{libs}");

    return %tools;

}

1;





