/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config-synaptiks.h"
#include "synaptikskcm.h"
#include "synaptiksconfiguration.h"
#include "synaptikswidgets.h"
#include <KIcon>
#include <KTabWidget>
#include <KPageWidget>
#include <KPluginFactory>
#include <KAboutData>
#include <KLocalizedString>
#include <QtDBus/QDBusInterface>
#include <QtDBus/QDBusReply>
#include <QtGui/QHBoxLayout>

using namespace synaptiks;

K_PLUGIN_FACTORY(SynaptiksKCMFactory, registerPlugin<SynaptiksKCM>();)
K_EXPORT_PLUGIN(SynaptiksKCMFactory("kcmsynaptiks"))


SynaptiksKCM::SynaptiksKCM(QWidget *parent, const QList<QVariant> &args):
    KCModule(SynaptiksKCMFactory::componentData(), parent, args) {
    KGlobal::locale()->insertCatalog("synaptiks");

    KAboutData *about = new KAboutData(
        "synaptiks", 0, ki18n("synaptiks"), SYNAPTIKS_VERSION,
        ki18n("synaptiks – touchpad management"), KAboutData::License_BSD,
        ki18n("Copyright © 2009, 2010 Sebastian Wiesner"), KLocalizedString(),
        "http://synaptiks.lunaryorn.de/");
    about->addAuthor(ki18n("Sebastian Wiesner"), ki18n("Programmer"),
                     "basti.wiesner@gmx.net");
    about->addCredit(ki18n("Valentyn Pavliuchenko"),
                     ki18n("Debian packaging, russian translation, "
                           "bug reporting and testing"),
                     "valentyn.pavliuchenko@gmail.com");
    about->setTranslator(ki18nc("NAME OF TRANSLATORS", "Your names"),
                         ki18nc("EMAIL OF TRANSLATORS", "Your emails"));
    this->setAboutData(about);
    this->setQuickHelp(
        i18nc("@info:tooltip synaptiks kcmodule",
              "<title><application>synaptiks</application> "
              "configuration</title>"
              "<para>This module lets you configure the "
              "<application>synaptiks</application> daemon, which in turn "
              "manages and configures your touchpad.</para>"));

    // set main layout for this widget
    this->setLayout(new QHBoxLayout(this));
    QDBusInterface daemon("org.kde.kded", "/modules/synaptiks",
                          "org.kde.Synaptiks");
    QDBusReply<bool> touchpadAvailable = daemon.call("isTouchpadAvailable");
    if (!touchpadAvailable.isValid()) {
        // daemon is apparently not running
        this->showError(
            i18nc("@info connectivity error",
                  "<warning>The <application>synaptiks</application> "
                  "daemon is not running.</warning>"));
    } else if (!touchpadAvailable.value()) {
        // daemon is running, but reported, that no touchpad was found, so
        // lets get the error message from the daemon
        QDBusReply<QString> message = daemon.call(
            "touchpadNotAvailableMessage");
        this->showError(message.value());
    } else {
        // daemon is running and has a touchpad, so load configuration and
        // setup user interface
        this->config = new SynaptiksConfiguration();
        // read the defaults provided by the daemon ...
        this->config->setSharedConfig(
            KSharedConfig::openConfig("synaptiksrc-defaults"));
        this->config->readConfig();
        // ... update all defaults in the configuration scheme ...
        foreach (KConfigSkeletonItem *item, this->config->items()) {
            item->swapDefault();
        }
        // ... and finally read the actual configuration
        this->config->setSharedConfig(
            KSharedConfig::openConfig("synaptiksrc"));
        this->config->readConfig();
        this->setupUi();
    }
}

void SynaptiksKCM::setupUi() {
    KPageWidget *configWidget = new KPageWidget(this);
    this->layout()->addWidget(configWidget);

    // setup the touchpad configuration page
    KTabWidget *touchpadConfig = new KTabWidget(configWidget);
    QList<QWidget *> pages;
    pages << new GeneralPage(configWidget)
          << new MotionPage(configWidget)
          << new ScrollingPage(configWidget)
          << new TappingPage(configWidget);
    foreach (QWidget *page, pages) {
        touchpadConfig->addTab(page, page->windowTitle());
    }
    touchpadConfig->setWindowTitle(
        i18nc("@title:window", "Touchpad configuration"));
    pages.clear();

    // and the complete configuration widget
    QWidget *notifyConfig = new NotificationsPage("synaptiks",
                                                  configWidget);
    this->connect(notifyConfig, SIGNAL(changed(bool)),
                  SIGNAL(changed(bool)));
    pages << touchpadConfig << new AutoTouchpadControlPage(configWidget)
          << notifyConfig << new TouchpadInformationPage(configWidget);
    QList<KPageWidgetItem *> items;
    foreach(QWidget *page, pages) {
        items << configWidget->addPage(page, page->windowTitle());
    }
    items.at(0)->setIcon(KIcon("configure"));
    items.at(1)->setIcon(KIcon("configure"));
    items.at(2)->setIcon(KIcon("preferences-desktop-notification"));
    items.at(3)->setIcon(KIcon("hwinfo"));
    this->addConfig(this->config, configWidget);
}

void SynaptiksKCM::showError(const QString &message) {
    QWidget *widget = this->findChild<QWidget*>();
    if (widget)
        delete widget;
    widget = new ErrorPage(message, this);
    this->layout()->addWidget(widget);
}

void SynaptiksKCM::save() {
    if (!this->findChild<ErrorPage*>()) {
        kDebug() << "saving configuration";
        NotificationsPage *notifyConfig =
            this->findChild<NotificationsPage *>();
        Q_ASSERT(notifyConfig);
        notifyConfig->saveNotifications();
        KCModule::save();
        QDBusInterface synaptiks("org.kde.synaptiks",
                                 "/modules/synaptiks",
                                 "org.kde.Synaptiks");
        synaptiks.call("reparseConfiguration");
    } else {
        kWarning() << "error page widget found, not saving config";
    }
}

#include "moc_synaptikskcm.cpp"
