/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "synaptiksapplet.h"
#include "ui_userinterfacepage.h"
#include <Plasma/ToolTipManager>
#include <Plasma/ConfigLoader>
#include <KCMultiDialog>
#include <KConfigDialog>
#include <KAction>
#include <KStandardAction>
#include <KIcon>
#include <KStandardDirs>
#include <QtGui/QPainter>
#include <QtGui/QWidget>
#include <QtDBus/QDBusReply>
#include <QtCore/QFile>


K_EXPORT_PLASMA_APPLET(synaptiks, synaptiks::SynaptiksApplet);


using namespace synaptiks;


enum DoubleClickAction {
    NoAction,
    ConfigDialogAction,
    SwitchTouchpadAction
};


SynaptiksApplet::SynaptiksApplet(QObject *parent, const QVariantList &args):
    Plasma::Applet(parent, args), touchpadManager(0) {
    this->m_svg.setImagePath(
        KStandardDirs::locate("data", "synaptiks/pics/touchpad.svg"));
    this->setAspectRatioMode(Plasma::ConstrainedSquare);
}

SynaptiksApplet::~SynaptiksApplet() {
}

void SynaptiksApplet::paintInterface(
    QPainter *painter, const QStyleOptionGraphicsItem *option,
    const QRect &contentsRect) {
    Q_UNUSED(option);
    if (this->touchpadManager.isNull()) {
        return;
    }
    painter->setRenderHint(QPainter::SmoothPixmapTransform);
    painter->setRenderHint(QPainter::Antialiasing);
    // Now we draw the applet, starting with our svg
    this->m_svg.resize(contentsRect.width(), contentsRect.height());
    painter->save();
    painter->translate(contentsRect.topLeft());
    this->m_svg.paint(painter, m_svg.elementRect("touchpad") , "touchpad");

    QDBusReply<bool> isOn = this->touchpadManager->call("isTouchpadOn");
    if (!isOn.isValid()) {
        this->m_svg.paint(painter, m_svg.elementRect("unavailable"),
                          "unavailable");
        this->showTouchpadError(isOn.error());
    } else if (!isOn.value()) {
        m_svg.paint(painter, m_svg.elementRect("off"), "off");
    }
    painter->restore();
}

void SynaptiksApplet::createConfigurationInterface(KConfigDialog *parent) {
    QWidget *page = new QWidget(parent);
    Ui::UserInterfacePage ui;
    ui.setupUi(page);
    parent->addPage(page, this->m_config, page->windowTitle(),
                    "preferences-other");
}

void SynaptiksApplet::init() {
    QString scheme = KStandardDirs::locate(
        "kcfg", "plasma-applet-synaptiks.kcfg");
    Q_ASSERT(!scheme.isNull());
    QFile schemeFile(scheme);
    KConfigGroup config = this->config();
    this->m_config = new Plasma::ConfigLoader(&config, &schemeFile);
    this->connect(this->m_config, SIGNAL(configChanged()),
                  SLOT(configChanged()));

    KAction *configureTouchpad = KStandardAction::preferences(
        this, SLOT(showTouchpadConfiguration()), this);
    configureTouchpad->setObjectName("configureTouchpad");
    this->addAction(configureTouchpad->objectName(), configureTouchpad);

    QDBusInterface daemon("org.kde.kded", "/modules/synaptiks",
                          "org.kde.Synaptiks");
    QDBusReply<bool> touchpadAvailable = daemon.call("isTouchpadAvailable");
    if (!touchpadAvailable.isValid()) {
        // daemon is apparently not running
        this->setFailedToLaunch(
            true, i18nc(
                "@info connectivity error",
                "<warning>The <application>synaptiks</application> "
                "daemon is not running.</warning>"));
    } else if (!touchpadAvailable.value()) {
        // daemon is running, but reported, that no touchpad is available,
        // so get the error message from the daemon
        QDBusReply<QString> message = daemon.call(
            "touchpadNotAvailableMessage");
        this->setFailedToLaunch(true, message.value());
    } else {
        // daemon is running, and has a touchpad, so get us connected to it
        this->touchpadManager = new QDBusInterface(
            "org.kde.synaptiks", "/TouchpadManager",
            "org.kde.TouchpadManager");
        this->connect(this->touchpadManager,
                      SIGNAL(touchpadSwitched(bool, const QString&,
                                              const QDBusVariant&)),
                      SLOT(touchpadSwitched(bool, const QString&,
                                            const QDBusVariant&)));
    }
}

void SynaptiksApplet::mouseDoubleClickEvent(
    QGraphicsSceneMouseEvent *event) {
    Q_UNUSED(event);
    KConfigSkeletonItem *item;
    item = this->m_config->findItemByName("DoubleClickAction");
    Q_ASSERT(item);
    switch (item->property().toInt()) {
    case ConfigDialogAction:
        this->showTouchpadConfiguration();
        break;
    case SwitchTouchpadAction:
        this->switchTouchpad();
        break;
    }
}

void SynaptiksApplet::touchpadSwitched(bool on, const QString &reason,
                                       const QDBusVariant &closure) {
    Plasma::ToolTipContent tooltip;
    tooltip.setMainText(
        on? i18nc("@info:tooltip applet main text", "Touchpad on"):
        i18nc("@info:tooltip applet applet main text", "Touchpad off"));
    if (reason == "keyboard") {
        tooltip.setSubText(
            on? i18nc("@info:tooltip applet sub text", "Keyboard inactive"):
            i18nc("@info:tooltip applet sub text", "Keyboard active"));
    } else if (reason == "mouse") {
        QString mouse = closure.variant().toString();
        tooltip.setSubText(
            on? i18nc("@info:tooltip applet sub text",
                      "<resource>%1</resource> unplugged", mouse):
            i18nc("@info:tooltip applet sub text",
                  "<resource>%1</resource> plugged", mouse));
    }
    Plasma::ToolTipManager::self()->setContent(this, tooltip);
    this->update();
}

void SynaptiksApplet::switchTouchpad() {
    QDBusReply<bool> isOn = this->touchpadManager->call("isTouchpadOn");
    if (!isOn.isValid()) {
        this->showTouchpadError(isOn.error());
    } else {
        QDBusReply<void> reply = this->touchpadManager->call(
            "setTouchpadOn", !isOn.value(), "interactive",
            QVariant::fromValue(QDBusVariant("")));
        if (!reply.isValid()) {
            this->showTouchpadError(reply.error());
        }
    }
}

void SynaptiksApplet::showTouchpadConfiguration() const {
    KCMultiDialog *dlg = new KCMultiDialog();
    dlg->setWindowTitle(i18nc("@title:window applet touchpad configuration",
                              "Touchpad configuration"));
    dlg->addModule("synaptiks");
    dlg->show();
    dlg->connect(dlg, SIGNAL(finished(int)), SLOT(deleteLater()));
}

void SynaptiksApplet::showTouchpadError(const QDBusError &error) {
    kWarning() << error;
    Plasma::ToolTipContent tooltip;
    tooltip.setMainText(
        i18nc("@info:tooltip applet main text", "Touchpad error"));
    tooltip.setSubText(error.message());
    tooltip.setImage(KIcon("dialog-warning").pixmap(
                         IconSize(KIconLoader::Desktop)));
    Plasma::ToolTipManager::self()->setContent(this, tooltip);
}

#include "moc_synaptiksapplet.cpp"
