//
// C++ Implementation: accesspointlistview
//
// Description: 
//
//
// Author: Ivn Forcada Atienza <ivan@swscanner.org>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "accesspointlistview.h"
#include "mainwindow.h"

AccessPointListView::AccessPointListView(QWidget * parent, const char * name, WFlags f)
	: QListView(parent, name, f)
{

	QFont fontBold;
	fontBold.setBold(TRUE);

	addColumn( "  (" + i18n( "OFF" ) + ")" );
	addColumn( i18n( "ESSID" ) );
	addColumn( i18n( "MAC" ) );
	addColumn( i18n( "VENDOR" ) );
	addColumn( i18n( "WEP" ) );
	addColumn( i18n( "CHANNEL" ) );
	addColumn( i18n( "MODE" ) );
	addColumn( i18n( "SIGNAL" ) );
	addColumn( i18n( "SIGNAL(MAX)" ) );
	addColumn( i18n( "NOISE" ) );
	addColumn( i18n( "SNR" ) );
	addColumn( i18n( "SNR(MAX)" ) );
	header()->setFont(fontBold);
	header()->setLabel( 0, SmallIcon("stop"), "  (" + i18n( "OFF" ) + ")" );
	header()->setLabel( 1, i18n( "ESSID" ) );
	header()->setLabel( 2, i18n( "MAC" ) );
	header()->setLabel( 3, i18n( "VENDOR" ) );
	header()->setLabel( 4, i18n( "WEP" ) );
	header()->setLabel( 5, i18n( "CHANNEL" ) );
	header()->setLabel( 6, i18n( "MODE" ) );
	header()->setLabel( 7, i18n( "SIGNAL" ) );
	header()->setLabel( 8, i18n( "SIGNAL(MAX)" ) );
	header()->setLabel( 9, i18n( "NOISE" ) );
	header()->setLabel( 10, i18n( "SNR" ) );
	header()->setLabel( 11, i18n( "SNR(MAX)" ) );
	setMinimumSize( QSize( 400, 140 ) );
	//setLineWidth( 1 );
	setSelectionMode( QListView::Single );
	setShowSortIndicator( TRUE );
	setItemMargin( 1 );
	setRootIsDecorated( FALSE );
	setSorting(1);
	setPreserve( TRUE );
	setSoundEnabled( FALSE );
	adjustColumns();
	lstFilter = 0;
}

QListView * AccessPointListView::Filter()
{
	if (lstFilter)
		return lstFilter;
	else
	{
		lstFilter = (TreeFilterListView*) parentWidget()->child("lstFilter"); 
		return lstFilter;
	}
}

void AccessPointListView::adjustColumns()
{
	// Adjust the columns width
	for (int i=0;i < columns(); i++)
		adjustColumn(i);

}

// Returns the column index of a field name(header)
int AccessPointListView::getCol(QString title)
{
        QHeader* head = header();

        for (int i=0 ; i < head->count() ; i++)
                if (title == head->label(i))
                        return i;
        return -1;
}

// Return the number of visible childs of the table
uint AccessPointListView::visiblesCount()
{
	QListViewItem * it = firstChild();
	uint j = 0;

        while ( it )
	{
                if ( it->isVisible() )
                        j++;
		it=it->nextSibling();
	}
	return j;
}

//Check if a MAC is already displayed in the current table
bool AccessPointListView::IsInTable (QString MAC)
{
	if ( findItem(MAC, getCol(i18n("MAC"))) == 0)
		return false;
	else
		return true;
}

/* This function updates a previously existing row, taking into account
if doesn't pass the filter, when it hides the row */
void AccessPointListView::updateRow(APEntry* ap)
{
        char temp[20];
        int snrnew, snrold, sig;
        QString mac = ap->getMAC(temp);
        QPixmap pix  = getPix(ap->getSignal());
	QListViewItem * it = findItem(mac,getCol(i18n("MAC")));
	TreeFilterListView * filter = (TreeFilterListView *)Filter();
	QString soundPath = KStandardDirs::kfsstnd_defaultprefix() + "/" + KStandardDirs::kde_default("data") + "swscanner/sounds/";

	it->setPixmap(0,pix);

	sig = it->text(getCol(i18n("SIGNAL(MAX)"))).toInt();
	snrold = it->text(getCol(i18n("SNR(MAX)"))).toInt();
	snrnew = ap->getSignal() - ap->getNoise();		
	if (ap->getSignal() > sig)
		it->setText(getCol(i18n("SIGNAL(MAX)")),QVariant(ap->getSignal()).toString());
	if (snrnew >  snrold)
		it->setText(getCol(i18n("SNR(MAX)")),QVariant(snrnew).toString());
	it->setText(getCol(i18n("SIGNAL")),QVariant(ap->getSignal()).toString());
	it->setText(getCol(i18n("NOISE")),QVariant(ap->getNoise()).toString());
	it->setText(getCol(i18n("SNR")),QVariant(snrnew).toString());

	if ( preserve && filter->passTheFilter(it) )
		it->setVisible(TRUE);
	else if ( !preserve && filter->passTheFilter(it) && ( it->text(getCol(i18n("SIGNAL")))!=i18n("N.A.")) )
	{
		if (! it->isVisible() && soundEnabled)
			KAudioPlayer::play(soundPath + "swscanner_appear.ogg");
		it->setVisible(TRUE);
	}
	else
		it->setVisible(FALSE);
}

void AccessPointListView::addNewRow(QListViewItem* row)
{
	bool filt;
	QString wep, mac, channel, essid;
        QListViewItem * it;
	TreeFilterListView * filter = (TreeFilterListView *)Filter();
	KIconLoader *loader = KGlobal::iconLoader();
	QPixmap icon;
	QString soundPath = KStandardDirs::kfsstnd_defaultprefix() + "/" + KStandardDirs::kde_default("data") + "swscanner/sounds/";
	
	if ( row->text(getCol(i18n("SIGNAL"))) == "" || row->text(getCol(i18n("SIGNAL"))) == i18n("N.A.") )
		 icon = loader->loadIcon("signal_none", KIcon::User);
	else
		 icon = getPix(row->text(getCol(i18n("SIGNAL"))).toInt());

	row->setPixmap( 0, icon );

	if (soundEnabled)
		KAudioPlayer::play(soundPath + "swscanner_appear.ogg");

	filt = filter->passTheFilter(row);
        if ( preserve && filt )
                row->setVisible(TRUE);
        else if ( !preserve && filt && (row->text(getCol(i18n("SIGNAL")))!=i18n("N.A.")) )
                row->setVisible(TRUE);
        else
                row->setVisible(FALSE);

        //add the mac'x to the filter tree
        it=filter->findItem(i18n("MAC"), 0);
	mac=row->text(getCol(i18n("MAC")));	
        if (!IsInTree(it, mac))
                new QCheckListItem(it, mac);
        //add the channel to the filter, only if it isn't there yet
        it=filter->findItem(i18n("CHANNEL"), 0);
	channel=row->text(getCol(i18n("CHANNEL")));
        if (!IsInTree(it, channel))
                new QCheckListItem(it, channel);
        //the same with essid:
        it=filter->findItem(i18n("ESSID"), 0);
	essid=row->text(getCol(i18n("ESSID")));
        if (!IsInTree(it, essid))
                new QCheckListItem(it, essid);
	//and again with wep:
        it=filter->findItem(i18n("WEP"), 0);
	wep=row->text(getCol(i18n("WEP")));
        if (!IsInTree(it, wep ))
                new QCheckListItem(it, wep);
}

void AccessPointListView::addNewRow(APEntry* ap)
{
	char temp[255];
	char mac[20];
	QListViewItem * row;

	row = new QListViewItem( this );
	
        row->setText(getCol(i18n("ESSID")),ap->getEssid(temp));
        row->setText(getCol(i18n("MAC")),ap->getMAC(mac));

	if ( ! ap->getVendor(temp) )	// Calculate vendor if it is not already set
		row->setText(getCol(i18n("VENDOR")),getVendor(QString(mac)));
	else
		row->setText(getCol(i18n("VENDOR")),ap->getVendor(temp));

        row->setText(getCol(i18n("CHANNEL")),QVariant(ap->getChannel()).toString());
	row->setText(getCol(i18n("WEP")), ap->getWEP()?i18n("YES"):i18n("NO"));

	if ( ap->getSignalMax() == -999 ) // Not set
        	row->setText(getCol(i18n("SIGNAL(MAX)")),QVariant(ap->getSignal()).toString());
	else
		row->setText(getCol(i18n("SIGNAL(MAX)")),QVariant(ap->getSignalMax()).toString());

        row->setText(getCol(i18n("MODE")),ap->getModeString());
        row->setText(getCol(i18n("SIGNAL")),QVariant(ap->getSignal()).toString());
        row->setText(getCol(i18n("NOISE")),QVariant(ap->getNoise()).toString());
        row->setText(getCol(i18n("SNR")),QVariant(ap->getSignal()-ap->getNoise()).toString());
	if (ap->getSNRMax() == -999 ) // Not set
        	row->setText(getCol(i18n("SNR(MAX)")),QVariant(ap->getSignal()-ap->getNoise()).toString());
	else
		row->setText(getCol(i18n("SNR(MAX)")),QVariant(ap->getSNRMax()).toString());

	addNewRow(row);
}

//Search in the parentNode's childs, but not in the children's children
bool AccessPointListView::IsInTree (QListViewItem * parentNode, QString strToFind)
{
        QListViewItem * myChild = parentNode->firstChild();
        while( myChild ) {
                if (myChild->key(0,true) == strToFind)
                        return true;
                myChild = myChild->nextSibling();
        }
        return false;
}

void AccessPointListView::addNewAP(APEntry* ap)
{
        char mac[20];
	
        if (IsInTable(ap->getMAC(mac)))	// If the row already exists then update it with the new signal-noise
                updateRow(ap);
        else
                addNewRow(ap);
}

void AccessPointListView::repaintTable()
{
	APList * nodes = SWSglobals::nodes();

	for (uint i=0 ; i < nodes->count() ; i++)
                addNewAP( nodes->at(i) );	// adds or updates the list of ap's
}

QString AccessPointListView::getVendor(QString MAC)
{
	QString ouiFile = "/etc/swscanner/oui_sws.txt";
	QString line;
	QFile file(ouiFile);
	QString fileMAC;
	QString fileVendor;

	if (! file.exists() )
		cerr << "** Error: Cannot get vendor information: missing '" << ouiFile << "' file." << endl;

	if ( file.open( IO_ReadOnly ) )
	{
                QTextStream stream( &file );
                QString line;
                while ( !stream.atEnd() )
		{
			line=stream.readLine();
			fileMAC = line.left(8);
			fileMAC = fileMAC.replace('-', ':');
			if ( fileMAC == MAC.left(8) )
			{
				fileVendor = line.right(line.length()-9);
				break;
			}
		}
	}
	if (fileVendor.length() == 0)
		fileVendor = i18n("Unknown");
	return fileVendor.upper();
}

QString AccessPointListView::getSelectedESSID()
{
	QListViewItem * selected = selectedItem();
	QString essid = QString::null;
	
	if (selected)
		essid = selected->text(getCol(i18n("ESSID")));

	return essid;
}

QString AccessPointListView::getSelectedMAC()
{
	QListViewItem * selected = selectedItem();
	QString mac = QString::null;
	
	if (selected)
		mac = selected->text(getCol(i18n("MAC")));

	return mac;	
}

bool AccessPointListView::getSelectedWEP()
{
	QListViewItem * selected = selectedItem();
	bool wep = FALSE;
	
	if (selected)
		wep = (selected->text(getCol(i18n("WEP"))) == i18n("YES")) ? TRUE : FALSE;

	return wep;
}

int AccessPointListView::aplistFromTable(APList * list)
{
	QListViewItem * it = firstChild();
	APEntry * ap;
	while ( it ) 
	{
		if (!it->isVisible())
		{
			it = it->nextSibling();
			continue;
		}
		ap = new APEntry;
		ap->setEssid( (char*)it->text(getCol(i18n("ESSID"))).simplifyWhiteSpace().ascii() );
		ap->setMAC( (char*)it->text(getCol(i18n("MAC"))).simplifyWhiteSpace().ascii() );
		ap->setVendor( (char*)it->text(getCol(i18n("VENDOR"))).simplifyWhiteSpace().ascii() );
		ap->setWEP( it->text(getCol(i18n("WEP"))).simplifyWhiteSpace() == i18n("YES") ? TRUE : FALSE );
		ap->setChannel( it->text(getCol(i18n("CHANNEL"))).simplifyWhiteSpace().toUInt() );
		ap->setMode( it->text(getCol(i18n("MODE"))).simplifyWhiteSpace() == APEntry::getModeString(0) ? 0 : 1 );
		ap->setSignal( it->text(getCol(i18n("SIGNAL"))).simplifyWhiteSpace().toInt() );
		ap->setSignalMax( it->text(getCol(i18n("SIGNAL(MAX)"))).simplifyWhiteSpace().toInt() );
		ap->setNoise( it->text(getCol(i18n("NOISE"))).simplifyWhiteSpace().toInt() );
		ap->setSNRMax( it->text(getCol(i18n("SNR(MAX)"))).simplifyWhiteSpace().toInt() );
		list->append(ap);
		it = it->nextSibling();
	}
	return 0;
}

QPixmap getPix(int signal)
{
	QFile file;
	KIconLoader *loader = KGlobal::iconLoader();
	QPixmap icon;

	if ( (signal < -90) || ((signal > 0) && (signal < 5)) )
		icon = loader->loadIcon("signal_none", KIcon::User);
	else if ( ((signal >= -90) && (signal < -80)) || ((signal >= 5) && (signal < 25)) )
		icon = loader->loadIcon("./signal_low", KIcon::User);
	else if ( ((signal >= -80) && (signal < -70)) || ((signal >= 25) && (signal < 45)) )
		icon = loader->loadIcon("./signal_medium", KIcon::User);
	else if ( (signal >= -70) || (signal >= 40) )
		icon = loader->loadIcon("./signal_good", KIcon::User);
	
	return icon;
}
