###
# Copyright (c) 2002-2004, Jeremiah Fincher
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   * Redistributions of source code must retain the above copyright notice,
#     this list of conditions, and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright notice,
#     this list of conditions, and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#   * Neither the name of the author of this software nor the name of
#     contributors to this software may be used to endorse or promote products
#     derived from this software without specific prior written consent.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
###

"""
Various utility commands, mostly useful for manipulating nested commands.
"""

import supybot

__revision__ = "$Id: Utilities.py,v 1.64 2004/12/18 21:22:20 jemfinch Exp $"
__author__ = supybot.authors.jemfinch

import supybot.plugins as plugins

import types
import random
import string

import supybot.utils as utils
from supybot.commands import *
import supybot.ircmsgs as ircmsgs
import supybot.ircutils as ircutils
import supybot.callbacks as callbacks

class Utilities(callbacks.Privmsg):
    # Yes, I really do mean "requires no arguments" below.  "takes no
    # arguments" would probably lead people to think it was a useless command.
    def ignore(self, irc, msg, args):
        """requires no arguments

        Does nothing.  Useful sometimes for sequencing commands when you don't
        care about their non-error return values.
        """
        pass

    def reply(self, irc, msg, args, text):
        """<text>

        Replies with <text>.  Equivalent to the alias, 'echo $nick: $1'.
        """
        irc.reply(text, prefixName=True)
    reply = wrap(reply, ['text'])

    def success(self, irc, msg, args, text):
        """[<text>]

        Does nothing except to reply with a success message.  This is useful
        when you want to run multiple commands as nested commands, and don't
        care about their output as long as they're successful.  An error, of
        course, will break out of this command.  <text>, if given, will be
        appended to the end of the success message.
        """
        irc.replySuccess(text)
    success = wrap(success, [additional('text')])

    def last(self, irc, msg, args):
        """<text> [<text> ...]

        Returns the last argument given.  Useful when you'd like multiple
        nested commands to run, but only the output of the last one to be
        returned.
        """
        args = filter(None, args)
        if args:
            irc.reply(args[-1])
        else:
            raise callbacks.ArgumentError

    def strlen(self, irc, msg, args):
        """<text>

        Returns the length of <text>.
        """
        total = 0
        for arg in args:
            total += len(arg)
        total += len(args)-1 # spaces between the arguments.
        irc.reply(str(total))

    def echo(self, irc, msg, args, text):
        """<text>

        Returns the arguments given it.  Uses our standard substitute on the
        string(s) given to it; $nick (or $who), $randomNick, $randomInt,
        $botnick, $channel, $user, $host, $today, $now, and $randomDate are all
        handled appropriately.
        """
        text = ircutils.standardSubstitute(irc, msg, text)
        irc.reply(text, prefixName=False)
    echo = wrap(echo, ['text'])

    def shuffle(self, irc, msg, args, things):
        """<arg> [<arg> ...]

        Shuffles the arguments given it.
        """
        random.shuffle(things)
        irc.reply(' '.join(things))
    shuffle = wrap(shuffle, [many('anything')])

    def re(self, irc, msg, args, ff, text):
        """<regexp> <text>

        If <regexp> is of the form m/regexp/flags, returns the portion of
        <text> that matches the regexp.  If <regexp> is of the form
        s/regexp/replacement/flags, returns the result of applying such a
        regexp to <text>
        """
        if isinstance(ff, (types.FunctionType, types.MethodType)):
            f = ff
        else:
            f = lambda s: ff.search(s) and ff.search(s).group(0) or ''
        if f('') and len(f(' ')) > len(f(''))+1: # Matches the empty string.
            s = 'You probably don\'t want to match the empty string.'
            irc.error(s)
        else:
            irc.reply(f(text))
    re = wrap(re, [('checkCapability', 'trusted'),
                   first('regexpMatcher', 'regexpReplacer'),
                   'text'])

    def apply(self, irc, msg, args, command, rest):
        """<command> <text>

        Tokenizes <text> and calls <command> with the resulting arguments.
        """
        args = [token and token or '""' for token in rest]
        text = ' '.join(args)
        commands = command.split()
        commands = map(callbacks.canonicalName, commands)
        tokens = callbacks.tokenize(text)
        allTokens = commands + tokens
        self.Proxy(irc, msg, allTokens)
    apply = wrap(apply, ['something', many('anything')])


Class = Utilities

# vim:set shiftwidth=4 tabstop=8 expandtab textwidth=78:
