import urlparse
import pygtk
pygtk.require('2.0')
import gobject
import gtk
import gtk.glade
import gettext
from Feed import Feed
import FeedList
import feedfinder
import error
import straw
import Event
import utils
import SummaryParser
import MessageManager
import hig_alert
from xml.sax import saxutils
import time

_subscribe_window = None

def show(url=None):
    global _subscribe_window
    if _subscribe_window is None:
        glade_file = utils.find_glade_file()
        _subscribe_window = SubscribeDialog(gtk.glade.XML(glade_file, 'subscribe_window',
                                                      gettext.textdomain()))
    _subscribe_window.show(url)

class SubscribeDialog(Event.SignalEmitter):
    LOCATION_PAGE = 0
    PROGRESS_PAGE = 1
    AUTH_PAGE = 2
    MULTIPLE_FEED_PAGE = 3

    COLUMN_SUBSCRIBE = 0
    COLUMN_TITLE = 1
    COLUMN_FEED = 2
    COLUMN_PARSEDSUMMARY = 3
    COLUMN_STATUS_FLAG = 4
    COLUMN_URL = 5

    SUBSCRIBE_CANCELLED = False

    def __init__(self, widgets):
        Event.SignalEmitter.__init__(self)
        self._main_window = straw.main_window.get_window()
        self._window = widgets.get_widget('subscribe_window')
        self._window.set_transient_for(self._main_window)

        self._notebook = widgets.get_widget("subscribe_notebook")
        self._entry_url = widgets.get_widget("subscribe_location_entry")

        self._progress_bar = widgets.get_widget("subscribe_progress_bar")
        self._progress_label = widgets.get_widget("subscribe_progress_label")

        self._username_entry = widgets.get_widget("subscribe_username_entry")
        self._password_entry = widgets.get_widget("subscribe_password_entry")
        self._auth_label = widgets.get_widget("auth_label")

        self._ok_button = widgets.get_widget("subscribe_ok_button")

        self._treeview = widgets.get_widget("feeds_treeview")
        model = gtk.ListStore(gobject.TYPE_OBJECT, gobject.TYPE_STRING,
                              gobject.TYPE_PYOBJECT, gobject.TYPE_PYOBJECT, 
                              gobject.TYPE_BOOLEAN, gobject.TYPE_STRING)
        self._treeview.set_model(model)
        renderer = gtk.CellRendererToggle()
        column = gtk.TreeViewColumn(_('Subscribe'), renderer, active=self.COLUMN_STATUS_FLAG)
        self._treeview.append_column(column)
        renderer.connect('toggled', self.sticky_toggled)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_('Title'), renderer, markup=self.COLUMN_TITLE)
        self._treeview.append_column(column)

        self._finder = FeedFinder()
        self._finder.signal_connect(Event.FinderDoneSignal, self.finder_done_cb)
        self._finder.signal_connect(Event.AuthNeededSignal, self.auth_needed_cb)

        self._url = None # full uri
        self._location = None
        self._username = None
        self._password = None
        self._progress_timer = 0
        self._mf_ticked_feeds = dict() # mf: multiple feed

        nameFuncMap = dict()

        for key in dir(self.__class__):
            if key[:3] == 'on_':
                nameFuncMap[key] = getattr(self, key)

        widgets.signal_autoconnect(nameFuncMap)

    def show(self, url):
        self.SUBSCRIBE_CANCELLED = False
        if url is not None:
            self.feed_subscribe(url)
        else:
            self._location = ""
            self._notebook.set_current_page(self.LOCATION_PAGE)
            self._ok_button.show()
            self._entry_url.grab_focus()
        self._window.show()

    def hide(self):
        self._window.hide()
        self._entry_url.delete_text(0,-1)
        self._progress_timer = 0

    def on_subscribe_window_delete_event(self, *args):
        self.hide()
        return True

    def on_subscribe_cancel_button_clicked(self, event):
        if not (self._notebook.get_current_page() == self.LOCATION_PAGE):
            SUBSCRIBE_CANCELLED = True
        self.hide()

    def on_subscribe_location_entry_key_press_event(self, widget, event):
        if event.keyval == gtk.keysyms.Return:
            self.feed_subscribe(self._entry_url.get_text())

    def on_subscribe_ok_button_clicked(self, event):
        current_page = self._notebook.get_current_page()
        if current_page == self.LOCATION_PAGE:
            self.feed_subscribe(self._entry_url.get_text())
        elif current_page == self.AUTH_PAGE:
            self._username = self._username_entry.get_text()
            self._password = self._password_entry.get_text()
            self.use_http_backend()
        elif current_page == self.MULTIPLE_FEED_PAGE:
            for feed, ps in self._mf_ticked_feeds.itervalues():
                category = straw.main.get_visible_category() or None
                fl = FeedList.get_instance()
                fl.append(category, feed)
                feed.router.route_all(None, ps)
                feed.poll_done()
            self.hide()

    def show_progress(self):
        def _update_progress_bar():
            self._progress_bar.pulse()
            if not self._progress_timer:
                return False
            return True
        self._ok_button.hide()
        self._notebook.set_current_page(self.PROGRESS_PAGE)
        self._progress_timer = gtk.timeout_add(150, _update_progress_bar)

    def feed_subscribe(self, url):
        m = urlparse.urlsplit(url, "http")
        if m[1] == '': # bug in urlparse
            m = urlparse.urlsplit("//" + url, "http")
        scheme, loc, path, query, fragment = m
        auth_info = loc.split('@')

        if len(auth_info) > 1:
            # format is scheme://username@password:location
            self._location = auth_info[1]
            self._username, self._password = auth_info[0].split(':')
            self._url = urlparse.urlunsplit((scheme, loc, path, query, fragment))
        else:
            self._location = "%s://%s" % (scheme, loc)
            self._url = url

        if scheme == 'http':
            self.use_http_backend()
        else:
            self.use_vfs_backend(scheme)

    def use_http_backend(self):
        self.show_progress()
        self._progress_label.set_text(_("Connecting to '%s'") % self._location)
        self._finder.find(self._url, self._username, self._password)

    def auth_needed_cb(self, event):
        """
        Called when host needs authorization
        """
        self._notebook.set_current_page(self.AUTH_PAGE)
        self._auth_label.set_text(_("Enter username and password for '%s'" % self._location))
        self._username_entry.grab_focus()
        self._ok_button.show()

    def finder_done_cb(self, ev):
        """
        Called when FeedFinder is done looking for feeds
        """
        if ev.error:
            MessageManager.get_instance().post_message(str(ev.error))
            self.SUBSCRIBE_CANCELLED = True
            self.hide()
            return

        _feeds = list()
        num_feeds = len(ev.feeds)

        self._progress_label.set_text(_("Getting feed info ...."))

        def summary_poller_done_cb(event):
            _feeds.append(event.feed)

            self._progress_label.set_text(_("Processing %s of %s feeds") % (len(_feeds), num_feeds))

            if not self.SUBSCRIBE_CANCELLED:
                if len(_feeds) == num_feeds:
                    if num_feeds == 1:
                        feed, ps = event.feed
                        category = straw.main.get_visible_category() or None
                        fl = FeedList.get_instance()
                        fl.append(category, feed)
                        feed.router.route_all(None, ps)
                        feed.poll_done()
                        self.hide()
                    elif num_feeds > 1:
                        self.show_multiple_feeds_page(_feeds)
                elif event.error:
                    error.log(event.error)

        for url in ev.feeds:
            summary_poller = FeedSummaryPoller()
            summary_poller.signal_connect(Event.FeedSummaryDoneSignal, summary_poller_done_cb)
            summary_poller.poll(url, self._username, self._password)

    def show_multiple_feeds_page(self, feeds):
        self._notebook.set_current_page(self.MULTIPLE_FEED_PAGE)
        self._ok_button.show()
        self.display_choices(feeds)

    def sticky_toggled(self, cell, path):
        model = self._treeview.get_model()
        iter = model.get_iter((int(path),))
        feed = model.get_value(iter, self.COLUMN_FEED)
        ps = model.get_value(iter, self.COLUMN_PARSEDSUMMARY)
        sticky = not model.get_value(iter, self.COLUMN_STATUS_FLAG)
        model.set(iter, self.COLUMN_STATUS_FLAG, sticky)

        if feed.id in self._mf_ticked_feeds.iterkeys():
            del self._mf_ticked_feeds[feed.id]
        else:
            self._mf_ticked_feeds[feed.id] = (feed, ps)

    def display_choices(self, feeds):
        model = self._treeview.get_model()
        model.clear()
        for feed, ps in feeds:
            iter = model.append()
            escaped_text = "<b>%s</b>\n%s" % (saxutils.escape(feed.title), saxutils.escape(feed.channel_description))
            escaped_text = "%s\n%s" % (escaped_text.strip(), saxutils.escape(feed.access_info[0]))
            model.set(iter, self.COLUMN_TITLE, escaped_text,
                      self.COLUMN_FEED, feed, self.COLUMN_PARSEDSUMMARY, ps,
                      self.COLUMN_STATUS_FLAG, feed.id in self._mf_ticked_feeds,
                      self.COLUMN_URL, feed.access_info[0])

    def use_vfs_backend(self, scheme):
        hig_alert.reportError(_("Unsupported Protocol"),
                              _("Subscribing to '%s://' is not yet supported") % scheme)

class FeedSummaryPoller(Event.SignalEmitter):
    def __init__(self):
        Event.SignalEmitter.__init__(self)
        self.initialize_slots(Event.FeedSummaryDoneSignal)
        self._headers = dict()
        self._url = None
        self._username = None
        self._password = None
        self._feed = None

    def poll(self, url, username, password):
        self._url = url
        self._username = username
        self._password = password
        try:
            pc = straw.URLFetch.connection_manager.request(self._url, self,
                                                           self._headers, self._username, self._password, priority=straw.NetworkConstants.PRIORITY_RSS)
        except Exception, e:
            error.log_exc("Caught an exception while polling")
            self.http_failed(e)

    def http_results(self, status, header, data):
        msg = ""
        if status is None:
            msg = _("No data (%s)" % status[1])
        elif status[1] == 304:
            msg = "%s: %s" % (status[1], status[2])
        elif status[1] == 410 or status[1] == 404:
            msg = _("Unable to find the feed (%s: %s)") % (status[1], status[2])
        elif status[1] == 401:
            msg = _("Invalid username and password.")
            return
        elif status[1] > 299:
            msg = _("Updating feed resulted in abnormal status '%s' (code %d)") % (status[2].strip(), status[1].strip())

        if msg:
            self.http_failed(msg)
            return

        self._feed = Feed.create_new_feed("", self._url, self._username, self._password)
        parsed = SummaryParser.parse(data, self._feed)
        self._feed.title = utils.convert_entities(parsed.title)
        self._feed.channel_description = utils.convert_entities(parsed.description)
        self._feed.access_info = (self._url, self._username, self._password)
        self._feed.last_poll = int(time.time())
        self.emit_signal(Event.FeedSummaryDoneSignal(self, self._feed, parsed))
        return

    def http_failed(self, exception):
        error.log( str(exception))
        self.emit_signal(Event.FeedSummaryDoneSignal(self, self._feed, None, str(exception)))

class FeedFinder(Event.SignalEmitter):
    """ A wrapper to RSS finder that uses Straw's async polling stuff """
    def __init__(self):
        Event.SignalEmitter.__init__(self)
        self.initialize_slots(Event.FinderDoneSignal, Event.AuthNeededSignal)
        self._url = None
        self._title = None
        self._username = None
        self._password = None
        self._headers = {}

    def find(self, url, username="", password=""):
        self._url = feedfinder.makeFullURI(url)
        self._username = username
        self._password = password

        try:
            pc = straw.URLFetch.connection_manager.request(self._url, self,
                                                      self._headers, self._username, self._password,
                                                      priority=straw.NetworkConstants.PRIORITY_RSS)
        except Exception, e:
            error.log(e)
            self.http_failed(e)

    def http_results(self, status, header, data):
        errmsg = ""
        if status[1] == 401:
            self.emit_signal(Event.AuthNeededSignal(self))
            return
        elif status[1] == 403:
            self.http_failed(_("Forbidden"))
            return
        elif status[1] >= 404:
            errmsg = "%s: %s" % (status[1], status[2].strip())

        if feedfinder.couldBeFeedData(data):
            self.emit_signal(Event.FinderDoneSignal(self, [self._url]))
            return

        feeds = feedfinder.getLinks(data, self._url)
        feeds = filter(feedfinder.isFeed, feeds)
        if not feeds:
            links = feedfinder.getALinks(data, self._url)
            locallinks = feedfinder.getLocalLinks(links, self._url)

            # XXX: feedfinder.isfeed() blocks.
            feeds = filter(feedfinder.isFeed, filter(feedfinder.isFeedLink, locallinks))
            if not feeds:
                feeds = filter(feedfinder.isFeed, filter(feedfinder.isXMLRelatedLink, locallinks))

            if not feeds:
                feeds = filter(feedfinder.isFeed, filter(feedfinder.isFeedLink, links))

            if not feeds:
                feeds = filter(feedfinder.isFeed, filter(feedfinder.isXMLRelatedLink, links))

        # TODO: Write an Async transport for xmlpclib
        #if not feeds and querySyndic8:
            # still no luck, search Syndic8 for feeds (requires xmlrpclib)
        #    _debuglog('still no luck, searching Syndic8')
        #    feeds = getFeedsFromSyndic8(uri)

        self.emit_signal(Event.FinderDoneSignal(self, feeds, error=errmsg))

    def http_failed(self, exception):
        error.log(str(exception))
        self.emit_signal(Event.FinderDoneSignal(self, list(), error=str(exception)))

    def http_permanent_redirect(self, newloc):
        self._url = newloc
        self.find(self._url)
