# -*- coding: utf-8 -*-
# Copyright (c) 2007 Jimmy Rönnholm
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import os
import re
from subprocess import Popen, PIPE
from bootconfig import utils

class Splashy:

    """Represent the Splashy configuration."""

    def __init__(self):
        """Set the variables"""
        self.binary = '/sbin/splashy'
        self.config_command = '/sbin/splashy_config'
        self.config_file = '/etc/default/splashy'
        self.update_initramfs_command = 'update-initramfs -uk all'
        if not os.path.isfile(self.binary):
            raise SystemExit('Splashy not found.')
        self.update_initramfs = False
        self.must_update_initramfs = False

    def __get_info_lines(self):
        """Get info from splashy."""
        cmd = self.config_command + ' --info'
        pipe = Popen(cmd, shell=True, stdout=PIPE).stdout
        return pipe.readlines()

    def get_active_theme(self):
        """Return the name of the currently active Splashy theme."""
        lines = self.__get_info_lines()
        # This has the potential to break if splashy output change.
        return lines[1].strip()

    def set_active_theme(self, name):
        """Set the active Splashy theme."""
        cmd = self.config_command + ' --set-theme ' + name
        os.system(cmd)
        self.update_initramfs = True

    def get_all_themes(self):
        """Return a list of the available themes.
        
        Return -1 if failed.
        """
        lines = self.__get_info_lines()
        lines.reverse()
        themes = []
        for line in lines:
            if line[:4] != "    ":
                themes.reverse()
                return themes
            themes.append(line.strip())
        return -1

    def add_theme(self, path):
        """Add theme from path."""
        path = utils.fix_filename_spaces(path)
        cmd = self.config_command + ' --install-theme ' + path
        os.system(cmd)

    def remove_theme(self, name):
        """Remove theme."""
        cmd = self.config_command + ' --remove-theme ' + name
        os.system(cmd)

    def show_preview(self):
        """Show a preview of splashy."""
        cmd = self.binary + ' test'
        os.system(cmd)

    def get_use_initramfs(self):
        """Return whether Splashy is being put into initramfs."""
        lines = utils.read_lines_from_file(self.config_file)
        line_number = utils.get_line_number(self.config_file, 'ENABLE_INITRAMFS=')
        if line_number != -1:
            line = lines[line_number]
            number_filter = re.compile('[0-9]+')
            match = number_filter.search(line[17:])
            if match:
                if match.group() == '1':
                    return True
        return False

    def set_use_initramfs(self, active):
        """Set whether Splashy should be put into initramfs."""
        identifier = 'ENABLE_INITRAMFS='
        if active:
            line_end = '1'
        else:
            line_end = '0'
        lines = utils.read_lines_from_file(self.config_file)
        line_number = utils.get_line_number(self.config_file, identifier)
        if line_number == -1:
            lines.append('\n')
            line_number = len(lines) - 1
        line = identifier + line_end + '\n'
        lines[line_number] = line
        utils.write_lines_to_file(self.config_file, lines)
        self.must_update_initramfs = True

    def do_shutdown_tasks(self):
        """Do any post-config tasks necessary."""
        if self.must_update_initramfs:
            os.system(self.update_initramfs_command)
        elif self.get_use_initramfs():
            if self.update_initramfs:
                os.system(self.update_initramfs_command)

