// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  starplot.cc
  This is the file which includes the StarPlot main() function.
  However, this does little besides set up the main program window -
  most of the program functionality is defined in the other .cc files
  in this directory.
*/

#include <gtk/gtk.h>
#include "starplot.h"
#include "../version.h"

Rules starrules = initrules;
StarArray *stararray = NULL;
GTKViewer *viewer = NULL;
GtkWidget *drawing_area = NULL;
GtkWidget *statusline = NULL;

char *help_browser = NULL;
double hr_brightmag = HR_BRIGHT_MAG, hr_dimmag = HR_DIM_MAG;

/* Backing pixmap for drawing area */
static GdkPixmap *pixmap = NULL;

void redraw_display(GtkWidget *widget)
{
  if (viewer && stararray) {
    stararray->Display(viewer);
    gdk_draw_pixmap(widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		    pixmap,
		    0, 0, 0, 0,
		    widget->allocation.width, widget->allocation.height);
  }
}


/* Create a new backing pixmap of the appropriate size */
static gint configure_event( GtkWidget         *widget,
			     GdkEventConfigure *event )
{
  if (pixmap) {
    delete viewer;
    gdk_pixmap_unref (pixmap);
  }
  pixmap = gdk_pixmap_new(widget->window,
			  widget->allocation.width,
			  widget->allocation.height,
			  -1);
  viewer = new GTKViewer(pixmap);
  redraw_display(widget);
  return TRUE;
}


/* Redraw the screen from the backing pixmap */
static gint expose_event( GtkWidget      *widget,
			  GdkEventExpose *event )
{
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return FALSE;
}


void redraw_all(Changetype changetype)
{
  char statustext[200];
  stararray->SetRules(starrules, changetype);
  redraw_display(drawing_area);
  if (hr_viewer) hr_redraw_display(hr_drawing_area);
  if (info_clist) update_info (info_clist);

  if (starrules.ChartFileNames.size() == 1)
    snprintf(statustext, 200, "Viewing %d / %d stars of %s", stararray->size(),
  	     stararray->totalstars(), starrules.ChartFileNames[0]);
  else
    snprintf(statustext, 200, "Viewing %d / %d stars in %d files",
             stararray->size(), stararray->totalstars(),
	     starrules.ChartFileNames.size());	     
  statustext[199] = 0;
  gtk_label_set_text (GTK_LABEL (statusline), statustext);
}


/* Function to respond to mouse clicks on the star chart.  A left-click
 * will recenter the chart to the clicked star.  A right-click will pop up
 * a mini-dialog box with some information about the star.  A left-click
 * on a chart legend item will toggle the visibility of that spectral 
 * class.  If no star or legend item is near the mouse click location,
 * nothing will happen.
 */
static gint button_press_event( GtkWidget      *widget,
				GdkEventButton *event )
{
  bool found = false;
  int i;
  gdouble X = event->x, Y = event->y;

  if (! (pixmap && stararray)) return FALSE;

  /* check for star legend item */
  if (event->button == 1 && starrules.ChartLegend) {
    int xbase = (viewer->width() > 80) ? viewer->width() - 75 : 5;

    if (pow(X - (xbase + LEGEND_OFFSETS[10][0]), 2)
    	+ pow(Y - LEGEND_OFFSETS[10][1], 2) < 25) {
      starrules.StarClasses[0] = !starrules.StarClasses[0];
      redraw_all(FILTER_CHANGE);
      return TRUE;
    }

    for (unsigned int i = 0; i < 10; i++) {
      if (pow(X - (xbase + LEGEND_OFFSETS[i][0]), 2) 
          + pow(Y - LEGEND_OFFSETS[i][1], 2) < 25) {
	starrules.StarClasses[i] = !starrules.StarClasses[i];
	redraw_all(FILTER_CHANGE);
	return TRUE;
      }
    }
  }

  /* if not a legend item, find the star that was clicked on */
  for (i = stararray->size() - 1; i >= 0; i--) {
    int x = (*stararray)[i].GetXPixel(), y = (*stararray)[i].GetYPixel();
    int r = (*stararray)[i].GetRPixel();

    if (pow(X - x, 2) + pow(Y - y, 2) < r * r) {
      found = true;
      break;
    }
  }
    
  if (found) {
    if (event->button == 1) {
      /* recenter the chart on the clicked star */
      starrules.ChartLocation = (*stararray)[i].GetStarXYZ();
      redraw_all(LOCATION_CHANGE);
    }
    else if (event->button == 3 || event->button == 2) {
      /* display information about the star */
      my_gtk_star_popup((*stararray)[i]);
    }
  }

  return TRUE;
}


gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  starplot_quit();
  return FALSE;
}


int main( int argc, char *argv[] )
{
  GtkWidget *window;
  GtkWidget *mainmenu;
  GtkWidget *mainbox, *hbox, *statusbar;
  GtkWidget *buttonbar;

  /* extract GTK command-line flags */
  gtk_init(&argc, &argv);

  // The HR diagram drawing area must be NULL before HR window is initialized
  hr_viewer = NULL;

  /* This sets the window title and border width: */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "StarPlot v. " STARPLOT_VERSION);
  gtk_container_set_border_width (GTK_CONTAINER (window), 0);
  my_gtk_main_menu (window, &mainmenu);
  my_gtk_push_buttons (window, &buttonbar);

  /* Set a delete_event handler for ending the program */
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), NULL);

  /* Set up the drawing area */
  drawing_area = gtk_drawing_area_new();
  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), 
			 STARPLOT_WIDTH, STARPLOT_HEIGHT);
  gtk_widget_show (drawing_area);

  /* Signals used to handle backing pixmap */
  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT(drawing_area),"configure_event",
		      (GtkSignalFunc) configure_event, NULL);
  
  /* Event signals */
  gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
		      (GtkSignalFunc) button_press_event, NULL);
  gtk_widget_set_events (drawing_area, GDK_EXPOSURE_MASK
			 | GDK_LEAVE_NOTIFY_MASK
			 | GDK_BUTTON_PRESS_MASK
			 | GDK_POINTER_MOTION_MASK
			 | GDK_POINTER_MOTION_HINT_MASK);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (hbox), buttonbar, FALSE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), drawing_area, TRUE, TRUE, 0);

  statusline = gtk_label_new ("");
  gtk_misc_set_alignment (GTK_MISC (statusline), (float)0.0, (float)0.0);
  gtk_widget_show (statusline);
  statusbar = gtk_frame_new (0);
  gtk_frame_set_shadow_type (GTK_FRAME (statusbar), GTK_SHADOW_OUT);
  gtk_widget_show (statusbar);
  gtk_container_add (GTK_CONTAINER (statusbar), statusline);

  mainbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (mainbox), 1);
  gtk_container_add (GTK_CONTAINER (window), mainbox);
  gtk_box_pack_start (GTK_BOX (mainbox), mainmenu, FALSE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), hbox, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), statusbar, FALSE, TRUE, 0);

  stararray = new StarArray();
  gtk_widget_realize (drawing_area);

  /* load settings from starplot RC file, if it exists */
  read_rc_file();

  /* assume any command-line arguments are data files to use */
  if (argc > 1) {
    starrules.ChartFileNames = StringList(argv[1]);
    for (int i = 2; i < argc; i++)
      starrules.ChartFileNames.append(argv[i]);
  }

  redraw_all(FILE_CHANGE);

  gtk_widget_show (drawing_area);
  gtk_widget_show (mainmenu);
  gtk_widget_show (buttonbar);
  gtk_widget_show (mainbox);
  gtk_widget_show (window);

  gtk_main ();
  return 0;
}
