// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  hrdiagram.cc
  This file contains code for viewing a Hertzsprung-Russell diagram of the
  stars being shown in the main program window.
*/

#include <gtk/gtk.h>
#include "starplot.h"


/* Functions to make sure the HR Diagram window stays updated */

GtkWidget *hr_window = NULL;
GTKViewer *hr_viewer = NULL;
GtkWidget *hr_drawing_area = NULL;
static GdkPixmap *hr_pixmap = NULL;


void hr_redraw_display(GtkWidget *widget)
{
  if (hr_viewer && stararray) {
    stararray->Diagram(hr_viewer, hr_brightmag, hr_dimmag);
    gdk_draw_pixmap(widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		    hr_pixmap,
		    0, 0, 0, 0,
		    widget->allocation.width, widget->allocation.height);
  }
}


/* Create a new backing pixmap of the appropriate size */
static gint configure_event( GtkWidget         *widget,
			     GdkEventConfigure *event )
{
  if (hr_pixmap) {
    delete hr_viewer;
    gdk_pixmap_unref (hr_pixmap);
  }
  hr_pixmap = gdk_pixmap_new(widget->window,
			     widget->allocation.width,
			     widget->allocation.height,
			     -1);
  hr_viewer = new GTKViewer(hr_pixmap);
  hr_redraw_display(widget);
  return TRUE;
}


/* Redraw the screen from the backing pixmap */
static gint expose_event( GtkWidget      *widget,
			  GdkEventExpose *event )
{
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  hr_pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return FALSE;
}


/* function to exit the HR Diagram window */
void hr_quit() 
{
  delete hr_viewer;
  hr_viewer = NULL;
  gtk_widget_destroy(hr_window);
  return;
}


/* functions allowing to set the magnitude bounds of the HR Diagram */

/* callback functions: */
void hr_mag_callback(GtkWidget *widget, gpointer magnitude_data) 
{
  GtkWidget **data = (GtkWidget **)magnitude_data;
  hr_brightmag = myatof(gtk_entry_get_text (GTK_ENTRY (data[0])));
  hr_dimmag =    myatof(gtk_entry_get_text (GTK_ENTRY (data[1])));
  hr_redraw_display(hr_drawing_area);
}

void hr_mag_defaults(GtkWidget *widget, 
		     gpointer data /*just here so the type checker is happy*/) 
{
  hr_brightmag = HR_BRIGHT_MAG;
  hr_dimmag = HR_DIM_MAG;
  hr_redraw_display(hr_drawing_area);
}

/* the function displaying the dialog box */
void hr_settings()
{
  GtkWidget *hr_set_dialog, *vbox, *table, *labels[2];
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *entrybox[2];
  char entryboxtext[2][10];
  
  // The window
  hr_set_dialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_policy (GTK_WINDOW (hr_set_dialog), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (hr_set_dialog), 
			"StarPlot: HR Diagram Scale");
  gtk_window_set_modal (GTK_WINDOW (hr_set_dialog), TRUE);
  gtk_signal_connect (GTK_OBJECT (hr_set_dialog), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

  // Pack it vertically
  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
  gtk_container_add (GTK_CONTAINER (hr_set_dialog), vbox);
  gtk_widget_show (vbox);

  // will put in a table:
  table = gtk_table_new (2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, FALSE, 0);
  gtk_widget_show (table);

  labels[0] = gtk_label_new ("Magnitude at top of y axis:");
  labels[1] = gtk_label_new ("Magnitude at bottom of y axis:");
  snprintf(entryboxtext[0], 10, "%+2.1f", hr_brightmag);
  snprintf(entryboxtext[1], 10, "%+2.1f", hr_dimmag);

  for (unsigned int i = 0; i < 2; i++) {
    gtk_misc_set_alignment (GTK_MISC (labels[i]), (float)0.0, (float)0.0);
    gtk_widget_show (labels[i]);

    entryboxtext[i][9] = 0;
    stripspace(entryboxtext[i]);
    entrybox[i] = gtk_entry_new_with_max_length (8);
    gtk_widget_set_usize (entrybox[i], 50, 20);
    gtk_entry_set_text (GTK_ENTRY (entrybox[i]), entryboxtext[i]);
    gtk_widget_show (entrybox[i]);

    gtk_table_attach (GTK_TABLE (table), labels[i], 0, 1, i, i + 1,
		      (GtkAttachOptions)GTK_FILL, (GtkAttachOptions)GTK_FILL,
		      5, 5);
    gtk_table_attach (GTK_TABLE (table), entrybox[i], 1, 2, i, i + 1,
		      (GtkAttachOptions)0, (GtkAttachOptions)0, 5, 5);
  }

  // set up the buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, vbox);
  gtk_signal_connect (GTK_OBJECT (OK), "clicked", 
		      GTK_SIGNAL_FUNC (hr_mag_callback),
		      entrybox);
  gtk_signal_connect (GTK_OBJECT (defaults), "clicked", 
		      GTK_SIGNAL_FUNC (hr_mag_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, hr_set_dialog);

  gtk_window_set_focus (GTK_WINDOW (hr_set_dialog), OK);
  gtk_widget_show (hr_set_dialog);
}


/* Here are all the items in the StarPlot HR Diagram window's menu. 
 *  Unimplemented items have been commented out.
 *  Note: C++ compilers don't like having non-void callback functions in the
 *  GtkItemFactoryEntry struct.  It interferes with strict type checking.
 */

GtkItemFactoryEntry hr_menu_items[] = {     /*       fn.        */
  /* menu path             acc. key    callback fn. param  type */
  { "/_File",                   0,     0,             0, "<Branch>" },
  //  { "/File/_Save Window as PNG...", 0, hr_save,       0,   "" },
  { "/File/_Close Window",      0,     hr_quit,       0, "" },

  { "/_Options",                0,     0,             0, "<Branch>" },
  { "/Options/Scale _Y Axis...",0,     hr_settings,   0, "" },

  // star label and star diameter submenus
  OPTIONS_LABEL_MENU, OPTIONS_DIAMETER_MENU // #defined in menu.h
};

GtkWidget *hr_options_label_widgets[OPTIONS_LABEL_MENU_SIZE];
GtkWidget *hr_options_diameter_widgets[OPTIONS_DIAMETER_MENU_SIZE];


/* hr_main_menu(): This function prepares the StarPlot HR Diagram window
   menu bar. */

void hr_main_menu (GtkWidget *hr_window, GtkWidget **menubar)
{
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (hr_menu_items) / sizeof (hr_menu_items[0]);
  
  accel_group = gtk_accel_group_new ();
       
  /* This function initializes the item factory.
     Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU,
     or GTK_TYPE_OPTION_MENU.
     Param 2: The path of the menu.
     Param 3: A pointer to a gtk_accel_group.  The item factory sets up
     the accelerator table while generating menus.
  */

  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", 
				       accel_group);

  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items (item_factory, nmenu_items, 
				 hr_menu_items, NULL);

  /* Attach the new accelerator group to the window. */
  gtk_window_add_accel_group (GTK_WINDOW (hr_window), accel_group);

  /* Make the options menu radio button widgets globally available and set
   *  them to have the same state as the main menu options menu */
  for (unsigned int i = 0; i < OPTIONS_LABEL_MENU_SIZE; i++) {
    GtkWidget *item;
    item = gtk_item_factory_get_widget (item_factory,
					options_label_names[i]);
    hr_options_label_widgets[i] = item;
    
    if ((GTK_CHECK_MENU_ITEM (options_label_widgets[i]))->active)
      set_item(&starrules.StarLabels, starrules.StarLabels, item, false);
  }

  for (unsigned int i = 0; i < OPTIONS_DIAMETER_MENU_SIZE; i++) {
    GtkWidget *item;
    item = gtk_item_factory_get_widget (item_factory,
					options_diameter_names[i]);
    hr_options_diameter_widgets[i] = item;

    if ((GTK_CHECK_MENU_ITEM (options_diameter_widgets[i]))->active)
      set_item(&starrules.StarDiameters, starrules.StarDiameters, item, false);
  }

  if (menubar) {
    /* Finally, return the actual menu bar created by the item factory. */ 
    *menubar = gtk_item_factory_get_widget (item_factory, "<main>");
  }
}


/* The main function for the HR Diagram window */

void hrdiagram()
{
  GtkWidget *mainbox = NULL;
  GtkWidget *menubar;

  // If the HR window is already open, do nothing but put it in front
  if (hr_viewer) {
    gdk_window_raise(hr_window->window);
    return;
  }

  /* This sets the window title and border width: */
  hr_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (hr_window), "StarPlot: HR Diagram");
  gtk_container_set_border_width (GTK_CONTAINER (hr_window), 0);

  /* Set a delete_event handler for closing the window */
  gtk_signal_connect (GTK_OBJECT (hr_window), "delete_event",
		      GTK_SIGNAL_FUNC (hr_quit), NULL);

  /* Set the menus for this window. */
  hr_main_menu (hr_window, &menubar);

  /* Set up the drawing area */
  hr_drawing_area = gtk_drawing_area_new();
  gtk_drawing_area_size (GTK_DRAWING_AREA (hr_drawing_area), 
			 HR_WIDTH, HR_HEIGHT);

  /* Signals used to handle backing pixmap */
  gtk_signal_connect (GTK_OBJECT (hr_drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT(hr_drawing_area), "configure_event",
		      (GtkSignalFunc) configure_event, NULL);
  
  /* Event signals */
  //  gtk_signal_connect (GTK_OBJECT (hr_drawing_area), "button_press_event",
  //		      (GtkSignalFunc) button_press_event, NULL);
  gtk_widget_set_events (hr_drawing_area, GDK_EXPOSURE_MASK
			 | GDK_LEAVE_NOTIFY_MASK
			 | GDK_BUTTON_PRESS_MASK
			 | GDK_POINTER_MOTION_MASK
			 | GDK_POINTER_MOTION_HINT_MASK);

  /* finish putting widgets together */
  mainbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (mainbox), 1);
  gtk_container_add (GTK_CONTAINER (hr_window), mainbox);
  gtk_box_pack_start (GTK_BOX (mainbox), menubar, FALSE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), hr_drawing_area, TRUE, TRUE, 0);

  gtk_widget_realize (hr_drawing_area);
  hr_redraw_display (hr_drawing_area);

  gtk_widget_show (hr_drawing_area);
  gtk_widget_show (menubar);
  gtk_widget_show (mainbox);
  gtk_widget_show (hr_window);

  return;
}
