/*
 * Copyright (C) 2005  Stig Venaas <venaas@uninett.no>
 * $Id: asmping.c,v 1.4 2005/11/29 16:27:26 sv Exp $
 *
 * Contributions:
 * Solaris support by Alexander Gall <gall@switch.ch>
 * Initial Windows support by Nick Lamb <njl@ecs.soton.ac.uk>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */

#include "ssmping.h"

extern int optind;
extern char *optarg;

void joingroup(int s, struct sockaddr *grp, uint32_t intface, struct sockaddr *ifaddr) {
    int e;
#ifdef MCAST_JOIN_GROUP
    int level;
    socklen_t addrlen;
    struct group_req greq;
#endif

    if (grp && ifaddr && ifaddr->sa_family != grp->sa_family) {
	fprintf(stderr, "joingroup failed, group and interface must be of same address family\n");
	exit(1);
    }
    
    switch (grp->sa_family) {
    case AF_INET:
#ifdef MCAST_JOIN_GROUP
	addrlen = sizeof(struct sockaddr_in);
	level = IPPROTO_IP;
#else
	{
	    struct ip_mreq mreq;
	    memset(&mreq, 0, sizeof(mreq));
	    mreq.imr_multiaddr = ((struct sockaddr_in *)grp)->sin_addr;
	    mreq.imr_interface = ((struct sockaddr_in *)ifaddr)->sin_addr;
	    e = setsockopt(s, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq, sizeof(mreq));
	}
#endif
	break;
    case AF_INET6:
#ifdef MCAST_JOIN_GROUP
	addrlen = sizeof(struct sockaddr_in6);
	level = IPPROTO_IPV6;
#else
	{
	    struct ipv6_mreq mreq6;
	    memset(&mreq6, 0, sizeof(mreq6));
	    mreq6.ipv6mr_multiaddr = ((struct sockaddr_in6 *)grp)->sin6_addr;
	    mreq6.ipv6mr_interface = intface;
	    e = setsockopt(s, IPPROTO_IPV6, IPV6_JOIN_GROUP, (char *)&mreq6, sizeof(mreq6));
	}
#endif
	break;
    default:
	fprintf(stderr, "joingroup failed, unsupported address family\n");
	exit(1);
    }

#ifdef MCAST_JOIN_GROUP
    memset(&greq, 0, sizeof(greq));
    memcpy(&greq.gr_group, grp, addrlen);
    greq.gr_interface = intface;
    e = setsockopt(s, level, MCAST_JOIN_GROUP, (char *)&greq, sizeof(greq));
#endif

    if (e < 0)
	errx("Failed to join multicast group");
}

int main(int argc, char **argv) {
    int famarg, family, count, us, ms;
    char *pingee, *group, source[INET6_ADDRSTRLEN];
    uint32_t intface;
    struct sockaddr_storage name, ucaddr, mcaddr, grpaddr;
    size_t namelen;
#ifdef WIN32
    WORD wVersionRequested;
    WSADATA wsaData;

    wVersionRequested = MAKEWORD( 2, 0 );
    WSAStartup(wVersionRequested, &wsaData);
    /* lots of complicated Win32 error checking expected - sigh - */
#endif

    parseargs(argc, argv, ASMMODE, &pingee, &famarg, &intface, &count, &group);
    family = famarg;
    
    /* Should rewrite this */
    ms = name2addrsock(group, &family, &mcaddr);
    if (ms < 0)
	errx("Failed to create socket for %s", group);

    us = name2addrsock(pingee, &family, &ucaddr);
    /* If this fails it might be that pingee is v4 and group is name
       with both AAAA and A, and AAAA is preferred */
    if (us < 0 && famarg == AF_UNSPEC) {
	/* Retry with the other family for both */
	close(ms);
	family = family == AF_INET ? AF_INET6 : AF_INET;
	ms = name2addrsock(group, &family, &mcaddr);
	if (ms < 0)
	    errx("Failed to create socket for %s", group);
	us = name2addrsock(pingee, &family, &ucaddr);
    }
    if (us < 0)
	errx("Failed to create socket for %s", pingee);

    prep_sock(family, us);
    prep_sock(family, ms);

    findsrc((struct sockaddr *)&name, (struct sockaddr *)&ucaddr);
    namelen = SOCKADDR_SIZE(name);
    
    setport((struct sockaddr *)&name, 0);

    if (bind(us, (struct sockaddr *)&name, namelen) < 0)
	errx("bind");

    if (connect(us, (struct sockaddr *)&ucaddr, namelen) < 0)
	errx("connect");

    if (getsockname(us, (struct sockaddr *)&name, &namelen) == -1)
	errx("getsockname");
    
    grpaddr = name;
    setaddr(&grpaddr, group ? &mcaddr : NULL, "ff3e::4321:1234", "232.43.211.234");
#ifdef WIN32
    {
	struct sockaddr_storage any = name;
	setaddr(&any, NULL, "::", "0.0.0.0");
	if (bind(ms, (struct sockaddr *)&any, namelen) < 0)
	    errx("bind [INADDR_ANY]");
    }
#else    
    if (bind(ms, (struct sockaddr *)&grpaddr, namelen) < 0)
	errx("bind [multicast]");
#endif
     /* using name to specify interface is wrong, only problem for old API */
    joingroup(ms, (struct sockaddr *)&grpaddr, intface, (struct sockaddr *)&name);
    strcpy(source, addr2string((struct sockaddr *)&ucaddr, namelen));
    printf("ssmping joined (S,G) = (%s,%s)\n", source,
	   addr2string((struct sockaddr *)&grpaddr, namelen));
    printf("pinging S from %s\n", addr2string((struct sockaddr *)&name, namelen));

    return doit(count, us, ms, &ucaddr, &grpaddr, source);
}    
