<?php
/**
 * Euc-jp decoding functions
 *
 * This file contains euc-jp decoding function that is needed to read
 * euc-jp encoded mails in non-euc-jp locale.
 *
 * Original data taken from:
 *  ftp://ftp.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/JIS0208.TXT
 *
 *   Name:             JIS X 0208 (1990) to Unicode
 *   Unicode version:  1.1
 *   Table version:    0.9
 *   Table format:     Format A
 *   Date:             8 March 1994
 *
 * Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 * This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 * No claims are made as to fitness for any particular purpose.  No
 * warranties of any kind are expressed or implied.  The recipient
 * agrees to determine applicability of information provided.  If this
 * file has been provided on optical media by Unicode, Inc., the sole
 * remedy for any claim will be exchange of defective media within 90
 * days of receipt.
 *
 * Unicode, Inc. hereby grants the right to freely use the information
 * supplied in this file in the creation of products supporting the
 * Unicode Standard, and to make copies of this file in any form for
 * internal or external distribution as long as this notice remains
 * attached.
 *
 * @copyright (c) 2004-2005 The SquirrelMail Project Team
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version $Id: euc_jp.php,v 1.10 2005/11/17 21:10:27 tokul Exp $
 * @package decode
 * @subpackage eastasia
 */

/**
 * Decode euc-jp encoded string
 * @param string $string Encoded string
 * @param boolean $save_html don't html encode special characters if true
 * @return string $string Decoded string
 */
function charset_decode_euc_jp($string,$save_html=false) {
    global $aggressive_decoding, $squirrelmail_language;

    // ja_JP uses own functions
    if ($squirrelmail_language=='ja_JP')
        return $string;

    // don't do decoding when there are no 8bit symbols
    if (! sq_is8bit($string,'euc-jp'))
        return $string;

    // this is CPU intensive task. Use recode functions if they are available. 
    if (function_exists('recode_string')) {
        // if string is already sanitized, undo htmlspecial chars
        if (! $save_html)
            $string=str_replace(array('&amp;','&quot;','&lt;','&gt;'),array('&','"','<','>'),$string);

        $string = recode_string("euc-jp..html",$string);

        // if string sanitizing is not needed, undo htmlspecialchars applied by recode.
        if ($save_html)
            $string=str_replace(array('&amp;','&quot;','&lt;','&gt;'),array('&','"','<','>'),$string);

        return $string;
    }

    /**
     * iconv does not support html target, but internal utf-8 decoding is 
     * faster than pure php implementation.
     */
    if (function_exists('iconv') && file_exists(SM_PATH . 'functions/decode/utf_8.php') ) {
        include_once(SM_PATH . 'functions/decode/utf_8.php');
        $string = iconv('euc-jp','utf-8',$string);
        return charset_decode_utf_8($string);
    }

    // try mbstring
    // TODO: check sanitizing of html special chars.
    if (function_exists('mb_convert_encoding') && 
        function_exists('sq_mb_list_encodings') &&
        check_php_version(4,3,0) &&
        in_array('euc-jp',sq_mb_list_encodings())) {
        return mb_convert_encoding($string,'HTML-ENTITIES','EUC-JP');
    }

    if (!$aggressive_decoding) return $string;

    // don't do decoding when there are no 8bit symbols
    if (! sq_is8bit($string,'euc-jp'))
        return $string;

    $eucjp = array(
        "\xA1\xA1" => '&#12288;',
        "\xA1\xA2" => '&#12289;',
        "\xA1\xA3" => '&#12290;',
        "\xA1\xA4" => '&#65292;',
        "\xA1\xA5" => '&#65294;',
        "\xA1\xA6" => '&#12539;',
        "\xA1\xA7" => '&#65306;',
        "\xA1\xA8" => '&#65307;',
        "\xA1\xA9" => '&#65311;',
        "\xA1\xAA" => '&#65281;',
        "\xA1\xAB" => '&#12443;',
        "\xA1\xAC" => '&#12444;',
        "\xA1\xAD" => '&#180;',
        "\xA1\xAE" => '&#65344;',
        "\xA1\xAF" => '&#168;',
        "\xA1\xB0" => '&#65342;',
        "\xA1\xB1" => '&#65507;',
        "\xA1\xB2" => '&#65343;',
        "\xA1\xB3" => '&#12541;',
        "\xA1\xB4" => '&#12542;',
        "\xA1\xB5" => '&#12445;',
        "\xA1\xB6" => '&#12446;',
        "\xA1\xB7" => '&#12291;',
        "\xA1\xB8" => '&#20189;',
        "\xA1\xB9" => '&#12293;',
        "\xA1\xBA" => '&#12294;',
        "\xA1\xBB" => '&#12295;',
        "\xA1\xBC" => '&#12540;',
        "\xA1\xBD" => '&#8213;',
        "\xA1\xBE" => '&#8208;',
        "\xA1\xBF" => '&#65295;',
        "\xA1\xC0" => '&#92;',
        "\xA1\xC1" => '&#12316;',
        "\xA1\xC2" => '&#8214;',
        "\xA1\xC3" => '&#65372;',
        "\xA1\xC4" => '&#8230;',
        "\xA1\xC5" => '&#8229;',
        "\xA1\xC6" => '&#8216;',
        "\xA1\xC7" => '&#8217;',
        "\xA1\xC8" => '&#8220;',
        "\xA1\xC9" => '&#8221;',
        "\xA1\xCA" => '&#65288;',
        "\xA1\xCB" => '&#65289;',
        "\xA1\xCC" => '&#12308;',
        "\xA1\xCD" => '&#12309;',
        "\xA1\xCE" => '&#65339;',
        "\xA1\xCF" => '&#65341;',
        "\xA1\xD0" => '&#65371;',
        "\xA1\xD1" => '&#65373;',
        "\xA1\xD2" => '&#12296;',
        "\xA1\xD3" => '&#12297;',
        "\xA1\xD4" => '&#12298;',
        "\xA1\xD5" => '&#12299;',
        "\xA1\xD6" => '&#12300;',
        "\xA1\xD7" => '&#12301;',
        "\xA1\xD8" => '&#12302;',
        "\xA1\xD9" => '&#12303;',
        "\xA1\xDA" => '&#12304;',
        "\xA1\xDB" => '&#12305;',
        "\xA1\xDC" => '&#65291;',
        "\xA1\xDD" => '&#8722;',
        "\xA1\xDE" => '&#177;',
        "\xA1\xDF" => '&#215;',
        "\xA1\xE0" => '&#247;',
        "\xA1\xE1" => '&#65309;',
        "\xA1\xE2" => '&#8800;',
        "\xA1\xE3" => '&#65308;',
        "\xA1\xE4" => '&#65310;',
        "\xA1\xE5" => '&#8806;',
        "\xA1\xE6" => '&#8807;',
        "\xA1\xE7" => '&#8734;',
        "\xA1\xE8" => '&#8756;',
        "\xA1\xE9" => '&#9794;',
        "\xA1\xEA" => '&#9792;',
        "\xA1\xEB" => '&#176;',
        "\xA1\xEC" => '&#8242;',
        "\xA1\xED" => '&#8243;',
        "\xA1\xEE" => '&#8451;',
        "\xA1\xEF" => '&#65509;',
        "\xA1\xF0" => '&#65284;',
        "\xA1\xF1" => '&#162;',
        "\xA1\xF2" => '&#163;',
        "\xA1\xF3" => '&#65285;',
        "\xA1\xF4" => '&#65283;',
        "\xA1\xF5" => '&#65286;',
        "\xA1\xF6" => '&#65290;',
        "\xA1\xF7" => '&#65312;',
        "\xA1\xF8" => '&#167;',
        "\xA1\xF9" => '&#9734;',
        "\xA1\xFA" => '&#9733;',
        "\xA1\xFB" => '&#9675;',
        "\xA1\xFC" => '&#9679;',
        "\xA1\xFD" => '&#9678;',
        "\xA1\xFE" => '&#9671;',
        "\xA2\xA1" => '&#9670;',
        "\xA2\xA2" => '&#9633;',
        "\xA2\xA3" => '&#9632;',
        "\xA2\xA4" => '&#9651;',
        "\xA2\xA5" => '&#9650;',
        "\xA2\xA6" => '&#9661;',
        "\xA2\xA7" => '&#9660;',
        "\xA2\xA8" => '&#8251;',
        "\xA2\xA9" => '&#12306;',
        "\xA2\xAA" => '&#8594;',
        "\xA2\xAB" => '&#8592;',
        "\xA2\xAC" => '&#8593;',
        "\xA2\xAD" => '&#8595;',
        "\xA2\xAE" => '&#12307;',
        "\xA2\xBA" => '&#8712;',
        "\xA2\xBB" => '&#8715;',
        "\xA2\xBC" => '&#8838;',
        "\xA2\xBD" => '&#8839;',
        "\xA2\xBE" => '&#8834;',
        "\xA2\xBF" => '&#8835;',
        "\xA2\xC0" => '&#8746;',
        "\xA2\xC1" => '&#8745;',
        "\xA2\xCA" => '&#8743;',
        "\xA2\xCB" => '&#8744;',
        "\xA2\xCC" => '&#172;',
        "\xA2\xCD" => '&#8658;',
        "\xA2\xCE" => '&#8660;',
        "\xA2\xCF" => '&#8704;',
        "\xA2\xD0" => '&#8707;',
        "\xA2\xDC" => '&#8736;',
        "\xA2\xDD" => '&#8869;',
        "\xA2\xDE" => '&#8978;',
        "\xA2\xDF" => '&#8706;',
        "\xA2\xE0" => '&#8711;',
        "\xA2\xE1" => '&#8801;',
        "\xA2\xE2" => '&#8786;',
        "\xA2\xE3" => '&#8810;',
        "\xA2\xE4" => '&#8811;',
        "\xA2\xE5" => '&#8730;',
        "\xA2\xE6" => '&#8765;',
        "\xA2\xE7" => '&#8733;',
        "\xA2\xE8" => '&#8757;',
        "\xA2\xE9" => '&#8747;',
        "\xA2\xEA" => '&#8748;',
        "\xA2\xF2" => '&#8491;',
        "\xA2\xF3" => '&#8240;',
        "\xA2\xF4" => '&#9839;',
        "\xA2\xF5" => '&#9837;',
        "\xA2\xF6" => '&#9834;',
        "\xA2\xF7" => '&#8224;',
        "\xA2\xF8" => '&#8225;',
        "\xA2\xF9" => '&#182;',
        "\xA2\xFE" => '&#9711;',
        "\xA3\xB0" => '&#65296;',
        "\xA3\xB1" => '&#65297;',
        "\xA3\xB2" => '&#65298;',
        "\xA3\xB3" => '&#65299;',
        "\xA3\xB4" => '&#65300;',
        "\xA3\xB5" => '&#65301;',
        "\xA3\xB6" => '&#65302;',
        "\xA3\xB7" => '&#65303;',
        "\xA3\xB8" => '&#65304;',
        "\xA3\xB9" => '&#65305;',
        "\xA3\xC1" => '&#65313;',
        "\xA3\xC2" => '&#65314;',
        "\xA3\xC3" => '&#65315;',
        "\xA3\xC4" => '&#65316;',
        "\xA3\xC5" => '&#65317;',
        "\xA3\xC6" => '&#65318;',
        "\xA3\xC7" => '&#65319;',
        "\xA3\xC8" => '&#65320;',
        "\xA3\xC9" => '&#65321;',
        "\xA3\xCA" => '&#65322;',
        "\xA3\xCB" => '&#65323;',
        "\xA3\xCC" => '&#65324;',
        "\xA3\xCD" => '&#65325;',
        "\xA3\xCE" => '&#65326;',
        "\xA3\xCF" => '&#65327;',
        "\xA3\xD0" => '&#65328;',
        "\xA3\xD1" => '&#65329;',
        "\xA3\xD2" => '&#65330;',
        "\xA3\xD3" => '&#65331;',
        "\xA3\xD4" => '&#65332;',
        "\xA3\xD5" => '&#65333;',
        "\xA3\xD6" => '&#65334;',
        "\xA3\xD7" => '&#65335;',
        "\xA3\xD8" => '&#65336;',
        "\xA3\xD9" => '&#65337;',
        "\xA3\xDA" => '&#65338;',
        "\xA3\xE1" => '&#65345;',
        "\xA3\xE2" => '&#65346;',
        "\xA3\xE3" => '&#65347;',
        "\xA3\xE4" => '&#65348;',
        "\xA3\xE5" => '&#65349;',
        "\xA3\xE6" => '&#65350;',
        "\xA3\xE7" => '&#65351;',
        "\xA3\xE8" => '&#65352;',
        "\xA3\xE9" => '&#65353;',
        "\xA3\xEA" => '&#65354;',
        "\xA3\xEB" => '&#65355;',
        "\xA3\xEC" => '&#65356;',
        "\xA3\xED" => '&#65357;',
        "\xA3\xEE" => '&#65358;',
        "\xA3\xEF" => '&#65359;',
        "\xA3\xF0" => '&#65360;',
        "\xA3\xF1" => '&#65361;',
        "\xA3\xF2" => '&#65362;',
        "\xA3\xF3" => '&#65363;',
        "\xA3\xF4" => '&#65364;',
        "\xA3\xF5" => '&#65365;',
        "\xA3\xF6" => '&#65366;',
        "\xA3\xF7" => '&#65367;',
        "\xA3\xF8" => '&#65368;',
        "\xA3\xF9" => '&#65369;',
        "\xA3\xFA" => '&#65370;',
        "\xA4\xA1" => '&#12353;',
        "\xA4\xA2" => '&#12354;',
        "\xA4\xA3" => '&#12355;',
        "\xA4\xA4" => '&#12356;',
        "\xA4\xA5" => '&#12357;',
        "\xA4\xA6" => '&#12358;',
        "\xA4\xA7" => '&#12359;',
        "\xA4\xA8" => '&#12360;',
        "\xA4\xA9" => '&#12361;',
        "\xA4\xAA" => '&#12362;',
        "\xA4\xAB" => '&#12363;',
        "\xA4\xAC" => '&#12364;',
        "\xA4\xAD" => '&#12365;',
        "\xA4\xAE" => '&#12366;',
        "\xA4\xAF" => '&#12367;',
        "\xA4\xB0" => '&#12368;',
        "\xA4\xB1" => '&#12369;',
        "\xA4\xB2" => '&#12370;',
        "\xA4\xB3" => '&#12371;',
        "\xA4\xB4" => '&#12372;',
        "\xA4\xB5" => '&#12373;',
        "\xA4\xB6" => '&#12374;',
        "\xA4\xB7" => '&#12375;',
        "\xA4\xB8" => '&#12376;',
        "\xA4\xB9" => '&#12377;',
        "\xA4\xBA" => '&#12378;',
        "\xA4\xBB" => '&#12379;',
        "\xA4\xBC" => '&#12380;',
        "\xA4\xBD" => '&#12381;',
        "\xA4\xBE" => '&#12382;',
        "\xA4\xBF" => '&#12383;',
        "\xA4\xC0" => '&#12384;',
        "\xA4\xC1" => '&#12385;',
        "\xA4\xC2" => '&#12386;',
        "\xA4\xC3" => '&#12387;',
        "\xA4\xC4" => '&#12388;',
        "\xA4\xC5" => '&#12389;',
        "\xA4\xC6" => '&#12390;',
        "\xA4\xC7" => '&#12391;',
        "\xA4\xC8" => '&#12392;',
        "\xA4\xC9" => '&#12393;',
        "\xA4\xCA" => '&#12394;',
        "\xA4\xCB" => '&#12395;',
        "\xA4\xCC" => '&#12396;',
        "\xA4\xCD" => '&#12397;',
        "\xA4\xCE" => '&#12398;',
        "\xA4\xCF" => '&#12399;',
        "\xA4\xD0" => '&#12400;',
        "\xA4\xD1" => '&#12401;',
        "\xA4\xD2" => '&#12402;',
        "\xA4\xD3" => '&#12403;',
        "\xA4\xD4" => '&#12404;',
        "\xA4\xD5" => '&#12405;',
        "\xA4\xD6" => '&#12406;',
        "\xA4\xD7" => '&#12407;',
        "\xA4\xD8" => '&#12408;',
        "\xA4\xD9" => '&#12409;',
        "\xA4\xDA" => '&#12410;',
        "\xA4\xDB" => '&#12411;',
        "\xA4\xDC" => '&#12412;',
        "\xA4\xDD" => '&#12413;',
        "\xA4\xDE" => '&#12414;',
        "\xA4\xDF" => '&#12415;',
        "\xA4\xE0" => '&#12416;',
        "\xA4\xE1" => '&#12417;',
        "\xA4\xE2" => '&#12418;',
        "\xA4\xE3" => '&#12419;',
        "\xA4\xE4" => '&#12420;',
        "\xA4\xE5" => '&#12421;',
        "\xA4\xE6" => '&#12422;',
        "\xA4\xE7" => '&#12423;',
        "\xA4\xE8" => '&#12424;',
        "\xA4\xE9" => '&#12425;',
        "\xA4\xEA" => '&#12426;',
        "\xA4\xEB" => '&#12427;',
        "\xA4\xEC" => '&#12428;',
        "\xA4\xED" => '&#12429;',
        "\xA4\xEE" => '&#12430;',
        "\xA4\xEF" => '&#12431;',
        "\xA4\xF0" => '&#12432;',
        "\xA4\xF1" => '&#12433;',
        "\xA4\xF2" => '&#12434;',
        "\xA4\xF3" => '&#12435;',
        "\xA5\xA1" => '&#12449;',
        "\xA5\xA2" => '&#12450;',
        "\xA5\xA3" => '&#12451;',
        "\xA5\xA4" => '&#12452;',
        "\xA5\xA5" => '&#12453;',
        "\xA5\xA6" => '&#12454;',
        "\xA5\xA7" => '&#12455;',
        "\xA5\xA8" => '&#12456;',
        "\xA5\xA9" => '&#12457;',
        "\xA5\xAA" => '&#12458;',
        "\xA5\xAB" => '&#12459;',
        "\xA5\xAC" => '&#12460;',
        "\xA5\xAD" => '&#12461;',
        "\xA5\xAE" => '&#12462;',
        "\xA5\xAF" => '&#12463;',
        "\xA5\xB0" => '&#12464;',
        "\xA5\xB1" => '&#12465;',
        "\xA5\xB2" => '&#12466;',
        "\xA5\xB3" => '&#12467;',
        "\xA5\xB4" => '&#12468;',
        "\xA5\xB5" => '&#12469;',
        "\xA5\xB6" => '&#12470;',
        "\xA5\xB7" => '&#12471;',
        "\xA5\xB8" => '&#12472;',
        "\xA5\xB9" => '&#12473;',
        "\xA5\xBA" => '&#12474;',
        "\xA5\xBB" => '&#12475;',
        "\xA5\xBC" => '&#12476;',
        "\xA5\xBD" => '&#12477;',
        "\xA5\xBE" => '&#12478;',
        "\xA5\xBF" => '&#12479;',
        "\xA5\xC0" => '&#12480;',
        "\xA5\xC1" => '&#12481;',
        "\xA5\xC2" => '&#12482;',
        "\xA5\xC3" => '&#12483;',
        "\xA5\xC4" => '&#12484;',
        "\xA5\xC5" => '&#12485;',
        "\xA5\xC6" => '&#12486;',
        "\xA5\xC7" => '&#12487;',
        "\xA5\xC8" => '&#12488;',
        "\xA5\xC9" => '&#12489;',
        "\xA5\xCA" => '&#12490;',
        "\xA5\xCB" => '&#12491;',
        "\xA5\xCC" => '&#12492;',
        "\xA5\xCD" => '&#12493;',
        "\xA5\xCE" => '&#12494;',
        "\xA5\xCF" => '&#12495;',
        "\xA5\xD0" => '&#12496;',
        "\xA5\xD1" => '&#12497;',
        "\xA5\xD2" => '&#12498;',
        "\xA5\xD3" => '&#12499;',
        "\xA5\xD4" => '&#12500;',
        "\xA5\xD5" => '&#12501;',
        "\xA5\xD6" => '&#12502;',
        "\xA5\xD7" => '&#12503;',
        "\xA5\xD8" => '&#12504;',
        "\xA5\xD9" => '&#12505;',
        "\xA5\xDA" => '&#12506;',
        "\xA5\xDB" => '&#12507;',
        "\xA5\xDC" => '&#12508;',
        "\xA5\xDD" => '&#12509;',
        "\xA5\xDE" => '&#12510;',
        "\xA5\xDF" => '&#12511;',
        "\xA5\xE0" => '&#12512;',
        "\xA5\xE1" => '&#12513;',
        "\xA5\xE2" => '&#12514;',
        "\xA5\xE3" => '&#12515;',
        "\xA5\xE4" => '&#12516;',
        "\xA5\xE5" => '&#12517;',
        "\xA5\xE6" => '&#12518;',
        "\xA5\xE7" => '&#12519;',
        "\xA5\xE8" => '&#12520;',
        "\xA5\xE9" => '&#12521;',
        "\xA5\xEA" => '&#12522;',
        "\xA5\xEB" => '&#12523;',
        "\xA5\xEC" => '&#12524;',
        "\xA5\xED" => '&#12525;',
        "\xA5\xEE" => '&#12526;',
        "\xA5\xEF" => '&#12527;',
        "\xA5\xF0" => '&#12528;',
        "\xA5\xF1" => '&#12529;',
        "\xA5\xF2" => '&#12530;',
        "\xA5\xF3" => '&#12531;',
        "\xA5\xF4" => '&#12532;',
        "\xA5\xF5" => '&#12533;',
        "\xA5\xF6" => '&#12534;',
        "\xA6\xA1" => '&#913;',
        "\xA6\xA2" => '&#914;',
        "\xA6\xA3" => '&#915;',
        "\xA6\xA4" => '&#916;',
        "\xA6\xA5" => '&#917;',
        "\xA6\xA6" => '&#918;',
        "\xA6\xA7" => '&#919;',
        "\xA6\xA8" => '&#920;',
        "\xA6\xA9" => '&#921;',
        "\xA6\xAA" => '&#922;',
        "\xA6\xAB" => '&#923;',
        "\xA6\xAC" => '&#924;',
        "\xA6\xAD" => '&#925;',
        "\xA6\xAE" => '&#926;',
        "\xA6\xAF" => '&#927;',
        "\xA6\xB0" => '&#928;',
        "\xA6\xB1" => '&#929;',
        "\xA6\xB2" => '&#931;',
        "\xA6\xB3" => '&#932;',
        "\xA6\xB4" => '&#933;',
        "\xA6\xB5" => '&#934;',
        "\xA6\xB6" => '&#935;',
        "\xA6\xB7" => '&#936;',
        "\xA6\xB8" => '&#937;',
        "\xA6\xC1" => '&#945;',
        "\xA6\xC2" => '&#946;',
        "\xA6\xC3" => '&#947;',
        "\xA6\xC4" => '&#948;',
        "\xA6\xC5" => '&#949;',
        "\xA6\xC6" => '&#950;',
        "\xA6\xC7" => '&#951;',
        "\xA6\xC8" => '&#952;',
        "\xA6\xC9" => '&#953;',
        "\xA6\xCA" => '&#954;',
        "\xA6\xCB" => '&#955;',
        "\xA6\xCC" => '&#956;',
        "\xA6\xCD" => '&#957;',
        "\xA6\xCE" => '&#958;',
        "\xA6\xCF" => '&#959;',
        "\xA6\xD0" => '&#960;',
        "\xA6\xD1" => '&#961;',
        "\xA6\xD2" => '&#963;',
        "\xA6\xD3" => '&#964;',
        "\xA6\xD4" => '&#965;',
        "\xA6\xD5" => '&#966;',
        "\xA6\xD6" => '&#967;',
        "\xA6\xD7" => '&#968;',
        "\xA6\xD8" => '&#969;',
        "\xA7\xA1" => '&#1040;',
        "\xA7\xA2" => '&#1041;',
        "\xA7\xA3" => '&#1042;',
        "\xA7\xA4" => '&#1043;',
        "\xA7\xA5" => '&#1044;',
        "\xA7\xA6" => '&#1045;',
        "\xA7\xA7" => '&#1025;',
        "\xA7\xA8" => '&#1046;',
        "\xA7\xA9" => '&#1047;',
        "\xA7\xAA" => '&#1048;',
        "\xA7\xAB" => '&#1049;',
        "\xA7\xAC" => '&#1050;',
        "\xA7\xAD" => '&#1051;',
        "\xA7\xAE" => '&#1052;',
        "\xA7\xAF" => '&#1053;',
        "\xA7\xB0" => '&#1054;',
        "\xA7\xB1" => '&#1055;',
        "\xA7\xB2" => '&#1056;',
        "\xA7\xB3" => '&#1057;',
        "\xA7\xB4" => '&#1058;',
        "\xA7\xB5" => '&#1059;',
        "\xA7\xB6" => '&#1060;',
        "\xA7\xB7" => '&#1061;',
        "\xA7\xB8" => '&#1062;',
        "\xA7\xB9" => '&#1063;',
        "\xA7\xBA" => '&#1064;',
        "\xA7\xBB" => '&#1065;',
        "\xA7\xBC" => '&#1066;',
        "\xA7\xBD" => '&#1067;',
        "\xA7\xBE" => '&#1068;',
        "\xA7\xBF" => '&#1069;',
        "\xA7\xC0" => '&#1070;',
        "\xA7\xC1" => '&#1071;',
        "\xA7\xD1" => '&#1072;',
        "\xA7\xD2" => '&#1073;',
        "\xA7\xD3" => '&#1074;',
        "\xA7\xD4" => '&#1075;',
        "\xA7\xD5" => '&#1076;',
        "\xA7\xD6" => '&#1077;',
        "\xA7\xD7" => '&#1105;',
        "\xA7\xD8" => '&#1078;',
        "\xA7\xD9" => '&#1079;',
        "\xA7\xDA" => '&#1080;',
        "\xA7\xDB" => '&#1081;',
        "\xA7\xDC" => '&#1082;',
        "\xA7\xDD" => '&#1083;',
        "\xA7\xDE" => '&#1084;',
        "\xA7\xDF" => '&#1085;',
        "\xA7\xE0" => '&#1086;',
        "\xA7\xE1" => '&#1087;',
        "\xA7\xE2" => '&#1088;',
        "\xA7\xE3" => '&#1089;',
        "\xA7\xE4" => '&#1090;',
        "\xA7\xE5" => '&#1091;',
        "\xA7\xE6" => '&#1092;',
        "\xA7\xE7" => '&#1093;',
        "\xA7\xE8" => '&#1094;',
        "\xA7\xE9" => '&#1095;',
        "\xA7\xEA" => '&#1096;',
        "\xA7\xEB" => '&#1097;',
        "\xA7\xEC" => '&#1098;',
        "\xA7\xED" => '&#1099;',
        "\xA7\xEE" => '&#1100;',
        "\xA7\xEF" => '&#1101;',
        "\xA7\xF0" => '&#1102;',
        "\xA7\xF1" => '&#1103;',
        "\xA8\xA1" => '&#9472;',
        "\xA8\xA2" => '&#9474;',
        "\xA8\xA3" => '&#9484;',
        "\xA8\xA4" => '&#9488;',
        "\xA8\xA5" => '&#9496;',
        "\xA8\xA6" => '&#9492;',
        "\xA8\xA7" => '&#9500;',
        "\xA8\xA8" => '&#9516;',
        "\xA8\xA9" => '&#9508;',
        "\xA8\xAA" => '&#9524;',
        "\xA8\xAB" => '&#9532;',
        "\xA8\xAC" => '&#9473;',
        "\xA8\xAD" => '&#9475;',
        "\xA8\xAE" => '&#9487;',
        "\xA8\xAF" => '&#9491;',
        "\xA8\xB0" => '&#9499;',
        "\xA8\xB1" => '&#9495;',
        "\xA8\xB2" => '&#9507;',
        "\xA8\xB3" => '&#9523;',
        "\xA8\xB4" => '&#9515;',
        "\xA8\xB5" => '&#9531;',
        "\xA8\xB6" => '&#9547;',
        "\xA8\xB7" => '&#9504;',
        "\xA8\xB8" => '&#9519;',
        "\xA8\xB9" => '&#9512;',
        "\xA8\xBA" => '&#9527;',
        "\xA8\xBB" => '&#9535;',
        "\xA8\xBC" => '&#9501;',
        "\xA8\xBD" => '&#9520;',
        "\xA8\xBE" => '&#9509;',
        "\xA8\xBF" => '&#9528;',
        "\xA8\xC0" => '&#9538;',
        "\xB0\xA1" => '&#20124;',
        "\xB0\xA2" => '&#21782;',
        "\xB0\xA3" => '&#23043;',
        "\xB0\xA4" => '&#38463;',
        "\xB0\xA5" => '&#21696;',
        "\xB0\xA6" => '&#24859;',
        "\xB0\xA7" => '&#25384;',
        "\xB0\xA8" => '&#23030;',
        "\xB0\xA9" => '&#36898;',
        "\xB0\xAA" => '&#33909;',
        "\xB0\xAB" => '&#33564;',
        "\xB0\xAC" => '&#31312;',
        "\xB0\xAD" => '&#24746;',
        "\xB0\xAE" => '&#25569;',
        "\xB0\xAF" => '&#28197;',
        "\xB0\xB0" => '&#26093;',
        "\xB0\xB1" => '&#33894;',
        "\xB0\xB2" => '&#33446;',
        "\xB0\xB3" => '&#39925;',
        "\xB0\xB4" => '&#26771;',
        "\xB0\xB5" => '&#22311;',
        "\xB0\xB6" => '&#26017;',
        "\xB0\xB7" => '&#25201;',
        "\xB0\xB8" => '&#23451;',
        "\xB0\xB9" => '&#22992;',
        "\xB0\xBA" => '&#34427;',
        "\xB0\xBB" => '&#39156;',
        "\xB0\xBC" => '&#32098;',
        "\xB0\xBD" => '&#32190;',
        "\xB0\xBE" => '&#39822;',
        "\xB0\xBF" => '&#25110;',
        "\xB0\xC0" => '&#31903;',
        "\xB0\xC1" => '&#34999;',
        "\xB0\xC2" => '&#23433;',
        "\xB0\xC3" => '&#24245;',
        "\xB0\xC4" => '&#25353;',
        "\xB0\xC5" => '&#26263;',
        "\xB0\xC6" => '&#26696;',
        "\xB0\xC7" => '&#38343;',
        "\xB0\xC8" => '&#38797;',
        "\xB0\xC9" => '&#26447;',
        "\xB0\xCA" => '&#20197;',
        "\xB0\xCB" => '&#20234;',
        "\xB0\xCC" => '&#20301;',
        "\xB0\xCD" => '&#20381;',
        "\xB0\xCE" => '&#20553;',
        "\xB0\xCF" => '&#22258;',
        "\xB0\xD0" => '&#22839;',
        "\xB0\xD1" => '&#22996;',
        "\xB0\xD2" => '&#23041;',
        "\xB0\xD3" => '&#23561;',
        "\xB0\xD4" => '&#24799;',
        "\xB0\xD5" => '&#24847;',
        "\xB0\xD6" => '&#24944;',
        "\xB0\xD7" => '&#26131;',
        "\xB0\xD8" => '&#26885;',
        "\xB0\xD9" => '&#28858;',
        "\xB0\xDA" => '&#30031;',
        "\xB0\xDB" => '&#30064;',
        "\xB0\xDC" => '&#31227;',
        "\xB0\xDD" => '&#32173;',
        "\xB0\xDE" => '&#32239;',
        "\xB0\xDF" => '&#32963;',
        "\xB0\xE0" => '&#33806;',
        "\xB0\xE1" => '&#34915;',
        "\xB0\xE2" => '&#35586;',
        "\xB0\xE3" => '&#36949;',
        "\xB0\xE4" => '&#36986;',
        "\xB0\xE5" => '&#21307;',
        "\xB0\xE6" => '&#20117;',
        "\xB0\xE7" => '&#20133;',
        "\xB0\xE8" => '&#22495;',
        "\xB0\xE9" => '&#32946;',
        "\xB0\xEA" => '&#37057;',
        "\xB0\xEB" => '&#30959;',
        "\xB0\xEC" => '&#19968;',
        "\xB0\xED" => '&#22769;',
        "\xB0\xEE" => '&#28322;',
        "\xB0\xEF" => '&#36920;',
        "\xB0\xF0" => '&#31282;',
        "\xB0\xF1" => '&#33576;',
        "\xB0\xF2" => '&#33419;',
        "\xB0\xF3" => '&#39983;',
        "\xB0\xF4" => '&#20801;',
        "\xB0\xF5" => '&#21360;',
        "\xB0\xF6" => '&#21693;',
        "\xB0\xF7" => '&#21729;',
        "\xB0\xF8" => '&#22240;',
        "\xB0\xF9" => '&#23035;',
        "\xB0\xFA" => '&#24341;',
        "\xB0\xFB" => '&#39154;',
        "\xB0\xFC" => '&#28139;',
        "\xB0\xFD" => '&#32996;',
        "\xB0\xFE" => '&#34093;',
        "\xB1\xA1" => '&#38498;',
        "\xB1\xA2" => '&#38512;',
        "\xB1\xA3" => '&#38560;',
        "\xB1\xA4" => '&#38907;',
        "\xB1\xA5" => '&#21515;',
        "\xB1\xA6" => '&#21491;',
        "\xB1\xA7" => '&#23431;',
        "\xB1\xA8" => '&#28879;',
        "\xB1\xA9" => '&#32701;',
        "\xB1\xAA" => '&#36802;',
        "\xB1\xAB" => '&#38632;',
        "\xB1\xAC" => '&#21359;',
        "\xB1\xAD" => '&#40284;',
        "\xB1\xAE" => '&#31418;',
        "\xB1\xAF" => '&#19985;',
        "\xB1\xB0" => '&#30867;',
        "\xB1\xB1" => '&#33276;',
        "\xB1\xB2" => '&#28198;',
        "\xB1\xB3" => '&#22040;',
        "\xB1\xB4" => '&#21764;',
        "\xB1\xB5" => '&#27421;',
        "\xB1\xB6" => '&#34074;',
        "\xB1\xB7" => '&#39995;',
        "\xB1\xB8" => '&#23013;',
        "\xB1\xB9" => '&#21417;',
        "\xB1\xBA" => '&#28006;',
        "\xB1\xBB" => '&#29916;',
        "\xB1\xBC" => '&#38287;',
        "\xB1\xBD" => '&#22082;',
        "\xB1\xBE" => '&#20113;',
        "\xB1\xBF" => '&#36939;',
        "\xB1\xC0" => '&#38642;',
        "\xB1\xC1" => '&#33615;',
        "\xB1\xC2" => '&#39180;',
        "\xB1\xC3" => '&#21473;',
        "\xB1\xC4" => '&#21942;',
        "\xB1\xC5" => '&#23344;',
        "\xB1\xC6" => '&#24433;',
        "\xB1\xC7" => '&#26144;',
        "\xB1\xC8" => '&#26355;',
        "\xB1\xC9" => '&#26628;',
        "\xB1\xCA" => '&#27704;',
        "\xB1\xCB" => '&#27891;',
        "\xB1\xCC" => '&#27945;',
        "\xB1\xCD" => '&#29787;',
        "\xB1\xCE" => '&#30408;',
        "\xB1\xCF" => '&#31310;',
        "\xB1\xD0" => '&#38964;',
        "\xB1\xD1" => '&#33521;',
        "\xB1\xD2" => '&#34907;',
        "\xB1\xD3" => '&#35424;',
        "\xB1\xD4" => '&#37613;',
        "\xB1\xD5" => '&#28082;',
        "\xB1\xD6" => '&#30123;',
        "\xB1\xD7" => '&#30410;',
        "\xB1\xD8" => '&#39365;',
        "\xB1\xD9" => '&#24742;',
        "\xB1\xDA" => '&#35585;',
        "\xB1\xDB" => '&#36234;',
        "\xB1\xDC" => '&#38322;',
        "\xB1\xDD" => '&#27022;',
        "\xB1\xDE" => '&#21421;',
        "\xB1\xDF" => '&#20870;',
        "\xB1\xE0" => '&#22290;',
        "\xB1\xE1" => '&#22576;',
        "\xB1\xE2" => '&#22852;',
        "\xB1\xE3" => '&#23476;',
        "\xB1\xE4" => '&#24310;',
        "\xB1\xE5" => '&#24616;',
        "\xB1\xE6" => '&#25513;',
        "\xB1\xE7" => '&#25588;',
        "\xB1\xE8" => '&#27839;',
        "\xB1\xE9" => '&#28436;',
        "\xB1\xEA" => '&#28814;',
        "\xB1\xEB" => '&#28948;',
        "\xB1\xEC" => '&#29017;',
        "\xB1\xED" => '&#29141;',
        "\xB1\xEE" => '&#29503;',
        "\xB1\xEF" => '&#32257;',
        "\xB1\xF0" => '&#33398;',
        "\xB1\xF1" => '&#33489;',
        "\xB1\xF2" => '&#34199;',
        "\xB1\xF3" => '&#36960;',
        "\xB1\xF4" => '&#37467;',
        "\xB1\xF5" => '&#40219;',
        "\xB1\xF6" => '&#22633;',
        "\xB1\xF7" => '&#26044;',
        "\xB1\xF8" => '&#27738;',
        "\xB1\xF9" => '&#29989;',
        "\xB1\xFA" => '&#20985;',
        "\xB1\xFB" => '&#22830;',
        "\xB1\xFC" => '&#22885;',
        "\xB1\xFD" => '&#24448;',
        "\xB1\xFE" => '&#24540;',
        "\xB2\xA1" => '&#25276;',
        "\xB2\xA2" => '&#26106;',
        "\xB2\xA3" => '&#27178;',
        "\xB2\xA4" => '&#27431;',
        "\xB2\xA5" => '&#27572;',
        "\xB2\xA6" => '&#29579;',
        "\xB2\xA7" => '&#32705;',
        "\xB2\xA8" => '&#35158;',
        "\xB2\xA9" => '&#40236;',
        "\xB2\xAA" => '&#40206;',
        "\xB2\xAB" => '&#40644;',
        "\xB2\xAC" => '&#23713;',
        "\xB2\xAD" => '&#27798;',
        "\xB2\xAE" => '&#33659;',
        "\xB2\xAF" => '&#20740;',
        "\xB2\xB0" => '&#23627;',
        "\xB2\xB1" => '&#25014;',
        "\xB2\xB2" => '&#33222;',
        "\xB2\xB3" => '&#26742;',
        "\xB2\xB4" => '&#29281;',
        "\xB2\xB5" => '&#20057;',
        "\xB2\xB6" => '&#20474;',
        "\xB2\xB7" => '&#21368;',
        "\xB2\xB8" => '&#24681;',
        "\xB2\xB9" => '&#28201;',
        "\xB2\xBA" => '&#31311;',
        "\xB2\xBB" => '&#38899;',
        "\xB2\xBC" => '&#19979;',
        "\xB2\xBD" => '&#21270;',
        "\xB2\xBE" => '&#20206;',
        "\xB2\xBF" => '&#20309;',
        "\xB2\xC0" => '&#20285;',
        "\xB2\xC1" => '&#20385;',
        "\xB2\xC2" => '&#20339;',
        "\xB2\xC3" => '&#21152;',
        "\xB2\xC4" => '&#21487;',
        "\xB2\xC5" => '&#22025;',
        "\xB2\xC6" => '&#22799;',
        "\xB2\xC7" => '&#23233;',
        "\xB2\xC8" => '&#23478;',
        "\xB2\xC9" => '&#23521;',
        "\xB2\xCA" => '&#31185;',
        "\xB2\xCB" => '&#26247;',
        "\xB2\xCC" => '&#26524;',
        "\xB2\xCD" => '&#26550;',
        "\xB2\xCE" => '&#27468;',
        "\xB2\xCF" => '&#27827;',
        "\xB2\xD0" => '&#28779;',
        "\xB2\xD1" => '&#29634;',
        "\xB2\xD2" => '&#31117;',
        "\xB2\xD3" => '&#31166;',
        "\xB2\xD4" => '&#31292;',
        "\xB2\xD5" => '&#31623;',
        "\xB2\xD6" => '&#33457;',
        "\xB2\xD7" => '&#33499;',
        "\xB2\xD8" => '&#33540;',
        "\xB2\xD9" => '&#33655;',
        "\xB2\xDA" => '&#33775;',
        "\xB2\xDB" => '&#33747;',
        "\xB2\xDC" => '&#34662;',
        "\xB2\xDD" => '&#35506;',
        "\xB2\xDE" => '&#22057;',
        "\xB2\xDF" => '&#36008;',
        "\xB2\xE0" => '&#36838;',
        "\xB2\xE1" => '&#36942;',
        "\xB2\xE2" => '&#38686;',
        "\xB2\xE3" => '&#34442;',
        "\xB2\xE4" => '&#20420;',
        "\xB2\xE5" => '&#23784;',
        "\xB2\xE6" => '&#25105;',
        "\xB2\xE7" => '&#29273;',
        "\xB2\xE8" => '&#30011;',
        "\xB2\xE9" => '&#33253;',
        "\xB2\xEA" => '&#33469;',
        "\xB2\xEB" => '&#34558;',
        "\xB2\xEC" => '&#36032;',
        "\xB2\xED" => '&#38597;',
        "\xB2\xEE" => '&#39187;',
        "\xB2\xEF" => '&#39381;',
        "\xB2\xF0" => '&#20171;',
        "\xB2\xF1" => '&#20250;',
        "\xB2\xF2" => '&#35299;',
        "\xB2\xF3" => '&#22238;',
        "\xB2\xF4" => '&#22602;',
        "\xB2\xF5" => '&#22730;',
        "\xB2\xF6" => '&#24315;',
        "\xB2\xF7" => '&#24555;',
        "\xB2\xF8" => '&#24618;',
        "\xB2\xF9" => '&#24724;',
        "\xB2\xFA" => '&#24674;',
        "\xB2\xFB" => '&#25040;',
        "\xB2\xFC" => '&#25106;',
        "\xB2\xFD" => '&#25296;',
        "\xB2\xFE" => '&#25913;',
        "\xB3\xA1" => '&#39745;',
        "\xB3\xA2" => '&#26214;',
        "\xB3\xA3" => '&#26800;',
        "\xB3\xA4" => '&#28023;',
        "\xB3\xA5" => '&#28784;',
        "\xB3\xA6" => '&#30028;',
        "\xB3\xA7" => '&#30342;',
        "\xB3\xA8" => '&#32117;',
        "\xB3\xA9" => '&#33445;',
        "\xB3\xAA" => '&#34809;',
        "\xB3\xAB" => '&#38283;',
        "\xB3\xAC" => '&#38542;',
        "\xB3\xAD" => '&#35997;',
        "\xB3\xAE" => '&#20977;',
        "\xB3\xAF" => '&#21182;',
        "\xB3\xB0" => '&#22806;',
        "\xB3\xB1" => '&#21683;',
        "\xB3\xB2" => '&#23475;',
        "\xB3\xB3" => '&#23830;',
        "\xB3\xB4" => '&#24936;',
        "\xB3\xB5" => '&#27010;',
        "\xB3\xB6" => '&#28079;',
        "\xB3\xB7" => '&#30861;',
        "\xB3\xB8" => '&#33995;',
        "\xB3\xB9" => '&#34903;',
        "\xB3\xBA" => '&#35442;',
        "\xB3\xBB" => '&#37799;',
        "\xB3\xBC" => '&#39608;',
        "\xB3\xBD" => '&#28012;',
        "\xB3\xBE" => '&#39336;',
        "\xB3\xBF" => '&#34521;',
        "\xB3\xC0" => '&#22435;',
        "\xB3\xC1" => '&#26623;',
        "\xB3\xC2" => '&#34510;',
        "\xB3\xC3" => '&#37390;',
        "\xB3\xC4" => '&#21123;',
        "\xB3\xC5" => '&#22151;',
        "\xB3\xC6" => '&#21508;',
        "\xB3\xC7" => '&#24275;',
        "\xB3\xC8" => '&#25313;',
        "\xB3\xC9" => '&#25785;',
        "\xB3\xCA" => '&#26684;',
        "\xB3\xCB" => '&#26680;',
        "\xB3\xCC" => '&#27579;',
        "\xB3\xCD" => '&#29554;',
        "\xB3\xCE" => '&#30906;',
        "\xB3\xCF" => '&#31339;',
        "\xB3\xD0" => '&#35226;',
        "\xB3\xD1" => '&#35282;',
        "\xB3\xD2" => '&#36203;',
        "\xB3\xD3" => '&#36611;',
        "\xB3\xD4" => '&#37101;',
        "\xB3\xD5" => '&#38307;',
        "\xB3\xD6" => '&#38548;',
        "\xB3\xD7" => '&#38761;',
        "\xB3\xD8" => '&#23398;',
        "\xB3\xD9" => '&#23731;',
        "\xB3\xDA" => '&#27005;',
        "\xB3\xDB" => '&#38989;',
        "\xB3\xDC" => '&#38990;',
        "\xB3\xDD" => '&#25499;',
        "\xB3\xDE" => '&#31520;',
        "\xB3\xDF" => '&#27179;',
        "\xB3\xE0" => '&#27263;',
        "\xB3\xE1" => '&#26806;',
        "\xB3\xE2" => '&#39949;',
        "\xB3\xE3" => '&#28511;',
        "\xB3\xE4" => '&#21106;',
        "\xB3\xE5" => '&#21917;',
        "\xB3\xE6" => '&#24688;',
        "\xB3\xE7" => '&#25324;',
        "\xB3\xE8" => '&#27963;',
        "\xB3\xE9" => '&#28167;',
        "\xB3\xEA" => '&#28369;',
        "\xB3\xEB" => '&#33883;',
        "\xB3\xEC" => '&#35088;',
        "\xB3\xED" => '&#36676;',
        "\xB3\xEE" => '&#19988;',
        "\xB3\xEF" => '&#39993;',
        "\xB3\xF0" => '&#21494;',
        "\xB3\xF1" => '&#26907;',
        "\xB3\xF2" => '&#27194;',
        "\xB3\xF3" => '&#38788;',
        "\xB3\xF4" => '&#26666;',
        "\xB3\xF5" => '&#20828;',
        "\xB3\xF6" => '&#31427;',
        "\xB3\xF7" => '&#33970;',
        "\xB3\xF8" => '&#37340;',
        "\xB3\xF9" => '&#37772;',
        "\xB3\xFA" => '&#22107;',
        "\xB3\xFB" => '&#40232;',
        "\xB3\xFC" => '&#26658;',
        "\xB3\xFD" => '&#33541;',
        "\xB3\xFE" => '&#33841;',
        "\xB4\xA1" => '&#31909;',
        "\xB4\xA2" => '&#21000;',
        "\xB4\xA3" => '&#33477;',
        "\xB4\xA4" => '&#29926;',
        "\xB4\xA5" => '&#20094;',
        "\xB4\xA6" => '&#20355;',
        "\xB4\xA7" => '&#20896;',
        "\xB4\xA8" => '&#23506;',
        "\xB4\xA9" => '&#21002;',
        "\xB4\xAA" => '&#21208;',
        "\xB4\xAB" => '&#21223;',
        "\xB4\xAC" => '&#24059;',
        "\xB4\xAD" => '&#21914;',
        "\xB4\xAE" => '&#22570;',
        "\xB4\xAF" => '&#23014;',
        "\xB4\xB0" => '&#23436;',
        "\xB4\xB1" => '&#23448;',
        "\xB4\xB2" => '&#23515;',
        "\xB4\xB3" => '&#24178;',
        "\xB4\xB4" => '&#24185;',
        "\xB4\xB5" => '&#24739;',
        "\xB4\xB6" => '&#24863;',
        "\xB4\xB7" => '&#24931;',
        "\xB4\xB8" => '&#25022;',
        "\xB4\xB9" => '&#25563;',
        "\xB4\xBA" => '&#25954;',
        "\xB4\xBB" => '&#26577;',
        "\xB4\xBC" => '&#26707;',
        "\xB4\xBD" => '&#26874;',
        "\xB4\xBE" => '&#27454;',
        "\xB4\xBF" => '&#27475;',
        "\xB4\xC0" => '&#27735;',
        "\xB4\xC1" => '&#28450;',
        "\xB4\xC2" => '&#28567;',
        "\xB4\xC3" => '&#28485;',
        "\xB4\xC4" => '&#29872;',
        "\xB4\xC5" => '&#29976;',
        "\xB4\xC6" => '&#30435;',
        "\xB4\xC7" => '&#30475;',
        "\xB4\xC8" => '&#31487;',
        "\xB4\xC9" => '&#31649;',
        "\xB4\xCA" => '&#31777;',
        "\xB4\xCB" => '&#32233;',
        "\xB4\xCC" => '&#32566;',
        "\xB4\xCD" => '&#32752;',
        "\xB4\xCE" => '&#32925;',
        "\xB4\xCF" => '&#33382;',
        "\xB4\xD0" => '&#33694;',
        "\xB4\xD1" => '&#35251;',
        "\xB4\xD2" => '&#35532;',
        "\xB4\xD3" => '&#36011;',
        "\xB4\xD4" => '&#36996;',
        "\xB4\xD5" => '&#37969;',
        "\xB4\xD6" => '&#38291;',
        "\xB4\xD7" => '&#38289;',
        "\xB4\xD8" => '&#38306;',
        "\xB4\xD9" => '&#38501;',
        "\xB4\xDA" => '&#38867;',
        "\xB4\xDB" => '&#39208;',
        "\xB4\xDC" => '&#33304;',
        "\xB4\xDD" => '&#20024;',
        "\xB4\xDE" => '&#21547;',
        "\xB4\xDF" => '&#23736;',
        "\xB4\xE0" => '&#24012;',
        "\xB4\xE1" => '&#29609;',
        "\xB4\xE2" => '&#30284;',
        "\xB4\xE3" => '&#30524;',
        "\xB4\xE4" => '&#23721;',
        "\xB4\xE5" => '&#32747;',
        "\xB4\xE6" => '&#36107;',
        "\xB4\xE7" => '&#38593;',
        "\xB4\xE8" => '&#38929;',
        "\xB4\xE9" => '&#38996;',
        "\xB4\xEA" => '&#39000;',
        "\xB4\xEB" => '&#20225;',
        "\xB4\xEC" => '&#20238;',
        "\xB4\xED" => '&#21361;',
        "\xB4\xEE" => '&#21916;',
        "\xB4\xEF" => '&#22120;',
        "\xB4\xF0" => '&#22522;',
        "\xB4\xF1" => '&#22855;',
        "\xB4\xF2" => '&#23305;',
        "\xB4\xF3" => '&#23492;',
        "\xB4\xF4" => '&#23696;',
        "\xB4\xF5" => '&#24076;',
        "\xB4\xF6" => '&#24190;',
        "\xB4\xF7" => '&#24524;',
        "\xB4\xF8" => '&#25582;',
        "\xB4\xF9" => '&#26426;',
        "\xB4\xFA" => '&#26071;',
        "\xB4\xFB" => '&#26082;',
        "\xB4\xFC" => '&#26399;',
        "\xB4\xFD" => '&#26827;',
        "\xB4\xFE" => '&#26820;',
        "\xB5\xA1" => '&#27231;',
        "\xB5\xA2" => '&#24112;',
        "\xB5\xA3" => '&#27589;',
        "\xB5\xA4" => '&#27671;',
        "\xB5\xA5" => '&#27773;',
        "\xB5\xA6" => '&#30079;',
        "\xB5\xA7" => '&#31048;',
        "\xB5\xA8" => '&#23395;',
        "\xB5\xA9" => '&#31232;',
        "\xB5\xAA" => '&#32000;',
        "\xB5\xAB" => '&#24509;',
        "\xB5\xAC" => '&#35215;',
        "\xB5\xAD" => '&#35352;',
        "\xB5\xAE" => '&#36020;',
        "\xB5\xAF" => '&#36215;',
        "\xB5\xB0" => '&#36556;',
        "\xB5\xB1" => '&#36637;',
        "\xB5\xB2" => '&#39138;',
        "\xB5\xB3" => '&#39438;',
        "\xB5\xB4" => '&#39740;',
        "\xB5\xB5" => '&#20096;',
        "\xB5\xB6" => '&#20605;',
        "\xB5\xB7" => '&#20736;',
        "\xB5\xB8" => '&#22931;',
        "\xB5\xB9" => '&#23452;',
        "\xB5\xBA" => '&#25135;',
        "\xB5\xBB" => '&#25216;',
        "\xB5\xBC" => '&#25836;',
        "\xB5\xBD" => '&#27450;',
        "\xB5\xBE" => '&#29344;',
        "\xB5\xBF" => '&#30097;',
        "\xB5\xC0" => '&#31047;',
        "\xB5\xC1" => '&#32681;',
        "\xB5\xC2" => '&#34811;',
        "\xB5\xC3" => '&#35516;',
        "\xB5\xC4" => '&#35696;',
        "\xB5\xC5" => '&#25516;',
        "\xB5\xC6" => '&#33738;',
        "\xB5\xC7" => '&#38816;',
        "\xB5\xC8" => '&#21513;',
        "\xB5\xC9" => '&#21507;',
        "\xB5\xCA" => '&#21931;',
        "\xB5\xCB" => '&#26708;',
        "\xB5\xCC" => '&#27224;',
        "\xB5\xCD" => '&#35440;',
        "\xB5\xCE" => '&#30759;',
        "\xB5\xCF" => '&#26485;',
        "\xB5\xD0" => '&#40653;',
        "\xB5\xD1" => '&#21364;',
        "\xB5\xD2" => '&#23458;',
        "\xB5\xD3" => '&#33050;',
        "\xB5\xD4" => '&#34384;',
        "\xB5\xD5" => '&#36870;',
        "\xB5\xD6" => '&#19992;',
        "\xB5\xD7" => '&#20037;',
        "\xB5\xD8" => '&#20167;',
        "\xB5\xD9" => '&#20241;',
        "\xB5\xDA" => '&#21450;',
        "\xB5\xDB" => '&#21560;',
        "\xB5\xDC" => '&#23470;',
        "\xB5\xDD" => '&#24339;',
        "\xB5\xDE" => '&#24613;',
        "\xB5\xDF" => '&#25937;',
        "\xB5\xE0" => '&#26429;',
        "\xB5\xE1" => '&#27714;',
        "\xB5\xE2" => '&#27762;',
        "\xB5\xE3" => '&#27875;',
        "\xB5\xE4" => '&#28792;',
        "\xB5\xE5" => '&#29699;',
        "\xB5\xE6" => '&#31350;',
        "\xB5\xE7" => '&#31406;',
        "\xB5\xE8" => '&#31496;',
        "\xB5\xE9" => '&#32026;',
        "\xB5\xEA" => '&#31998;',
        "\xB5\xEB" => '&#32102;',
        "\xB5\xEC" => '&#26087;',
        "\xB5\xED" => '&#29275;',
        "\xB5\xEE" => '&#21435;',
        "\xB5\xEF" => '&#23621;',
        "\xB5\xF0" => '&#24040;',
        "\xB5\xF1" => '&#25298;',
        "\xB5\xF2" => '&#25312;',
        "\xB5\xF3" => '&#25369;',
        "\xB5\xF4" => '&#28192;',
        "\xB5\xF5" => '&#34394;',
        "\xB5\xF6" => '&#35377;',
        "\xB5\xF7" => '&#36317;',
        "\xB5\xF8" => '&#37624;',
        "\xB5\xF9" => '&#28417;',
        "\xB5\xFA" => '&#31142;',
        "\xB5\xFB" => '&#39770;',
        "\xB5\xFC" => '&#20136;',
        "\xB5\xFD" => '&#20139;',
        "\xB5\xFE" => '&#20140;',
        "\xB6\xA1" => '&#20379;',
        "\xB6\xA2" => '&#20384;',
        "\xB6\xA3" => '&#20689;',
        "\xB6\xA4" => '&#20807;',
        "\xB6\xA5" => '&#31478;',
        "\xB6\xA6" => '&#20849;',
        "\xB6\xA7" => '&#20982;',
        "\xB6\xA8" => '&#21332;',
        "\xB6\xA9" => '&#21281;',
        "\xB6\xAA" => '&#21375;',
        "\xB6\xAB" => '&#21483;',
        "\xB6\xAC" => '&#21932;',
        "\xB6\xAD" => '&#22659;',
        "\xB6\xAE" => '&#23777;',
        "\xB6\xAF" => '&#24375;',
        "\xB6\xB0" => '&#24394;',
        "\xB6\xB1" => '&#24623;',
        "\xB6\xB2" => '&#24656;',
        "\xB6\xB3" => '&#24685;',
        "\xB6\xB4" => '&#25375;',
        "\xB6\xB5" => '&#25945;',
        "\xB6\xB6" => '&#27211;',
        "\xB6\xB7" => '&#27841;',
        "\xB6\xB8" => '&#29378;',
        "\xB6\xB9" => '&#29421;',
        "\xB6\xBA" => '&#30703;',
        "\xB6\xBB" => '&#33016;',
        "\xB6\xBC" => '&#33029;',
        "\xB6\xBD" => '&#33288;',
        "\xB6\xBE" => '&#34126;',
        "\xB6\xBF" => '&#37111;',
        "\xB6\xC0" => '&#37857;',
        "\xB6\xC1" => '&#38911;',
        "\xB6\xC2" => '&#39255;',
        "\xB6\xC3" => '&#39514;',
        "\xB6\xC4" => '&#20208;',
        "\xB6\xC5" => '&#20957;',
        "\xB6\xC6" => '&#23597;',
        "\xB6\xC7" => '&#26241;',
        "\xB6\xC8" => '&#26989;',
        "\xB6\xC9" => '&#23616;',
        "\xB6\xCA" => '&#26354;',
        "\xB6\xCB" => '&#26997;',
        "\xB6\xCC" => '&#29577;',
        "\xB6\xCD" => '&#26704;',
        "\xB6\xCE" => '&#31873;',
        "\xB6\xCF" => '&#20677;',
        "\xB6\xD0" => '&#21220;',
        "\xB6\xD1" => '&#22343;',
        "\xB6\xD2" => '&#24062;',
        "\xB6\xD3" => '&#37670;',
        "\xB6\xD4" => '&#26020;',
        "\xB6\xD5" => '&#27427;',
        "\xB6\xD6" => '&#27453;',
        "\xB6\xD7" => '&#29748;',
        "\xB6\xD8" => '&#31105;',
        "\xB6\xD9" => '&#31165;',
        "\xB6\xDA" => '&#31563;',
        "\xB6\xDB" => '&#32202;',
        "\xB6\xDC" => '&#33465;',
        "\xB6\xDD" => '&#33740;',
        "\xB6\xDE" => '&#34943;',
        "\xB6\xDF" => '&#35167;',
        "\xB6\xE0" => '&#35641;',
        "\xB6\xE1" => '&#36817;',
        "\xB6\xE2" => '&#37329;',
        "\xB6\xE3" => '&#21535;',
        "\xB6\xE4" => '&#37504;',
        "\xB6\xE5" => '&#20061;',
        "\xB6\xE6" => '&#20534;',
        "\xB6\xE7" => '&#21477;',
        "\xB6\xE8" => '&#21306;',
        "\xB6\xE9" => '&#29399;',
        "\xB6\xEA" => '&#29590;',
        "\xB6\xEB" => '&#30697;',
        "\xB6\xEC" => '&#33510;',
        "\xB6\xED" => '&#36527;',
        "\xB6\xEE" => '&#39366;',
        "\xB6\xEF" => '&#39368;',
        "\xB6\xF0" => '&#39378;',
        "\xB6\xF1" => '&#20855;',
        "\xB6\xF2" => '&#24858;',
        "\xB6\xF3" => '&#34398;',
        "\xB6\xF4" => '&#21936;',
        "\xB6\xF5" => '&#31354;',
        "\xB6\xF6" => '&#20598;',
        "\xB6\xF7" => '&#23507;',
        "\xB6\xF8" => '&#36935;',
        "\xB6\xF9" => '&#38533;',
        "\xB6\xFA" => '&#20018;',
        "\xB6\xFB" => '&#27355;',
        "\xB6\xFC" => '&#37351;',
        "\xB6\xFD" => '&#23633;',
        "\xB6\xFE" => '&#23624;',
        "\xB7\xA1" => '&#25496;',
        "\xB7\xA2" => '&#31391;',
        "\xB7\xA3" => '&#27795;',
        "\xB7\xA4" => '&#38772;',
        "\xB7\xA5" => '&#36705;',
        "\xB7\xA6" => '&#31402;',
        "\xB7\xA7" => '&#29066;',
        "\xB7\xA8" => '&#38536;',
        "\xB7\xA9" => '&#31874;',
        "\xB7\xAA" => '&#26647;',
        "\xB7\xAB" => '&#32368;',
        "\xB7\xAC" => '&#26705;',
        "\xB7\xAD" => '&#37740;',
        "\xB7\xAE" => '&#21234;',
        "\xB7\xAF" => '&#21531;',
        "\xB7\xB0" => '&#34219;',
        "\xB7\xB1" => '&#35347;',
        "\xB7\xB2" => '&#32676;',
        "\xB7\xB3" => '&#36557;',
        "\xB7\xB4" => '&#37089;',
        "\xB7\xB5" => '&#21350;',
        "\xB7\xB6" => '&#34952;',
        "\xB7\xB7" => '&#31041;',
        "\xB7\xB8" => '&#20418;',
        "\xB7\xB9" => '&#20670;',
        "\xB7\xBA" => '&#21009;',
        "\xB7\xBB" => '&#20804;',
        "\xB7\xBC" => '&#21843;',
        "\xB7\xBD" => '&#22317;',
        "\xB7\xBE" => '&#29674;',
        "\xB7\xBF" => '&#22411;',
        "\xB7\xC0" => '&#22865;',
        "\xB7\xC1" => '&#24418;',
        "\xB7\xC2" => '&#24452;',
        "\xB7\xC3" => '&#24693;',
        "\xB7\xC4" => '&#24950;',
        "\xB7\xC5" => '&#24935;',
        "\xB7\xC6" => '&#25001;',
        "\xB7\xC7" => '&#25522;',
        "\xB7\xC8" => '&#25658;',
        "\xB7\xC9" => '&#25964;',
        "\xB7\xCA" => '&#26223;',
        "\xB7\xCB" => '&#26690;',
        "\xB7\xCC" => '&#28179;',
        "\xB7\xCD" => '&#30054;',
        "\xB7\xCE" => '&#31293;',
        "\xB7\xCF" => '&#31995;',
        "\xB7\xD0" => '&#32076;',
        "\xB7\xD1" => '&#32153;',
        "\xB7\xD2" => '&#32331;',
        "\xB7\xD3" => '&#32619;',
        "\xB7\xD4" => '&#33550;',
        "\xB7\xD5" => '&#33610;',
        "\xB7\xD6" => '&#34509;',
        "\xB7\xD7" => '&#35336;',
        "\xB7\xD8" => '&#35427;',
        "\xB7\xD9" => '&#35686;',
        "\xB7\xDA" => '&#36605;',
        "\xB7\xDB" => '&#38938;',
        "\xB7\xDC" => '&#40335;',
        "\xB7\xDD" => '&#33464;',
        "\xB7\xDE" => '&#36814;',
        "\xB7\xDF" => '&#39912;',
        "\xB7\xE0" => '&#21127;',
        "\xB7\xE1" => '&#25119;',
        "\xB7\xE2" => '&#25731;',
        "\xB7\xE3" => '&#28608;',
        "\xB7\xE4" => '&#38553;',
        "\xB7\xE5" => '&#26689;',
        "\xB7\xE6" => '&#20625;',
        "\xB7\xE7" => '&#27424;',
        "\xB7\xE8" => '&#27770;',
        "\xB7\xE9" => '&#28500;',
        "\xB7\xEA" => '&#31348;',
        "\xB7\xEB" => '&#32080;',
        "\xB7\xEC" => '&#34880;',
        "\xB7\xED" => '&#35363;',
        "\xB7\xEE" => '&#26376;',
        "\xB7\xEF" => '&#20214;',
        "\xB7\xF0" => '&#20537;',
        "\xB7\xF1" => '&#20518;',
        "\xB7\xF2" => '&#20581;',
        "\xB7\xF3" => '&#20860;',
        "\xB7\xF4" => '&#21048;',
        "\xB7\xF5" => '&#21091;',
        "\xB7\xF6" => '&#21927;',
        "\xB7\xF7" => '&#22287;',
        "\xB7\xF8" => '&#22533;',
        "\xB7\xF9" => '&#23244;',
        "\xB7\xFA" => '&#24314;',
        "\xB7\xFB" => '&#25010;',
        "\xB7\xFC" => '&#25080;',
        "\xB7\xFD" => '&#25331;',
        "\xB7\xFE" => '&#25458;',
        "\xB8\xA1" => '&#26908;',
        "\xB8\xA2" => '&#27177;',
        "\xB8\xA3" => '&#29309;',
        "\xB8\xA4" => '&#29356;',
        "\xB8\xA5" => '&#29486;',
        "\xB8\xA6" => '&#30740;',
        "\xB8\xA7" => '&#30831;',
        "\xB8\xA8" => '&#32121;',
        "\xB8\xA9" => '&#30476;',
        "\xB8\xAA" => '&#32937;',
        "\xB8\xAB" => '&#35211;',
        "\xB8\xAC" => '&#35609;',
        "\xB8\xAD" => '&#36066;',
        "\xB8\xAE" => '&#36562;',
        "\xB8\xAF" => '&#36963;',
        "\xB8\xB0" => '&#37749;',
        "\xB8\xB1" => '&#38522;',
        "\xB8\xB2" => '&#38997;',
        "\xB8\xB3" => '&#39443;',
        "\xB8\xB4" => '&#40568;',
        "\xB8\xB5" => '&#20803;',
        "\xB8\xB6" => '&#21407;',
        "\xB8\xB7" => '&#21427;',
        "\xB8\xB8" => '&#24187;',
        "\xB8\xB9" => '&#24358;',
        "\xB8\xBA" => '&#28187;',
        "\xB8\xBB" => '&#28304;',
        "\xB8\xBC" => '&#29572;',
        "\xB8\xBD" => '&#29694;',
        "\xB8\xBE" => '&#32067;',
        "\xB8\xBF" => '&#33335;',
        "\xB8\xC0" => '&#35328;',
        "\xB8\xC1" => '&#35578;',
        "\xB8\xC2" => '&#38480;',
        "\xB8\xC3" => '&#20046;',
        "\xB8\xC4" => '&#20491;',
        "\xB8\xC5" => '&#21476;',
        "\xB8\xC6" => '&#21628;',
        "\xB8\xC7" => '&#22266;',
        "\xB8\xC8" => '&#22993;',
        "\xB8\xC9" => '&#23396;',
        "\xB8\xCA" => '&#24049;',
        "\xB8\xCB" => '&#24235;',
        "\xB8\xCC" => '&#24359;',
        "\xB8\xCD" => '&#25144;',
        "\xB8\xCE" => '&#25925;',
        "\xB8\xCF" => '&#26543;',
        "\xB8\xD0" => '&#28246;',
        "\xB8\xD1" => '&#29392;',
        "\xB8\xD2" => '&#31946;',
        "\xB8\xD3" => '&#34996;',
        "\xB8\xD4" => '&#32929;',
        "\xB8\xD5" => '&#32993;',
        "\xB8\xD6" => '&#33776;',
        "\xB8\xD7" => '&#34382;',
        "\xB8\xD8" => '&#35463;',
        "\xB8\xD9" => '&#36328;',
        "\xB8\xDA" => '&#37431;',
        "\xB8\xDB" => '&#38599;',
        "\xB8\xDC" => '&#39015;',
        "\xB8\xDD" => '&#40723;',
        "\xB8\xDE" => '&#20116;',
        "\xB8\xDF" => '&#20114;',
        "\xB8\xE0" => '&#20237;',
        "\xB8\xE1" => '&#21320;',
        "\xB8\xE2" => '&#21577;',
        "\xB8\xE3" => '&#21566;',
        "\xB8\xE4" => '&#23087;',
        "\xB8\xE5" => '&#24460;',
        "\xB8\xE6" => '&#24481;',
        "\xB8\xE7" => '&#24735;',
        "\xB8\xE8" => '&#26791;',
        "\xB8\xE9" => '&#27278;',
        "\xB8\xEA" => '&#29786;',
        "\xB8\xEB" => '&#30849;',
        "\xB8\xEC" => '&#35486;',
        "\xB8\xED" => '&#35492;',
        "\xB8\xEE" => '&#35703;',
        "\xB8\xEF" => '&#37264;',
        "\xB8\xF0" => '&#20062;',
        "\xB8\xF1" => '&#39881;',
        "\xB8\xF2" => '&#20132;',
        "\xB8\xF3" => '&#20348;',
        "\xB8\xF4" => '&#20399;',
        "\xB8\xF5" => '&#20505;',
        "\xB8\xF6" => '&#20502;',
        "\xB8\xF7" => '&#20809;',
        "\xB8\xF8" => '&#20844;',
        "\xB8\xF9" => '&#21151;',
        "\xB8\xFA" => '&#21177;',
        "\xB8\xFB" => '&#21246;',
        "\xB8\xFC" => '&#21402;',
        "\xB8\xFD" => '&#21475;',
        "\xB8\xFE" => '&#21521;',
        "\xB9\xA1" => '&#21518;',
        "\xB9\xA2" => '&#21897;',
        "\xB9\xA3" => '&#22353;',
        "\xB9\xA4" => '&#22434;',
        "\xB9\xA5" => '&#22909;',
        "\xB9\xA6" => '&#23380;',
        "\xB9\xA7" => '&#23389;',
        "\xB9\xA8" => '&#23439;',
        "\xB9\xA9" => '&#24037;',
        "\xB9\xAA" => '&#24039;',
        "\xB9\xAB" => '&#24055;',
        "\xB9\xAC" => '&#24184;',
        "\xB9\xAD" => '&#24195;',
        "\xB9\xAE" => '&#24218;',
        "\xB9\xAF" => '&#24247;',
        "\xB9\xB0" => '&#24344;',
        "\xB9\xB1" => '&#24658;',
        "\xB9\xB2" => '&#24908;',
        "\xB9\xB3" => '&#25239;',
        "\xB9\xB4" => '&#25304;',
        "\xB9\xB5" => '&#25511;',
        "\xB9\xB6" => '&#25915;',
        "\xB9\xB7" => '&#26114;',
        "\xB9\xB8" => '&#26179;',
        "\xB9\xB9" => '&#26356;',
        "\xB9\xBA" => '&#26477;',
        "\xB9\xBB" => '&#26657;',
        "\xB9\xBC" => '&#26775;',
        "\xB9\xBD" => '&#27083;',
        "\xB9\xBE" => '&#27743;',
        "\xB9\xBF" => '&#27946;',
        "\xB9\xC0" => '&#28009;',
        "\xB9\xC1" => '&#28207;',
        "\xB9\xC2" => '&#28317;',
        "\xB9\xC3" => '&#30002;',
        "\xB9\xC4" => '&#30343;',
        "\xB9\xC5" => '&#30828;',
        "\xB9\xC6" => '&#31295;',
        "\xB9\xC7" => '&#31968;',
        "\xB9\xC8" => '&#32005;',
        "\xB9\xC9" => '&#32024;',
        "\xB9\xCA" => '&#32094;',
        "\xB9\xCB" => '&#32177;',
        "\xB9\xCC" => '&#32789;',
        "\xB9\xCD" => '&#32771;',
        "\xB9\xCE" => '&#32943;',
        "\xB9\xCF" => '&#32945;',
        "\xB9\xD0" => '&#33108;',
        "\xB9\xD1" => '&#33167;',
        "\xB9\xD2" => '&#33322;',
        "\xB9\xD3" => '&#33618;',
        "\xB9\xD4" => '&#34892;',
        "\xB9\xD5" => '&#34913;',
        "\xB9\xD6" => '&#35611;',
        "\xB9\xD7" => '&#36002;',
        "\xB9\xD8" => '&#36092;',
        "\xB9\xD9" => '&#37066;',
        "\xB9\xDA" => '&#37237;',
        "\xB9\xDB" => '&#37489;',
        "\xB9\xDC" => '&#30783;',
        "\xB9\xDD" => '&#37628;',
        "\xB9\xDE" => '&#38308;',
        "\xB9\xDF" => '&#38477;',
        "\xB9\xE0" => '&#38917;',
        "\xB9\xE1" => '&#39321;',
        "\xB9\xE2" => '&#39640;',
        "\xB9\xE3" => '&#40251;',
        "\xB9\xE4" => '&#21083;',
        "\xB9\xE5" => '&#21163;',
        "\xB9\xE6" => '&#21495;',
        "\xB9\xE7" => '&#21512;',
        "\xB9\xE8" => '&#22741;',
        "\xB9\xE9" => '&#25335;',
        "\xB9\xEA" => '&#28640;',
        "\xB9\xEB" => '&#35946;',
        "\xB9\xEC" => '&#36703;',
        "\xB9\xED" => '&#40633;',
        "\xB9\xEE" => '&#20811;',
        "\xB9\xEF" => '&#21051;',
        "\xB9\xF0" => '&#21578;',
        "\xB9\xF1" => '&#22269;',
        "\xB9\xF2" => '&#31296;',
        "\xB9\xF3" => '&#37239;',
        "\xB9\xF4" => '&#40288;',
        "\xB9\xF5" => '&#40658;',
        "\xB9\xF6" => '&#29508;',
        "\xB9\xF7" => '&#28425;',
        "\xB9\xF8" => '&#33136;',
        "\xB9\xF9" => '&#29969;',
        "\xB9\xFA" => '&#24573;',
        "\xB9\xFB" => '&#24794;',
        "\xB9\xFC" => '&#39592;',
        "\xB9\xFD" => '&#29403;',
        "\xB9\xFE" => '&#36796;',
        "\xBA\xA1" => '&#27492;',
        "\xBA\xA2" => '&#38915;',
        "\xBA\xA3" => '&#20170;',
        "\xBA\xA4" => '&#22256;',
        "\xBA\xA5" => '&#22372;',
        "\xBA\xA6" => '&#22718;',
        "\xBA\xA7" => '&#23130;',
        "\xBA\xA8" => '&#24680;',
        "\xBA\xA9" => '&#25031;',
        "\xBA\xAA" => '&#26127;',
        "\xBA\xAB" => '&#26118;',
        "\xBA\xAC" => '&#26681;',
        "\xBA\xAD" => '&#26801;',
        "\xBA\xAE" => '&#28151;',
        "\xBA\xAF" => '&#30165;',
        "\xBA\xB0" => '&#32058;',
        "\xBA\xB1" => '&#33390;',
        "\xBA\xB2" => '&#39746;',
        "\xBA\xB3" => '&#20123;',
        "\xBA\xB4" => '&#20304;',
        "\xBA\xB5" => '&#21449;',
        "\xBA\xB6" => '&#21766;',
        "\xBA\xB7" => '&#23919;',
        "\xBA\xB8" => '&#24038;',
        "\xBA\xB9" => '&#24046;',
        "\xBA\xBA" => '&#26619;',
        "\xBA\xBB" => '&#27801;',
        "\xBA\xBC" => '&#29811;',
        "\xBA\xBD" => '&#30722;',
        "\xBA\xBE" => '&#35408;',
        "\xBA\xBF" => '&#37782;',
        "\xBA\xC0" => '&#35039;',
        "\xBA\xC1" => '&#22352;',
        "\xBA\xC2" => '&#24231;',
        "\xBA\xC3" => '&#25387;',
        "\xBA\xC4" => '&#20661;',
        "\xBA\xC5" => '&#20652;',
        "\xBA\xC6" => '&#20877;',
        "\xBA\xC7" => '&#26368;',
        "\xBA\xC8" => '&#21705;',
        "\xBA\xC9" => '&#22622;',
        "\xBA\xCA" => '&#22971;',
        "\xBA\xCB" => '&#23472;',
        "\xBA\xCC" => '&#24425;',
        "\xBA\xCD" => '&#25165;',
        "\xBA\xCE" => '&#25505;',
        "\xBA\xCF" => '&#26685;',
        "\xBA\xD0" => '&#27507;',
        "\xBA\xD1" => '&#28168;',
        "\xBA\xD2" => '&#28797;',
        "\xBA\xD3" => '&#37319;',
        "\xBA\xD4" => '&#29312;',
        "\xBA\xD5" => '&#30741;',
        "\xBA\xD6" => '&#30758;',
        "\xBA\xD7" => '&#31085;',
        "\xBA\xD8" => '&#25998;',
        "\xBA\xD9" => '&#32048;',
        "\xBA\xDA" => '&#33756;',
        "\xBA\xDB" => '&#35009;',
        "\xBA\xDC" => '&#36617;',
        "\xBA\xDD" => '&#38555;',
        "\xBA\xDE" => '&#21092;',
        "\xBA\xDF" => '&#22312;',
        "\xBA\xE0" => '&#26448;',
        "\xBA\xE1" => '&#32618;',
        "\xBA\xE2" => '&#36001;',
        "\xBA\xE3" => '&#20916;',
        "\xBA\xE4" => '&#22338;',
        "\xBA\xE5" => '&#38442;',
        "\xBA\xE6" => '&#22586;',
        "\xBA\xE7" => '&#27018;',
        "\xBA\xE8" => '&#32948;',
        "\xBA\xE9" => '&#21682;',
        "\xBA\xEA" => '&#23822;',
        "\xBA\xEB" => '&#22524;',
        "\xBA\xEC" => '&#30869;',
        "\xBA\xED" => '&#40442;',
        "\xBA\xEE" => '&#20316;',
        "\xBA\xEF" => '&#21066;',
        "\xBA\xF0" => '&#21643;',
        "\xBA\xF1" => '&#25662;',
        "\xBA\xF2" => '&#26152;',
        "\xBA\xF3" => '&#26388;',
        "\xBA\xF4" => '&#26613;',
        "\xBA\xF5" => '&#31364;',
        "\xBA\xF6" => '&#31574;',
        "\xBA\xF7" => '&#32034;',
        "\xBA\xF8" => '&#37679;',
        "\xBA\xF9" => '&#26716;',
        "\xBA\xFA" => '&#39853;',
        "\xBA\xFB" => '&#31545;',
        "\xBA\xFC" => '&#21273;',
        "\xBA\xFD" => '&#20874;',
        "\xBA\xFE" => '&#21047;',
        "\xBB\xA1" => '&#23519;',
        "\xBB\xA2" => '&#25334;',
        "\xBB\xA3" => '&#25774;',
        "\xBB\xA4" => '&#25830;',
        "\xBB\xA5" => '&#26413;',
        "\xBB\xA6" => '&#27578;',
        "\xBB\xA7" => '&#34217;',
        "\xBB\xA8" => '&#38609;',
        "\xBB\xA9" => '&#30352;',
        "\xBB\xAA" => '&#39894;',
        "\xBB\xAB" => '&#25420;',
        "\xBB\xAC" => '&#37638;',
        "\xBB\xAD" => '&#39851;',
        "\xBB\xAE" => '&#30399;',
        "\xBB\xAF" => '&#26194;',
        "\xBB\xB0" => '&#19977;',
        "\xBB\xB1" => '&#20632;',
        "\xBB\xB2" => '&#21442;',
        "\xBB\xB3" => '&#23665;',
        "\xBB\xB4" => '&#24808;',
        "\xBB\xB5" => '&#25746;',
        "\xBB\xB6" => '&#25955;',
        "\xBB\xB7" => '&#26719;',
        "\xBB\xB8" => '&#29158;',
        "\xBB\xB9" => '&#29642;',
        "\xBB\xBA" => '&#29987;',
        "\xBB\xBB" => '&#31639;',
        "\xBB\xBC" => '&#32386;',
        "\xBB\xBD" => '&#34453;',
        "\xBB\xBE" => '&#35715;',
        "\xBB\xBF" => '&#36059;',
        "\xBB\xC0" => '&#37240;',
        "\xBB\xC1" => '&#39184;',
        "\xBB\xC2" => '&#26028;',
        "\xBB\xC3" => '&#26283;',
        "\xBB\xC4" => '&#27531;',
        "\xBB\xC5" => '&#20181;',
        "\xBB\xC6" => '&#20180;',
        "\xBB\xC7" => '&#20282;',
        "\xBB\xC8" => '&#20351;',
        "\xBB\xC9" => '&#21050;',
        "\xBB\xCA" => '&#21496;',
        "\xBB\xCB" => '&#21490;',
        "\xBB\xCC" => '&#21987;',
        "\xBB\xCD" => '&#22235;',
        "\xBB\xCE" => '&#22763;',
        "\xBB\xCF" => '&#22987;',
        "\xBB\xD0" => '&#22985;',
        "\xBB\xD1" => '&#23039;',
        "\xBB\xD2" => '&#23376;',
        "\xBB\xD3" => '&#23629;',
        "\xBB\xD4" => '&#24066;',
        "\xBB\xD5" => '&#24107;',
        "\xBB\xD6" => '&#24535;',
        "\xBB\xD7" => '&#24605;',
        "\xBB\xD8" => '&#25351;',
        "\xBB\xD9" => '&#25903;',
        "\xBB\xDA" => '&#23388;',
        "\xBB\xDB" => '&#26031;',
        "\xBB\xDC" => '&#26045;',
        "\xBB\xDD" => '&#26088;',
        "\xBB\xDE" => '&#26525;',
        "\xBB\xDF" => '&#27490;',
        "\xBB\xE0" => '&#27515;',
        "\xBB\xE1" => '&#27663;',
        "\xBB\xE2" => '&#29509;',
        "\xBB\xE3" => '&#31049;',
        "\xBB\xE4" => '&#31169;',
        "\xBB\xE5" => '&#31992;',
        "\xBB\xE6" => '&#32025;',
        "\xBB\xE7" => '&#32043;',
        "\xBB\xE8" => '&#32930;',
        "\xBB\xE9" => '&#33026;',
        "\xBB\xEA" => '&#33267;',
        "\xBB\xEB" => '&#35222;',
        "\xBB\xEC" => '&#35422;',
        "\xBB\xED" => '&#35433;',
        "\xBB\xEE" => '&#35430;',
        "\xBB\xEF" => '&#35468;',
        "\xBB\xF0" => '&#35566;',
        "\xBB\xF1" => '&#36039;',
        "\xBB\xF2" => '&#36060;',
        "\xBB\xF3" => '&#38604;',
        "\xBB\xF4" => '&#39164;',
        "\xBB\xF5" => '&#27503;',
        "\xBB\xF6" => '&#20107;',
        "\xBB\xF7" => '&#20284;',
        "\xBB\xF8" => '&#20365;',
        "\xBB\xF9" => '&#20816;',
        "\xBB\xFA" => '&#23383;',
        "\xBB\xFB" => '&#23546;',
        "\xBB\xFC" => '&#24904;',
        "\xBB\xFD" => '&#25345;',
        "\xBB\xFE" => '&#26178;',
        "\xBC\xA1" => '&#27425;',
        "\xBC\xA2" => '&#28363;',
        "\xBC\xA3" => '&#27835;',
        "\xBC\xA4" => '&#29246;',
        "\xBC\xA5" => '&#29885;',
        "\xBC\xA6" => '&#30164;',
        "\xBC\xA7" => '&#30913;',
        "\xBC\xA8" => '&#31034;',
        "\xBC\xA9" => '&#32780;',
        "\xBC\xAA" => '&#32819;',
        "\xBC\xAB" => '&#33258;',
        "\xBC\xAC" => '&#33940;',
        "\xBC\xAD" => '&#36766;',
        "\xBC\xAE" => '&#27728;',
        "\xBC\xAF" => '&#40575;',
        "\xBC\xB0" => '&#24335;',
        "\xBC\xB1" => '&#35672;',
        "\xBC\xB2" => '&#40235;',
        "\xBC\xB3" => '&#31482;',
        "\xBC\xB4" => '&#36600;',
        "\xBC\xB5" => '&#23437;',
        "\xBC\xB6" => '&#38635;',
        "\xBC\xB7" => '&#19971;',
        "\xBC\xB8" => '&#21489;',
        "\xBC\xB9" => '&#22519;',
        "\xBC\xBA" => '&#22833;',
        "\xBC\xBB" => '&#23241;',
        "\xBC\xBC" => '&#23460;',
        "\xBC\xBD" => '&#24713;',
        "\xBC\xBE" => '&#28287;',
        "\xBC\xBF" => '&#28422;',
        "\xBC\xC0" => '&#30142;',
        "\xBC\xC1" => '&#36074;',
        "\xBC\xC2" => '&#23455;',
        "\xBC\xC3" => '&#34048;',
        "\xBC\xC4" => '&#31712;',
        "\xBC\xC5" => '&#20594;',
        "\xBC\xC6" => '&#26612;',
        "\xBC\xC7" => '&#33437;',
        "\xBC\xC8" => '&#23649;',
        "\xBC\xC9" => '&#34122;',
        "\xBC\xCA" => '&#32286;',
        "\xBC\xCB" => '&#33294;',
        "\xBC\xCC" => '&#20889;',
        "\xBC\xCD" => '&#23556;',
        "\xBC\xCE" => '&#25448;',
        "\xBC\xCF" => '&#36198;',
        "\xBC\xD0" => '&#26012;',
        "\xBC\xD1" => '&#29038;',
        "\xBC\xD2" => '&#31038;',
        "\xBC\xD3" => '&#32023;',
        "\xBC\xD4" => '&#32773;',
        "\xBC\xD5" => '&#35613;',
        "\xBC\xD6" => '&#36554;',
        "\xBC\xD7" => '&#36974;',
        "\xBC\xD8" => '&#34503;',
        "\xBC\xD9" => '&#37034;',
        "\xBC\xDA" => '&#20511;',
        "\xBC\xDB" => '&#21242;',
        "\xBC\xDC" => '&#23610;',
        "\xBC\xDD" => '&#26451;',
        "\xBC\xDE" => '&#28796;',
        "\xBC\xDF" => '&#29237;',
        "\xBC\xE0" => '&#37196;',
        "\xBC\xE1" => '&#37320;',
        "\xBC\xE2" => '&#37675;',
        "\xBC\xE3" => '&#33509;',
        "\xBC\xE4" => '&#23490;',
        "\xBC\xE5" => '&#24369;',
        "\xBC\xE6" => '&#24825;',
        "\xBC\xE7" => '&#20027;',
        "\xBC\xE8" => '&#21462;',
        "\xBC\xE9" => '&#23432;',
        "\xBC\xEA" => '&#25163;',
        "\xBC\xEB" => '&#26417;',
        "\xBC\xEC" => '&#27530;',
        "\xBC\xED" => '&#29417;',
        "\xBC\xEE" => '&#29664;',
        "\xBC\xEF" => '&#31278;',
        "\xBC\xF0" => '&#33131;',
        "\xBC\xF1" => '&#36259;',
        "\xBC\xF2" => '&#37202;',
        "\xBC\xF3" => '&#39318;',
        "\xBC\xF4" => '&#20754;',
        "\xBC\xF5" => '&#21463;',
        "\xBC\xF6" => '&#21610;',
        "\xBC\xF7" => '&#23551;',
        "\xBC\xF8" => '&#25480;',
        "\xBC\xF9" => '&#27193;',
        "\xBC\xFA" => '&#32172;',
        "\xBC\xFB" => '&#38656;',
        "\xBC\xFC" => '&#22234;',
        "\xBC\xFD" => '&#21454;',
        "\xBC\xFE" => '&#21608;',
        "\xBD\xA1" => '&#23447;',
        "\xBD\xA2" => '&#23601;',
        "\xBD\xA3" => '&#24030;',
        "\xBD\xA4" => '&#20462;',
        "\xBD\xA5" => '&#24833;',
        "\xBD\xA6" => '&#25342;',
        "\xBD\xA7" => '&#27954;',
        "\xBD\xA8" => '&#31168;',
        "\xBD\xA9" => '&#31179;',
        "\xBD\xAA" => '&#32066;',
        "\xBD\xAB" => '&#32333;',
        "\xBD\xAC" => '&#32722;',
        "\xBD\xAD" => '&#33261;',
        "\xBD\xAE" => '&#33311;',
        "\xBD\xAF" => '&#33936;',
        "\xBD\xB0" => '&#34886;',
        "\xBD\xB1" => '&#35186;',
        "\xBD\xB2" => '&#35728;',
        "\xBD\xB3" => '&#36468;',
        "\xBD\xB4" => '&#36655;',
        "\xBD\xB5" => '&#36913;',
        "\xBD\xB6" => '&#37195;',
        "\xBD\xB7" => '&#37228;',
        "\xBD\xB8" => '&#38598;',
        "\xBD\xB9" => '&#37276;',
        "\xBD\xBA" => '&#20160;',
        "\xBD\xBB" => '&#20303;',
        "\xBD\xBC" => '&#20805;',
        "\xBD\xBD" => '&#21313;',
        "\xBD\xBE" => '&#24467;',
        "\xBD\xBF" => '&#25102;',
        "\xBD\xC0" => '&#26580;',
        "\xBD\xC1" => '&#27713;',
        "\xBD\xC2" => '&#28171;',
        "\xBD\xC3" => '&#29539;',
        "\xBD\xC4" => '&#32294;',
        "\xBD\xC5" => '&#37325;',
        "\xBD\xC6" => '&#37507;',
        "\xBD\xC7" => '&#21460;',
        "\xBD\xC8" => '&#22809;',
        "\xBD\xC9" => '&#23487;',
        "\xBD\xCA" => '&#28113;',
        "\xBD\xCB" => '&#31069;',
        "\xBD\xCC" => '&#32302;',
        "\xBD\xCD" => '&#31899;',
        "\xBD\xCE" => '&#22654;',
        "\xBD\xCF" => '&#29087;',
        "\xBD\xD0" => '&#20986;',
        "\xBD\xD1" => '&#34899;',
        "\xBD\xD2" => '&#36848;',
        "\xBD\xD3" => '&#20426;',
        "\xBD\xD4" => '&#23803;',
        "\xBD\xD5" => '&#26149;',
        "\xBD\xD6" => '&#30636;',
        "\xBD\xD7" => '&#31459;',
        "\xBD\xD8" => '&#33308;',
        "\xBD\xD9" => '&#39423;',
        "\xBD\xDA" => '&#20934;',
        "\xBD\xDB" => '&#24490;',
        "\xBD\xDC" => '&#26092;',
        "\xBD\xDD" => '&#26991;',
        "\xBD\xDE" => '&#27529;',
        "\xBD\xDF" => '&#28147;',
        "\xBD\xE0" => '&#28310;',
        "\xBD\xE1" => '&#28516;',
        "\xBD\xE2" => '&#30462;',
        "\xBD\xE3" => '&#32020;',
        "\xBD\xE4" => '&#24033;',
        "\xBD\xE5" => '&#36981;',
        "\xBD\xE6" => '&#37255;',
        "\xBD\xE7" => '&#38918;',
        "\xBD\xE8" => '&#20966;',
        "\xBD\xE9" => '&#21021;',
        "\xBD\xEA" => '&#25152;',
        "\xBD\xEB" => '&#26257;',
        "\xBD\xEC" => '&#26329;',
        "\xBD\xED" => '&#28186;',
        "\xBD\xEE" => '&#24246;',
        "\xBD\xEF" => '&#32210;',
        "\xBD\xF0" => '&#32626;',
        "\xBD\xF1" => '&#26360;',
        "\xBD\xF2" => '&#34223;',
        "\xBD\xF3" => '&#34295;',
        "\xBD\xF4" => '&#35576;',
        "\xBD\xF5" => '&#21161;',
        "\xBD\xF6" => '&#21465;',
        "\xBD\xF7" => '&#22899;',
        "\xBD\xF8" => '&#24207;',
        "\xBD\xF9" => '&#24464;',
        "\xBD\xFA" => '&#24661;',
        "\xBD\xFB" => '&#37604;',
        "\xBD\xFC" => '&#38500;',
        "\xBD\xFD" => '&#20663;',
        "\xBD\xFE" => '&#20767;',
        "\xBE\xA1" => '&#21213;',
        "\xBE\xA2" => '&#21280;',
        "\xBE\xA3" => '&#21319;',
        "\xBE\xA4" => '&#21484;',
        "\xBE\xA5" => '&#21736;',
        "\xBE\xA6" => '&#21830;',
        "\xBE\xA7" => '&#21809;',
        "\xBE\xA8" => '&#22039;',
        "\xBE\xA9" => '&#22888;',
        "\xBE\xAA" => '&#22974;',
        "\xBE\xAB" => '&#23100;',
        "\xBE\xAC" => '&#23477;',
        "\xBE\xAD" => '&#23558;',
        "\xBE\xAE" => '&#23567;',
        "\xBE\xAF" => '&#23569;',
        "\xBE\xB0" => '&#23578;',
        "\xBE\xB1" => '&#24196;',
        "\xBE\xB2" => '&#24202;',
        "\xBE\xB3" => '&#24288;',
        "\xBE\xB4" => '&#24432;',
        "\xBE\xB5" => '&#25215;',
        "\xBE\xB6" => '&#25220;',
        "\xBE\xB7" => '&#25307;',
        "\xBE\xB8" => '&#25484;',
        "\xBE\xB9" => '&#25463;',
        "\xBE\xBA" => '&#26119;',
        "\xBE\xBB" => '&#26124;',
        "\xBE\xBC" => '&#26157;',
        "\xBE\xBD" => '&#26230;',
        "\xBE\xBE" => '&#26494;',
        "\xBE\xBF" => '&#26786;',
        "\xBE\xC0" => '&#27167;',
        "\xBE\xC1" => '&#27189;',
        "\xBE\xC2" => '&#27836;',
        "\xBE\xC3" => '&#28040;',
        "\xBE\xC4" => '&#28169;',
        "\xBE\xC5" => '&#28248;',
        "\xBE\xC6" => '&#28988;',
        "\xBE\xC7" => '&#28966;',
        "\xBE\xC8" => '&#29031;',
        "\xBE\xC9" => '&#30151;',
        "\xBE\xCA" => '&#30465;',
        "\xBE\xCB" => '&#30813;',
        "\xBE\xCC" => '&#30977;',
        "\xBE\xCD" => '&#31077;',
        "\xBE\xCE" => '&#31216;',
        "\xBE\xCF" => '&#31456;',
        "\xBE\xD0" => '&#31505;',
        "\xBE\xD1" => '&#31911;',
        "\xBE\xD2" => '&#32057;',
        "\xBE\xD3" => '&#32918;',
        "\xBE\xD4" => '&#33750;',
        "\xBE\xD5" => '&#33931;',
        "\xBE\xD6" => '&#34121;',
        "\xBE\xD7" => '&#34909;',
        "\xBE\xD8" => '&#35059;',
        "\xBE\xD9" => '&#35359;',
        "\xBE\xDA" => '&#35388;',
        "\xBE\xDB" => '&#35412;',
        "\xBE\xDC" => '&#35443;',
        "\xBE\xDD" => '&#35937;',
        "\xBE\xDE" => '&#36062;',
        "\xBE\xDF" => '&#37284;',
        "\xBE\xE0" => '&#37478;',
        "\xBE\xE1" => '&#37758;',
        "\xBE\xE2" => '&#37912;',
        "\xBE\xE3" => '&#38556;',
        "\xBE\xE4" => '&#38808;',
        "\xBE\xE5" => '&#19978;',
        "\xBE\xE6" => '&#19976;',
        "\xBE\xE7" => '&#19998;',
        "\xBE\xE8" => '&#20055;',
        "\xBE\xE9" => '&#20887;',
        "\xBE\xEA" => '&#21104;',
        "\xBE\xEB" => '&#22478;',
        "\xBE\xEC" => '&#22580;',
        "\xBE\xED" => '&#22732;',
        "\xBE\xEE" => '&#23330;',
        "\xBE\xEF" => '&#24120;',
        "\xBE\xF0" => '&#24773;',
        "\xBE\xF1" => '&#25854;',
        "\xBE\xF2" => '&#26465;',
        "\xBE\xF3" => '&#26454;',
        "\xBE\xF4" => '&#27972;',
        "\xBE\xF5" => '&#29366;',
        "\xBE\xF6" => '&#30067;',
        "\xBE\xF7" => '&#31331;',
        "\xBE\xF8" => '&#33976;',
        "\xBE\xF9" => '&#35698;',
        "\xBE\xFA" => '&#37304;',
        "\xBE\xFB" => '&#37664;',
        "\xBE\xFC" => '&#22065;',
        "\xBE\xFD" => '&#22516;',
        "\xBE\xFE" => '&#39166;',
        "\xBF\xA1" => '&#25325;',
        "\xBF\xA2" => '&#26893;',
        "\xBF\xA3" => '&#27542;',
        "\xBF\xA4" => '&#29165;',
        "\xBF\xA5" => '&#32340;',
        "\xBF\xA6" => '&#32887;',
        "\xBF\xA7" => '&#33394;',
        "\xBF\xA8" => '&#35302;',
        "\xBF\xA9" => '&#39135;',
        "\xBF\xAA" => '&#34645;',
        "\xBF\xAB" => '&#36785;',
        "\xBF\xAC" => '&#23611;',
        "\xBF\xAD" => '&#20280;',
        "\xBF\xAE" => '&#20449;',
        "\xBF\xAF" => '&#20405;',
        "\xBF\xB0" => '&#21767;',
        "\xBF\xB1" => '&#23072;',
        "\xBF\xB2" => '&#23517;',
        "\xBF\xB3" => '&#23529;',
        "\xBF\xB4" => '&#24515;',
        "\xBF\xB5" => '&#24910;',
        "\xBF\xB6" => '&#25391;',
        "\xBF\xB7" => '&#26032;',
        "\xBF\xB8" => '&#26187;',
        "\xBF\xB9" => '&#26862;',
        "\xBF\xBA" => '&#27035;',
        "\xBF\xBB" => '&#28024;',
        "\xBF\xBC" => '&#28145;',
        "\xBF\xBD" => '&#30003;',
        "\xBF\xBE" => '&#30137;',
        "\xBF\xBF" => '&#30495;',
        "\xBF\xC0" => '&#31070;',
        "\xBF\xC1" => '&#31206;',
        "\xBF\xC2" => '&#32051;',
        "\xBF\xC3" => '&#33251;',
        "\xBF\xC4" => '&#33455;',
        "\xBF\xC5" => '&#34218;',
        "\xBF\xC6" => '&#35242;',
        "\xBF\xC7" => '&#35386;',
        "\xBF\xC8" => '&#36523;',
        "\xBF\xC9" => '&#36763;',
        "\xBF\xCA" => '&#36914;',
        "\xBF\xCB" => '&#37341;',
        "\xBF\xCC" => '&#38663;',
        "\xBF\xCD" => '&#20154;',
        "\xBF\xCE" => '&#20161;',
        "\xBF\xCF" => '&#20995;',
        "\xBF\xD0" => '&#22645;',
        "\xBF\xD1" => '&#22764;',
        "\xBF\xD2" => '&#23563;',
        "\xBF\xD3" => '&#29978;',
        "\xBF\xD4" => '&#23613;',
        "\xBF\xD5" => '&#33102;',
        "\xBF\xD6" => '&#35338;',
        "\xBF\xD7" => '&#36805;',
        "\xBF\xD8" => '&#38499;',
        "\xBF\xD9" => '&#38765;',
        "\xBF\xDA" => '&#31525;',
        "\xBF\xDB" => '&#35535;',
        "\xBF\xDC" => '&#38920;',
        "\xBF\xDD" => '&#37218;',
        "\xBF\xDE" => '&#22259;',
        "\xBF\xDF" => '&#21416;',
        "\xBF\xE0" => '&#36887;',
        "\xBF\xE1" => '&#21561;',
        "\xBF\xE2" => '&#22402;',
        "\xBF\xE3" => '&#24101;',
        "\xBF\xE4" => '&#25512;',
        "\xBF\xE5" => '&#27700;',
        "\xBF\xE6" => '&#28810;',
        "\xBF\xE7" => '&#30561;',
        "\xBF\xE8" => '&#31883;',
        "\xBF\xE9" => '&#32736;',
        "\xBF\xEA" => '&#34928;',
        "\xBF\xEB" => '&#36930;',
        "\xBF\xEC" => '&#37204;',
        "\xBF\xED" => '&#37648;',
        "\xBF\xEE" => '&#37656;',
        "\xBF\xEF" => '&#38543;',
        "\xBF\xF0" => '&#29790;',
        "\xBF\xF1" => '&#39620;',
        "\xBF\xF2" => '&#23815;',
        "\xBF\xF3" => '&#23913;',
        "\xBF\xF4" => '&#25968;',
        "\xBF\xF5" => '&#26530;',
        "\xBF\xF6" => '&#36264;',
        "\xBF\xF7" => '&#38619;',
        "\xBF\xF8" => '&#25454;',
        "\xBF\xF9" => '&#26441;',
        "\xBF\xFA" => '&#26905;',
        "\xBF\xFB" => '&#33733;',
        "\xBF\xFC" => '&#38935;',
        "\xBF\xFD" => '&#38592;',
        "\xBF\xFE" => '&#35070;',
        "\xC0\xA1" => '&#28548;',
        "\xC0\xA2" => '&#25722;',
        "\xC0\xA3" => '&#23544;',
        "\xC0\xA4" => '&#19990;',
        "\xC0\xA5" => '&#28716;',
        "\xC0\xA6" => '&#30045;',
        "\xC0\xA7" => '&#26159;',
        "\xC0\xA8" => '&#20932;',
        "\xC0\xA9" => '&#21046;',
        "\xC0\xAA" => '&#21218;',
        "\xC0\xAB" => '&#22995;',
        "\xC0\xAC" => '&#24449;',
        "\xC0\xAD" => '&#24615;',
        "\xC0\xAE" => '&#25104;',
        "\xC0\xAF" => '&#25919;',
        "\xC0\xB0" => '&#25972;',
        "\xC0\xB1" => '&#26143;',
        "\xC0\xB2" => '&#26228;',
        "\xC0\xB3" => '&#26866;',
        "\xC0\xB4" => '&#26646;',
        "\xC0\xB5" => '&#27491;',
        "\xC0\xB6" => '&#28165;',
        "\xC0\xB7" => '&#29298;',
        "\xC0\xB8" => '&#29983;',
        "\xC0\xB9" => '&#30427;',
        "\xC0\xBA" => '&#31934;',
        "\xC0\xBB" => '&#32854;',
        "\xC0\xBC" => '&#22768;',
        "\xC0\xBD" => '&#35069;',
        "\xC0\xBE" => '&#35199;',
        "\xC0\xBF" => '&#35488;',
        "\xC0\xC0" => '&#35475;',
        "\xC0\xC1" => '&#35531;',
        "\xC0\xC2" => '&#36893;',
        "\xC0\xC3" => '&#37266;',
        "\xC0\xC4" => '&#38738;',
        "\xC0\xC5" => '&#38745;',
        "\xC0\xC6" => '&#25993;',
        "\xC0\xC7" => '&#31246;',
        "\xC0\xC8" => '&#33030;',
        "\xC0\xC9" => '&#38587;',
        "\xC0\xCA" => '&#24109;',
        "\xC0\xCB" => '&#24796;',
        "\xC0\xCC" => '&#25114;',
        "\xC0\xCD" => '&#26021;',
        "\xC0\xCE" => '&#26132;',
        "\xC0\xCF" => '&#26512;',
        "\xC0\xD0" => '&#30707;',
        "\xC0\xD1" => '&#31309;',
        "\xC0\xD2" => '&#31821;',
        "\xC0\xD3" => '&#32318;',
        "\xC0\xD4" => '&#33034;',
        "\xC0\xD5" => '&#36012;',
        "\xC0\xD6" => '&#36196;',
        "\xC0\xD7" => '&#36321;',
        "\xC0\xD8" => '&#36447;',
        "\xC0\xD9" => '&#30889;',
        "\xC0\xDA" => '&#20999;',
        "\xC0\xDB" => '&#25305;',
        "\xC0\xDC" => '&#25509;',
        "\xC0\xDD" => '&#25666;',
        "\xC0\xDE" => '&#25240;',
        "\xC0\xDF" => '&#35373;',
        "\xC0\xE0" => '&#31363;',
        "\xC0\xE1" => '&#31680;',
        "\xC0\xE2" => '&#35500;',
        "\xC0\xE3" => '&#38634;',
        "\xC0\xE4" => '&#32118;',
        "\xC0\xE5" => '&#33292;',
        "\xC0\xE6" => '&#34633;',
        "\xC0\xE7" => '&#20185;',
        "\xC0\xE8" => '&#20808;',
        "\xC0\xE9" => '&#21315;',
        "\xC0\xEA" => '&#21344;',
        "\xC0\xEB" => '&#23459;',
        "\xC0\xEC" => '&#23554;',
        "\xC0\xED" => '&#23574;',
        "\xC0\xEE" => '&#24029;',
        "\xC0\xEF" => '&#25126;',
        "\xC0\xF0" => '&#25159;',
        "\xC0\xF1" => '&#25776;',
        "\xC0\xF2" => '&#26643;',
        "\xC0\xF3" => '&#26676;',
        "\xC0\xF4" => '&#27849;',
        "\xC0\xF5" => '&#27973;',
        "\xC0\xF6" => '&#27927;',
        "\xC0\xF7" => '&#26579;',
        "\xC0\xF8" => '&#28508;',
        "\xC0\xF9" => '&#29006;',
        "\xC0\xFA" => '&#29053;',
        "\xC0\xFB" => '&#26059;',
        "\xC0\xFC" => '&#31359;',
        "\xC0\xFD" => '&#31661;',
        "\xC0\xFE" => '&#32218;',
        "\xC1\xA1" => '&#32330;',
        "\xC1\xA2" => '&#32680;',
        "\xC1\xA3" => '&#33146;',
        "\xC1\xA4" => '&#33307;',
        "\xC1\xA5" => '&#33337;',
        "\xC1\xA6" => '&#34214;',
        "\xC1\xA7" => '&#35438;',
        "\xC1\xA8" => '&#36046;',
        "\xC1\xA9" => '&#36341;',
        "\xC1\xAA" => '&#36984;',
        "\xC1\xAB" => '&#36983;',
        "\xC1\xAC" => '&#37549;',
        "\xC1\xAD" => '&#37521;',
        "\xC1\xAE" => '&#38275;',
        "\xC1\xAF" => '&#39854;',
        "\xC1\xB0" => '&#21069;',
        "\xC1\xB1" => '&#21892;',
        "\xC1\xB2" => '&#28472;',
        "\xC1\xB3" => '&#28982;',
        "\xC1\xB4" => '&#20840;',
        "\xC1\xB5" => '&#31109;',
        "\xC1\xB6" => '&#32341;',
        "\xC1\xB7" => '&#33203;',
        "\xC1\xB8" => '&#31950;',
        "\xC1\xB9" => '&#22092;',
        "\xC1\xBA" => '&#22609;',
        "\xC1\xBB" => '&#23720;',
        "\xC1\xBC" => '&#25514;',
        "\xC1\xBD" => '&#26366;',
        "\xC1\xBE" => '&#26365;',
        "\xC1\xBF" => '&#26970;',
        "\xC1\xC0" => '&#29401;',
        "\xC1\xC1" => '&#30095;',
        "\xC1\xC2" => '&#30094;',
        "\xC1\xC3" => '&#30990;',
        "\xC1\xC4" => '&#31062;',
        "\xC1\xC5" => '&#31199;',
        "\xC1\xC6" => '&#31895;',
        "\xC1\xC7" => '&#32032;',
        "\xC1\xC8" => '&#32068;',
        "\xC1\xC9" => '&#34311;',
        "\xC1\xCA" => '&#35380;',
        "\xC1\xCB" => '&#38459;',
        "\xC1\xCC" => '&#36961;',
        "\xC1\xCD" => '&#40736;',
        "\xC1\xCE" => '&#20711;',
        "\xC1\xCF" => '&#21109;',
        "\xC1\xD0" => '&#21452;',
        "\xC1\xD1" => '&#21474;',
        "\xC1\xD2" => '&#20489;',
        "\xC1\xD3" => '&#21930;',
        "\xC1\xD4" => '&#22766;',
        "\xC1\xD5" => '&#22863;',
        "\xC1\xD6" => '&#29245;',
        "\xC1\xD7" => '&#23435;',
        "\xC1\xD8" => '&#23652;',
        "\xC1\xD9" => '&#21277;',
        "\xC1\xDA" => '&#24803;',
        "\xC1\xDB" => '&#24819;',
        "\xC1\xDC" => '&#25436;',
        "\xC1\xDD" => '&#25475;',
        "\xC1\xDE" => '&#25407;',
        "\xC1\xDF" => '&#25531;',
        "\xC1\xE0" => '&#25805;',
        "\xC1\xE1" => '&#26089;',
        "\xC1\xE2" => '&#26361;',
        "\xC1\xE3" => '&#24035;',
        "\xC1\xE4" => '&#27085;',
        "\xC1\xE5" => '&#27133;',
        "\xC1\xE6" => '&#28437;',
        "\xC1\xE7" => '&#29157;',
        "\xC1\xE8" => '&#20105;',
        "\xC1\xE9" => '&#30185;',
        "\xC1\xEA" => '&#30456;',
        "\xC1\xEB" => '&#31379;',
        "\xC1\xEC" => '&#31967;',
        "\xC1\xED" => '&#32207;',
        "\xC1\xEE" => '&#32156;',
        "\xC1\xEF" => '&#32865;',
        "\xC1\xF0" => '&#33609;',
        "\xC1\xF1" => '&#33624;',
        "\xC1\xF2" => '&#33900;',
        "\xC1\xF3" => '&#33980;',
        "\xC1\xF4" => '&#34299;',
        "\xC1\xF5" => '&#35013;',
        "\xC1\xF6" => '&#36208;',
        "\xC1\xF7" => '&#36865;',
        "\xC1\xF8" => '&#36973;',
        "\xC1\xF9" => '&#37783;',
        "\xC1\xFA" => '&#38684;',
        "\xC1\xFB" => '&#39442;',
        "\xC1\xFC" => '&#20687;',
        "\xC1\xFD" => '&#22679;',
        "\xC1\xFE" => '&#24974;',
        "\xC2\xA1" => '&#33235;',
        "\xC2\xA2" => '&#34101;',
        "\xC2\xA3" => '&#36104;',
        "\xC2\xA4" => '&#36896;',
        "\xC2\xA5" => '&#20419;',
        "\xC2\xA6" => '&#20596;',
        "\xC2\xA7" => '&#21063;',
        "\xC2\xA8" => '&#21363;',
        "\xC2\xA9" => '&#24687;',
        "\xC2\xAA" => '&#25417;',
        "\xC2\xAB" => '&#26463;',
        "\xC2\xAC" => '&#28204;',
        "\xC2\xAD" => '&#36275;',
        "\xC2\xAE" => '&#36895;',
        "\xC2\xAF" => '&#20439;',
        "\xC2\xB0" => '&#23646;',
        "\xC2\xB1" => '&#36042;',
        "\xC2\xB2" => '&#26063;',
        "\xC2\xB3" => '&#32154;',
        "\xC2\xB4" => '&#21330;',
        "\xC2\xB5" => '&#34966;',
        "\xC2\xB6" => '&#20854;',
        "\xC2\xB7" => '&#25539;',
        "\xC2\xB8" => '&#23384;',
        "\xC2\xB9" => '&#23403;',
        "\xC2\xBA" => '&#23562;',
        "\xC2\xBB" => '&#25613;',
        "\xC2\xBC" => '&#26449;',
        "\xC2\xBD" => '&#36956;',
        "\xC2\xBE" => '&#20182;',
        "\xC2\xBF" => '&#22810;',
        "\xC2\xC0" => '&#22826;',
        "\xC2\xC1" => '&#27760;',
        "\xC2\xC2" => '&#35409;',
        "\xC2\xC3" => '&#21822;',
        "\xC2\xC4" => '&#22549;',
        "\xC2\xC5" => '&#22949;',
        "\xC2\xC6" => '&#24816;',
        "\xC2\xC7" => '&#25171;',
        "\xC2\xC8" => '&#26561;',
        "\xC2\xC9" => '&#33333;',
        "\xC2\xCA" => '&#26965;',
        "\xC2\xCB" => '&#38464;',
        "\xC2\xCC" => '&#39364;',
        "\xC2\xCD" => '&#39464;',
        "\xC2\xCE" => '&#20307;',
        "\xC2\xCF" => '&#22534;',
        "\xC2\xD0" => '&#23550;',
        "\xC2\xD1" => '&#32784;',
        "\xC2\xD2" => '&#23729;',
        "\xC2\xD3" => '&#24111;',
        "\xC2\xD4" => '&#24453;',
        "\xC2\xD5" => '&#24608;',
        "\xC2\xD6" => '&#24907;',
        "\xC2\xD7" => '&#25140;',
        "\xC2\xD8" => '&#26367;',
        "\xC2\xD9" => '&#27888;',
        "\xC2\xDA" => '&#28382;',
        "\xC2\xDB" => '&#32974;',
        "\xC2\xDC" => '&#33151;',
        "\xC2\xDD" => '&#33492;',
        "\xC2\xDE" => '&#34955;',
        "\xC2\xDF" => '&#36024;',
        "\xC2\xE0" => '&#36864;',
        "\xC2\xE1" => '&#36910;',
        "\xC2\xE2" => '&#38538;',
        "\xC2\xE3" => '&#40667;',
        "\xC2\xE4" => '&#39899;',
        "\xC2\xE5" => '&#20195;',
        "\xC2\xE6" => '&#21488;',
        "\xC2\xE7" => '&#22823;',
        "\xC2\xE8" => '&#31532;',
        "\xC2\xE9" => '&#37261;',
        "\xC2\xEA" => '&#38988;',
        "\xC2\xEB" => '&#40441;',
        "\xC2\xEC" => '&#28381;',
        "\xC2\xED" => '&#28711;',
        "\xC2\xEE" => '&#21331;',
        "\xC2\xEF" => '&#21828;',
        "\xC2\xF0" => '&#23429;',
        "\xC2\xF1" => '&#25176;',
        "\xC2\xF2" => '&#25246;',
        "\xC2\xF3" => '&#25299;',
        "\xC2\xF4" => '&#27810;',
        "\xC2\xF5" => '&#28655;',
        "\xC2\xF6" => '&#29730;',
        "\xC2\xF7" => '&#35351;',
        "\xC2\xF8" => '&#37944;',
        "\xC2\xF9" => '&#28609;',
        "\xC2\xFA" => '&#35582;',
        "\xC2\xFB" => '&#33592;',
        "\xC2\xFC" => '&#20967;',
        "\xC2\xFD" => '&#34552;',
        "\xC2\xFE" => '&#21482;',
        "\xC3\xA1" => '&#21481;',
        "\xC3\xA2" => '&#20294;',
        "\xC3\xA3" => '&#36948;',
        "\xC3\xA4" => '&#36784;',
        "\xC3\xA5" => '&#22890;',
        "\xC3\xA6" => '&#33073;',
        "\xC3\xA7" => '&#24061;',
        "\xC3\xA8" => '&#31466;',
        "\xC3\xA9" => '&#36799;',
        "\xC3\xAA" => '&#26842;',
        "\xC3\xAB" => '&#35895;',
        "\xC3\xAC" => '&#29432;',
        "\xC3\xAD" => '&#40008;',
        "\xC3\xAE" => '&#27197;',
        "\xC3\xAF" => '&#35504;',
        "\xC3\xB0" => '&#20025;',
        "\xC3\xB1" => '&#21336;',
        "\xC3\xB2" => '&#22022;',
        "\xC3\xB3" => '&#22374;',
        "\xC3\xB4" => '&#25285;',
        "\xC3\xB5" => '&#25506;',
        "\xC3\xB6" => '&#26086;',
        "\xC3\xB7" => '&#27470;',
        "\xC3\xB8" => '&#28129;',
        "\xC3\xB9" => '&#28251;',
        "\xC3\xBA" => '&#28845;',
        "\xC3\xBB" => '&#30701;',
        "\xC3\xBC" => '&#31471;',
        "\xC3\xBD" => '&#31658;',
        "\xC3\xBE" => '&#32187;',
        "\xC3\xBF" => '&#32829;',
        "\xC3\xC0" => '&#32966;',
        "\xC3\xC1" => '&#34507;',
        "\xC3\xC2" => '&#35477;',
        "\xC3\xC3" => '&#37723;',
        "\xC3\xC4" => '&#22243;',
        "\xC3\xC5" => '&#22727;',
        "\xC3\xC6" => '&#24382;',
        "\xC3\xC7" => '&#26029;',
        "\xC3\xC8" => '&#26262;',
        "\xC3\xC9" => '&#27264;',
        "\xC3\xCA" => '&#27573;',
        "\xC3\xCB" => '&#30007;',
        "\xC3\xCC" => '&#35527;',
        "\xC3\xCD" => '&#20516;',
        "\xC3\xCE" => '&#30693;',
        "\xC3\xCF" => '&#22320;',
        "\xC3\xD0" => '&#24347;',
        "\xC3\xD1" => '&#24677;',
        "\xC3\xD2" => '&#26234;',
        "\xC3\xD3" => '&#27744;',
        "\xC3\xD4" => '&#30196;',
        "\xC3\xD5" => '&#31258;',
        "\xC3\xD6" => '&#32622;',
        "\xC3\xD7" => '&#33268;',
        "\xC3\xD8" => '&#34584;',
        "\xC3\xD9" => '&#36933;',
        "\xC3\xDA" => '&#39347;',
        "\xC3\xDB" => '&#31689;',
        "\xC3\xDC" => '&#30044;',
        "\xC3\xDD" => '&#31481;',
        "\xC3\xDE" => '&#31569;',
        "\xC3\xDF" => '&#33988;',
        "\xC3\xE0" => '&#36880;',
        "\xC3\xE1" => '&#31209;',
        "\xC3\xE2" => '&#31378;',
        "\xC3\xE3" => '&#33590;',
        "\xC3\xE4" => '&#23265;',
        "\xC3\xE5" => '&#30528;',
        "\xC3\xE6" => '&#20013;',
        "\xC3\xE7" => '&#20210;',
        "\xC3\xE8" => '&#23449;',
        "\xC3\xE9" => '&#24544;',
        "\xC3\xEA" => '&#25277;',
        "\xC3\xEB" => '&#26172;',
        "\xC3\xEC" => '&#26609;',
        "\xC3\xED" => '&#27880;',
        "\xC3\xEE" => '&#34411;',
        "\xC3\xEF" => '&#34935;',
        "\xC3\xF0" => '&#35387;',
        "\xC3\xF1" => '&#37198;',
        "\xC3\xF2" => '&#37619;',
        "\xC3\xF3" => '&#39376;',
        "\xC3\xF4" => '&#27159;',
        "\xC3\xF5" => '&#28710;',
        "\xC3\xF6" => '&#29482;',
        "\xC3\xF7" => '&#33511;',
        "\xC3\xF8" => '&#33879;',
        "\xC3\xF9" => '&#36015;',
        "\xC3\xFA" => '&#19969;',
        "\xC3\xFB" => '&#20806;',
        "\xC3\xFC" => '&#20939;',
        "\xC3\xFD" => '&#21899;',
        "\xC3\xFE" => '&#23541;',
        "\xC4\xA1" => '&#24086;',
        "\xC4\xA2" => '&#24115;',
        "\xC4\xA3" => '&#24193;',
        "\xC4\xA4" => '&#24340;',
        "\xC4\xA5" => '&#24373;',
        "\xC4\xA6" => '&#24427;',
        "\xC4\xA7" => '&#24500;',
        "\xC4\xA8" => '&#25074;',
        "\xC4\xA9" => '&#25361;',
        "\xC4\xAA" => '&#26274;',
        "\xC4\xAB" => '&#26397;',
        "\xC4\xAC" => '&#28526;',
        "\xC4\xAD" => '&#29266;',
        "\xC4\xAE" => '&#30010;',
        "\xC4\xAF" => '&#30522;',
        "\xC4\xB0" => '&#32884;',
        "\xC4\xB1" => '&#33081;',
        "\xC4\xB2" => '&#33144;',
        "\xC4\xB3" => '&#34678;',
        "\xC4\xB4" => '&#35519;',
        "\xC4\xB5" => '&#35548;',
        "\xC4\xB6" => '&#36229;',
        "\xC4\xB7" => '&#36339;',
        "\xC4\xB8" => '&#37530;',
        "\xC4\xB9" => '&#38263;',
        "\xC4\xBA" => '&#38914;',
        "\xC4\xBB" => '&#40165;',
        "\xC4\xBC" => '&#21189;',
        "\xC4\xBD" => '&#25431;',
        "\xC4\xBE" => '&#30452;',
        "\xC4\xBF" => '&#26389;',
        "\xC4\xC0" => '&#27784;',
        "\xC4\xC1" => '&#29645;',
        "\xC4\xC2" => '&#36035;',
        "\xC4\xC3" => '&#37806;',
        "\xC4\xC4" => '&#38515;',
        "\xC4\xC5" => '&#27941;',
        "\xC4\xC6" => '&#22684;',
        "\xC4\xC7" => '&#26894;',
        "\xC4\xC8" => '&#27084;',
        "\xC4\xC9" => '&#36861;',
        "\xC4\xCA" => '&#37786;',
        "\xC4\xCB" => '&#30171;',
        "\xC4\xCC" => '&#36890;',
        "\xC4\xCD" => '&#22618;',
        "\xC4\xCE" => '&#26626;',
        "\xC4\xCF" => '&#25524;',
        "\xC4\xD0" => '&#27131;',
        "\xC4\xD1" => '&#20291;',
        "\xC4\xD2" => '&#28460;',
        "\xC4\xD3" => '&#26584;',
        "\xC4\xD4" => '&#36795;',
        "\xC4\xD5" => '&#34086;',
        "\xC4\xD6" => '&#32180;',
        "\xC4\xD7" => '&#37716;',
        "\xC4\xD8" => '&#26943;',
        "\xC4\xD9" => '&#28528;',
        "\xC4\xDA" => '&#22378;',
        "\xC4\xDB" => '&#22775;',
        "\xC4\xDC" => '&#23340;',
        "\xC4\xDD" => '&#32044;',
        "\xC4\xDE" => '&#29226;',
        "\xC4\xDF" => '&#21514;',
        "\xC4\xE0" => '&#37347;',
        "\xC4\xE1" => '&#40372;',
        "\xC4\xE2" => '&#20141;',
        "\xC4\xE3" => '&#20302;',
        "\xC4\xE4" => '&#20572;',
        "\xC4\xE5" => '&#20597;',
        "\xC4\xE6" => '&#21059;',
        "\xC4\xE7" => '&#35998;',
        "\xC4\xE8" => '&#21576;',
        "\xC4\xE9" => '&#22564;',
        "\xC4\xEA" => '&#23450;',
        "\xC4\xEB" => '&#24093;',
        "\xC4\xEC" => '&#24213;',
        "\xC4\xED" => '&#24237;',
        "\xC4\xEE" => '&#24311;',
        "\xC4\xEF" => '&#24351;',
        "\xC4\xF0" => '&#24716;',
        "\xC4\xF1" => '&#25269;',
        "\xC4\xF2" => '&#25402;',
        "\xC4\xF3" => '&#25552;',
        "\xC4\xF4" => '&#26799;',
        "\xC4\xF5" => '&#27712;',
        "\xC4\xF6" => '&#30855;',
        "\xC4\xF7" => '&#31118;',
        "\xC4\xF8" => '&#31243;',
        "\xC4\xF9" => '&#32224;',
        "\xC4\xFA" => '&#33351;',
        "\xC4\xFB" => '&#35330;',
        "\xC4\xFC" => '&#35558;',
        "\xC4\xFD" => '&#36420;',
        "\xC4\xFE" => '&#36883;',
        "\xC5\xA1" => '&#37048;',
        "\xC5\xA2" => '&#37165;',
        "\xC5\xA3" => '&#37336;',
        "\xC5\xA4" => '&#40718;',
        "\xC5\xA5" => '&#27877;',
        "\xC5\xA6" => '&#25688;',
        "\xC5\xA7" => '&#25826;',
        "\xC5\xA8" => '&#25973;',
        "\xC5\xA9" => '&#28404;',
        "\xC5\xAA" => '&#30340;',
        "\xC5\xAB" => '&#31515;',
        "\xC5\xAC" => '&#36969;',
        "\xC5\xAD" => '&#37841;',
        "\xC5\xAE" => '&#28346;',
        "\xC5\xAF" => '&#21746;',
        "\xC5\xB0" => '&#24505;',
        "\xC5\xB1" => '&#25764;',
        "\xC5\xB2" => '&#36685;',
        "\xC5\xB3" => '&#36845;',
        "\xC5\xB4" => '&#37444;',
        "\xC5\xB5" => '&#20856;',
        "\xC5\xB6" => '&#22635;',
        "\xC5\xB7" => '&#22825;',
        "\xC5\xB8" => '&#23637;',
        "\xC5\xB9" => '&#24215;',
        "\xC5\xBA" => '&#28155;',
        "\xC5\xBB" => '&#32399;',
        "\xC5\xBC" => '&#29980;',
        "\xC5\xBD" => '&#36028;',
        "\xC5\xBE" => '&#36578;',
        "\xC5\xBF" => '&#39003;',
        "\xC5\xC0" => '&#28857;',
        "\xC5\xC1" => '&#20253;',
        "\xC5\xC2" => '&#27583;',
        "\xC5\xC3" => '&#28593;',
        "\xC5\xC4" => '&#30000;',
        "\xC5\xC5" => '&#38651;',
        "\xC5\xC6" => '&#20814;',
        "\xC5\xC7" => '&#21520;',
        "\xC5\xC8" => '&#22581;',
        "\xC5\xC9" => '&#22615;',
        "\xC5\xCA" => '&#22956;',
        "\xC5\xCB" => '&#23648;',
        "\xC5\xCC" => '&#24466;',
        "\xC5\xCD" => '&#26007;',
        "\xC5\xCE" => '&#26460;',
        "\xC5\xCF" => '&#28193;',
        "\xC5\xD0" => '&#30331;',
        "\xC5\xD1" => '&#33759;',
        "\xC5\xD2" => '&#36077;',
        "\xC5\xD3" => '&#36884;',
        "\xC5\xD4" => '&#37117;',
        "\xC5\xD5" => '&#37709;',
        "\xC5\xD6" => '&#30757;',
        "\xC5\xD7" => '&#30778;',
        "\xC5\xD8" => '&#21162;',
        "\xC5\xD9" => '&#24230;',
        "\xC5\xDA" => '&#22303;',
        "\xC5\xDB" => '&#22900;',
        "\xC5\xDC" => '&#24594;',
        "\xC5\xDD" => '&#20498;',
        "\xC5\xDE" => '&#20826;',
        "\xC5\xDF" => '&#20908;',
        "\xC5\xE0" => '&#20941;',
        "\xC5\xE1" => '&#20992;',
        "\xC5\xE2" => '&#21776;',
        "\xC5\xE3" => '&#22612;',
        "\xC5\xE4" => '&#22616;',
        "\xC5\xE5" => '&#22871;',
        "\xC5\xE6" => '&#23445;',
        "\xC5\xE7" => '&#23798;',
        "\xC5\xE8" => '&#23947;',
        "\xC5\xE9" => '&#24764;',
        "\xC5\xEA" => '&#25237;',
        "\xC5\xEB" => '&#25645;',
        "\xC5\xEC" => '&#26481;',
        "\xC5\xED" => '&#26691;',
        "\xC5\xEE" => '&#26812;',
        "\xC5\xEF" => '&#26847;',
        "\xC5\xF0" => '&#30423;',
        "\xC5\xF1" => '&#28120;',
        "\xC5\xF2" => '&#28271;',
        "\xC5\xF3" => '&#28059;',
        "\xC5\xF4" => '&#28783;',
        "\xC5\xF5" => '&#29128;',
        "\xC5\xF6" => '&#24403;',
        "\xC5\xF7" => '&#30168;',
        "\xC5\xF8" => '&#31095;',
        "\xC5\xF9" => '&#31561;',
        "\xC5\xFA" => '&#31572;',
        "\xC5\xFB" => '&#31570;',
        "\xC5\xFC" => '&#31958;',
        "\xC5\xFD" => '&#32113;',
        "\xC5\xFE" => '&#21040;',
        "\xC6\xA1" => '&#33891;',
        "\xC6\xA2" => '&#34153;',
        "\xC6\xA3" => '&#34276;',
        "\xC6\xA4" => '&#35342;',
        "\xC6\xA5" => '&#35588;',
        "\xC6\xA6" => '&#35910;',
        "\xC6\xA7" => '&#36367;',
        "\xC6\xA8" => '&#36867;',
        "\xC6\xA9" => '&#36879;',
        "\xC6\xAA" => '&#37913;',
        "\xC6\xAB" => '&#38518;',
        "\xC6\xAC" => '&#38957;',
        "\xC6\xAD" => '&#39472;',
        "\xC6\xAE" => '&#38360;',
        "\xC6\xAF" => '&#20685;',
        "\xC6\xB0" => '&#21205;',
        "\xC6\xB1" => '&#21516;',
        "\xC6\xB2" => '&#22530;',
        "\xC6\xB3" => '&#23566;',
        "\xC6\xB4" => '&#24999;',
        "\xC6\xB5" => '&#25758;',
        "\xC6\xB6" => '&#27934;',
        "\xC6\xB7" => '&#30643;',
        "\xC6\xB8" => '&#31461;',
        "\xC6\xB9" => '&#33012;',
        "\xC6\xBA" => '&#33796;',
        "\xC6\xBB" => '&#36947;',
        "\xC6\xBC" => '&#37509;',
        "\xC6\xBD" => '&#23776;',
        "\xC6\xBE" => '&#40199;',
        "\xC6\xBF" => '&#21311;',
        "\xC6\xC0" => '&#24471;',
        "\xC6\xC1" => '&#24499;',
        "\xC6\xC2" => '&#28060;',
        "\xC6\xC3" => '&#29305;',
        "\xC6\xC4" => '&#30563;',
        "\xC6\xC5" => '&#31167;',
        "\xC6\xC6" => '&#31716;',
        "\xC6\xC7" => '&#27602;',
        "\xC6\xC8" => '&#29420;',
        "\xC6\xC9" => '&#35501;',
        "\xC6\xCA" => '&#26627;',
        "\xC6\xCB" => '&#27233;',
        "\xC6\xCC" => '&#20984;',
        "\xC6\xCD" => '&#31361;',
        "\xC6\xCE" => '&#26932;',
        "\xC6\xCF" => '&#23626;',
        "\xC6\xD0" => '&#40182;',
        "\xC6\xD1" => '&#33515;',
        "\xC6\xD2" => '&#23493;',
        "\xC6\xD3" => '&#37193;',
        "\xC6\xD4" => '&#28702;',
        "\xC6\xD5" => '&#22136;',
        "\xC6\xD6" => '&#23663;',
        "\xC6\xD7" => '&#24775;',
        "\xC6\xD8" => '&#25958;',
        "\xC6\xD9" => '&#27788;',
        "\xC6\xDA" => '&#35930;',
        "\xC6\xDB" => '&#36929;',
        "\xC6\xDC" => '&#38931;',
        "\xC6\xDD" => '&#21585;',
        "\xC6\xDE" => '&#26311;',
        "\xC6\xDF" => '&#37389;',
        "\xC6\xE0" => '&#22856;',
        "\xC6\xE1" => '&#37027;',
        "\xC6\xE2" => '&#20869;',
        "\xC6\xE3" => '&#20045;',
        "\xC6\xE4" => '&#20970;',
        "\xC6\xE5" => '&#34201;',
        "\xC6\xE6" => '&#35598;',
        "\xC6\xE7" => '&#28760;',
        "\xC6\xE8" => '&#25466;',
        "\xC6\xE9" => '&#37707;',
        "\xC6\xEA" => '&#26978;',
        "\xC6\xEB" => '&#39348;',
        "\xC6\xEC" => '&#32260;',
        "\xC6\xED" => '&#30071;',
        "\xC6\xEE" => '&#21335;',
        "\xC6\xEF" => '&#26976;',
        "\xC6\xF0" => '&#36575;',
        "\xC6\xF1" => '&#38627;',
        "\xC6\xF2" => '&#27741;',
        "\xC6\xF3" => '&#20108;',
        "\xC6\xF4" => '&#23612;',
        "\xC6\xF5" => '&#24336;',
        "\xC6\xF6" => '&#36841;',
        "\xC6\xF7" => '&#21250;',
        "\xC6\xF8" => '&#36049;',
        "\xC6\xF9" => '&#32905;',
        "\xC6\xFA" => '&#34425;',
        "\xC6\xFB" => '&#24319;',
        "\xC6\xFC" => '&#26085;',
        "\xC6\xFD" => '&#20083;',
        "\xC6\xFE" => '&#20837;',
        "\xC7\xA1" => '&#22914;',
        "\xC7\xA2" => '&#23615;',
        "\xC7\xA3" => '&#38894;',
        "\xC7\xA4" => '&#20219;',
        "\xC7\xA5" => '&#22922;',
        "\xC7\xA6" => '&#24525;',
        "\xC7\xA7" => '&#35469;',
        "\xC7\xA8" => '&#28641;',
        "\xC7\xA9" => '&#31152;',
        "\xC7\xAA" => '&#31074;',
        "\xC7\xAB" => '&#23527;',
        "\xC7\xAC" => '&#33905;',
        "\xC7\xAD" => '&#29483;',
        "\xC7\xAE" => '&#29105;',
        "\xC7\xAF" => '&#24180;',
        "\xC7\xB0" => '&#24565;',
        "\xC7\xB1" => '&#25467;',
        "\xC7\xB2" => '&#25754;',
        "\xC7\xB3" => '&#29123;',
        "\xC7\xB4" => '&#31896;',
        "\xC7\xB5" => '&#20035;',
        "\xC7\xB6" => '&#24316;',
        "\xC7\xB7" => '&#20043;',
        "\xC7\xB8" => '&#22492;',
        "\xC7\xB9" => '&#22178;',
        "\xC7\xBA" => '&#24745;',
        "\xC7\xBB" => '&#28611;',
        "\xC7\xBC" => '&#32013;',
        "\xC7\xBD" => '&#33021;',
        "\xC7\xBE" => '&#33075;',
        "\xC7\xBF" => '&#33215;',
        "\xC7\xC0" => '&#36786;',
        "\xC7\xC1" => '&#35223;',
        "\xC7\xC2" => '&#34468;',
        "\xC7\xC3" => '&#24052;',
        "\xC7\xC4" => '&#25226;',
        "\xC7\xC5" => '&#25773;',
        "\xC7\xC6" => '&#35207;',
        "\xC7\xC7" => '&#26487;',
        "\xC7\xC8" => '&#27874;',
        "\xC7\xC9" => '&#27966;',
        "\xC7\xCA" => '&#29750;',
        "\xC7\xCB" => '&#30772;',
        "\xC7\xCC" => '&#23110;',
        "\xC7\xCD" => '&#32629;',
        "\xC7\xCE" => '&#33453;',
        "\xC7\xCF" => '&#39340;',
        "\xC7\xD0" => '&#20467;',
        "\xC7\xD1" => '&#24259;',
        "\xC7\xD2" => '&#25309;',
        "\xC7\xD3" => '&#25490;',
        "\xC7\xD4" => '&#25943;',
        "\xC7\xD5" => '&#26479;',
        "\xC7\xD6" => '&#30403;',
        "\xC7\xD7" => '&#29260;',
        "\xC7\xD8" => '&#32972;',
        "\xC7\xD9" => '&#32954;',
        "\xC7\xDA" => '&#36649;',
        "\xC7\xDB" => '&#37197;',
        "\xC7\xDC" => '&#20493;',
        "\xC7\xDD" => '&#22521;',
        "\xC7\xDE" => '&#23186;',
        "\xC7\xDF" => '&#26757;',
        "\xC7\xE0" => '&#26995;',
        "\xC7\xE1" => '&#29028;',
        "\xC7\xE2" => '&#29437;',
        "\xC7\xE3" => '&#36023;',
        "\xC7\xE4" => '&#22770;',
        "\xC7\xE5" => '&#36064;',
        "\xC7\xE6" => '&#38506;',
        "\xC7\xE7" => '&#36889;',
        "\xC7\xE8" => '&#34687;',
        "\xC7\xE9" => '&#31204;',
        "\xC7\xEA" => '&#30695;',
        "\xC7\xEB" => '&#33833;',
        "\xC7\xEC" => '&#20271;',
        "\xC7\xED" => '&#21093;',
        "\xC7\xEE" => '&#21338;',
        "\xC7\xEF" => '&#25293;',
        "\xC7\xF0" => '&#26575;',
        "\xC7\xF1" => '&#27850;',
        "\xC7\xF2" => '&#30333;',
        "\xC7\xF3" => '&#31636;',
        "\xC7\xF4" => '&#31893;',
        "\xC7\xF5" => '&#33334;',
        "\xC7\xF6" => '&#34180;',
        "\xC7\xF7" => '&#36843;',
        "\xC7\xF8" => '&#26333;',
        "\xC7\xF9" => '&#28448;',
        "\xC7\xFA" => '&#29190;',
        "\xC7\xFB" => '&#32283;',
        "\xC7\xFC" => '&#33707;',
        "\xC7\xFD" => '&#39361;',
        "\xC7\xFE" => '&#40614;',
        "\xC8\xA1" => '&#20989;',
        "\xC8\xA2" => '&#31665;',
        "\xC8\xA3" => '&#30834;',
        "\xC8\xA4" => '&#31672;',
        "\xC8\xA5" => '&#32903;',
        "\xC8\xA6" => '&#31560;',
        "\xC8\xA7" => '&#27368;',
        "\xC8\xA8" => '&#24161;',
        "\xC8\xA9" => '&#32908;',
        "\xC8\xAA" => '&#30033;',
        "\xC8\xAB" => '&#30048;',
        "\xC8\xAC" => '&#20843;',
        "\xC8\xAD" => '&#37474;',
        "\xC8\xAE" => '&#28300;',
        "\xC8\xAF" => '&#30330;',
        "\xC8\xB0" => '&#37271;',
        "\xC8\xB1" => '&#39658;',
        "\xC8\xB2" => '&#20240;',
        "\xC8\xB3" => '&#32624;',
        "\xC8\xB4" => '&#25244;',
        "\xC8\xB5" => '&#31567;',
        "\xC8\xB6" => '&#38309;',
        "\xC8\xB7" => '&#40169;',
        "\xC8\xB8" => '&#22138;',
        "\xC8\xB9" => '&#22617;',
        "\xC8\xBA" => '&#34532;',
        "\xC8\xBB" => '&#38588;',
        "\xC8\xBC" => '&#20276;',
        "\xC8\xBD" => '&#21028;',
        "\xC8\xBE" => '&#21322;',
        "\xC8\xBF" => '&#21453;',
        "\xC8\xC0" => '&#21467;',
        "\xC8\xC1" => '&#24070;',
        "\xC8\xC2" => '&#25644;',
        "\xC8\xC3" => '&#26001;',
        "\xC8\xC4" => '&#26495;',
        "\xC8\xC5" => '&#27710;',
        "\xC8\xC6" => '&#27726;',
        "\xC8\xC7" => '&#29256;',
        "\xC8\xC8" => '&#29359;',
        "\xC8\xC9" => '&#29677;',
        "\xC8\xCA" => '&#30036;',
        "\xC8\xCB" => '&#32321;',
        "\xC8\xCC" => '&#33324;',
        "\xC8\xCD" => '&#34281;',
        "\xC8\xCE" => '&#36009;',
        "\xC8\xCF" => '&#31684;',
        "\xC8\xD0" => '&#37318;',
        "\xC8\xD1" => '&#29033;',
        "\xC8\xD2" => '&#38930;',
        "\xC8\xD3" => '&#39151;',
        "\xC8\xD4" => '&#25405;',
        "\xC8\xD5" => '&#26217;',
        "\xC8\xD6" => '&#30058;',
        "\xC8\xD7" => '&#30436;',
        "\xC8\xD8" => '&#30928;',
        "\xC8\xD9" => '&#34115;',
        "\xC8\xDA" => '&#34542;',
        "\xC8\xDB" => '&#21290;',
        "\xC8\xDC" => '&#21329;',
        "\xC8\xDD" => '&#21542;',
        "\xC8\xDE" => '&#22915;',
        "\xC8\xDF" => '&#24199;',
        "\xC8\xE0" => '&#24444;',
        "\xC8\xE1" => '&#24754;',
        "\xC8\xE2" => '&#25161;',
        "\xC8\xE3" => '&#25209;',
        "\xC8\xE4" => '&#25259;',
        "\xC8\xE5" => '&#26000;',
        "\xC8\xE6" => '&#27604;',
        "\xC8\xE7" => '&#27852;',
        "\xC8\xE8" => '&#30130;',
        "\xC8\xE9" => '&#30382;',
        "\xC8\xEA" => '&#30865;',
        "\xC8\xEB" => '&#31192;',
        "\xC8\xEC" => '&#32203;',
        "\xC8\xED" => '&#32631;',
        "\xC8\xEE" => '&#32933;',
        "\xC8\xEF" => '&#34987;',
        "\xC8\xF0" => '&#35513;',
        "\xC8\xF1" => '&#36027;',
        "\xC8\xF2" => '&#36991;',
        "\xC8\xF3" => '&#38750;',
        "\xC8\xF4" => '&#39131;',
        "\xC8\xF5" => '&#27147;',
        "\xC8\xF6" => '&#31800;',
        "\xC8\xF7" => '&#20633;',
        "\xC8\xF8" => '&#23614;',
        "\xC8\xF9" => '&#24494;',
        "\xC8\xFA" => '&#26503;',
        "\xC8\xFB" => '&#27608;',
        "\xC8\xFC" => '&#29749;',
        "\xC8\xFD" => '&#30473;',
        "\xC8\xFE" => '&#32654;',
        "\xC9\xA1" => '&#40763;',
        "\xC9\xA2" => '&#26570;',
        "\xC9\xA3" => '&#31255;',
        "\xC9\xA4" => '&#21305;',
        "\xC9\xA5" => '&#30091;',
        "\xC9\xA6" => '&#39661;',
        "\xC9\xA7" => '&#24422;',
        "\xC9\xA8" => '&#33181;',
        "\xC9\xA9" => '&#33777;',
        "\xC9\xAA" => '&#32920;',
        "\xC9\xAB" => '&#24380;',
        "\xC9\xAC" => '&#24517;',
        "\xC9\xAD" => '&#30050;',
        "\xC9\xAE" => '&#31558;',
        "\xC9\xAF" => '&#36924;',
        "\xC9\xB0" => '&#26727;',
        "\xC9\xB1" => '&#23019;',
        "\xC9\xB2" => '&#23195;',
        "\xC9\xB3" => '&#32016;',
        "\xC9\xB4" => '&#30334;',
        "\xC9\xB5" => '&#35628;',
        "\xC9\xB6" => '&#20469;',
        "\xC9\xB7" => '&#24426;',
        "\xC9\xB8" => '&#27161;',
        "\xC9\xB9" => '&#27703;',
        "\xC9\xBA" => '&#28418;',
        "\xC9\xBB" => '&#29922;',
        "\xC9\xBC" => '&#31080;',
        "\xC9\xBD" => '&#34920;',
        "\xC9\xBE" => '&#35413;',
        "\xC9\xBF" => '&#35961;',
        "\xC9\xC0" => '&#24287;',
        "\xC9\xC1" => '&#25551;',
        "\xC9\xC2" => '&#30149;',
        "\xC9\xC3" => '&#31186;',
        "\xC9\xC4" => '&#33495;',
        "\xC9\xC5" => '&#37672;',
        "\xC9\xC6" => '&#37618;',
        "\xC9\xC7" => '&#33948;',
        "\xC9\xC8" => '&#34541;',
        "\xC9\xC9" => '&#39981;',
        "\xC9\xCA" => '&#21697;',
        "\xC9\xCB" => '&#24428;',
        "\xC9\xCC" => '&#25996;',
        "\xC9\xCD" => '&#27996;',
        "\xC9\xCE" => '&#28693;',
        "\xC9\xCF" => '&#36007;',
        "\xC9\xD0" => '&#36051;',
        "\xC9\xD1" => '&#38971;',
        "\xC9\xD2" => '&#25935;',
        "\xC9\xD3" => '&#29942;',
        "\xC9\xD4" => '&#19981;',
        "\xC9\xD5" => '&#20184;',
        "\xC9\xD6" => '&#22496;',
        "\xC9\xD7" => '&#22827;',
        "\xC9\xD8" => '&#23142;',
        "\xC9\xD9" => '&#23500;',
        "\xC9\xDA" => '&#20904;',
        "\xC9\xDB" => '&#24067;',
        "\xC9\xDC" => '&#24220;',
        "\xC9\xDD" => '&#24598;',
        "\xC9\xDE" => '&#25206;',
        "\xC9\xDF" => '&#25975;',
        "\xC9\xE0" => '&#26023;',
        "\xC9\xE1" => '&#26222;',
        "\xC9\xE2" => '&#28014;',
        "\xC9\xE3" => '&#29238;',
        "\xC9\xE4" => '&#31526;',
        "\xC9\xE5" => '&#33104;',
        "\xC9\xE6" => '&#33178;',
        "\xC9\xE7" => '&#33433;',
        "\xC9\xE8" => '&#35676;',
        "\xC9\xE9" => '&#36000;',
        "\xC9\xEA" => '&#36070;',
        "\xC9\xEB" => '&#36212;',
        "\xC9\xEC" => '&#38428;',
        "\xC9\xED" => '&#38468;',
        "\xC9\xEE" => '&#20398;',
        "\xC9\xEF" => '&#25771;',
        "\xC9\xF0" => '&#27494;',
        "\xC9\xF1" => '&#33310;',
        "\xC9\xF2" => '&#33889;',
        "\xC9\xF3" => '&#34154;',
        "\xC9\xF4" => '&#37096;',
        "\xC9\xF5" => '&#23553;',
        "\xC9\xF6" => '&#26963;',
        "\xC9\xF7" => '&#39080;',
        "\xC9\xF8" => '&#33914;',
        "\xC9\xF9" => '&#34135;',
        "\xC9\xFA" => '&#20239;',
        "\xC9\xFB" => '&#21103;',
        "\xC9\xFC" => '&#24489;',
        "\xC9\xFD" => '&#24133;',
        "\xC9\xFE" => '&#26381;',
        "\xCA\xA1" => '&#31119;',
        "\xCA\xA2" => '&#33145;',
        "\xCA\xA3" => '&#35079;',
        "\xCA\xA4" => '&#35206;',
        "\xCA\xA5" => '&#28149;',
        "\xCA\xA6" => '&#24343;',
        "\xCA\xA7" => '&#25173;',
        "\xCA\xA8" => '&#27832;',
        "\xCA\xA9" => '&#20175;',
        "\xCA\xAA" => '&#29289;',
        "\xCA\xAB" => '&#39826;',
        "\xCA\xAC" => '&#20998;',
        "\xCA\xAD" => '&#21563;',
        "\xCA\xAE" => '&#22132;',
        "\xCA\xAF" => '&#22707;',
        "\xCA\xB0" => '&#24996;',
        "\xCA\xB1" => '&#25198;',
        "\xCA\xB2" => '&#28954;',
        "\xCA\xB3" => '&#22894;',
        "\xCA\xB4" => '&#31881;',
        "\xCA\xB5" => '&#31966;',
        "\xCA\xB6" => '&#32027;',
        "\xCA\xB7" => '&#38640;',
        "\xCA\xB8" => '&#25991;',
        "\xCA\xB9" => '&#32862;',
        "\xCA\xBA" => '&#19993;',
        "\xCA\xBB" => '&#20341;',
        "\xCA\xBC" => '&#20853;',
        "\xCA\xBD" => '&#22592;',
        "\xCA\xBE" => '&#24163;',
        "\xCA\xBF" => '&#24179;',
        "\xCA\xC0" => '&#24330;',
        "\xCA\xC1" => '&#26564;',
        "\xCA\xC2" => '&#20006;',
        "\xCA\xC3" => '&#34109;',
        "\xCA\xC4" => '&#38281;',
        "\xCA\xC5" => '&#38491;',
        "\xCA\xC6" => '&#31859;',
        "\xCA\xC7" => '&#38913;',
        "\xCA\xC8" => '&#20731;',
        "\xCA\xC9" => '&#22721;',
        "\xCA\xCA" => '&#30294;',
        "\xCA\xCB" => '&#30887;',
        "\xCA\xCC" => '&#21029;',
        "\xCA\xCD" => '&#30629;',
        "\xCA\xCE" => '&#34065;',
        "\xCA\xCF" => '&#31622;',
        "\xCA\xD0" => '&#20559;',
        "\xCA\xD1" => '&#22793;',
        "\xCA\xD2" => '&#29255;',
        "\xCA\xD3" => '&#31687;',
        "\xCA\xD4" => '&#32232;',
        "\xCA\xD5" => '&#36794;',
        "\xCA\xD6" => '&#36820;',
        "\xCA\xD7" => '&#36941;',
        "\xCA\xD8" => '&#20415;',
        "\xCA\xD9" => '&#21193;',
        "\xCA\xDA" => '&#23081;',
        "\xCA\xDB" => '&#24321;',
        "\xCA\xDC" => '&#38829;',
        "\xCA\xDD" => '&#20445;',
        "\xCA\xDE" => '&#33303;',
        "\xCA\xDF" => '&#37610;',
        "\xCA\xE0" => '&#22275;',
        "\xCA\xE1" => '&#25429;',
        "\xCA\xE2" => '&#27497;',
        "\xCA\xE3" => '&#29995;',
        "\xCA\xE4" => '&#35036;',
        "\xCA\xE5" => '&#36628;',
        "\xCA\xE6" => '&#31298;',
        "\xCA\xE7" => '&#21215;',
        "\xCA\xE8" => '&#22675;',
        "\xCA\xE9" => '&#24917;',
        "\xCA\xEA" => '&#25098;',
        "\xCA\xEB" => '&#26286;',
        "\xCA\xEC" => '&#27597;',
        "\xCA\xED" => '&#31807;',
        "\xCA\xEE" => '&#33769;',
        "\xCA\xEF" => '&#20515;',
        "\xCA\xF0" => '&#20472;',
        "\xCA\xF1" => '&#21253;',
        "\xCA\xF2" => '&#21574;',
        "\xCA\xF3" => '&#22577;',
        "\xCA\xF4" => '&#22857;',
        "\xCA\xF5" => '&#23453;',
        "\xCA\xF6" => '&#23792;',
        "\xCA\xF7" => '&#23791;',
        "\xCA\xF8" => '&#23849;',
        "\xCA\xF9" => '&#24214;',
        "\xCA\xFA" => '&#25265;',
        "\xCA\xFB" => '&#25447;',
        "\xCA\xFC" => '&#25918;',
        "\xCA\xFD" => '&#26041;',
        "\xCA\xFE" => '&#26379;',
        "\xCB\xA1" => '&#27861;',
        "\xCB\xA2" => '&#27873;',
        "\xCB\xA3" => '&#28921;',
        "\xCB\xA4" => '&#30770;',
        "\xCB\xA5" => '&#32299;',
        "\xCB\xA6" => '&#32990;',
        "\xCB\xA7" => '&#33459;',
        "\xCB\xA8" => '&#33804;',
        "\xCB\xA9" => '&#34028;',
        "\xCB\xAA" => '&#34562;',
        "\xCB\xAB" => '&#35090;',
        "\xCB\xAC" => '&#35370;',
        "\xCB\xAD" => '&#35914;',
        "\xCB\xAE" => '&#37030;',
        "\xCB\xAF" => '&#37586;',
        "\xCB\xB0" => '&#39165;',
        "\xCB\xB1" => '&#40179;',
        "\xCB\xB2" => '&#40300;',
        "\xCB\xB3" => '&#20047;',
        "\xCB\xB4" => '&#20129;',
        "\xCB\xB5" => '&#20621;',
        "\xCB\xB6" => '&#21078;',
        "\xCB\xB7" => '&#22346;',
        "\xCB\xB8" => '&#22952;',
        "\xCB\xB9" => '&#24125;',
        "\xCB\xBA" => '&#24536;',
        "\xCB\xBB" => '&#24537;',
        "\xCB\xBC" => '&#25151;',
        "\xCB\xBD" => '&#26292;',
        "\xCB\xBE" => '&#26395;',
        "\xCB\xBF" => '&#26576;',
        "\xCB\xC0" => '&#26834;',
        "\xCB\xC1" => '&#20882;',
        "\xCB\xC2" => '&#32033;',
        "\xCB\xC3" => '&#32938;',
        "\xCB\xC4" => '&#33192;',
        "\xCB\xC5" => '&#35584;',
        "\xCB\xC6" => '&#35980;',
        "\xCB\xC7" => '&#36031;',
        "\xCB\xC8" => '&#37502;',
        "\xCB\xC9" => '&#38450;',
        "\xCB\xCA" => '&#21536;',
        "\xCB\xCB" => '&#38956;',
        "\xCB\xCC" => '&#21271;',
        "\xCB\xCD" => '&#20693;',
        "\xCB\xCE" => '&#21340;',
        "\xCB\xCF" => '&#22696;',
        "\xCB\xD0" => '&#25778;',
        "\xCB\xD1" => '&#26420;',
        "\xCB\xD2" => '&#29287;',
        "\xCB\xD3" => '&#30566;',
        "\xCB\xD4" => '&#31302;',
        "\xCB\xD5" => '&#37350;',
        "\xCB\xD6" => '&#21187;',
        "\xCB\xD7" => '&#27809;',
        "\xCB\xD8" => '&#27526;',
        "\xCB\xD9" => '&#22528;',
        "\xCB\xDA" => '&#24140;',
        "\xCB\xDB" => '&#22868;',
        "\xCB\xDC" => '&#26412;',
        "\xCB\xDD" => '&#32763;',
        "\xCB\xDE" => '&#20961;',
        "\xCB\xDF" => '&#30406;',
        "\xCB\xE0" => '&#25705;',
        "\xCB\xE1" => '&#30952;',
        "\xCB\xE2" => '&#39764;',
        "\xCB\xE3" => '&#40635;',
        "\xCB\xE4" => '&#22475;',
        "\xCB\xE5" => '&#22969;',
        "\xCB\xE6" => '&#26151;',
        "\xCB\xE7" => '&#26522;',
        "\xCB\xE8" => '&#27598;',
        "\xCB\xE9" => '&#21737;',
        "\xCB\xEA" => '&#27097;',
        "\xCB\xEB" => '&#24149;',
        "\xCB\xEC" => '&#33180;',
        "\xCB\xED" => '&#26517;',
        "\xCB\xEE" => '&#39850;',
        "\xCB\xEF" => '&#26622;',
        "\xCB\xF0" => '&#40018;',
        "\xCB\xF1" => '&#26717;',
        "\xCB\xF2" => '&#20134;',
        "\xCB\xF3" => '&#20451;',
        "\xCB\xF4" => '&#21448;',
        "\xCB\xF5" => '&#25273;',
        "\xCB\xF6" => '&#26411;',
        "\xCB\xF7" => '&#27819;',
        "\xCB\xF8" => '&#36804;',
        "\xCB\xF9" => '&#20397;',
        "\xCB\xFA" => '&#32365;',
        "\xCB\xFB" => '&#40639;',
        "\xCB\xFC" => '&#19975;',
        "\xCB\xFD" => '&#24930;',
        "\xCB\xFE" => '&#28288;',
        "\xCC\xA1" => '&#28459;',
        "\xCC\xA2" => '&#34067;',
        "\xCC\xA3" => '&#21619;',
        "\xCC\xA4" => '&#26410;',
        "\xCC\xA5" => '&#39749;',
        "\xCC\xA6" => '&#24051;',
        "\xCC\xA7" => '&#31637;',
        "\xCC\xA8" => '&#23724;',
        "\xCC\xA9" => '&#23494;',
        "\xCC\xAA" => '&#34588;',
        "\xCC\xAB" => '&#28234;',
        "\xCC\xAC" => '&#34001;',
        "\xCC\xAD" => '&#31252;',
        "\xCC\xAE" => '&#33032;',
        "\xCC\xAF" => '&#22937;',
        "\xCC\xB0" => '&#31885;',
        "\xCC\xB1" => '&#27665;',
        "\xCC\xB2" => '&#30496;',
        "\xCC\xB3" => '&#21209;',
        "\xCC\xB4" => '&#22818;',
        "\xCC\xB5" => '&#28961;',
        "\xCC\xB6" => '&#29279;',
        "\xCC\xB7" => '&#30683;',
        "\xCC\xB8" => '&#38695;',
        "\xCC\xB9" => '&#40289;',
        "\xCC\xBA" => '&#26891;',
        "\xCC\xBB" => '&#23167;',
        "\xCC\xBC" => '&#23064;',
        "\xCC\xBD" => '&#20901;',
        "\xCC\xBE" => '&#21517;',
        "\xCC\xBF" => '&#21629;',
        "\xCC\xC0" => '&#26126;',
        "\xCC\xC1" => '&#30431;',
        "\xCC\xC2" => '&#36855;',
        "\xCC\xC3" => '&#37528;',
        "\xCC\xC4" => '&#40180;',
        "\xCC\xC5" => '&#23018;',
        "\xCC\xC6" => '&#29277;',
        "\xCC\xC7" => '&#28357;',
        "\xCC\xC8" => '&#20813;',
        "\xCC\xC9" => '&#26825;',
        "\xCC\xCA" => '&#32191;',
        "\xCC\xCB" => '&#32236;',
        "\xCC\xCC" => '&#38754;',
        "\xCC\xCD" => '&#40634;',
        "\xCC\xCE" => '&#25720;',
        "\xCC\xCF" => '&#27169;',
        "\xCC\xD0" => '&#33538;',
        "\xCC\xD1" => '&#22916;',
        "\xCC\xD2" => '&#23391;',
        "\xCC\xD3" => '&#27611;',
        "\xCC\xD4" => '&#29467;',
        "\xCC\xD5" => '&#30450;',
        "\xCC\xD6" => '&#32178;',
        "\xCC\xD7" => '&#32791;',
        "\xCC\xD8" => '&#33945;',
        "\xCC\xD9" => '&#20786;',
        "\xCC\xDA" => '&#26408;',
        "\xCC\xDB" => '&#40665;',
        "\xCC\xDC" => '&#30446;',
        "\xCC\xDD" => '&#26466;',
        "\xCC\xDE" => '&#21247;',
        "\xCC\xDF" => '&#39173;',
        "\xCC\xE0" => '&#23588;',
        "\xCC\xE1" => '&#25147;',
        "\xCC\xE2" => '&#31870;',
        "\xCC\xE3" => '&#36016;',
        "\xCC\xE4" => '&#21839;',
        "\xCC\xE5" => '&#24758;',
        "\xCC\xE6" => '&#32011;',
        "\xCC\xE7" => '&#38272;',
        "\xCC\xE8" => '&#21249;',
        "\xCC\xE9" => '&#20063;',
        "\xCC\xEA" => '&#20918;',
        "\xCC\xEB" => '&#22812;',
        "\xCC\xEC" => '&#29242;',
        "\xCC\xED" => '&#32822;',
        "\xCC\xEE" => '&#37326;',
        "\xCC\xEF" => '&#24357;',
        "\xCC\xF0" => '&#30690;',
        "\xCC\xF1" => '&#21380;',
        "\xCC\xF2" => '&#24441;',
        "\xCC\xF3" => '&#32004;',
        "\xCC\xF4" => '&#34220;',
        "\xCC\xF5" => '&#35379;',
        "\xCC\xF6" => '&#36493;',
        "\xCC\xF7" => '&#38742;',
        "\xCC\xF8" => '&#26611;',
        "\xCC\xF9" => '&#34222;',
        "\xCC\xFA" => '&#37971;',
        "\xCC\xFB" => '&#24841;',
        "\xCC\xFC" => '&#24840;',
        "\xCC\xFD" => '&#27833;',
        "\xCC\xFE" => '&#30290;',
        "\xCD\xA1" => '&#35565;',
        "\xCD\xA2" => '&#36664;',
        "\xCD\xA3" => '&#21807;',
        "\xCD\xA4" => '&#20305;',
        "\xCD\xA5" => '&#20778;',
        "\xCD\xA6" => '&#21191;',
        "\xCD\xA7" => '&#21451;',
        "\xCD\xA8" => '&#23461;',
        "\xCD\xA9" => '&#24189;',
        "\xCD\xAA" => '&#24736;',
        "\xCD\xAB" => '&#24962;',
        "\xCD\xAC" => '&#25558;',
        "\xCD\xAD" => '&#26377;',
        "\xCD\xAE" => '&#26586;',
        "\xCD\xAF" => '&#28263;',
        "\xCD\xB0" => '&#28044;',
        "\xCD\xB1" => '&#29494;',
        "\xCD\xB2" => '&#29495;',
        "\xCD\xB3" => '&#30001;',
        "\xCD\xB4" => '&#31056;',
        "\xCD\xB5" => '&#35029;',
        "\xCD\xB6" => '&#35480;',
        "\xCD\xB7" => '&#36938;',
        "\xCD\xB8" => '&#37009;',
        "\xCD\xB9" => '&#37109;',
        "\xCD\xBA" => '&#38596;',
        "\xCD\xBB" => '&#34701;',
        "\xCD\xBC" => '&#22805;',
        "\xCD\xBD" => '&#20104;',
        "\xCD\xBE" => '&#20313;',
        "\xCD\xBF" => '&#19982;',
        "\xCD\xC0" => '&#35465;',
        "\xCD\xC1" => '&#36671;',
        "\xCD\xC2" => '&#38928;',
        "\xCD\xC3" => '&#20653;',
        "\xCD\xC4" => '&#24188;',
        "\xCD\xC5" => '&#22934;',
        "\xCD\xC6" => '&#23481;',
        "\xCD\xC7" => '&#24248;',
        "\xCD\xC8" => '&#25562;',
        "\xCD\xC9" => '&#25594;',
        "\xCD\xCA" => '&#25793;',
        "\xCD\xCB" => '&#26332;',
        "\xCD\xCC" => '&#26954;',
        "\xCD\xCD" => '&#27096;',
        "\xCD\xCE" => '&#27915;',
        "\xCD\xCF" => '&#28342;',
        "\xCD\xD0" => '&#29076;',
        "\xCD\xD1" => '&#29992;',
        "\xCD\xD2" => '&#31407;',
        "\xCD\xD3" => '&#32650;',
        "\xCD\xD4" => '&#32768;',
        "\xCD\xD5" => '&#33865;',
        "\xCD\xD6" => '&#33993;',
        "\xCD\xD7" => '&#35201;',
        "\xCD\xD8" => '&#35617;',
        "\xCD\xD9" => '&#36362;',
        "\xCD\xDA" => '&#36965;',
        "\xCD\xDB" => '&#38525;',
        "\xCD\xDC" => '&#39178;',
        "\xCD\xDD" => '&#24958;',
        "\xCD\xDE" => '&#25233;',
        "\xCD\xDF" => '&#27442;',
        "\xCD\xE0" => '&#27779;',
        "\xCD\xE1" => '&#28020;',
        "\xCD\xE2" => '&#32716;',
        "\xCD\xE3" => '&#32764;',
        "\xCD\xE4" => '&#28096;',
        "\xCD\xE5" => '&#32645;',
        "\xCD\xE6" => '&#34746;',
        "\xCD\xE7" => '&#35064;',
        "\xCD\xE8" => '&#26469;',
        "\xCD\xE9" => '&#33713;',
        "\xCD\xEA" => '&#38972;',
        "\xCD\xEB" => '&#38647;',
        "\xCD\xEC" => '&#27931;',
        "\xCD\xED" => '&#32097;',
        "\xCD\xEE" => '&#33853;',
        "\xCD\xEF" => '&#37226;',
        "\xCD\xF0" => '&#20081;',
        "\xCD\xF1" => '&#21365;',
        "\xCD\xF2" => '&#23888;',
        "\xCD\xF3" => '&#27396;',
        "\xCD\xF4" => '&#28651;',
        "\xCD\xF5" => '&#34253;',
        "\xCD\xF6" => '&#34349;',
        "\xCD\xF7" => '&#35239;',
        "\xCD\xF8" => '&#21033;',
        "\xCD\xF9" => '&#21519;',
        "\xCD\xFA" => '&#23653;',
        "\xCD\xFB" => '&#26446;',
        "\xCD\xFC" => '&#26792;',
        "\xCD\xFD" => '&#29702;',
        "\xCD\xFE" => '&#29827;',
        "\xCE\xA1" => '&#30178;',
        "\xCE\xA2" => '&#35023;',
        "\xCE\xA3" => '&#35041;',
        "\xCE\xA4" => '&#37324;',
        "\xCE\xA5" => '&#38626;',
        "\xCE\xA6" => '&#38520;',
        "\xCE\xA7" => '&#24459;',
        "\xCE\xA8" => '&#29575;',
        "\xCE\xA9" => '&#31435;',
        "\xCE\xAA" => '&#33870;',
        "\xCE\xAB" => '&#25504;',
        "\xCE\xAC" => '&#30053;',
        "\xCE\xAD" => '&#21129;',
        "\xCE\xAE" => '&#27969;',
        "\xCE\xAF" => '&#28316;',
        "\xCE\xB0" => '&#29705;',
        "\xCE\xB1" => '&#30041;',
        "\xCE\xB2" => '&#30827;',
        "\xCE\xB3" => '&#31890;',
        "\xCE\xB4" => '&#38534;',
        "\xCE\xB5" => '&#31452;',
        "\xCE\xB6" => '&#40845;',
        "\xCE\xB7" => '&#20406;',
        "\xCE\xB8" => '&#24942;',
        "\xCE\xB9" => '&#26053;',
        "\xCE\xBA" => '&#34396;',
        "\xCE\xBB" => '&#20102;',
        "\xCE\xBC" => '&#20142;',
        "\xCE\xBD" => '&#20698;',
        "\xCE\xBE" => '&#20001;',
        "\xCE\xBF" => '&#20940;',
        "\xCE\xC0" => '&#23534;',
        "\xCE\xC1" => '&#26009;',
        "\xCE\xC2" => '&#26753;',
        "\xCE\xC3" => '&#28092;',
        "\xCE\xC4" => '&#29471;',
        "\xCE\xC5" => '&#30274;',
        "\xCE\xC6" => '&#30637;',
        "\xCE\xC7" => '&#31260;',
        "\xCE\xC8" => '&#31975;',
        "\xCE\xC9" => '&#33391;',
        "\xCE\xCA" => '&#35538;',
        "\xCE\xCB" => '&#36988;',
        "\xCE\xCC" => '&#37327;',
        "\xCE\xCD" => '&#38517;',
        "\xCE\xCE" => '&#38936;',
        "\xCE\xCF" => '&#21147;',
        "\xCE\xD0" => '&#32209;',
        "\xCE\xD1" => '&#20523;',
        "\xCE\xD2" => '&#21400;',
        "\xCE\xD3" => '&#26519;',
        "\xCE\xD4" => '&#28107;',
        "\xCE\xD5" => '&#29136;',
        "\xCE\xD6" => '&#29747;',
        "\xCE\xD7" => '&#33256;',
        "\xCE\xD8" => '&#36650;',
        "\xCE\xD9" => '&#38563;',
        "\xCE\xDA" => '&#40023;',
        "\xCE\xDB" => '&#40607;',
        "\xCE\xDC" => '&#29792;',
        "\xCE\xDD" => '&#22593;',
        "\xCE\xDE" => '&#28057;',
        "\xCE\xDF" => '&#32047;',
        "\xCE\xE0" => '&#39006;',
        "\xCE\xE1" => '&#20196;',
        "\xCE\xE2" => '&#20278;',
        "\xCE\xE3" => '&#20363;',
        "\xCE\xE4" => '&#20919;',
        "\xCE\xE5" => '&#21169;',
        "\xCE\xE6" => '&#23994;',
        "\xCE\xE7" => '&#24604;',
        "\xCE\xE8" => '&#29618;',
        "\xCE\xE9" => '&#31036;',
        "\xCE\xEA" => '&#33491;',
        "\xCE\xEB" => '&#37428;',
        "\xCE\xEC" => '&#38583;',
        "\xCE\xED" => '&#38646;',
        "\xCE\xEE" => '&#38666;',
        "\xCE\xEF" => '&#40599;',
        "\xCE\xF0" => '&#40802;',
        "\xCE\xF1" => '&#26278;',
        "\xCE\xF2" => '&#27508;',
        "\xCE\xF3" => '&#21015;',
        "\xCE\xF4" => '&#21155;',
        "\xCE\xF5" => '&#28872;',
        "\xCE\xF6" => '&#35010;',
        "\xCE\xF7" => '&#24265;',
        "\xCE\xF8" => '&#24651;',
        "\xCE\xF9" => '&#24976;',
        "\xCE\xFA" => '&#28451;',
        "\xCE\xFB" => '&#29001;',
        "\xCE\xFC" => '&#31806;',
        "\xCE\xFD" => '&#32244;',
        "\xCE\xFE" => '&#32879;',
        "\xCF\xA1" => '&#34030;',
        "\xCF\xA2" => '&#36899;',
        "\xCF\xA3" => '&#37676;',
        "\xCF\xA4" => '&#21570;',
        "\xCF\xA5" => '&#39791;',
        "\xCF\xA6" => '&#27347;',
        "\xCF\xA7" => '&#28809;',
        "\xCF\xA8" => '&#36034;',
        "\xCF\xA9" => '&#36335;',
        "\xCF\xAA" => '&#38706;',
        "\xCF\xAB" => '&#21172;',
        "\xCF\xAC" => '&#23105;',
        "\xCF\xAD" => '&#24266;',
        "\xCF\xAE" => '&#24324;',
        "\xCF\xAF" => '&#26391;',
        "\xCF\xB0" => '&#27004;',
        "\xCF\xB1" => '&#27028;',
        "\xCF\xB2" => '&#28010;',
        "\xCF\xB3" => '&#28431;',
        "\xCF\xB4" => '&#29282;',
        "\xCF\xB5" => '&#29436;',
        "\xCF\xB6" => '&#31725;',
        "\xCF\xB7" => '&#32769;',
        "\xCF\xB8" => '&#32894;',
        "\xCF\xB9" => '&#34635;',
        "\xCF\xBA" => '&#37070;',
        "\xCF\xBB" => '&#20845;',
        "\xCF\xBC" => '&#40595;',
        "\xCF\xBD" => '&#31108;',
        "\xCF\xBE" => '&#32907;',
        "\xCF\xBF" => '&#37682;',
        "\xCF\xC0" => '&#35542;',
        "\xCF\xC1" => '&#20525;',
        "\xCF\xC2" => '&#21644;',
        "\xCF\xC3" => '&#35441;',
        "\xCF\xC4" => '&#27498;',
        "\xCF\xC5" => '&#36036;',
        "\xCF\xC6" => '&#33031;',
        "\xCF\xC7" => '&#24785;',
        "\xCF\xC8" => '&#26528;',
        "\xCF\xC9" => '&#40434;',
        "\xCF\xCA" => '&#20121;',
        "\xCF\xCB" => '&#20120;',
        "\xCF\xCC" => '&#39952;',
        "\xCF\xCD" => '&#35435;',
        "\xCF\xCE" => '&#34241;',
        "\xCF\xCF" => '&#34152;',
        "\xCF\xD0" => '&#26880;',
        "\xCF\xD1" => '&#28286;',
        "\xCF\xD2" => '&#30871;',
        "\xCF\xD3" => '&#33109;',
        "\xD0\xA1" => '&#24332;',
        "\xD0\xA2" => '&#19984;',
        "\xD0\xA3" => '&#19989;',
        "\xD0\xA4" => '&#20010;',
        "\xD0\xA5" => '&#20017;',
        "\xD0\xA6" => '&#20022;',
        "\xD0\xA7" => '&#20028;',
        "\xD0\xA8" => '&#20031;',
        "\xD0\xA9" => '&#20034;',
        "\xD0\xAA" => '&#20054;',
        "\xD0\xAB" => '&#20056;',
        "\xD0\xAC" => '&#20098;',
        "\xD0\xAD" => '&#20101;',
        "\xD0\xAE" => '&#35947;',
        "\xD0\xAF" => '&#20106;',
        "\xD0\xB0" => '&#33298;',
        "\xD0\xB1" => '&#24333;',
        "\xD0\xB2" => '&#20110;',
        "\xD0\xB3" => '&#20126;',
        "\xD0\xB4" => '&#20127;',
        "\xD0\xB5" => '&#20128;',
        "\xD0\xB6" => '&#20130;',
        "\xD0\xB7" => '&#20144;',
        "\xD0\xB8" => '&#20147;',
        "\xD0\xB9" => '&#20150;',
        "\xD0\xBA" => '&#20174;',
        "\xD0\xBB" => '&#20173;',
        "\xD0\xBC" => '&#20164;',
        "\xD0\xBD" => '&#20166;',
        "\xD0\xBE" => '&#20162;',
        "\xD0\xBF" => '&#20183;',
        "\xD0\xC0" => '&#20190;',
        "\xD0\xC1" => '&#20205;',
        "\xD0\xC2" => '&#20191;',
        "\xD0\xC3" => '&#20215;',
        "\xD0\xC4" => '&#20233;',
        "\xD0\xC5" => '&#20314;',
        "\xD0\xC6" => '&#20272;',
        "\xD0\xC7" => '&#20315;',
        "\xD0\xC8" => '&#20317;',
        "\xD0\xC9" => '&#20311;',
        "\xD0\xCA" => '&#20295;',
        "\xD0\xCB" => '&#20342;',
        "\xD0\xCC" => '&#20360;',
        "\xD0\xCD" => '&#20367;',
        "\xD0\xCE" => '&#20376;',
        "\xD0\xCF" => '&#20347;',
        "\xD0\xD0" => '&#20329;',
        "\xD0\xD1" => '&#20336;',
        "\xD0\xD2" => '&#20369;',
        "\xD0\xD3" => '&#20335;',
        "\xD0\xD4" => '&#20358;',
        "\xD0\xD5" => '&#20374;',
        "\xD0\xD6" => '&#20760;',
        "\xD0\xD7" => '&#20436;',
        "\xD0\xD8" => '&#20447;',
        "\xD0\xD9" => '&#20430;',
        "\xD0\xDA" => '&#20440;',
        "\xD0\xDB" => '&#20443;',
        "\xD0\xDC" => '&#20433;',
        "\xD0\xDD" => '&#20442;',
        "\xD0\xDE" => '&#20432;',
        "\xD0\xDF" => '&#20452;',
        "\xD0\xE0" => '&#20453;',
        "\xD0\xE1" => '&#20506;',
        "\xD0\xE2" => '&#20520;',
        "\xD0\xE3" => '&#20500;',
        "\xD0\xE4" => '&#20522;',
        "\xD0\xE5" => '&#20517;',
        "\xD0\xE6" => '&#20485;',
        "\xD0\xE7" => '&#20252;',
        "\xD0\xE8" => '&#20470;',
        "\xD0\xE9" => '&#20513;',
        "\xD0\xEA" => '&#20521;',
        "\xD0\xEB" => '&#20524;',
        "\xD0\xEC" => '&#20478;',
        "\xD0\xED" => '&#20463;',
        "\xD0\xEE" => '&#20497;',
        "\xD0\xEF" => '&#20486;',
        "\xD0\xF0" => '&#20547;',
        "\xD0\xF1" => '&#20551;',
        "\xD0\xF2" => '&#26371;',
        "\xD0\xF3" => '&#20565;',
        "\xD0\xF4" => '&#20560;',
        "\xD0\xF5" => '&#20552;',
        "\xD0\xF6" => '&#20570;',
        "\xD0\xF7" => '&#20566;',
        "\xD0\xF8" => '&#20588;',
        "\xD0\xF9" => '&#20600;',
        "\xD0\xFA" => '&#20608;',
        "\xD0\xFB" => '&#20634;',
        "\xD0\xFC" => '&#20613;',
        "\xD0\xFD" => '&#20660;',
        "\xD0\xFE" => '&#20658;',
        "\xD1\xA1" => '&#20681;',
        "\xD1\xA2" => '&#20682;',
        "\xD1\xA3" => '&#20659;',
        "\xD1\xA4" => '&#20674;',
        "\xD1\xA5" => '&#20694;',
        "\xD1\xA6" => '&#20702;',
        "\xD1\xA7" => '&#20709;',
        "\xD1\xA8" => '&#20717;',
        "\xD1\xA9" => '&#20707;',
        "\xD1\xAA" => '&#20718;',
        "\xD1\xAB" => '&#20729;',
        "\xD1\xAC" => '&#20725;',
        "\xD1\xAD" => '&#20745;',
        "\xD1\xAE" => '&#20737;',
        "\xD1\xAF" => '&#20738;',
        "\xD1\xB0" => '&#20758;',
        "\xD1\xB1" => '&#20757;',
        "\xD1\xB2" => '&#20756;',
        "\xD1\xB3" => '&#20762;',
        "\xD1\xB4" => '&#20769;',
        "\xD1\xB5" => '&#20794;',
        "\xD1\xB6" => '&#20791;',
        "\xD1\xB7" => '&#20796;',
        "\xD1\xB8" => '&#20795;',
        "\xD1\xB9" => '&#20799;',
        "\xD1\xBA" => '&#20800;',
        "\xD1\xBB" => '&#20818;',
        "\xD1\xBC" => '&#20812;',
        "\xD1\xBD" => '&#20820;',
        "\xD1\xBE" => '&#20834;',
        "\xD1\xBF" => '&#31480;',
        "\xD1\xC0" => '&#20841;',
        "\xD1\xC1" => '&#20842;',
        "\xD1\xC2" => '&#20846;',
        "\xD1\xC3" => '&#20864;',
        "\xD1\xC4" => '&#20866;',
        "\xD1\xC5" => '&#22232;',
        "\xD1\xC6" => '&#20876;',
        "\xD1\xC7" => '&#20873;',
        "\xD1\xC8" => '&#20879;',
        "\xD1\xC9" => '&#20881;',
        "\xD1\xCA" => '&#20883;',
        "\xD1\xCB" => '&#20885;',
        "\xD1\xCC" => '&#20886;',
        "\xD1\xCD" => '&#20900;',
        "\xD1\xCE" => '&#20902;',
        "\xD1\xCF" => '&#20898;',
        "\xD1\xD0" => '&#20905;',
        "\xD1\xD1" => '&#20906;',
        "\xD1\xD2" => '&#20907;',
        "\xD1\xD3" => '&#20915;',
        "\xD1\xD4" => '&#20913;',
        "\xD1\xD5" => '&#20914;',
        "\xD1\xD6" => '&#20912;',
        "\xD1\xD7" => '&#20917;',
        "\xD1\xD8" => '&#20925;',
        "\xD1\xD9" => '&#20933;',
        "\xD1\xDA" => '&#20937;',
        "\xD1\xDB" => '&#20955;',
        "\xD1\xDC" => '&#20960;',
        "\xD1\xDD" => '&#34389;',
        "\xD1\xDE" => '&#20969;',
        "\xD1\xDF" => '&#20973;',
        "\xD1\xE0" => '&#20976;',
        "\xD1\xE1" => '&#20981;',
        "\xD1\xE2" => '&#20990;',
        "\xD1\xE3" => '&#20996;',
        "\xD1\xE4" => '&#21003;',
        "\xD1\xE5" => '&#21012;',
        "\xD1\xE6" => '&#21006;',
        "\xD1\xE7" => '&#21031;',
        "\xD1\xE8" => '&#21034;',
        "\xD1\xE9" => '&#21038;',
        "\xD1\xEA" => '&#21043;',
        "\xD1\xEB" => '&#21049;',
        "\xD1\xEC" => '&#21071;',
        "\xD1\xED" => '&#21060;',
        "\xD1\xEE" => '&#21067;',
        "\xD1\xEF" => '&#21068;',
        "\xD1\xF0" => '&#21086;',
        "\xD1\xF1" => '&#21076;',
        "\xD1\xF2" => '&#21098;',
        "\xD1\xF3" => '&#21108;',
        "\xD1\xF4" => '&#21097;',
        "\xD1\xF5" => '&#21107;',
        "\xD1\xF6" => '&#21119;',
        "\xD1\xF7" => '&#21117;',
        "\xD1\xF8" => '&#21133;',
        "\xD1\xF9" => '&#21140;',
        "\xD1\xFA" => '&#21138;',
        "\xD1\xFB" => '&#21105;',
        "\xD1\xFC" => '&#21128;',
        "\xD1\xFD" => '&#21137;',
        "\xD1\xFE" => '&#36776;',
        "\xD2\xA1" => '&#36775;',
        "\xD2\xA2" => '&#21164;',
        "\xD2\xA3" => '&#21165;',
        "\xD2\xA4" => '&#21180;',
        "\xD2\xA5" => '&#21173;',
        "\xD2\xA6" => '&#21185;',
        "\xD2\xA7" => '&#21197;',
        "\xD2\xA8" => '&#21207;',
        "\xD2\xA9" => '&#21214;',
        "\xD2\xAA" => '&#21219;',
        "\xD2\xAB" => '&#21222;',
        "\xD2\xAC" => '&#39149;',
        "\xD2\xAD" => '&#21216;',
        "\xD2\xAE" => '&#21235;',
        "\xD2\xAF" => '&#21237;',
        "\xD2\xB0" => '&#21240;',
        "\xD2\xB1" => '&#21241;',
        "\xD2\xB2" => '&#21254;',
        "\xD2\xB3" => '&#21256;',
        "\xD2\xB4" => '&#30008;',
        "\xD2\xB5" => '&#21261;',
        "\xD2\xB6" => '&#21264;',
        "\xD2\xB7" => '&#21263;',
        "\xD2\xB8" => '&#21269;',
        "\xD2\xB9" => '&#21274;',
        "\xD2\xBA" => '&#21283;',
        "\xD2\xBB" => '&#21295;',
        "\xD2\xBC" => '&#21297;',
        "\xD2\xBD" => '&#21299;',
        "\xD2\xBE" => '&#21304;',
        "\xD2\xBF" => '&#21312;',
        "\xD2\xC0" => '&#21318;',
        "\xD2\xC1" => '&#21317;',
        "\xD2\xC2" => '&#19991;',
        "\xD2\xC3" => '&#21321;',
        "\xD2\xC4" => '&#21325;',
        "\xD2\xC5" => '&#20950;',
        "\xD2\xC6" => '&#21342;',
        "\xD2\xC7" => '&#21353;',
        "\xD2\xC8" => '&#21358;',
        "\xD2\xC9" => '&#22808;',
        "\xD2\xCA" => '&#21371;',
        "\xD2\xCB" => '&#21367;',
        "\xD2\xCC" => '&#21378;',
        "\xD2\xCD" => '&#21398;',
        "\xD2\xCE" => '&#21408;',
        "\xD2\xCF" => '&#21414;',
        "\xD2\xD0" => '&#21413;',
        "\xD2\xD1" => '&#21422;',
        "\xD2\xD2" => '&#21424;',
        "\xD2\xD3" => '&#21430;',
        "\xD2\xD4" => '&#21443;',
        "\xD2\xD5" => '&#31762;',
        "\xD2\xD6" => '&#38617;',
        "\xD2\xD7" => '&#21471;',
        "\xD2\xD8" => '&#26364;',
        "\xD2\xD9" => '&#29166;',
        "\xD2\xDA" => '&#21486;',
        "\xD2\xDB" => '&#21480;',
        "\xD2\xDC" => '&#21485;',
        "\xD2\xDD" => '&#21498;',
        "\xD2\xDE" => '&#21505;',
        "\xD2\xDF" => '&#21565;',
        "\xD2\xE0" => '&#21568;',
        "\xD2\xE1" => '&#21548;',
        "\xD2\xE2" => '&#21549;',
        "\xD2\xE3" => '&#21564;',
        "\xD2\xE4" => '&#21550;',
        "\xD2\xE5" => '&#21558;',
        "\xD2\xE6" => '&#21545;',
        "\xD2\xE7" => '&#21533;',
        "\xD2\xE8" => '&#21582;',
        "\xD2\xE9" => '&#21647;',
        "\xD2\xEA" => '&#21621;',
        "\xD2\xEB" => '&#21646;',
        "\xD2\xEC" => '&#21599;',
        "\xD2\xED" => '&#21617;',
        "\xD2\xEE" => '&#21623;',
        "\xD2\xEF" => '&#21616;',
        "\xD2\xF0" => '&#21650;',
        "\xD2\xF1" => '&#21627;',
        "\xD2\xF2" => '&#21632;',
        "\xD2\xF3" => '&#21622;',
        "\xD2\xF4" => '&#21636;',
        "\xD2\xF5" => '&#21648;',
        "\xD2\xF6" => '&#21638;',
        "\xD2\xF7" => '&#21703;',
        "\xD2\xF8" => '&#21666;',
        "\xD2\xF9" => '&#21688;',
        "\xD2\xFA" => '&#21669;',
        "\xD2\xFB" => '&#21676;',
        "\xD2\xFC" => '&#21700;',
        "\xD2\xFD" => '&#21704;',
        "\xD2\xFE" => '&#21672;',
        "\xD3\xA1" => '&#21675;',
        "\xD3\xA2" => '&#21698;',
        "\xD3\xA3" => '&#21668;',
        "\xD3\xA4" => '&#21694;',
        "\xD3\xA5" => '&#21692;',
        "\xD3\xA6" => '&#21720;',
        "\xD3\xA7" => '&#21733;',
        "\xD3\xA8" => '&#21734;',
        "\xD3\xA9" => '&#21775;',
        "\xD3\xAA" => '&#21780;',
        "\xD3\xAB" => '&#21757;',
        "\xD3\xAC" => '&#21742;',
        "\xD3\xAD" => '&#21741;',
        "\xD3\xAE" => '&#21754;',
        "\xD3\xAF" => '&#21730;',
        "\xD3\xB0" => '&#21817;',
        "\xD3\xB1" => '&#21824;',
        "\xD3\xB2" => '&#21859;',
        "\xD3\xB3" => '&#21836;',
        "\xD3\xB4" => '&#21806;',
        "\xD3\xB5" => '&#21852;',
        "\xD3\xB6" => '&#21829;',
        "\xD3\xB7" => '&#21846;',
        "\xD3\xB8" => '&#21847;',
        "\xD3\xB9" => '&#21816;',
        "\xD3\xBA" => '&#21811;',
        "\xD3\xBB" => '&#21853;',
        "\xD3\xBC" => '&#21913;',
        "\xD3\xBD" => '&#21888;',
        "\xD3\xBE" => '&#21679;',
        "\xD3\xBF" => '&#21898;',
        "\xD3\xC0" => '&#21919;',
        "\xD3\xC1" => '&#21883;',
        "\xD3\xC2" => '&#21886;',
        "\xD3\xC3" => '&#21912;',
        "\xD3\xC4" => '&#21918;',
        "\xD3\xC5" => '&#21934;',
        "\xD3\xC6" => '&#21884;',
        "\xD3\xC7" => '&#21891;',
        "\xD3\xC8" => '&#21929;',
        "\xD3\xC9" => '&#21895;',
        "\xD3\xCA" => '&#21928;',
        "\xD3\xCB" => '&#21978;',
        "\xD3\xCC" => '&#21957;',
        "\xD3\xCD" => '&#21983;',
        "\xD3\xCE" => '&#21956;',
        "\xD3\xCF" => '&#21980;',
        "\xD3\xD0" => '&#21988;',
        "\xD3\xD1" => '&#21972;',
        "\xD3\xD2" => '&#22036;',
        "\xD3\xD3" => '&#22007;',
        "\xD3\xD4" => '&#22038;',
        "\xD3\xD5" => '&#22014;',
        "\xD3\xD6" => '&#22013;',
        "\xD3\xD7" => '&#22043;',
        "\xD3\xD8" => '&#22009;',
        "\xD3\xD9" => '&#22094;',
        "\xD3\xDA" => '&#22096;',
        "\xD3\xDB" => '&#29151;',
        "\xD3\xDC" => '&#22068;',
        "\xD3\xDD" => '&#22070;',
        "\xD3\xDE" => '&#22066;',
        "\xD3\xDF" => '&#22072;',
        "\xD3\xE0" => '&#22123;',
        "\xD3\xE1" => '&#22116;',
        "\xD3\xE2" => '&#22063;',
        "\xD3\xE3" => '&#22124;',
        "\xD3\xE4" => '&#22122;',
        "\xD3\xE5" => '&#22150;',
        "\xD3\xE6" => '&#22144;',
        "\xD3\xE7" => '&#22154;',
        "\xD3\xE8" => '&#22176;',
        "\xD3\xE9" => '&#22164;',
        "\xD3\xEA" => '&#22159;',
        "\xD3\xEB" => '&#22181;',
        "\xD3\xEC" => '&#22190;',
        "\xD3\xED" => '&#22198;',
        "\xD3\xEE" => '&#22196;',
        "\xD3\xEF" => '&#22210;',
        "\xD3\xF0" => '&#22204;',
        "\xD3\xF1" => '&#22209;',
        "\xD3\xF2" => '&#22211;',
        "\xD3\xF3" => '&#22208;',
        "\xD3\xF4" => '&#22216;',
        "\xD3\xF5" => '&#22222;',
        "\xD3\xF6" => '&#22225;',
        "\xD3\xF7" => '&#22227;',
        "\xD3\xF8" => '&#22231;',
        "\xD3\xF9" => '&#22254;',
        "\xD3\xFA" => '&#22265;',
        "\xD3\xFB" => '&#22272;',
        "\xD3\xFC" => '&#22271;',
        "\xD3\xFD" => '&#22276;',
        "\xD3\xFE" => '&#22281;',
        "\xD4\xA1" => '&#22280;',
        "\xD4\xA2" => '&#22283;',
        "\xD4\xA3" => '&#22285;',
        "\xD4\xA4" => '&#22291;',
        "\xD4\xA5" => '&#22296;',
        "\xD4\xA6" => '&#22294;',
        "\xD4\xA7" => '&#21959;',
        "\xD4\xA8" => '&#22300;',
        "\xD4\xA9" => '&#22310;',
        "\xD4\xAA" => '&#22327;',
        "\xD4\xAB" => '&#22328;',
        "\xD4\xAC" => '&#22350;',
        "\xD4\xAD" => '&#22331;',
        "\xD4\xAE" => '&#22336;',
        "\xD4\xAF" => '&#22351;',
        "\xD4\xB0" => '&#22377;',
        "\xD4\xB1" => '&#22464;',
        "\xD4\xB2" => '&#22408;',
        "\xD4\xB3" => '&#22369;',
        "\xD4\xB4" => '&#22399;',
        "\xD4\xB5" => '&#22409;',
        "\xD4\xB6" => '&#22419;',
        "\xD4\xB7" => '&#22432;',
        "\xD4\xB8" => '&#22451;',
        "\xD4\xB9" => '&#22436;',
        "\xD4\xBA" => '&#22442;',
        "\xD4\xBB" => '&#22448;',
        "\xD4\xBC" => '&#22467;',
        "\xD4\xBD" => '&#22470;',
        "\xD4\xBE" => '&#22484;',
        "\xD4\xBF" => '&#22482;',
        "\xD4\xC0" => '&#22483;',
        "\xD4\xC1" => '&#22538;',
        "\xD4\xC2" => '&#22486;',
        "\xD4\xC3" => '&#22499;',
        "\xD4\xC4" => '&#22539;',
        "\xD4\xC5" => '&#22553;',
        "\xD4\xC6" => '&#22557;',
        "\xD4\xC7" => '&#22642;',
        "\xD4\xC8" => '&#22561;',
        "\xD4\xC9" => '&#22626;',
        "\xD4\xCA" => '&#22603;',
        "\xD4\xCB" => '&#22640;',
        "\xD4\xCC" => '&#27584;',
        "\xD4\xCD" => '&#22610;',
        "\xD4\xCE" => '&#22589;',
        "\xD4\xCF" => '&#22649;',
        "\xD4\xD0" => '&#22661;',
        "\xD4\xD1" => '&#22713;',
        "\xD4\xD2" => '&#22687;',
        "\xD4\xD3" => '&#22699;',
        "\xD4\xD4" => '&#22714;',
        "\xD4\xD5" => '&#22750;',
        "\xD4\xD6" => '&#22715;',
        "\xD4\xD7" => '&#22712;',
        "\xD4\xD8" => '&#22702;',
        "\xD4\xD9" => '&#22725;',
        "\xD4\xDA" => '&#22739;',
        "\xD4\xDB" => '&#22737;',
        "\xD4\xDC" => '&#22743;',
        "\xD4\xDD" => '&#22745;',
        "\xD4\xDE" => '&#22744;',
        "\xD4\xDF" => '&#22757;',
        "\xD4\xE0" => '&#22748;',
        "\xD4\xE1" => '&#22756;',
        "\xD4\xE2" => '&#22751;',
        "\xD4\xE3" => '&#22767;',
        "\xD4\xE4" => '&#22778;',
        "\xD4\xE5" => '&#22777;',
        "\xD4\xE6" => '&#22779;',
        "\xD4\xE7" => '&#22780;',
        "\xD4\xE8" => '&#22781;',
        "\xD4\xE9" => '&#22786;',
        "\xD4\xEA" => '&#22794;',
        "\xD4\xEB" => '&#22800;',
        "\xD4\xEC" => '&#22811;',
        "\xD4\xED" => '&#26790;',
        "\xD4\xEE" => '&#22821;',
        "\xD4\xEF" => '&#22828;',
        "\xD4\xF0" => '&#22829;',
        "\xD4\xF1" => '&#22834;',
        "\xD4\xF2" => '&#22840;',
        "\xD4\xF3" => '&#22846;',
        "\xD4\xF4" => '&#31442;',
        "\xD4\xF5" => '&#22869;',
        "\xD4\xF6" => '&#22864;',
        "\xD4\xF7" => '&#22862;',
        "\xD4\xF8" => '&#22874;',
        "\xD4\xF9" => '&#22872;',
        "\xD4\xFA" => '&#22882;',
        "\xD4\xFB" => '&#22880;',
        "\xD4\xFC" => '&#22887;',
        "\xD4\xFD" => '&#22892;',
        "\xD4\xFE" => '&#22889;',
        "\xD5\xA1" => '&#22904;',
        "\xD5\xA2" => '&#22913;',
        "\xD5\xA3" => '&#22941;',
        "\xD5\xA4" => '&#20318;',
        "\xD5\xA5" => '&#20395;',
        "\xD5\xA6" => '&#22947;',
        "\xD5\xA7" => '&#22962;',
        "\xD5\xA8" => '&#22982;',
        "\xD5\xA9" => '&#23016;',
        "\xD5\xAA" => '&#23004;',
        "\xD5\xAB" => '&#22925;',
        "\xD5\xAC" => '&#23001;',
        "\xD5\xAD" => '&#23002;',
        "\xD5\xAE" => '&#23077;',
        "\xD5\xAF" => '&#23071;',
        "\xD5\xB0" => '&#23057;',
        "\xD5\xB1" => '&#23068;',
        "\xD5\xB2" => '&#23049;',
        "\xD5\xB3" => '&#23066;',
        "\xD5\xB4" => '&#23104;',
        "\xD5\xB5" => '&#23148;',
        "\xD5\xB6" => '&#23113;',
        "\xD5\xB7" => '&#23093;',
        "\xD5\xB8" => '&#23094;',
        "\xD5\xB9" => '&#23138;',
        "\xD5\xBA" => '&#23146;',
        "\xD5\xBB" => '&#23194;',
        "\xD5\xBC" => '&#23228;',
        "\xD5\xBD" => '&#23230;',
        "\xD5\xBE" => '&#23243;',
        "\xD5\xBF" => '&#23234;',
        "\xD5\xC0" => '&#23229;',
        "\xD5\xC1" => '&#23267;',
        "\xD5\xC2" => '&#23255;',
        "\xD5\xC3" => '&#23270;',
        "\xD5\xC4" => '&#23273;',
        "\xD5\xC5" => '&#23254;',
        "\xD5\xC6" => '&#23290;',
        "\xD5\xC7" => '&#23291;',
        "\xD5\xC8" => '&#23308;',
        "\xD5\xC9" => '&#23307;',
        "\xD5\xCA" => '&#23318;',
        "\xD5\xCB" => '&#23346;',
        "\xD5\xCC" => '&#23248;',
        "\xD5\xCD" => '&#23338;',
        "\xD5\xCE" => '&#23350;',
        "\xD5\xCF" => '&#23358;',
        "\xD5\xD0" => '&#23363;',
        "\xD5\xD1" => '&#23365;',
        "\xD5\xD2" => '&#23360;',
        "\xD5\xD3" => '&#23377;',
        "\xD5\xD4" => '&#23381;',
        "\xD5\xD5" => '&#23386;',
        "\xD5\xD6" => '&#23387;',
        "\xD5\xD7" => '&#23397;',
        "\xD5\xD8" => '&#23401;',
        "\xD5\xD9" => '&#23408;',
        "\xD5\xDA" => '&#23411;',
        "\xD5\xDB" => '&#23413;',
        "\xD5\xDC" => '&#23416;',
        "\xD5\xDD" => '&#25992;',
        "\xD5\xDE" => '&#23418;',
        "\xD5\xDF" => '&#23424;',
        "\xD5\xE0" => '&#23427;',
        "\xD5\xE1" => '&#23462;',
        "\xD5\xE2" => '&#23480;',
        "\xD5\xE3" => '&#23491;',
        "\xD5\xE4" => '&#23495;',
        "\xD5\xE5" => '&#23497;',
        "\xD5\xE6" => '&#23508;',
        "\xD5\xE7" => '&#23504;',
        "\xD5\xE8" => '&#23524;',
        "\xD5\xE9" => '&#23526;',
        "\xD5\xEA" => '&#23522;',
        "\xD5\xEB" => '&#23518;',
        "\xD5\xEC" => '&#23525;',
        "\xD5\xED" => '&#23531;',
        "\xD5\xEE" => '&#23536;',
        "\xD5\xEF" => '&#23542;',
        "\xD5\xF0" => '&#23539;',
        "\xD5\xF1" => '&#23557;',
        "\xD5\xF2" => '&#23559;',
        "\xD5\xF3" => '&#23560;',
        "\xD5\xF4" => '&#23565;',
        "\xD5\xF5" => '&#23571;',
        "\xD5\xF6" => '&#23584;',
        "\xD5\xF7" => '&#23586;',
        "\xD5\xF8" => '&#23592;',
        "\xD5\xF9" => '&#23608;',
        "\xD5\xFA" => '&#23609;',
        "\xD5\xFB" => '&#23617;',
        "\xD5\xFC" => '&#23622;',
        "\xD5\xFD" => '&#23630;',
        "\xD5\xFE" => '&#23635;',
        "\xD6\xA1" => '&#23632;',
        "\xD6\xA2" => '&#23631;',
        "\xD6\xA3" => '&#23409;',
        "\xD6\xA4" => '&#23660;',
        "\xD6\xA5" => '&#23662;',
        "\xD6\xA6" => '&#20066;',
        "\xD6\xA7" => '&#23670;',
        "\xD6\xA8" => '&#23673;',
        "\xD6\xA9" => '&#23692;',
        "\xD6\xAA" => '&#23697;',
        "\xD6\xAB" => '&#23700;',
        "\xD6\xAC" => '&#22939;',
        "\xD6\xAD" => '&#23723;',
        "\xD6\xAE" => '&#23739;',
        "\xD6\xAF" => '&#23734;',
        "\xD6\xB0" => '&#23740;',
        "\xD6\xB1" => '&#23735;',
        "\xD6\xB2" => '&#23749;',
        "\xD6\xB3" => '&#23742;',
        "\xD6\xB4" => '&#23751;',
        "\xD6\xB5" => '&#23769;',
        "\xD6\xB6" => '&#23785;',
        "\xD6\xB7" => '&#23805;',
        "\xD6\xB8" => '&#23802;',
        "\xD6\xB9" => '&#23789;',
        "\xD6\xBA" => '&#23948;',
        "\xD6\xBB" => '&#23786;',
        "\xD6\xBC" => '&#23819;',
        "\xD6\xBD" => '&#23829;',
        "\xD6\xBE" => '&#23831;',
        "\xD6\xBF" => '&#23900;',
        "\xD6\xC0" => '&#23839;',
        "\xD6\xC1" => '&#23835;',
        "\xD6\xC2" => '&#23825;',
        "\xD6\xC3" => '&#23828;',
        "\xD6\xC4" => '&#23842;',
        "\xD6\xC5" => '&#23834;',
        "\xD6\xC6" => '&#23833;',
        "\xD6\xC7" => '&#23832;',
        "\xD6\xC8" => '&#23884;',
        "\xD6\xC9" => '&#23890;',
        "\xD6\xCA" => '&#23886;',
        "\xD6\xCB" => '&#23883;',
        "\xD6\xCC" => '&#23916;',
        "\xD6\xCD" => '&#23923;',
        "\xD6\xCE" => '&#23926;',
        "\xD6\xCF" => '&#23943;',
        "\xD6\xD0" => '&#23940;',
        "\xD6\xD1" => '&#23938;',
        "\xD6\xD2" => '&#23970;',
        "\xD6\xD3" => '&#23965;',
        "\xD6\xD4" => '&#23980;',
        "\xD6\xD5" => '&#23982;',
        "\xD6\xD6" => '&#23997;',
        "\xD6\xD7" => '&#23952;',
        "\xD6\xD8" => '&#23991;',
        "\xD6\xD9" => '&#23996;',
        "\xD6\xDA" => '&#24009;',
        "\xD6\xDB" => '&#24013;',
        "\xD6\xDC" => '&#24019;',
        "\xD6\xDD" => '&#24018;',
        "\xD6\xDE" => '&#24022;',
        "\xD6\xDF" => '&#24027;',
        "\xD6\xE0" => '&#24043;',
        "\xD6\xE1" => '&#24050;',
        "\xD6\xE2" => '&#24053;',
        "\xD6\xE3" => '&#24075;',
        "\xD6\xE4" => '&#24090;',
        "\xD6\xE5" => '&#24089;',
        "\xD6\xE6" => '&#24081;',
        "\xD6\xE7" => '&#24091;',
        "\xD6\xE8" => '&#24118;',
        "\xD6\xE9" => '&#24119;',
        "\xD6\xEA" => '&#24132;',
        "\xD6\xEB" => '&#24131;',
        "\xD6\xEC" => '&#24128;',
        "\xD6\xED" => '&#24142;',
        "\xD6\xEE" => '&#24151;',
        "\xD6\xEF" => '&#24148;',
        "\xD6\xF0" => '&#24159;',
        "\xD6\xF1" => '&#24162;',
        "\xD6\xF2" => '&#24164;',
        "\xD6\xF3" => '&#24135;',
        "\xD6\xF4" => '&#24181;',
        "\xD6\xF5" => '&#24182;',
        "\xD6\xF6" => '&#24186;',
        "\xD6\xF7" => '&#40636;',
        "\xD6\xF8" => '&#24191;',
        "\xD6\xF9" => '&#24224;',
        "\xD6\xFA" => '&#24257;',
        "\xD6\xFB" => '&#24258;',
        "\xD6\xFC" => '&#24264;',
        "\xD6\xFD" => '&#24272;',
        "\xD6\xFE" => '&#24271;',
        "\xD7\xA1" => '&#24278;',
        "\xD7\xA2" => '&#24291;',
        "\xD7\xA3" => '&#24285;',
        "\xD7\xA4" => '&#24282;',
        "\xD7\xA5" => '&#24283;',
        "\xD7\xA6" => '&#24290;',
        "\xD7\xA7" => '&#24289;',
        "\xD7\xA8" => '&#24296;',
        "\xD7\xA9" => '&#24297;',
        "\xD7\xAA" => '&#24300;',
        "\xD7\xAB" => '&#24305;',
        "\xD7\xAC" => '&#24307;',
        "\xD7\xAD" => '&#24304;',
        "\xD7\xAE" => '&#24308;',
        "\xD7\xAF" => '&#24312;',
        "\xD7\xB0" => '&#24318;',
        "\xD7\xB1" => '&#24323;',
        "\xD7\xB2" => '&#24329;',
        "\xD7\xB3" => '&#24413;',
        "\xD7\xB4" => '&#24412;',
        "\xD7\xB5" => '&#24331;',
        "\xD7\xB6" => '&#24337;',
        "\xD7\xB7" => '&#24342;',
        "\xD7\xB8" => '&#24361;',
        "\xD7\xB9" => '&#24365;',
        "\xD7\xBA" => '&#24376;',
        "\xD7\xBB" => '&#24385;',
        "\xD7\xBC" => '&#24392;',
        "\xD7\xBD" => '&#24396;',
        "\xD7\xBE" => '&#24398;',
        "\xD7\xBF" => '&#24367;',
        "\xD7\xC0" => '&#24401;',
        "\xD7\xC1" => '&#24406;',
        "\xD7\xC2" => '&#24407;',
        "\xD7\xC3" => '&#24409;',
        "\xD7\xC4" => '&#24417;',
        "\xD7\xC5" => '&#24429;',
        "\xD7\xC6" => '&#24435;',
        "\xD7\xC7" => '&#24439;',
        "\xD7\xC8" => '&#24451;',
        "\xD7\xC9" => '&#24450;',
        "\xD7\xCA" => '&#24447;',
        "\xD7\xCB" => '&#24458;',
        "\xD7\xCC" => '&#24456;',
        "\xD7\xCD" => '&#24465;',
        "\xD7\xCE" => '&#24455;',
        "\xD7\xCF" => '&#24478;',
        "\xD7\xD0" => '&#24473;',
        "\xD7\xD1" => '&#24472;',
        "\xD7\xD2" => '&#24480;',
        "\xD7\xD3" => '&#24488;',
        "\xD7\xD4" => '&#24493;',
        "\xD7\xD5" => '&#24508;',
        "\xD7\xD6" => '&#24534;',
        "\xD7\xD7" => '&#24571;',
        "\xD7\xD8" => '&#24548;',
        "\xD7\xD9" => '&#24568;',
        "\xD7\xDA" => '&#24561;',
        "\xD7\xDB" => '&#24541;',
        "\xD7\xDC" => '&#24755;',
        "\xD7\xDD" => '&#24575;',
        "\xD7\xDE" => '&#24609;',
        "\xD7\xDF" => '&#24672;',
        "\xD7\xE0" => '&#24601;',
        "\xD7\xE1" => '&#24592;',
        "\xD7\xE2" => '&#24617;',
        "\xD7\xE3" => '&#24590;',
        "\xD7\xE4" => '&#24625;',
        "\xD7\xE5" => '&#24603;',
        "\xD7\xE6" => '&#24597;',
        "\xD7\xE7" => '&#24619;',
        "\xD7\xE8" => '&#24614;',
        "\xD7\xE9" => '&#24591;',
        "\xD7\xEA" => '&#24634;',
        "\xD7\xEB" => '&#24666;',
        "\xD7\xEC" => '&#24641;',
        "\xD7\xED" => '&#24682;',
        "\xD7\xEE" => '&#24695;',
        "\xD7\xEF" => '&#24671;',
        "\xD7\xF0" => '&#24650;',
        "\xD7\xF1" => '&#24646;',
        "\xD7\xF2" => '&#24653;',
        "\xD7\xF3" => '&#24675;',
        "\xD7\xF4" => '&#24643;',
        "\xD7\xF5" => '&#24676;',
        "\xD7\xF6" => '&#24642;',
        "\xD7\xF7" => '&#24684;',
        "\xD7\xF8" => '&#24683;',
        "\xD7\xF9" => '&#24665;',
        "\xD7\xFA" => '&#24705;',
        "\xD7\xFB" => '&#24717;',
        "\xD7\xFC" => '&#24807;',
        "\xD7\xFD" => '&#24707;',
        "\xD7\xFE" => '&#24730;',
        "\xD8\xA1" => '&#24708;',
        "\xD8\xA2" => '&#24731;',
        "\xD8\xA3" => '&#24726;',
        "\xD8\xA4" => '&#24727;',
        "\xD8\xA5" => '&#24722;',
        "\xD8\xA6" => '&#24743;',
        "\xD8\xA7" => '&#24715;',
        "\xD8\xA8" => '&#24801;',
        "\xD8\xA9" => '&#24760;',
        "\xD8\xAA" => '&#24800;',
        "\xD8\xAB" => '&#24787;',
        "\xD8\xAC" => '&#24756;',
        "\xD8\xAD" => '&#24560;',
        "\xD8\xAE" => '&#24765;',
        "\xD8\xAF" => '&#24774;',
        "\xD8\xB0" => '&#24757;',
        "\xD8\xB1" => '&#24792;',
        "\xD8\xB2" => '&#24909;',
        "\xD8\xB3" => '&#24853;',
        "\xD8\xB4" => '&#24838;',
        "\xD8\xB5" => '&#24822;',
        "\xD8\xB6" => '&#24823;',
        "\xD8\xB7" => '&#24832;',
        "\xD8\xB8" => '&#24820;',
        "\xD8\xB9" => '&#24826;',
        "\xD8\xBA" => '&#24835;',
        "\xD8\xBB" => '&#24865;',
        "\xD8\xBC" => '&#24827;',
        "\xD8\xBD" => '&#24817;',
        "\xD8\xBE" => '&#24845;',
        "\xD8\xBF" => '&#24846;',
        "\xD8\xC0" => '&#24903;',
        "\xD8\xC1" => '&#24894;',
        "\xD8\xC2" => '&#24872;',
        "\xD8\xC3" => '&#24871;',
        "\xD8\xC4" => '&#24906;',
        "\xD8\xC5" => '&#24895;',
        "\xD8\xC6" => '&#24892;',
        "\xD8\xC7" => '&#24876;',
        "\xD8\xC8" => '&#24884;',
        "\xD8\xC9" => '&#24893;',
        "\xD8\xCA" => '&#24898;',
        "\xD8\xCB" => '&#24900;',
        "\xD8\xCC" => '&#24947;',
        "\xD8\xCD" => '&#24951;',
        "\xD8\xCE" => '&#24920;',
        "\xD8\xCF" => '&#24921;',
        "\xD8\xD0" => '&#24922;',
        "\xD8\xD1" => '&#24939;',
        "\xD8\xD2" => '&#24948;',
        "\xD8\xD3" => '&#24943;',
        "\xD8\xD4" => '&#24933;',
        "\xD8\xD5" => '&#24945;',
        "\xD8\xD6" => '&#24927;',
        "\xD8\xD7" => '&#24925;',
        "\xD8\xD8" => '&#24915;',
        "\xD8\xD9" => '&#24949;',
        "\xD8\xDA" => '&#24985;',
        "\xD8\xDB" => '&#24982;',
        "\xD8\xDC" => '&#24967;',
        "\xD8\xDD" => '&#25004;',
        "\xD8\xDE" => '&#24980;',
        "\xD8\xDF" => '&#24986;',
        "\xD8\xE0" => '&#24970;',
        "\xD8\xE1" => '&#24977;',
        "\xD8\xE2" => '&#25003;',
        "\xD8\xE3" => '&#25006;',
        "\xD8\xE4" => '&#25036;',
        "\xD8\xE5" => '&#25034;',
        "\xD8\xE6" => '&#25033;',
        "\xD8\xE7" => '&#25079;',
        "\xD8\xE8" => '&#25032;',
        "\xD8\xE9" => '&#25027;',
        "\xD8\xEA" => '&#25030;',
        "\xD8\xEB" => '&#25018;',
        "\xD8\xEC" => '&#25035;',
        "\xD8\xED" => '&#32633;',
        "\xD8\xEE" => '&#25037;',
        "\xD8\xEF" => '&#25062;',
        "\xD8\xF0" => '&#25059;',
        "\xD8\xF1" => '&#25078;',
        "\xD8\xF2" => '&#25082;',
        "\xD8\xF3" => '&#25076;',
        "\xD8\xF4" => '&#25087;',
        "\xD8\xF5" => '&#25085;',
        "\xD8\xF6" => '&#25084;',
        "\xD8\xF7" => '&#25086;',
        "\xD8\xF8" => '&#25088;',
        "\xD8\xF9" => '&#25096;',
        "\xD8\xFA" => '&#25097;',
        "\xD8\xFB" => '&#25101;',
        "\xD8\xFC" => '&#25100;',
        "\xD8\xFD" => '&#25108;',
        "\xD8\xFE" => '&#25115;',
        "\xD9\xA1" => '&#25118;',
        "\xD9\xA2" => '&#25121;',
        "\xD9\xA3" => '&#25130;',
        "\xD9\xA4" => '&#25134;',
        "\xD9\xA5" => '&#25136;',
        "\xD9\xA6" => '&#25138;',
        "\xD9\xA7" => '&#25139;',
        "\xD9\xA8" => '&#25153;',
        "\xD9\xA9" => '&#25166;',
        "\xD9\xAA" => '&#25182;',
        "\xD9\xAB" => '&#25187;',
        "\xD9\xAC" => '&#25179;',
        "\xD9\xAD" => '&#25184;',
        "\xD9\xAE" => '&#25192;',
        "\xD9\xAF" => '&#25212;',
        "\xD9\xB0" => '&#25218;',
        "\xD9\xB1" => '&#25225;',
        "\xD9\xB2" => '&#25214;',
        "\xD9\xB3" => '&#25234;',
        "\xD9\xB4" => '&#25235;',
        "\xD9\xB5" => '&#25238;',
        "\xD9\xB6" => '&#25300;',
        "\xD9\xB7" => '&#25219;',
        "\xD9\xB8" => '&#25236;',
        "\xD9\xB9" => '&#25303;',
        "\xD9\xBA" => '&#25297;',
        "\xD9\xBB" => '&#25275;',
        "\xD9\xBC" => '&#25295;',
        "\xD9\xBD" => '&#25343;',
        "\xD9\xBE" => '&#25286;',
        "\xD9\xBF" => '&#25812;',
        "\xD9\xC0" => '&#25288;',
        "\xD9\xC1" => '&#25308;',
        "\xD9\xC2" => '&#25292;',
        "\xD9\xC3" => '&#25290;',
        "\xD9\xC4" => '&#25282;',
        "\xD9\xC5" => '&#25287;',
        "\xD9\xC6" => '&#25243;',
        "\xD9\xC7" => '&#25289;',
        "\xD9\xC8" => '&#25356;',
        "\xD9\xC9" => '&#25326;',
        "\xD9\xCA" => '&#25329;',
        "\xD9\xCB" => '&#25383;',
        "\xD9\xCC" => '&#25346;',
        "\xD9\xCD" => '&#25352;',
        "\xD9\xCE" => '&#25327;',
        "\xD9\xCF" => '&#25333;',
        "\xD9\xD0" => '&#25424;',
        "\xD9\xD1" => '&#25406;',
        "\xD9\xD2" => '&#25421;',
        "\xD9\xD3" => '&#25628;',
        "\xD9\xD4" => '&#25423;',
        "\xD9\xD5" => '&#25494;',
        "\xD9\xD6" => '&#25486;',
        "\xD9\xD7" => '&#25472;',
        "\xD9\xD8" => '&#25515;',
        "\xD9\xD9" => '&#25462;',
        "\xD9\xDA" => '&#25507;',
        "\xD9\xDB" => '&#25487;',
        "\xD9\xDC" => '&#25481;',
        "\xD9\xDD" => '&#25503;',
        "\xD9\xDE" => '&#25525;',
        "\xD9\xDF" => '&#25451;',
        "\xD9\xE0" => '&#25449;',
        "\xD9\xE1" => '&#25534;',
        "\xD9\xE2" => '&#25577;',
        "\xD9\xE3" => '&#25536;',
        "\xD9\xE4" => '&#25542;',
        "\xD9\xE5" => '&#25571;',
        "\xD9\xE6" => '&#25545;',
        "\xD9\xE7" => '&#25554;',
        "\xD9\xE8" => '&#25590;',
        "\xD9\xE9" => '&#25540;',
        "\xD9\xEA" => '&#25622;',
        "\xD9\xEB" => '&#25652;',
        "\xD9\xEC" => '&#25606;',
        "\xD9\xED" => '&#25619;',
        "\xD9\xEE" => '&#25638;',
        "\xD9\xEF" => '&#25654;',
        "\xD9\xF0" => '&#25885;',
        "\xD9\xF1" => '&#25623;',
        "\xD9\xF2" => '&#25640;',
        "\xD9\xF3" => '&#25615;',
        "\xD9\xF4" => '&#25703;',
        "\xD9\xF5" => '&#25711;',
        "\xD9\xF6" => '&#25718;',
        "\xD9\xF7" => '&#25678;',
        "\xD9\xF8" => '&#25898;',
        "\xD9\xF9" => '&#25749;',
        "\xD9\xFA" => '&#25747;',
        "\xD9\xFB" => '&#25765;',
        "\xD9\xFC" => '&#25769;',
        "\xD9\xFD" => '&#25736;',
        "\xD9\xFE" => '&#25788;',
        "\xDA\xA1" => '&#25818;',
        "\xDA\xA2" => '&#25810;',
        "\xDA\xA3" => '&#25797;',
        "\xDA\xA4" => '&#25799;',
        "\xDA\xA5" => '&#25787;',
        "\xDA\xA6" => '&#25816;',
        "\xDA\xA7" => '&#25794;',
        "\xDA\xA8" => '&#25841;',
        "\xDA\xA9" => '&#25831;',
        "\xDA\xAA" => '&#33289;',
        "\xDA\xAB" => '&#25824;',
        "\xDA\xAC" => '&#25825;',
        "\xDA\xAD" => '&#25260;',
        "\xDA\xAE" => '&#25827;',
        "\xDA\xAF" => '&#25839;',
        "\xDA\xB0" => '&#25900;',
        "\xDA\xB1" => '&#25846;',
        "\xDA\xB2" => '&#25844;',
        "\xDA\xB3" => '&#25842;',
        "\xDA\xB4" => '&#25850;',
        "\xDA\xB5" => '&#25856;',
        "\xDA\xB6" => '&#25853;',
        "\xDA\xB7" => '&#25880;',
        "\xDA\xB8" => '&#25884;',
        "\xDA\xB9" => '&#25861;',
        "\xDA\xBA" => '&#25892;',
        "\xDA\xBB" => '&#25891;',
        "\xDA\xBC" => '&#25899;',
        "\xDA\xBD" => '&#25908;',
        "\xDA\xBE" => '&#25909;',
        "\xDA\xBF" => '&#25911;',
        "\xDA\xC0" => '&#25910;',
        "\xDA\xC1" => '&#25912;',
        "\xDA\xC2" => '&#30027;',
        "\xDA\xC3" => '&#25928;',
        "\xDA\xC4" => '&#25942;',
        "\xDA\xC5" => '&#25941;',
        "\xDA\xC6" => '&#25933;',
        "\xDA\xC7" => '&#25944;',
        "\xDA\xC8" => '&#25950;',
        "\xDA\xC9" => '&#25949;',
        "\xDA\xCA" => '&#25970;',
        "\xDA\xCB" => '&#25976;',
        "\xDA\xCC" => '&#25986;',
        "\xDA\xCD" => '&#25987;',
        "\xDA\xCE" => '&#35722;',
        "\xDA\xCF" => '&#26011;',
        "\xDA\xD0" => '&#26015;',
        "\xDA\xD1" => '&#26027;',
        "\xDA\xD2" => '&#26039;',
        "\xDA\xD3" => '&#26051;',
        "\xDA\xD4" => '&#26054;',
        "\xDA\xD5" => '&#26049;',
        "\xDA\xD6" => '&#26052;',
        "\xDA\xD7" => '&#26060;',
        "\xDA\xD8" => '&#26066;',
        "\xDA\xD9" => '&#26075;',
        "\xDA\xDA" => '&#26073;',
        "\xDA\xDB" => '&#26080;',
        "\xDA\xDC" => '&#26081;',
        "\xDA\xDD" => '&#26097;',
        "\xDA\xDE" => '&#26482;',
        "\xDA\xDF" => '&#26122;',
        "\xDA\xE0" => '&#26115;',
        "\xDA\xE1" => '&#26107;',
        "\xDA\xE2" => '&#26483;',
        "\xDA\xE3" => '&#26165;',
        "\xDA\xE4" => '&#26166;',
        "\xDA\xE5" => '&#26164;',
        "\xDA\xE6" => '&#26140;',
        "\xDA\xE7" => '&#26191;',
        "\xDA\xE8" => '&#26180;',
        "\xDA\xE9" => '&#26185;',
        "\xDA\xEA" => '&#26177;',
        "\xDA\xEB" => '&#26206;',
        "\xDA\xEC" => '&#26205;',
        "\xDA\xED" => '&#26212;',
        "\xDA\xEE" => '&#26215;',
        "\xDA\xEF" => '&#26216;',
        "\xDA\xF0" => '&#26207;',
        "\xDA\xF1" => '&#26210;',
        "\xDA\xF2" => '&#26224;',
        "\xDA\xF3" => '&#26243;',
        "\xDA\xF4" => '&#26248;',
        "\xDA\xF5" => '&#26254;',
        "\xDA\xF6" => '&#26249;',
        "\xDA\xF7" => '&#26244;',
        "\xDA\xF8" => '&#26264;',
        "\xDA\xF9" => '&#26269;',
        "\xDA\xFA" => '&#26305;',
        "\xDA\xFB" => '&#26297;',
        "\xDA\xFC" => '&#26313;',
        "\xDA\xFD" => '&#26302;',
        "\xDA\xFE" => '&#26300;',
        "\xDB\xA1" => '&#26308;',
        "\xDB\xA2" => '&#26296;',
        "\xDB\xA3" => '&#26326;',
        "\xDB\xA4" => '&#26330;',
        "\xDB\xA5" => '&#26336;',
        "\xDB\xA6" => '&#26175;',
        "\xDB\xA7" => '&#26342;',
        "\xDB\xA8" => '&#26345;',
        "\xDB\xA9" => '&#26352;',
        "\xDB\xAA" => '&#26357;',
        "\xDB\xAB" => '&#26359;',
        "\xDB\xAC" => '&#26383;',
        "\xDB\xAD" => '&#26390;',
        "\xDB\xAE" => '&#26398;',
        "\xDB\xAF" => '&#26406;',
        "\xDB\xB0" => '&#26407;',
        "\xDB\xB1" => '&#38712;',
        "\xDB\xB2" => '&#26414;',
        "\xDB\xB3" => '&#26431;',
        "\xDB\xB4" => '&#26422;',
        "\xDB\xB5" => '&#26433;',
        "\xDB\xB6" => '&#26424;',
        "\xDB\xB7" => '&#26423;',
        "\xDB\xB8" => '&#26438;',
        "\xDB\xB9" => '&#26462;',
        "\xDB\xBA" => '&#26464;',
        "\xDB\xBB" => '&#26457;',
        "\xDB\xBC" => '&#26467;',
        "\xDB\xBD" => '&#26468;',
        "\xDB\xBE" => '&#26505;',
        "\xDB\xBF" => '&#26480;',
        "\xDB\xC0" => '&#26537;',
        "\xDB\xC1" => '&#26492;',
        "\xDB\xC2" => '&#26474;',
        "\xDB\xC3" => '&#26508;',
        "\xDB\xC4" => '&#26507;',
        "\xDB\xC5" => '&#26534;',
        "\xDB\xC6" => '&#26529;',
        "\xDB\xC7" => '&#26501;',
        "\xDB\xC8" => '&#26551;',
        "\xDB\xC9" => '&#26607;',
        "\xDB\xCA" => '&#26548;',
        "\xDB\xCB" => '&#26604;',
        "\xDB\xCC" => '&#26547;',
        "\xDB\xCD" => '&#26601;',
        "\xDB\xCE" => '&#26552;',
        "\xDB\xCF" => '&#26596;',
        "\xDB\xD0" => '&#26590;',
        "\xDB\xD1" => '&#26589;',
        "\xDB\xD2" => '&#26594;',
        "\xDB\xD3" => '&#26606;',
        "\xDB\xD4" => '&#26553;',
        "\xDB\xD5" => '&#26574;',
        "\xDB\xD6" => '&#26566;',
        "\xDB\xD7" => '&#26599;',
        "\xDB\xD8" => '&#27292;',
        "\xDB\xD9" => '&#26654;',
        "\xDB\xDA" => '&#26694;',
        "\xDB\xDB" => '&#26665;',
        "\xDB\xDC" => '&#26688;',
        "\xDB\xDD" => '&#26701;',
        "\xDB\xDE" => '&#26674;',
        "\xDB\xDF" => '&#26702;',
        "\xDB\xE0" => '&#26803;',
        "\xDB\xE1" => '&#26667;',
        "\xDB\xE2" => '&#26713;',
        "\xDB\xE3" => '&#26723;',
        "\xDB\xE4" => '&#26743;',
        "\xDB\xE5" => '&#26751;',
        "\xDB\xE6" => '&#26783;',
        "\xDB\xE7" => '&#26767;',
        "\xDB\xE8" => '&#26797;',
        "\xDB\xE9" => '&#26772;',
        "\xDB\xEA" => '&#26781;',
        "\xDB\xEB" => '&#26779;',
        "\xDB\xEC" => '&#26755;',
        "\xDB\xED" => '&#27310;',
        "\xDB\xEE" => '&#26809;',
        "\xDB\xEF" => '&#26740;',
        "\xDB\xF0" => '&#26805;',
        "\xDB\xF1" => '&#26784;',
        "\xDB\xF2" => '&#26810;',
        "\xDB\xF3" => '&#26895;',
        "\xDB\xF4" => '&#26765;',
        "\xDB\xF5" => '&#26750;',
        "\xDB\xF6" => '&#26881;',
        "\xDB\xF7" => '&#26826;',
        "\xDB\xF8" => '&#26888;',
        "\xDB\xF9" => '&#26840;',
        "\xDB\xFA" => '&#26914;',
        "\xDB\xFB" => '&#26918;',
        "\xDB\xFC" => '&#26849;',
        "\xDB\xFD" => '&#26892;',
        "\xDB\xFE" => '&#26829;',
        "\xDC\xA1" => '&#26836;',
        "\xDC\xA2" => '&#26855;',
        "\xDC\xA3" => '&#26837;',
        "\xDC\xA4" => '&#26934;',
        "\xDC\xA5" => '&#26898;',
        "\xDC\xA6" => '&#26884;',
        "\xDC\xA7" => '&#26839;',
        "\xDC\xA8" => '&#26851;',
        "\xDC\xA9" => '&#26917;',
        "\xDC\xAA" => '&#26873;',
        "\xDC\xAB" => '&#26848;',
        "\xDC\xAC" => '&#26863;',
        "\xDC\xAD" => '&#26920;',
        "\xDC\xAE" => '&#26922;',
        "\xDC\xAF" => '&#26906;',
        "\xDC\xB0" => '&#26915;',
        "\xDC\xB1" => '&#26913;',
        "\xDC\xB2" => '&#26822;',
        "\xDC\xB3" => '&#27001;',
        "\xDC\xB4" => '&#26999;',
        "\xDC\xB5" => '&#26972;',
        "\xDC\xB6" => '&#27000;',
        "\xDC\xB7" => '&#26987;',
        "\xDC\xB8" => '&#26964;',
        "\xDC\xB9" => '&#27006;',
        "\xDC\xBA" => '&#26990;',
        "\xDC\xBB" => '&#26937;',
        "\xDC\xBC" => '&#26996;',
        "\xDC\xBD" => '&#26941;',
        "\xDC\xBE" => '&#26969;',
        "\xDC\xBF" => '&#26928;',
        "\xDC\xC0" => '&#26977;',
        "\xDC\xC1" => '&#26974;',
        "\xDC\xC2" => '&#26973;',
        "\xDC\xC3" => '&#27009;',
        "\xDC\xC4" => '&#26986;',
        "\xDC\xC5" => '&#27058;',
        "\xDC\xC6" => '&#27054;',
        "\xDC\xC7" => '&#27088;',
        "\xDC\xC8" => '&#27071;',
        "\xDC\xC9" => '&#27073;',
        "\xDC\xCA" => '&#27091;',
        "\xDC\xCB" => '&#27070;',
        "\xDC\xCC" => '&#27086;',
        "\xDC\xCD" => '&#23528;',
        "\xDC\xCE" => '&#27082;',
        "\xDC\xCF" => '&#27101;',
        "\xDC\xD0" => '&#27067;',
        "\xDC\xD1" => '&#27075;',
        "\xDC\xD2" => '&#27047;',
        "\xDC\xD3" => '&#27182;',
        "\xDC\xD4" => '&#27025;',
        "\xDC\xD5" => '&#27040;',
        "\xDC\xD6" => '&#27036;',
        "\xDC\xD7" => '&#27029;',
        "\xDC\xD8" => '&#27060;',
        "\xDC\xD9" => '&#27102;',
        "\xDC\xDA" => '&#27112;',
        "\xDC\xDB" => '&#27138;',
        "\xDC\xDC" => '&#27163;',
        "\xDC\xDD" => '&#27135;',
        "\xDC\xDE" => '&#27402;',
        "\xDC\xDF" => '&#27129;',
        "\xDC\xE0" => '&#27122;',
        "\xDC\xE1" => '&#27111;',
        "\xDC\xE2" => '&#27141;',
        "\xDC\xE3" => '&#27057;',
        "\xDC\xE4" => '&#27166;',
        "\xDC\xE5" => '&#27117;',
        "\xDC\xE6" => '&#27156;',
        "\xDC\xE7" => '&#27115;',
        "\xDC\xE8" => '&#27146;',
        "\xDC\xE9" => '&#27154;',
        "\xDC\xEA" => '&#27329;',
        "\xDC\xEB" => '&#27171;',
        "\xDC\xEC" => '&#27155;',
        "\xDC\xED" => '&#27204;',
        "\xDC\xEE" => '&#27148;',
        "\xDC\xEF" => '&#27250;',
        "\xDC\xF0" => '&#27190;',
        "\xDC\xF1" => '&#27256;',
        "\xDC\xF2" => '&#27207;',
        "\xDC\xF3" => '&#27234;',
        "\xDC\xF4" => '&#27225;',
        "\xDC\xF5" => '&#27238;',
        "\xDC\xF6" => '&#27208;',
        "\xDC\xF7" => '&#27192;',
        "\xDC\xF8" => '&#27170;',
        "\xDC\xF9" => '&#27280;',
        "\xDC\xFA" => '&#27277;',
        "\xDC\xFB" => '&#27296;',
        "\xDC\xFC" => '&#27268;',
        "\xDC\xFD" => '&#27298;',
        "\xDC\xFE" => '&#27299;',
        "\xDD\xA1" => '&#27287;',
        "\xDD\xA2" => '&#34327;',
        "\xDD\xA3" => '&#27323;',
        "\xDD\xA4" => '&#27331;',
        "\xDD\xA5" => '&#27330;',
        "\xDD\xA6" => '&#27320;',
        "\xDD\xA7" => '&#27315;',
        "\xDD\xA8" => '&#27308;',
        "\xDD\xA9" => '&#27358;',
        "\xDD\xAA" => '&#27345;',
        "\xDD\xAB" => '&#27359;',
        "\xDD\xAC" => '&#27306;',
        "\xDD\xAD" => '&#27354;',
        "\xDD\xAE" => '&#27370;',
        "\xDD\xAF" => '&#27387;',
        "\xDD\xB0" => '&#27397;',
        "\xDD\xB1" => '&#34326;',
        "\xDD\xB2" => '&#27386;',
        "\xDD\xB3" => '&#27410;',
        "\xDD\xB4" => '&#27414;',
        "\xDD\xB5" => '&#39729;',
        "\xDD\xB6" => '&#27423;',
        "\xDD\xB7" => '&#27448;',
        "\xDD\xB8" => '&#27447;',
        "\xDD\xB9" => '&#30428;',
        "\xDD\xBA" => '&#27449;',
        "\xDD\xBB" => '&#39150;',
        "\xDD\xBC" => '&#27463;',
        "\xDD\xBD" => '&#27459;',
        "\xDD\xBE" => '&#27465;',
        "\xDD\xBF" => '&#27472;',
        "\xDD\xC0" => '&#27481;',
        "\xDD\xC1" => '&#27476;',
        "\xDD\xC2" => '&#27483;',
        "\xDD\xC3" => '&#27487;',
        "\xDD\xC4" => '&#27489;',
        "\xDD\xC5" => '&#27512;',
        "\xDD\xC6" => '&#27513;',
        "\xDD\xC7" => '&#27519;',
        "\xDD\xC8" => '&#27520;',
        "\xDD\xC9" => '&#27524;',
        "\xDD\xCA" => '&#27523;',
        "\xDD\xCB" => '&#27533;',
        "\xDD\xCC" => '&#27544;',
        "\xDD\xCD" => '&#27541;',
        "\xDD\xCE" => '&#27550;',
        "\xDD\xCF" => '&#27556;',
        "\xDD\xD0" => '&#27562;',
        "\xDD\xD1" => '&#27563;',
        "\xDD\xD2" => '&#27567;',
        "\xDD\xD3" => '&#27570;',
        "\xDD\xD4" => '&#27569;',
        "\xDD\xD5" => '&#27571;',
        "\xDD\xD6" => '&#27575;',
        "\xDD\xD7" => '&#27580;',
        "\xDD\xD8" => '&#27590;',
        "\xDD\xD9" => '&#27595;',
        "\xDD\xDA" => '&#27603;',
        "\xDD\xDB" => '&#27615;',
        "\xDD\xDC" => '&#27628;',
        "\xDD\xDD" => '&#27627;',
        "\xDD\xDE" => '&#27635;',
        "\xDD\xDF" => '&#27631;',
        "\xDD\xE0" => '&#40638;',
        "\xDD\xE1" => '&#27656;',
        "\xDD\xE2" => '&#27667;',
        "\xDD\xE3" => '&#27668;',
        "\xDD\xE4" => '&#27675;',
        "\xDD\xE5" => '&#27684;',
        "\xDD\xE6" => '&#27683;',
        "\xDD\xE7" => '&#27742;',
        "\xDD\xE8" => '&#27733;',
        "\xDD\xE9" => '&#27746;',
        "\xDD\xEA" => '&#27754;',
        "\xDD\xEB" => '&#27778;',
        "\xDD\xEC" => '&#27789;',
        "\xDD\xED" => '&#27802;',
        "\xDD\xEE" => '&#27777;',
        "\xDD\xEF" => '&#27803;',
        "\xDD\xF0" => '&#27774;',
        "\xDD\xF1" => '&#27752;',
        "\xDD\xF2" => '&#27763;',
        "\xDD\xF3" => '&#27794;',
        "\xDD\xF4" => '&#27792;',
        "\xDD\xF5" => '&#27844;',
        "\xDD\xF6" => '&#27889;',
        "\xDD\xF7" => '&#27859;',
        "\xDD\xF8" => '&#27837;',
        "\xDD\xF9" => '&#27863;',
        "\xDD\xFA" => '&#27845;',
        "\xDD\xFB" => '&#27869;',
        "\xDD\xFC" => '&#27822;',
        "\xDD\xFD" => '&#27825;',
        "\xDD\xFE" => '&#27838;',
        "\xDE\xA1" => '&#27834;',
        "\xDE\xA2" => '&#27867;',
        "\xDE\xA3" => '&#27887;',
        "\xDE\xA4" => '&#27865;',
        "\xDE\xA5" => '&#27882;',
        "\xDE\xA6" => '&#27935;',
        "\xDE\xA7" => '&#34893;',
        "\xDE\xA8" => '&#27958;',
        "\xDE\xA9" => '&#27947;',
        "\xDE\xAA" => '&#27965;',
        "\xDE\xAB" => '&#27960;',
        "\xDE\xAC" => '&#27929;',
        "\xDE\xAD" => '&#27957;',
        "\xDE\xAE" => '&#27955;',
        "\xDE\xAF" => '&#27922;',
        "\xDE\xB0" => '&#27916;',
        "\xDE\xB1" => '&#28003;',
        "\xDE\xB2" => '&#28051;',
        "\xDE\xB3" => '&#28004;',
        "\xDE\xB4" => '&#27994;',
        "\xDE\xB5" => '&#28025;',
        "\xDE\xB6" => '&#27993;',
        "\xDE\xB7" => '&#28046;',
        "\xDE\xB8" => '&#28053;',
        "\xDE\xB9" => '&#28644;',
        "\xDE\xBA" => '&#28037;',
        "\xDE\xBB" => '&#28153;',
        "\xDE\xBC" => '&#28181;',
        "\xDE\xBD" => '&#28170;',
        "\xDE\xBE" => '&#28085;',
        "\xDE\xBF" => '&#28103;',
        "\xDE\xC0" => '&#28134;',
        "\xDE\xC1" => '&#28088;',
        "\xDE\xC2" => '&#28102;',
        "\xDE\xC3" => '&#28140;',
        "\xDE\xC4" => '&#28126;',
        "\xDE\xC5" => '&#28108;',
        "\xDE\xC6" => '&#28136;',
        "\xDE\xC7" => '&#28114;',
        "\xDE\xC8" => '&#28101;',
        "\xDE\xC9" => '&#28154;',
        "\xDE\xCA" => '&#28121;',
        "\xDE\xCB" => '&#28132;',
        "\xDE\xCC" => '&#28117;',
        "\xDE\xCD" => '&#28138;',
        "\xDE\xCE" => '&#28142;',
        "\xDE\xCF" => '&#28205;',
        "\xDE\xD0" => '&#28270;',
        "\xDE\xD1" => '&#28206;',
        "\xDE\xD2" => '&#28185;',
        "\xDE\xD3" => '&#28274;',
        "\xDE\xD4" => '&#28255;',
        "\xDE\xD5" => '&#28222;',
        "\xDE\xD6" => '&#28195;',
        "\xDE\xD7" => '&#28267;',
        "\xDE\xD8" => '&#28203;',
        "\xDE\xD9" => '&#28278;',
        "\xDE\xDA" => '&#28237;',
        "\xDE\xDB" => '&#28191;',
        "\xDE\xDC" => '&#28227;',
        "\xDE\xDD" => '&#28218;',
        "\xDE\xDE" => '&#28238;',
        "\xDE\xDF" => '&#28196;',
        "\xDE\xE0" => '&#28415;',
        "\xDE\xE1" => '&#28189;',
        "\xDE\xE2" => '&#28216;',
        "\xDE\xE3" => '&#28290;',
        "\xDE\xE4" => '&#28330;',
        "\xDE\xE5" => '&#28312;',
        "\xDE\xE6" => '&#28361;',
        "\xDE\xE7" => '&#28343;',
        "\xDE\xE8" => '&#28371;',
        "\xDE\xE9" => '&#28349;',
        "\xDE\xEA" => '&#28335;',
        "\xDE\xEB" => '&#28356;',
        "\xDE\xEC" => '&#28338;',
        "\xDE\xED" => '&#28372;',
        "\xDE\xEE" => '&#28373;',
        "\xDE\xEF" => '&#28303;',
        "\xDE\xF0" => '&#28325;',
        "\xDE\xF1" => '&#28354;',
        "\xDE\xF2" => '&#28319;',
        "\xDE\xF3" => '&#28481;',
        "\xDE\xF4" => '&#28433;',
        "\xDE\xF5" => '&#28748;',
        "\xDE\xF6" => '&#28396;',
        "\xDE\xF7" => '&#28408;',
        "\xDE\xF8" => '&#28414;',
        "\xDE\xF9" => '&#28479;',
        "\xDE\xFA" => '&#28402;',
        "\xDE\xFB" => '&#28465;',
        "\xDE\xFC" => '&#28399;',
        "\xDE\xFD" => '&#28466;',
        "\xDE\xFE" => '&#28364;',
        "\xDF\xA1" => '&#28478;',
        "\xDF\xA2" => '&#28435;',
        "\xDF\xA3" => '&#28407;',
        "\xDF\xA4" => '&#28550;',
        "\xDF\xA5" => '&#28538;',
        "\xDF\xA6" => '&#28536;',
        "\xDF\xA7" => '&#28545;',
        "\xDF\xA8" => '&#28544;',
        "\xDF\xA9" => '&#28527;',
        "\xDF\xAA" => '&#28507;',
        "\xDF\xAB" => '&#28659;',
        "\xDF\xAC" => '&#28525;',
        "\xDF\xAD" => '&#28546;',
        "\xDF\xAE" => '&#28540;',
        "\xDF\xAF" => '&#28504;',
        "\xDF\xB0" => '&#28558;',
        "\xDF\xB1" => '&#28561;',
        "\xDF\xB2" => '&#28610;',
        "\xDF\xB3" => '&#28518;',
        "\xDF\xB4" => '&#28595;',
        "\xDF\xB5" => '&#28579;',
        "\xDF\xB6" => '&#28577;',
        "\xDF\xB7" => '&#28580;',
        "\xDF\xB8" => '&#28601;',
        "\xDF\xB9" => '&#28614;',
        "\xDF\xBA" => '&#28586;',
        "\xDF\xBB" => '&#28639;',
        "\xDF\xBC" => '&#28629;',
        "\xDF\xBD" => '&#28652;',
        "\xDF\xBE" => '&#28628;',
        "\xDF\xBF" => '&#28632;',
        "\xDF\xC0" => '&#28657;',
        "\xDF\xC1" => '&#28654;',
        "\xDF\xC2" => '&#28635;',
        "\xDF\xC3" => '&#28681;',
        "\xDF\xC4" => '&#28683;',
        "\xDF\xC5" => '&#28666;',
        "\xDF\xC6" => '&#28689;',
        "\xDF\xC7" => '&#28673;',
        "\xDF\xC8" => '&#28687;',
        "\xDF\xC9" => '&#28670;',
        "\xDF\xCA" => '&#28699;',
        "\xDF\xCB" => '&#28698;',
        "\xDF\xCC" => '&#28532;',
        "\xDF\xCD" => '&#28701;',
        "\xDF\xCE" => '&#28696;',
        "\xDF\xCF" => '&#28703;',
        "\xDF\xD0" => '&#28720;',
        "\xDF\xD1" => '&#28734;',
        "\xDF\xD2" => '&#28722;',
        "\xDF\xD3" => '&#28753;',
        "\xDF\xD4" => '&#28771;',
        "\xDF\xD5" => '&#28825;',
        "\xDF\xD6" => '&#28818;',
        "\xDF\xD7" => '&#28847;',
        "\xDF\xD8" => '&#28913;',
        "\xDF\xD9" => '&#28844;',
        "\xDF\xDA" => '&#28856;',
        "\xDF\xDB" => '&#28851;',
        "\xDF\xDC" => '&#28846;',
        "\xDF\xDD" => '&#28895;',
        "\xDF\xDE" => '&#28875;',
        "\xDF\xDF" => '&#28893;',
        "\xDF\xE0" => '&#28889;',
        "\xDF\xE1" => '&#28937;',
        "\xDF\xE2" => '&#28925;',
        "\xDF\xE3" => '&#28956;',
        "\xDF\xE4" => '&#28953;',
        "\xDF\xE5" => '&#29029;',
        "\xDF\xE6" => '&#29013;',
        "\xDF\xE7" => '&#29064;',
        "\xDF\xE8" => '&#29030;',
        "\xDF\xE9" => '&#29026;',
        "\xDF\xEA" => '&#29004;',
        "\xDF\xEB" => '&#29014;',
        "\xDF\xEC" => '&#29036;',
        "\xDF\xED" => '&#29071;',
        "\xDF\xEE" => '&#29179;',
        "\xDF\xEF" => '&#29060;',
        "\xDF\xF0" => '&#29077;',
        "\xDF\xF1" => '&#29096;',
        "\xDF\xF2" => '&#29100;',
        "\xDF\xF3" => '&#29143;',
        "\xDF\xF4" => '&#29113;',
        "\xDF\xF5" => '&#29118;',
        "\xDF\xF6" => '&#29138;',
        "\xDF\xF7" => '&#29129;',
        "\xDF\xF8" => '&#29140;',
        "\xDF\xF9" => '&#29134;',
        "\xDF\xFA" => '&#29152;',
        "\xDF\xFB" => '&#29164;',
        "\xDF\xFC" => '&#29159;',
        "\xDF\xFD" => '&#29173;',
        "\xDF\xFE" => '&#29180;',
        "\xE0\xA1" => '&#29177;',
        "\xE0\xA2" => '&#29183;',
        "\xE0\xA3" => '&#29197;',
        "\xE0\xA4" => '&#29200;',
        "\xE0\xA5" => '&#29211;',
        "\xE0\xA6" => '&#29224;',
        "\xE0\xA7" => '&#29229;',
        "\xE0\xA8" => '&#29228;',
        "\xE0\xA9" => '&#29232;',
        "\xE0\xAA" => '&#29234;',
        "\xE0\xAB" => '&#29243;',
        "\xE0\xAC" => '&#29244;',
        "\xE0\xAD" => '&#29247;',
        "\xE0\xAE" => '&#29248;',
        "\xE0\xAF" => '&#29254;',
        "\xE0\xB0" => '&#29259;',
        "\xE0\xB1" => '&#29272;',
        "\xE0\xB2" => '&#29300;',
        "\xE0\xB3" => '&#29310;',
        "\xE0\xB4" => '&#29314;',
        "\xE0\xB5" => '&#29313;',
        "\xE0\xB6" => '&#29319;',
        "\xE0\xB7" => '&#29330;',
        "\xE0\xB8" => '&#29334;',
        "\xE0\xB9" => '&#29346;',
        "\xE0\xBA" => '&#29351;',
        "\xE0\xBB" => '&#29369;',
        "\xE0\xBC" => '&#29362;',
        "\xE0\xBD" => '&#29379;',
        "\xE0\xBE" => '&#29382;',
        "\xE0\xBF" => '&#29380;',
        "\xE0\xC0" => '&#29390;',
        "\xE0\xC1" => '&#29394;',
        "\xE0\xC2" => '&#29410;',
        "\xE0\xC3" => '&#29408;',
        "\xE0\xC4" => '&#29409;',
        "\xE0\xC5" => '&#29433;',
        "\xE0\xC6" => '&#29431;',
        "\xE0\xC7" => '&#20495;',
        "\xE0\xC8" => '&#29463;',
        "\xE0\xC9" => '&#29450;',
        "\xE0\xCA" => '&#29468;',
        "\xE0\xCB" => '&#29462;',
        "\xE0\xCC" => '&#29469;',
        "\xE0\xCD" => '&#29492;',
        "\xE0\xCE" => '&#29487;',
        "\xE0\xCF" => '&#29481;',
        "\xE0\xD0" => '&#29477;',
        "\xE0\xD1" => '&#29502;',
        "\xE0\xD2" => '&#29518;',
        "\xE0\xD3" => '&#29519;',
        "\xE0\xD4" => '&#40664;',
        "\xE0\xD5" => '&#29527;',
        "\xE0\xD6" => '&#29546;',
        "\xE0\xD7" => '&#29544;',
        "\xE0\xD8" => '&#29552;',
        "\xE0\xD9" => '&#29560;',
        "\xE0\xDA" => '&#29557;',
        "\xE0\xDB" => '&#29563;',
        "\xE0\xDC" => '&#29562;',
        "\xE0\xDD" => '&#29640;',
        "\xE0\xDE" => '&#29619;',
        "\xE0\xDF" => '&#29646;',
        "\xE0\xE0" => '&#29627;',
        "\xE0\xE1" => '&#29632;',
        "\xE0\xE2" => '&#29669;',
        "\xE0\xE3" => '&#29678;',
        "\xE0\xE4" => '&#29662;',
        "\xE0\xE5" => '&#29858;',
        "\xE0\xE6" => '&#29701;',
        "\xE0\xE7" => '&#29807;',
        "\xE0\xE8" => '&#29733;',
        "\xE0\xE9" => '&#29688;',
        "\xE0\xEA" => '&#29746;',
        "\xE0\xEB" => '&#29754;',
        "\xE0\xEC" => '&#29781;',
        "\xE0\xED" => '&#29759;',
        "\xE0\xEE" => '&#29791;',
        "\xE0\xEF" => '&#29785;',
        "\xE0\xF0" => '&#29761;',
        "\xE0\xF1" => '&#29788;',
        "\xE0\xF2" => '&#29801;',
        "\xE0\xF3" => '&#29808;',
        "\xE0\xF4" => '&#29795;',
        "\xE0\xF5" => '&#29802;',
        "\xE0\xF6" => '&#29814;',
        "\xE0\xF7" => '&#29822;',
        "\xE0\xF8" => '&#29835;',
        "\xE0\xF9" => '&#29854;',
        "\xE0\xFA" => '&#29863;',
        "\xE0\xFB" => '&#29898;',
        "\xE0\xFC" => '&#29903;',
        "\xE0\xFD" => '&#29908;',
        "\xE0\xFE" => '&#29681;',
        "\xE1\xA1" => '&#29920;',
        "\xE1\xA2" => '&#29923;',
        "\xE1\xA3" => '&#29927;',
        "\xE1\xA4" => '&#29929;',
        "\xE1\xA5" => '&#29934;',
        "\xE1\xA6" => '&#29938;',
        "\xE1\xA7" => '&#29936;',
        "\xE1\xA8" => '&#29937;',
        "\xE1\xA9" => '&#29944;',
        "\xE1\xAA" => '&#29943;',
        "\xE1\xAB" => '&#29956;',
        "\xE1\xAC" => '&#29955;',
        "\xE1\xAD" => '&#29957;',
        "\xE1\xAE" => '&#29964;',
        "\xE1\xAF" => '&#29966;',
        "\xE1\xB0" => '&#29965;',
        "\xE1\xB1" => '&#29973;',
        "\xE1\xB2" => '&#29971;',
        "\xE1\xB3" => '&#29982;',
        "\xE1\xB4" => '&#29990;',
        "\xE1\xB5" => '&#29996;',
        "\xE1\xB6" => '&#30012;',
        "\xE1\xB7" => '&#30020;',
        "\xE1\xB8" => '&#30029;',
        "\xE1\xB9" => '&#30026;',
        "\xE1\xBA" => '&#30025;',
        "\xE1\xBB" => '&#30043;',
        "\xE1\xBC" => '&#30022;',
        "\xE1\xBD" => '&#30042;',
        "\xE1\xBE" => '&#30057;',
        "\xE1\xBF" => '&#30052;',
        "\xE1\xC0" => '&#30055;',
        "\xE1\xC1" => '&#30059;',
        "\xE1\xC2" => '&#30061;',
        "\xE1\xC3" => '&#30072;',
        "\xE1\xC4" => '&#30070;',
        "\xE1\xC5" => '&#30086;',
        "\xE1\xC6" => '&#30087;',
        "\xE1\xC7" => '&#30068;',
        "\xE1\xC8" => '&#30090;',
        "\xE1\xC9" => '&#30089;',
        "\xE1\xCA" => '&#30082;',
        "\xE1\xCB" => '&#30100;',
        "\xE1\xCC" => '&#30106;',
        "\xE1\xCD" => '&#30109;',
        "\xE1\xCE" => '&#30117;',
        "\xE1\xCF" => '&#30115;',
        "\xE1\xD0" => '&#30146;',
        "\xE1\xD1" => '&#30131;',
        "\xE1\xD2" => '&#30147;',
        "\xE1\xD3" => '&#30133;',
        "\xE1\xD4" => '&#30141;',
        "\xE1\xD5" => '&#30136;',
        "\xE1\xD6" => '&#30140;',
        "\xE1\xD7" => '&#30129;',
        "\xE1\xD8" => '&#30157;',
        "\xE1\xD9" => '&#30154;',
        "\xE1\xDA" => '&#30162;',
        "\xE1\xDB" => '&#30169;',
        "\xE1\xDC" => '&#30179;',
        "\xE1\xDD" => '&#30174;',
        "\xE1\xDE" => '&#30206;',
        "\xE1\xDF" => '&#30207;',
        "\xE1\xE0" => '&#30204;',
        "\xE1\xE1" => '&#30209;',
        "\xE1\xE2" => '&#30192;',
        "\xE1\xE3" => '&#30202;',
        "\xE1\xE4" => '&#30194;',
        "\xE1\xE5" => '&#30195;',
        "\xE1\xE6" => '&#30219;',
        "\xE1\xE7" => '&#30221;',
        "\xE1\xE8" => '&#30217;',
        "\xE1\xE9" => '&#30239;',
        "\xE1\xEA" => '&#30247;',
        "\xE1\xEB" => '&#30240;',
        "\xE1\xEC" => '&#30241;',
        "\xE1\xED" => '&#30242;',
        "\xE1\xEE" => '&#30244;',
        "\xE1\xEF" => '&#30260;',
        "\xE1\xF0" => '&#30256;',
        "\xE1\xF1" => '&#30267;',
        "\xE1\xF2" => '&#30279;',
        "\xE1\xF3" => '&#30280;',
        "\xE1\xF4" => '&#30278;',
        "\xE1\xF5" => '&#30300;',
        "\xE1\xF6" => '&#30296;',
        "\xE1\xF7" => '&#30305;',
        "\xE1\xF8" => '&#30306;',
        "\xE1\xF9" => '&#30312;',
        "\xE1\xFA" => '&#30313;',
        "\xE1\xFB" => '&#30314;',
        "\xE1\xFC" => '&#30311;',
        "\xE1\xFD" => '&#30316;',
        "\xE1\xFE" => '&#30320;',
        "\xE2\xA1" => '&#30322;',
        "\xE2\xA2" => '&#30326;',
        "\xE2\xA3" => '&#30328;',
        "\xE2\xA4" => '&#30332;',
        "\xE2\xA5" => '&#30336;',
        "\xE2\xA6" => '&#30339;',
        "\xE2\xA7" => '&#30344;',
        "\xE2\xA8" => '&#30347;',
        "\xE2\xA9" => '&#30350;',
        "\xE2\xAA" => '&#30358;',
        "\xE2\xAB" => '&#30355;',
        "\xE2\xAC" => '&#30361;',
        "\xE2\xAD" => '&#30362;',
        "\xE2\xAE" => '&#30384;',
        "\xE2\xAF" => '&#30388;',
        "\xE2\xB0" => '&#30392;',
        "\xE2\xB1" => '&#30393;',
        "\xE2\xB2" => '&#30394;',
        "\xE2\xB3" => '&#30402;',
        "\xE2\xB4" => '&#30413;',
        "\xE2\xB5" => '&#30422;',
        "\xE2\xB6" => '&#30418;',
        "\xE2\xB7" => '&#30430;',
        "\xE2\xB8" => '&#30433;',
        "\xE2\xB9" => '&#30437;',
        "\xE2\xBA" => '&#30439;',
        "\xE2\xBB" => '&#30442;',
        "\xE2\xBC" => '&#34351;',
        "\xE2\xBD" => '&#30459;',
        "\xE2\xBE" => '&#30472;',
        "\xE2\xBF" => '&#30471;',
        "\xE2\xC0" => '&#30468;',
        "\xE2\xC1" => '&#30505;',
        "\xE2\xC2" => '&#30500;',
        "\xE2\xC3" => '&#30494;',
        "\xE2\xC4" => '&#30501;',
        "\xE2\xC5" => '&#30502;',
        "\xE2\xC6" => '&#30491;',
        "\xE2\xC7" => '&#30519;',
        "\xE2\xC8" => '&#30520;',
        "\xE2\xC9" => '&#30535;',
        "\xE2\xCA" => '&#30554;',
        "\xE2\xCB" => '&#30568;',
        "\xE2\xCC" => '&#30571;',
        "\xE2\xCD" => '&#30555;',
        "\xE2\xCE" => '&#30565;',
        "\xE2\xCF" => '&#30591;',
        "\xE2\xD0" => '&#30590;',
        "\xE2\xD1" => '&#30585;',
        "\xE2\xD2" => '&#30606;',
        "\xE2\xD3" => '&#30603;',
        "\xE2\xD4" => '&#30609;',
        "\xE2\xD5" => '&#30624;',
        "\xE2\xD6" => '&#30622;',
        "\xE2\xD7" => '&#30640;',
        "\xE2\xD8" => '&#30646;',
        "\xE2\xD9" => '&#30649;',
        "\xE2\xDA" => '&#30655;',
        "\xE2\xDB" => '&#30652;',
        "\xE2\xDC" => '&#30653;',
        "\xE2\xDD" => '&#30651;',
        "\xE2\xDE" => '&#30663;',
        "\xE2\xDF" => '&#30669;',
        "\xE2\xE0" => '&#30679;',
        "\xE2\xE1" => '&#30682;',
        "\xE2\xE2" => '&#30684;',
        "\xE2\xE3" => '&#30691;',
        "\xE2\xE4" => '&#30702;',
        "\xE2\xE5" => '&#30716;',
        "\xE2\xE6" => '&#30732;',
        "\xE2\xE7" => '&#30738;',
        "\xE2\xE8" => '&#31014;',
        "\xE2\xE9" => '&#30752;',
        "\xE2\xEA" => '&#31018;',
        "\xE2\xEB" => '&#30789;',
        "\xE2\xEC" => '&#30862;',
        "\xE2\xED" => '&#30836;',
        "\xE2\xEE" => '&#30854;',
        "\xE2\xEF" => '&#30844;',
        "\xE2\xF0" => '&#30874;',
        "\xE2\xF1" => '&#30860;',
        "\xE2\xF2" => '&#30883;',
        "\xE2\xF3" => '&#30901;',
        "\xE2\xF4" => '&#30890;',
        "\xE2\xF5" => '&#30895;',
        "\xE2\xF6" => '&#30929;',
        "\xE2\xF7" => '&#30918;',
        "\xE2\xF8" => '&#30923;',
        "\xE2\xF9" => '&#30932;',
        "\xE2\xFA" => '&#30910;',
        "\xE2\xFB" => '&#30908;',
        "\xE2\xFC" => '&#30917;',
        "\xE2\xFD" => '&#30922;',
        "\xE2\xFE" => '&#30956;',
        "\xE3\xA1" => '&#30951;',
        "\xE3\xA2" => '&#30938;',
        "\xE3\xA3" => '&#30973;',
        "\xE3\xA4" => '&#30964;',
        "\xE3\xA5" => '&#30983;',
        "\xE3\xA6" => '&#30994;',
        "\xE3\xA7" => '&#30993;',
        "\xE3\xA8" => '&#31001;',
        "\xE3\xA9" => '&#31020;',
        "\xE3\xAA" => '&#31019;',
        "\xE3\xAB" => '&#31040;',
        "\xE3\xAC" => '&#31072;',
        "\xE3\xAD" => '&#31063;',
        "\xE3\xAE" => '&#31071;',
        "\xE3\xAF" => '&#31066;',
        "\xE3\xB0" => '&#31061;',
        "\xE3\xB1" => '&#31059;',
        "\xE3\xB2" => '&#31098;',
        "\xE3\xB3" => '&#31103;',
        "\xE3\xB4" => '&#31114;',
        "\xE3\xB5" => '&#31133;',
        "\xE3\xB6" => '&#31143;',
        "\xE3\xB7" => '&#40779;',
        "\xE3\xB8" => '&#31146;',
        "\xE3\xB9" => '&#31150;',
        "\xE3\xBA" => '&#31155;',
        "\xE3\xBB" => '&#31161;',
        "\xE3\xBC" => '&#31162;',
        "\xE3\xBD" => '&#31177;',
        "\xE3\xBE" => '&#31189;',
        "\xE3\xBF" => '&#31207;',
        "\xE3\xC0" => '&#31212;',
        "\xE3\xC1" => '&#31201;',
        "\xE3\xC2" => '&#31203;',
        "\xE3\xC3" => '&#31240;',
        "\xE3\xC4" => '&#31245;',
        "\xE3\xC5" => '&#31256;',
        "\xE3\xC6" => '&#31257;',
        "\xE3\xC7" => '&#31264;',
        "\xE3\xC8" => '&#31263;',
        "\xE3\xC9" => '&#31104;',
        "\xE3\xCA" => '&#31281;',
        "\xE3\xCB" => '&#31291;',
        "\xE3\xCC" => '&#31294;',
        "\xE3\xCD" => '&#31287;',
        "\xE3\xCE" => '&#31299;',
        "\xE3\xCF" => '&#31319;',
        "\xE3\xD0" => '&#31305;',
        "\xE3\xD1" => '&#31329;',
        "\xE3\xD2" => '&#31330;',
        "\xE3\xD3" => '&#31337;',
        "\xE3\xD4" => '&#40861;',
        "\xE3\xD5" => '&#31344;',
        "\xE3\xD6" => '&#31353;',
        "\xE3\xD7" => '&#31357;',
        "\xE3\xD8" => '&#31368;',
        "\xE3\xD9" => '&#31383;',
        "\xE3\xDA" => '&#31381;',
        "\xE3\xDB" => '&#31384;',
        "\xE3\xDC" => '&#31382;',
        "\xE3\xDD" => '&#31401;',
        "\xE3\xDE" => '&#31432;',
        "\xE3\xDF" => '&#31408;',
        "\xE3\xE0" => '&#31414;',
        "\xE3\xE1" => '&#31429;',
        "\xE3\xE2" => '&#31428;',
        "\xE3\xE3" => '&#31423;',
        "\xE3\xE4" => '&#36995;',
        "\xE3\xE5" => '&#31431;',
        "\xE3\xE6" => '&#31434;',
        "\xE3\xE7" => '&#31437;',
        "\xE3\xE8" => '&#31439;',
        "\xE3\xE9" => '&#31445;',
        "\xE3\xEA" => '&#31443;',
        "\xE3\xEB" => '&#31449;',
        "\xE3\xEC" => '&#31450;',
        "\xE3\xED" => '&#31453;',
        "\xE3\xEE" => '&#31457;',
        "\xE3\xEF" => '&#31458;',
        "\xE3\xF0" => '&#31462;',
        "\xE3\xF1" => '&#31469;',
        "\xE3\xF2" => '&#31472;',
        "\xE3\xF3" => '&#31490;',
        "\xE3\xF4" => '&#31503;',
        "\xE3\xF5" => '&#31498;',
        "\xE3\xF6" => '&#31494;',
        "\xE3\xF7" => '&#31539;',
        "\xE3\xF8" => '&#31512;',
        "\xE3\xF9" => '&#31513;',
        "\xE3\xFA" => '&#31518;',
        "\xE3\xFB" => '&#31541;',
        "\xE3\xFC" => '&#31528;',
        "\xE3\xFD" => '&#31542;',
        "\xE3\xFE" => '&#31568;',
        "\xE4\xA1" => '&#31610;',
        "\xE4\xA2" => '&#31492;',
        "\xE4\xA3" => '&#31565;',
        "\xE4\xA4" => '&#31499;',
        "\xE4\xA5" => '&#31564;',
        "\xE4\xA6" => '&#31557;',
        "\xE4\xA7" => '&#31605;',
        "\xE4\xA8" => '&#31589;',
        "\xE4\xA9" => '&#31604;',
        "\xE4\xAA" => '&#31591;',
        "\xE4\xAB" => '&#31600;',
        "\xE4\xAC" => '&#31601;',
        "\xE4\xAD" => '&#31596;',
        "\xE4\xAE" => '&#31598;',
        "\xE4\xAF" => '&#31645;',
        "\xE4\xB0" => '&#31640;',
        "\xE4\xB1" => '&#31647;',
        "\xE4\xB2" => '&#31629;',
        "\xE4\xB3" => '&#31644;',
        "\xE4\xB4" => '&#31642;',
        "\xE4\xB5" => '&#31627;',
        "\xE4\xB6" => '&#31634;',
        "\xE4\xB7" => '&#31631;',
        "\xE4\xB8" => '&#31581;',
        "\xE4\xB9" => '&#31641;',
        "\xE4\xBA" => '&#31691;',
        "\xE4\xBB" => '&#31681;',
        "\xE4\xBC" => '&#31692;',
        "\xE4\xBD" => '&#31695;',
        "\xE4\xBE" => '&#31668;',
        "\xE4\xBF" => '&#31686;',
        "\xE4\xC0" => '&#31709;',
        "\xE4\xC1" => '&#31721;',
        "\xE4\xC2" => '&#31761;',
        "\xE4\xC3" => '&#31764;',
        "\xE4\xC4" => '&#31718;',
        "\xE4\xC5" => '&#31717;',
        "\xE4\xC6" => '&#31840;',
        "\xE4\xC7" => '&#31744;',
        "\xE4\xC8" => '&#31751;',
        "\xE4\xC9" => '&#31763;',
        "\xE4\xCA" => '&#31731;',
        "\xE4\xCB" => '&#31735;',
        "\xE4\xCC" => '&#31767;',
        "\xE4\xCD" => '&#31757;',
        "\xE4\xCE" => '&#31734;',
        "\xE4\xCF" => '&#31779;',
        "\xE4\xD0" => '&#31783;',
        "\xE4\xD1" => '&#31786;',
        "\xE4\xD2" => '&#31775;',
        "\xE4\xD3" => '&#31799;',
        "\xE4\xD4" => '&#31787;',
        "\xE4\xD5" => '&#31805;',
        "\xE4\xD6" => '&#31820;',
        "\xE4\xD7" => '&#31811;',
        "\xE4\xD8" => '&#31828;',
        "\xE4\xD9" => '&#31823;',
        "\xE4\xDA" => '&#31808;',
        "\xE4\xDB" => '&#31824;',
        "\xE4\xDC" => '&#31832;',
        "\xE4\xDD" => '&#31839;',
        "\xE4\xDE" => '&#31844;',
        "\xE4\xDF" => '&#31830;',
        "\xE4\xE0" => '&#31845;',
        "\xE4\xE1" => '&#31852;',
        "\xE4\xE2" => '&#31861;',
        "\xE4\xE3" => '&#31875;',
        "\xE4\xE4" => '&#31888;',
        "\xE4\xE5" => '&#31908;',
        "\xE4\xE6" => '&#31917;',
        "\xE4\xE7" => '&#31906;',
        "\xE4\xE8" => '&#31915;',
        "\xE4\xE9" => '&#31905;',
        "\xE4\xEA" => '&#31912;',
        "\xE4\xEB" => '&#31923;',
        "\xE4\xEC" => '&#31922;',
        "\xE4\xED" => '&#31921;',
        "\xE4\xEE" => '&#31918;',
        "\xE4\xEF" => '&#31929;',
        "\xE4\xF0" => '&#31933;',
        "\xE4\xF1" => '&#31936;',
        "\xE4\xF2" => '&#31941;',
        "\xE4\xF3" => '&#31938;',
        "\xE4\xF4" => '&#31960;',
        "\xE4\xF5" => '&#31954;',
        "\xE4\xF6" => '&#31964;',
        "\xE4\xF7" => '&#31970;',
        "\xE4\xF8" => '&#39739;',
        "\xE4\xF9" => '&#31983;',
        "\xE4\xFA" => '&#31986;',
        "\xE4\xFB" => '&#31988;',
        "\xE4\xFC" => '&#31990;',
        "\xE4\xFD" => '&#31994;',
        "\xE4\xFE" => '&#32006;',
        "\xE5\xA1" => '&#32002;',
        "\xE5\xA2" => '&#32028;',
        "\xE5\xA3" => '&#32021;',
        "\xE5\xA4" => '&#32010;',
        "\xE5\xA5" => '&#32069;',
        "\xE5\xA6" => '&#32075;',
        "\xE5\xA7" => '&#32046;',
        "\xE5\xA8" => '&#32050;',
        "\xE5\xA9" => '&#32063;',
        "\xE5\xAA" => '&#32053;',
        "\xE5\xAB" => '&#32070;',
        "\xE5\xAC" => '&#32115;',
        "\xE5\xAD" => '&#32086;',
        "\xE5\xAE" => '&#32078;',
        "\xE5\xAF" => '&#32114;',
        "\xE5\xB0" => '&#32104;',
        "\xE5\xB1" => '&#32110;',
        "\xE5\xB2" => '&#32079;',
        "\xE5\xB3" => '&#32099;',
        "\xE5\xB4" => '&#32147;',
        "\xE5\xB5" => '&#32137;',
        "\xE5\xB6" => '&#32091;',
        "\xE5\xB7" => '&#32143;',
        "\xE5\xB8" => '&#32125;',
        "\xE5\xB9" => '&#32155;',
        "\xE5\xBA" => '&#32186;',
        "\xE5\xBB" => '&#32174;',
        "\xE5\xBC" => '&#32163;',
        "\xE5\xBD" => '&#32181;',
        "\xE5\xBE" => '&#32199;',
        "\xE5\xBF" => '&#32189;',
        "\xE5\xC0" => '&#32171;',
        "\xE5\xC1" => '&#32317;',
        "\xE5\xC2" => '&#32162;',
        "\xE5\xC3" => '&#32175;',
        "\xE5\xC4" => '&#32220;',
        "\xE5\xC5" => '&#32184;',
        "\xE5\xC6" => '&#32159;',
        "\xE5\xC7" => '&#32176;',
        "\xE5\xC8" => '&#32216;',
        "\xE5\xC9" => '&#32221;',
        "\xE5\xCA" => '&#32228;',
        "\xE5\xCB" => '&#32222;',
        "\xE5\xCC" => '&#32251;',
        "\xE5\xCD" => '&#32242;',
        "\xE5\xCE" => '&#32225;',
        "\xE5\xCF" => '&#32261;',
        "\xE5\xD0" => '&#32266;',
        "\xE5\xD1" => '&#32291;',
        "\xE5\xD2" => '&#32289;',
        "\xE5\xD3" => '&#32274;',
        "\xE5\xD4" => '&#32305;',
        "\xE5\xD5" => '&#32287;',
        "\xE5\xD6" => '&#32265;',
        "\xE5\xD7" => '&#32267;',
        "\xE5\xD8" => '&#32290;',
        "\xE5\xD9" => '&#32326;',
        "\xE5\xDA" => '&#32358;',
        "\xE5\xDB" => '&#32315;',
        "\xE5\xDC" => '&#32309;',
        "\xE5\xDD" => '&#32313;',
        "\xE5\xDE" => '&#32323;',
        "\xE5\xDF" => '&#32311;',
        "\xE5\xE0" => '&#32306;',
        "\xE5\xE1" => '&#32314;',
        "\xE5\xE2" => '&#32359;',
        "\xE5\xE3" => '&#32349;',
        "\xE5\xE4" => '&#32342;',
        "\xE5\xE5" => '&#32350;',
        "\xE5\xE6" => '&#32345;',
        "\xE5\xE7" => '&#32346;',
        "\xE5\xE8" => '&#32377;',
        "\xE5\xE9" => '&#32362;',
        "\xE5\xEA" => '&#32361;',
        "\xE5\xEB" => '&#32380;',
        "\xE5\xEC" => '&#32379;',
        "\xE5\xED" => '&#32387;',
        "\xE5\xEE" => '&#32213;',
        "\xE5\xEF" => '&#32381;',
        "\xE5\xF0" => '&#36782;',
        "\xE5\xF1" => '&#32383;',
        "\xE5\xF2" => '&#32392;',
        "\xE5\xF3" => '&#32393;',
        "\xE5\xF4" => '&#32396;',
        "\xE5\xF5" => '&#32402;',
        "\xE5\xF6" => '&#32400;',
        "\xE5\xF7" => '&#32403;',
        "\xE5\xF8" => '&#32404;',
        "\xE5\xF9" => '&#32406;',
        "\xE5\xFA" => '&#32398;',
        "\xE5\xFB" => '&#32411;',
        "\xE5\xFC" => '&#32412;',
        "\xE5\xFD" => '&#32568;',
        "\xE5\xFE" => '&#32570;',
        "\xE6\xA1" => '&#32581;',
        "\xE6\xA2" => '&#32588;',
        "\xE6\xA3" => '&#32589;',
        "\xE6\xA4" => '&#32590;',
        "\xE6\xA5" => '&#32592;',
        "\xE6\xA6" => '&#32593;',
        "\xE6\xA7" => '&#32597;',
        "\xE6\xA8" => '&#32596;',
        "\xE6\xA9" => '&#32600;',
        "\xE6\xAA" => '&#32607;',
        "\xE6\xAB" => '&#32608;',
        "\xE6\xAC" => '&#32616;',
        "\xE6\xAD" => '&#32617;',
        "\xE6\xAE" => '&#32615;',
        "\xE6\xAF" => '&#32632;',
        "\xE6\xB0" => '&#32642;',
        "\xE6\xB1" => '&#32646;',
        "\xE6\xB2" => '&#32643;',
        "\xE6\xB3" => '&#32648;',
        "\xE6\xB4" => '&#32647;',
        "\xE6\xB5" => '&#32652;',
        "\xE6\xB6" => '&#32660;',
        "\xE6\xB7" => '&#32670;',
        "\xE6\xB8" => '&#32669;',
        "\xE6\xB9" => '&#32666;',
        "\xE6\xBA" => '&#32675;',
        "\xE6\xBB" => '&#32687;',
        "\xE6\xBC" => '&#32690;',
        "\xE6\xBD" => '&#32697;',
        "\xE6\xBE" => '&#32686;',
        "\xE6\xBF" => '&#32694;',
        "\xE6\xC0" => '&#32696;',
        "\xE6\xC1" => '&#35697;',
        "\xE6\xC2" => '&#32709;',
        "\xE6\xC3" => '&#32710;',
        "\xE6\xC4" => '&#32714;',
        "\xE6\xC5" => '&#32725;',
        "\xE6\xC6" => '&#32724;',
        "\xE6\xC7" => '&#32737;',
        "\xE6\xC8" => '&#32742;',
        "\xE6\xC9" => '&#32745;',
        "\xE6\xCA" => '&#32755;',
        "\xE6\xCB" => '&#32761;',
        "\xE6\xCC" => '&#39132;',
        "\xE6\xCD" => '&#32774;',
        "\xE6\xCE" => '&#32772;',
        "\xE6\xCF" => '&#32779;',
        "\xE6\xD0" => '&#32786;',
        "\xE6\xD1" => '&#32792;',
        "\xE6\xD2" => '&#32793;',
        "\xE6\xD3" => '&#32796;',
        "\xE6\xD4" => '&#32801;',
        "\xE6\xD5" => '&#32808;',
        "\xE6\xD6" => '&#32831;',
        "\xE6\xD7" => '&#32827;',
        "\xE6\xD8" => '&#32842;',
        "\xE6\xD9" => '&#32838;',
        "\xE6\xDA" => '&#32850;',
        "\xE6\xDB" => '&#32856;',
        "\xE6\xDC" => '&#32858;',
        "\xE6\xDD" => '&#32863;',
        "\xE6\xDE" => '&#32866;',
        "\xE6\xDF" => '&#32872;',
        "\xE6\xE0" => '&#32883;',
        "\xE6\xE1" => '&#32882;',
        "\xE6\xE2" => '&#32880;',
        "\xE6\xE3" => '&#32886;',
        "\xE6\xE4" => '&#32889;',
        "\xE6\xE5" => '&#32893;',
        "\xE6\xE6" => '&#32895;',
        "\xE6\xE7" => '&#32900;',
        "\xE6\xE8" => '&#32902;',
        "\xE6\xE9" => '&#32901;',
        "\xE6\xEA" => '&#32923;',
        "\xE6\xEB" => '&#32915;',
        "\xE6\xEC" => '&#32922;',
        "\xE6\xED" => '&#32941;',
        "\xE6\xEE" => '&#20880;',
        "\xE6\xEF" => '&#32940;',
        "\xE6\xF0" => '&#32987;',
        "\xE6\xF1" => '&#32997;',
        "\xE6\xF2" => '&#32985;',
        "\xE6\xF3" => '&#32989;',
        "\xE6\xF4" => '&#32964;',
        "\xE6\xF5" => '&#32986;',
        "\xE6\xF6" => '&#32982;',
        "\xE6\xF7" => '&#33033;',
        "\xE6\xF8" => '&#33007;',
        "\xE6\xF9" => '&#33009;',
        "\xE6\xFA" => '&#33051;',
        "\xE6\xFB" => '&#33065;',
        "\xE6\xFC" => '&#33059;',
        "\xE6\xFD" => '&#33071;',
        "\xE6\xFE" => '&#33099;',
        "\xE7\xA1" => '&#38539;',
        "\xE7\xA2" => '&#33094;',
        "\xE7\xA3" => '&#33086;',
        "\xE7\xA4" => '&#33107;',
        "\xE7\xA5" => '&#33105;',
        "\xE7\xA6" => '&#33020;',
        "\xE7\xA7" => '&#33137;',
        "\xE7\xA8" => '&#33134;',
        "\xE7\xA9" => '&#33125;',
        "\xE7\xAA" => '&#33126;',
        "\xE7\xAB" => '&#33140;',
        "\xE7\xAC" => '&#33155;',
        "\xE7\xAD" => '&#33160;',
        "\xE7\xAE" => '&#33162;',
        "\xE7\xAF" => '&#33152;',
        "\xE7\xB0" => '&#33154;',
        "\xE7\xB1" => '&#33184;',
        "\xE7\xB2" => '&#33173;',
        "\xE7\xB3" => '&#33188;',
        "\xE7\xB4" => '&#33187;',
        "\xE7\xB5" => '&#33119;',
        "\xE7\xB6" => '&#33171;',
        "\xE7\xB7" => '&#33193;',
        "\xE7\xB8" => '&#33200;',
        "\xE7\xB9" => '&#33205;',
        "\xE7\xBA" => '&#33214;',
        "\xE7\xBB" => '&#33208;',
        "\xE7\xBC" => '&#33213;',
        "\xE7\xBD" => '&#33216;',
        "\xE7\xBE" => '&#33218;',
        "\xE7\xBF" => '&#33210;',
        "\xE7\xC0" => '&#33225;',
        "\xE7\xC1" => '&#33229;',
        "\xE7\xC2" => '&#33233;',
        "\xE7\xC3" => '&#33241;',
        "\xE7\xC4" => '&#33240;',
        "\xE7\xC5" => '&#33224;',
        "\xE7\xC6" => '&#33242;',
        "\xE7\xC7" => '&#33247;',
        "\xE7\xC8" => '&#33248;',
        "\xE7\xC9" => '&#33255;',
        "\xE7\xCA" => '&#33274;',
        "\xE7\xCB" => '&#33275;',
        "\xE7\xCC" => '&#33278;',
        "\xE7\xCD" => '&#33281;',
        "\xE7\xCE" => '&#33282;',
        "\xE7\xCF" => '&#33285;',
        "\xE7\xD0" => '&#33287;',
        "\xE7\xD1" => '&#33290;',
        "\xE7\xD2" => '&#33293;',
        "\xE7\xD3" => '&#33296;',
        "\xE7\xD4" => '&#33302;',
        "\xE7\xD5" => '&#33321;',
        "\xE7\xD6" => '&#33323;',
        "\xE7\xD7" => '&#33336;',
        "\xE7\xD8" => '&#33331;',
        "\xE7\xD9" => '&#33344;',
        "\xE7\xDA" => '&#33369;',
        "\xE7\xDB" => '&#33368;',
        "\xE7\xDC" => '&#33373;',
        "\xE7\xDD" => '&#33370;',
        "\xE7\xDE" => '&#33375;',
        "\xE7\xDF" => '&#33380;',
        "\xE7\xE0" => '&#33378;',
        "\xE7\xE1" => '&#33384;',
        "\xE7\xE2" => '&#33386;',
        "\xE7\xE3" => '&#33387;',
        "\xE7\xE4" => '&#33326;',
        "\xE7\xE5" => '&#33393;',
        "\xE7\xE6" => '&#33399;',
        "\xE7\xE7" => '&#33400;',
        "\xE7\xE8" => '&#33406;',
        "\xE7\xE9" => '&#33421;',
        "\xE7\xEA" => '&#33426;',
        "\xE7\xEB" => '&#33451;',
        "\xE7\xEC" => '&#33439;',
        "\xE7\xED" => '&#33467;',
        "\xE7\xEE" => '&#33452;',
        "\xE7\xEF" => '&#33505;',
        "\xE7\xF0" => '&#33507;',
        "\xE7\xF1" => '&#33503;',
        "\xE7\xF2" => '&#33490;',
        "\xE7\xF3" => '&#33524;',
        "\xE7\xF4" => '&#33523;',
        "\xE7\xF5" => '&#33530;',
        "\xE7\xF6" => '&#33683;',
        "\xE7\xF7" => '&#33539;',
        "\xE7\xF8" => '&#33531;',
        "\xE7\xF9" => '&#33529;',
        "\xE7\xFA" => '&#33502;',
        "\xE7\xFB" => '&#33542;',
        "\xE7\xFC" => '&#33500;',
        "\xE7\xFD" => '&#33545;',
        "\xE7\xFE" => '&#33497;',
        "\xE8\xA1" => '&#33589;',
        "\xE8\xA2" => '&#33588;',
        "\xE8\xA3" => '&#33558;',
        "\xE8\xA4" => '&#33586;',
        "\xE8\xA5" => '&#33585;',
        "\xE8\xA6" => '&#33600;',
        "\xE8\xA7" => '&#33593;',
        "\xE8\xA8" => '&#33616;',
        "\xE8\xA9" => '&#33605;',
        "\xE8\xAA" => '&#33583;',
        "\xE8\xAB" => '&#33579;',
        "\xE8\xAC" => '&#33559;',
        "\xE8\xAD" => '&#33560;',
        "\xE8\xAE" => '&#33669;',
        "\xE8\xAF" => '&#33690;',
        "\xE8\xB0" => '&#33706;',
        "\xE8\xB1" => '&#33695;',
        "\xE8\xB2" => '&#33698;',
        "\xE8\xB3" => '&#33686;',
        "\xE8\xB4" => '&#33571;',
        "\xE8\xB5" => '&#33678;',
        "\xE8\xB6" => '&#33671;',
        "\xE8\xB7" => '&#33674;',
        "\xE8\xB8" => '&#33660;',
        "\xE8\xB9" => '&#33717;',
        "\xE8\xBA" => '&#33651;',
        "\xE8\xBB" => '&#33653;',
        "\xE8\xBC" => '&#33696;',
        "\xE8\xBD" => '&#33673;',
        "\xE8\xBE" => '&#33704;',
        "\xE8\xBF" => '&#33780;',
        "\xE8\xC0" => '&#33811;',
        "\xE8\xC1" => '&#33771;',
        "\xE8\xC2" => '&#33742;',
        "\xE8\xC3" => '&#33789;',
        "\xE8\xC4" => '&#33795;',
        "\xE8\xC5" => '&#33752;',
        "\xE8\xC6" => '&#33803;',
        "\xE8\xC7" => '&#33729;',
        "\xE8\xC8" => '&#33783;',
        "\xE8\xC9" => '&#33799;',
        "\xE8\xCA" => '&#33760;',
        "\xE8\xCB" => '&#33778;',
        "\xE8\xCC" => '&#33805;',
        "\xE8\xCD" => '&#33826;',
        "\xE8\xCE" => '&#33824;',
        "\xE8\xCF" => '&#33725;',
        "\xE8\xD0" => '&#33848;',
        "\xE8\xD1" => '&#34054;',
        "\xE8\xD2" => '&#33787;',
        "\xE8\xD3" => '&#33901;',
        "\xE8\xD4" => '&#33834;',
        "\xE8\xD5" => '&#33852;',
        "\xE8\xD6" => '&#34138;',
        "\xE8\xD7" => '&#33924;',
        "\xE8\xD8" => '&#33911;',
        "\xE8\xD9" => '&#33899;',
        "\xE8\xDA" => '&#33965;',
        "\xE8\xDB" => '&#33902;',
        "\xE8\xDC" => '&#33922;',
        "\xE8\xDD" => '&#33897;',
        "\xE8\xDE" => '&#33862;',
        "\xE8\xDF" => '&#33836;',
        "\xE8\xE0" => '&#33903;',
        "\xE8\xE1" => '&#33913;',
        "\xE8\xE2" => '&#33845;',
        "\xE8\xE3" => '&#33994;',
        "\xE8\xE4" => '&#33890;',
        "\xE8\xE5" => '&#33977;',
        "\xE8\xE6" => '&#33983;',
        "\xE8\xE7" => '&#33951;',
        "\xE8\xE8" => '&#34009;',
        "\xE8\xE9" => '&#33997;',
        "\xE8\xEA" => '&#33979;',
        "\xE8\xEB" => '&#34010;',
        "\xE8\xEC" => '&#34000;',
        "\xE8\xED" => '&#33985;',
        "\xE8\xEE" => '&#33990;',
        "\xE8\xEF" => '&#34006;',
        "\xE8\xF0" => '&#33953;',
        "\xE8\xF1" => '&#34081;',
        "\xE8\xF2" => '&#34047;',
        "\xE8\xF3" => '&#34036;',
        "\xE8\xF4" => '&#34071;',
        "\xE8\xF5" => '&#34072;',
        "\xE8\xF6" => '&#34092;',
        "\xE8\xF7" => '&#34079;',
        "\xE8\xF8" => '&#34069;',
        "\xE8\xF9" => '&#34068;',
        "\xE8\xFA" => '&#34044;',
        "\xE8\xFB" => '&#34112;',
        "\xE8\xFC" => '&#34147;',
        "\xE8\xFD" => '&#34136;',
        "\xE8\xFE" => '&#34120;',
        "\xE9\xA1" => '&#34113;',
        "\xE9\xA2" => '&#34306;',
        "\xE9\xA3" => '&#34123;',
        "\xE9\xA4" => '&#34133;',
        "\xE9\xA5" => '&#34176;',
        "\xE9\xA6" => '&#34212;',
        "\xE9\xA7" => '&#34184;',
        "\xE9\xA8" => '&#34193;',
        "\xE9\xA9" => '&#34186;',
        "\xE9\xAA" => '&#34216;',
        "\xE9\xAB" => '&#34157;',
        "\xE9\xAC" => '&#34196;',
        "\xE9\xAD" => '&#34203;',
        "\xE9\xAE" => '&#34282;',
        "\xE9\xAF" => '&#34183;',
        "\xE9\xB0" => '&#34204;',
        "\xE9\xB1" => '&#34167;',
        "\xE9\xB2" => '&#34174;',
        "\xE9\xB3" => '&#34192;',
        "\xE9\xB4" => '&#34249;',
        "\xE9\xB5" => '&#34234;',
        "\xE9\xB6" => '&#34255;',
        "\xE9\xB7" => '&#34233;',
        "\xE9\xB8" => '&#34256;',
        "\xE9\xB9" => '&#34261;',
        "\xE9\xBA" => '&#34269;',
        "\xE9\xBB" => '&#34277;',
        "\xE9\xBC" => '&#34268;',
        "\xE9\xBD" => '&#34297;',
        "\xE9\xBE" => '&#34314;',
        "\xE9\xBF" => '&#34323;',
        "\xE9\xC0" => '&#34315;',
        "\xE9\xC1" => '&#34302;',
        "\xE9\xC2" => '&#34298;',
        "\xE9\xC3" => '&#34310;',
        "\xE9\xC4" => '&#34338;',
        "\xE9\xC5" => '&#34330;',
        "\xE9\xC6" => '&#34352;',
        "\xE9\xC7" => '&#34367;',
        "\xE9\xC8" => '&#34381;',
        "\xE9\xC9" => '&#20053;',
        "\xE9\xCA" => '&#34388;',
        "\xE9\xCB" => '&#34399;',
        "\xE9\xCC" => '&#34407;',
        "\xE9\xCD" => '&#34417;',
        "\xE9\xCE" => '&#34451;',
        "\xE9\xCF" => '&#34467;',
        "\xE9\xD0" => '&#34473;',
        "\xE9\xD1" => '&#34474;',
        "\xE9\xD2" => '&#34443;',
        "\xE9\xD3" => '&#34444;',
        "\xE9\xD4" => '&#34486;',
        "\xE9\xD5" => '&#34479;',
        "\xE9\xD6" => '&#34500;',
        "\xE9\xD7" => '&#34502;',
        "\xE9\xD8" => '&#34480;',
        "\xE9\xD9" => '&#34505;',
        "\xE9\xDA" => '&#34851;',
        "\xE9\xDB" => '&#34475;',
        "\xE9\xDC" => '&#34516;',
        "\xE9\xDD" => '&#34526;',
        "\xE9\xDE" => '&#34537;',
        "\xE9\xDF" => '&#34540;',
        "\xE9\xE0" => '&#34527;',
        "\xE9\xE1" => '&#34523;',
        "\xE9\xE2" => '&#34543;',
        "\xE9\xE3" => '&#34578;',
        "\xE9\xE4" => '&#34566;',
        "\xE9\xE5" => '&#34568;',
        "\xE9\xE6" => '&#34560;',
        "\xE9\xE7" => '&#34563;',
        "\xE9\xE8" => '&#34555;',
        "\xE9\xE9" => '&#34577;',
        "\xE9\xEA" => '&#34569;',
        "\xE9\xEB" => '&#34573;',
        "\xE9\xEC" => '&#34553;',
        "\xE9\xED" => '&#34570;',
        "\xE9\xEE" => '&#34612;',
        "\xE9\xEF" => '&#34623;',
        "\xE9\xF0" => '&#34615;',
        "\xE9\xF1" => '&#34619;',
        "\xE9\xF2" => '&#34597;',
        "\xE9\xF3" => '&#34601;',
        "\xE9\xF4" => '&#34586;',
        "\xE9\xF5" => '&#34656;',
        "\xE9\xF6" => '&#34655;',
        "\xE9\xF7" => '&#34680;',
        "\xE9\xF8" => '&#34636;',
        "\xE9\xF9" => '&#34638;',
        "\xE9\xFA" => '&#34676;',
        "\xE9\xFB" => '&#34647;',
        "\xE9\xFC" => '&#34664;',
        "\xE9\xFD" => '&#34670;',
        "\xE9\xFE" => '&#34649;',
        "\xEA\xA1" => '&#34643;',
        "\xEA\xA2" => '&#34659;',
        "\xEA\xA3" => '&#34666;',
        "\xEA\xA4" => '&#34821;',
        "\xEA\xA5" => '&#34722;',
        "\xEA\xA6" => '&#34719;',
        "\xEA\xA7" => '&#34690;',
        "\xEA\xA8" => '&#34735;',
        "\xEA\xA9" => '&#34763;',
        "\xEA\xAA" => '&#34749;',
        "\xEA\xAB" => '&#34752;',
        "\xEA\xAC" => '&#34768;',
        "\xEA\xAD" => '&#38614;',
        "\xEA\xAE" => '&#34731;',
        "\xEA\xAF" => '&#34756;',
        "\xEA\xB0" => '&#34739;',
        "\xEA\xB1" => '&#34759;',
        "\xEA\xB2" => '&#34758;',
        "\xEA\xB3" => '&#34747;',
        "\xEA\xB4" => '&#34799;',
        "\xEA\xB5" => '&#34802;',
        "\xEA\xB6" => '&#34784;',
        "\xEA\xB7" => '&#34831;',
        "\xEA\xB8" => '&#34829;',
        "\xEA\xB9" => '&#34814;',
        "\xEA\xBA" => '&#34806;',
        "\xEA\xBB" => '&#34807;',
        "\xEA\xBC" => '&#34830;',
        "\xEA\xBD" => '&#34770;',
        "\xEA\xBE" => '&#34833;',
        "\xEA\xBF" => '&#34838;',
        "\xEA\xC0" => '&#34837;',
        "\xEA\xC1" => '&#34850;',
        "\xEA\xC2" => '&#34849;',
        "\xEA\xC3" => '&#34865;',
        "\xEA\xC4" => '&#34870;',
        "\xEA\xC5" => '&#34873;',
        "\xEA\xC6" => '&#34855;',
        "\xEA\xC7" => '&#34875;',
        "\xEA\xC8" => '&#34884;',
        "\xEA\xC9" => '&#34882;',
        "\xEA\xCA" => '&#34898;',
        "\xEA\xCB" => '&#34905;',
        "\xEA\xCC" => '&#34910;',
        "\xEA\xCD" => '&#34914;',
        "\xEA\xCE" => '&#34923;',
        "\xEA\xCF" => '&#34945;',
        "\xEA\xD0" => '&#34942;',
        "\xEA\xD1" => '&#34974;',
        "\xEA\xD2" => '&#34933;',
        "\xEA\xD3" => '&#34941;',
        "\xEA\xD4" => '&#34997;',
        "\xEA\xD5" => '&#34930;',
        "\xEA\xD6" => '&#34946;',
        "\xEA\xD7" => '&#34967;',
        "\xEA\xD8" => '&#34962;',
        "\xEA\xD9" => '&#34990;',
        "\xEA\xDA" => '&#34969;',
        "\xEA\xDB" => '&#34978;',
        "\xEA\xDC" => '&#34957;',
        "\xEA\xDD" => '&#34980;',
        "\xEA\xDE" => '&#34992;',
        "\xEA\xDF" => '&#35007;',
        "\xEA\xE0" => '&#34993;',
        "\xEA\xE1" => '&#35011;',
        "\xEA\xE2" => '&#35012;',
        "\xEA\xE3" => '&#35028;',
        "\xEA\xE4" => '&#35032;',
        "\xEA\xE5" => '&#35033;',
        "\xEA\xE6" => '&#35037;',
        "\xEA\xE7" => '&#35065;',
        "\xEA\xE8" => '&#35074;',
        "\xEA\xE9" => '&#35068;',
        "\xEA\xEA" => '&#35060;',
        "\xEA\xEB" => '&#35048;',
        "\xEA\xEC" => '&#35058;',
        "\xEA\xED" => '&#35076;',
        "\xEA\xEE" => '&#35084;',
        "\xEA\xEF" => '&#35082;',
        "\xEA\xF0" => '&#35091;',
        "\xEA\xF1" => '&#35139;',
        "\xEA\xF2" => '&#35102;',
        "\xEA\xF3" => '&#35109;',
        "\xEA\xF4" => '&#35114;',
        "\xEA\xF5" => '&#35115;',
        "\xEA\xF6" => '&#35137;',
        "\xEA\xF7" => '&#35140;',
        "\xEA\xF8" => '&#35131;',
        "\xEA\xF9" => '&#35126;',
        "\xEA\xFA" => '&#35128;',
        "\xEA\xFB" => '&#35148;',
        "\xEA\xFC" => '&#35101;',
        "\xEA\xFD" => '&#35168;',
        "\xEA\xFE" => '&#35166;',
        "\xEB\xA1" => '&#35174;',
        "\xEB\xA2" => '&#35172;',
        "\xEB\xA3" => '&#35181;',
        "\xEB\xA4" => '&#35178;',
        "\xEB\xA5" => '&#35183;',
        "\xEB\xA6" => '&#35188;',
        "\xEB\xA7" => '&#35191;',
        "\xEB\xA8" => '&#35198;',
        "\xEB\xA9" => '&#35203;',
        "\xEB\xAA" => '&#35208;',
        "\xEB\xAB" => '&#35210;',
        "\xEB\xAC" => '&#35219;',
        "\xEB\xAD" => '&#35224;',
        "\xEB\xAE" => '&#35233;',
        "\xEB\xAF" => '&#35241;',
        "\xEB\xB0" => '&#35238;',
        "\xEB\xB1" => '&#35244;',
        "\xEB\xB2" => '&#35247;',
        "\xEB\xB3" => '&#35250;',
        "\xEB\xB4" => '&#35258;',
        "\xEB\xB5" => '&#35261;',
        "\xEB\xB6" => '&#35263;',
        "\xEB\xB7" => '&#35264;',
        "\xEB\xB8" => '&#35290;',
        "\xEB\xB9" => '&#35292;',
        "\xEB\xBA" => '&#35293;',
        "\xEB\xBB" => '&#35303;',
        "\xEB\xBC" => '&#35316;',
        "\xEB\xBD" => '&#35320;',
        "\xEB\xBE" => '&#35331;',
        "\xEB\xBF" => '&#35350;',
        "\xEB\xC0" => '&#35344;',
        "\xEB\xC1" => '&#35340;',
        "\xEB\xC2" => '&#35355;',
        "\xEB\xC3" => '&#35357;',
        "\xEB\xC4" => '&#35365;',
        "\xEB\xC5" => '&#35382;',
        "\xEB\xC6" => '&#35393;',
        "\xEB\xC7" => '&#35419;',
        "\xEB\xC8" => '&#35410;',
        "\xEB\xC9" => '&#35398;',
        "\xEB\xCA" => '&#35400;',
        "\xEB\xCB" => '&#35452;',
        "\xEB\xCC" => '&#35437;',
        "\xEB\xCD" => '&#35436;',
        "\xEB\xCE" => '&#35426;',
        "\xEB\xCF" => '&#35461;',
        "\xEB\xD0" => '&#35458;',
        "\xEB\xD1" => '&#35460;',
        "\xEB\xD2" => '&#35496;',
        "\xEB\xD3" => '&#35489;',
        "\xEB\xD4" => '&#35473;',
        "\xEB\xD5" => '&#35493;',
        "\xEB\xD6" => '&#35494;',
        "\xEB\xD7" => '&#35482;',
        "\xEB\xD8" => '&#35491;',
        "\xEB\xD9" => '&#35524;',
        "\xEB\xDA" => '&#35533;',
        "\xEB\xDB" => '&#35522;',
        "\xEB\xDC" => '&#35546;',
        "\xEB\xDD" => '&#35563;',
        "\xEB\xDE" => '&#35571;',
        "\xEB\xDF" => '&#35559;',
        "\xEB\xE0" => '&#35556;',
        "\xEB\xE1" => '&#35569;',
        "\xEB\xE2" => '&#35604;',
        "\xEB\xE3" => '&#35552;',
        "\xEB\xE4" => '&#35554;',
        "\xEB\xE5" => '&#35575;',
        "\xEB\xE6" => '&#35550;',
        "\xEB\xE7" => '&#35547;',
        "\xEB\xE8" => '&#35596;',
        "\xEB\xE9" => '&#35591;',
        "\xEB\xEA" => '&#35610;',
        "\xEB\xEB" => '&#35553;',
        "\xEB\xEC" => '&#35606;',
        "\xEB\xED" => '&#35600;',
        "\xEB\xEE" => '&#35607;',
        "\xEB\xEF" => '&#35616;',
        "\xEB\xF0" => '&#35635;',
        "\xEB\xF1" => '&#38827;',
        "\xEB\xF2" => '&#35622;',
        "\xEB\xF3" => '&#35627;',
        "\xEB\xF4" => '&#35646;',
        "\xEB\xF5" => '&#35624;',
        "\xEB\xF6" => '&#35649;',
        "\xEB\xF7" => '&#35660;',
        "\xEB\xF8" => '&#35663;',
        "\xEB\xF9" => '&#35662;',
        "\xEB\xFA" => '&#35657;',
        "\xEB\xFB" => '&#35670;',
        "\xEB\xFC" => '&#35675;',
        "\xEB\xFD" => '&#35674;',
        "\xEB\xFE" => '&#35691;',
        "\xEC\xA1" => '&#35679;',
        "\xEC\xA2" => '&#35692;',
        "\xEC\xA3" => '&#35695;',
        "\xEC\xA4" => '&#35700;',
        "\xEC\xA5" => '&#35709;',
        "\xEC\xA6" => '&#35712;',
        "\xEC\xA7" => '&#35724;',
        "\xEC\xA8" => '&#35726;',
        "\xEC\xA9" => '&#35730;',
        "\xEC\xAA" => '&#35731;',
        "\xEC\xAB" => '&#35734;',
        "\xEC\xAC" => '&#35737;',
        "\xEC\xAD" => '&#35738;',
        "\xEC\xAE" => '&#35898;',
        "\xEC\xAF" => '&#35905;',
        "\xEC\xB0" => '&#35903;',
        "\xEC\xB1" => '&#35912;',
        "\xEC\xB2" => '&#35916;',
        "\xEC\xB3" => '&#35918;',
        "\xEC\xB4" => '&#35920;',
        "\xEC\xB5" => '&#35925;',
        "\xEC\xB6" => '&#35938;',
        "\xEC\xB7" => '&#35948;',
        "\xEC\xB8" => '&#35960;',
        "\xEC\xB9" => '&#35962;',
        "\xEC\xBA" => '&#35970;',
        "\xEC\xBB" => '&#35977;',
        "\xEC\xBC" => '&#35973;',
        "\xEC\xBD" => '&#35978;',
        "\xEC\xBE" => '&#35981;',
        "\xEC\xBF" => '&#35982;',
        "\xEC\xC0" => '&#35988;',
        "\xEC\xC1" => '&#35964;',
        "\xEC\xC2" => '&#35992;',
        "\xEC\xC3" => '&#25117;',
        "\xEC\xC4" => '&#36013;',
        "\xEC\xC5" => '&#36010;',
        "\xEC\xC6" => '&#36029;',
        "\xEC\xC7" => '&#36018;',
        "\xEC\xC8" => '&#36019;',
        "\xEC\xC9" => '&#36014;',
        "\xEC\xCA" => '&#36022;',
        "\xEC\xCB" => '&#36040;',
        "\xEC\xCC" => '&#36033;',
        "\xEC\xCD" => '&#36068;',
        "\xEC\xCE" => '&#36067;',
        "\xEC\xCF" => '&#36058;',
        "\xEC\xD0" => '&#36093;',
        "\xEC\xD1" => '&#36090;',
        "\xEC\xD2" => '&#36091;',
        "\xEC\xD3" => '&#36100;',
        "\xEC\xD4" => '&#36101;',
        "\xEC\xD5" => '&#36106;',
        "\xEC\xD6" => '&#36103;',
        "\xEC\xD7" => '&#36111;',
        "\xEC\xD8" => '&#36109;',
        "\xEC\xD9" => '&#36112;',
        "\xEC\xDA" => '&#40782;',
        "\xEC\xDB" => '&#36115;',
        "\xEC\xDC" => '&#36045;',
        "\xEC\xDD" => '&#36116;',
        "\xEC\xDE" => '&#36118;',
        "\xEC\xDF" => '&#36199;',
        "\xEC\xE0" => '&#36205;',
        "\xEC\xE1" => '&#36209;',
        "\xEC\xE2" => '&#36211;',
        "\xEC\xE3" => '&#36225;',
        "\xEC\xE4" => '&#36249;',
        "\xEC\xE5" => '&#36290;',
        "\xEC\xE6" => '&#36286;',
        "\xEC\xE7" => '&#36282;',
        "\xEC\xE8" => '&#36303;',
        "\xEC\xE9" => '&#36314;',
        "\xEC\xEA" => '&#36310;',
        "\xEC\xEB" => '&#36300;',
        "\xEC\xEC" => '&#36315;',
        "\xEC\xED" => '&#36299;',
        "\xEC\xEE" => '&#36330;',
        "\xEC\xEF" => '&#36331;',
        "\xEC\xF0" => '&#36319;',
        "\xEC\xF1" => '&#36323;',
        "\xEC\xF2" => '&#36348;',
        "\xEC\xF3" => '&#36360;',
        "\xEC\xF4" => '&#36361;',
        "\xEC\xF5" => '&#36351;',
        "\xEC\xF6" => '&#36381;',
        "\xEC\xF7" => '&#36382;',
        "\xEC\xF8" => '&#36368;',
        "\xEC\xF9" => '&#36383;',
        "\xEC\xFA" => '&#36418;',
        "\xEC\xFB" => '&#36405;',
        "\xEC\xFC" => '&#36400;',
        "\xEC\xFD" => '&#36404;',
        "\xEC\xFE" => '&#36426;',
        "\xED\xA1" => '&#36423;',
        "\xED\xA2" => '&#36425;',
        "\xED\xA3" => '&#36428;',
        "\xED\xA4" => '&#36432;',
        "\xED\xA5" => '&#36424;',
        "\xED\xA6" => '&#36441;',
        "\xED\xA7" => '&#36452;',
        "\xED\xA8" => '&#36448;',
        "\xED\xA9" => '&#36394;',
        "\xED\xAA" => '&#36451;',
        "\xED\xAB" => '&#36437;',
        "\xED\xAC" => '&#36470;',
        "\xED\xAD" => '&#36466;',
        "\xED\xAE" => '&#36476;',
        "\xED\xAF" => '&#36481;',
        "\xED\xB0" => '&#36487;',
        "\xED\xB1" => '&#36485;',
        "\xED\xB2" => '&#36484;',
        "\xED\xB3" => '&#36491;',
        "\xED\xB4" => '&#36490;',
        "\xED\xB5" => '&#36499;',
        "\xED\xB6" => '&#36497;',
        "\xED\xB7" => '&#36500;',
        "\xED\xB8" => '&#36505;',
        "\xED\xB9" => '&#36522;',
        "\xED\xBA" => '&#36513;',
        "\xED\xBB" => '&#36524;',
        "\xED\xBC" => '&#36528;',
        "\xED\xBD" => '&#36550;',
        "\xED\xBE" => '&#36529;',
        "\xED\xBF" => '&#36542;',
        "\xED\xC0" => '&#36549;',
        "\xED\xC1" => '&#36552;',
        "\xED\xC2" => '&#36555;',
        "\xED\xC3" => '&#36571;',
        "\xED\xC4" => '&#36579;',
        "\xED\xC5" => '&#36604;',
        "\xED\xC6" => '&#36603;',
        "\xED\xC7" => '&#36587;',
        "\xED\xC8" => '&#36606;',
        "\xED\xC9" => '&#36618;',
        "\xED\xCA" => '&#36613;',
        "\xED\xCB" => '&#36629;',
        "\xED\xCC" => '&#36626;',
        "\xED\xCD" => '&#36633;',
        "\xED\xCE" => '&#36627;',
        "\xED\xCF" => '&#36636;',
        "\xED\xD0" => '&#36639;',
        "\xED\xD1" => '&#36635;',
        "\xED\xD2" => '&#36620;',
        "\xED\xD3" => '&#36646;',
        "\xED\xD4" => '&#36659;',
        "\xED\xD5" => '&#36667;',
        "\xED\xD6" => '&#36665;',
        "\xED\xD7" => '&#36677;',
        "\xED\xD8" => '&#36674;',
        "\xED\xD9" => '&#36670;',
        "\xED\xDA" => '&#36684;',
        "\xED\xDB" => '&#36681;',
        "\xED\xDC" => '&#36678;',
        "\xED\xDD" => '&#36686;',
        "\xED\xDE" => '&#36695;',
        "\xED\xDF" => '&#36700;',
        "\xED\xE0" => '&#36706;',
        "\xED\xE1" => '&#36707;',
        "\xED\xE2" => '&#36708;',
        "\xED\xE3" => '&#36764;',
        "\xED\xE4" => '&#36767;',
        "\xED\xE5" => '&#36771;',
        "\xED\xE6" => '&#36781;',
        "\xED\xE7" => '&#36783;',
        "\xED\xE8" => '&#36791;',
        "\xED\xE9" => '&#36826;',
        "\xED\xEA" => '&#36837;',
        "\xED\xEB" => '&#36834;',
        "\xED\xEC" => '&#36842;',
        "\xED\xED" => '&#36847;',
        "\xED\xEE" => '&#36999;',
        "\xED\xEF" => '&#36852;',
        "\xED\xF0" => '&#36869;',
        "\xED\xF1" => '&#36857;',
        "\xED\xF2" => '&#36858;',
        "\xED\xF3" => '&#36881;',
        "\xED\xF4" => '&#36885;',
        "\xED\xF5" => '&#36897;',
        "\xED\xF6" => '&#36877;',
        "\xED\xF7" => '&#36894;',
        "\xED\xF8" => '&#36886;',
        "\xED\xF9" => '&#36875;',
        "\xED\xFA" => '&#36903;',
        "\xED\xFB" => '&#36918;',
        "\xED\xFC" => '&#36917;',
        "\xED\xFD" => '&#36921;',
        "\xED\xFE" => '&#36856;',
        "\xEE\xA1" => '&#36943;',
        "\xEE\xA2" => '&#36944;',
        "\xEE\xA3" => '&#36945;',
        "\xEE\xA4" => '&#36946;',
        "\xEE\xA5" => '&#36878;',
        "\xEE\xA6" => '&#36937;',
        "\xEE\xA7" => '&#36926;',
        "\xEE\xA8" => '&#36950;',
        "\xEE\xA9" => '&#36952;',
        "\xEE\xAA" => '&#36958;',
        "\xEE\xAB" => '&#36968;',
        "\xEE\xAC" => '&#36975;',
        "\xEE\xAD" => '&#36982;',
        "\xEE\xAE" => '&#38568;',
        "\xEE\xAF" => '&#36978;',
        "\xEE\xB0" => '&#36994;',
        "\xEE\xB1" => '&#36989;',
        "\xEE\xB2" => '&#36993;',
        "\xEE\xB3" => '&#36992;',
        "\xEE\xB4" => '&#37002;',
        "\xEE\xB5" => '&#37001;',
        "\xEE\xB6" => '&#37007;',
        "\xEE\xB7" => '&#37032;',
        "\xEE\xB8" => '&#37039;',
        "\xEE\xB9" => '&#37041;',
        "\xEE\xBA" => '&#37045;',
        "\xEE\xBB" => '&#37090;',
        "\xEE\xBC" => '&#37092;',
        "\xEE\xBD" => '&#25160;',
        "\xEE\xBE" => '&#37083;',
        "\xEE\xBF" => '&#37122;',
        "\xEE\xC0" => '&#37138;',
        "\xEE\xC1" => '&#37145;',
        "\xEE\xC2" => '&#37170;',
        "\xEE\xC3" => '&#37168;',
        "\xEE\xC4" => '&#37194;',
        "\xEE\xC5" => '&#37206;',
        "\xEE\xC6" => '&#37208;',
        "\xEE\xC7" => '&#37219;',
        "\xEE\xC8" => '&#37221;',
        "\xEE\xC9" => '&#37225;',
        "\xEE\xCA" => '&#37235;',
        "\xEE\xCB" => '&#37234;',
        "\xEE\xCC" => '&#37259;',
        "\xEE\xCD" => '&#37257;',
        "\xEE\xCE" => '&#37250;',
        "\xEE\xCF" => '&#37282;',
        "\xEE\xD0" => '&#37291;',
        "\xEE\xD1" => '&#37295;',
        "\xEE\xD2" => '&#37290;',
        "\xEE\xD3" => '&#37301;',
        "\xEE\xD4" => '&#37300;',
        "\xEE\xD5" => '&#37306;',
        "\xEE\xD6" => '&#37312;',
        "\xEE\xD7" => '&#37313;',
        "\xEE\xD8" => '&#37321;',
        "\xEE\xD9" => '&#37323;',
        "\xEE\xDA" => '&#37328;',
        "\xEE\xDB" => '&#37334;',
        "\xEE\xDC" => '&#37343;',
        "\xEE\xDD" => '&#37345;',
        "\xEE\xDE" => '&#37339;',
        "\xEE\xDF" => '&#37372;',
        "\xEE\xE0" => '&#37365;',
        "\xEE\xE1" => '&#37366;',
        "\xEE\xE2" => '&#37406;',
        "\xEE\xE3" => '&#37375;',
        "\xEE\xE4" => '&#37396;',
        "\xEE\xE5" => '&#37420;',
        "\xEE\xE6" => '&#37397;',
        "\xEE\xE7" => '&#37393;',
        "\xEE\xE8" => '&#37470;',
        "\xEE\xE9" => '&#37463;',
        "\xEE\xEA" => '&#37445;',
        "\xEE\xEB" => '&#37449;',
        "\xEE\xEC" => '&#37476;',
        "\xEE\xED" => '&#37448;',
        "\xEE\xEE" => '&#37525;',
        "\xEE\xEF" => '&#37439;',
        "\xEE\xF0" => '&#37451;',
        "\xEE\xF1" => '&#37456;',
        "\xEE\xF2" => '&#37532;',
        "\xEE\xF3" => '&#37526;',
        "\xEE\xF4" => '&#37523;',
        "\xEE\xF5" => '&#37531;',
        "\xEE\xF6" => '&#37466;',
        "\xEE\xF7" => '&#37583;',
        "\xEE\xF8" => '&#37561;',
        "\xEE\xF9" => '&#37559;',
        "\xEE\xFA" => '&#37609;',
        "\xEE\xFB" => '&#37647;',
        "\xEE\xFC" => '&#37626;',
        "\xEE\xFD" => '&#37700;',
        "\xEE\xFE" => '&#37678;',
        "\xEF\xA1" => '&#37657;',
        "\xEF\xA2" => '&#37666;',
        "\xEF\xA3" => '&#37658;',
        "\xEF\xA4" => '&#37667;',
        "\xEF\xA5" => '&#37690;',
        "\xEF\xA6" => '&#37685;',
        "\xEF\xA7" => '&#37691;',
        "\xEF\xA8" => '&#37724;',
        "\xEF\xA9" => '&#37728;',
        "\xEF\xAA" => '&#37756;',
        "\xEF\xAB" => '&#37742;',
        "\xEF\xAC" => '&#37718;',
        "\xEF\xAD" => '&#37808;',
        "\xEF\xAE" => '&#37804;',
        "\xEF\xAF" => '&#37805;',
        "\xEF\xB0" => '&#37780;',
        "\xEF\xB1" => '&#37817;',
        "\xEF\xB2" => '&#37846;',
        "\xEF\xB3" => '&#37847;',
        "\xEF\xB4" => '&#37864;',
        "\xEF\xB5" => '&#37861;',
        "\xEF\xB6" => '&#37848;',
        "\xEF\xB7" => '&#37827;',
        "\xEF\xB8" => '&#37853;',
        "\xEF\xB9" => '&#37840;',
        "\xEF\xBA" => '&#37832;',
        "\xEF\xBB" => '&#37860;',
        "\xEF\xBC" => '&#37914;',
        "\xEF\xBD" => '&#37908;',
        "\xEF\xBE" => '&#37907;',
        "\xEF\xBF" => '&#37891;',
        "\xEF\xC0" => '&#37895;',
        "\xEF\xC1" => '&#37904;',
        "\xEF\xC2" => '&#37942;',
        "\xEF\xC3" => '&#37931;',
        "\xEF\xC4" => '&#37941;',
        "\xEF\xC5" => '&#37921;',
        "\xEF\xC6" => '&#37946;',
        "\xEF\xC7" => '&#37953;',
        "\xEF\xC8" => '&#37970;',
        "\xEF\xC9" => '&#37956;',
        "\xEF\xCA" => '&#37979;',
        "\xEF\xCB" => '&#37984;',
        "\xEF\xCC" => '&#37986;',
        "\xEF\xCD" => '&#37982;',
        "\xEF\xCE" => '&#37994;',
        "\xEF\xCF" => '&#37417;',
        "\xEF\xD0" => '&#38000;',
        "\xEF\xD1" => '&#38005;',
        "\xEF\xD2" => '&#38007;',
        "\xEF\xD3" => '&#38013;',
        "\xEF\xD4" => '&#37978;',
        "\xEF\xD5" => '&#38012;',
        "\xEF\xD6" => '&#38014;',
        "\xEF\xD7" => '&#38017;',
        "\xEF\xD8" => '&#38015;',
        "\xEF\xD9" => '&#38274;',
        "\xEF\xDA" => '&#38279;',
        "\xEF\xDB" => '&#38282;',
        "\xEF\xDC" => '&#38292;',
        "\xEF\xDD" => '&#38294;',
        "\xEF\xDE" => '&#38296;',
        "\xEF\xDF" => '&#38297;',
        "\xEF\xE0" => '&#38304;',
        "\xEF\xE1" => '&#38312;',
        "\xEF\xE2" => '&#38311;',
        "\xEF\xE3" => '&#38317;',
        "\xEF\xE4" => '&#38332;',
        "\xEF\xE5" => '&#38331;',
        "\xEF\xE6" => '&#38329;',
        "\xEF\xE7" => '&#38334;',
        "\xEF\xE8" => '&#38346;',
        "\xEF\xE9" => '&#28662;',
        "\xEF\xEA" => '&#38339;',
        "\xEF\xEB" => '&#38349;',
        "\xEF\xEC" => '&#38348;',
        "\xEF\xED" => '&#38357;',
        "\xEF\xEE" => '&#38356;',
        "\xEF\xEF" => '&#38358;',
        "\xEF\xF0" => '&#38364;',
        "\xEF\xF1" => '&#38369;',
        "\xEF\xF2" => '&#38373;',
        "\xEF\xF3" => '&#38370;',
        "\xEF\xF4" => '&#38433;',
        "\xEF\xF5" => '&#38440;',
        "\xEF\xF6" => '&#38446;',
        "\xEF\xF7" => '&#38447;',
        "\xEF\xF8" => '&#38466;',
        "\xEF\xF9" => '&#38476;',
        "\xEF\xFA" => '&#38479;',
        "\xEF\xFB" => '&#38475;',
        "\xEF\xFC" => '&#38519;',
        "\xEF\xFD" => '&#38492;',
        "\xEF\xFE" => '&#38494;',
        "\xF0\xA1" => '&#38493;',
        "\xF0\xA2" => '&#38495;',
        "\xF0\xA3" => '&#38502;',
        "\xF0\xA4" => '&#38514;',
        "\xF0\xA5" => '&#38508;',
        "\xF0\xA6" => '&#38541;',
        "\xF0\xA7" => '&#38552;',
        "\xF0\xA8" => '&#38549;',
        "\xF0\xA9" => '&#38551;',
        "\xF0\xAA" => '&#38570;',
        "\xF0\xAB" => '&#38567;',
        "\xF0\xAC" => '&#38577;',
        "\xF0\xAD" => '&#38578;',
        "\xF0\xAE" => '&#38576;',
        "\xF0\xAF" => '&#38580;',
        "\xF0\xB0" => '&#38582;',
        "\xF0\xB1" => '&#38584;',
        "\xF0\xB2" => '&#38585;',
        "\xF0\xB3" => '&#38606;',
        "\xF0\xB4" => '&#38603;',
        "\xF0\xB5" => '&#38601;',
        "\xF0\xB6" => '&#38605;',
        "\xF0\xB7" => '&#35149;',
        "\xF0\xB8" => '&#38620;',
        "\xF0\xB9" => '&#38669;',
        "\xF0\xBA" => '&#38613;',
        "\xF0\xBB" => '&#38649;',
        "\xF0\xBC" => '&#38660;',
        "\xF0\xBD" => '&#38662;',
        "\xF0\xBE" => '&#38664;',
        "\xF0\xBF" => '&#38675;',
        "\xF0\xC0" => '&#38670;',
        "\xF0\xC1" => '&#38673;',
        "\xF0\xC2" => '&#38671;',
        "\xF0\xC3" => '&#38678;',
        "\xF0\xC4" => '&#38681;',
        "\xF0\xC5" => '&#38692;',
        "\xF0\xC6" => '&#38698;',
        "\xF0\xC7" => '&#38704;',
        "\xF0\xC8" => '&#38713;',
        "\xF0\xC9" => '&#38717;',
        "\xF0\xCA" => '&#38718;',
        "\xF0\xCB" => '&#38724;',
        "\xF0\xCC" => '&#38726;',
        "\xF0\xCD" => '&#38728;',
        "\xF0\xCE" => '&#38722;',
        "\xF0\xCF" => '&#38729;',
        "\xF0\xD0" => '&#38748;',
        "\xF0\xD1" => '&#38752;',
        "\xF0\xD2" => '&#38756;',
        "\xF0\xD3" => '&#38758;',
        "\xF0\xD4" => '&#38760;',
        "\xF0\xD5" => '&#21202;',
        "\xF0\xD6" => '&#38763;',
        "\xF0\xD7" => '&#38769;',
        "\xF0\xD8" => '&#38777;',
        "\xF0\xD9" => '&#38789;',
        "\xF0\xDA" => '&#38780;',
        "\xF0\xDB" => '&#38785;',
        "\xF0\xDC" => '&#38778;',
        "\xF0\xDD" => '&#38790;',
        "\xF0\xDE" => '&#38795;',
        "\xF0\xDF" => '&#38799;',
        "\xF0\xE0" => '&#38800;',
        "\xF0\xE1" => '&#38812;',
        "\xF0\xE2" => '&#38824;',
        "\xF0\xE3" => '&#38822;',
        "\xF0\xE4" => '&#38819;',
        "\xF0\xE5" => '&#38835;',
        "\xF0\xE6" => '&#38836;',
        "\xF0\xE7" => '&#38851;',
        "\xF0\xE8" => '&#38854;',
        "\xF0\xE9" => '&#38856;',
        "\xF0\xEA" => '&#38859;',
        "\xF0\xEB" => '&#38876;',
        "\xF0\xEC" => '&#38893;',
        "\xF0\xED" => '&#40783;',
        "\xF0\xEE" => '&#38898;',
        "\xF0\xEF" => '&#31455;',
        "\xF0\xF0" => '&#38902;',
        "\xF0\xF1" => '&#38901;',
        "\xF0\xF2" => '&#38927;',
        "\xF0\xF3" => '&#38924;',
        "\xF0\xF4" => '&#38968;',
        "\xF0\xF5" => '&#38948;',
        "\xF0\xF6" => '&#38945;',
        "\xF0\xF7" => '&#38967;',
        "\xF0\xF8" => '&#38973;',
        "\xF0\xF9" => '&#38982;',
        "\xF0\xFA" => '&#38991;',
        "\xF0\xFB" => '&#38987;',
        "\xF0\xFC" => '&#39019;',
        "\xF0\xFD" => '&#39023;',
        "\xF0\xFE" => '&#39024;',
        "\xF1\xA1" => '&#39025;',
        "\xF1\xA2" => '&#39028;',
        "\xF1\xA3" => '&#39027;',
        "\xF1\xA4" => '&#39082;',
        "\xF1\xA5" => '&#39087;',
        "\xF1\xA6" => '&#39089;',
        "\xF1\xA7" => '&#39094;',
        "\xF1\xA8" => '&#39108;',
        "\xF1\xA9" => '&#39107;',
        "\xF1\xAA" => '&#39110;',
        "\xF1\xAB" => '&#39145;',
        "\xF1\xAC" => '&#39147;',
        "\xF1\xAD" => '&#39171;',
        "\xF1\xAE" => '&#39177;',
        "\xF1\xAF" => '&#39186;',
        "\xF1\xB0" => '&#39188;',
        "\xF1\xB1" => '&#39192;',
        "\xF1\xB2" => '&#39201;',
        "\xF1\xB3" => '&#39197;',
        "\xF1\xB4" => '&#39198;',
        "\xF1\xB5" => '&#39204;',
        "\xF1\xB6" => '&#39200;',
        "\xF1\xB7" => '&#39212;',
        "\xF1\xB8" => '&#39214;',
        "\xF1\xB9" => '&#39229;',
        "\xF1\xBA" => '&#39230;',
        "\xF1\xBB" => '&#39234;',
        "\xF1\xBC" => '&#39241;',
        "\xF1\xBD" => '&#39237;',
        "\xF1\xBE" => '&#39248;',
        "\xF1\xBF" => '&#39243;',
        "\xF1\xC0" => '&#39249;',
        "\xF1\xC1" => '&#39250;',
        "\xF1\xC2" => '&#39244;',
        "\xF1\xC3" => '&#39253;',
        "\xF1\xC4" => '&#39319;',
        "\xF1\xC5" => '&#39320;',
        "\xF1\xC6" => '&#39333;',
        "\xF1\xC7" => '&#39341;',
        "\xF1\xC8" => '&#39342;',
        "\xF1\xC9" => '&#39356;',
        "\xF1\xCA" => '&#39391;',
        "\xF1\xCB" => '&#39387;',
        "\xF1\xCC" => '&#39389;',
        "\xF1\xCD" => '&#39384;',
        "\xF1\xCE" => '&#39377;',
        "\xF1\xCF" => '&#39405;',
        "\xF1\xD0" => '&#39406;',
        "\xF1\xD1" => '&#39409;',
        "\xF1\xD2" => '&#39410;',
        "\xF1\xD3" => '&#39419;',
        "\xF1\xD4" => '&#39416;',
        "\xF1\xD5" => '&#39425;',
        "\xF1\xD6" => '&#39439;',
        "\xF1\xD7" => '&#39429;',
        "\xF1\xD8" => '&#39394;',
        "\xF1\xD9" => '&#39449;',
        "\xF1\xDA" => '&#39467;',
        "\xF1\xDB" => '&#39479;',
        "\xF1\xDC" => '&#39493;',
        "\xF1\xDD" => '&#39490;',
        "\xF1\xDE" => '&#39488;',
        "\xF1\xDF" => '&#39491;',
        "\xF1\xE0" => '&#39486;',
        "\xF1\xE1" => '&#39509;',
        "\xF1\xE2" => '&#39501;',
        "\xF1\xE3" => '&#39515;',
        "\xF1\xE4" => '&#39511;',
        "\xF1\xE5" => '&#39519;',
        "\xF1\xE6" => '&#39522;',
        "\xF1\xE7" => '&#39525;',
        "\xF1\xE8" => '&#39524;',
        "\xF1\xE9" => '&#39529;',
        "\xF1\xEA" => '&#39531;',
        "\xF1\xEB" => '&#39530;',
        "\xF1\xEC" => '&#39597;',
        "\xF1\xED" => '&#39600;',
        "\xF1\xEE" => '&#39612;',
        "\xF1\xEF" => '&#39616;',
        "\xF1\xF0" => '&#39631;',
        "\xF1\xF1" => '&#39633;',
        "\xF1\xF2" => '&#39635;',
        "\xF1\xF3" => '&#39636;',
        "\xF1\xF4" => '&#39646;',
        "\xF1\xF5" => '&#39647;',
        "\xF1\xF6" => '&#39650;',
        "\xF1\xF7" => '&#39651;',
        "\xF1\xF8" => '&#39654;',
        "\xF1\xF9" => '&#39663;',
        "\xF1\xFA" => '&#39659;',
        "\xF1\xFB" => '&#39662;',
        "\xF1\xFC" => '&#39668;',
        "\xF1\xFD" => '&#39665;',
        "\xF1\xFE" => '&#39671;',
        "\xF2\xA1" => '&#39675;',
        "\xF2\xA2" => '&#39686;',
        "\xF2\xA3" => '&#39704;',
        "\xF2\xA4" => '&#39706;',
        "\xF2\xA5" => '&#39711;',
        "\xF2\xA6" => '&#39714;',
        "\xF2\xA7" => '&#39715;',
        "\xF2\xA8" => '&#39717;',
        "\xF2\xA9" => '&#39719;',
        "\xF2\xAA" => '&#39720;',
        "\xF2\xAB" => '&#39721;',
        "\xF2\xAC" => '&#39722;',
        "\xF2\xAD" => '&#39726;',
        "\xF2\xAE" => '&#39727;',
        "\xF2\xAF" => '&#39730;',
        "\xF2\xB0" => '&#39748;',
        "\xF2\xB1" => '&#39747;',
        "\xF2\xB2" => '&#39759;',
        "\xF2\xB3" => '&#39757;',
        "\xF2\xB4" => '&#39758;',
        "\xF2\xB5" => '&#39761;',
        "\xF2\xB6" => '&#39768;',
        "\xF2\xB7" => '&#39796;',
        "\xF2\xB8" => '&#39827;',
        "\xF2\xB9" => '&#39811;',
        "\xF2\xBA" => '&#39825;',
        "\xF2\xBB" => '&#39830;',
        "\xF2\xBC" => '&#39831;',
        "\xF2\xBD" => '&#39839;',
        "\xF2\xBE" => '&#39840;',
        "\xF2\xBF" => '&#39848;',
        "\xF2\xC0" => '&#39860;',
        "\xF2\xC1" => '&#39872;',
        "\xF2\xC2" => '&#39882;',
        "\xF2\xC3" => '&#39865;',
        "\xF2\xC4" => '&#39878;',
        "\xF2\xC5" => '&#39887;',
        "\xF2\xC6" => '&#39889;',
        "\xF2\xC7" => '&#39890;',
        "\xF2\xC8" => '&#39907;',
        "\xF2\xC9" => '&#39906;',
        "\xF2\xCA" => '&#39908;',
        "\xF2\xCB" => '&#39892;',
        "\xF2\xCC" => '&#39905;',
        "\xF2\xCD" => '&#39994;',
        "\xF2\xCE" => '&#39922;',
        "\xF2\xCF" => '&#39921;',
        "\xF2\xD0" => '&#39920;',
        "\xF2\xD1" => '&#39957;',
        "\xF2\xD2" => '&#39956;',
        "\xF2\xD3" => '&#39945;',
        "\xF2\xD4" => '&#39955;',
        "\xF2\xD5" => '&#39948;',
        "\xF2\xD6" => '&#39942;',
        "\xF2\xD7" => '&#39944;',
        "\xF2\xD8" => '&#39954;',
        "\xF2\xD9" => '&#39946;',
        "\xF2\xDA" => '&#39940;',
        "\xF2\xDB" => '&#39982;',
        "\xF2\xDC" => '&#39963;',
        "\xF2\xDD" => '&#39973;',
        "\xF2\xDE" => '&#39972;',
        "\xF2\xDF" => '&#39969;',
        "\xF2\xE0" => '&#39984;',
        "\xF2\xE1" => '&#40007;',
        "\xF2\xE2" => '&#39986;',
        "\xF2\xE3" => '&#40006;',
        "\xF2\xE4" => '&#39998;',
        "\xF2\xE5" => '&#40026;',
        "\xF2\xE6" => '&#40032;',
        "\xF2\xE7" => '&#40039;',
        "\xF2\xE8" => '&#40054;',
        "\xF2\xE9" => '&#40056;',
        "\xF2\xEA" => '&#40167;',
        "\xF2\xEB" => '&#40172;',
        "\xF2\xEC" => '&#40176;',
        "\xF2\xED" => '&#40201;',
        "\xF2\xEE" => '&#40200;',
        "\xF2\xEF" => '&#40171;',
        "\xF2\xF0" => '&#40195;',
        "\xF2\xF1" => '&#40198;',
        "\xF2\xF2" => '&#40234;',
        "\xF2\xF3" => '&#40230;',
        "\xF2\xF4" => '&#40367;',
        "\xF2\xF5" => '&#40227;',
        "\xF2\xF6" => '&#40223;',
        "\xF2\xF7" => '&#40260;',
        "\xF2\xF8" => '&#40213;',
        "\xF2\xF9" => '&#40210;',
        "\xF2\xFA" => '&#40257;',
        "\xF2\xFB" => '&#40255;',
        "\xF2\xFC" => '&#40254;',
        "\xF2\xFD" => '&#40262;',
        "\xF2\xFE" => '&#40264;',
        "\xF3\xA1" => '&#40285;',
        "\xF3\xA2" => '&#40286;',
        "\xF3\xA3" => '&#40292;',
        "\xF3\xA4" => '&#40273;',
        "\xF3\xA5" => '&#40272;',
        "\xF3\xA6" => '&#40281;',
        "\xF3\xA7" => '&#40306;',
        "\xF3\xA8" => '&#40329;',
        "\xF3\xA9" => '&#40327;',
        "\xF3\xAA" => '&#40363;',
        "\xF3\xAB" => '&#40303;',
        "\xF3\xAC" => '&#40314;',
        "\xF3\xAD" => '&#40346;',
        "\xF3\xAE" => '&#40356;',
        "\xF3\xAF" => '&#40361;',
        "\xF3\xB0" => '&#40370;',
        "\xF3\xB1" => '&#40388;',
        "\xF3\xB2" => '&#40385;',
        "\xF3\xB3" => '&#40379;',
        "\xF3\xB4" => '&#40376;',
        "\xF3\xB5" => '&#40378;',
        "\xF3\xB6" => '&#40390;',
        "\xF3\xB7" => '&#40399;',
        "\xF3\xB8" => '&#40386;',
        "\xF3\xB9" => '&#40409;',
        "\xF3\xBA" => '&#40403;',
        "\xF3\xBB" => '&#40440;',
        "\xF3\xBC" => '&#40422;',
        "\xF3\xBD" => '&#40429;',
        "\xF3\xBE" => '&#40431;',
        "\xF3\xBF" => '&#40445;',
        "\xF3\xC0" => '&#40474;',
        "\xF3\xC1" => '&#40475;',
        "\xF3\xC2" => '&#40478;',
        "\xF3\xC3" => '&#40565;',
        "\xF3\xC4" => '&#40569;',
        "\xF3\xC5" => '&#40573;',
        "\xF3\xC6" => '&#40577;',
        "\xF3\xC7" => '&#40584;',
        "\xF3\xC8" => '&#40587;',
        "\xF3\xC9" => '&#40588;',
        "\xF3\xCA" => '&#40594;',
        "\xF3\xCB" => '&#40597;',
        "\xF3\xCC" => '&#40593;',
        "\xF3\xCD" => '&#40605;',
        "\xF3\xCE" => '&#40613;',
        "\xF3\xCF" => '&#40617;',
        "\xF3\xD0" => '&#40632;',
        "\xF3\xD1" => '&#40618;',
        "\xF3\xD2" => '&#40621;',
        "\xF3\xD3" => '&#38753;',
        "\xF3\xD4" => '&#40652;',
        "\xF3\xD5" => '&#40654;',
        "\xF3\xD6" => '&#40655;',
        "\xF3\xD7" => '&#40656;',
        "\xF3\xD8" => '&#40660;',
        "\xF3\xD9" => '&#40668;',
        "\xF3\xDA" => '&#40670;',
        "\xF3\xDB" => '&#40669;',
        "\xF3\xDC" => '&#40672;',
        "\xF3\xDD" => '&#40677;',
        "\xF3\xDE" => '&#40680;',
        "\xF3\xDF" => '&#40687;',
        "\xF3\xE0" => '&#40692;',
        "\xF3\xE1" => '&#40694;',
        "\xF3\xE2" => '&#40695;',
        "\xF3\xE3" => '&#40697;',
        "\xF3\xE4" => '&#40699;',
        "\xF3\xE5" => '&#40700;',
        "\xF3\xE6" => '&#40701;',
        "\xF3\xE7" => '&#40711;',
        "\xF3\xE8" => '&#40712;',
        "\xF3\xE9" => '&#30391;',
        "\xF3\xEA" => '&#40725;',
        "\xF3\xEB" => '&#40737;',
        "\xF3\xEC" => '&#40748;',
        "\xF3\xED" => '&#40766;',
        "\xF3\xEE" => '&#40778;',
        "\xF3\xEF" => '&#40786;',
        "\xF3\xF0" => '&#40788;',
        "\xF3\xF1" => '&#40803;',
        "\xF3\xF2" => '&#40799;',
        "\xF3\xF3" => '&#40800;',
        "\xF3\xF4" => '&#40801;',
        "\xF3\xF5" => '&#40806;',
        "\xF3\xF6" => '&#40807;',
        "\xF3\xF7" => '&#40812;',
        "\xF3\xF8" => '&#40810;',
        "\xF3\xF9" => '&#40823;',
        "\xF3\xFA" => '&#40818;',
        "\xF3\xFB" => '&#40822;',
        "\xF3\xFC" => '&#40853;',
        "\xF3\xFD" => '&#40860;',
        "\xF3\xFE" => '&#40864;',
        "\xF4\xA1" => '&#22575;',
        "\xF4\xA2" => '&#27079;',
        "\xF4\xA3" => '&#36953;',
        "\xF4\xA4" => '&#29796;',
        "\xF4\xA5" => '&#20956;',
        "\xF4\xA6" => '&#29081;');

  $index=0;
  $ret="";

  while ( $index < strlen($string)) {
    if ( ereg("[\241-\364]", $string[$index])) {
	$ret.= str_replace(array_keys($eucjp), array_values($eucjp), $string[$index] . $string[$index+1]);
	$index=$index+2;
    } else {
	$ret.= $string[$index];
        $index=$index+1;
    }
  }
  
  return $ret;
}
?>
