#ifdef __GNUC__
#  pragma implementation "sppc.H"
#endif

#include "sppc.H"
#include "postscript.H"
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>

const int CMD_MAX = 80;
const int STR_MAX = 1024;
const int CURVES_MAX = 100;
static const char *ErrorProlog = "*** spcc: ";

inline greal min(greal x, greal y) {return (x < y) ? x : y;}
inline greal max(greal x, greal y) {return (x > y) ? x : y;}

static real min(const Tentry* p)
{
	if (!p->data) return 0;
	int i;
	real result = 0;
	const int n = p->ncols * p->nrows;
	for (i=0; i<n; i++) if (finite(p->data[i])) {result = p->data[i]; break;}
	if (i >= n) return 0;
	for (; i<n; i++) if (finite(p->data[i]) && p->data[i] < result) result = p->data[i];
	return result;
}

static real max(const Tentry* p)
{
	if (!p->data) return 0;
	int i;
	real result = 0;
	const int n = p->ncols * p->nrows;
	for (i=0; i<n; i++) if (finite(p->data[i])) {result = p->data[i]; break;}
	if (i >= n) return 0;
	for (; i<n; i++) if (finite(p->data[i]) && p->data[i] > result) result = p->data[i];
	return result;
#if 0
	if (!p->data) return 0;
	int i;
	real result = p->data[0];
	const int n = p->ncols * p->nrows;
	for (i=1; i<n; i++) if (p->data[i] > result) result = p->data[i];
	return result;
#endif
}

void Tentry::print_string(ostream& o) const
{
	int i,j;
	if (nrows > 1) o << "[";
	for (i=0; i<nrows; i++) {
		o << '"';
		for (j=0; j<ncols; j++) {
			o.put(char(int(data[i*ncols+j]+0.5)));
		}
		o << '"';
		if (i<nrows-1) o << ',';
	}
	if (nrows > 1) o << "]";
}

void Tentry::print_numeric(ostream& o) const
{
	int i,j;
	o << "[" << nrows << ' ' << ncols;
	for (i=0; i<nrows; i++) for (j=0; j<ncols; j++) o << ' ' << data[i*ncols+j];
	o << "]";
}

class PrintString {
private:
	const Tentry *ptr;
public:
	PrintString(const Tentry& e) :ptr(&e) {}
	friend ostream& operator<<(ostream& o, const PrintString& S) {S.ptr->print_string(o); return o;}
};

class PrintNumeric {
private:
	const Tentry *ptr;
public:
	PrintNumeric(const Tentry& e) :ptr(&e) {}
	friend ostream& operator<<(ostream& o, const PrintNumeric& S) {S.ptr->print_numeric(o); return o;}
};

void Tentry::print(ostream& o) const
{
	switch (type) {
	case CURVE: o << "curve\n"; break;
	case CURVE_RIGHT: o << "curve_right\n"; break;
	case PCOLOR: o << "pcolor\n"; break;
	case PANEL: o << "panel\n"; break;
	case TEXTS: o << "texts\n"; break;
	case TEXTS_UT: o << "texts_UT\n"; break;
	case MARK: o << "mark\n"; break;
	case PLOT: o << "plot\n"; break;
	case TITLE: o << "title=" << PrintString(*this) << "\n"; break;
	case X: o << "x=" << PrintNumeric(*this) << "\n"; break;
	case Y: o << "y=" << PrintNumeric(*this) << "\n"; break;
	case Y_RIGHT: o << "y_right=" << PrintNumeric(*this) << "\n"; break;
	case Z: o << "z=" << PrintNumeric(*this) << "\n"; break;
	case TEXTVEC: o << "textvec=" << PrintNumeric(*this) << "\n"; break;
	case LEFTLABEL: o << "leftlabel=" << PrintString(*this) << "\n"; break;
	case RIGHTLABEL: o << "rightlabel=" << PrintString(*this) << "\n"; break;
	case XLEFTLABEL: o << "xleftlabel=" << PrintString(*this) << "\n"; break;
	case XRIGHTLABEL: o << "xrightlabel=" << PrintString(*this) << "\n"; break;
	case LABEL: o << "label=" << PrintString(*this) << "\n"; break;
	case LINECOLOR: o << "linecolor=" << PrintNumeric(*this) << "\n"; break;
	case MARKERCOLOR: o << "markercolor=" << PrintNumeric(*this) << "\n"; break;
	case LINEWIDTH: o << "linewidth=" << PrintNumeric(*this) << "\n"; break;
	case LINETYPE: o << "linetype=" << PrintNumeric(*this) << "\n"; break;
	case MARKERTYPE: o << "markertype=" << PrintNumeric(*this) << "\n"; break;
	case MARKERSIZE: o << "markersize=" << PrintNumeric(*this) << "\n"; break;
	case ENDPOINT_X: o << "endpoint_x=" << PrintNumeric(*this) << "\n"; break;
	case ENDPOINT_Y: o << "endpoint_y=" << PrintNumeric(*this) << "\n"; break;
	case FILLCOLOR: o << "fillcolor=" << PrintNumeric(*this) << "\n"; break;
	case FOREGROUND: o << "foreground=" << PrintNumeric(*this) << "\n"; break;
	case BACKGROUND: o << "background=" << PrintNumeric(*this) << "\n"; break;
	case FRAMEWIDTH: o << "framewidth=" << PrintNumeric(*this) << "\n"; break;
	case XMIN: o << "xmin=" << PrintNumeric(*this) << "\n"; break;
	case XMAX: o << "xmax=" << PrintNumeric(*this) << "\n"; break;
	case YMIN: o << "ymin=" << PrintNumeric(*this) << "\n"; break;
	case YMAX: o << "ymax=" << PrintNumeric(*this) << "\n"; break;
	case YMIN_RIGHT: o << "ymin_right=" << PrintNumeric(*this) << "\n"; break;
	case YMAX_RIGHT: o << "ymax_right=" << PrintNumeric(*this) << "\n"; break;
	case ZMIN: o << "zmin=" << PrintNumeric(*this) << "\n"; break;
	case ZMAX: o << "zmax=" << PrintNumeric(*this) << "\n"; break;
	case GRID: o << "grid=" << PrintNumeric(*this) << "\n"; break;
	case ARROW: o << "arrow=" << PrintNumeric(*this) << "\n"; break;
	case INTERP: o << "interp=" << PrintNumeric(*this) << "\n"; break;
	case SHOWNEG: o << "showneg=" << PrintNumeric(*this) << "\n"; break;
	case HOURAXIS: o << "houraxis=" << PrintNumeric(*this) << "\n"; break;
	case HOURAXIS_X: o << "houraxis_x=" << PrintNumeric(*this) << "\n"; break;
	case TITLEFONTSIZE: o << "titlefontsize=" << PrintNumeric(*this) << "\n"; break;
	case LABELFONTSIZE: o << "labelfontsize=" << PrintNumeric(*this) << "\n"; break;
	case AXISFONTSIZE: o << "axisfontsize=" << PrintNumeric(*this) << "\n"; break;
	case ANNOTFONTSIZE: o << "annotfontsize=" << PrintNumeric(*this) << "\n"; break;
	case XLOG: o << "xlog=" << PrintNumeric(*this) << "\n"; break;
	case YLOG: o << "ylog=" << PrintNumeric(*this) << "\n"; break;
	case YLOG_RIGHT: o << "ylog_right=" << PrintNumeric(*this) << "\n"; break;
	case ZLOG: o << "zlog=" << PrintNumeric(*this) << "\n"; break;
	case PALETTE: o << "palette=" << PrintNumeric(*this) << "\n"; break;
	default: o << "<***unknown type***>\n";
	}
}

Tsppc::Tsppc()
	: lst(0), last(0)
{
	
}

void Tsppc::add(Tentry* p)
{
	if (last)
		last->next = p;
	else
		lst = p;
	p->next = 0;
	last = p;
}

static void Expected(char ch)
{
	cerr << ErrorProlog << "Expected '" << ch << "'\n";
}

static bool ReadValue(char ch, istream& in, Tentry *p)
{
	if (ch != '=') {Expected('='); return false;}
	in.get(ch);
	if (ch == '"') {
		// Read string value
		char s[STR_MAX+1];
		int i = 0;
		while (!in.eof() && i<STR_MAX) {
			in.get(ch);
			if (ch == '"') break;
			if (ch == '\\') in.get(ch);		// backslash preceding a doublequote does not end string but is the doublequote
			if (ch == '\n') {
				cerr << ErrorProlog << "String ended with a newline\n";
				return false;
			}
			s[i++] = ch;
		}
		if (in.eof()) {
			cerr << ErrorProlog << "EOF encountered while reading string\n";
			return false;
		}
		if (i >= STR_MAX) {
			cerr << ErrorProlog << "String too long\n";
			return false;
		}
		s[i] = '\0';
		p->nrows = 1;
		p->ncols = strlen(s);
		p->data = new real [p->ncols];
		for (i=0; i<p->ncols; i++) p->data[i] = int(s[i]);
	} else if (ch == '[') {
		// Read ASCII numeric value
		in >> p->nrows >> p->ncols;
		if (p->nrows < 1 || p->ncols < 1) {
			cerr << ErrorProlog << "Bad value(s) for nrows,ncols (" << p->nrows << "," << p->ncols << ")\n";
			return false;
		}
		p->data = new real [p->nrows*p->ncols];
		int i,j;
		for (i=0; i<p->nrows; i++) for (j=0; j<p->ncols; j++) {
			in >> p->data[i*p->ncols+j];
		}
		in >> ch;
		if (ch != ']') {Expected(']'); return false;}
	} else if (ch == '(') {
		// Read binary (double precision) numeric value
		in >> p->nrows >> p->ncols;
		if (p->nrows < 1 || p->ncols < 1) {
			cerr << ErrorProlog << "Bad value(s) for nrows,ncols (" << p->nrows << "," << p->ncols << ")\n";
			return false;
		}
		in.get(ch);
		if (ch != '\n') {cerr << ErrorProlog << "Expected newline before binary but got '" << ch << "'\n"; return false;}
		const int N = p->nrows*p->ncols;
		p->data = new real [N];
		if (sizeof(double) != sizeof(real)) {
			double *const buff = new double [N];
			in.read((char*)buff,sizeof(double)*N);
			int i;
			for (i=0; i<N; i++) p->data[i] = buff[i];
			delete [] buff;
		} else {
			in.read((char*)p->data,sizeof(double)*N);
		}
		in.get(ch);
		if (ch != ')') {Expected(')'); return false;}
	}
	in.get(ch);
	if (ch != '\n') {cerr << ErrorProlog << "Expected newline but got '" << ch << "'\n"; return false;}
	return true;
}

bool Tsppc::read(istream& in)
{
	char command[CMD_MAX+1];
	int i;
	char ch;
	while (!in.eof()) {
		i = 0;
		do {
			in.get(ch);
			command[i++] = ch;
		} while (i < CMD_MAX && (isalpha(ch) || ch == '_') && !in.eof());
		if (in.eof()) break;
		command[--i] = '\0';
		Tentry *p = new Tentry;
		bool iscommand = false;
		if (!strcmp(command,"title")) {
			p->type = TITLE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"x")) {
			p->type = X;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"y")) {
			p->type = Y;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"y_right")) {
			p->type = Y_RIGHT;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"z")) {
			p->type = Z;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"textvec")) {
			p->type = TEXTVEC;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"leftlabel")) {
			p->type = LEFTLABEL;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"rightlabel")) {
			p->type = RIGHTLABEL;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"xleftlabel")) {
			p->type = XLEFTLABEL;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"xrightlabel")) {
			p->type = XRIGHTLABEL;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"label")) {
			p->type = LABEL;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"linecolor")) {
			p->type = LINECOLOR;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"markercolor")) {
			p->type = MARKERCOLOR;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"linewidth")) {
			p->type = LINEWIDTH;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"linetype")) {
			p->type = LINETYPE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"markertype")) {
			p->type = MARKERTYPE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"markersize")) {
			p->type = MARKERSIZE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"endpoint_x")) {
			p->type = ENDPOINT_X;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"endpoint_y")) {
			p->type = ENDPOINT_Y;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"fillcolor")) {
			p->type = FILLCOLOR;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"foreground")) {
			p->type = FOREGROUND;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"framewidth")) {
			p->type = FRAMEWIDTH;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"background")) {
			p->type = BACKGROUND;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"xmin")) {
			p->type = XMIN;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"xmax")) {
			p->type = XMAX;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ymin")) {
			p->type = YMIN;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ymax")) {
			p->type = YMAX;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ymin_right")) {
			p->type = YMIN_RIGHT;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ymax_right")) {
			p->type = YMAX_RIGHT;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"zmin")) {
			p->type = ZMIN;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"zmax")) {
			p->type = ZMAX;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"grid")) {
			p->type = GRID;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"arrow")) {
			p->type = ARROW;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"interp")) {
			p->type = INTERP;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"showneg")) {
			p->type = SHOWNEG;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"houraxis")) {
			p->type = HOURAXIS;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"houraxis_x")) {
			p->type = HOURAXIS_X;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"titlefontsize")) {
			p->type = TITLEFONTSIZE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"labelfontsize")) {
			p->type = LABELFONTSIZE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"axisfontsize")) {
			p->type = AXISFONTSIZE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"annotfontsize")) {
			p->type = ANNOTFONTSIZE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"xlog")) {
			p->type = XLOG;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ylog")) {
			p->type = YLOG;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"ylog_right")) {
			p->type = YLOG_RIGHT;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"palette")) {
			p->type = PALETTE;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"zlog")) {
			p->type = ZLOG;
			if (!ReadValue(ch,in,p)) return false;
		} else if (!strcmp(command,"curve")) {
			p->type = CURVE;
			iscommand = true;
		} else if (!strcmp(command,"curve_right")) {
			p->type = CURVE_RIGHT;
			iscommand = true;
		} else if (!strcmp(command,"pcolor")) {
			p->type = PCOLOR;
			iscommand = true;
		} else if (!strcmp(command,"panel")) {
			p->type = PANEL;
			iscommand = true;
		} else if (!strcmp(command,"texts")) {
			p->type = TEXTS;
			iscommand = true;
		} else if (!strcmp(command,"texts_UT")) {
			p->type = TEXTS_UT;
			iscommand = true;
		} else if (!strcmp(command,"mark")) {
			p->type = MARK;
			iscommand = true;
		} else if (!strcmp(command,"plot")) {
			p->type = PLOT;
			iscommand = true;
		} else {
			cerr << ErrorProlog << "Unrecognized command \"" << command << "\"\n";
			return false;
		}
		add(p);
//		cout << "Added: "; p->print(cout);
		if (iscommand) {
			if (ch != '\n') {
				cerr << ErrorProlog << "Expected newline but got '" << ch << "'\n";
				return false;
			}
		}
	}
	LogProcessing();
	return true;
}

void Tsppc::print(ostream& o) const
{
	Tentry *p;
	for (p=lst; p; p=p->next) p->print(o);
}

static void GetString(const Tentry* p, char*& s)
{
	char *result = new char [p->ncols + 1];
	int i;
	for (i=0; i<p->ncols; i++) result[i] = char(int(p->data[i]+0.5));
	result[p->ncols] = '\0';
	if (s) delete [] s;
	s = result;
}

static void GetScalar(const Tentry *p, real& x)
{
	if (p->nrows != 1 || p->ncols != 1) {cerr << ErrorProlog << "Scalar expected\n"; x = 0; return;}
	x = p->data[0];
}

static void GetColor(const Tentry *p, real c[3])
{
	if (p->nrows != 1 || p->ncols != 3) {
		cerr << ErrorProlog << "3-vector (RGB color) expected\n";
		c[0] = c[1] = c[2] = 0;
		return;
	}
	c[0] = p->data[0];
	c[1] = p->data[1];
	c[2] = p->data[2];
}

static void DeleteString(char*& s) {if (s) delete [] s; s = 0;}

static void FindYRange(const Tentry *start, greal& ymin, greal& ymax, greal& ymin_right, greal& ymax_right)
{
	const Tentry *p;
	ymin = 1e30; ymax = -1e30;
	ymin_right = ymin; ymax_right = ymax;
	greal set_ymin, set_ymax, set_ymin_right, set_ymax_right;
	bool YminSet=false, YmaxSet=false, YminRightSet=false, YmaxRightSet=false;
	for (p=start; p && p->type != PANEL && p->type != PLOT; p=p->next) {
		switch (p->type) {
		case Y:
			ymin = min(ymin,min(p));
			ymax = max(ymax,max(p));
			break;
		case Y_RIGHT:
			ymin_right = min(ymin_right,min(p));
			ymax_right = max(ymax_right,max(p));
			break;
		case YMIN:
			GetScalar(p,set_ymin);
			YminSet = true;
			break;
		case YMAX:
			GetScalar(p,set_ymax);
			YmaxSet = true;
			break;
		case YMIN_RIGHT:
			GetScalar(p,set_ymin_right);
			YminRightSet = true;
			break;
		case YMAX_RIGHT:
			GetScalar(p,set_ymax_right);
			YmaxRightSet = true;
			break;
		default:;
		}
	}
	greal tinyy = 0.02*(ymax-ymin);
	if (YminSet)
		ymin  = set_ymin;
	else
		ymin-= tinyy;
	if (YmaxSet)
		ymax = set_ymax;
	else
		ymax+= tinyy;
	greal tinyy_right = 0.02*(ymax_right - ymin_right);
	if (YminRightSet)
		ymin_right  = set_ymin_right;
	else
		ymin_right-= tinyy_right;
	if (YmaxRightSet)
		ymax_right = set_ymax_right;
	else
		ymax_right+= tinyy_right;
}

static void FindZRange(const Tentry *start, greal& zmin, greal& zmax)
{
	const Tentry *p;
	zmin = 1e30; zmax = -1e30;
	greal set_zmin, set_zmax;
	bool ZminSet=false, ZmaxSet=false;
	for (p=start; p && p->type != PANEL && p->type != PLOT; p=p->next) {
		if (p->type == Z) {
			zmin = min(zmin,min(p));
			zmax = max(zmax,max(p));
		} else if (p->type == ZMIN) {
			GetScalar(p,set_zmin);
			ZminSet = true;
		} else if (p->type == ZMAX) {
			GetScalar(p,set_zmax);
			ZmaxSet = true;
		}
	}
	greal tinyz = 0.01*(zmax-zmin);
	if (ZminSet)
		zmin = set_zmin;
	else
		zmin-= tinyz;
	if (ZmaxSet)
		zmax = set_zmax;
	else
		zmax+= tinyz;
}

static void ReplaceByLog(Tentry *p, bool showneg)
{
	int i;
	const int n = p->ncols*p->nrows;
	const real invlg10 = 1.0/log(10.0);
	const real nan = log(-1);
//	cerr << "ReplaceByLog: showneg=" << showneg << "\n";
	for (i=0; i<n; i++) {
		if (!showneg && p->data[i] <= 0) {
			p->data[i] = nan;
			continue;
		}
		if (p->data[i] < 1e-15)
			p->data[i] = -15;
		else
			p->data[i] = log(p->data[i])*invlg10;
	}
}

static real FindClosestMatch(const real x[], const real y[], int n, real xval)
{
	int i;
	real absdiff;
	real min_absdiff = fabs(x[0]-xval);
	int min_ind = 0;
	for (i=0; i<n; i++) {
		absdiff = fabs(x[i] - xval);
		if (absdiff < min_absdiff) {min_absdiff = absdiff; min_ind = i;}
	}
	if (min_ind > 0
		&& (x[min_ind-1] <= xval && xval <= x[min_ind] && x[min_ind-1] < x[min_ind]
			|| x[min_ind-1] >= xval && xval >= x[min_ind] && x[min_ind-1] > x[min_ind])) {
		const real t = (xval - x[min_ind-1])/(x[min_ind] - x[min_ind-1]);
		return (1-t)*y[min_ind-1] + t*y[min_ind];
	} else if (min_ind < n-1
			   && (x[min_ind] <= xval && xval <= x[min_ind+1] && x[min_ind] < x[min_ind+1]
				   || x[min_ind] >= xval && xval >= x[min_ind+1] && x[min_ind] > x[min_ind+1])) {
		const real t = (xval - x[min_ind])/(x[min_ind+1] - x[min_ind]);
		return (1-t)*y[min_ind] + t*y[min_ind+1];
	}
	return y[min_ind];
}

void Tsppc::LogProcessing()
{
	Tentry *p;
	// find out xlog setting
	bool xlog = false, showneg = false;
	for (p=lst; p && p->type != PLOT; p=p->next)
		if (p->type == XLOG) {
			greal xlog1; GetScalar(p,xlog1); xlog = (xlog1!=0.0);
		}
	// ylog processing: replace Y objects by their logarithms where ylog=true
	// zlog processing: replace PCOLOR objects by their logarithms where zlog=true
	bool ylog = false, ylog_right = false, zlog = false;
	for (p=lst; p && p->type != PLOT; p=p->next) {
		switch (p->type) {
		case SHOWNEG:
		{
			greal showneg1;
			GetScalar(p,showneg1);
			showneg = (showneg1 != 0.0);
		}
		break;
		case YLOG:
		{
			greal ylog1;
			GetScalar(p,ylog1);
			ylog = (ylog1 != 0.0);
		}
		break;
		case YLOG_RIGHT:
		{
			greal ylog_right_1;
			GetScalar(p,ylog_right_1);
			ylog_right = (ylog_right_1 != 0.0);
		}
		break;
		case ZLOG:
		{
			greal zlog1;
			GetScalar(p,zlog1);
			zlog = (zlog1 != 0.0);
		}
		break;
		case Y:
			if (ylog) ReplaceByLog(p,showneg);
			break;
		case Y_RIGHT:
			if (ylog_right) ReplaceByLog(p,showneg);
			break;
		case Z:
			if (zlog) ReplaceByLog(p,showneg);
			break;
		case PANEL:
			ylog = false;		// a PANEL will reset ylog and zlog
			zlog = false;
			break;
		case X:
			if (xlog) ReplaceByLog(p,showneg);
			break;
		default:;
		}
	}

}

static void setlinetype(Tgraph2D& g, greal x)
{
	Tgraph2D::TLineType lt = Tgraph2D::LINE_SOLID;
	const int linetype = int(x + 0.5);
	if (linetype == 0)
		lt = Tgraph2D::LINE_NONE;
	else if (linetype == 1)
		lt = Tgraph2D::LINE_SOLID;
	else if (linetype == 2)
		lt = Tgraph2D::LINE_DASHED;
	else if (linetype == 3)
		lt = Tgraph2D::LINE_DOTTED;
	else if (linetype == 4)
		lt = Tgraph2D::LINE_DOTDASHED;
	g.setlinetype(lt);
}

void Tsppc::plot(Tgraph2D& g, greal Xmax, greal Ymax,
				 greal titlefontsize, greal labelfontsize, greal axisfontsize, greal annotfontsize,
				 bool DrawCopyrightNotice) const
{
	const greal minXmaxYmax = min(Xmax,Ymax);
	const greal tmargin = 0.05*minXmaxYmax;
	const greal bmargin = 0.1*minXmaxYmax;
	const greal lmargin = 0.15*minXmaxYmax;
	const greal leftlabelpos = 0.35*lmargin;
	const greal textheight = 1.3*axisfontsize;
	greal rmargin = 0.13*minXmaxYmax;
	const Tentry *p;
	int npanels,ntexts;
	bool has_pcolors = false, has_rightcurves = false, has_title = false, xlog = false;
	for (p=lst,npanels=0,ntexts=0; p && p->type != PLOT; p=p->next)
		switch (p->type) {
		case PANEL: npanels++; break;
		case PCOLOR: has_pcolors = true; break;
		case CURVE_RIGHT: has_rightcurves = true; break;
		case TITLE: if (p->ncols > 0) has_title = true; break;
		case XLOG: {greal xlog1; GetScalar(p,xlog1); xlog = (xlog1!=0.0);} break;
		case TEXTS: ntexts++; break;
		default:;
		}
	if (npanels == 0) return;
	const greal rightlabelpos = Xmax-(has_pcolors||has_rightcurves ? 0.35 : 0.5)*rmargin;
	const greal xticklen = 0.02*minXmaxYmax;
	const greal yticklen = xticklen*(0.5 + 0.5/npanels);
	const greal barticklen = 0.01*minXmaxYmax;
	const greal titleheight = has_title ? yticklen+(1.3+1.3/npanels)*titlefontsize /*0.1*minXmaxYmax*/ : 0;
	if (has_pcolors) rmargin+= 0.06*minXmaxYmax;
	const greal lefttextlabelpos = 0.75*lmargin;
	const greal righttextlabelpos = Xmax - 0.75*rmargin;
//	cerr << npanels << " panels\n";
	const greal intramargin = (0.25+0.25/npanels)*0.1*minXmaxYmax;
	const greal panelheight = (Ymax-tmargin-bmargin-(ntexts+0.5)*textheight-titleheight-(npanels-1)*intramargin)/greal(npanels);
	if (panelheight < 0.01*Ymax) return;
//	cerr << "panelheight=" << panelheight << "\n";
	// Autodetect x range, find framewidth, foreground and background color
	greal xmin = 1e30, xmax = -1e30;
	greal foreground[3] = {0,0,0}, background[3] = {1,1,1};
	greal framewidth = 1;
	for (p=lst; p && p->type != PLOT; p=p->next) {
		if (p->type == X) {
			xmin = min(xmin,min(p));
			xmax = max(xmax,max(p));
		} else if (p->type == FOREGROUND)
			GetColor(p,foreground);
		else if (p->type == BACKGROUND)
			GetColor(p,background);
		else if (p->type == FRAMEWIDTH)
			GetScalar(p,framewidth);
	}
	greal linecolor[3] = {foreground[0],foreground[1],foreground[2]};
	greal fillcolor[3] = {background[0],background[1],background[2]};
	greal markercolor[3] = {foreground[0],foreground[1],foreground[2]};
	const greal default_markersize = 6;
	greal endpoint_x=0, endpoint_y=0, arrow=0.0;
	bool endpoint_x_given=false, endpoint_y_given=false;
	greal linewidth=framewidth, linetype=1, markertype=1, draw_grid_lines=0, interp=0, houraxis=0, houraxis_x=0;
	greal markersize=default_markersize;
	const greal tinyx = 0.013*(xmax-xmin);
	xmin-= tinyx; xmax+= tinyx;
//	cerr << "autodetect xrange = " << xmin << " .. " << xmax << "\n";
	// Draw panels
	int npanels_plotted = 0, ntexts_plotted = 0;
	char *title = 0, *leftlabel = 0, *rightlabel = 0, *xleftlabel = 0, *xrightlabel = 0, *label = 0;
	const Tentry *xs[CURVES_MAX], *ys[CURVES_MAX], *ys_right[CURVES_MAX], *zs[CURVES_MAX], *textvecs[CURVES_MAX];
	int i;
	for (i=0; i<CURVES_MAX; i++) xs[i] = ys[i] = zs[i] = textvecs[i] = 0;
	int curves_done = 0;
	greal ymin,ymax,ymin_right,ymax_right,zmin,zmax;
	FindYRange(lst,ymin,ymax,ymin_right,ymax_right);
	FindZRange(lst,zmin,zmax);
	bool has_drawn_any_pcolor = false;
//	bool include_minor_ticks = panelheight/(2*framewidth) > 50; /*npanels < 5);*/
	TMinorTicks include_minor_ticks = MINORTICKS_AUTO;
	if (background[0]!=1 || background[1]!=1 || background[2]!=1) {
		g.comment("Fill the page with background color");
		g.setcolor(background);
		g.fillrrect(0,0,Xmax,Ymax);
	}
	g.setcolor(foreground);
	g.setlinewidth(framewidth);
	bool ylog = false, ylog_right = false, zlog = false;
	bool panel_had_any_rightcurve = false;
	for (p=lst; p && p->type != PLOT; p=p->next) {
		switch (p->type) {
		case TITLE:
			GetString(p,title);
			break;
		case XLEFTLABEL:
			GetString(p,xleftlabel);
			break;
		case XRIGHTLABEL:
			GetString(p,xrightlabel);
			break;
		case LEFTLABEL:
			GetString(p,leftlabel);
			break;
		case RIGHTLABEL:
			GetString(p,rightlabel);
			break;
		case LABEL:
			GetString(p,label);
			break;
		case X:
			xs[curves_done] = p;
			break;
		case Y:
			ys[curves_done] = p;
			break;
		case Y_RIGHT:
			ys_right[curves_done] = p;
			break;
		case Z:
			zs[curves_done] = p;
			break;
		case TEXTVEC:
			textvecs[curves_done] = p;
			break;
		case GRID:
			GetScalar(p,draw_grid_lines);
			break;
		case ARROW:
			GetScalar(p,arrow);
			break;
		case INTERP:
			GetScalar(p,interp);
			break;
		case HOURAXIS:
			GetScalar(p,houraxis);
			break;
		case HOURAXIS_X:
			GetScalar(p,houraxis_x);
			break;
		case LINECOLOR:
			GetColor(p,linecolor);
			break;
		case MARKERCOLOR:
			GetColor(p,markercolor);
			break;
		case LINEWIDTH:
			GetScalar(p,linewidth);
			break;
		case LINETYPE:
			GetScalar(p,linetype);
			linetype = floor(linetype+0.5);
			break;
		case MARKERTYPE:
			GetScalar(p,markertype);
			markertype = floor(markertype+0.5);
			break;
		case MARKERSIZE:
			GetScalar(p,markersize);
			break;
		case ENDPOINT_X:
			GetScalar(p,endpoint_x);
			endpoint_x_given = true;
			break;
		case ENDPOINT_Y:
			GetScalar(p,endpoint_y);
			endpoint_y_given = true;
			break;
		case FILLCOLOR:
			GetColor(p,fillcolor);
			break;
		case XMIN:
			GetScalar(p,xmin);
			break;
		case XMAX:
			GetScalar(p,xmax);
			break;
		case YLOG:
		{
			greal ylog1;
			GetScalar(p,ylog1);
			ylog = (ylog1 != 0.0);
		}
		break;
		case ZLOG:
		{
			greal zlog1;
			GetScalar(p,zlog1);
			zlog = (zlog1 != 0.0);
		}
		break;
		case PALETTE:
		{
			if (p->nrows == 1) {
				g.setpalette(Tgraph2D::PALETTE_RAINBOW);
			} else {
				if (p->ncols != Ncolors || p->nrows != 3) {
					cerr << ErrorProlog << "palette: value is no 3 by " << Ncolors << " matrix (it is "
						 << p->nrows << "x" << p->ncols << ")\n";
					continue;
				}
				int pal[3][Ncolors];
				int i,c;
				for (c=0; c<3; c++) for (i=0; i<Ncolors; i++) {
					pal[c][i] = int(p->data[c*Ncolors+i] + 0.5);
					if (pal[c][i] < 0) pal[c][i] = 0;
					if (pal[c][i] > Ncolors-1) pal[c][i] = Ncolors-1;
				}
				g.setpalette(pal);
			}
		}
		break;
		case FRAMEWIDTH:
		case FOREGROUND:
		case BACKGROUND:
		case XLOG:
		case YMIN:
		case YMAX:
		case YMIN_RIGHT:
		case YMAX_RIGHT:
		case ZMIN:
		case ZMAX:
		case TITLEFONTSIZE:
		case LABELFONTSIZE:
		case SHOWNEG:
		case AXISFONTSIZE:
		case ANNOTFONTSIZE:
			break;
		case CURVE:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin);
			// yy == top boundary of panel
			int xcurv_ind;
			for (xcurv_ind=curves_done; xcurv_ind>=0; xcurv_ind--) if (xs[xcurv_ind]) break;
			if (xcurv_ind<0 || !xs[xcurv_ind] || !ys[curves_done]) {
				cerr << ErrorProlog << "curve: x or y undefined\n";
				break;
			}
			if (xs[xcurv_ind]->ncols != ys[curves_done]->ncols) {
				cerr << ErrorProlog << "curve: x and y vectors of unequal lengths ("
					 << xs[xcurv_ind]->ncols << "," << ys[curves_done]->ncols << ")\n";
			}
			g.comment("Draw curve");
			g.begin_clipped(lmargin,yy-panelheight,Xmax-rmargin,yy);
			if (fillcolor[0]!=background[0] || fillcolor[1]!=background[1] || fillcolor[2]!=background[2]) {
				g.comment("Draw filled part of the curve");
				g.setcolor(fillcolor);
				g.filledcurve(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,ys[curves_done]->data,
							  lmargin,yy-panelheight,Xmax-rmargin,yy,
							  xmin,xmax,ymin,ymax);
				g.comment("Draw the curve itself");
			}
			setlinetype(g,linetype);
			g.setlinewidth(linewidth);
			g.setcolor(linecolor);
			g.curve(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,ys[curves_done]->data,
					lmargin,yy-panelheight,Xmax-rmargin,yy,
					xmin,xmax,ymin,ymax, (int(markertype+0.5)-1) % 13 + 1,markersize);
			g.setlinetype(Tgraph2D::LINE_SOLID);
			g.setlinewidth(framewidth);
			g.setcolor(foreground);
			g.end_clipped();
			ys[curves_done] = 0;
			curves_done++;
			if (curves_done >= CURVES_MAX) {
				cerr << ErrorProlog << "curve: Too many curves or pcolors in single panel\n";
				return;
			}
		}
		break;
		case MARK:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin);
			if (xs[curves_done]->nrows != 1 || xs[curves_done]->ncols != 1 ||
				ys[curves_done]->nrows != 1 || ys[curves_done]->ncols != 1) {
				cerr << ErrorProlog << "mark: x or y not scalars\n";
				break;
			}
			g.comment("Mark");
			const greal xmapped = (Xmax-rmargin-lmargin)*(xs[curves_done]->data[0]-xmin)/(xmax-xmin) + lmargin;
			const greal ymapped = panelheight*(ys[curves_done]->data[0]-ymin)/(ymax-ymin) + yy-panelheight;
			const greal xmapped_endpoint = (Xmax-rmargin-lmargin)*(endpoint_x-xmin)/(xmax-xmin) + lmargin;
			const greal ymapped_endpoint = panelheight*(endpoint_y-ymin)/(ymax-ymin) + yy-panelheight;
			g.setcolor(markercolor);
			if (markertype != 0.0) {
				g.mark(xmapped,ymapped,(int(markertype+0.5)-1) % 13 + 1,markersize);
			}
			if (label && *label) {
				g.setfont(Tgraph2D::FONT_ANNOTATION);
				if (endpoint_x_given && endpoint_y_given) {
					greal vx = xmapped - xmapped_endpoint;
					greal vy = ymapped - ymapped_endpoint;
					greal norm;
					if (vx == 0 && vy == 0) norm = 1; else norm = 1.0/sqrt(vx*vx + vy*vy);
					vx*= norm; vy*= norm;
					int Xoffset=0,Yoffset=0;
					if (vx < -0.5) Xoffset = -1; else if (vx > 0.5) Xoffset = 1;
					if (vy < -0.5) Yoffset = -1; else if (vy > 0.5) Yoffset = 1;
					if (markertype == 0.0)
						g.text(xmapped,ymapped,label,Xoffset,Yoffset,false);
					else
						g.text(xmapped+0.5*markersize*vx,ymapped+0.5*markersize*vy,label,Xoffset,Yoffset,false);
				} else {
					if (markertype == 0.0)
						g.text(xmapped,ymapped,label,0,0,false);
					else
						g.text(xmapped+0.6*markersize,ymapped,label ? label : "",1,0,false);
				}
			}
			if (endpoint_x_given && endpoint_y_given) {
				g.setlinewidth(linewidth);
				setlinetype(g,linetype);
				g.setcolor(linecolor);
				g.line(xmapped,ymapped,xmapped_endpoint,ymapped_endpoint);
				if (arrow != 0.0) {
					greal headsize = 6*linewidth;
					if (headsize < 13) headsize = 13;
					const greal dullness = 0.3;
					greal ux,uy,tx,ty,norm;
					ux = xmapped_endpoint - xmapped;
					uy = ymapped_endpoint - ymapped;
					if (ux==0 && uy==0)
						norm = 1;
					else {
						norm = sqrt(ux*ux + uy*uy);
						if (headsize > 0.6*norm) headsize = 0.6*norm;
						norm = 1.0/norm;
					}
					ux*= norm; uy*= norm;
					tx = uy;
					ty = -ux;
					g.line(xmapped_endpoint,ymapped_endpoint,
						   xmapped_endpoint-ux*headsize+dullness*tx*headsize,
						   ymapped_endpoint-uy*headsize+dullness*ty*headsize);
					g.line(xmapped_endpoint,ymapped_endpoint,
						   xmapped_endpoint-ux*headsize-dullness*tx*headsize,
						   ymapped_endpoint-uy*headsize-dullness*ty*headsize);
				}
				g.setlinewidth(framewidth);
			}
			g.setcolor(foreground);
			DeleteString(label);
			endpoint_x_given = endpoint_y_given = false;
		}
		break;
		case CURVE_RIGHT:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin);
			// yy == top boundary of panel
			int xcurv_ind;
			for (xcurv_ind=curves_done; xcurv_ind>=0; xcurv_ind--) if (xs[xcurv_ind]) break;
			if (xcurv_ind<0 || !xs[xcurv_ind] || !ys_right[curves_done]) {
				cerr << ErrorProlog << "curve_right: x or y_right undefined\n";
				break;
			}
			if (xs[xcurv_ind]->ncols != ys_right[curves_done]->ncols) {
				cerr << ErrorProlog << "curve_right: x and y_right vectors of unequal lengths ("
					 << xs[xcurv_ind]->ncols << "," << ys_right[curves_done]->ncols << ")\n";
			}
			g.comment("Draw curve_right");
			g.begin_clipped(lmargin,yy-panelheight,Xmax-rmargin,yy);
			if (fillcolor[0]!=background[0] || fillcolor[1]!=background[1] || fillcolor[2]!=background[2]) {
				g.comment("Draw filled part of the curve_right");
				g.setcolor(fillcolor);
				g.filledcurve(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,ys_right[curves_done]->data,
							  lmargin,yy-panelheight,Xmax-rmargin,yy,
							  xmin,xmax,ymin_right,ymax_right);
				g.comment("Draw the curve_right itself");
			}
			Tgraph2D::TLineType lt = Tgraph2D::LINE_SOLID;
			if (linetype == 0)
				lt = Tgraph2D::LINE_NONE;
			else if (linetype == 1)
				lt = Tgraph2D::LINE_SOLID;
			else if (linetype == 2)
				lt = Tgraph2D::LINE_DASHED;
			else if (linetype == 3)
				lt = Tgraph2D::LINE_DOTTED;
			else if (linetype == 4)
				lt = Tgraph2D::LINE_DOTDASHED;
			g.setlinetype(lt);
			g.setlinewidth(linewidth);
			g.setcolor(linecolor);
			g.curve(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,ys_right[curves_done]->data,
					lmargin,yy-panelheight,Xmax-rmargin,yy,
					xmin,xmax,ymin_right,ymax_right,(int(markertype+0.5)-1) % 13 + 1,markersize);
			g.setlinetype(Tgraph2D::LINE_SOLID);
			g.setlinewidth(framewidth);
			g.setcolor(foreground);
			g.end_clipped();
			ys_right[curves_done] = 0;
			curves_done++;
			if (curves_done >= CURVES_MAX) {
				cerr << ErrorProlog << "curve_right: Too many curves or pcolors in single panel\n";
				return;
			}
			panel_had_any_rightcurve = true;
		}
		break;
		case PCOLOR:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin);
			// yy == top boundary of panel
			int xcurv_ind,ycurv_ind;
			for (xcurv_ind=curves_done; xcurv_ind>=0; xcurv_ind--) if (xs[xcurv_ind]) break;
			for (ycurv_ind=curves_done; ycurv_ind>=0; ycurv_ind--) if (ys[ycurv_ind]) break;
			if (xcurv_ind<0 || ycurv_ind<0 || !xs[xcurv_ind] || !ys[ycurv_ind]) {
				cerr << ErrorProlog << "pcolor: x or y undefined\n";
				break;
			}
			if (!zs[curves_done]) {
				cerr << ErrorProlog << "pcolor: z undefined\n";
				break;
			}
			if (xs[xcurv_ind]->ncols != zs[curves_done]->nrows || ys[ycurv_ind]->ncols != zs[curves_done]->ncols) {
				cerr << ErrorProlog << "pcolor: x and y vector lengths do not match with z-matrix (xlen="
					 << xs[xcurv_ind]->ncols << ",ylen=" << ys[curves_done]->ncols << ", zsize="
					 << zs[curves_done]->nrows << "x" << zs[curves_done]->ncols << ")\n";
				break;
			}
			g.comment("Draw pcolor");
			g.begin_clipped(lmargin,yy-panelheight,Xmax-rmargin,yy);
			if (interp!=0.0)
				g.pcolor(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,
						 ys[ycurv_ind]->ncols,ys[ycurv_ind]->data,
						 zs[curves_done]->data,
						 zmin, zmax,
						 lmargin,yy-panelheight,Xmax-rmargin,yy,
						 xmin,xmax,ymin,ymax,
						 false,true);
			g.pcolor(xs[xcurv_ind]->ncols,xs[xcurv_ind]->data,
					 ys[ycurv_ind]->ncols,ys[ycurv_ind]->data,
					 zs[curves_done]->data,
					 zmin, zmax,
					 lmargin,yy-panelheight,Xmax-rmargin,yy,
					 xmin,xmax,ymin,ymax,
					 interp!=0.0);
			g.end_clipped();
			zs[curves_done] = 0;
			curves_done++;
			has_drawn_any_pcolor = true;
			if (curves_done >= CURVES_MAX) {
				cerr << ErrorProlog << "curve: Too many curves or pcolors in single panel\n";
				return;
			}
		}
		break;
		case PANEL:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin);
			g.comment("Draw panel box");
			g.setcolor(foreground);
			g.rrect(lmargin,yy,
					Xmax-lmargin-rmargin,-panelheight);
			g.setfont(Tgraph2D::FONT_AXISLABEL);
			// Upper X-axis, always unlabelled, and only for upmost panel:
			if (npanels_plotted==0) {
				g.comment("Draw upper X-axis");
				g.axis(lmargin,yy,Xmax-rmargin,yy,0,xticklen, xlog, xmin,xmax, false,houraxis_x!=0.0);
			}
			// Lower X-axis, labelled only for lowest panel:
			g.comment("Draw lower X-axis");
			g.axis(lmargin,yy-panelheight,Xmax-rmargin,yy-panelheight,
				   0,-xticklen, xlog, xmin,xmax, npanels_plotted==npanels-1,houraxis_x!=0.0);
			if (npanels_plotted == npanels-1) {
				if (xleftlabel) {
					g.comment("Draw left label of x-axis");
					g.text(lefttextlabelpos,yy-panelheight-xticklen*1.3,xleftlabel,-1,-1);
				}
				if (xrightlabel) {
					g.comment("Draw right label of x-axis\n");
					g.text(righttextlabelpos,yy-panelheight-xticklen*1.3,xrightlabel,1,-1);
				}
				DeleteString(xleftlabel);
				DeleteString(xrightlabel);
			}
			if (draw_grid_lines != 0.0) {
				g.comment("Draw vertical grid lines");
				g.setlinetype(Tgraph2D::LINE_DOTTED);
				g.setlinewidth(0.5*framewidth);
				g.axis(lmargin,yy-panelheight,Xmax-rmargin,yy-panelheight, 0,panelheight, xlog, xmin,xmax,
					   false,houraxis_x!=0.0,MINORTICKS_FALSE);
				g.setlinetype(Tgraph2D::LINE_SOLID);
				g.setlinewidth(framewidth);
			}
			// Left Y-axis:
			g.comment("Draw left Y-axis");
			g.axis(lmargin,yy-panelheight,lmargin,yy,
				   -yticklen,0, ylog, ymin,ymax, true,false,include_minor_ticks);
			if (draw_grid_lines != 0.0) {
				g.comment("Draw horizontal grid lines");
				g.setlinetype(Tgraph2D::LINE_DOTTED);
				g.setlinewidth(0.5*framewidth);
				g.axis(lmargin,yy-panelheight,lmargin,yy,Xmax-rmargin-lmargin,0,ylog,ymin,ymax, false,false,MINORTICKS_FALSE);
				g.setlinetype(Tgraph2D::LINE_SOLID);
				g.setlinewidth(framewidth);
			}
			if (panel_had_any_rightcurve) {
				// Right Y-axis:
				g.comment("Draw right Y-axis");
//				cerr << "Drawing right y-axis, ylog_right=" << ylog_right << ",ymin_right=" << ymin_right
//					 << ",ymax_right=" << ymax_right << ",include_minor_ticks=" << include_minor_ticks << "\n";
				g.axis(Xmax-rmargin,yy-panelheight,Xmax
					   -rmargin,yy,
					   yticklen,0, ylog_right, ymin_right,ymax_right, true,false,include_minor_ticks);
			}
			if (leftlabel) {
				g.comment("Draw left sidelabel");
				g.setfont(Tgraph2D::FONT_SIDELABEL);
				g.text(leftlabelpos,yy-0.5*panelheight,leftlabel,0,0,true);
			}
			if (rightlabel) {
				g.comment("Draw right sidelabel");
				g.setfont(Tgraph2D::FONT_SIDELABEL);
				g.text(rightlabelpos,yy-0.5*panelheight,rightlabel,0,0,true);
			}
			if (has_drawn_any_pcolor) {
				g.comment("Draw vertical color bar");
				const real xdata[2] = {0,1};
				real ydata[256];
				real zdata[2][256];
				int i;
				for (i=0; i<256; i++) {
					ydata[i] = i;
					zdata[0][i] = zdata[1][i] = i;
				}
				const greal panelx1=Xmax-rmargin+0.025*minXmaxYmax;
				const greal panely1=yy-panelheight;
				const greal panelx2=Xmax-rmargin+0.05*minXmaxYmax;
				const greal panely2=yy;
				g.begin_clipped(panelx1,panely1,panelx2,panely2);
				g.pcolor(2,xdata,
						 256,ydata,
						 &zdata[0][0],
						 0, 255,
						 panelx1,panely1,panelx2,panely2,
						 0,1,0,255,
						 false);
				g.end_clipped();
				g.comment("Draw frame around vertical color bar");
				g.setcolor(foreground);
				g.rect(panelx1,panely1,panelx2,panely2);
				g.comment("Draw tick marks to vertical color bar");
				g.setfont(Tgraph2D::FONT_AXISLABEL);
				g.axis(panelx2,panely1,panelx2,panely2, barticklen,0, zlog, zmin,zmax, true,false,include_minor_ticks);
			}
			DeleteString(leftlabel);
			DeleteString(rightlabel);
			DeleteString(label);
			linewidth = framewidth;
			linecolor[0] = foreground[0];
			linecolor[1] = foreground[1];
			linecolor[2] = foreground[2];
			fillcolor[0] = background[0];
			fillcolor[1] = background[1];
			fillcolor[2] = background[2];
			markercolor[0] = foreground[0];
			markercolor[1] = foreground[1];
			markercolor[2] = foreground[2];
			linetype = 1;
			markertype = 1;
			markersize = default_markersize;
			endpoint_x_given = endpoint_y_given = false;
			arrow = 0.0;
			ylog = false;
			ylog_right = false;
			zlog = false;
			FindYRange(p->next,ymin,ymax,ymin_right,ymax_right);
			FindZRange(p->next,zmin,zmax);
			has_drawn_any_pcolor = false;
			for (i=0; i<=curves_done; i++) xs[i] = ys[i] = zs[i] = textvecs[i] = 0;
			curves_done = 0;
			houraxis = 0;
			panel_had_any_rightcurve = false;
			npanels_plotted++;
		}
		break;
		case TEXTS:
		{
			const greal yy = Ymax-tmargin-titleheight-npanels_plotted*(panelheight+intramargin)-(0.5+ntexts_plotted)*textheight;
			// yy = top boundary of box where to draw text labels
			int xcurv_ind;
			for (xcurv_ind=curves_done; xcurv_ind>=0; xcurv_ind--) if (xs[xcurv_ind]) break;
			if (xcurv_ind<0 || !xs[xcurv_ind]) {
				cerr << ErrorProlog << "texts: x undefined\n";
				break;
			}
			int textvec_ind;
			for (textvec_ind=curves_done; textvec_ind>=0; textvec_ind--) if (textvecs[textvec_ind]) break;
			if (textvec_ind<0 || !textvecs[textvec_ind]) {
				cerr << ErrorProlog << "texts: textvec undefined\n";
				break;
			}
			if (xs[xcurv_ind]->ncols != textvecs[textvec_ind]->ncols) {
				cerr << ErrorProlog << "texts: x and textvec vectors of unequal lengths ("
					 << xs[xcurv_ind]->ncols << "," << textvecs[textvec_ind]->ncols << ")\n";
			}
			g.setfont(Tgraph2D::FONT_AXISLABEL);
			g.comment("Draw text labels");
			g.DrawAxis1(lmargin,yy-textheight,Xmax-rmargin,yy-textheight,
						0,-xticklen, xlog, xmin,xmax, 1,true,houraxis_x!=0.0,false);
			char lab[80];
			if (houraxis != 0.0) {
				real *const vals = new real [N_selected_xticks];
				bool drawseconds = false;
				bool drawfractseconds = false;
				const real drawseconds_limit = 10.0/60.0;
				const real drawfractseconds_limit = 10.0/3600.0;
				for (i=0; i<N_selected_xticks; i++) {
					vals[i] = 
						FindClosestMatch(xs[xcurv_ind]->data,textvecs[textvec_ind]->data,
										 xs[xcurv_ind]->ncols,selected_xticks_uvalues[i]);
					if (i > 0 && fabs(vals[i] - vals[i-1]) < drawseconds_limit) drawseconds = true;
					if (i > 0 && fabs(vals[i] - vals[i-1]) < drawfractseconds_limit) drawfractseconds = true;
				}
				for (i=0; i<N_selected_xticks; i++) {
					const real val = vals[i];
					int hh = int(floor(val));
					real mm = (val-hh)*60.0;
					if (drawseconds) {
						const real ss = (mm - floor(mm))*60.0;
						if (drawfractseconds)
							sprintf(lab,"%.2d:%.2d:%1.5g",hh,int(floor(mm)),ss);
						else
							sprintf(lab,"%.2d:%.2d:%.2d",hh,int(floor(mm)),int(floor(ss)+0.5));
					} else {
						if (int(mm+0.5)==60) {hh++; mm=0;}
						sprintf(lab,"%.2d:%.2d",hh,int(mm+0.5));
					}
					g.text(selected_xticks[i],yy-textheight,lab,0,1);
				}
				delete [] vals;
			} else {
				// not houraxis, ordinary decimal labels
				for (i=0; i<N_selected_xticks; i++) {
					const real val = 
						FindClosestMatch(xs[xcurv_ind]->data,textvecs[textvec_ind]->data,
										 xs[xcurv_ind]->ncols,selected_xticks_uvalues[i]);
					sprintf(lab,"%1.4g",val);
					g.text(selected_xticks[i],yy-textheight,lab,0,1);
				}
			}
			if (leftlabel) g.text(lefttextlabelpos,yy-textheight,leftlabel,-1,1);
			if (rightlabel) g.text(righttextlabelpos,yy-textheight,rightlabel,1,1);
			DeleteString(leftlabel);
			DeleteString(rightlabel);
			linewidth = framewidth;
			linecolor[0] = foreground[0];
			linecolor[1] = foreground[1];
			linecolor[2] = foreground[2];
			fillcolor[0] = background[0];
			fillcolor[1] = background[1];
			fillcolor[2] = background[2];
			linetype = 1;
			ylog = false;
			ylog_right = false;
			zlog = false;
			FindYRange(p->next,ymin,ymax,ymin_right,ymax_right);
			FindZRange(p->next,zmin,zmax);
			has_drawn_any_pcolor = false;
			for (i=0; i<=curves_done; i++) xs[i] = ys[i] = zs[i] = textvecs[i] = 0;
			curves_done = 0;
			houraxis = 0;
			ntexts_plotted++;
		}
		break;
		default:
			cerr << "unknown type in plot\n";
		}
	}
	if (title) {
		g.comment("Draw the title");
		g.setcolor(foreground);
		g.setfont(Tgraph2D::FONT_TOPLABEL);
		g.text(0.5*Xmax,Ymax-tmargin-titlefontsize,title,0,1);
		delete [] title;
	}
	if (DrawCopyrightNotice) {
		g.setfont(Tgraph2D::FONT_COPYRIGHT);
		g.text(Xmax-20,Ymax-20,"SPPC (c) 2000 /Pj",-1,-1);
	}
}

bool Tsppc::plotPS(ostream& o, real paperwidth_pts, real paperheight_pts, bool landscape, const char *paper)
{
	const greal Xmax=paperwidth_pts /*600*/, Ymax=paperheight_pts /*800*/;
	Tentry *p;
	greal titlefontsize = 18, labelfontsize = 14, axisfontsize = 12, annotfontsize = 12;
	for (p=lst; p && p->type != PLOT; p=p->next) {
		if (p->type == TITLEFONTSIZE) GetScalar(p,titlefontsize);
		if (p->type == LABELFONTSIZE) GetScalar(p,labelfontsize);
		if (p->type == AXISFONTSIZE) GetScalar(p,axisfontsize);
		if (p->type == ANNOTFONTSIZE) GetScalar(p,annotfontsize);
	}
	TPostScript ps(o,Xmax,Ymax,titlefontsize,labelfontsize,axisfontsize,annotfontsize,landscape,paper);
	plot(ps,Xmax,Ymax,titlefontsize,labelfontsize,axisfontsize,annotfontsize,true);
	return true;
}
