-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--  Synopsis:                                                                 --
--                                                                            --
--  Procedure to analyse a .rsm file                                          --
--                                                                            --
--------------------------------------------------------------------------------

separate (VCS)
procedure Analyse_Riposte_Summary_File
  (Report_File             : in     SPARK_IO.File_Type;
   Filename                : in     E_Strings.T;
   Error_In_RSM_File       :    out Boolean;
   File_Error              :    out E_Strings.T;
   Temp_Riposte_Error_File : in     SPARK_IO.File_Type)
is

   --  String and error conctants.
   Str_File_Corrupt : constant String := "Riposte summary file corrupt: Could not parse line";
   Str_Cannot_Open  : constant String := "Cannot open Riposte summary file";

   type Riposte_Status_T is (Riposte_Error, Riposte_True, Riposte_True_Sortof, Riposte_False, Riposte_Undischarged);

   --  As defined in the Riposte User Manual
   type Riposte_CSV_Line_T is record
      Unit                              : E_Strings.T;
      Verdict                           : Riposte_Status_T;
      Verdict_Error                     : Boolean;
      Verdict_True                      : Boolean;
      Verdict_Sound                     : Boolean;
      Verdict_Complete                  : Boolean;
      Verdict_Affected_By_Undefinedness : Boolean;
      Verdict_Time                      : Natural;
      Verdict_Ground_Time               : Integer;
      Verdict_Solved_Time               : Integer;
   end record;

   Invalid_CSV_Line : constant Riposte_CSV_Line_T :=
     Riposte_CSV_Line_T'
     (Unit                              => E_Strings.Empty_String,
      Verdict                           => Riposte_Error,
      Verdict_Error                     => True,
      Verdict_True                      => False,
      Verdict_Sound                     => True,
      Verdict_Complete                  => False,
      Verdict_Affected_By_Undefinedness => False,
      Verdict_Time                      => 0,
      Verdict_Ground_Time               => -1,
      Verdict_Solved_Time               => -1);

   File_Status          : SPARK_IO.File_Status;
   Riposte_Summary_File : SPARK_IO.File_Type;
   Line_Read            : E_Strings.T;
   Trimmed_Line         : E_Strings.T;
   Success              : Boolean;
   CSV_Line             : Riposte_CSV_Line_T;
   Error_Flag_Mentioned : Boolean;

   function Unqote (E_Str : E_Strings.T) return E_Strings.T is
      L      : E_Strings.Lengths;
      Retval : E_Strings.T;
   begin
      L := E_Strings.Get_Length (E_Str);
      if L >= 2
        and then E_Strings.Get_Element (E_Str, E_Strings.Positions'First) = '"'
        and then E_Strings.Get_Element (E_Str, L) = '"' then
         Retval := E_Strings.Section (E_Str     => E_Str,
                                      Start_Pos => E_Strings.Positions'First + 1,
                                      Length    => L - 2);
      else
         Retval := E_Str;
      end if;
      return Retval;
   end Unqote;

   procedure Parse_CSV_String
     (CSV_Line          : in     E_Strings.T;
      CSV_Line_Position : in out E_Strings.Positions;
      The_String        :    out E_Strings.T;
      Ok                :    out Boolean;
      Expect_EOL        : in     Boolean)
   --# derives CSV_Line_Position,
   --#         Ok,
   --#         The_String        from CSV_Line,
   --#                                CSV_Line_Position,
   --#                                Expect_EOL;
   --# pre CSV_Line_Position <= E_Strings.Get_Length (CSV_Line) + 1;
   --# post CSV_Line_Position <= E_Strings.Get_Length (CSV_Line) + 1;
   is
      Is_Quoted_String : Boolean;
      In_Quoted_String : Boolean;
      Comma_Found      : Boolean;
      EOL_Found        : Boolean;
      End_Position     : E_Strings.Positions;
      Tmp_String       : E_Strings.T;
   begin
      Is_Quoted_String := E_Strings.Get_Element (CSV_Line, CSV_Line_Position) = '"';
      In_Quoted_String := Is_Quoted_String;
      Comma_Found      := False;
      End_Position     := CSV_Line_Position;
      for I in E_Strings.Positions range CSV_Line_Position .. E_Strings.Get_Length (CSV_Line) loop
         --# assert E_Strings.Get_Length (CSV_Line) <= E_Strings.Lengths'Last;
         End_Position := I;
         case E_Strings.Get_Element (CSV_Line, I) is
            when '"' =>
               if In_Quoted_String and I > CSV_Line_Position then
                  In_Quoted_String := False;
               end if;
            when ',' =>
               if not In_Quoted_String then
                  Comma_Found := True;
               end if;
            when others =>
               null;
         end case;
         exit when Comma_Found;
      end loop;

      --# assert End_Position >= CSV_Line_Position and End_Position <= E_Strings.Get_Length (CSV_Line) + 1
      --#   and (Comma_Found -> (End_Position <= E_Strings.Get_Length (CSV_Line)));

      --  Work out if we hit the end of line.
      EOL_Found := not Comma_Found or else E_Strings.Get_Length (CSV_Line) = E_Strings.Lengths'Last;

      --  Make sure we found a comma if we were looking for one.
      Ok := Expect_EOL = EOL_Found;

      --  Make sure any quoted strings are OK.
      Ok := Ok and not In_Quoted_String;

      if Ok then
         Tmp_String :=
           E_Strings.Section (E_Str     => CSV_Line,
                              Start_Pos => CSV_Line_Position,
                              Length    => End_Position - CSV_Line_Position);

         --  Strip away the quotes, if necessary.
         if Is_Quoted_String then
            The_String := Unqote (Tmp_String);
         else
            The_String := Tmp_String;
         end if;

         --  Jump over the comma.
         if not EOL_Found then
            CSV_Line_Position := End_Position + 1;
         end if;
      else
         The_String := E_Strings.Empty_String;
      end if;
   end Parse_CSV_String;

   function Parse_Natural (E_Str          : E_Strings.T;
                           Value_On_Error : Natural) return Natural is
      Tmp_Integer : Integer;
      The_Natural : Natural;
      Tmp_Stop    : Integer;
   begin
      The_Natural := Value_On_Error;
      if E_Strings.Get_Length (E_Str) > 0 then
         --# accept F, 10, Tmp_Stop, "We don't care about Stop at the moment";
         E_Strings.Get_Int_From_String
           (Source   => E_Str,
            Item     => Tmp_Integer,
            Start_Pt => E_Strings.Positions'First,
            Stop     => Tmp_Stop);
         --# end accept;

         -- TODO: Check that Tmp_Stop = E_Strings.Get_Length (Tmp_String) ?

         if Tmp_Integer >= Natural'First then
            The_Natural := Natural'(Tmp_Integer);
         end if;
      end if;
      --# accept F, 33, Tmp_Stop, "We don't care about Stop at the moment";
      return The_Natural;
   end Parse_Natural;

   function Parse_Riposte_Status (E_Str          : E_Strings.T;
                                  Value_On_Error : Riposte_Status_T) return Riposte_Status_T is
      The_Riposte_Status : Riposte_Status_T;
   begin
      The_Riposte_Status := Value_On_Error;
      if E_Strings.Eq1_String (E_Str, "true") then
         The_Riposte_Status := Riposte_True;
      elsif E_Strings.Eq1_String (E_Str, "true_but_affected_by_undefinedness") then
         The_Riposte_Status := Riposte_True_Sortof;
      elsif E_Strings.Eq1_String (E_Str, "false") then
         The_Riposte_Status := Riposte_False;
      elsif E_Strings.Eq1_String (E_Str, "undischarged") then
         The_Riposte_Status := Riposte_Undischarged;
      elsif E_Strings.Eq1_String (E_Str, "error") then
         The_Riposte_Status := Riposte_Error;
      end if;
      return The_Riposte_Status;
   end Parse_Riposte_Status;

   procedure Parse_Riposte_CSV_Line (The_Line   : in     E_Strings.T;
                                     The_Record :    out Riposte_CSV_Line_T;
                                     Ok         :    out Boolean)
   --# derives Ok,
   --#         The_Record from The_Line;
   --# pre E_Strings.Get_Length (The_Line) >= 1;
   is
      Current_Position : E_Strings.Positions := E_Strings.Positions'First;
      Tmp              : E_Strings.T;

      Num_CSV_Entries : constant Natural := 10;
      subtype CSV_Record_Index is Natural range 1 .. Num_CSV_Entries;
   begin
      The_Record := Invalid_CSV_Line;

      for I in CSV_Record_Index
      --# assert Current_Position <= E_Strings.Get_Length (The_Line) + 1;
      loop
         Parse_CSV_String
           (CSV_Line          => The_Line,
            CSV_Line_Position => Current_Position,
            The_String        => Tmp,
            Ok                => Ok,
            Expect_EOL        => (I = CSV_Record_Index'Last));
         exit when not Ok;
         --  Each I will map to each number given in the Riposte user
         --  manual describing the "summary" proof artefact.
         case I is
            when 1 =>
               The_Record.Unit := Tmp;
            when 2 =>
               The_Record.Verdict := Parse_Riposte_Status (Tmp, Riposte_Error);
            when 3 =>
               The_Record.Verdict_Error := Parse_Natural (Tmp, 1) /= 0;
            when 4 =>
               The_Record.Verdict_True := Parse_Natural (Tmp, 0) /= 0;
            when 5 =>
               The_Record.Verdict_Sound := Parse_Natural (Tmp, 1) /= 0;
            when 6 =>
               The_Record.Verdict_Complete := Parse_Natural (Tmp, 0) /= 0;
            when 7 =>
               The_Record.Verdict_Affected_By_Undefinedness := Parse_Natural (Tmp, 0) /= 0;
            when 8 =>
               The_Record.Verdict_Time := Parse_Natural (Tmp, 0);
            when 9 =>
               if E_Strings.Get_Length (Tmp) > 0 then
                  The_Record.Verdict_Time := Parse_Natural (Tmp, 0);
               end if;
            when 10 =>
               if E_Strings.Get_Length (Tmp) > 0 then
                  The_Record.Verdict_Time := Parse_Natural (Tmp, 0);
               end if;
         end case;
      end loop;
   end Parse_Riposte_CSV_Line;

begin -- Analyse_Riposte_Summary_File
   Error_In_RSM_File    := False;
   File_Error           := E_Strings.Empty_String;
   Riposte_Summary_File := SPARK_IO.Null_File;
   Error_Flag_Mentioned := False;

   -- open Riposte results file
   E_Strings.Open
     (File         => Riposte_Summary_File,
      Mode_Of_File => SPARK_IO.In_File,
      Name_Of_File => Filename,
      Form_Of_File => "",
      Status       => File_Status);
   if File_Status /= SPARK_IO.Ok then
      Error_In_RSM_File := True;
      File_Error        := E_Strings.Copy_String (Str_Cannot_Open);
      FatalErrors.Process (FatalErrors.Could_Not_Open_Input_File, E_Strings.Empty_String);
   end if;

   loop
      Read_Next_Non_Blank_Line (File      => Riposte_Summary_File,
                                Success   => Success,
                                File_Line => Line_Read);
      exit when not Success;
      --# assert Success;

      Trimmed_Line := E_Strings.Trim (Line_Read);
      Success      := E_Strings.Get_Length (Trimmed_Line) >= 1;

      if Success then
         Parse_Riposte_CSV_Line (The_Line   => E_Strings.Trim (Line_Read),
                                 The_Record => CSV_Line,
                                 Ok         => Success);
      else
         CSV_Line := Invalid_CSV_Line;
      end if;

      if not Success then
         --  Notify stdout.
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "************* " & Str_File_Corrupt & " ************", 0);
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "*** Offending line was: [", 0);
         E_Strings.Put_String (SPARK_IO.Standard_Output, Trimmed_Line);
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "]", 0);

         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);

         --  Also put somthing in the report file.
         SPARK_IO.Put_String (Report_File, "*** " & Str_File_Corrupt & " ***", 0);

         --  And finally set error flags.
         File_Error        := E_Strings.Copy_String (Str_File_Corrupt);
         Error_In_RSM_File := True;
      end if;

      --# assert True;

      if not Success then
         null;
      else
         case CSV_Line.Verdict is
            when Riposte_True =>
               VCHeap.Set_VC_State (CSV_Line.Unit, VCDetails.VC_Proved_By_Riposte);
            when Riposte_False =>
               VCHeap.Set_VC_State (CSV_Line.Unit, VCDetails.VC_False);
            when Riposte_True_Sortof | Riposte_Undischarged =>
               --  We don't do anything in this case.
               null;
            when Riposte_Error =>
               --  This means riposte could not fully parse the VC or
               --  encountered some other kind of error. We will flag
               --  this up and include it in the final
               --  summary. However, we will only do this once.
               if not Error_Flag_Mentioned then
                  E_Strings.Put_String (File  => Temp_Riposte_Error_File,
                                        E_Str => PathFormatter.Format (Filename));
                  SPARK_IO.Put_Char (Temp_Riposte_Error_File, ' ');
                  SPARK_IO.Put_Char (Temp_Riposte_Error_File, '(');
                  SPARK_IO.Put_String (Temp_Riposte_Error_File, "Error returned by Riposte.", 0);
                  SPARK_IO.Put_Line (Temp_Riposte_Error_File, ")", 0);
                  Error_Flag_Mentioned := True;
               end if;
         end case;
      end if;
   end loop;

   --# accept F, 10, File_Status, "We don't care anymore since we've got everything we came for." &
   --#        F, 10, Riposte_Summary_File, "Same as above.";
   SPARK_IO.Close (Riposte_Summary_File, File_Status);
   --# end accept;

   --# accept Flow, 601, FatalErrors.State, Temp_Riposte_Error_File, "False coupling through SPARK_IO" &
   --#        Flow, 601, VCHeap.State, Temp_Riposte_Error_File, "False coupling through SPARK_IO";
end Analyse_Riposte_Summary_File;
