-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Text_IO;

separate (Sem.Walk_Expression_P)
package body Exp_Stack is

   procedure Debug_Item (Op   : in String;
                         Item : in Sem.Exp_Record)
   --# derives null from Item,
   --#                   Op;
   is
      --# hide Debug_Item;
      Str : E_Strings.T;
   begin
      Ada.Text_IO.Put_Line (Op);
      if Item.Type_Symbol /= Dictionary.NullSymbol then
         Ada.Text_IO.Put ("   Sem.Exp_Record.TypeSymbol = ");
         Dictionary.GenerateSimpleName (Item.Type_Symbol, ".", Str);
         E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                             E_Str => Str);
      end if;
      if Item.Other_Symbol /= Dictionary.NullSymbol then
         Ada.Text_IO.Put ("   Sem.Exp_Record.OtherSymbol = ");
         Dictionary.GenerateSimpleName (Item.Other_Symbol, ".", Str);
         E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                             E_Str => Str);
      end if;
      if Item.Stream_Symbol /= Dictionary.NullSymbol then
         Ada.Text_IO.Put ("   Sem.Exp_Record.StreamSymbol = ");
         Dictionary.GenerateSimpleName (Item.Stream_Symbol, ".", Str);
         E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                             E_Str => Str);
      end if;
      if Item.Tagged_Parameter_Symbol /= Dictionary.NullSymbol then
         Ada.Text_IO.Put ("   Sem.Exp_Record.TaggedParameterSymbol = ");
         Dictionary.GenerateSimpleName (Item.Tagged_Parameter_Symbol, ".", Str);
         E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                             E_Str => Str);
      end if;
      if Item.Variable_Symbol /= Dictionary.NullSymbol then
         Ada.Text_IO.Put ("   Sem.Exp_Record.VariableSymbol = ");
         Dictionary.GenerateSimpleName (Item.Variable_Symbol, ".", Str);
         E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                             E_Str => Str);
      end if;
   end Debug_Item;

   procedure Debug_Stack (Stack : in Exp_Stack_Type)
   --# derives null from Stack;
   is
      --# hide Debug_Stack;
   begin
      Ada.Text_IO.Put_Line ("----------------------------------------");
      for I in Index_Range'First .. Stack.Top_Ptr loop
         Debug_Item (Op   => "Exp_Stack.Debug_Stack",
                     Item => Stack.S (I));
      end loop;
      Ada.Text_IO.Put_Line ("----------------------------------------");
   end Debug_Stack;

   procedure Init (Stack : out Exp_Stack_Type) is
   begin
      Stack.Top_Ptr := 0;
      --# accept Flow, 32, Stack.S, "Safe partial initialisation" &
      --#        Flow, 31, Stack.S, "Safe partial initialisation" &
      --#        Flow, 602, Stack, Stack.S, "Safe partial initialisation";
   end Init;

   procedure Push (X     : in     Sem.Exp_Record;
                   Stack : in out Exp_Stack_Type) is
   begin
      if CommandLineData.Content.Debug.Expressions then
         Debug_Item ("Exp_Stack.Push", X);
      end if;

      if Stack.Top_Ptr < ExaminerConstants.WalkExpStackMax then
         Stack.Top_Ptr           := Stack.Top_Ptr + 1;
         Stack.S (Stack.Top_Ptr) := X;
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Expression_Stack_Overflow,
                                   Msg     => "in Exp_Stack.Push");
      end if;

      if CommandLineData.Content.Debug.Expressions then
         Debug_Stack (Stack);
      end if;
   end Push;

   procedure Pop (Item  :    out Sem.Exp_Record;
                  Stack : in out Exp_Stack_Type) is
   begin
      if Stack.Top_Ptr > 0 then
         Item          := Stack.S (Stack.Top_Ptr);
         Stack.Top_Ptr := Stack.Top_Ptr - 1;

         if CommandLineData.Content.Debug.Expressions then
            Debug_Item ("Exp_Stack.Pop", Item);
            Debug_Stack (Stack);
         end if;

      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Expression_Stack_Underflow,
                                   Msg     => "in Exp_Stack.Pop");
         Item := Stack.S (Stack.Top_Ptr);
      end if;
   end Pop;

   function Top (Stack : Exp_Stack_Type) return Sem.Exp_Record is
      Result : Sem.Exp_Record;
   begin
      if Stack.Top_Ptr > 0 then
         Result := Stack.S (Stack.Top_Ptr);
      else
         Result := Sem.Null_Exp_Record;
      end if;
      return Result;
   end Top;

   function Has_One_Entry (Stack : Exp_Stack_Type) return Boolean is
   begin
      return Stack.Top_Ptr = 1;
   end Has_One_Entry;

   function Is_Empty (Stack : Exp_Stack_Type) return Boolean is
   begin
      return Stack.Top_Ptr = 0;
   end Is_Empty;

end Exp_Stack;
