/*
 * (C) Copyright 2002, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Local configuration
#include "config.h"

// Implementation
#include "XMLNode.hpp"

// STDC++
#include <cassert>

// Local
#include "XMLException.hpp"


namespace SP {
namespace GXML {

XMLNode::XMLNode()
{
	node_ = NULL;
}

XMLNode::XMLNode( const xmlNodePtr node )
{
	if ( node == NULL )
	{
		throw XMLException( XMLException::XMLNODE_NO_NODE );
	}
	assert( node );
	node_ = node;
}

XMLNode::~XMLNode()
{}

int
XMLNode::getType()
{
	return node_->type;
}

XMLNode
XMLNode::getNext()
{
	if ( node_->next == NULL ) {
		throw XMLException( XMLException::XMLNODE_LAST_NODE );
	}
	assert( node_->next );
	XMLNode result( node_->next );
	return result;
}

XMLNode
XMLNode::getPrev()
{
	if ( node_->prev == NULL ) {
		throw XMLException( XMLException::XMLNODE_FIRST_NODE );
	}
	assert( node_->prev );
	XMLNode result( node_->prev );
	return result;
}

XMLNode
XMLNode::getParent()
{
	if ( node_->parent == NULL ) {
		throw XMLException( XMLException::XMLNODE_NO_PARENT );
	}
	assert( node_->parent );
	XMLNode result( node_->parent );
	return result;
}

XMLNode
XMLNode::getLast()
{
	if ( node_->last == NULL ) {
		// node_->last points to the last child of the node
		throw XMLException( XMLException::XMLNODE_NO_CHILD );
	}
	assert( node_->last );
	XMLNode result( node_->last );
	return result;
}

XMLNode
XMLNode::getChild()
{
	if ( node_->children == NULL )
	{
		// should only happen if the node has no content AND no child nodes.
		throw XMLException( XMLException::XMLNODE_NO_CHILD );
	}
	assert( node_->children );
	XMLNode result( node_->children );
	return result;
}

std::string
XMLNode::getName()
{
	// node with no name should not exist
	assert( node_->name );
	return (std::string)(char*)node_->name;
}

std::string
XMLNode::getContent()
{
	if ( node_->children == NULL ) {
		throw XMLException( XMLException::XMLNODE_NO_CHILD );
	}
	assert( node_->children );
	// if node has content it will be the content of the first child (text-node)
	if ( node_->children->content == NULL ) {
		throw XMLException( XMLException::XMLNODE_NO_CONTENT );
	}
	assert( node_->children->content );
	return (std::string)(char*)node_->children->content;
}


std::string
XMLNode::getAttribute(const std::string& name)
{
	if (node_->properties == 0)
	{
		throw XMLException(XMLException::XMLNODE_NO_ATTRIBUTE);
	}

	// search for attribute by 'name'
	xmlAttrPtr attr(node_->properties);

	while (attr != 0 && std::strcmp((char *)attr->name, name.c_str()) != 0)
		attr = attr->next;

	if (!attr)
		throw XMLException(XMLException::XMLNODE_NO_ATTRIBUTE);

	// Attribute found
	assert( attr->children );
	if (attr->children->content == 0)
	{
		// will never happen, cause child of attribute is always text-node (although it might be empty)
		throw XMLException(XMLException::XMLNODE_NO_CONTENT);
	}
	assert( attr->children->content );
	return (std::string)(char*)attr->children->content;
}

std::string
XMLNode::getNodeDump()
{
	xmlBufferPtr nodeDump = xmlBufferCreate();
	if ( !nodeDump ) {
		throw XMLException( XMLException::XMLNODE_BUFFER_CREATE );
	}
	if ( node_ ) {
		xmlNodeDump( nodeDump, node_->doc, node_, 0, 0 );
		if ( !nodeDump->content ) {
			xmlBufferFree( nodeDump );
			throw XMLException( XMLException::XMLNODE_NODE_DUMP );
		}
	} else {
		throw XMLException( XMLException::XMLNODE_NODE_DUMP );
	}

	std::string result((char*)nodeDump->content);
	xmlBufferFree( nodeDump );
	return result;
}

xmlNodePtr XMLNode::getNodePtr()
{
	return node_;
}

}}


