# Forward declarations
cdef class GeomObject(_soya.CoordSyst)
cdef class SpaceBase(GeomObject)

# World
cdef class _World(_soya._World):
    """Dynamics world.
    
    The world object is a container for rigid bodies and joints.
    
    
    Constructor::
    
      World()
    """

    cdef public double round_duration
    cdef dWorldID wid
#    cdef readonly spaces

    def __new__(self, _soya._World parent=None):
        self.round_duration = 0.030 # Same as the Soya idler
        self.wid = dWorldCreate()

#    def __init__(self, _soya._World parent=None):
#        self.spaces = []
#        _soya._World.__init__(self, parent)

    def __dealloc__(self):
        if self.wid!=NULL:
            dWorldDestroy(self.wid)

    property gravity:
        """setGravity(gravity)
 
        Set the world's global gravity vector.

        @param gravity: Gravity vector
        @type gravity: 3-sequence of floats
        """
        def __set__(self, gravity):
            dWorldSetGravity(self.wid, gravity[0], gravity[1], gravity[2])
    
        def __get__(self):
            cdef dVector3 g
            dWorldGetGravity(self.wid, g)
            return (g[0],g[1],g[2])

    property erp:
        """setERP(erp)

        Set the global ERP value, that controls how much error
        correction is performed in each time step. Typical values are
        in the range 0.1-0.8. The default is 0.2.

        @param erp: Global ERP value
        @type erp: float
        """
        def __set__(self, erp):
            dWorldSetERP(self.wid, erp)

        def __get__(self):
            return dWorldGetERP(self.wid)

    property cfm:
        """setCFM(cfm)

        Set the global CFM (constraint force mixing) value. Typical
        values are in the range 10E-9 - 1. The default is 10E-5 if
        single precision is being used, or 10E-10 if double precision
        is being used.

        @param cfm: Constraint force mixing value
        @type cfm: float
        """
        def __set__(self, cfm):
            dWorldSetCFM(self.wid, cfm)

        def __get__(self):
            return dWorldGetCFM(self.wid)

    # step
    def step(self, stepsize):
        """step(stepsize)

        Step the world. This uses a "big matrix" method that takes
        time on the order of O(m3) and memory on the order of O(m2), where m
        is the total number of constraint rows.

        For large systems this will use a lot of memory and can be
        very slow, but this is currently the most accurate method.

        @param stepsize: Time step
        @type stepsize: float
        """

        dWorldStep(self.wid, stepsize)

    # quickStep
    def quickStep(self, stepsize):
        """quickStep(stepsize)
        
        Step the world. This uses an iterative method that takes time
        on the order of O(m*N) and memory on the order of O(m), where m is
        the total number of constraint rows and N is the number of
        iterations.

        For large systems this is a lot faster than dWorldStep, but it
        is less accurate.

        @param stepsize: Time step
        @type stepsize: float        
        """
        dWorldQuickStep(self.wid, stepsize)

    property quickstep_num_iterations:
        """setQuickStepNumIterations(num)
        
        Set the number of iterations that the QuickStep method
        performs per step. More iterations will give a more accurate
        solution, but will take longer to compute. The default is 20
        iterations.

        @param num: Number of iterations
        @type num: int
        """
        def __set__(self, num):
        
            dWorldSetQuickStepNumIterations(self.wid, num)

        def __get__(self):
            return dWorldGetQuickStepNumIterations(self.wid)

    property contact_max_correcting_velocity:
        """setContactMaxCorrectingVel(vel)

        Set the maximum correcting velocity that contacts are allowed
        to generate. The default value is infinity (i.e. no
        limit). Reducing this value can help prevent "popping" of
        deeply embedded objects.

        @param vel: Maximum correcting velocity
        @type vel: float
        """
        def __set__(self, vel):
            dWorldSetContactMaxCorrectingVel(self.wid, vel)

        def __get__(self):
            return dWorldGetContactMaxCorrectingVel(self.wid)

    property contact_surface_layer:
        """setContactSurfaceLayer(depth)

        Set the depth of the surface layer around all geometry
        objects. Contacts are allowed to sink into the surface layer
        up to the given depth before coming to rest. The default value
        is zero. Increasing this to some small value (e.g. 0.001) can
        help prevent jittering problems due to contacts being
        repeatedly made and broken.

        @param depth: Surface layer depth
        @type depth: float
        """
        def __set__(self, depth):
            dWorldSetContactSurfaceLayer(self.wid, depth)

        def __get__(self):
            return dWorldGetContactSurfaceLayer(self.wid)

    property auto_disable:
        """setAutoDisableFlag(flag)
        
        Set the default auto-disable flag for newly created bodies.

        @param flag: True = Do auto disable
        @type flag: bool
        """
        def __set__(self, flag):
            dWorldSetAutoDisableFlag(self.wid, flag)
        
        def __get__(self):
            return dWorldGetAutoDisableFlag(self.wid)

    property auto_disable_linear_threshold:
        """setAutoDisableLinearThreshold(threshold)
        
        Set the default auto-disable linear threshold for newly created
        bodies.

        @param threshold: Linear threshold
        @type threshold: float
        """
        def __set__(self, threshold):
            dWorldSetAutoDisableLinearThreshold(self.wid, threshold)

        def __get__(self):
            return dWorldGetAutoDisableLinearThreshold(self.wid)

    property auto_disable_angular_threshold:
        """setAutoDisableAngularThreshold(threshold)
        
        Set the default auto-disable angular threshold for newly created
        bodies.

        @param threshold: Angular threshold
        @type threshold: float
        """
        def __set__(self, threshold):
            dWorldSetAutoDisableAngularThreshold(self.wid, threshold)

        def __get__(self):
            return dWorldGetAutoDisableAngularThreshold(self.wid)

    property auto_disable_steps:
        """setAutoDisableSteps(steps)
        
        Set the default auto-disable steps for newly created bodies.

        @param steps: Auto disable steps
        @type steps: int
        """
        def __set__(self, steps):
            dWorldSetAutoDisableSteps(self.wid, steps)

        def __get__(self):
            return dWorldGetAutoDisableSteps(self.wid)

    property auto_disable_time:
        """setAutoDisableTime(time)
        
        Set the default auto-disable time for newly created bodies.

        @param time: Auto disable time
        @type time: float
        """
        def __set__(self, time):
            dWorldSetAutoDisableTime(self.wid, time)

        def __get__(self):
            return dWorldGetAutoDisableTime(self.wid)

    # impulseToForce
    def impulseToForce(self, stepsize, impulse):
        """impulseToForce(stepsize, impulse) -> 3-tuple

        If you want to apply a linear or angular impulse to a rigid
        body, instead of a force or a torque, then you can use this
        function to convert the desired impulse into a force/torque
        vector before calling the dBodyAdd... function.

        @param stepsize: Time step
        @param impulse: Impulse vector
        @type stepsize: float
        @type impulse: 3-tuple of floats
        """
        cdef dVector3 force
        dWorldImpulseToForce(self.wid, stepsize, impulse[0], impulse[1], impulse[2], force)
        return (force[0], force[1], force[2])

#    def add_space(self, SpaceBase space not None):
#        self.spaces.append(space)

    def begin_round(self):
        """Step the simulation. This is done before the children get
        begin_round"""

        # The user needs to handle collisions
        #for space in self.spaces:
        #    space.collide()

        dWorldQuickStep(self.wid, self.round_duration)

        # Make sure all the children get the end of the round
        _soya._World.begin_round(self)

