#!/usr/bin/env python
"""Curses based test client for SooperLooper interfacing
with ecasound.

2002, Jesse Chappell <gro.jesse@essej.org>
"""

import sys,os
import curses
from curses import textpad

import string
import time
import eci
import traceback

# if you have a low-latency kernel and run as root you 
# should change the -b:256 down to -b:128 or -b:64 for lower latency

ECASTART = '-b:256 -n:SooperLooper -X -z:nointbuf -z:noxruns -z:nodb -z:nopsr' + \
	   ' -a:default -f:s16_le,1,44100 -i:/dev/dsp' + \
	   ' -a:default -f:s16_le,1,44100 -o:/dev/dsp' + \
	   ' -a:default' + \
	   ' -el:SooperLooper,0,1,1,1,1,0,-1,0,0,0,0,0'

MIDILINE = ' -km:7.00,0.00,127.00,32.00,0.00,1.00'


MULTIPORT =       7

QUANTMODEPORT = 10
ROUNDMODEPORT = 11
REDOTAPMODEPORT = 12

STATEPORT =       13
LOOPLENPORT =     14
LOOPPOSPORT =     15
CYCLELENPORT =    16
SECSFREEPORT =    17
SECSTOTALPORT =   18

MULTI_UNDO =      0
MULTI_REDO =      1   # posibbly MULTI_DELAY if toggled, or REDO_TOG if muted
MULTI_REPLACE =   2   # QUANT_TOG in mute mode
MULTI_REVERSE =   3   # ROUND_TOG in mute mode
MULTI_SCRATCH =   4   # unmute continue in mute mode
MULTI_RECORD  =   5   
MULTI_OVERDUB  =  6
MULTI_MULTIPLY =  7
MULTI_INSERT   =  8  # oneshot from mute mode
MULTI_MUTE    =   9  


DEBUGFILE = 'sltester.debug'

multimap = { 0:'Undo',
	     1:'Redo',
	     2:'Replace',
	     3:'Reverse',
	     4:'Scratch',
	     5:'Record',
	     6:'Overdub',
	     7:'Multiply',
	     8:'Insert',
	     9:'Mute' }


STATE_OFF        = 0
STATE_TRIG_START = 1
STATE_RECORD     = 2
STATE_TRIG_STOP  = 3
STATE_PLAY       = 4
STATE_OVERDUB    = 5
STATE_MULTIPLY   = 6
STATE_INSERT     = 7
STATE_REPLACE    = 8
STATE_DELAY      = 9
STATE_MUTE       = 10
STATE_SCRATCH    = 11
STATE_ONESHOT    = 12

statemap = { 0:'Off',
	     1:'TrigRec Start',
	     2:'RECORDING',
	     3:'TrigRec Stop',     	     
	     4:'PLAYING',
	     5:'OVERDUBBING',
	     6:'MULTIPLYING',
	     7:'INSERTING',
	     8:'REPLACING',
	     9:'TAP DELAY',
	     10:'MUTED',
	     11:'SCRATCHING',
	     12:'ONE SHOT'
	     }

BLOCK_TIME = 100  # millisecs
WAIT_TIME = 0.05  # secs


class sltester:
    def __init__(self):

	self.eca = eci.ECI()

	try:
	    self.eca('cs-add default')
	    self.eca('cs-select default')	    

	    initline = ECASTART

	    if '-m' in sys.argv:
		# midi mode
		self.midimode = 1
		initline = initline + MIDILINE
	    else:
		self.midimode = 0
		sys.stderr.write(str(sys.argv))

	    self.eca(initline)

	    self.eca('cs-connect')

	    self.eca('cop-select 1')
	    self.eca('copp-select %d' % MULTIPORT)		


	except eci.ECIError:
	    self._debug_exc()


	# set up curses
	self.stdscr = curses.initscr()

	# screen must be at least 69 chars wide by 41 high
	# for proper display
	maxy,maxx = self.stdscr.getmaxyx()
	if maxy < 42 or maxx < 69:
	    cleanup()
	    print 'Sorry, your terminal must be at least 69x40 for proper display'
	    sys.exit(2)

	try:
	    curses.curs_set(0) # hide cursor
	except:
	    pass




	self.status_tog = 0

	self.cleaned = 0
	self.taptoggled = 0	

	self.state = 0
	self.loop_pos = 0
	self.loop_len = 0
	self.cycle_len = 0
	self.roundmode = 0
	self.quantmode = 0				
	self.mem_total_secs = 0
	self.mem_free_secs = 0	

	curses.noecho()
	curses.cbreak()


	self.h = 25
	self.ph = (self.stdscr.getmaxyx()[0] - 26)
	self.w = self.stdscr.getmaxyx()[1]

	# split it vertically
	self.infowin = self.stdscr.subwin(self.h, self.w / 2, 0, 0)
	self.usewin = self.stdscr.subwin(self.h, int(self.w / 2) - 1, 0, int(self.w / 2) + 1)
	self.paramwin = self.stdscr.subwin(self.ph, self.w, self.h , 0)
	
	self.usewin.box()
	self.paramwin.box()


	self.infowin.addstr(0, 0, 'Events:\n\n')
	self.infowin.move(2, 0)

	# usage
	self.redrawUsage()

	self.paramwin.scrollok(1)
	self.paramwin.idlok(1)	
	self.infowin.scrollok(1)
	self.infowin.idlok(1)	



	# this row has current state
	#  STATE      LoopPos     LoopLen    CycleLen
	
	# make edit boxes in param win

	self.threshrow = 7
	twin = self.paramwin.derwin(1,10,self.threshrow, 45)
	self.threshbox = textpad.Textbox(twin)

	self.dryrow = 8
	dwin = self.paramwin.derwin(1,10,self.dryrow, 45)
	self.drybox = textpad.Textbox(dwin)

	self.wetrow = 9
	win = self.paramwin.derwin(1,10,self.wetrow, 45)
	self.wetbox = textpad.Textbox(win)

	self.feedrow = 10
	win = self.paramwin.derwin(1,10,self.feedrow, 45)
	self.feedbox = textpad.Textbox(win)

	self.raterow = 11
	win = self.paramwin.derwin(1,10, self.raterow, 45)
	self.ratebox = textpad.Textbox(win)


	self.quantrow = 12
	self.roundrow = 13
	self.taptogrow = 14


	self.paramwin.refresh()
	self.usewin.refresh()
	self.infowin.refresh()

	
    def go(self):
	# main event loop
	inittime = time.time()
	
	self.infowin.addstr('%g: SooperLooper started\n' % 0.0)
	self.infowin.refresh()
		
	self.redrawParams()

	self.eca('start')

	while 1:

	    if self.status_tog:
		ch = self.getch(BLOCK_TIME)
	    else:
		# block
		ch = self.getch()

	    if ch == -1:
		# timeout update params
		oldstate = self.state
		self.updateState()

		self.redrawState(1)
		
		if self.state != oldstate:
		    # do full redraw if state changed
		    self.redrawUsage()

		continue
		
	    if ch == ord('q'):
		break
	    elif ch == ord('t'):
		# edit threshold
		self.paramwin.addstr(self.threshrow ,30, 'changing to:')
		self.paramwin.refresh()
		val = self.threshbox.edit()
		try:
		    val = float(val)
		    self.eca('copp-select 1')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)

		except:
		    traceback.print_exc()
		self.redrawParams()

	    elif ch == ord('d'):
		# edit dry
		self.paramwin.addstr(self.dryrow ,30, 'changing to:')
		self.paramwin.refresh()
		val = self.drybox.edit()
		try:
		    val = float(val)
		    self.eca('copp-select 2')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)

		except:
		    traceback.print_exc()
		self.redrawParams()

	    elif ch == ord('w'):
		# edit wet
		self.paramwin.addstr(self.wetrow ,30, 'changing to:')
		self.paramwin.refresh()
		val = self.wetbox.edit()
		try:
		    val = float(val)
		    self.eca('copp-select 3')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)

		except:
		    traceback.print_exc()

		self.redrawParams()
	    elif ch == ord('f'):
		# edit dry
		self.paramwin.addstr(self.feedrow ,30, 'changing to:')
		self.paramwin.refresh()
		val = self.feedbox.edit()
		try:
		    val = float(val)
		    self.eca('copp-select 4')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)


		except:
		    traceback.print_exc()

		self.redrawParams()
	    elif ch == ord('r'):
		# edit dry
		self.paramwin.addstr(self.raterow ,30, 'changing to:')
		self.paramwin.refresh()
		val = self.ratebox.edit()
		try:
		    val = float(val)
		    self.eca('copp-select 5')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)


		except:
		    traceback.print_exc()

		self.redrawParams()

	    elif ch == ord('z'):
		# toggle quantize
		val = not self.quantmode
		
		try:
		    val = float(val)
		    self.eca('copp-select 10')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)
		except:
		    traceback.print_exc()

		self.redrawParams()

	    elif ch == ord('n'):
		# edit quantize
		val = not self.roundmode
		try:
		    val = float(val)
		    self.eca('copp-select 11')
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)
		except:
		    traceback.print_exc()

		self.redrawParams()
	    elif ch == ord('p'):
		# toggle tap tog mode
		val = not self.taptoggled
		try:
		    val = float(val)
		    self.eca('copp-select %d' % REDOTAPMODEPORT)
		    self.eca('copp-set %f' % val)

		    time.sleep(WAIT_TIME)
		    oldtog = self.taptoggled
		    self.updateState()
		    if oldtog != self.taptoggled:
			self.redrawUsage()
		    
		except:
		    traceback.print_exc()

		self.redrawParams()

	    elif ch == ord('s'):
		# toggle status showing
		self.status_tog = not self.status_tog
		self.redrawUsage()
		self.redrawParams()

	    # looking for numbers between 0 and 9
	    num = ch - 48

	    if num >= 0 and num < 10:
		# change value of MULTIPORT
		str = multimap[num]

		if self.state == STATE_MUTE:
		    if num == MULTI_REPLACE:
			str = 'QuantizeMode Toggle'
		    elif num == MULTI_REVERSE:
			str = 'RoundMode Toggle'
		    elif num == MULTI_SCRATCH:
			str = 'Unmute from top'
		    elif num == MULTI_INSERT:
			str = 'Oneshot play'
		    elif num == MULTI_UNDO:
			str = 'Undo All'


		if self.state == STATE_SCRATCH and num == MULTI_REVERSE:
		    str = 'Toggle Rate Active'

		if self.taptoggled and num == MULTI_REDO:
		    str = 'Tap Tempo'
		if self.state == STATE_DELAY and num == MULTI_REPLACE:
		    str = 'Hold Mode Toggle'



		try:
		    self.eca('copp-select %d' % MULTIPORT)

		    self.eca('copp-set %d' % num)
		    # wait a little bit just to make sure it took
		    # then set it out of range
		    time.sleep(WAIT_TIME)

		    self.eca('copp-set -1')

		    time.sleep(WAIT_TIME)

		    
		except eci.ECIError:
		    self._debug_exc()

		self.infowin.addstr('%.1f: %s pressed\n' % (self.loop_pos, str))
		self.infowin.refresh()


		oldstate = self.state

		self.updateState()

		self.redrawParams()

		if self.state != oldstate:
		    self.redrawUsage()

		    
	self.eca('stop')
	

    def updateState(self):
	try:
	    # get new state and outparams
	    self.eca('copp-select %d' % STATEPORT)
	    self.state = int(self.eca('copp-get'))
	    
	    self.eca('copp-select %d' % LOOPPOSPORT)
	    self.loop_pos = self.eca('copp-get')
	    
	    self.eca('copp-select %d' % LOOPLENPORT)
	    self.loop_len = self.eca('copp-get')
	    
	    self.eca('copp-select %d' % CYCLELENPORT)
	    self.cycle_len = self.eca('copp-get')
	    
	    self.eca('copp-select %d' % QUANTMODEPORT)
	    self.quantmode = self.eca('copp-get')
	    self.eca('copp-select %d' % ROUNDMODEPORT)
	    self.roundmode = self.eca('copp-get')

	    self.eca('copp-select %d' % REDOTAPMODEPORT)
	    self.taptoggled = int(self.eca('copp-get'))


	    self.eca('copp-select %d' % SECSTOTALPORT)
	    self.mem_total_secs = self.eca('copp-get')
	    self.eca('copp-select %d' % SECSFREEPORT)
	    self.mem_free_secs = self.eca('copp-get')
	    

	except:
	    traceback.print_exc()

    def redrawUsage(self):

	self.usewin.clear()
	self.usewin.box()
	
	if self.state in (STATE_MUTE, STATE_ONESHOT):
	    self.usewin.addstr(0, 0, 'SooperLooper Demo Usage ')
	    
	    self.usewin.addstr(2, 2,  '5 - Record')
	    self.usewin.addstr(3, 2,  '6 - Overdub')
	    self.usewin.addstr(4, 2,  '7 - Multiply')
	    self.usewin.addstr(5, 2,  '8 - OneShot')
	    self.usewin.addstr(6, 2,  '9 - Unmute continue')
	    self.usewin.addstr(7, 2,  '0 - Undo All')
	    self.usewin.addstr(8, 2,  '1 - Redo/Delay Toggle')
	    self.usewin.addstr(9, 2,  '2 - QuantizeMode Toggle')
	    self.usewin.addstr(10, 2, '3 - RoundMode Toggle')
	    self.usewin.addstr(11, 2, '4 - Unmute from top')		

	else:
	    self.usewin.addstr(0, 0, 'SooperLooper Usage ')
	    
	    self.usewin.addstr(2, 2,  '5 - Record')
	    self.usewin.addstr(3, 2,  '6 - Overdub')
	    self.usewin.addstr(4, 2,  '7 - Multiply')
	    self.usewin.addstr(5, 2,  '8 - Insert')
	    self.usewin.addstr(6, 2,  '9 - Mute')
	    self.usewin.addstr(7, 2,  '0 - Undo')

	    if self.taptoggled:
		self.usewin.addstr(8, 2,  '1 - Tap Tempo Trig')
		if self.state == STATE_DELAY:
		    self.usewin.addstr(9, 2,  '2 - Hold Mode Toggle')
		else:
		    self.usewin.addstr(9, 2,  '2 - Replace')		    
	    else:
		self.usewin.addstr(8, 2,  '1 - Redo')
		self.usewin.addstr(9, 2,  '2 - Replace')

	    if self.state == STATE_SCRATCH:
		self.usewin.addstr(10, 2, '3 - Toggle Rate Active')
	    else:
		self.usewin.addstr(10, 2, '3 - Reverse')		

	    if self.state == STATE_DELAY:
		self.usewin.addstr(11, 2, '4 - No function')		
	    else:
		self.usewin.addstr(11, 2, '4 - ScratchMode')		

	self.usewin.addstr(13, 2, 'q - Quit')
	self.usewin.addstr(14, 2, 't - Modify Threshold')
	self.usewin.addstr(15, 2, 'd - Modify Dry Level')
	self.usewin.addstr(16, 2, 'w - Modify Wet Level')
	self.usewin.addstr(17, 2, 'f - Modify Feedback')	
	self.usewin.addstr(18, 2, 'r - Modify Base Rate in SM')	
	self.usewin.addstr(19, 2, 'z - Toggle QuantizeMode')
	self.usewin.addstr(20, 2, 'n - Toggle RoundMode')		
	self.usewin.addstr(21, 2, 'p - Toggle Redo is TAP')		

	if self.status_tog:
	    self.usewin.addstr(23, 2, 's - Hide Status')		
	else:
	    self.usewin.addstr(23, 2, 's - Show Status (beware xruns)')		
	

	self.usewin.refresh()


    def redrawState(self, refresh=0):
	#  STATE      LoopPos     LoopLen    CycleLen

	if self.status_tog:
	    statestr = statemap.get(self.state, 'Unknown')
	    self.paramwin.addstr(2, 2, 'Status: %-13s' % statestr[:13])
	    
	    self.paramwin.addstr(2, 23,
				 '  LoopPos: %7.2f' % self.loop_pos)
	    self.paramwin.addstr(3, 23,
				 '  LoopLen: %7.2f' % self.loop_len)

	    if self.cycle_len > 0 and self.state not in (STATE_RECORD, STATE_TRIG_STOP):
		self.paramwin.addstr(2, 43,
				     '  CyclePos: %7d' % ( int(self.loop_pos / self.cycle_len) + 1))

	    self.paramwin.addstr(3, 43,
				 '  CycleLen: %7.2f' % self.cycle_len)
	    
	    # loop mem
	    self.paramwin.addstr(4, 23,
				 '  FreeTime:  %7.2f' % self.mem_free_secs)

	    self.paramwin.addstr(4, 43,
				 '  TotalTime: %7.2f' % self.mem_total_secs)
	    

	    
	    if refresh:
		self.paramwin.refresh()

    def redrawParams(self):
	self.paramwin.clear()
	self.paramwin.box()

	self.redrawState()

	self.paramwin.addstr(5, 2, 'Basic Parameters ')

	try:
	
	    self.eca('copp-select 1')
	    val = self.eca('copp-get')	
	    self.paramwin.addstr(self.threshrow ,2, 'Threshold    =  %g' % val) 

	    self.eca('copp-select 2')
	    val = self.eca('copp-get')	
	    self.paramwin.addstr(self.dryrow ,2,    'Dry Level    =  %g' % val) 

	    self.eca('copp-select 3')
	    val = self.eca('copp-get')	
	    self.paramwin.addstr(self.wetrow ,2,    'Wet Level    =  %g' % val) 

	    self.eca('copp-select 4')
	    val = self.eca('copp-get')	
	    self.paramwin.addstr(self.feedrow ,2,   'Feedback     =  %g' % val) 

	    self.eca('copp-select 5')
	    val = self.eca('copp-get')	
	    self.paramwin.addstr(self.raterow ,2,   'Rate in SM   =  %g' % val) 

	    self.eca('copp-select 10')
	    self.quantmode = int(self.eca('copp-get'))
	    if self.quantmode:
		self.paramwin.addstr(self.quantrow ,2,  'QuantizeMode =  On') 
	    else:
		self.paramwin.addstr(self.quantrow ,2,  'QuantizeMode =  Off') 
		
		
	    self.eca('copp-select 11')
	    self.roundmode = int(self.eca('copp-get'))
	    if self.roundmode:
		self.paramwin.addstr(self.roundrow ,2,  'RoundMode    =  On') 
	    else:
		self.paramwin.addstr(self.roundrow ,2,  'RoundMode    =  Off') 

	    self.eca('copp-select 12')
	    self.taptoggled = int(self.eca('copp-get'))
	    if self.taptoggled:
		self.paramwin.addstr(self.taptogrow ,2, 'Redo is Tap  =  On') 
	    else:
		self.paramwin.addstr(self.taptogrow ,2, 'Redo is Tap  =  Off') 

		
	    self.paramwin.refresh()
	    self.infowin.refresh()

	except:
	    traceback.print_exc()
	    
		    
    def getch(self, timeout=-1):
	"""Returns a single keypress.  If noblock is true, it might return
	0 signifying no data available."""

	self.stdscr.timeout(timeout)

	try:
	    ret = self.stdscr.getch()
	except:
	    traceback.print_exc()
	    ret = -1

	return ret


    def clear(self, refresh=1):
	self.stdscr.clear()

	if refresh:
	    self.stdscr.refresh()


    def debug(self, str):
	sys.stderr.write(str)
	

    def _debug_exc(self):
	self.debug(string.join(traceback.format_exception(sys.exc_type,
								 sys.exc_value,
								 sys.exc_traceback)) + '\n')
	
def cleanup():
    curses.nocbreak()
    curses.echo()
    curses.endwin()



def main():

    try:
	slt = sltester()
	
	slt.go()
	cleanup()

    except SystemExit:
	cleanup()
    except:
	traceback.print_exc()
	cleanup()

if __name__=="__main__":
    main()
