#include <stdlib.h>
#include <string.h>
#include <pcre.h>
#include <pcre++.h>
#include <vector>
#include <string>

/*
 * C++ wrapper dla biblioteki PCRE
 *
 * Mikolaj Rydzewski, miki@ceti.pl
 *
 * $Id: mr_pcre.h,v 1.5 2004/08/16 22:47:46 miki Exp $
 *
 * Zmiany:
 *	17.8.2004
 *	dodanie wsparcia dla biblioteki pcre++, wycofanie sie z starej klasy
 *
 * 	11.3.2004
 * 	import do CVS
 */

class PcreWrap {
public:
	typedef std::vector<std::vector<std::string> > Array;
	
	static bool pcre_match_all(
				const std::string &regex,
				const std::string &buffer,
				const std::string &options, 
				PcreWrap::Array &matches
			) {
		pcrepp::Pcre pcre(regex, options);
		int pos = 0;
		bool found = false;
		
		matches.clear();
		
		while (pcre.search(buffer, pos)) {
			found = true;
			std::vector<std::string> *substrings = pcre.get_sub_strings();
			if (substrings) matches.push_back(*substrings);
			pos = pcre.get_match_end() + 1;
		}
		
		return found;
	}
	
	static bool pcre_match(
				const std::string &regex,
				const std::string &buffer,
				const std::string &options 
			) {
		pcrepp::Pcre pcre(regex, options);
		
		return pcre.search(buffer);
	}
};
 
 
#define PCRE_NUM_ELEMS 10

class PCRE {
	pcre *re;
	const char *error;
	int erroroffset, *ovector, vector_size;

public:
	PCRE(int options=0, int size=PCRE_NUM_ELEMS) {
		ovector = NULL;
		re = NULL;
		error = NULL;
		ReallocVector(size);
	}
	PCRE(const char *pattern, int options=0, int size=PCRE_NUM_ELEMS) {
		ovector = NULL;
		re = NULL;
		ReallocVector(size);
		Compile(pattern, options);
	}
	~PCRE() {
		if (ovector) delete ovector;
		if (re) pcre_free(re);
	}

	int ReallocVector(int size=PCRE_NUM_ELEMS) {
		if (ovector) delete ovector;
		vector_size = size*3;
		ovector = new int[vector_size];
		return (ovector != NULL);
	}

	int VectorSize() { return vector_size; };
	
	const char *Error() { return error; };
	
	int ErrorOffset() { return erroroffset; };
	
	int Compile(const char *pattern, int options=0) {
		error = NULL;
		re = pcre_compile(pattern, options, &error, &erroroffset, NULL);
		return re != NULL;
	}

	int Matches(const char *subject) {
		return Exec(subject) >= 0;
	}
	
	int Exec(const char *subject, int subject_length=0, int start_offset=0,
			int options=0) {
		if (!subject_length) subject_length = strlen(subject);
		return pcre_exec(re, NULL, subject, subject_length,
				start_offset, options, ovector, vector_size);
	}

	int ExecNext(const char *subject, int subject_length=0) {
		for(;;) {
			int rc, options = 0, start_offset = ovector[1];

			if (ovector[0] == ovector[1]) {
				if (ovector[0] == subject_length)
					return PCRE_ERROR_NOMATCH;
				options = PCRE_NOTEMPTY | PCRE_ANCHORED;
			}

			rc = Exec(subject, subject_length, start_offset, options);
			if (rc == PCRE_ERROR_NOMATCH) {
				if (options == 0) return PCRE_ERROR_NOMATCH;
				ovector[1] = start_offset + 1;
				continue;
			}

			if (rc < 0) {
				return rc;
			}

			return rc;
		}
	}

	int operator[](int offset) {
		return (offset >=0 && offset < vector_size)
			? ovector[offset] : -1;
	}

	int Start(int offset) { return ovector[2*offset]; };

	int Length(int offset) { return ovector[2*offset+1] - ovector[2*offset]; };

};

#ifdef __test
#include <iostream>
int main(int argc, char **argv) {
	char pattern[80], buf[80];

	cout << "Podaj wyrazenie: ";
	cin >> pattern;
	PCRE regexp(pattern);
	for(;;) {
		cout << "Podaj lancuch ('q' konczy): ";
		cin >> buf;
		if (buf[0]=='q') break;
		int ret = regexp.Exec(buf);
		if (ret < 0) cout << "Cos sie nie udalo\n";
		else {
			cout << "Udalo sie, znaleziono " << ret << " miejsc\n";
			for(int i=0; i<ret; i++) {
				cout << i << ": " << regexp[2*i] << 
					", " << regexp[2*i+1]-regexp[2*i] << endl;
			}

			while ((ret = regexp.ExecNext(buf)) >= 0) {
				cout << "kolejne znalezisko\n";
				for(i=0; i<ret; i++) {
					cout << i << ": " << regexp[2*i] << 
					", " << regexp[2*i+1]-regexp[2*i] << endl;
				}

			}
		}
	}

	return 0;
}
#endif
