/*
 * Copyright (c) 1998, 1999, 2000, 2001
 *	Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * The SIP parser.
 */

%{
#include <stdlib.h>
#include <string.h>

#include "sip.h"


#define	MAX_NESTED_IF		10
#define	MAX_NESTED_SCOPE	10

#define	isMainModule(m)		(currentSpec -> module == (m))


/* The different attribute name types. */

typedef enum {
	func_attr,
	var_attr,
	enum_attr,
	class_attr
} attrType;


/* Table of Python methods that can go in the PyMethods section. */

static specialPyMethod pyMethods[] = {
	{"__repr__",		SPM_METH_REPR,0,	"PyObject *",	1},
	{"__str__",		SPM_METH_STR,0,		"PyObject *",	1},
	{"__cmp__",		SPM_METH_CMP,0,		"int",		2},
	{"__hash__",		SPM_METH_HASH,0,	"long",		1},
	{"__call__",		SPM_METH_CALL,0,	"PyObject *",	3},
	{"__richcompare__",	SPM_METH_RICH,0,	"PyObject *",	-1, 0x02010000},
	{NULL}
};


/* Table of Python methods that can go in the PyNumberMethods section. */

static specialPyMethod pyNumMethods[] = {
	{"__add__",		SPM_NUM_ADD,0,		"PyObject *",	2},
	{"__sub__",		SPM_NUM_SUB,0,		"PyObject *",	2},
	{"__mul__",		SPM_NUM_MUL,0,		"PyObject *",	2},
	{"__div__",		SPM_NUM_DIV,0,		"PyObject *",	2},
	{"__mod__",		SPM_NUM_MOD,0,		"PyObject *",	2},
	{"__divmod__",		SPM_NUM_DIVMOD,0,	"PyObject *",	2},
	{"__pow__",		SPM_NUM_POW,0,		"PyObject *",	3},
	{"__neg__",		SPM_NUM_NEG,0,		"PyObject *",	1},
	{"__pos__",		SPM_NUM_POS,0,		"PyObject *",	1},
	{"__abs__",		SPM_NUM_ABS,0,		"PyObject *",	1},
	{"__nonzero__",		SPM_NUM_NONZERO,0,	"int",		1},
	{"__invert__",		SPM_NUM_INVERT,0,	"PyObject *",	1},
	{"__lshift__",		SPM_NUM_LSHIFT,0,	"PyObject *",	2},
	{"__rshift__",		SPM_NUM_RSHIFT,0,	"PyObject *",	2},
	{"__and__",		SPM_NUM_AND,0,		"PyObject *",	2},
	{"__xor__",		SPM_NUM_XOR,0,		"PyObject *",	2},
	{"__or__",		SPM_NUM_OR,0,		"PyObject *",	2},
	{"__coerce__",		SPM_NUM_COERCE,0,	"int",		-1},
	{"__int__",		SPM_NUM_INT,0,		"PyObject *",	1},
	{"__long__",		SPM_NUM_LONG,0,		"PyObject *",	1},
	{"__float__",		SPM_NUM_FLOAT,0,	"PyObject *",	1},
	{"__oct__",		SPM_NUM_OCT,0,		"PyObject *",	1},
	{"__hex__",		SPM_NUM_HEX,0,		"PyObject *",	1},
	{"__iadd__",		SPM_NUM_IADD,0,		"PyObject *",	2, 0x02000000},
	{"__isub__",		SPM_NUM_ISUB,0,		"PyObject *",	2, 0x02000000},
	{"__imul__",		SPM_NUM_IMUL,0,		"PyObject *",	2, 0x02000000},
	{"__idiv__",		SPM_NUM_IDIV,0,		"PyObject *",	2, 0x02000000},
	{"__imod__",		SPM_NUM_IMOD,0,		"PyObject *",	2, 0x02000000},
	{"__ipow__",		SPM_NUM_IPOW,0,		"PyObject *",	3, 0x02000000},
	{"__ilshift__",		SPM_NUM_ILSHIFT,0,	"PyObject *",	2, 0x02000000},
	{"__irshift__",		SPM_NUM_IRSHIFT,0,	"PyObject *",	2, 0x02000000},
	{"__iand__",		SPM_NUM_IAND,0,		"PyObject *",	2, 0x02000000},
	{"__ixor__",		0,SPM_NUM_IXOR,		"PyObject *",	2, 0x02000000},
	{"__ior__",		0,SPM_NUM_IOR,		"PyObject *",	2, 0x02000000},
	{NULL}
};


/* Table of Python methods that can go in the PySequenceMethods section. */

static specialPyMethod pySeqMethods[] = {
	{"__len__",		SPM_SEQ_LEN,0,		"int",		1},
	{"__add__",		SPM_SEQ_CONCAT,0,	"PyObject *",	2},
	{"__mul__",		SPM_SEQ_REPEAT,0,	"PyObject *",	-1},
	{"__getitem__",		SPM_SEQ_GETITEM,0,	"PyObject *",	-1},
	{"__setitem__",		SPM_SEQ_SETITEM,0,	"int",		-1},
	{"__getslice__",	SPM_SEQ_GETSLICE,0,	"PyObject *",	-1},
	{"__setslice__",	SPM_SEQ_SETSLICE,0,	"int",		-1},
	{"__contains__",	SPM_SEQ_CONTAINS,0,	"int",		2, 0x01060000},
	{"__iadd__",		SPM_SEQ_ICONCAT,0,	"PyObject *",	2, 0x02000000},
	{"__imul__",		SPM_SEQ_IREPEAT,0,	"PyObject *",	-1, 0x02000000},
	{NULL}
};


/* Table of Python methods that can go in the PyMappingMethods section. */

static specialPyMethod pyMapMethods[] = {
	{"__len__",		SPM_MAP_LEN,0,		"int",		1},
	{"__getitem__",		SPM_MAP_GETITEM,0,	"PyObject *",	2},
	{"__setitem__",		SPM_MAP_SETITEM,0,	"int",		3},
	{NULL}
};


static sipSpec *currentSpec;		/* The current spec being parsed. */
static stringList *timeSlotList;	/* The list of required primaries. */
static moduleDef *currentModule;	/* The current module being parsed. */
static mappedTypeDef *currentMappedType;	/* The current mapped type. */
static int sectionFlags;		/* The current section flags. */
static int currentOverIsVirt;		/* Set if the overload is virtual. */
static int currentIsStatic;		/* Set if the current is static. */
static int importDepth;			/* The nr. of nested imports. */
static char *previousFile;		/* The file just parsed. */
static parserContext newContext;	/* The new pending context. */
static int currentVersion;		/* The index of the version. */
static versionRange versScope[MAX_NESTED_IF];	/* Current version nest. */
static classVersDef *scopeStack[MAX_NESTED_SCOPE];	/* The scope stack. */
static int sectFlagsStack[MAX_NESTED_SCOPE];	/* The section flags stack. */
static int currentScopeIdx;		/* The scope stack index. */


static nameDef *cacheName(sipSpec *,moduleDef *,char *);
static versionDef *getVersion(void);
static void checkIfaceFile(sipSpec *,ifaceFileDef *,versionDef *,argDef *);
static ifaceFileDef *findIfaceFile(sipSpec *,scopedNameDef *);
static classDef *findClass(sipSpec *,scopedNameDef *);
static classVersDef *newClass(sipSpec *,versionDef *,scopedNameDef *);
static classVersDef *findNamespace(sipSpec *,versionDef *,scopedNameDef *);
static void finishClass(classVersDef *);
static mappedTypeDef *newMappedType(sipSpec *,argDef *,versionDef *);
static void newEnum(sipSpec *,versionDef *,moduleDef *,char *,int,enumValueDef *);
static void newTypedef(sipSpec *,versionDef *,char *,argDef *);
static void newVar(sipSpec *,versionDef *,moduleDef *,char *,int,argDef *,codeBlock *);
static void newSpecialMethod(versionDef *,char *,int,codeBlock *);
static void newCtor(versionDef *,char *,int,funcArgs *,codeBlock *);
static void newFunction(sipSpec *,versionDef *,moduleDef *,int,int,int,
			argDef *,char *,funcArgs *,int,int,optFlags *,
			codeBlock *,codeBlock *);
static optFlag *findOptFlag(optFlags *,char *,flagType);
static memberDef *findFunction(sipSpec *,versionDef *,moduleDef *,classVersDef *,char *);
static void checkAttributes(sipSpec *,classVersDef *,nameDef *,versionDef *,attrType);
static void newModule(FILE *,char *,int);
static void appendCodeBlock(codeBlock **,codeBlock *);
static void appendToClassList(classList **,classDef *);
static void parseFile(FILE *,char *,moduleDef *,int);
static void handleEOF(void);
static void handleEOM(void);
static versionQual *findQualifier(versionQual *,char *);
static scopedNameDef *text2scopedName(char *);
static void pushScope(classVersDef *);
static void popScope(void);
static classVersDef *currentScope(void);
static versionQual *newVersion(sipSpec *,moduleDef *,int,char *,char *,char *);
static void newImport(sipSpec *,char *,int);
%}

%union {
	char		qchar;
	char		*text;
	long		number;
	double		real;
	argDef		memArg;
	funcArgs	arglist;
	codeBlock	*codeb;
	valueDef	value;
	valueDef	*valp;
	optFlags	optflags;
	optFlag		flag;
	scopedNameDef	*scpvalp;
	enumValueDef	*evalp;
	fcallDef	fcall;
	versionRange	versrange;
}

%token			TK_DOC
%token			TK_EXPORTEDDOC
%token			TK_MAKEFILE
%token			TK_VARCODE
%token			TK_CPPCODE
%token			TK_PREPYCODE
%token			TK_PYCODE
%token 			TK_COPYING
%token			TK_MAPPEDTYPE
%token <codeb>		TK_CODEBLOCK
%token			TK_IF
%token			TK_END
%token <text>		TK_NAME
%token <text>		TK_PYMETHODNAME
%token <text>		TK_FILENAME
%token <text>		TK_STRING
%token			TK_VIRTUALCODE
%token			TK_MEMBERCODE
%token			TK_FROMTYPE
%token			TK_CANTOTYPE
%token			TK_TOTYPE
%token			TK_TOSUBCLASS
%token			TK_INCLUDE
%token			TK_IMPORT
%token			TK_IMPORTTL
%token			TK_VERSIONCODE
%token			TK_HEADERCODE
%token			TK_EXPHEADERCODE
%token			TK_MODULE
%token			TK_CLASS
%token			TK_STRUCT
%token			TK_PUBLIC
%token			TK_PROTECTED
%token			TK_PRIVATE
%token			TK_SIGNALS
%token			TK_SLOTS
%token			TK_PYMETHODS
%token			TK_PYNUMMETHODS
%token			TK_PYSEQMETHODS
%token			TK_PYMAPMETHODS
%token			TK_BOOL
%token			TK_SHORT
%token			TK_INT
%token			TK_LONG
%token			TK_FLOAT
%token			TK_DOUBLE
%token			TK_CHAR
%token			TK_VOID
%token			TK_VIRTUAL
%token			TK_ENUM
%token			TK_UNSIGNED
%token			TK_SCOPE
%token			TK_CONST
%token			TK_STATIC
%token			TK_SIPSIGNAL
%token			TK_SIPRXCON
%token			TK_SIPRXDIS
%token			TK_SIPSLOTCON
%token			TK_SIPSLOTDIS
%token			TK_SIPARRAY
%token			TK_SIPUARRAY
%token			TK_SIPARRAYSIZE
%token			TK_SIPARRAYUSIZE
%token <number>		TK_NUMBER
%token <real>		TK_REAL
%token			TK_TYPEDEF
%token			TK_NAMESPACE
%token			TK_TIMELINE
%token			TK_SECVERSION
%token			TK_EXPOSE
%token <qchar>		TK_QCHAR

%type <memArg>		argvalue
%type <memArg>		argtype
%type <memArg>		cpptype
%type <memArg>		basetype
%type <arglist>		arglist
%type <arglist>		rawarglist
%type <arglist>		basetypelist
%type <number>		optslot
%type <number>		optref
%type <number>		optconst
%type <number>		optabstract
%type <number>		deref
%type <value>		simplevalue
%type <valp>		value
%type <valp>		expr
%type <valp>		optassign
%type <codeb>		optvarcode
%type <codeb>		hdrcode
%type <codeb>		cppcode
%type <codeb>		codeblock
%type <codeb>		optvirtualcode
%type <codeb>		membercode
%type <text>		optname
%type <optflags>	optflags
%type <optflags>	flaglist
%type <flag>		flag
%type <flag>		flagvalue
%type <qchar>		optunop
%type <qchar>		binop
%type <scpvalp>		scopepart
%type <scpvalp>		scopedname
%type <evalp>		enumvalue
%type <evalp>		enumvalues
%type <evalp>		optenumvalues
%type <fcall>		valuelist
%type <versrange>	versionrange

%%

specification: 	statement
	|	specification statement
	;

statement:	{
			/*
			 * We don't do these in parserEOF() because the parser
			 * is reading ahead and that would be too early.
			 */

			if (previousFile != NULL)
			{
				handleEOF();

				if (newContext.prevmod != NULL)
					handleEOM();

				free(previousFile);
				previousFile = NULL;
			}
	} modstatement
	;

modstatement:	module
	|	copying
	|	include
	|	import
	|	importtl
	|	timeline
	|	secversion
	|	expose
	|	versioncode
	|	hdrcode {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> hdrcode,$1);
		}
	|	exphdrcode
	|	cppcode {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> cppcode,$1);
		}
	|	prepycode
	|	pycode
	|	doc
	|	exporteddoc
	|	makefile
	|	mappedtype
	|	nsstatement
	;

nsstatement:	ifstart
	|	ifend
	|	namespace
	|	class
	|	opaqueclass
	|	typedef
	|	enum
	|	function
	|	variable
	;

mappedtype:	TK_MAPPEDTYPE basetype {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
				currentMappedType = newMappedType(currentSpec,&$2,vd);
		} '{' mtbody '}' ';' {
			if (getVersion() != NULL)
			{
				if (currentMappedType -> convfromcode == NULL)
					yyerror("%MappedType must have a %ConvertFromTypeCode directive");

				if (currentMappedType -> convtocode == NULL)
					yyerror("%MappedType must have a %ConvertToTypeCode directive");

				if (currentMappedType -> canconvtocode == NULL)
					yyerror("%MappedType must have a %CanConvertToTypeCode directive");

				currentMappedType = NULL;
			}
		}
	;

mtbody:		mtline
	|	mtbody mtline
	;

mtline: 	hdrcode {
			if (getVersion() != NULL)
				appendCodeBlock(&currentMappedType -> hdrcode,$1);
		}
	|	TK_FROMTYPE codeblock {
			if (getVersion() != NULL)
			{
				if (currentMappedType -> convfromcode != NULL)
					yyerror("%MappedType has more than one %ConvertFromTypeCode directive");

				currentMappedType -> convfromcode = $2;
			}
		}
	|	TK_CANTOTYPE codeblock {
			if (getVersion() != NULL)
			{
				if (currentMappedType -> canconvtocode != NULL)
					yyerror("%MappedType has more than one %CanConvertToTypeCode directive");

				currentMappedType -> canconvtocode = $2;
			}
		}
	|	TK_TOTYPE codeblock {
			if (getVersion() != NULL)
			{
				if (currentMappedType -> convtocode != NULL)
					yyerror("%MappedType has more than one %ConvertToTypeCode directive");

				currentMappedType -> convtocode = $2;
			}
		}
	;

namespace:	TK_NAMESPACE TK_NAME {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				classVersDef *ns;
				scopedNameDef *fqname;

				fqname = text2scopedName($2);

				if ((ns = findNamespace(currentSpec,vd,fqname)) == NULL)
				{
					ns = newClass(currentSpec,vd,fqname);
			
					setCannotCreate(ns);
					setIsNamespace(ns);
				}

				sectionFlags = 0;

				pushScope(ns);
			}
		} '{' nsbody '}' ';' {
			/* Remove the namespace name. */

			if (getVersion() != NULL)
				popScope();
		}
	;

nsbody:		nsstatement
	|	nsbody nsstatement
	;

secversion:	TK_SECVERSION TK_NAME TK_STRING TK_STRING {
			newVersion(currentSpec,currentModule,-1,$2,$3,$4);
		}
	;

timeline:	TK_TIMELINE {
			if (currentModule -> nrTimeSlots >= 0)
				yyerror("%Timeline has already been defined for this module");

			currentModule -> nrTimeSlots = 0;
		}
		'{' versionlist '}' {
			/*
			 * If we don't know the primary version yet, use the
			 * latest one.
			 */

			if (currentModule -> required == NULL)
			{
				versionQual *vq;

				for (vq = currentModule -> timeSlots; vq != NULL; vq = vq -> next)
					if (vq -> order == currentModule -> nrTimeSlots - 1)
					{
						currentModule -> required = vq;
						break;
					}
			}
		}
	;

versionlist:	versionname
	|	versionlist versionname
	;

versionname:	TK_NAME {
			versionQual *vq;
			stringList *sl;

			vq = newVersion(currentSpec,currentModule,currentModule -> nrTimeSlots++,$1,NULL,NULL);

			/* See if this is a time slot we want. */

			for (sl = timeSlotList; sl != NULL; sl = sl -> next)
				if (strcmp(vq -> name,sl -> s) == 0)
				{
					if (currentModule -> required != NULL)
						fatal("Multiple primary versions requested for the same module\n");

					currentModule -> required = vq;
				}
		}
	;

ifstart:	TK_IF '(' versionrange ')' {
			if (currentVersion >= MAX_NESTED_IF)
				yyerror("Internal error: increase the value of MAX_NESTED_IF");

			versScope[currentVersion++] = $3;
		}
	;

versionrange:	TK_NAME {
			int i;

			if (($$.secondary = findQualifier(currentSpec -> secondaries,$1)) == NULL)
				yyerror("No such secondary version");

			/*
			 * Check there isn't already a secondary in scope and
			 * set the primary.
			 */

			$$.lowidx = 0;
			$$.uppidx = currentModule -> nrTimeSlots;

			for (i = 0; i < currentVersion; ++i)
				if (versScope[i].secondary != NULL)
					yyerror("Cannot nest secondary versions");
				else
				{
					/*
					 * Keep overwriting with more and more
					 * specific primaries.
					 */

					$$.lowidx = versScope[i].lowidx;
					$$.uppidx = versScope[i].uppidx;
				}
		}
	|	optname '-' optname {
			versionQual *vq;

			if ($1 == NULL)
				$$.lowidx = (currentVersion > 0 ?
					versScope[currentVersion - 1].lowidx :
					0);
			else if ((vq = findQualifier(currentModule -> timeSlots,$1)) == NULL)
				yyerror("Lower bound is not a primary version in this module");
			else
				$$.lowidx = vq -> order;

			if ($3 == NULL)
				$$.uppidx = (currentVersion > 0 ?
					versScope[currentVersion - 1].uppidx :
					currentModule -> nrTimeSlots);
			else if ((vq = findQualifier(currentModule -> timeSlots,$3)) == NULL)
				yyerror("Upper bound is not a primary version in this module");
			else
				$$.uppidx = vq -> order;

			/* Sanity checks on the bounds. */

			if ($$.lowidx == $$.uppidx)
				yyerror("Lower and upper bounds must be different");

			if ($$.lowidx > $$.uppidx)
				yyerror("Later version specified as lower bound");

			/* Set the secondary. */

			$$.secondary = (currentVersion > 0 ?
				versScope[currentVersion - 1].secondary :
				NULL);
		}
	;

ifend:		TK_END {
			if (currentVersion-- <= 0)
				yyerror("Too many %End directives");
		}
	;

expose:		TK_EXPOSE TK_NAME {
			/* Only bother if this is the main module. */

			if (isMainModule(currentModule))
			{
				expFuncDef *ef;

				/* Check it isn't already there. */

				for (ef = currentSpec -> exposed; ef != NULL; ef = ef -> next)
					if (strcmp(ef -> name,$2) == 0)
						yyerror("Function has already been exposed in this module");

				ef = sipMalloc(sizeof (expFuncDef));

				ef -> name = $2;
				ef -> next = currentSpec -> exposed;

				currentSpec -> exposed = ef;
			}
		}
	;

module:		TK_MODULE TK_NAME {
			/* Check the module hasn't already been defined. */

			moduleDef *mod;

			for (mod = currentSpec -> moduleList; mod != NULL; mod = mod -> next)
				if (mod -> name != NULL && strcmp(mod -> name -> text,$2) == 0)
					yyerror("Module is already defined");

			currentModule -> name = cacheName(currentSpec,currentModule,$2);
		}
	;

include:	TK_INCLUDE TK_FILENAME {
			parseFile(NULL,$2,NULL,FALSE);
		}
	;

import:		TK_IMPORT TK_FILENAME {
			newImport(currentSpec,$2,FALSE);
		}
	;

importtl:	TK_IMPORTTL TK_FILENAME {
			if (currentModule -> timeSlots != NULL)
				yyerror("Cannot %ImportWithTimeline after a primary version has been defined");

			newImport(currentSpec,$2,TRUE);
		}
	;

optvarcode:	{
			$$ = NULL;
		}
	|	TK_VARCODE codeblock {
			$$ = $2;
		}
	;

versioncode:	TK_VERSIONCODE codeblock {
			appendCodeBlock(&currentSpec -> versioncode,$2);
		}
	;

copying:	TK_COPYING codeblock {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> copying,$2);
		}
	;

hdrcode:	TK_HEADERCODE codeblock {
			$$ = $2;
		}
	;

exphdrcode:	TK_EXPHEADERCODE codeblock {
			appendCodeBlock(&currentSpec -> exphdrcode,$2);
		}
	;

cppcode:	TK_CPPCODE codeblock {
			$$ = $2;
		}
	;

prepycode:	TK_PREPYCODE codeblock {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> prepycode,$2);
		}
	;

pycode:		TK_PYCODE codeblock {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> pycode,$2);
		}
	;

doc:		TK_DOC codeblock {
			if (isMainModule(currentModule))
				appendCodeBlock(&currentSpec -> docs,$2);
		}
	;

exporteddoc:	TK_EXPORTEDDOC codeblock {
			appendCodeBlock(&currentSpec -> docs,$2);
		}
	;

makefile:	TK_MAKEFILE TK_FILENAME TK_FILENAME codeblock {
			if (isMainModule(currentModule))
			{
				mkTemplateDef *mtd;

				mtd = sipMalloc(sizeof (mkTemplateDef));

				mtd -> name = $2;
				mtd -> objext = $3;
				mtd -> templ = NULL;

				appendCodeBlock(&mtd -> templ,$4);

				mtd -> next = currentSpec -> mktemplates;
				currentSpec -> mktemplates = mtd;
			}
		}
	;

codeblock:	TK_CODEBLOCK TK_END
	;

enum:		TK_ENUM optname '{' optenumvalues '}' ';' {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				if (sectionFlags != 0 && (sectionFlags & ~(SECT_IS_PUBLIC | SECT_IS_PROT)) != 0)
					yyerror("Class enums must be in the public or protected sections");

				newEnum(currentSpec,vd,currentModule,$2,sectionFlags,$4);
			}
		}
	;

optname:	{
			$$ = NULL;
		}
	|	TK_NAME {
			$$ = $1;
		}
	;

optenumvalues:	{
			$$ = NULL;
		}
	|	enumvalues {
			$$ = $1;
		}
	;

enumvalues:	enumvalue
	|	enumvalues ',' enumvalue {
			enumValueDef *evd;

			/*
			 * Append the new value to the list so long as it
			 * isn't already there.
			 */

			for (evd = $1; evd -> name != $3 -> name && evd -> next != NULL; evd = evd -> next)
				;

			if (evd -> name == $3 -> name)
				yyerror("enum values given more that once in the same enum");

			evd -> next = $3;
		}
	;

enumvalue:	TK_NAME optassign {
			/*
			 * Note that we don't use the assigned value.  This is
			 * a hangover from when enums where generated in
			 * Python.  We can remove it when we have got around to
			 * updating all the .sip files.
			 */

			$$ = sipMalloc(sizeof (enumValueDef));
			$$ -> name = cacheName(currentSpec,currentModule,$1);
			$$ -> next = NULL;
		}
	;

optassign:	{
			$$ = NULL;
		}
	|	'=' expr {
			$$ = $2;
		}
	;

expr:		value
	|	expr binop value {
			valueDef *vd;
 
			if ($1 -> vtype == string_value || $3 -> vtype == string_value)
				yyerror("Invalid binary operator for string");
 
			/* Find the last value in the existing expression. */
 
			for (vd = $1; vd -> next != NULL; vd = vd -> next)
				;
 
			vd -> vbinop = $2;
			vd -> next = $3;

			$$ = $1;
		}
	;

binop:		'-' {
			$$ = '-';
		}
	|	'+' {
			$$ = '+';
		}
	|	'|' {
			$$ = '|';
		}
	;

optunop:	{
			$$ = '\0';
		}
	|	'!' {
			$$ = '!';
		}
	|	'~' {
			$$ = '~';
		}
	|	'-' {
			$$ = '-';
		}
	|	'+' {
			$$ = '+';
		}
	;

value:		optunop simplevalue {
			if ($1 != '\0' && $2.vtype == string_value)
				yyerror("Invalid unary operator for string");
 
			/*
			 * Convert the value to a simple expression on the
			 * heap.
			 */
 
			$$ = sipMalloc(sizeof (valueDef));
 
			*$$ = $2;
			$$ -> vunop = $1;
			$$ -> vbinop = '\0';
			$$ -> next = NULL;
		}
	;

scopedname:	scopepart
	|	scopedname TK_SCOPE scopepart {
			appendScopedName(&$1,$3);
		}
	;

scopepart: 	TK_NAME {
			$$ = text2scopePart($1);
		}
	;

simplevalue:	scopedname {
			/*
			 * We let the C++ compiler decide if the value is a
			 * valid one - no point in building a full C++ parser
			 * here.
			 */

			$$.vtype = scoped_value;
			$$.u.vscp = $1;
		}
	|	scopedname '(' valuelist ')' {
			fcallDef *fcd;

			fcd = sipMalloc(sizeof (fcallDef));
			*fcd = $3;
			fcd -> name = $1;

			$$.vtype = fcall_value;
			$$.u.fcd = fcd;
		}
	|	TK_REAL {
			$$.vtype = real_value;
			$$.u.vreal = $1;
		}
	|	TK_NUMBER {
			$$.vtype = numeric_value;
			$$.u.vnum = $1;
		}
	|	TK_STRING {
			$$.vtype = string_value;
			$$.u.vstr = cacheName(currentSpec,currentModule,$1);
		}
	|	TK_QCHAR {
			$$.vtype = qchar_value;
			$$.u.vqchar = $1;
		}
	;

valuelist:	{
			/* No values. */

			$$.nrArgs = 0;
		}
	|	value {
			/* The single or first value. */

			$$.args[0] = $1;
			$$.nrArgs = 1;
		}
	|	valuelist ',' value {
			/* Check that it wasn't ...(,value...). */

			if ($$.nrArgs == 0)
				yyerror("First argument to function call is missing");

			/* Check there is room. */

			if ($1.nrArgs == MAX_NR_ARGS)
				yyerror("Too many arguments to function call");

			$$ = $1;

			$$.args[$$.nrArgs] = $3;
			$$.nrArgs++;
		}
	;

typedef:	TK_TYPEDEF cpptype TK_NAME ';' {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
				newTypedef(currentSpec,vd,$3,&$2);
		}
	;

opaqueclass:	TK_CLASS scopedname ';' {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				classVersDef *cvd;

				cvd = newClass(currentSpec,vd,$2);

				setHasRealClassVers(cvd -> common -> iff);
				setCannotCreate(cvd);
				setIsOpaque(cvd);
			}
		}
	;

class:		TK_CLASS TK_NAME {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				classVersDef *cvd;

				cvd = newClass(currentSpec,vd,text2scopedName($2));

				setHasRealClassVers(cvd -> common -> iff);
				setCannotCreate(cvd);

				sectionFlags = SECT_IS_PRIVATE;

				pushScope(cvd);
			}
		} superclasses '{' classbody '}' ';' {
			if (getVersion() != NULL)
			{
				finishClass(currentScope());
				popScope();
			}
		}
	;

superclasses:
	|	':' superlist
	;

superlist:	superclass
	|	superlist ',' superclass
	;

superclass:	scopedname {
			if (getVersion() != NULL)
			{
				classDef *cd = findClass(currentSpec,$1);
				classVersDef *cvd = currentScope();

				appendToClassList(&cvd -> supers,cd);

				if (isMainModule(cvd -> common -> iff -> module))
					addToUsedList(&cvd -> used,cd -> iff);
			}
		}
	;

classbody:	classline
	|	classbody classline
	;

classline:	ifstart
	|	ifend
	|	namespace
	|	class
	|	opaqueclass
	|	typedef
	|	enum
	|	cppcode {
			if (getVersion() != NULL)
				appendCodeBlock(&currentScope() -> cppcode,$1);
		}
	|	hdrcode {
			if (getVersion() != NULL)
				appendCodeBlock(&currentScope() -> hdrcode,$1);
		}
	|	ctor
	|	dtor
	|	varmember
	|	TK_TOSUBCLASS codeblock {
			if (getVersion() != NULL)
			{
				classVersDef *cvd = currentScope();

				if (cvd -> convtosubcode != NULL)
					yyerror("Class has more than one %ConvertToSubClassCode directive");

				cvd -> convtosubcode = $2;
			}
		}
	|	TK_CANTOTYPE codeblock {
			if (getVersion() != NULL)
			{
				classVersDef *cvd = currentScope();

				if (cvd -> canconvtocode != NULL)
					yyerror("Class has more than one %CanConvertToTypeCode directive");

				cvd -> canconvtocode = $2;
			}
		}
	|	TK_TOTYPE codeblock {
			if (getVersion() != NULL)
			{
				classVersDef *cvd = currentScope();

				if (cvd -> convtocode != NULL)
					yyerror("Class has more than one %ConvertToTypeCode directive");

				cvd -> convtocode = $2;
			}
		}
	|	TK_PYMETHODNAME membercode {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				if ($2 == NULL)
					yyerror("Code to implement the special Python method must be given");

				newSpecialMethod(vd,$1,sectionFlags,$2);
			}
		}
	|	TK_PUBLIC optslot ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PUBLIC | $2;
		}
	|	TK_PROTECTED optslot ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PROT | $2;
		}
	|	TK_PRIVATE optslot ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PRIVATE | $2;
		}
	|	TK_SIGNALS ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_SIGNAL;
		}
	|	TK_PYMETHODS ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PYMETHOD;
		}
	|	TK_PYNUMMETHODS ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PYNUMMETHOD;
		}
	|	TK_PYSEQMETHODS ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PYSEQMETHOD;
		}
	|	TK_PYMAPMETHODS ':' {
			if (getVersion() != NULL)
				sectionFlags = SECT_IS_PYMAPMETHOD;
		}
	;

optslot:	{
			$$ = 0;
		}
	|	TK_SLOTS {
			$$ = SECT_IS_SLOT;
		}
	;

dtor:		'~' TK_NAME '(' ')' ';' membercode {
			if (getVersion() != NULL)
			{
				classVersDef *cvd = currentScope();

				if (strcmp(classVersBaseName(cvd),$2) != 0)
					yyerror("Destructor doesn't have the same name as its class");

				if (isDtor(cvd))
					yyerror("Destructor has already been defined");

				cvd -> dtorcode = $6;
				setIsDtor(cvd);

				if ((sectionFlags & SECT_IS_PUBLIC) == 0)
					setNoPublicDtor(cvd);
			}
		}
	;

ctor:		TK_NAME '(' arglist ')' ';' membercode {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				if ((sectionFlags & (SECT_IS_PUBLIC | SECT_IS_PROT | SECT_IS_PRIVATE)) == 0)
					yyerror("Constructor must be in the public, private or protected sections");

				newCtor(vd,$1,sectionFlags,&$3,$6);
			}

			free($1);
		}
	;

function:	cpptype TK_NAME '(' arglist ')' optconst optabstract optflags ';' membercode optvirtualcode {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				if (sectionFlags != 0 && (sectionFlags & (SECT_IS_PUBLIC | SECT_IS_PROT | SECT_IS_PRIVATE | SECT_IS_SLOT | SECT_IS_SIGNAL)) == 0)
					yyerror("Class function must be in the public, private, protected, slot or signal sections");

				newFunction(currentSpec,vd,currentModule,
					    sectionFlags,currentIsStatic,
					    currentOverIsVirt,&$1,$2,&$4,$6,$7,
					    &$8,$10,$11);
			}

			currentIsStatic = FALSE;
			currentOverIsVirt = FALSE;
		}
	;


optconst:	{
			$$ = FALSE;
		}
	|	TK_CONST {
			$$ = TRUE;
		}
	;

optabstract:	{
			$$ = 0;
		}
	|	'=' TK_NUMBER {
			if ($2 != 0)
				yyerror("Abstract virtual function '= 0' expected");

			$$ = TRUE;
		}
	;

optflags:	{
			$$.nrFlags = 0;
		}
	|	'/' flaglist '/' {
			$$ = $2;
		}
	;


flaglist:	flag {
			$$.flags[0] = $1;
			$$.nrFlags = 1;
		}
	|	flaglist ',' flag {
			/* Check there is room. */

			if ($1.nrFlags == MAX_NR_FLAGS)
				yyerror("Too many optional flags");

			$$ = $1;

			$$.flags[$$.nrFlags++] = $3;
		}
	;

flag:		TK_NAME {
			$$.ftype = bool_flag;
			$$.fname = $1;
		}
	|	TK_NAME '=' flagvalue {
			$$ = $3;
			$$.fname = $1;
		}
	;

flagvalue:	TK_NAME {
			$$.ftype = name_flag;
			$$.fvalue.sval = $1;
		}
	|	TK_STRING {
			$$.ftype = string_flag;
			$$.fvalue.sval = $1;
		}
	;

membercode:	{
			$$ = NULL;
		}
	|	TK_MEMBERCODE codeblock {
			$$ = $2;
		}
	;

optvirtualcode:	{
			$$ = NULL;
		}
	|	TK_VIRTUALCODE codeblock {
			$$ = $2;
		}
	;

arglist:	rawarglist {
			int a, nrsigs, nrrxcon, nrrxdis, nrslotcon, nrslotdis, nrarray, nrarraysize;

			nrsigs = nrrxcon = nrrxdis = nrslotcon = nrslotdis = nrarray = nrarraysize = 0;

			for (a = 0; a < $1.nrArgs; ++a)
				switch ($1.args[a].atype)
				{
				case signal_type:
					++nrsigs;
					break;

				case rxcon_type:
					++nrrxcon;
					break;

				case rxdis_type:
					++nrrxdis;
					break;

				case slotcon_type:
					++nrslotcon;
					break;

				case slotdis_type:
					++nrslotdis;
					break;

				case array_type:
				case uarray_type:
					++nrarray;
					break;

				case arraysize_type:
				case arrayusize_type:
					++nrarraysize;
					break;
				}

			if (nrsigs == 1 && nrrxcon == 0)
				yyerror("SIP_SIGNAL can only be used with SIP_RXOBJ_CON");

			if (nrsigs > 1)
				yyerror("SIP_SIGNAL can only be given once");

			if (nrrxcon != nrslotcon || nrrxcon > 1)
				yyerror("SIP_RXOBJ_CON and SIP_SLOT_CON must both be given and at most once");

			if (nrrxdis != nrslotdis || nrrxdis > 1)
				yyerror("SIP_RXOBJ_DIS and SIP_SLOT_DIS must both be given and at most once");

			if (nrarray != nrarraysize || nrarray > 1)
				yyerror("SIP_ARRAY/SIP_UARRAY and SIP_ARRAY_SIZE/SIP_ARRAY_USIZE must both be given and at most once");

			$$ = $1;
		}
	;

rawarglist:	{
			/* No arguments. */

			$$.nrArgs = 0;
		}
	|	argvalue {
			/* The single or first argument. */

			$$.args[0] = $1;
			$$.nrArgs = 1;
		}
	|	rawarglist ',' argvalue {
			/* Check that it wasn't ...(,arg...). */

			if ($$.nrArgs == 0)
				yyerror("First argument of list is missing");

			/*
			 * If this argument has no default value, then all
			 * previous ones mustn't either.
			 */

			if ($3.defval == NULL)
			{
				int a;

				for (a = 0; a < $1.nrArgs; ++a)
					if ($1.args[a].defval != NULL)
						yyerror("Compulsory argument given after optional argument");
			}

			/* Check there is room. */

			if ($1.nrArgs == MAX_NR_ARGS)
				yyerror("Internal error - increase the value of MAX_NR_ARGS");

			$$ = $1;

			$$.args[$$.nrArgs] = $3;
			$$.nrArgs++;
		}
	;

argvalue:	TK_SIPSIGNAL {
			$$.atype = signal_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPRXCON {
			$$.atype = rxcon_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPRXDIS {
			$$.atype = rxdis_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPSLOTCON '(' arglist ')' {
			$$.atype = slotcon_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			$$.u.sa = sipMalloc(sizeof (funcArgs));
			*$$.u.sa = $3;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPSLOTDIS '(' arglist ')' {
			$$.atype = slotdis_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			$$.u.sa = sipMalloc(sizeof (funcArgs));
			*$$.u.sa = $3;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPARRAY {
			$$.atype = array_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPUARRAY {
			$$.atype = uarray_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPARRAYSIZE {
			$$.atype = arraysize_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPARRAYUSIZE {
			$$.atype = arrayusize_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	argtype optassign {
			$$ = $1;
			$$.defval = $2;
		}
	;

varmember:	TK_STATIC {currentIsStatic = TRUE;} varmem
	|	varmem
	;

varmem:		member
	|	variable
	;

member:		TK_VIRTUAL {currentOverIsVirt = TRUE;} function
	|	function
	;

variable:	cpptype TK_NAME ';' optvarcode {
			versionDef *vd;

			if ((vd = getVersion()) != NULL)
			{
				/* Check the section. */

				if (sectionFlags != 0)
				{
					if ((sectionFlags & SECT_IS_PUBLIC) == 0)
						yyerror("Class variables must be in the public section");

					if (!currentIsStatic && $4 != NULL)
						yyerror("%VariableCode cannot be specified for non-static class variables");
				}

				newVar(currentSpec,vd,currentModule,$2,currentIsStatic,&$1,$4);
			}

			currentIsStatic = FALSE;
		}
	;

cpptype:	TK_CONST basetype deref optref {
			$$ = $2;
			$$.nrderefs = $3;
			$$.argflags = ARG_IS_CONST | $4;
		}
	|	basetype deref optref {
			$$ = $1;
			$$.nrderefs = $2;
			$$.argflags = $3;
		}
	;

argtype:	cpptype optflags {
			$$ = $1;

			if (findOptFlag(&$2,"Transfer",bool_flag) != NULL)
				$$.argflags |= ARG_XFERRED;

			if (findOptFlag(&$2,"TransferThis",bool_flag) != NULL)
				$$.argflags |= ARG_THIS_XFERRED;

			if (findOptFlag(&$2,"TransferBack",bool_flag) != NULL)
				$$.argflags |= ARG_XFERRED_BACK;

			if (findOptFlag(&$2,"Constrained",bool_flag) != NULL)
			{
				if ($$.atype != int_type)
					yyerror("/Constrained/ may only be specified for int types");

				$$.atype = cint_type;
			}
		}
	;

optref:		{
			$$ = 0;
		}
	|	'&' {
			$$ = ARG_IS_REF;
		}
	;

deref:		{
			$$ = 0;
		}
	|	deref '*' {
			$$ = $1 + 1;
		}
	;

basetype:	scopedname {
			$$.atype = defined_type;
			$$.u.snd = $1;
		}
	|	scopedname '<' basetypelist '>' {
			templateDef *td;

			td = sipMalloc(sizeof(templateDef));
			td -> fqname = $1;
			td -> types = $3;

			$$.atype = template_type;
			$$.u.td = td;
		}
	|	TK_STRUCT scopedname {
			$$.atype = struct_type;
			$$.u.sname = $2;
		}
	|	TK_UNSIGNED TK_SHORT {
			$$.atype = ushort_type;
		}
	|	TK_SHORT {
			$$.atype = short_type;
		}
	|	TK_UNSIGNED {
			$$.atype = uint_type;
		}
	|	TK_INT {
			$$.atype = int_type;
		}
	|	TK_LONG {
			$$.atype = long_type;
		}
	|	TK_UNSIGNED TK_LONG {
			$$.atype = ulong_type;
		}
	|	TK_FLOAT {
			$$.atype = float_type;
		}
	|	TK_DOUBLE {
			$$.atype = double_type;
		}
	|	TK_BOOL {
			$$.atype = bool_type;
		}
	|	TK_UNSIGNED TK_CHAR {
			$$.atype = ustring_type;
		}
	|	TK_CHAR {
			$$.atype = string_type;
		}
	|	TK_VOID {
			$$.atype = voidptr_type;
		}
	;

basetypelist:	basetype {
			/* The single or first type. */

			$$.args[0] = $1;
			$$.nrArgs = 1;
		}
	|	basetypelist ',' basetype {
			/* Check that it wasn't ,type... */

			if ($$.nrArgs == 0)
				yyerror("First type of list is missing");

			/* Check there is room. */

			if ($1.nrArgs == MAX_NR_ARGS)
				yyerror("Internal error - increase the value of MAX_NR_ARGS");

			$$ = $1;

			$$.args[$$.nrArgs] = $3;
			$$.nrArgs++;
		}
	;

%%


/*
 * Parse the specification.
 */

void parse(sipSpec *spec,FILE *fp,char *filename,char *cppMName,stringList *tsl)
{
        /* Initialise the spec. */
 
	spec -> cppmname = cppMName;
	spec -> namecache = NULL;
	spec -> moduleList = NULL;
	spec -> secondaries = NULL;
	spec -> ifacefiles = NULL;
	spec -> classes = NULL;
	spec -> classversions = NULL;
	spec -> mappedtypes = NULL;
	spec -> used = NULL;
	spec -> qobjclass = -1;
	spec -> exposed = NULL;
	spec -> enums = NULL;
	spec -> vars = NULL;
	spec -> othfuncs = NULL;
	spec -> overs = NULL;
	spec -> typedefs = NULL;
	spec -> versioncode = NULL;
	spec -> copying = NULL;
	spec -> hdrcode = NULL;
	spec -> exphdrcode = NULL;
	spec -> cppcode = NULL;
	spec -> prepycode = NULL;
	spec -> pycode = NULL;
	spec -> docs = NULL;
	spec -> mktemplates = NULL;
	spec -> sigargs = NULL;
	spec -> sigslots = FALSE;

	currentSpec = spec;
	timeSlotList = tsl;
	currentModule = NULL;
	currentMappedType = NULL;
	currentOverIsVirt = FALSE;
	currentIsStatic = FALSE;
	importDepth = -1;
	previousFile = NULL;
	currentVersion = 0;
	currentScopeIdx = 0;
	sectionFlags = 0;

	newModule(fp,filename,FALSE);
	spec -> module = currentModule;

	yyparse();

	handleEOF();
	handleEOM();
}


/*
 * Tell the parser that a complete file has now been read.
 */

void parserEOF(char *name,parserContext *pc)
{
	previousFile = sipStrdup(name);
	newContext = *pc;
}


/*
 * Append a class definition to a class list.  Append is needed specifically
 * for the list of super-classes because the order is important to Python.
 */

static void appendToClassList(classList **clp,classDef *cd)
{
	classList *new;

	/* Find the end of the list. */

	while (*clp != NULL)
		clp = &(*clp) -> next;

	new = sipMalloc(sizeof (classList));

	new -> c = cd;
	new -> next = NULL;

	*clp = new;
}


/*
 * Create a new module for the current specification and make it current.
 */

static void newModule(FILE *fp,char *filename,int inherit)
{
	parseFile(fp,filename,currentModule,inherit);

	currentModule = sipMalloc(sizeof (moduleDef));

	currentModule -> name = NULL;
	currentModule -> file = filename;
	currentModule -> depth = ++importDepth;
	currentModule -> timeSlots = NULL;
	currentModule -> nrTimeSlots = -1;
	currentModule -> required = NULL;
	currentModule -> next = currentSpec -> moduleList;

	currentSpec -> moduleList = currentModule;
}


/*
 * Switch to parsing a new file.
 */

static void parseFile(FILE *fp,char *name,moduleDef *prevmod,int inherit)
{
	parserContext pc;

	pc.ifdepth = currentVersion;
	pc.prevmod = prevmod;
	pc.inherit = inherit;

	setInputFile(fp,name,&pc);
}


/*
 * Find an interface file, or create a new one.
 */

static ifaceFileDef *findIfaceFile(sipSpec *pt,scopedNameDef *fqname)
{
	ifaceFileDef *iff;

	/* See if the name is already used. */

	for (iff = pt -> ifacefiles; iff != NULL; iff = iff -> next)
		if (sameScopedName(iff -> fqname,fqname))
			return iff;

	iff = sipMalloc(sizeof (ifaceFileDef));

	iff -> name = cacheName(pt,currentModule,scopedNameTail(fqname));
	iff -> fqname = fqname;
	iff -> module = NULL;
	iff -> ifacefileflags = 0;
	iff -> next = pt -> ifacefiles;

	pt -> ifacefiles = iff;

	return iff;
}


/*
 * Find a class definition in a parse tree.
 */

static classDef *findClass(sipSpec *pt,scopedNameDef *fqname)
{
	classDef *cd;
	ifaceFileDef *iff;

	/* See if the name is already used. */

	iff = findIfaceFile(pt,fqname);

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
		if (cd -> iff == iff)
			return cd;

	/*
	 * Create a new one.  For the moment we assume it will be defined in
	 * the same module.
	 */

	cd = sipMalloc(sizeof (classDef));

	cd -> iff = iff;
	cd -> classnr = -1;
	cd -> next = pt -> classes;

	pt -> classes = cd;

	return cd;
}


/*
 * Add an interface file to an interface file list if it isn't already there.
 */

void addToUsedList(ifaceFileList **ifflp,ifaceFileDef *iff)
{
	ifaceFileList *iffl;

	while ((iffl = *ifflp) != NULL)
	{
		/* Don't bother if it is already there. */

		if (iffl -> iff == iff)
			return;

		ifflp = &iffl -> next;
	}

	setIsUsed(iff);

	iffl = sipMalloc(sizeof (ifaceFileList));

	iffl -> iff = iff;
	iffl -> next = NULL;

	*ifflp = iffl;
}


/*
 * Check there is no module or version conflict for an interface file.
 */

static void checkIfaceFile(sipSpec *pt,ifaceFileDef *iff,versionDef *vd,argDef *ad)
{
	classVersDef *cvd;
	mappedTypeDef *mtd;

	if (iff -> module == NULL)
		iff -> module = currentModule;
	else if (iff -> module != currentModule)
	{
		fatalScopedName(iff -> fqname);
		fatal(" has already defined as a namespace, class or %%MappedType in another module\n");
	}

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		if (cvd -> common -> iff == iff && versionsOverlap(&cvd -> version,vd))
		{
			fatalScopedName(iff -> fqname);
			fatalVersion(pt,vd);
			fatal(" already defined as a class\n");
		}

	for (mtd = pt -> mappedtypes; mtd != NULL; mtd = mtd -> next)
		if (mtd -> iff == iff && versionsOverlap(&mtd -> version,vd))
		{
			/*
			 * We allow types based on the same template but with
			 * different arguments.
			 */

			if (ad != NULL && ad -> atype == template_type &&
			    !sameBaseType(ad,&mtd -> type,TRUE))
				continue;

			fatalScopedName(iff -> fqname);
			fatalVersion(pt,vd);
			fatal(" already defined as a %%MappedType\n");
		}

	if (isMainModule(iff -> module))
		addToUsedList(&pt -> used,iff);
}


/*
 * Find a namespace in the current module that matches the given name and
 * version.
 */

static classVersDef *findNamespace(sipSpec *pt,versionDef *vd,
				   scopedNameDef *fqname)
{
	classVersDef *cvd;

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		/* It must be a namespace. */

		if (!isNamespace(cvd))
			continue;

		/* It must be in the current module. */

		if (currentModule != cvd -> common -> iff -> module)
			continue;

		/* It must have the same name. */

		if (!sameScopedName(fqname,cvd -> common -> iff -> fqname))
			continue;

		/* It must have a compatible version. */

		if (!versionsOverlap(vd,&cvd -> version))
			continue;

		break;
	}

	return cvd;
}


/*
 * Find an undefined (or create a new) class definition in a parse tree.
 */

static classVersDef *newClass(sipSpec *pt,versionDef *vd,scopedNameDef *fqname)
{
	classVersDef *cvd;
	classDef *cd;

	cd = findClass(pt,fqname);

	setHasClassVers(cd -> iff);

	/* Check the version hasn't already been defined. */

	checkIfaceFile(pt,cd -> iff,vd,NULL);

	/* Create the particular version. */

	cvd = sipMalloc(sizeof (classVersDef));

	cvd -> classversflags = 0;
	cvd -> common = cd;
	cvd -> ecvd = currentScope();
	cvd -> version = *vd;
	cvd -> supers = NULL;
	cvd -> hierachy = NULL;
	cvd -> ctors = NULL;
	cvd -> dtorcode = NULL;
	cvd -> members = NULL;
	cvd -> overs = NULL;
	cvd -> vmembers = NULL;
	cvd -> visible = NULL;
	cvd -> cppcode = NULL;
	cvd -> hdrcode = NULL;
	cvd -> convtosubcode = NULL;
	cvd -> convtocode = NULL;
	cvd -> canconvtocode = NULL;
	cvd -> spms = NULL;
	cvd -> combmeth = 0;
	cvd -> combmethx = 0;
	cvd -> combnummeth = 0;
	cvd -> combnummethx = 0;
	cvd -> combseqmeth = 0;
	cvd -> combseqmethx = 0;
	cvd -> combmapmeth = 0;
	cvd -> combmapmethx = 0;
	cvd -> combcmpmeth = 0;
	cvd -> combcmpmethx = 0;
	cvd -> used = NULL;
	cvd -> next = pt -> classversions;

	pt -> classversions = cvd;

	return cvd;
}


/*
 * Tidy up after finishing a class definition.
 */

static void finishClass(classVersDef *cvd)
{
	overDef *od;

	if (cvd -> ctors == NULL)
	{
		/* Provide a default ctor. */

		cvd -> ctors = sipMalloc(sizeof (ctorDef));
 
		cvd -> ctors -> args.nrArgs = 0;
		cvd -> ctors -> cppcode = NULL;
		cvd -> ctors -> version = cvd -> version;
		cvd -> ctors -> next = NULL;

		resetCannotCreate(cvd);
	}

	/* We can't create this class if it has private abstract functions. */

	for (od = cvd -> overs; od != NULL; od = od -> next)
		if (isAbstract(od) && isPrivate(od))
		{
			setCannotCreate(cvd);
			break;
		}
}


/*
 * Create a new mapped type.
 */

static mappedTypeDef *newMappedType(sipSpec *pt,argDef *ad,versionDef *vd)
{
	mappedTypeDef *mtd;
	scopedNameDef *snd;
	ifaceFileDef *iff;

	/* Check that the type is one we want to map. */

	switch (ad -> atype)
	{
	case defined_type:
		snd = ad -> u.snd;
		break;

	case template_type:
		snd = ad -> u.td -> fqname;
		break;

	case struct_type:
		snd = ad -> u.sname;
		break;

	default:
		yyerror("Invalid type for %MappedType");
	}

	iff = findIfaceFile(pt,snd);

	setHasMappedType(iff);

	/* Check the version hasn't already been defined. */

	checkIfaceFile(pt,iff,vd,ad);

	/* Create a new mapped type. */

	mtd = sipMalloc(sizeof (mappedTypeDef));

	mtd -> type = *ad;
	mtd -> iff = iff;
	mtd -> version = *vd;
	mtd -> hdrcode = NULL;
	mtd -> convfromcode = NULL;
	mtd -> convtocode = NULL;
	mtd -> canconvtocode = NULL;
	mtd -> next = pt -> mappedtypes;

	pt -> mappedtypes = mtd;

	return mtd;
}


/*
 * Create a new enum.
 */

static void newEnum(sipSpec *pt,versionDef *vd,moduleDef *mod,char *name,int flags,enumValueDef *values)
{
	enumDef *ed;
	classVersDef *escope = currentScope();
	enumValueDef *evd;

	/* Check a version of a value doesn't already exist. */

	for (evd = values; evd != NULL; evd = evd -> next)
		checkAttributes(pt,escope,evd -> name,vd,enum_attr);

	ed = sipMalloc(sizeof (enumDef));

	ed -> fqname = (name != NULL ? text2scopedName(name) : NULL);
	ed -> ecvd = escope;
	ed -> module = mod;
	ed -> values = values;
	ed -> version = *vd;
	ed -> next = pt -> enums;

	pt -> enums = ed;
}


/*
 * Create a new typedef.
 */

static void newTypedef(sipSpec *pt,versionDef *vd,char *name,argDef *type)
{
	typedefDef *td;
	scopedNameDef *fqname = text2scopedName(name);

	/* Check a version doesn't already exist. */

	for (td = pt -> typedefs; td != NULL; td = td -> next)
		if (sameScopedName(td -> fqname,fqname) && versionsOverlap(&td -> version,vd))
		{
			fatalScopedName(fqname);
			fatalVersion(pt,vd);
			fatal(" already defined\n");
		}

	td = sipMalloc(sizeof (typedefDef));

	td -> fqname = fqname;
	td -> ecvd = currentScope();
	td -> type = *type;
	td -> version = *vd;
	td -> next = pt -> typedefs;

	pt -> typedefs = td;
}


/*
 * Create a new variable.
 */

static void newVar(sipSpec *pt,versionDef *vd,moduleDef *mod,char *name,
		   int isstatic,argDef *type,codeBlock *code)
{
	varDef *var;
	classVersDef *escope = currentScope();
	nameDef *nd = cacheName(pt,mod,name);

	checkAttributes(pt,escope,nd,vd,var_attr);

	var = sipMalloc(sizeof (varDef));

	var -> name = nd;
	var -> fqname = text2scopedName(name);
	var -> ecvd = escope;
	var -> module = mod;
	var -> varflags = 0;
	var -> type = *type;
	var -> version = *vd;
	var -> accessfunc = code;
	var -> next = pt -> vars;

	if (isstatic)
		setIsStaticVar(var);

	pt -> vars = var;
}


/*
 * Create a new special method.
 */

static void newSpecialMethod(versionDef *vd,char *name,int sectFlags,codeBlock *code)
{
	spmDef *sd;
	classVersDef *cvd = currentScope();
	int *comb, *combx;
	specialPyMethod *spm;

	/* Find the method according to the section. */

	if (sectFlags & SECT_IS_PYMETHOD)
	{
		spm = pyMethods;
		comb = &cvd -> combmeth;
		combx = &cvd -> combmethx;
	}
	else if (sectFlags & SECT_IS_PYNUMMETHOD)
	{
		spm = pyNumMethods;
		comb = &cvd -> combnummeth;
		combx = &cvd -> combnummethx;
	}
	else if (sectFlags & SECT_IS_PYSEQMETHOD)
	{
		spm = pySeqMethods;
		comb = &cvd -> combseqmeth;
		combx = &cvd -> combseqmethx;
	}
	else if (sectFlags & SECT_IS_PYMAPMETHOD)
	{
		spm = pyMapMethods;
		comb = &cvd -> combmapmeth;
		combx = &cvd -> combmapmethx;
	}
	else
		yyerror("Special Python methods must be in the PyMethods, PyNumberMethods, PySequenceMethods or PyMappingMethods sections");

	while (spm -> name != NULL)
	{
		if (strcmp(spm -> name,name) == 0)
			break;

		++spm;
	}

	if (spm -> name == NULL)
		yyerror("Invalid special Python method for this section");

	/* Check one of the same name (not ID) hasn't already been given. */

	for (sd = cvd -> spms; sd != NULL; sd = sd -> next)
		if (strcmp(sd -> spm -> name,name) == 0 && versionsOverlap(&sd -> version,vd))
			yyerror("Different versions of special Python method overlap");

	sd = sipMalloc(sizeof (spmDef));

	sd -> spm = spm;
	sd -> version = *vd;
	sd -> code = code;
	sd -> next = cvd -> spms;

	cvd -> spms = sd;

	*comb |= spm -> id;
	*combx |= spm -> idxtra;
}


/*
 * Create a new ctor.
 */

static void newCtor(versionDef *vd,char *name,int sectFlags,funcArgs *args,codeBlock *code)
{
	ctorDef *ct, **ctp;
	classVersDef *cvd = currentScope();

	/* Check the name of the constructor. */

	if (strcmp(classVersBaseName(cvd),name) != 0)
		yyerror("Constructor doesn't have the same name as its class");

	/*
	 * Find the end of the list and, at the same time, check for any
	 * existing one with the same signature and version.
	 */

	for (ctp = &cvd -> ctors; *ctp != NULL; ctp = &(*ctp) -> next)
		if (sameFuncArgs(&(*ctp) -> args,args,FALSE) && versionsOverlap(&(*ctp) -> version,vd))
			yyerror("Different versions of ctor with the same Python signature overlap");

	/* Add to the list of constructors. */

	ct = sipMalloc(sizeof (ctorDef));

	ct -> ctorflags = sectFlags;
	ct -> args = *args;
	ct -> cppcode = code;
	ct -> version = *vd;
	ct -> next = NULL;

	if (isProtectedCtor(ct))
		setIsComplex(cvd);

	if (!isPrivateCtor(ct))
		resetCannotCreate(cvd);

	*ctp = ct;
}


/*
 * Create a new function.
 */

static void newFunction(sipSpec *pt,versionDef *vd,moduleDef *mod,int sflags,
			int isstatic,int isvirt,argDef *type,char *name,
			funcArgs *args,int isconst,int isabstract,
			optFlags *optflgs,codeBlock *mcode,codeBlock *vcode)
{
	classVersDef *cvd = currentScope();
	char *pname;
	overDef *od, **odp, **headp;
	optFlag *of;

	headp = (cvd != NULL ?  &cvd -> overs : &pt -> overs);

	/* Use the C++ name if a Python name wasn't given. */

	if ((of = findOptFlag(optflgs,"PyName",name_flag)) != NULL)
		pname = of -> fvalue.sval;
	else
		pname = name;

	/* Create a new overload definition. */

	od = sipMalloc(sizeof (overDef));

	/* Set the overload flags. */

	od -> overflags = sflags;

	if (isProtected(od))
		setIsComplex(cvd);

	if ((isSlot(od) || isSignal(od)) && !isPrivate(od))
	{
		if (isSignal(od))
			setIsComplex(cvd);

		pt -> sigslots = TRUE;
	}

	if (isstatic)
	{
		if (!isPublic(od) || isSlot(od) || isSignal(od))
			yyerror("Static functions must be public");

		if (isvirt)
			yyerror("Static functions cannot be virtual");

		setIsStatic(od);
	}

	if (isconst)
		setIsConst(od);

	if (isabstract)
	{
		if (sflags == 0)
			yyerror("Non-class function specified as abstract");

		if (mcode != NULL)
			yyerror("Cannot provide member code for abstract functions");

		setIsAbstract(od);
	}

	if (findOptFlag(optflgs,"ReleaseLock",bool_flag) != NULL)
		setIsBlocking(od);

	if (isvirt)
	{
		if (isSignal(od))
			yyerror("Virtual signals aren't supported");

		setIsVirtual(od);
		setIsComplex(cvd);
	}
	else if (vcode != NULL)
		yyerror("Virtual code provided for non-virtual function");

	od -> cppname = name;
	od -> args = *args;
	od -> cppcode = mcode;
	od -> virtcode = vcode;
	od -> version = *vd;
	od -> common = findFunction(pt,vd,mod,cvd,pname);

	if ((of = findOptFlag(optflgs,"PreHook",name_flag)) != NULL)
		od -> prehook = of -> fvalue.sval;
	else
		od -> prehook = NULL;

	if ((of = findOptFlag(optflgs,"PostHook",name_flag)) != NULL)
		od -> posthook = of -> fvalue.sval;
	else
		od -> posthook = NULL;

	if (type -> atype == voidptr_type && type -> nrderefs == 0)
		od -> result = NULL;
	else
	{
		od -> result = sipMalloc(sizeof (argDef));
		*od -> result = *type;
	}

	od -> next = NULL;

	/*
	 * Find the end of the list and, at the same time, check for any
	 * existing one with the same signature and an overlapping version, and
	 * that if one is static they all are.
	 */

	for (odp = headp; *odp != NULL; odp = &(*odp) -> next)
	{
		if ((*odp) -> common != od -> common)
			continue;

		if (!sameOverload(*odp,od,FALSE))
			continue;

		if (versionsOverlap(&(*odp) -> version,vd))
			yyerror("Different versions of function with the same Python signature overlap");

		if (isStatic(*odp) != isStatic(od))
			yyerror("Cannot mix static and non-static member functions with the same Python name");
	}

	*odp = od;
}


/*
 * Cache a name in a module.
 */

static nameDef *cacheName(sipSpec *pt,moduleDef *mod,char *name)
{
	nameDef *nd;

	/* See if it already exists. */

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		if (strcmp(nd -> text,name) == 0)
		{
			/*
			 * Use the module that is deepest in the import
			 * hierachy.  This will guarantee it really exists.
			 */

			if (nd -> module -> depth < mod -> depth)
				nd -> module = mod;

			return nd;
		}

	/* Create a new one. */

	nd = sipMalloc(sizeof (nameDef));

	nd -> text = name;
	nd -> module = mod;
	nd -> next = pt -> namecache;

	pt -> namecache = nd;

	return nd;
}


/*
 * Find (or create) an overloaded function name.
 */

static memberDef *findFunction(sipSpec *pt,versionDef *vd,moduleDef *mod,
			       classVersDef *cvd,char *pname)
{
	memberDef *md, **flist;
	nameDef *pnd;

	/* Cache the name. */

	pnd = cacheName(pt,mod,pname);

	/* Check there is no name clash. */

	checkAttributes(pt,cvd,pnd,vd,func_attr);

	/* See if it already exists. */

	flist = (cvd != NULL ? &cvd -> members : &pt -> othfuncs);

	for (md = *flist; md != NULL; md = md -> next)
		if (md -> pyname == pnd)
			return md;

	/* Create a new one. */

	md = sipMalloc(sizeof (memberDef));

	md -> pyname = pnd;
	md -> module = mod;
	md -> next = *flist;

	*flist = md;

	return md;
}


/*
 * Get the current version or NULL if we are not interested in the current
 * version.  currentScope() is only valid if getVersion() returns non-NULL.
 */

static versionDef *getVersion(void)
{
	static versionDef vers;

	if (currentVersion > 0)
	{
		versionRange *vr = &versScope[currentVersion - 1];
		int req = currentModule -> required -> order;

		if (vr -> lowidx <= req && req < vr -> uppidx)
			vers.secondary = vr -> secondary;
		else
			return NULL;
	}
	else
		vers.secondary = NULL;

	return &vers;
}


/*
 * Search a set of flags for a particular one and check its type.
 */

static optFlag *findOptFlag(optFlags *flgs,char *name,flagType ft)
{
	int f;

	for (f = 0; f < flgs -> nrFlags; ++f)
		if (strcmp(flgs -> flags[f].fname,name) == 0)
		{
			if (flgs -> flags[f].ftype != ft)
				yyerror("Optional flag has a value of the wrong type");

			return &flgs -> flags[f];
		}

	return NULL;
}


/*
 * A name is going to be used as a Python attribute name with in a Python scope
 * (ie. a Python dictionary), so check against what we already know is going in
 * the same scope in case there is a clash.
 */

static void checkAttributes(sipSpec *pt,classVersDef *pyscope,nameDef *attr,
			    versionDef *attrvd,attrType at)
{
	enumDef *ed;
	varDef *vd;
	classVersDef *cvd;

	/* Check the enum values. */

	for (ed = pt -> enums; ed != NULL; ed = ed -> next)
	{
		enumValueDef *evd;

		if (ed -> ecvd != pyscope)
			continue;

		if (!versionsOverlap(&ed -> version,attrvd))
			continue;

		for (evd = ed -> values; evd != NULL; evd = evd -> next)
			if (evd -> name == attr)
				yyerror("There is already an enum value in scope with the same Python name");
	}

	/* Check the variables. */

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		if (vd -> ecvd != pyscope)
			continue;

		if (vd -> name != attr)
			continue;

		if (!versionsOverlap(&vd -> version,attrvd))
			continue;

		yyerror("There is already a variable in scope with the same Python name");
	}

	/*
	 * Only check the members if this attribute isn't a member because we
	 * can handle members with the same name in the same scope.
	 */

	if (at != func_attr)
	{
		memberDef *md, *membs;

		membs = (pyscope != NULL ? pyscope -> members : pt -> othfuncs);

		for (md = membs; md != NULL; md = md -> next)
		{
			overDef *od, *overs;

			if (md -> pyname != attr)
				continue;

			/* Check for version conflict with all overloads. */

			overs = (pyscope != NULL ? pyscope -> overs : pt -> overs);

			for (od = overs; od != NULL; od = od -> next)
			{
				if (od -> common != md)
					continue;

				if (!versionsOverlap(&od -> version,attrvd))
					continue;

				yyerror("There is already a function in scope with the same Python name");
			}
		}
	}

	/* Check the classes. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		if (cvd -> ecvd != pyscope)
			continue;

		if (cvd -> common -> iff -> name != attr)
			continue;

		if (!versionsOverlap(&cvd -> version,attrvd))
			continue;

		yyerror("There is already a class or namespace in scope with the same Python name");
	}
}


/*
 * Append a code block to a list of them.  Append is needed to give the
 * specifier easy control over the order of the documentation.
 */

static void appendCodeBlock(codeBlock **headp,codeBlock *new)
{
	while (*headp != NULL)
		headp = &(*headp) -> next;

	*headp = new;
}


/*
 * Check if two version qualifiers overlap, ie both might be true at the same
 * time.
 */

int versionsOverlap(versionDef *vd1,versionDef *vd2)
{
	/* Can't overlap if both have secondaries and they are different. */

	if (vd1 -> secondary != NULL && vd2 -> secondary != NULL && vd1 -> secondary != vd2 -> secondary)
		return FALSE;

	return TRUE;
}


/*
 * Handle the end of a fully parsed a file.
 */

static void handleEOF()
{
	/*
	 * Check that the number of nested if's is the same as when we started
	 * the file.
	 */

	if (currentVersion > newContext.ifdepth)
		fatal("Too many %%If statements in %s\n",previousFile);

	if (currentVersion < newContext.ifdepth)
		fatal("Too many %%End statements in %s\n",previousFile);
}


/*
 * Handle the end of a fully parsed a module.
 */

static void handleEOM()
{
	/* Check it has been named. */

	if (currentModule -> name == NULL)
		fatal("No %%Module has been specified for module defined in %s\n",previousFile);

	/* Check is has any primary versions ordered. */

	if (currentModule -> timeSlots != NULL && currentModule -> nrTimeSlots < 0)
		fatal("No %%PrimaryOrder has been specified for module with primary versions\n");

	/* Inherit the timeline if requested. */

	if (newContext.inherit)
	{
		newContext.prevmod -> timeSlots = currentModule -> timeSlots;
		newContext.prevmod -> nrTimeSlots = currentModule -> nrTimeSlots;
		newContext.prevmod -> required = currentModule -> required;
	}

	/* The previous module is now current. */

	currentModule = newContext.prevmod;

	--importDepth;
}


/*
 * Find an existing version qualifier in a list.
 */

static versionQual *findQualifier(versionQual *vq,char *name)
{
	while (vq != NULL)
	{
		if (strcmp(vq -> name,name) == 0)
			break;

		vq = vq -> next;
	}

	return vq;
}


/*
 * Return a copy of a scoped name.
 */

scopedNameDef *copyScopedName(scopedNameDef *snd)
{
	scopedNameDef *head;

	head = NULL;

	while (snd != NULL)
	{
		appendScopedName(&head,text2scopePart(snd -> name));
		snd = snd -> next;
	}

	return head;
}


/*
 * Append a name to a list of scopes.
 */

void appendScopedName(scopedNameDef **headp,scopedNameDef *newsnd)
{
	while (*headp != NULL)
		headp = &(*headp) -> next;

	*headp = newsnd;
}


/*
 * Free a scoped name - but not the text itself.
 */

void freeScopedName(scopedNameDef *snd)
{
	while (snd != NULL)
	{
		scopedNameDef *next = snd -> next;

		free(snd);

		snd = next;
	}
}


/*
 * Convert a text string to a scope part structure.
 */

scopedNameDef *text2scopePart(char *text)
{
	scopedNameDef *snd;

	snd = sipMalloc(sizeof (scopedNameDef));

	snd -> name = text;
	snd -> next = NULL;

	return snd;
}


/*
 * Convert a text string to a fully scoped name.
 */

static scopedNameDef *text2scopedName(char *text)
{
	classVersDef *cvd = currentScope();
	scopedNameDef *snd;

	snd = (cvd != NULL ? copyScopedName(cvd -> common -> iff -> fqname) : NULL);

	appendScopedName(&snd,text2scopePart(text));

	return snd;
}


/*
 * Return a pointer to the tail part of a scoped name.
 */

char *scopedNameTail(scopedNameDef *snd)
{
	if (snd == NULL)
		return NULL;

	while (snd -> next != NULL)
		snd = snd -> next;

	return snd -> name;
}


/*
 * Push the given scope onto the scope stack.
 */

static void pushScope(classVersDef *scope)
{
	if (currentScopeIdx >= MAX_NESTED_SCOPE)
		fatal("Internal error: increase the value of MAX_NESTED_SCOPE\n");

	scopeStack[currentScopeIdx] = scope;
	sectFlagsStack[currentScopeIdx] = sectionFlags;

	++currentScopeIdx;
}


/*
 * Pop the scope stack.
 */

static void popScope(void)
{
	if (currentScopeIdx > 0)
		sectionFlags = sectFlagsStack[currentScopeIdx--];
}


/*
 * Return the current scope.  currentScope() is only valid if getVersion()
 * returns non-NULL.
 */

static classVersDef *currentScope(void)
{
	return (currentScopeIdx > 0 ? scopeStack[currentScopeIdx - 1] : NULL);
}


/*
 * Create a new version qualifier.
 */

static versionQual *newVersion(sipSpec *pt,moduleDef *mod,int order,char *name,char *cpp,char *py)
{
	versionQual *vq, **headp;

	/* We check secondaries anyway. */

	if (findQualifier(pt -> secondaries,name) != NULL)
		yyerror("Version already defined as a secondary");

	/*
	 * If we are adding a secondary check primaries in every module.  If
	 * adding a primary then check other primaries defined in this module.
	 */

	if (order < 0)
	{
		moduleDef *md;

		for (md = pt -> moduleList; md != NULL; md = md -> next)
			if (findQualifier(md -> timeSlots,name) != NULL)
				yyerror("Secondary version already defined as a primary");

		headp = &pt -> secondaries;
	}
	else if (findQualifier(mod -> timeSlots,name) != NULL)
		yyerror("Primary version already defined in the same module");
	else
		headp = &mod -> timeSlots;

	vq = sipMalloc(sizeof (versionQual));

	vq -> name = name;
	vq -> module = mod;
	vq -> order = order;
	vq -> vqcpp = cpp;
	vq -> vqpy = py;
	vq -> next = *headp;

	*headp = vq;

	return vq;
}


/*
 * Create a new imported module.
 */

static void newImport(sipSpec *pt,char *name,int inherit)
{
	/* Ignore if the file has already been imported. */

	moduleDef *mod;

	for (mod = pt -> moduleList; mod != NULL; mod = mod -> next)
		if (strcmp(mod -> file,name) == 0)
			return;

	newModule(NULL,name,inherit);
}
