/*
    Copyright (C) 2003-2006 Teus Benschop.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


#include "libraries.h"
#include "versification.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "constants.h"
#include "utilities.h"


ustring current_element;
ustring current_value;
ustring current_book;
int current_chapter;
int current_verse;
vector<ustring> books;
ustring requested_book;
int requested_chapter;
int retrieved_chapter;
int retrieved_verse;
vector<int> retrieved_verses;


void versification_start_element_handler (GMarkupParseContext *context,
                                          const gchar         *element_name,
                                          const gchar        **attribute_names,
                                          const gchar        **attribute_values,
                                          gpointer             user_data,
                                          GError             **error)
{
  current_element = element_name;
  if (current_element == BOOK_TAG) {
    current_book = attribute_values [0];
  }
  current_value.clear();
}


void versification_end_element_handler (GMarkupParseContext *context,
                                        const gchar         *element_name,
                                        gpointer             user_data,
                                        GError             **error)
{
  current_value = trim (current_value);
  ustring element (element_name);
  if (element == CHAPTER_TAG) {
    current_chapter = convert_to_int (current_value);
  } 
  else if (element == VERSE_TAG) {
    current_verse = convert_to_int (current_value);
  }
  else if (element == BOOK_TAG) {
    books.push_back (current_book);
  }
  else if (element == LIMIT_TAG) {
    if (current_book == requested_book) {
      retrieved_chapter = current_chapter;
      retrieved_verses.push_back (current_verse);
      if (requested_chapter == current_chapter)
        retrieved_verse = current_verse;
    }
  }
}


void versification_text_handler (GMarkupParseContext *context,
                                 const gchar         *text,
                                 gsize                text_len,
                                 gpointer             user_data,
                                 GError             **error)
{
  current_value.append (text);
}



void versification_passthrough_handler (GMarkupParseContext *context,
                                        const gchar         *passthrough_text,
                                        gsize                text_len,
                                        gpointer             user_data,
                                        GError             **error)
{
}


void versification_error_handler (GMarkupParseContext *context,
                                  GError              *error,
                                  gpointer             user_data)
{
  cerr << error->message << endl;
}


vector<ustring> versification_get_systems ()
// Lists the available versification systems the library knows about.
{
  vector<ustring> result;
  ReadFiles rf (PACKAGE_DATA_DIR, "vrs_", ".xml");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    ustring filename;
    filename = gw_build_filename (PACKAGE_DATA_DIR, rf.files[i]);
    ReadText rt (filename, true, true);
    for (unsigned int i2 = 0; i2 < rt.lines.size(); i2++) {
      size_t position;
      position = rt.lines[i2].find ("versification name=");
      if (position != string::npos) {
        ustring name (rt.lines[i2]);
        position = name.find ("\"");
        name.erase (0, ++position);
        name.erase (name.length() - 2, 2);
        result.push_back (name);
        break;
      }
    }
  }    
  return result;
}


void versification_parse_data (const ustring & name)
{
  // Clear data.
  books.clear();
  retrieved_chapter = 0;
  retrieved_verse = 0;
  retrieved_verses.clear();
  // Read the versification file.
  string s;
  s = name;
  for (unsigned int i = 0; i < s.length(); i++)
    if (s[i] == ' ')
      s[i] = '_';
  ustring filename (s);
  filename = "vrs_" + filename.casefold() + ".xml";
  filename = gw_build_filename (PACKAGE_DATA_DIR, filename);
  bool was_error = false;
  gchar *contents;
  gsize  length;
  GError *error;
  error = NULL;
  if (!g_file_get_contents (filename.c_str(), &contents, &length, &error)) {
    cerr << error->message << endl;
    g_error_free (error);
    was_error = true;
  }
  // Set up parser.
  GMarkupParseContext *context;
  GMarkupParser parser = {
    versification_start_element_handler,
    versification_end_element_handler,
    versification_text_handler,
    versification_passthrough_handler,
    versification_error_handler
  };
  // Parse xml file.
  if (!was_error) {
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), NULL, NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error) {
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error)
    g_markup_parse_context_free (context);
}


vector<ustring> versification_get_books (const ustring & name)
{
  requested_book.clear();
  requested_chapter = -1;
  versification_parse_data (name);
  return books;
}


int versification_get_highest_chapter (const ustring & name, const ustring& book)
{
  requested_book = book;
  requested_chapter = -1;
  versification_parse_data (name);
  return retrieved_chapter;
}


int versification_get_highest_verse (const ustring & name, const ustring& book, int chapter)
{
  requested_book = book;
  requested_chapter = chapter;
  versification_parse_data (name);
  return retrieved_verse;
}


vector<int> versification_get_verses (const ustring & name, const ustring& book)
// Returns a list of highest verses per chapter in "book".
{
  requested_book = book;
  requested_chapter = -1;
  versification_parse_data (name);
  return retrieved_verses;
}
