#!/bin/sh

# tbstats.tcl:
#   Scid script to show the number of times each material configuration
#   covered by an endgame tablebase occurs in a given Scid database.
#   Usage: tbstats.tcl database-name
#      or: tcscid tbtasts.tcl database-name

# The "\" at the end of this line is necessary: \
exec tcscid "$0" "$@"

array set p {0 q  1 r  2 b  3 n  4 p}
set pieces {0 1 2 3 4}
set nopawns {0 1 2 3}

proc search {str} {
  array set w {Q 0 R 0 N 0 B 0 P 0}
  array set b {Q 0 R 0 N 0 B 0 P 0}
  set side w
  set arglist [split $str {}]

  foreach i $arglist {
    if {$i == "-"} {
      set side b
    } else {
      set i [string toupper $i]
      if {$side == "w"} { catch {incr w($i) }} else { catch {incr b($i)} }
    }
  }

  sc_searchMaterial \
    $w(Q) $w(Q) $b(Q) $b(Q) $w(R) $w(R) $b(R) $b(R) $w(B) $w(B) $b(B) $b(B) \
    $w(N) $w(N) $b(N) $b(N) 0 9 0 9 $w(P) $w(P) $b(P) $b(P) 1 2 0 999 1 Either
  return [sc_filter count]
}

proc newline {} {
  puts ""
}

proc indent {} {
  puts -nonewline "               "
}

proc tbstats {str} {
  puts -nonewline [format "  %7s %5d" [string toupper $str] [search $str]]
}


proc tb21 {} {
  global p pieces
  newline
  puts "# 2-1"
  foreach a $pieces {
    tbstats "k$p($a)-k"
  }
  newline
}

proc tb31 {} {
  global p pieces
  newline
  puts "# 3-1"
  foreach a $pieces {
    foreach b $pieces {
      if {$a <= $b} {
        tbstats "k$p($a)$p($b)-k"
      } else {
        indent
      }
    }
    newline
  }
}

proc tb22 {} {
  global p pieces
  newline
  puts "# 2-2"
  foreach a $pieces {
    foreach b $pieces {
      if {$a <= $b} {
        tbstats "k$p($a)-k$p($b)"
      } else {
        indent
      }
    }
    newline
  }
}

proc tb32 {} {
  global p pieces
  newline
  puts "# 3-2"
  foreach a $pieces {
    foreach b $pieces {
      if {$b < $a} { continue }
      foreach c $pieces {
        tbstats "k$p($a)$p($b)-k$p($c)"
      }
      newline
    }
  }
}

proc tb41 {} {
  global p pieces
  newline
  puts "# 4-1"
  foreach a $pieces {
    foreach b $pieces {
      if {$b < $a} { continue }
      foreach c $pieces {
        if {$c < $a  ||  $c < $b} {
          indent
        } else {
          tbstats "k$p($a)$p($b)$p($c)-k"
        }
      }
      newline
    }
  }
}

proc tb33 {{type nopawns}} {
  global p
  if {$type == "pawns"} {
    set pieces $::pieces
  } else {
    set pieces $::nopawns
  }
  newline
  puts -nonewline "# 3-3"
  if {$type != "pawns"} { puts -nonewline " (no pawns)" }
  newline
  foreach a $pieces {
    foreach b $pieces {
      if {$b <$a} { continue }
      foreach c $pieces {
        if {$c < $a} { continue }
        foreach d $pieces {
          if {$d < $c  ||  $d < $a  ||  $d < $b} {
            indent
          } else {
            tbstats "k$p($a)$p($b)-k$p($c)$p($d)"
          }
        }
        newline
      }
    }
  }
}

proc tb33n {} { tb33 nopawns }
proc tb33p {} { tb33 pawns }


proc usage {} {
  global argv0
  puts stderr "Usage: $argv0 <database> \[material ...\]"
  puts stderr "Example: $argv0 <database> 21 22 31 32 41"
  exit 1
}

### Main program:

if {[llength $argv] < 1} { usage }
set db [lindex $argv 0]
if {[catch { sc_base open $db }]} {
    puts stderr "Error: could not open the Scid database: $db"
    exit 1
}
set argv [lrange $argv 1 end]

puts "# Tablebase endgame frequency statistics"
puts "# Generated from the database \"$db\": [sc_base numGames] games"

if {[llength $argv] == 0} {
  tb21; tb22; tb31; tb32; tb41; #tb33n
} else {
  foreach i $argv {
    catch {tb$i}
  }
}

