# Samizdat mock classes
#
#   Copyright (c) 2002-2006  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

$samizdat_site_name = ENV['SAMIZDAT_SITE']

class MockConfig < SimpleDelegator
  include Singleton

  RDF = 'data/samizdat/rdf.yaml'
  DEFAULTS = 'data/samizdat/defaults.yaml'
  CONFIG = 'data/samizdat/config.yaml'
  XHTML = 'data/samizdat/xhtml.yaml'

  def initialize
    @config = File.open(RDF) {|f| YAML.load(f) }
    @config.merge! File.open(DEFAULTS) {|f| YAML.load(f) }
    @config.merge! File.open(CONFIG) {|f| YAML.load(f) }
    @xhtml = File.open(XHTML) {|f| YAML.load(f) }

    super @config
  end

  attr_reader :config, :drb, :xhtml

  def config_enabled?
    false
  end
end

def config
  MockConfig.instance
end

class MockSession
  def initialize
    @login = 'guest'
    @full_name = _(@login)
  end

  attr_accessor :member, :login, :full_name, :moderator

  def moderator?
    @moderator
  end
end

class MockRequest
  def initialize
    @session = MockSession.new
    @permissions = {}
    @moderate = false
    @options = {}
    @uri_prefix = ENV['SAMIZDAT_URI']
    @base = 'http://localhost' + @uri_prefix
    @content_location = 'content/'
    @route = '/'
    bindtextdomain('samizdat', nil, 'C', 'utf-8')
  end

  attr_accessor :session, :permissions, :moderate, :options, :uri_prefix, :base,
    :content_location, :route

  def cookie(name)
    nil
  end

  def access(action)
    @permissions[action]
  end

  def moderate?
    @moderate
  end
end

class MockMember
  def initialize
    @id = 1
    @login = 'test'
    @full_name = 'Test'
  end

  attr_accessor :id, :login, :full_name

  def guest?
    'guest' == @login
  end
end

class MockGuestMember < MockMember
  def initialize
    @id = nil
    @login = 'guest'
    @full_name = _(@login)
  end
end

class MockFromHash
  def initialize(params = {})
    @params = params
    @id = @params[:id]
  end

  attr_reader :id

  def method_missing(name, value = nil)
    if /=$/ =~ name.to_s
      @params[ name.to_s.sub(/=$/, '').to_sym ] = value
    else
      @params[name]
    end
  end
end

class MockContent < MockFromHash
  def initialize(params = {})
    super
    @params = {
      :title => '',
      :body => '',
      :format => nil,
      :html_full => nil,
      :html_short => nil,
      :inline? => true,
      :file_path => ''
    }.merge!(@params)
  end

  def inline?
    @params[:format].nil? or config['format']['inline'].include?(@params[:format])
  end

  def cacheable?
    inline? and @params[:format] != 'application/x-squish'
  end
end

class MockMessage < MockFromHash
  def initialize(params = {})
    super
    @params = {
      :content => MockContent.new(:id => params[:id]),
      :date => Time.now,
      :lang => nil,
      :creator => MockMember.new,
      :desc => nil,
      :parent => nil,
      :current => nil,
      :open => false,
      :nversions => 0,
      :translations => [],
      :nreplies => 0,
      :focuses => [],
      :nrelated => 0,
      :moderation_log => []
    }.merge!(@params)
  end
end
