/*
 * ARTI -- RTAI-compatible Adeos-based Real-Time Interface.
 *	   Based on ARTI for x86 and RTHAL for ARM.
 *
 * Original RTAI/x86 layer implementation:
 *   Copyright (c) 2000 Paolo Mantegazza (mantegazza@aero.polimi.it)
 *   Copyright (c) 2000 Steve Papacharalambous (stevep@zentropix.com)
 *   Copyright (c) 2000 Stuart Hughes
 *   and others.
 *
 * RTAI/x86 rewrite over Adeos:
 *   Copyright (c) 2002 Philippe Gerum (rpm@xenomai.org)
 *
 * Original RTAI/ARM RTHAL implementation:
 *   Copyright (c) 2000 Pierre Cloutier (pcloutier@poseidoncontrols.com)
 *   Copyright (c) 2001 Alex Zpke, SYSGO RTS GmbH (azu@sysgo.de)
 *   Copyright (c) 2002 Guennadi Liakhovetski DSA GmbH (gl@dsa-ac.de)
 *   Copyright (c) 2002 Steve Papacharalambous (stevep@zentropix.com)
 *   Copyright (c) 2002 Wolfgang Mller (wolfgang.mueller@dsa-ac.de)
 *   Copyright (c) 2003 Bernard Haible, Marconi Communications
 *   Copyright (c) 2003 Thomas Gleixner (tglx@linutronix.de)
 *   Copyright (c) 2003 Philippe Gerum (rpm@xenomai.org)
 *
 * RTAI/ARM over Adeos rewrite:
 *   Copyright (c) 2004-2005 Michael Neuhauser, Firmix Software GmbH (mike@firmix.at)
 *
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge MA 02139, USA; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef _RTAI_ASM_ARM_HAL_H
#define _RTAI_ASM_ARM_HAL_H

#include <rtai_types.h>
#include <asm/rtai_vectors.h>
#include <linux/bitops.h>
#ifndef __KERNEL__
#include <strings.h>
#endif

/* sanity checks */
#ifndef CONFIG_RTAI_ADEOS
#error "Sorry, you have to use a Kernel patched with Adeos for ARM"
#endif
#ifdef CONFIG_SMP
#error "Sorry, SMP is not supported on ARM"
#endif

#define RTAI_NR_CPUS	1

/* no, we don't have a 8254 to emulate TSC (sub-arch has to provide TSC stuff!) */
#undef DECLR_8254_TSC_EMULATION

extern inline unsigned long
ffnz(unsigned long word)
{
    return ffs(word) - 1;
}

extern inline unsigned long long
rtai_ulldiv(unsigned long long ull,
	    unsigned long uld,
	    unsigned long *r)
{
    unsigned long long q = ull / uld;
    *r = ull - q * uld;
    return q;
}

extern inline long long
rtai_llimd(long long ll, int mult, int div)
{
    long long q;
    int r;

    if (mult == div)
	return ll;

    ll *= mult;
    q = ll / div;
    r = ll - q * div;

    return (r + r) < div ? q : q + 1;
}

extern inline int
rtai_imuldiv(int i, int mult, int div)
{
    long long ll = i;
    int q;
    int r;

    if (mult == div)
	return i;

    ll *= mult;
    q = ll / div;
    r = ll - (long long)q * div;

    return (r + r) < div ? q : q + 1;
}

/* get architecture specific things (using <asm/...> breaks
 * build-in-seperate-directory feature!) */
#include <asm-arm/arch/rtai_arch.h>

#define RTAI_CPU_FREQ             	RTAI_TSC_FREQ
#define RTAI_CALIBRATED_CPU_FREQ  	RTAI_TSC_FREQ

#if defined(__KERNEL__) && !defined(__cplusplus)
#include <linux/sched.h>
#include <rtai_trace.h>
#include <asm/rtai_sched.h>
#include <asm/rtai_atomic.h>
#include <asm/rtai_fpu.h>

#define RTAI_DOMAIN_ID			0x52544149
#define RTAI_NR_SRQS			32

#define RTAI_TIME_LIMIT		   	0x7FFFFFFFFFFFFFFFLL

#define RTAI_IFLAG			(7)

#define rtai_cpuid()			adeos_processor_id()
#define rtai_tskext(idx)                ptd[idx]

/* Use these in hard real time code to achieve UP-atomicity. */
#define rtai_cli()			adeos_stall_pipeline_from(&rtai_domain)
#define rtai_sti()			adeos_unstall_pipeline_from(&rtai_domain)
#define rtai_save_flags_and_cli(x)	((x) = adeos_test_and_stall_pipeline_from(&rtai_domain))
#define rtai_restore_flags(x)		adeos_restore_pipeline_from(&rtai_domain,(x))
#define rtai_save_flags(x)		((x) = adeos_test_pipeline_from(&rtai_domain))

/* Use these when accessing the hardware. */
#define rtai_hw_disable()		adeos_hw_cli()
#define rtai_hw_enable()		adeos_hw_sti()
#define rtai_hw_lock(flags)		adeos_hw_local_irq_save(flags)
#define rtai_hw_unlock(flags)		adeos_hw_local_irq_restore(flags)
#define rtai_hw_flags(x)		adeos_hw_local_irq_flags(flags)
/* alternative names */
#define rtai_hw_cli()			rtai_hw_disable()
#define rtai_hw_sti()			rtai_hw_enable()
#define rtai_hw_save_flags_and_cli(f)	rtai_hw_lock(f)
#define rtai_hw_restore_flags(f)	rtai_hw_unlock(f)
#define rtai_hw_save_flags(f)		rtai_hw_flags(f)

#define __adeos_pend_uncond(irq, cpuid) adeos_propagate_irq(irq)

struct calibration_data {
    unsigned long cpu_freq;		/* TSC (i.e. rtai_rdtsc()) clock frequency, set in hal.c */
    int latency;			/* all other values are set by the scheduler, see there */
    int setup_time_TIMER_CPUNIT;
    int setup_time_TIMER_UNIT;
    int timers_tol[RTAI_NR_CPUS];
};
extern struct calibration_data		rtai_tunables;

struct rtai_switch_data {
    volatile unsigned long depth;
    volatile unsigned long oldflags;
};

/* x86 legacy type (needed because some general code still uses x86 specific stuff) */
struct apic_timer_setup_data {
    int mode;
    int count;
};

extern struct rt_times			rt_times;
extern struct rt_times			rt_smp_times[RTAI_NR_CPUS];
extern volatile unsigned long		rtai_cpu_realtime;
extern struct rtai_switch_data 		rtai_linux_context[RTAI_NR_CPUS];
extern int				rtai_adeos_ptdbase;
extern adomain_t			rtai_domain;

/* type of rt irq handler (general + timer) */
typedef void (*rt_irq_handler_t)(unsigned irq, void *cookie);
typedef void (*rt_timer_irq_handler_t)(void);

#ifdef CONFIG_ADEOS_THREADS

/* rtai_get_current() is Adeos-specific. Since real-time interrupt
   handlers are called on behalf of the RTAI domain stack, we cannot
   infere the "current" Linux task address using a CPU-register. We must use the
   suspended Linux domain's stack pointer instead. */
extern inline struct task_struct *
rtai_get_root_current(int cpuid)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    return (struct task_struct *)(((u_long)adp_root->esp[cpuid]) & (~8191UL));
#else /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0) */
    return ((struct thread_info *)(((u_long)adp_root->esp[cpuid]) & (~((THREAD_SIZE)-1))))->task;
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0) */
}

#else /* !CONFIG_ADEOS_THREADS */

extern inline struct task_struct *
rtai_get_root_current(int cpuid)
{
    return current;
}

#endif /* CONFIG_ADEOS_THREADS */

/* following macros only valid for UP */

#ifndef CONFIG_SMP

#define rt_spin_lock(lock)			(void)(lock) /* Not "unused variable". */
#define rt_spin_unlock(lock)			do { /* nop */ } while (0)
#define rt_spin_lock_irq(lock)	  		do { (void)(lock); rtai_cli(); } while (0)
#define rt_spin_unlock_irq(lock)  		rtai_sti()
#define rt_spin_lock_irqsave(lock)		({ unsigned long flags; (void)(lock); rtai_save_flags_and_cli(flags); flags; })
#define rt_spin_unlock_irqrestore(flags, lock)	rtai_restore_flags(flags)

#define rt_get_global_lock()	  		rtai_cli()
#define rt_release_global_lock()		do { /* nop */ } while (0)

#define rt_global_cli()  			rtai_cli()
#define rt_global_sti()  			rtai_sti()
#define rt_global_save_flags_and_cli()		({ unsigned long flags; rtai_save_flags_and_cli(flags); flags; })
#define rt_global_restore_flags(flags)		rtai_restore_flags(flags)
#define rt_global_save_flags(flags)		rtai_save_flags(*flags)

#endif /* !CONFIG_SMP */

extern inline void
rt_switch_to_real_time(int cpuid)
{
    ADEOS_PARANOIA_ASSERT(adeos_hw_irqs_disabled());
    TRACE_RTAI_SWITCHTO_RT(cpuid);
    if (!rtai_linux_context[cpuid].depth++) {
	rtai_linux_context[cpuid].oldflags = adeos_test_and_stall_pipeline_from(adp_root);
	__set_bit(cpuid, &rtai_cpu_realtime);
    } else {
	ADEOS_PARANOIA_ASSERT(test_bit(IPIPE_STALL_FLAG, &adp_root->cpudata[cpuid].status) != 0);
    }
}

extern inline void
rt_switch_to_linux(int cpuid)
{
    ADEOS_PARANOIA_ASSERT(adeos_hw_irqs_disabled());
    TRACE_RTAI_SWITCHTO_LINUX(cpuid);
    if (rtai_linux_context[cpuid].depth) {
	if (!--rtai_linux_context[cpuid].depth) {
	    __clear_bit(cpuid, &rtai_cpu_realtime);
	    adeos_restore_pipeline_nosync(adp_root, rtai_linux_context[cpuid].oldflags, cpuid);
	}
    }
}

#define in_hrt_mode(cpuid)  (test_bit(cpuid, &rtai_cpu_realtime))

/* Private interface -- internal use only
 * ====================================== */

extern unsigned long	rtai_critical_enter(void (*synch)(void));
extern void		rtai_critical_exit(unsigned long flags);
extern void		rtai_set_linux_task_priority(struct task_struct *task,
						     int policy, int prio);

#endif /* __KERNEL__ && !__cplusplus */

/* Public interface
 * ================ */

#ifdef __KERNEL__

#include <linux/kernel.h>

/* printk is safe to use with Adeos */
#define rt_printk	      		printk
#define rtai_print_to_screen  		printk

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

extern int		rt_request_irq(unsigned irq, rt_irq_handler_t handler, void *cookie, int retmode);
extern int		rt_release_irq(unsigned irq);
extern void		rt_set_irq_cookie(unsigned irq, void *cookie);
extern unsigned		rt_startup_irq(unsigned irq);
extern void		rt_shutdown_irq(unsigned irq);
extern void		rt_enable_irq(unsigned irq);
extern void		rt_disable_irq(unsigned irq);
extern void		rt_mask_and_ack_irq(unsigned irq);
extern void		rt_unmask_irq(unsigned irq);
extern void		rt_ack_irq(unsigned irq);
extern int		rt_request_linux_irq(unsigned irq,
					     irqreturn_t (*handler)(int irq, void *dev_id,
								    struct pt_regs *regs),
					     char *name, void *dev_id);
extern int		rt_free_linux_irq(unsigned irq, void *dev_id);
extern void		rt_pend_linux_irq(unsigned irq);
extern void		rt_pend_linux_srq(unsigned srq);
extern int		rt_request_srq(unsigned label, void (*k_handler)(void),
				       long long (*u_handler)(unsigned));
extern int		rt_free_srq(unsigned srq);
extern int		rt_assign_irq_to_cpu(int irq, unsigned long cpus_mask);
extern int		rt_reset_irq_to_sym_mode(int irq);
extern void		rt_request_timer_cpuid(rt_timer_irq_handler_t handler, unsigned tick, int cpuid);
extern int		rt_request_timer(rt_timer_irq_handler_t handler, unsigned tick, int use_apic);
extern void		rt_free_timer(void);
extern RT_TRAP_HANDLER	rt_set_trap_handler(RT_TRAP_HANDLER handler);
extern void		rt_mount(void);
extern void		rt_umount(void);
extern void		(*rt_set_ihook(void (*hookfn)(int)))(int);

/* deprecated calls */

extern inline int
rt_request_global_irq(unsigned irq, void (*handler)(void))
{
    /* cast is okay because retmode is 0 (i.e. irq-handling doesn't use return
     * value of handler, so it does not need to have one) */ 
    return rt_request_irq(irq, (rt_irq_handler_t)handler, NULL, 0);
}

extern inline int
rt_request_global_irq_ext(unsigned irq, void (*handler)(void), unsigned long cookie)
{
    /* cast is okay because retmode is 0 (i.e. irq-handling doesn't use return
     * value of handler, so it does not need to have one) */ 
    return rt_request_irq(irq, (rt_irq_handler_t)handler, (void *)cookie, 0);
}

extern inline void
rt_set_global_irq_ext(unsigned irq, int ext, unsigned long cookie)
{
    rt_set_irq_cookie(irq, (void *)cookie);
}

extern inline int
rt_free_global_irq(unsigned irq)
{
    return rt_release_irq(irq);
}

#ifdef __cplusplus
}
#endif /* __cplusplus */

/* include rtai_timer.h late (it might need something from above)
 * it has to provide:
 * - rtai_timer_irq_ack()
 * - rtai_rdtsc()
 * - void rt_set_timer_delay(int delay)
 */
#include <asm-arm/arch/rtai_timer.h>

#endif /* __KERNEL__ */

#include <asm/rtai_oldnames.h>

#endif /* !_RTAI_ASM_ARM_HAL_H */
