/*                                      -*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: util.c,v 1.12 2000/08/16 10:08:58 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@linuxcare.com>
 * Copyright (C) 1999 by Andrew Tridgell <tridge@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* FIXME: In the long term we need to step away from using these functions
   and towards something that will abort the request in a friendly way
   (message to the client, etc) if they fail. */


#define BUFSIZE 8192

#include "config.h"
#include "sysheaders.h"

#include "rproxy.h"
#include "trace.h"
#include "util.h"
#include "error.h"


char const *
str_or_null(char const *s, char const *alt)
{
    if (s)
        return s;
    else if (alt)
        return alt;
    else
        return "(null)";
}


/* like fgets but strip off trailing \n and \r */
char           *
fgets_strip(char *s, int size, FILE * f)
{
    if (!fgets(s, size, f))
        return NULL;
    trim_crlf(s);
    return s;
}


/* Strip off trailing cr/lf */
void
trim_crlf(char *s)
{
    int             l = strlen(s);

    while (l > 0 && (s[l - 1] == '\n' || s[l - 1] == '\r'))
        l--;
    s[l] = 0;
}


/* dup a string and exit if it fails */
char           *
xstrdup(char const *s)
{
    char           *d;

    d = strdup(s);
    if (!d) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "Out of memory in xstrdup");
        abort();
    }
    return d;
}


/* dup a number of characters from a string or abort.  The returned string is 
   always nul-terminated. */
char           *
xstrndup(char const *s, size_t len)
{
    /* NB strndup is a glibc extension */
    char           *d;

    d = strndup(s, len);
    if (!d) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "out of memory in xstrndup");
        abort();
    }
    return d;
}


/* realloc some memory exit if it fails */
void           *
xrealloc(void *p, size_t length)
{
    if (!p)
        return xmalloc(length);
    p = realloc(p, length);
    if (!p) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "out of memory in xrealloc");
        exit(1);
    }
    return p;
}


/* malloc some memory exit if it fails */
void           *
xmalloc(size_t length)
{
    void           *p = malloc(length);

    if (!p) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "out of memory in xmalloc");
        exit(1);
    }
    return p;
}

FILE *
xfdopen(int fd, char const *mode)
{
    FILE           *f = fdopen(fd, mode);

    if (!f) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "fdopen failed");
        exit(1);
    }
    return f;
}


/* Read everything from a stream into a newly allocated buffer. Returns
   negative if something broke. */
ssize_t read_and_alloc(FILE * f, char **buf)
{
    ssize_t         total = 0;
    ssize_t         bufsize = BUFSIZE;
    ssize_t         r;

    if (!(*buf = xmalloc(bufsize)))
        return -1;

    while ((r = fread(*buf + total, 1, BUFSIZE, f)) > 0) {
        total += r;
        bufsize = total + BUFSIZE;
        if (!(*buf = xrealloc(*buf, bufsize)))
            goto fail;
    }

    if (r == 0)                 /* eof */
        return total;

  fail:
    if (*buf) {
        free(*buf);
        *buf = NULL;
    }
    return -1;
}


/* read data form  stream, looping till all data is read */
ssize_t read_all(FILE * f, char *buf, size_t size)
{
    ssize_t         total = 0;

    while (size) {
        ssize_t         n = fread(buf, 1, size, f);

        if (n == -1)
            return -1;
        if (n == 0)
            break;
        size -= n;
        buf += n;
        total += n;
    }
    return total;
}

/***************************************************************************
  decode a base64 string in-place - simple and slow algorithm

  See rfc1521 for the specification of base64.
  ***************************************************************************/
size_t base64_decode(char *s)
{
    char const *b64 =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    int             bit_offset, byte_offset, idx, i, n;
    unsigned char  *d = (unsigned char *) s;
    char           *p;

    n = i = 0;

    while (*s && (p = strchr(b64, *s))) {
        idx = (int) (p - b64);
        byte_offset = (i * 6) / 8;
        bit_offset = (i * 6) % 8;
        d[byte_offset] &= ~((1 << (8 - bit_offset)) - 1);
        if (bit_offset < 3) {
            d[byte_offset] |= (idx << (2 - bit_offset));
            n = byte_offset + 1;
        } else {
            d[byte_offset] |= (idx >> (bit_offset - 2));
            d[byte_offset + 1] = 0;
            d[byte_offset + 1] |= (idx << (8 - (bit_offset - 2))) & 0xFF;
            n = byte_offset + 2;
        }
        s++;
        i++;
    }

    return n;
}

/***************************************************************************
encode a buffer as base64 - simple and slow algorithm
  ***************************************************************************/
void
base64_encode(unsigned char const *buf, int n, char *out)
{
    char const *b64 =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    int             bytes, i;

    /* work out how many bytes of output there are */
    bytes = ((n * 8) + 5) / 6;

    for (i = 0; i < bytes; i++) {
        int             byte = (i * 6) / 8;
        int             bit = (i * 6) % 8;

        if (bit < 3) {
            if (byte >= n)
                abort();
            *out = b64[(buf[byte] >> (2 - bit)) & 0x3F];
        } else {
            if (byte + 1 == n) {
                *out = b64[(buf[byte] << (bit - 2)) & 0x3F];
            } else {
                *out = b64[(buf[byte] << (bit - 2) |
                            buf[byte + 1] >> (10 - bit)) & 0x3F];
            }
        }
        out++;
    }
    *out = 0;
}


/* this is like vsnprintf but it always null terminates, so you can fit at
   most n-1 chars in */
int
vslprintf(char *str, int n, const char *format, va_list ap)
{
    int             ret = vsnprintf(str, n, format, ap);

    if (ret >= n || ret < 0) {
        str[n - 1] = 0;
        return -1;
    }
    str[ret] = 0;
    return ret;
}


/* like snprintf but always null terminates */
int
slprintf(char *str, int n, char const *format, ...)
{
    va_list         ap;
    int             ret;

    va_start(ap, format);
    ret = vslprintf(str, n, format, ap);
    va_end(ap);
    return ret;
}


/****************************************************************************
  Get the next token from a string, return False if none found
  handles double-quotes. 
Based on a routine by GJC@VILLAGE.COM. 
Extensively modified by Andrew.Tridgell@anu.edu.au
****************************************************************************/
int
next_token(char const **ptr, char *buff, char const *sep)
{
    char const     *s;
    int             quoted;
    static char const *last_ptr = NULL;

    if (!ptr)
        ptr = &last_ptr;
    if (!ptr)
        return 0;;

    s = *ptr;

    /* default to simple separators */
    if (!sep)
        sep = " \t\n\r";

    /* find the first non sep char */
    while (*s && strchr(sep, *s))
        s++;

    /* nothing left? */
    if (!*s)
        return (0);

    /* copy over the token */
    for (quoted = 0; *s && (quoted || !strchr(sep, *s)); s++) {
        if (*s == '\"')
            quoted = !quoted;
        else
            *buff++ = *s;
    }

    *ptr = (*s) ? s + 1 : s;
    *buff = 0;
    last_ptr = *ptr;

    return 1;
}


/* replace all occurrences of from with to */
void
string_replchar(char *s, const char from, const char to)
{
    while (*s) {
        if (*s == from)
            *s = to;
        s++;
    }
}


/****************************************************************************
substitute one substring for another in a string
****************************************************************************/
void
string_sub(char *s, const char *pattern, const char *insert)
{
    char           *p;
    size_t          ls, lp, li;

    if (!insert || !pattern || !s)
        return;

    ls = strlen(s);
    lp = strlen(pattern);
    li = strlen(insert);

    if (!*pattern)
        return;

    while (lp <= ls && (p = strstr(s, pattern))) {
        memmove(p + li, p + lp, ls + 1 - (PTR_DIFF(p, s) + lp));
        memcpy(p, insert, li);
        s = p + li;
        ls += (li - lp);
    }
}


/*******************************************************************
trim the specified elements off the front and back of a string
********************************************************************/
int
trim_string(char *s, const char *front, const char *back)
{
    int             ret = 0;
    size_t          front_len = (front && *front) ? strlen(front) : 0;
    size_t          back_len = (back && *back) ? strlen(back) : 0;
    size_t          s_len;

    while (front_len && strncmp(s, front, front_len) == 0) {
        char           *p = s;

        ret = 1;
        while (1) {
            if (!(*p = p[front_len]))
                break;
            p++;
        }
    }

    if (back_len) {
        s_len = strlen(s);
        while ((s_len >= back_len) &&
               (strncmp(s + s_len - back_len, back, back_len) == 0)) {
            ret = 1;
            s[s_len - back_len] = '\0';
            s_len = strlen(s);
        }
    }

    return (ret);
}


/* Return the length of the prefix common to A and B. */
int
string_countprefix(char const *a, char const *b)
{
    int             i;

    for (i = 0;; i++, a++, b++) {
        if (!*a || !*b || *a != *b)
            return i;
    }
}



/*
 * Resize a dynamically-allocated string and append.
 */
char *
x_strappend(char *buf, char const *to_append)
{
    size_t len, old_len = 0;

    len = strlen(to_append) + 1;
    if (buf) {
        old_len = strlen(buf);
        len += old_len;
    }

    buf = realloc(buf, len);
    if (!buf) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't allocate %d bytes for string buffer",
               len);
        exit(EXIT_NOMEM);
    }

    strcpy(buf + old_len, to_append);
    
    return buf;
}
