/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: serve.c,v 1.33 2000/08/25 07:28:15 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@humbug.org.au>
 * Copyright (C) 1999 tridge@samba.org
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* XXX: If the resource already has a content-encoding of `gzip' (say)
 * then at the moment we don't do anything.  Since most gzips can't be
 * handled by rsync this is arguably the most useful behaviour.  We
 * could layer encodings, but I'm not sure it'd help.
 *
 * TODO: Don't check the URL if we have an upstream proxy but rather
 * assume it can handle ftp and gopher and so on. */

#include "config.h"
#include "sysheaders.h"

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "rproxy.h"

#include "cache.h"
#include "util.h"
#include "urllib.h"
#include "header.h"
#include "msgpage.h"
#include "trace.h"
#include "myname.h"
#include "xferlog.h"
#include "logfile.h"
#include "error.h"
#include "pass.h"
#include "decreq.h"


/* TODO: Send HTTP_LENGTH_REQUIRED if we don't get a request content-length

   From rfc2616: ``If the client is sending data, a server implementation
   using TCP SHOULD be careful to ensure that the client acknowledges receipt 
   of the packet(s) containing the response, before the server closes the
   input connection.''

   How interesting!  I wonder how we make sure of that? */

/* 
 * Handling of aborted requests:
 * 
 * TCP/IP Illustrated Vol2 s25.2: when a socket is closed, TCP checks
 * whether SO_LINGER is set and the linger time is zero.  If so, the
 * connection is aborted with an RST instead of TCP's normal close.
 *
 * RFC2616 s10.4:
 *
 * If the client is sending data, a server implementation using TCP
 * SHOULD be careful to ensure that the client acknowledges receipt of
 * the packet(s) containing the response, before the server closes the
 * input connection. If the client continues sending data to the
 * server after the close, the server's TCP stack will send a reset
 * packet to the client, which may erase the client's unacknowledged
 * input buffers before they can be read and interpreted by the HTTP
 * application.
 */


static char const *
rproxy_via_line(void)
{
    static char     via_line[200];

    /* TODO: Allow the user to hide their hostname/port if they want?  */
    slprintf(via_line, sizeof(via_line) - 1,
	     "1.0 %s:%d (%s/%s)",
	     config.my_hostname, config.listen_port, PACKAGE, VERSION);
    return via_line;
}


static void
ignore_sigpipe(void)
{
    struct sigaction act;

    bzero(&act, sizeof act);
    act.sa_handler = SIG_IGN;
    act.sa_flags = SA_RESTART;
    sigaction(SIGPIPE, &act, NULL);
}



/* Indicate the encodings that we will accept. */
static int
add_encoding_header(request_t * req)
{
    char           *h;

    /* FIXME: just using strstr is not a very good test for whether the
       string is present. */
    h = header_content(req->headers, "Accept-Encoding");

    /* FIXME: Instead, use an idempotent-add-token routine. */
    if (!h || !strstr(h, HSYNC_ENCODING_TOKEN))
	header_add_list(req->headers, "Accept-Encoding",
			HSYNC_ENCODING_TOKEN);

    return 0;
}


static int
serve_add_via_header(request_t * req)
{
    header_add_list(req->headers, "Via", rproxy_via_line());
    return 0;
}


/*
 * We don't do persistent connections at the moment (sorry), so this
 * is added to the request *and* response to make sure nobody thinks
 * we do.
 */
static void
serve_set_connection_headers(request_t *req)
{
    header_add(req->headers, "Connection", "close");
    header_remove(req->headers, "Proxy-Connection");
}


/* Examine the client's request, set fields based on headers from the
   clients. */
static int
examine_request(request_t * req)
{
    if (!will_do_version(req)) {
	rp_request_failed(req,
		       HTTP_VERSION_NOT_SUPPORTED,
		       PROGRAM ": %s not supported\n", req->http_version);
	return -1;
    }

    /* FIXME: If there are multiple signature headers, concatenate them. */

    req->req_signature = header_content(req->headers, SIGNATURE_HEADER);
    /* XXX: header_content_all */
    if (req->req_signature) {
	/* Decode base64 in place. */
	req->req_signature = xstrdup(req->req_signature);
	req->req_signature_len = base64_decode(req->req_signature);
    } else if (req->f_old_sig) {
	size_t          len;
	char           *sig_block;

	len = read_and_alloc(req->f_old_sig, &sig_block);
	if (len >= 8) {
	    char           *sig64 = malloc(2 * len);

	    base64_encode(sig_block, len, sig64);
	    trace(LOGAREA_PROTO, "retrieved signature %s", sig64);
	    header_add(req->headers, SIGNATURE_HEADER, sig64);
	    free(sig64);
	    free(sig_block);
	}
    }

    req->client_does_hsync =
	header_list_contains_token(req->headers, "Accept-Encoding",
				   HSYNC_ENCODING_TOKEN);

#ifdef RPROXY_RESPECT_NO_CACHE
    /* Probably this should be off; we ought to do particularly well on files 
       which are normally uncacheable. */
    req->client_wants_direct =
	header_list_contains_token(req->headers, "Pragma", "no-cache");
    if (req->client_wants_direct)
	trace(LOGAREA_PROTO, "client wants no-cache");
#endif				/* RPROXY_RESPECT_NO_CACHE */

    return 0;
}


static int
examine_response(request_t * req)
{
    char           *encoding;

    header_set(req->headers, "Connection", "close");
    header_add_list(req->headers, "Via", rproxy_via_line());
    req->content_length = header_ival(req->headers, "Content-Length");

    encoding = header_content(req->headers, "Content-Encoding");
    req->resp_content_type = header_content(req->headers, "Content-Type");

    /* FIXME: This is no good if the server sends its own
     * Content-Encoding tags.  We need instead to split this into a
     * list and look at the last element of it; also flag a warning if
     * an intermediate entry is hsync. */
    req->resp_rsync_encoded = encoding
	&& !strcasecmp(encoding, HSYNC_ENCODING_TOKEN);

    return 0;
}


/*
 * This is called on a new connection.  in_fd and out_fd are the socket.
 * (Well, they may not quite be a socket if we're running from stdin.)
 * 
 * We're probably in a child process, but not if we're in inetd mode.
 */
int
rp_serve(int read_fd, int write_fd, struct sockaddr *client_addr)
{
    request_t       real_req;
    request_t      *const req = &real_req;

    /* We ignore SIGPIPE, and instead handle EPIPE when we go to do
     * IO.  That's the plan, anyhow. */
    ignore_sigpipe();

    bzero(req, sizeof(request_t));

    rp_req_set_client_addr(req, client_addr);
    rp_req_find_local_addr(read_fd, req);

    rp_process_name("reading from %s:%d", 
                    req->client_addr_str, req->client_port);

    req->fd_from_client = read_fd;
    req->fd_to_client = write_fd;
    req->f_from_client = xfdopen(read_fd, "r");
    req->f_to_client = xfdopen(write_fd, "w");
    req->headers = header_set_new();
    setbuffer(req->f_to_client, NULL, BUFFER_SIZE);

    read_request_line(req);
    if (request_parse_line(req) != DONE)
	goto out;
    if (!request_check_protocol(req))
	goto out;
    if (!request_check_method(req))
	goto out;
    /* TODO: At this point, if the request method is CONNECT, then we
     * have to give up normal procession (freeing any allocated data)
     * and instead switch to a data-pumping mode.  Do we already have
     * code to do that? */
     
    request_parse_url(req);

    header_clear(req->headers);
    if (header_load(req->headers, req->f_from_client) < 0) {
	rp_request_failed(req,
		       HTTP_BAD_REQUEST,
		       PROGRAM ": couldn't read headers from client");
	return -1;
    }
    if (examine_request(req) < 0)
	return -1;

    if (want_to_decode(req)) {
	trace(LOGAREA_PROTO, "I want to get an hsync encoded response");
	/* prepare the headers to go upstream */
	add_encoding_header(req);
	cache_open(req, req->url, &req->f_old_cache, &req->f_old_sig);
	if (req->f_old_sig)
	    add_old_sig_header(req);
    }

    serve_add_via_header(req);
    serve_set_connection_headers(req);

    if (config.remove_sig)
	header_remove(req->headers, SIGNATURE_HEADER);

    /* send out the request to the upstream proxy */
    req->f_upstream = send_upstream(req);

    /* clear the headers */
    header_clear(req->headers);

    /* read the first line of the response */
    read_response_status_line(req);

    /* read the headers */
    header_load(req->headers, req->f_upstream);
    examine_response(req);
    serve_add_via_header(req);
    serve_set_connection_headers(req);

    if (will_decode_hsync(req)) {
	trace(LOGAREA_PROTO, "I will decode a hsync response");
	decode_hsync_request(req);
    } else if (will_encode_hsync(req)) {
	trace(LOGAREA_PROTO, "I will encode a hsync response");
	encode_hsync_request(req);
    } else {
	/* For whatever reason we don't want to encode or decode this
	   request. */
	rp_pass_through(req);
    }
    
    rp_log_stats(req);

  out:
    rp_request_free(req);
    /* FIXME: We can arrive here without doing anything; we still need
     * to work out a systematic way to handle and report errors. */
    /* TODO: If the transfer failed, log that too. */
    rp_log_transfer(req);
    rp_log(LOGAREA_PROTO, LOG_INFO, "completed request %s",
           req->req_line);

    return 0;
}
