
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

/* -*- mode: C; c-basic-offset: 4  -*- */ 
#ifndef __RLIST_H__
#define __RLIST_H__

/* Documentation!
 *
 * In order to use these lists, you must create your own structure to be
 * used as the entrys of the list.  This structure *MUST* start with
 * the first entry as 'RLIST_HEADER;'
 *
 * For Example:
 *
 * typedef struct _FooBar FooBar;
 * struct _FooBar {
 *	    RLIST_HEADER;
 *	    char *mydata;
 * }
 *
 * You can create the toplevel RList structure by using rlist_new
 * () (freeing with rlist_free ()), or you can allocate the list
 * structure yourself, and call RLIST_INIT(list) on it.  This
 * initializes the list, setting up the tail and head pointers.
 *
 */

typedef struct _RListHeader	RListHeader;
typedef struct _RListEntry	RListEntry;
typedef struct _RList		RList;

struct _RListHeader {
    RListEntry *next;
    RListEntry *prev;
};
#define RLIST_HEADER RListHeader rlist_header

struct _RListEntry {
    RLIST_HEADER; 
};

struct _RList {
    RListEntry *head;
    RListEntry *tail;
};


/* Initialize a newly created list structure */
#define RLIST_INIT(list) \
    (list)->head = (list)->tail = NULL;

#define RLIST_DESTRUCT(list) \
    (list)->head = (list)->tail = NULL;


/* See if a list is empty */
#define rlist_empty(list) \
    (list == NULL || (list)->head == NULL)

/* Get the first entry of the list */
#define rlist_first(list) \
    ((void *) (list)->head)

/* Get the last entry of the list */
#define rlist_last(list) \
    ((void *) (list)->tail)


/* Given an entry, get the next entry */
#define rlist_next(entry) \
    ((void *) (entry)->rlist_header.next)

/* Given an entry, get the previous entry */
#define rlist_prev(entry) \
    ((void *) (entry)->rlist_header.prev)


/* Insert a new entry at the start of the list */
#define rlist_prepend(list, entry) \
    do { \
        RList *roy_list_tmp = (list); \
        RListEntry *roy_list_entry_tmp = (void *) (entry); \
        \
        if (roy_list_tmp->head != NULL) { \
            roy_list_tmp->head->rlist_header.prev = roy_list_entry_tmp; \
                roy_list_entry_tmp->rlist_header.next = roy_list_tmp->head; \
                roy_list_entry_tmp->rlist_header.prev = NULL; \
                roy_list_tmp->head = roy_list_entry_tmp; \
        } else { \
            roy_list_tmp->head = roy_list_entry_tmp; \
            roy_list_tmp->head->rlist_header.next = NULL; \
            roy_list_tmp->head->rlist_header.prev = NULL; \
            roy_list_tmp->tail = roy_list_tmp->head; \
        } \
    } while (0)

/* Append a new entry to the end of the list */
#define rlist_append(list, entry) \
    do { \
        RList *roy_list_tmp = (list); \
        RListEntry *roy_list_entry_tmp = (void *) (entry); \
        \
        if (roy_list_tmp->tail != NULL) { \
            roy_list_entry_tmp->rlist_header.next = NULL; \
            roy_list_entry_tmp->rlist_header.prev = roy_list_tmp->tail; \
            roy_list_tmp->tail->rlist_header.next = roy_list_entry_tmp; \
            roy_list_tmp->tail = roy_list_tmp->tail->rlist_header.next; \
        } else { \
            roy_list_tmp->head = roy_list_entry_tmp; \
            roy_list_tmp->head->rlist_header.next = NULL; \
            roy_list_tmp->head->rlist_header.prev = NULL; \
            roy_list_tmp->tail = roy_list_tmp->head; \
        } \
    } while (0)

/* Remove a given entry from the list */
#define rlist_remove(list, entry) \
    do { \
        RList *roy_list_tmp = (list); \
        RListEntry *roy_list_entry_tmp = (void *) (entry); \
        \
        if (roy_list_entry_tmp == roy_list_tmp->head) { \
            roy_list_tmp->head = roy_list_tmp->head->rlist_header.next; \
            \
            if (roy_list_tmp->head == NULL) \
                roy_list_tmp->tail = roy_list_tmp->head; \
            else \
                roy_list_tmp->head->rlist_header.prev = NULL; \
        } else { \
            roy_list_entry_tmp->rlist_header.prev->rlist_header.next = \
			        roy_list_entry_tmp->rlist_header.next; \
            \
            if (roy_list_entry_tmp->rlist_header.next == NULL) \
                roy_list_tmp->tail = roy_list_entry_tmp->rlist_header.prev; \
            else \
                roy_list_entry_tmp->rlist_header.next->rlist_header.prev = \
				    roy_list_entry_tmp->rlist_header.prev; \
        } \
    } while (0)


/* Walk through a list, assigning 'variable' to the
 * list item through each iteration. */
#define RLIST_FOREACH(list, variable) \
    do { \
       	RListEntry *tmp_rlist_next_entry; \
	    for (variable = (void *) rlist_first(list), \
	       	tmp_rlist_next_entry = (void *) variable ? rlist_next (variable) : NULL; \
		variable != NULL; \
		variable = (void *) tmp_rlist_next_entry, \
		tmp_rlist_next_entry = tmp_rlist_next_entry ? \
		rlist_next(tmp_rlist_next_entry) : NULL)


/* Same as above, but start at the end, and walk backwards */
#define RLIST_FOREACH_REVERSE(list, variable) \
    do { \
	RListEntry *tmp_rlist_prev_entry; \
	    for (variable = (void *) rlist_last(list), \
		tmp_rlist_prev_entry = (void *) variable ? rlist_prev (variable) : NULL; \
		variable != NULL; \
		variable = (void *) tmp_rlist_prev_entry, \
		tmp_rlist_prev_entry = tmp_rlist_prev_entry ? rlist_prev(tmp_rlist_prev_entry) : NULL)
            
/* Create a new toplevel struct to hold a list */
RList *
rlist_new (void);
    
/* Free that toplevel struct as allocated above.  This will
 * not free the items of the list */
void
rlist_free (RList *list);

void *
rlist_nth (RList *list, unsigned int nth);

/* returns the number of elements in the list. 0-based */
int
rlist_len (RList *list);


/* compare two lists and return -1 if they are of different lengths but the
   first matching segments are the same, or the result of the comparator
   function as soon it does not evaluate to 0 for a pair of entries. */
typedef int (*RListComparatorFunction) (RListEntry *entry_a, RListEntry *entry_b);

int
rlist_compare (RList *a, RList *b, RListComparatorFunction comparator);


#endif /* __RLIST_H__ */



