/*
 * CP3D.cpp
 * $Id: CP3D.cpp,v 1.4 2003/03/05 13:59:09 jhirche Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Methods of the CP3D class
//  Purpose     : 

/** documentation stuff

  @author Michael Meissner
  
  @version 0.0 //see cvs docu
    
*/
    
    
// System
///////////
#include <iostream>

// Own
////////
#include "GeoGeneric.h"
#include "CP3D.h"
#include "CP4D.h"


#ifndef MIN
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a,b) (((a) > (b)) ? (a) : (b))
#endif

// Initialisation of statics:
double CP3D::epsilon= DOUBLE_EPSILON; //EPS_CP3D;



//////////////////////////////
// 
// MEMBER FUNCTIONS
//
//////////////////////////////

// Function   : print
// Parameters : 
// Purpose    : print all elements of CP3D to cerr
// Comments   : 
void CP3D::print() const
/*******************************************************************/
{  
  cerr << "(" << m_ard[0] << "," << m_ard[1] << "," << m_ard[2] << ")";
}



//////////////////////////////
// 
// OPERATORS
//
//////////////////////////////

// Function   : operator CP4D()
// Parameters : 
// Purpose    : for casting a CP3D to a CP4D
// Comments   : the fourth component becomes 1.0
CP3D::operator CP4D()  const
/*******************************************************************/
{
  return CP4D(m_ard[0], 
              m_ard[1], 
	      m_ard[2], 
	      1.0);
}



// Function   : operator==
// Parameters : 
// Purpose    : Compare two points for being equal
// Comments   : They are eaul if their difference is <= CP3D::epsilon
int CP3D::operator==(const CP3D& pnt) const
/*******************************************************************/
{
  return((fabs(m_ard[0]-pnt.m_ard[0]) < epsilon) &&
         (fabs(m_ard[1]-pnt.m_ard[1]) < epsilon) &&
         (fabs(m_ard[2]-pnt.m_ard[2]) < epsilon));
}



// Function   : operator!=
// Parameters : 
// Purpose    : Compare two points for not being equal
// Comments   : They differ if at least one component differs by <= CP3D::epsilon
int CP3D::operator!=(const CP3D& pnt) const 
/*******************************************************************/
{
  return  !( *this == pnt);
}



// Function   : operator+=
// Parameters : 
// Purpose    : add a vector to a point
// Comments   : 
CP3D& CP3D::operator+=(const CV3D& v)
/*******************************************************************/
{
   m_ard[0] += v[0];  
   m_ard[1] += v[1];  
   m_ard[2] += v[2];
   return *this;
}



// Function   : operator-=
// Parameters : 
// Purpose    : substract a vector from a point
// Comments   : 
CP3D& CP3D::operator-=(const CV3D& v)
/*******************************************************************/
{
   m_ard[0] -= v[0];  
   m_ard[1] -= v[1];  
   m_ard[2] -= v[2];
   return *this;
}



// Function   : operator*=
// Parameters : 
// Purpose    : multiply a vector to a point (HMM)
// Comments   : 
CP3D& CP3D::operator*=(const CV3D& v)
/*******************************************************************/
{
   m_ard[0] *= v[0];  
   m_ard[1] *= v[1];  
   m_ard[2] *= v[2];
   return *this;
}



// Function   : operator*=
// Parameters : 
// Purpose    : multiply a point by a scalar
// Comments   : 
CP3D& CP3D::operator*=(double rdFactor)
/*******************************************************************/
{
   m_ard[0] *= rdFactor;  
   m_ard[1] *= rdFactor;  
   m_ard[2] *= rdFactor;
   return *this;
}



// Function   : operator/=
// Parameters : 
// Purpose    : divide a point by a scalar
// Comments   : 
CP3D& CP3D::operator/=(double rdFactor)
/*******************************************************************/
{
   m_ard[0] /= rdFactor;  
   m_ard[1] /= rdFactor;  
   m_ard[2] /= rdFactor;
   return *this;
}



// Function   : operator+
// Parameters : 
// Purpose    : add a vector to a point
// Comments   : 
CP3D CP3D::operator+(const CV3D& v) const
/*******************************************************************/
{
  return CP3D(m_ard[0]+v[0],
              m_ard[1]+v[1],
	      m_ard[2]+v[2]);
}



// Function   : operator-
// Parameters : 
// Purpose    : substract a vector from a point
// Comments   : 
CP3D CP3D::operator-(const CV3D& v) const
/*******************************************************************/
{
  return CP3D(m_ard[0]-v[0],
              m_ard[1]-v[1],
	      m_ard[2]-v[2]);
}



// Function   : operator+
// Parameters : 
// Purpose    : add a point to a point
// Comments   : 
CP3D CP3D::operator+(const CP3D& pnt) const
/*******************************************************************/
{
  return CP3D(m_ard[0] + pnt.m_ard[0], 
              m_ard[1] + pnt.m_ard[1], 
	      m_ard[2] + pnt.m_ard[2]);
}



// Function   : operator-
// Parameters : 
// Purpose    : substract a point from a point
// Comments   : 
CV3D CP3D::operator-(const CP3D& p1) const 
/*******************************************************************/
{
  return CV3D(m_ard[0]-p1[0],
              m_ard[1]-p1[1],
	      m_ard[2]-p1[2]);
}



// Function   : operator*
// Parameters : 
// Purpose    : multiply a point by a vector
// Comments   : 
CP3D CP3D::operator*(const CV3D& vec) const
/*******************************************************************/
{
  return CP3D(m_ard[0] * vec[0], 
              m_ard[1] * vec[1], 
	      m_ard[2] * vec[2]);
}



// Function   : operator*
// Parameters : 
// Purpose    : multiply a point by a scalar
// Comments   : 
CP3D CP3D::operator*(double rdFactor) const
/*******************************************************************/
{
  return CP3D(m_ard[0] * rdFactor, 
              m_ard[1] * rdFactor, 
	      m_ard[2] * rdFactor);
}



// Function   : operator/
// Parameters : 
// Purpose    : divide a point by a vector
// Comments   : 
CP3D CP3D::operator/(const CV3D& vec) const
/*******************************************************************/
{
  return CP3D(m_ard[0] / vec[0], 
              m_ard[1] / vec[1], 
	      m_ard[2] / vec[2]);
}



// Function   : operator/
// Parameters : 
// Purpose    : divide a point by a scalar
// Comments   : 
CP3D CP3D::operator/(double rdFactor) const
/*******************************************************************/
{
  return CP3D(m_ard[0] / rdFactor, 
              m_ard[1] / rdFactor, 
	      m_ard[2] / rdFactor);
}



//////////////////////////////
// 
// MEMBER FUNCTIONS
//
//////////////////////////////

// Function  : getAbsMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP3D::getAbsMinComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]),
         vz = fabs(m_ard[2]);

  return (vx < vy) ? ((vx < vz) ? 0 : 2) : ((vy < vz) ? 1 : 2);
}



// Function  : getMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP3D::getMinComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1],
         vz = m_ard[2];

  return (vx < vy) ? ((vx < vz) ? 0 : 2) : ((vy < vz) ? 1 : 2);
}



// Function  : getAbsMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP3D::getAbsMaxComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]),
         vz = fabs(m_ard[2]);

  return (vx > vy) ? ((vx > vz) ? 0 : 2) : ((vy > vz) ? 1 : 2);
}



// Function  : getMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP3D::getMaxComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1],
         vz = m_ard[2];

  return (vx > vy) ? ((vx > vz) ? 0 : 2) : ((vy > vz) ? 1 : 2);
}



//////////////////////////////
// 
// HELPER FUNCTIONS
//
//////////////////////////////

// Function   : Min
// Parameters : const CP3D& Point1, const CP3D& Point2
// Purpose    : Returns a new point that has the minimum coordinates of both points.
// Comments   : 
CP3D MidPoint(const CP3D& Point1, const CP3D& Point2)
/*******************************************************************/
{ 
  return CP3D(Point1 + 0.5 * CV3D(Point2[0] - Point1[0],
                                  Point2[1] - Point1[1],
				  Point2[2] - Point1[2]));
}



// Function   : Min
// Parameters : const CP3D& Point1, const CP3D& Point2
// Purpose    : Returns a new point that has the minimum coordinates of both points.
// Comments   : 
CP3D Min(const CP3D& Point1, const CP3D& Point2)
/*******************************************************************/
{ 
  return CP3D(MIN(Point1[0], Point2[0]), 
              MIN(Point1[1], Point2[1]), 
	      MIN(Point1[2], Point2[2])); 
}



// Function   : Max
// Parameters : const CP3D& Point1, const CP3D& Point2
// Purpose    : Returns a new point that has the maximum coordinates of both points.
// Comments   : 
CP3D Max(const CP3D& Point1, const CP3D& Point2)
/*******************************************************************/
{ 
  return CP3D(MAX(Point1[0], Point2[0]), 
              MAX(Point1[1], Point2[1]), 
	      MAX(Point1[2], Point2[2])); 
}



// Function   : operator*
// Parameters : double rfValue, const CP3D& Point
// Purpose    : Returns a new scaled that 
// Comments   : 
CP3D operator*(double rfValue, const CP3D& Point)
/*******************************************************************/
{
   return CP3D(rfValue * Point[0], 
               rfValue * Point[1],
               rfValue * Point[2]);
}
