/*
 * CP2D.cpp
 * $Id: CP2D.cpp,v 1.3 2003/03/05 13:59:09 jhirche Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Methods of the CP2D class
//  Purpose     : 

/** documentation stuff

  @author Michael Meissner
  
  @version 0.0 //see cvs docu
    
*/
    
    
// System
///////////
#include <iostream>

// Own
////////
#include "GeoGeneric.h"
#include "CP2D.h"
#include "CV2D.h"
#include "CP4D.h"


#ifndef MIN
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a,b) (((a) > (b)) ? (a) : (b))
#endif

// Initialisation of statics:
double CP2D::epsilon= DOUBLE_EPSILON; //EPS_CP2D;



//////////////////////////////
// 
// MEMBER FUNCTIONS
//
//////////////////////////////

// Function   : print
// Parameters : 
// Purpose    : print all elements of CP2D to cerr
// Comments   : 
void CP2D::print() const
/*******************************************************************/
{  
  cerr << "(" << m_ard[0] << "," << m_ard[1] << ")";
}



//////////////////////////////
// 
// OPERATORS
//
//////////////////////////////

// Function   : operator CP3D()
// Parameters : 
// Purpose    : for casting a CP2D to a CP3D
// Comments   : the fourth component becomes 0.0
CP2D::operator CP3D()  const
/*******************************************************************/
{
  return CP3D(m_ard[0], 
              m_ard[1], 
	      0.0);
}



// Function   : operator==
// Parameters : 
// Purpose    : Compare two points for being equal
// Comments   : They are eaul if their difference is <= CP2D::epsilon
int CP2D::operator==(const CP2D& pnt) const
/*******************************************************************/
{
  return((fabs(m_ard[0]-pnt.m_ard[0]) < epsilon) &&
         (fabs(m_ard[1]-pnt.m_ard[1]) < epsilon));
}



// Function   : operator!=
// Parameters : 
// Purpose    : Compare two points for not being equal
// Comments   : They differ if at least one component differs by <= CP2D::epsilon
int CP2D::operator!=(const CP2D& pnt) const 
/*******************************************************************/
{
  return  !( *this == pnt);
}



// Function   : operator+=
// Parameters : 
// Purpose    : add a vector to a point
// Comments   : 
CP2D& CP2D::operator+=(const CV2D& v)
/*******************************************************************/
{
   m_ard[0] += v[0];  
   m_ard[1] += v[1];  
   return *this;
}



// Function   : operator-=
// Parameters : 
// Purpose    : substract a vector from a point
// Comments   : 
CP2D& CP2D::operator-=(const CV2D& v)
/*******************************************************************/
{
   m_ard[0] -= v[0];  
   m_ard[1] -= v[1];  
   return *this;
}



// Function   : operator*=
// Parameters : 
// Purpose    : multiply a vector to a point (HMM)
// Comments   : 
CP2D& CP2D::operator*=(const CV2D& v)
/*******************************************************************/
{
   m_ard[0] *= v[0];  
   m_ard[1] *= v[1];  
   return *this;
}



// Function   : operator*=
// Parameters : 
// Purpose    : multiply a point by a scalar
// Comments   : 
CP2D& CP2D::operator*=(double rdFactor)
/*******************************************************************/
{
   m_ard[0] *= rdFactor;  
   m_ard[1] *= rdFactor;  
   return *this;
}



// Function   : operator/=
// Parameters : 
// Purpose    : divide a point by a scalar
// Comments   : 
CP2D& CP2D::operator/=(double rdFactor)
/*******************************************************************/
{
   m_ard[0] /= rdFactor;  
   m_ard[1] /= rdFactor;  
   return *this;
}



// Function   : operator+
// Parameters : 
// Purpose    : add a vector to a point
// Comments   : 
CP2D CP2D::operator+(const CV2D& v) const
/*******************************************************************/
{
  return CP2D(m_ard[0]+v[0],
              m_ard[1]+v[1]);
}



// Function   : operator-
// Parameters : 
// Purpose    : substract a vector from a point
// Comments   : 
CP2D CP2D::operator-(const CV2D& v) const
/*******************************************************************/
{
  return CP2D(m_ard[0]-v[0],
              m_ard[1]-v[1]);
}



// Function   : operator+
// Parameters : 
// Purpose    : add a point to a point
// Comments   : 
CP2D CP2D::operator+(const CP2D& pnt) const
/*******************************************************************/
{
  return CP2D(m_ard[0] + pnt.m_ard[0], 
              m_ard[1] + pnt.m_ard[1]);
}



// Function   : operator-
// Parameters : 
// Purpose    : substract a point from a point
// Comments   : 
CV2D CP2D::operator-(const CP2D& p1) const 
/*******************************************************************/
{
  return CV2D(m_ard[0]-p1[0],
              m_ard[1]-p1[1]);
}



// Function   : operator*
// Parameters : 
// Purpose    : multiply a point by a vector
// Comments   : 
CP2D CP2D::operator*(const CV2D& vec) const
/*******************************************************************/
{
  return CP2D(m_ard[0] * vec[0], 
              m_ard[1] * vec[1]);
}



// Function   : operator*
// Parameters : 
// Purpose    : multiply a point by a scalar
// Comments   : 
CP2D CP2D::operator*(double rdFactor) const
/*******************************************************************/
{
  return CP2D(m_ard[0] * rdFactor, 
              m_ard[1] * rdFactor);
}



// Function   : operator/
// Parameters : 
// Purpose    : divide a point by a vector
// Comments   : 
CP2D CP2D::operator/(const CV2D& vec) const
/*******************************************************************/
{
  return CP2D(m_ard[0] / vec[0], 
              m_ard[1] / vec[1]);
}



// Function   : operator/
// Parameters : 
// Purpose    : divide a point by a scalar
// Comments   : 
CP2D CP2D::operator/(double rdFactor) const
/*******************************************************************/
{
  return CP2D(m_ard[0] / rdFactor, 
              m_ard[1] / rdFactor);
}



//////////////////////////////
// 
// MEMBER FUNCTIONS
//
//////////////////////////////

// Function  : getAbsMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP2D::getAbsMinComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]);

  return (vx < vy) ? 0 : 1;
}



// Function  : getMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP2D::getMinComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1];

  return (vx < vy) ? 0 : 1;
}



// Function  : getAbsMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP2D::getAbsMaxComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]);

  return (vx > vy) ? 0 : 1;
}



// Function  : getMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CP2D::getMaxComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1];

  return (vx > vy) ? 0 : 1;
}



//////////////////////////////
// 
// HELPER FUNCTIONS
//
//////////////////////////////

// Function   : Min
// Parameters : const CP2D& Point1, const CP2D& Point2
// Purpose    : Returns a new point that has the minimum coordinates of both points.
// Comments   : 
CP2D MidPoint(const CP2D& Point1, const CP2D& Point2)
/*******************************************************************/
{ 
  return CP2D(Point1 + 0.5 * CV2D(Point2[0] - Point1[0],
                                  Point2[1] - Point1[1]));
}



// Function   : Min
// Parameters : const CP2D& Point1, const CP2D& Point2
// Purpose    : Returns a new point that has the minimum coordinates of both points.
// Comments   : 
CP2D Min(const CP2D& Point1, const CP2D& Point2)
/*******************************************************************/
{ 
  return CP2D(MIN(Point1[0], Point2[0]), 
              MIN(Point1[1], Point2[1]));
}



// Function   : Max
// Parameters : const CP2D& Point1, const CP2D& Point2
// Purpose    : Returns a new point that has the maximum coordinates of both points.
// Comments   : 
CP2D Max(const CP2D& Point1, const CP2D& Point2)
/*******************************************************************/
{ 
  return CP2D(MAX(Point1[0], Point2[0]), 
              MAX(Point1[1], Point2[1]));
}

